@name `inject()` must be called from an injection context
@category runtime
@shortDescription `inject()` must be called from an injection context such as a constructor, a factory function, a field initializer, or a function used with `EnvironmentInjector#runInContext`.

@description
You see this error when you try to use the `inject()` function outside of the allowed injection context. The injection context is available during the class creation and initialization. It is also available to functions
used with `EnvironmentInjector#runInContext`.

In practice the `inject()` calls are allowed in a constructor, a constructor parameter and a field initializer:

```typescript
@Injectable({providedIn: 'root'})
export class Car {
  radio: Radio|undefined;

  // OK: field initializer
  spareTyre = inject(Tyre);
  
  constructor() {
    // OK: constructor body
    this.radio = inject(Radio);
  }
}
```

It is also legal to call `inject` from a provider's factory:

```typescript
providers: [
  {provide: Car, useFactory: () => {
    // OK: a class factory
    const engine = inject(Engine);
    return new Car(engine);
  }}
]
```

Calls to the `inject()` function outside of the class creation or `runInContext` will result in error. Most notably, calls to `inject()` are disallowed after a class instance was created, in methods (including lifecycle hooks):

```typescript
@Component({ ... })
export class CarComponent {
  ngOnInit() {
    // ERROR: too late, the component instance was already created
    const engine = inject(Engine);
    engine.start();
  }
}
```

@debugging

Work backwards from the stack trace of the error to identify a place where the disallowed call to `inject()` is located. 

To fix the error move the `inject()` call to an allowed place (usually a class constructor or a field initializer).

<!-- links -->

<!-- external links -->

<!-- end links -->

@reviewed 2022-05-27
