/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */

import {runfiles} from '@bazel/runfiles';
import fs from 'fs';
import path from 'path';
import shx from 'shelljs';

import {matchesObjectWithOrder} from './test_utils';

// Resolve the "npm_package" directory by using the runfile resolution. Note that we need to
// resolve the "package.json" of the package since otherwise NodeJS would resolve the "main"
// file, which is not necessarily at the root of the "npm_package".
shx.cd(path.dirname(runfiles.resolve('angular/packages/common/npm_package/package.json')));

describe('@angular/common ng_package', () => {
  describe('should have the locales files', () => {
    it('/locales', () => {
      const files = shx.ls('locales').stdout.split('\n');
      expect(files.some(n => n.endsWith('.d.ts'))).toBe(true, `.d.ts files don't exist`);
      expect(files.some(n => n.endsWith('.mjs'))).toBe(true, `.mjs files don't exist`);
    });
    it('/locales/extra', () => {
      const files = shx.ls('locales/extra').stdout.split('\n');
      expect(files.some(n => n.endsWith('.d.ts'))).toBe(true, `.d.ts files don't exist`);
      expect(files.some(n => n.endsWith('.mjs'))).toBe(true, `.mjs files don't exist`);
    });
  });

  it('should have right fesm files', () => {
    const expected = [
      'common.mjs',
      'common.mjs.map',
      'http',
      'http.mjs',
      'http.mjs.map',
      'http/testing.mjs',
      'http/testing.mjs.map',
      'testing.mjs',
      'testing.mjs.map',
      'upgrade.mjs',
      'upgrade.mjs.map',
    ];
    expect(shx.ls('-R', 'fesm2015').stdout.split('\n').filter(n => !!n).sort()).toEqual(expected);
    expect(shx.ls('-R', 'fesm2020').stdout.split('\n').filter(n => !!n).sort()).toEqual(expected);
  });

  it('should have the correct source map paths', () => {
    expect(shx.grep('sourceMappingURL', 'fesm2020/common.mjs'))
        .toMatch('//# sourceMappingURL=common.mjs.map');
    expect(shx.grep('sourceMappingURL', 'fesm2020/http.mjs'))
        .toMatch('//# sourceMappingURL=http.mjs.map');
    expect(shx.grep('sourceMappingURL', 'fesm2020/http/testing.mjs'))
        .toMatch('//# sourceMappingURL=testing.mjs.map');
    expect(shx.grep('sourceMappingURL', 'fesm2020/testing.mjs'))
        .toMatch('//# sourceMappingURL=testing.mjs.map');
    expect(shx.grep('sourceMappingURL', 'fesm2020/upgrade.mjs'))
        .toMatch('//# sourceMappingURL=upgrade.mjs.map');
  });

  describe('should have module resolution properties in the package.json file for', () => {
    interface PackageJson {
      main: string;
      fesm2015: string;
      es2020: string;
      module: string;
      typings: string;
      exports: object;
    }
    // https://github.com/angular/common-builds/blob/master/package.json
    it('/', () => {
      const actual =
          JSON.parse(fs.readFileSync('package.json', {encoding: 'utf-8'})) as PackageJson;

      expect(actual).toEqual(jasmine.objectContaining({
        module: `./fesm2015/common.mjs`,
        es2020: `./fesm2020/common.mjs`,
        esm2020: `./esm2020/common.mjs`,
        fesm2020: `./fesm2020/common.mjs`,
        fesm2015: `./fesm2015/common.mjs`,
        typings: `./index.d.ts`,
        exports: matchesObjectWithOrder({
          './locales/global/*': {default: './locales/global/*.js'},
          './locales/*': {default: './locales/*.mjs'},
          './package.json': {default: './package.json'},
          '.': {
            types: './index.d.ts',
            esm2020: './esm2020/common.mjs',
            es2020: './fesm2020/common.mjs',
            es2015: './fesm2015/common.mjs',
            node: './fesm2015/common.mjs',
            default: './fesm2020/common.mjs'
          },
          './http': {
            types: './http/index.d.ts',
            esm2020: './esm2020/http/http.mjs',
            es2020: './fesm2020/http.mjs',
            es2015: './fesm2015/http.mjs',
            node: './fesm2015/http.mjs',
            default: './fesm2020/http.mjs'
          },
          './http/testing': {
            types: './http/testing/index.d.ts',
            esm2020: './esm2020/http/testing/testing.mjs',
            es2020: './fesm2020/http/testing.mjs',
            es2015: './fesm2015/http/testing.mjs',
            node: './fesm2015/http/testing.mjs',
            default: './fesm2020/http/testing.mjs'
          },
          './testing': {
            types: './testing/index.d.ts',
            esm2020: './esm2020/testing/testing.mjs',
            es2020: './fesm2020/testing.mjs',
            es2015: './fesm2015/testing.mjs',
            node: './fesm2015/testing.mjs',
            default: './fesm2020/testing.mjs'
          },
          './upgrade': {
            types: './upgrade/index.d.ts',
            esm2020: './esm2020/upgrade/upgrade.mjs',
            es2020: './fesm2020/upgrade.mjs',
            es2015: './fesm2015/upgrade.mjs',
            node: './fesm2015/upgrade.mjs',
            default: './fesm2020/upgrade.mjs'
          }
        }),
      }));
    });
    // https://github.com/angular/common-builds/blob/master/http
    it('/http', () => {
      expect(fs.existsSync('http/index.d.ts')).toBe(true);
    });
    // https://github.com/angular/common-builds/blob/master/testing
    it('/testing', () => {
      expect(fs.existsSync('testing/index.d.ts')).toBe(true);
    });
    // https://github.com/angular/common-builds/blob/master/http/testing
    it('/http/testing', () => {
      expect(fs.existsSync('http/testing/index.d.ts')).toBe(true);
    });
    // https://github.com/angular/common-builds/blob/master/upgrade
    it('/upgrade', () => {
      expect(fs.existsSync('upgrade/index.d.ts')).toBe(true);
    });
  });
});
