/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import ts from 'typescript';

import {ClassDeclaration, Declaration, Decorator, ReflectionHost} from '../../../src/ngtsc/reflection';
import {SymbolWithValueDeclaration} from '../../../src/ngtsc/util/src/typescript';

/**
 * The symbol corresponding to a "class" declaration. I.e. a `ts.Symbol` whose `valueDeclaration` is
 * a `ClassDeclaration`.
 */
export type ClassSymbol = ts.Symbol&{valueDeclaration: ClassDeclaration};

/**
 * A representation of a class that accounts for the potential existence of two `ClassSymbol`s for a
 * given class, as the compiled JavaScript bundles that ngcc reflects on can have two declarations.
 */
export interface NgccClassSymbol {
  /**
   * The name of the class.
   */
  name: string;

  /**
   * Represents the symbol corresponding with the outer declaration of the class. This should be
   * considered the public class symbol, i.e. its declaration is visible to the rest of the program.
   */
  declaration: ClassSymbol;

  /**
   * Represents the symbol corresponding with the inner declaration of the class, referred to as its
   * "implementation". This is not necessarily a `ClassSymbol` but rather just a `ts.Symbol`, as the
   * inner declaration does not need to satisfy the requirements imposed on a publicly visible class
   * declaration.
   */
  implementation: SymbolWithValueDeclaration;

  /**
   * Represents the symbol corresponding to a variable within a class IIFE that may be used to
   * attach static properties or decorated.
   */
  adjacent?: SymbolWithValueDeclaration;
}

/**
 * A reflection host that has extra methods for looking at non-Typescript package formats
 */
export interface NgccReflectionHost extends ReflectionHost {
  /**
   * Find a symbol for a declaration that we think is a class.
   * @param declaration The declaration whose symbol we are finding
   * @returns the symbol for the declaration or `undefined` if it is not
   * a "class" or has no symbol.
   */
  getClassSymbol(declaration: ts.Node): NgccClassSymbol|undefined;

  /**
   * Retrieves all decorators of a given class symbol.
   * @param symbol Class symbol that can refer to a declaration which can hold decorators.
   * @returns An array of decorators or null if none are declared.
   */
  getDecoratorsOfSymbol(symbol: NgccClassSymbol): Decorator[]|null;

  /**
   * Retrieves all class symbols of a given source file.
   * @param sourceFile The source file to search for classes.
   * @returns An array of found class symbols.
   */
  findClassSymbols(sourceFile: ts.SourceFile): NgccClassSymbol[];

  /**
   * Find the last node that is relevant to the specified class.
   *
   * As well as the main declaration, classes can have additional statements such as static
   * properties (`SomeClass.staticProp = ...;`) and decorators (`__decorate(SomeClass, ...);`).
   * It is useful to know exactly where the class "ends" so that we can inject additional
   * statements after that point.
   *
   * @param classSymbol The class whose statements we want.
   */
  getEndOfClass(classSymbol: NgccClassSymbol): ts.Node;

  /**
   * Check whether a `Declaration` corresponds with a known declaration and set its `known` property
   * to the appropriate `KnownDeclaration`.
   *
   * @param decl The `Declaration` to check.
   * @return The passed in `Declaration` (potentially enhanced with a `KnownDeclaration`).
   */
  detectKnownDeclaration<T extends Declaration>(decl: T): T;
}
