#
# spec file
#
# Copyright (c) 2024 SUSE LLC
#
# All modifications and additions to the file contributed by third parties
# remain the property of their copyright owners, unless otherwise agreed
# upon. The license for this file, and modifications and additions to the
# file, is the same license as for the pristine package itself (unless the
# license for the pristine package is not an Open Source License, in which
# case the license is the MIT License). An "Open Source License" is a
# license that conforms to the Open Source Definition (Version 1.9)
# published by the Open Source Initiative.

# Please submit bugfixes or comments via https://bugs.opensuse.org/
#


# Run with osc --with=fast_build to have a shorter turnaround
# It will avoid building some parts of glibc
%bcond_with    fast_build

%bcond_with ringdisabled

%define flavor %{nil}

# We need to map from flavor to cross-arch, but as we need the
# result in BuildRequires where the build service evaluates, we
# can use only simple RPM expressions, no lua, no shell, no '{expand:'
# expression :-/  Ideally we'd like to just strip the 'cross_' prefix,
# but we can't.  So enumerate the possibilities for now.
%define cross_arch %{cross_cpu}
%if "%flavor" == "cross-aarch64"
%define cross_cpu aarch64
%endif
%if "%flavor" == "cross-riscv64"
%define cross_cpu riscv64
%endif
%if "%flavor" == "cross-s390x"
%define cross_cpu s390x
%endif
%if "%flavor" == "cross-ppc64le"
%define cross_arch ppc64le
%define cross_cpu powerpc64le
%endif

%if 0%{?cross_cpu:1}
%define binutils_os %{cross_cpu}-suse-linux
# use same sysroot as in binutils.spec
%define sysroot %{_prefix}/%{binutils_os}/sys-root
%endif

%if 0%{?suse_version} >= 1550
%bcond_without usrmerged
%else
%bcond_with usrmerged
%endif

%if 0%{?gcc_version} < 13
%define with_gcc 13
%endif

%bcond_with build_all
%define build_main 1
%define build_utils %{with build_all}
%define build_testsuite %{with build_all}
%define build_cross 0
%if "%flavor" == "utils"
%if %{with ringdisabled}
ExclusiveArch:  do_not_build
%endif
%define build_main 0
%define build_utils 1
%define build_testsuite 0
%endif
%if "%flavor" == "testsuite"
%if %{with ringdisabled}
ExclusiveArch:  do_not_build
%endif
%define build_main 0
%define build_utils 0
%define build_testsuite 1
%endif
%if 0%{?cross_cpu:1}
%define build_main 0
%define build_utils 0
%define build_testsuite 0
%define build_cross 1
%undefine _build_create_debug
ExcludeArch:    %{cross_arch}
%if 0%{?suse_version} < 1600
ExclusiveArch:  do_not_build
%endif
%endif
%define host_arch %{?cross_cpu}%{!?cross_cpu:%{_target_cpu}}

# Enable support for livepatching.
%define have_livepatching_support 0
%if %{build_cross}
%if "%{cross_arch}" == "x86_64" ||  "%{cross_arch}" == "ppc64le"
%define have_livepatching_support 1
%endif
%else
%ifarch x86_64 ppc64le
%define have_livepatching_support 1
%endif
%endif

%if %{have_livepatching_support}
%bcond_without livepatching
%else
%bcond_with livepatching
%endif

%if %{build_main}
%define name_suffix %{nil}
%else
%define name_suffix -%{flavor}-src
%endif

%define __filter_GLIBC_PRIVATE 1
%ifarch i686
# For i686 let's only build what's different from i586, so
# no need to build documentation
%define build_profile 1
%define build_locales 1
%define build_html 0
%else
%if %{with fast_build} || %{build_cross} || %{build_utils} && %{without build_all}
%define build_profile 0
%define build_locales 0
%define build_html 0
%else
# Default:
%define build_profile 1
%define build_locales 1
%define build_html 1
%endif
%endif

%define build_variants %{build_main}

%define disable_assert 0
%define enable_stackguard_randomization 1
# glibc requires at least kernel 3.2
# Bump to 4.3 to enable use of direct socketcalls on x86-32 and s390x
%define enablekernel 4.3
# some architectures need a newer kernel
%ifarch riscv64
%define enablekernel 4.15
%endif

Name:           glibc%{name_suffix}
Summary:        Standard Shared Libraries (from the GNU C Library)
License:        GPL-2.0-or-later AND LGPL-2.1-or-later AND LGPL-2.1-or-later WITH GCC-exception-2.0
Group:          System/Libraries
Version:        2.38
Release:        150600.14.37.1
%define git_id 36f2487f13
%define libversion %version
URL:            https://www.gnu.org/software/libc/libc.html
BuildRoot:      %{_tmppath}/%{name}-%{version}-build
Source:         https://ftp.gnu.org/pub/gnu/glibc/glibc-%{version}.tar.xz
Source1:        https://ftp.gnu.org/pub/gnu/glibc/glibc-%{version}.tar.xz.sig
Source2:        http://savannah.gnu.org/project/memberlist-gpgkeys.php?group=libc&download=1#/glibc.keyring
Source4:        manpages.tar.bz2
Source5:        nsswitch.conf
Source7:        bindresvport.blacklist
Source9:        glibc.rpmlintrc
Source10:       baselibs.conf
# For systemd
Source20:       nscd.conf
Source21:       nscd.service
Source22:       nscd.sysusers

%if %{build_main}
# ngpt was used in 8.1 and SLES8
Obsoletes:      ngpt < 2.2.2
Obsoletes:      ngpt-devel < 2.2.2
Provides:       ngpt = 2.2.2
Provides:       ngpt-devel = 2.2.2
Conflicts:      kernel < %{enablekernel}
%if %{with usrmerged}
# make sure we have post-usrmerge filesystem package
Conflicts:      filesystem < 15.6
%endif
# bug437293 - handle update from SLES10 on PowerPC
%ifarch ppc64
Obsoletes:      glibc-64bit
%endif
%ifarch ppc
Obsoletes:      glibc-32bit
%endif
%ifarch armv6hl armv7hl
# The old runtime linker link gets not provided by rpm find.provides, but it exists
Provides:       ld-linux.so.3
Provides:       ld-linux.so.3(GLIBC_2.4)
%endif
Requires(pre):  filesystem
Recommends:     glibc-extra
Provides:       rtld(GNU_HASH)
%endif
%if %{build_utils}
Requires:       glibc = %{version}
%endif
BuildRoot:      %{_tmppath}/%{name}-%{version}-build
BuildRequires:  audit-devel
BuildRequires:  bison
BuildRequires:  fdupes
BuildRequires:  libcap-devel
BuildRequires:  libselinux-devel
BuildRequires:  makeinfo
BuildRequires:  python3-base
BuildRequires:  systemd-rpm-macros
BuildRequires:  systemtap-headers
BuildRequires:  sysuser-tools
BuildRequires:  xz
%if 0%{?with_gcc:1}
BuildRequires:  gcc%{with_gcc}
%endif
%if %{build_testsuite}
BuildRequires:  gcc%{?with_gcc}-c++
BuildRequires:  gdb
BuildRequires:  glibc-devel-static
BuildRequires:  libidn2-0
BuildRequires:  libstdc++-devel
BuildRequires:  python3-pexpect
%endif
%if %{build_utils}
BuildRequires:  gd-devel
BuildRequires:  libpng-devel
BuildRequires:  zlib-devel
%endif
%if %{build_cross}
BuildRequires:  cross-%{cross_arch}-gcc%{!?with_gcc:%{gcc_version}}%{?with_gcc}-bootstrap
BuildRequires:  cross-%{cross_arch}-linux-glibc-devel
%endif
%if "%flavor" == "i686"
ExclusiveArch:  i586 i686
BuildArch:      i686
# Sync only this build counter with the main build
#!BcntSyncTag:  glibc
%endif

###
# Patches are ordered in the following groups:
# Patches that we will never upstream or which have not been looked at: 0-999
# Patches taken from upstream: 1000-2000
# Patches that are going upstream, waiting approval: 2000-3000
###

###
# Patches that upstream will not accept
###

###
# openSUSE specific patches - won't go upstream
###
### openSUSE extensions, configuration
# PATCH-FIX-OPENSUSE Fix path for nscd databases
Patch6:         glibc-2.3.3-nscd-db-path.diff
# PATCH-FIX-OPENSUSE Fix path for nss_db (bnc#753657) - aj@suse.de
Patch7:         nss-db-path.patch
# PATCH-FIX-OPENSUSE adjust nscd.conf
Patch8:         glibc-nscd.conf.patch
# PATCH-FIX-OPENSUSE -- add some extra information to version output - kukuk@suse.de
Patch10:        glibc-version.diff
# PATCH-FIX-OPENSUSE -- Make --no-archive default for localedef - kukuk@suse.de
Patch13:        glibc-2.3.2.no_archive.diff
# PATCH-FIX-OPENSUSE -- add blacklist for bindresvport
Patch14:        glibc-bindresvport-blacklist.diff
# PATCH-FIX-OPENSUSE prefer -lang rpm packages
Patch15:        glibc-2.3.90-langpackdir.diff
# PATCH-FEATURE-SLE Use nscd user for nscd
Patch19:        nscd-server-user.patch
# PATCH-FEATURE-SLE read nsswich.conf from /usr
Patch20:        glibc-nsswitch-usr.diff
# PATCH-FEATURE-SLE Add ULP prologue into ASM functions
Patch21:        ulp-prologue-into-asm-functions.patch
# PATCH-FIX-SLE Reduce chance of crash when using nscd GETFDHST
Patch22:        nscd-gethst-race.patch

### Locale related patches
# PATCH-FIX-OPENSUSE Add additional locales
Patch100:       add-locales.patch
# PATCH-FIX-OPENSUSE -- Add no_NO back (XXX: Still needed?)
Patch102:       glibc-2.4.90-no_NO.diff
# PATCH-FIX-OPENSUSE -- Renames for China
Patch103:       glibc-2.4-china.diff

### Network related patches
# PATCH-FIX-OPENSUSE Warn about usage of mdns in resolv.conv
Patch304:       glibc-resolv-mdnshint.diff
# PATCH-FIX-OPENSUSE disable rewriting ::1 to 127.0.0.1 for /etc/hosts bnc#684534, bnc#706719
Patch306:       glibc-fix-double-loopback.diff

###
# Patches from upstream
###
# PATCH-FIX-UPSTREAM iconv: restore verbosity with unrecognized encoding names (BZ #30694)
Patch1000:      iconv-error-verbosity.patch
# PATCH-FIX-UPSTREAM x86: Fix for cache computation on AMD legacy cpus
Patch1001:      cache-amd-legacy.patch
# PATCH-FIX-UPSTREAM x86: Fix incorrect scope of setting `shared_per_thread` (BZ# 30745)
Patch1002:      cache-intel-shared.patch
# PATCH-FIX-UPSTREAM malloc: Enable merging of remainders in memalign, remove bin scanning from memalign (BZ #30723)
Patch1003:      posix-memalign-fragmentation.patch
# PATCH-FIX-UPSTREAM intl: Treat C.UTF-8 locale like C locale (BZ #16621)
Patch1004:      intl-c-utf-8-like-c-locale.patch
# PATCH-FIX-UPSTREAM io: Fix record locking contants for powerpc64 with __USE_FILE_OFFSET64 (BZ #30804)
Patch1005:      ppc64-flock-fob64.patch
# PATCH-FIX-UPSTREAM libio: Fix oversized __io_vtables
Patch1006:      libio-io-vtables.patch
# PATCH-FIX-UPSTREAM elf: Do not run constructors for proxy objects
Patch1007:      call-init-proxy-objects.patch
# PATCH-FIX-UPSTREAM Stack read overflow with large TCP responses in no-aaaa mode (CVE-2023-4527, BZ #30842)
Patch1008:      no-aaaa-read-overflow.patch
# PATCH-FIX-UPSTREAM getaddrinfo: Fix use after free in getcanonname (CVE-2023-4806, BZ #30843)
Patch1009:      getcanonname-use-after-free.patch
# PATCH-FIX-UPSTREAM Fix leak in getaddrinfo introduced by the fix for CVE-2023-4806 (CVE-2023-5156, BZ #30884)
Patch1010:      getaddrinfo-memory-leak.patch
# PATCH-FIX-UPSTREAM io: Do not implement fstat with fstatat
Patch1011:      fstat-implementation.patch
# PATCH-FIX-UPSTREAM Propagate GLIBC_TUNABLES in setxid binaries
Patch1012:      setxid-propagate-glibc-tunables.patch
# PATCH-FIX-UPSTREAM tunables: Terminate if end of input is reached (CVE-2023-4911)
Patch1013:      tunables-string-parsing.patch
# PATCH-FIX-UPSTREAM add GB18030-2022 charmap and test the entire GB18030 charmap (BZ #30243)
Patch1014:      gb18030-2022.patch
# PATCH-FIX-UPSTREAM aarch64: correct CFI in rawmemchr (BZ #31113)
Patch1015:      aarch64-rawmemchr-unwind.patch
# PATCH-FIX-UPSTREAM sysdeps: sem_open: Clear O_CREAT when semaphore file is expected to exist (BZ #30789)
Patch1016:      sem-open-o-creat.patch
# PATCH-FIX-UPSTREAM elf: Fix wrong break removal from 8ee878592c
Patch1017:      ldconfig-process-elf-file.patch
# PATCH-FIX-UPSTREAM elf: Fix TLS modid reuse generation assignment (BZ #29039)
Patch1018:      tls-modid-reuse.patch
# PATCH-FIX-UPSTREAM getaddrinfo: translate ENOMEM to EAI_MEMORY (BZ #31163)
Patch1019:      getaddrinfo-eai-memory.patch
# PATCH-FIX-UPSTREAM libio: Check remaining buffer size in _IO_wdo_write (BZ #31183)
Patch1020:      libio-wdo-write.patch
# PATCH-FIX-UPSTREAM syslog: Fix heap buffer overflow in __vsyslog_internal (CVE-2023-6246, CVE-2023-6779, CVE-2023-6780)
Patch1021:      syslog-buffer-overflow.patch
# PATCH-FIX-UPSTREAM qsort: handle degenerated compare function
Patch1022:      qsort-invalid-cmp.patch
# PATCH-FIX-UPSTREAM S390: Do not clobber r7 in clone (BZ #31402)
Patch1023:      s390-clone-error-clobber-r7.patch
# PATCH-FIX-UPSTREAM duplocale: protect use of global locale (BZ #23970)
Patch1024:      duplocale-global-locale.patch
# PATCH-FIX-UPSTREAM malloc: Use __get_nprocs on arena_get2 (BZ #30945)
Patch1025:      malloc-arena-get2.patch
# PATCH-FIX-UPSTREAM linux: Use rseq area unconditionally in sched_getcpu (BZ #31479)
Patch1026:      sched-getcpu-rseq-area.patch
# PATCH-FIX-UPSTREAM iconv: ISO-2022-CN-EXT: fix out-of-bound writes when writing escape sequence (CVE-2024-2961)
Patch1027:      iconv-iso-2022-cn-ext.patch
# PATCH-FIX-UPSTREAM nscd: Stack-based buffer overflow in netgroup cache (CVE-2024-33599, BZ #31677)
Patch1028:      glibc-CVE-2024-33599-nscd-Stack-based-buffer-overflow-in-n.patch
# PATCH-FIX-UPSTREAM nscd: Do not send missing not found response in addgetnetgrentX (CVE-2024-33600, BZ #31678)
Patch1029:      glibc-CVE-2024-33600-nscd-Do-not-send-missing-not-found-re.patch
# PATCH-FIX-UPSTREAM nscd: Avoid null pointer crashes after notfound response (CVE-2024-33600, BZ #31678) 
Patch1030:      glibc-CVE-2024-33600-nscd-Avoid-null-pointer-crashes-after.patch
# PATCH-FIX-UPSTREAM nscd netgroup: Use two buffers in addgetnetgrentX (CVE-2024-33601, CVE-2024-33602, BZ #31680)
Patch1031:      glibc-CVE-2024-33601-CVE-2024-33602-nscd-netgroup-Use-two.patch
# PATCH-FIX-UPSTREAM nscd: Use time_t for return type of addgetnetgrentX (CVE-2024-33602)
Patch1032:      nscd-netgroup-cache-timeout.patch
# PATCH-FIX-UPSTREAM s390x: Fix segfault in wcsncmp (BZ #31934)
Patch1033:      s390x-wcsncmp.patch
# PATCH-FIX-UPSTREAM Linux: Switch back to assembly syscall wrapper for prctl (BZ #29770)
Patch1034:      prctl-syscall-wrapper.patch
# PATCH-FIX-UPSTREAM Fix underallocation of abort_msg_s struct (CVE-2025-0395, BZ #32582)
Patch1035:      assert-message-allocation.patch
# PATCH-FIX-UPSTREAM pthreads NPTL: lost wakeup fix 2 (BZ #25847)
Patch1036:      pthread-wakeup.patch
# PATCH-FIX-UPSTREAM elf: Ignore LD_LIBRARY_PATH and debug env var for setuid for static (CVE-2025-4802)
Patch1037:      static-setuid-ld-library-path.patch
# PATCH-FIX-UPSTREAM posix: Fix double-free after allocation failure in regcomp (BZ #33185)
Patch1038:      regcomp-double-free.patch

###
# Patches awaiting upstream approval
###
# PATCH-FIX-UPSTREAM Avoid concurrency problem in ldconfig (BZ #23973)
Patch2000:      ldconfig-concurrency.patch

# Non-glibc patches
# PATCH-FIX-OPENSUSE Remove debianisms from manpages
Patch3000:      manpages.patch

%description
The GNU C Library provides the most important standard libraries used
by nearly all programs: the standard C library, the standard math
library, and the POSIX thread library. A system is not functional
without these libraries.

%package -n glibc-utils
Summary:        Development utilities from the GNU C Library
License:        LGPL-2.1-or-later
Group:          Development/Languages/C and C++
Requires:       glibc = %{version}

%description -n glibc-utils
The glibc-utils package contains mtrace, a memory leak tracer and
xtrace, a function call tracer which can be helpful during program
debugging.

If you are unsure if you need this, do not install this package.

%package -n glibc-testsuite
Summary:        Testsuite results from the GNU C Library
License:        LGPL-2.1-or-later
Group:          Development/Languages/C and C++

%description -n glibc-testsuite
This package contains the testsuite results from the GNU C Library.

%if %{build_main}
%package info
Summary:        Info Files for the GNU C Library
License:        GFDL-1.1-only
Group:          Documentation/Other
Requires(post): %{install_info_prereq}
Requires(preun):%{install_info_prereq}
BuildArch:      noarch

%description info
This package contains the documentation for the GNU C library stored as
info files. Due to a lack of resources, this documentation is not
complete and is partially out of date.

%package html
Summary:        HTML Documentation for the GNU C Library
License:        GFDL-1.1-only
Group:          Documentation/HTML
BuildArch:      noarch

%description html
This package contains the HTML documentation for the GNU C library. Due
to a lack of resources, this documentation is not complete and is
partially out of date.

%package i18ndata
Summary:        Database Sources for 'locale'
License:        GPL-2.0-or-later AND MIT
Group:          System/Libraries
BuildArch:      noarch

%description i18ndata
This package contains the data needed to build the locale data files to
use the internationalization features of the GNU libc. It is normally
not necessary to install this packages, the data files are already
created.

%package locale-base
Summary:        en_US Locale Data for Localized Programs
License:        GPL-2.0-or-later AND MIT AND LGPL-2.1-or-later
Group:          System/Libraries
Requires(post): /bin/cat
Requires:       glibc = %{version}

%description locale-base
Locale data for the internationalisation features of the GNU C library.
This package contains only the U.S. English locale.

%package locale
Summary:        Locale Data for Localized Programs
License:        GPL-2.0-or-later AND MIT AND LGPL-2.1-or-later
Group:          System/Libraries
Requires:       glibc-locale-base = %{version}
# bug437293
%ifarch ppc64
Obsoletes:      glibc-locale-64bit
%endif
%ifarch ppc
Obsoletes:      glibc-locale-32bit
%endif

%description locale
Locale data for the internationalisation features of the GNU C library.

%package -n nscd
Summary:        Name Service Caching Daemon
License:        GPL-2.0-or-later
Group:          System/Daemons
Provides:       glibc:/usr/sbin/nscd
Requires:       glibc = %{version}
Obsoletes:      unscd <= 0.48
%{?sysusers_requires}
%{?systemd_requires}

%description -n nscd
Nscd caches name service lookups and can dramatically improve
performance with NIS, NIS+, and LDAP.

%package profile
Summary:        Libc Profiling and Debugging Versions
License:        GPL-2.0-or-later AND LGPL-2.1-or-later AND LGPL-2.1-or-later WITH GCC-exception-2.0
Group:          Development/Libraries/C and C++
Requires:       glibc = %{version}
# bug437293
%ifarch ppc64
Obsoletes:      glibc-profile-64bit
%endif
%ifarch ppc
Obsoletes:      glibc-profile-32bit
%endif

%description profile
This package contains special versions of the GNU C library which are
necessary for profiling and debugging.

%package devel
Summary:        Include Files and Libraries Mandatory for Development
License:        BSD-3-Clause AND LGPL-2.1-or-later AND LGPL-2.1-or-later WITH GCC-exception-2.0 AND GPL-2.0-or-later
Group:          Development/Libraries/C and C++
Obsoletes:      epoll = 1.0
Provides:       epoll < 1.0
# bug437293
%ifarch ppc64
Obsoletes:      glibc-devel-64bit
%endif
%ifarch ppc
Obsoletes:      glibc-devel-32bit
%endif
Requires:       glibc = %{version}
Requires:       libxcrypt-devel
Requires:       linux-kernel-headers

%description devel
These libraries are needed to develop programs which use the standard C
library.

%package devel-static
Summary:        C library static libraries for -static linking
License:        BSD-3-Clause AND LGPL-2.1-or-later AND LGPL-2.1-or-later WITH GCC-exception-2.0 AND GPL-2.0-or-later
Group:          Development/Libraries/C and C++
Requires:       %{name}-devel = %{version}
Requires:       libxcrypt-devel-static
# Provide Fedora name for package to make packaging easier
Provides:       %{name}-static = %{version}

%description devel-static
The glibc-devel-static package contains the C library static libraries
for -static linking.  You don't need these, unless you link statically,
which is highly discouraged.

%package extra
# makedb requires libselinux. We add this program in a separate
# package so that glibc does not require libselinux.
Summary:        Extra binaries from GNU C Library
License:        LGPL-2.1-or-later
Group:          Development/Libraries/C and C++
Requires:       glibc = %{version}

%description extra
The glibc-extra package contains some extra binaries for glibc that
are not essential but recommend for use.

makedb: A program to create a database for nss

%lang_package
%endif

%package -n cross-%{cross_arch}-glibc-devel
Summary:        Include Files and Libraries Mandatory for Development
License:        BSD-3-Clause AND LGPL-2.1-or-later AND LGPL-2.1-or-later WITH GCC-exception-2.0 AND GPL-2.0-or-later
Group:          Development/Libraries/C and C++
Requires:       cross-%{cross_arch}-linux-glibc-devel
BuildArch:      noarch
AutoReqProv:    off

%description -n cross-%{cross_arch}-glibc-devel
These libraries are needed to develop programs which use the standard C
library in a cross compilation setting.

%package -n libnsl1
Summary:        Legacy Network Support Library (NIS)
License:        LGPL-2.1-or-later
Group:          System/Libraries

%description -n libnsl1
Network Support Library for legacy architectures.  This library does not
have support for IPv6.

%if 0%{suse_version} >= 1500
%define make_output_sync -Oline
%endif

%prep
%autosetup -n glibc-%{version} -a 4 -p1

rm -f manpages/catchsegv.1

%build
# Disable LTO due to a usage of top-level assembler that
#  causes LTO issues (boo#1138807).
%define _lto_cflags %{nil}
if [ -x /bin/uname.bin ]; then
	/bin/uname.bin -a
else
	uname -a
fi
uptime || :
ulimit -a
nice
# We do not want configure to figure out the system its building one
# to support a common ground and thus set build and host ourself.
target="%{host_arch}-suse-linux"
case " %arm " in
  *" %{host_arch} "*) target="%{host_arch}-suse-linux-gnueabi" ;;
esac
%ifarch %arm
%define build %{_target_cpu}-suse-linux-gnueabi
%else
%define build %{_target_cpu}-suse-linux
%endif
# Don't use as-needed, it breaks glibc assumptions
# Before enabling it, run the testsuite and verify that it
# passes completely
export SUSE_ASNEEDED=0
# This is controlled by --enable-bind-now.
export SUSE_ZNOW=0
# Adjust glibc version.h
echo "#define CONFHOST \"${target}\"" >> version.h
echo "#define GITID \"%{git_id}\"" >> version.h
#
# Default CFLAGS and Compiler
#
enable_stack_protector=
BuildFlags=
tmp="%{optflags}"
for opt in $tmp; do
  case $opt in
    -fstack-protector-*) enable_stack_protector=${opt#-fstack-protector-} ;;
    -fstack-protector) enable_stack_protector=yes ;;
    -D_FORTIFY_SOURCE=*) enable_fortify_source=${opt#-D_FORTIFY_SOURCE=} ;;
    -ffortify=* | *_FORTIFY_SOURCE*) ;;
%if "%flavor" == "i686"
    *i586*) BuildFlags+=" ${opt/i586/i686}" ;;
%endif
%if %{build_cross}
    -m*) ;;  # remove all machine specific options for crosses
%endif
    *) BuildFlags+=" $opt" ;;
  esac
done
%if "%flavor" == "i686"
BuildFlags+=" -march=i686 -mtune=generic"
%else
%if !%{build_cross}
%ifarch i586
# workaround for unaligned stack in java-11-openjdk and mariadb
# by avoiding use of SSE
BuildFlags+=" -march=i586"
%endif
%endif
%endif
%if 0%{?with_gcc:1}
BuildCC="gcc-%{with_gcc}"
BuildCCplus="g++-%{with_gcc}"
%else
BuildCC="%__cc"
BuildCCplus="%__cxx"
%endif

#
#now overwrite for some architectures
#
%if %{build_cross}
BuildCC=%{cross_cpu}-suse-linux-gcc
BuildCCplus=%{cross_cpu}-suse-linux-g++
%else
%ifarch sparc64
	BuildFlags="-O2 -mcpu=ultrasparc -mvis -fcall-used-g6"
	BuildCC="gcc -m64"
	BuildCCplus="$BuildCCplus -m64"
%endif
%ifarch sparc
	BuildFlags="$BuildFlags -fcall-used-g6"
	BuildCC="gcc -m32"
	BuildCCplus="$BuildCCplus -m32"
%endif
%ifarch sparcv9
	BuildFlags="$BuildFlags -mcpu=ultrasparc -fcall-used-g6"
	BuildCC="gcc -m32"
	BuildCCplus="$BuildCCplus -m32"
%endif
%ifarch alphaev6
	BuildFlags="-mcpu=ev6"
%endif
%ifarch ppc ppc64
	BuildFlags="$(echo $BuildFlags | sed 's#-mminimal-toc##')"
	BuildFlags+=" -mtune=power5"
%endif
%ifarch ppc64
	BuildCC="$BuildCC -m64"
	BuildCCplus="$BuildCCplus -m64"
%endif
%ifarch hppa
	BuildFlags="$BuildFlags -mpa-risc-1-1 -fstrict-aliasing"
%endif
%if %{disable_assert}
	BuildFlags="$BuildFlags -DNDEBUG=1"
%endif
%ifarch mipsel
	# fails to build otherwise - need to recheck and fix
	%define enable_stackguard_randomization 0
%endif
%endif

#
# Build base glibc
#
mkdir cc-base
cd cc-base
%if %{build_profile}
profile="--enable-profile"
%else
profile="--disable-profile"
%endif

../configure \
	CFLAGS="$BuildFlags" BUILD_CFLAGS="$BuildFlags" \
	CC="$BuildCC" CXX="$BuildCCplus" \
	--prefix=%{_prefix} \
	--libexecdir=%{_libexecdir} --infodir=%{_infodir} \
        $profile \
	--build=%{build} --host=${target} \
%if %{build_cross}
	--with-headers=%{sysroot}/usr/include \
%else
%ifarch armv7hl ppc ppc64 ppc64le i686 x86_64 sparc sparc64 s390 s390x
	--enable-multi-arch \
%endif
%ifarch aarch64
	--enable-memory-tagging \
%endif
%ifarch mipsel
	--without-fp \
%endif
%ifarch ppc
	--with-cpu=power4 \
%endif
%ifarch ppc64p7
	--with-cpu=power7 \
%endif
%ifarch x86_64
%if %suse_version > 1500
	--enable-cet \
%endif
%endif
	--enable-systemtap \
%endif
%if %{enable_stackguard_randomization}
	--enable-stackguard-randomization \
%endif
	${enable_stack_protector:+--enable-stack-protector=$enable_stack_protector} \
%if !%{build_cross}
	${enable_fortify_source:+--enable-fortify-source=$enable_fortify_source} \
%endif
	--enable-tunables \
	--enable-kernel=%{enablekernel} \
	--with-bugurl=http://bugs.opensuse.org \
	--enable-bind-now \
	--disable-timezone-tools \
%if %{with livepatching}
	--enable-userspace-livepatch \
%endif
	--disable-crypt || \
  {
    rc=$?;
    echo "------- BEGIN config.log ------";
    %{__cat} config.log;
    echo "------- END config.log ------";
    exit $rc;
  }

%if %{build_main} && %{with livepatching}
# Append necessary flags for livepatch support, if enabled.  Only objects
# included in shared libraries should be prepared for live patching.
echo 'CFLAGS-.os += -fdump-ipa-clones' \
     >> Makeconfig
%endif

make %{?_smp_mflags} %{?make_output_sync}

%if %{build_main} && 0%{?sle_version}
%ifarch x86_64 i686 s390x ppc64le
# Horrible workaround for bsc#1221482
%ifarch x86_64 i686
archsub=x86
%endif
%ifarch s390x
archsub=s390
%endif
%ifarch ppc64le
archsub=powerpc
%endif
xstatbuild ()
{
  gcc -O2 -fpic -I ../sysdeps/unix/sysv/linux/$archsub -xc - -c -o $1stat$2.oS <<EOF
#include <bits/wordsize.h>
#include <xstatver.h>
int __$1xstat$2 (int, $3, void *);

int
__attribute__ ((visibility ("hidden")))
$1stat$2 ($3 file, void *buf)
{
  return __$1xstat$2 (_STAT_VER, file, buf);
}
EOF
  ar r libc_nonshared.a $1stat$2.oS
}
xstatbuild "" "" "const char *"
xstatbuild f "" int
xstatbuild l "" "const char *"
%ifarch i686
xstatbuild "" 64 "const char *"
xstatbuild f 64 int
xstatbuild l 64 "const char *"
%endif
%endif
%endif

cd ..

#
# Build html documentation
#
%if %{build_html}
make %{?_smp_mflags} %{?make_output_sync} -C cc-base html
%endif

# sysusers.d
%sysusers_generate_pre %{SOURCE22} nscd nscd.conf

%check
%if %{build_testsuite}
# The testsuite will fail if asneeded is used
export SUSE_ASNEEDED=0
# The testsuite will fail if -znow is used
export SUSE_ZNOW=0
# Increase timeout
export TIMEOUTFACTOR=16
# The testsuite does its own malloc checking
unset MALLOC_CHECK_
make %{?_smp_mflags} %{?make_output_sync} -C cc-base -k check || {
  cd cc-base
  o=$-
  set +x
  for sum in subdir-tests.sum */subdir-tests.sum; do
    while read s t; do
      case $s in
	XPASS:|PASS:)
	  echo ++++++ $s $t ++++++
	  ;;
	*) # X?FAIL:
	  echo ------ $s $t ------
	  test ! -f $t.out || cat $t.out
	  ;;
	esac
    done < $sum
  done
  set -$o
  # Fail build if there where compilation errors during testsuite run
  test -f tests.sum
}
%else
# This has to pass on all platforms!
# Exceptions:
# None!
make %{?_smp_mflags} %{?make_output_sync} -C cc-base check-abi
make %{?_smp_mflags} %{?make_output_sync} -C cc-base test t=elf/check-localplt
%endif

%define rtldlib %{_lib}
# Each architecture has a different name for the dynamic linker:
%ifarch %arm
%ifarch armv6hl armv7hl
%define rtld_name ld-linux-armhf.so.3
# Keep compatibility link
%define rtld_oldname ld-linux.so.3
%else
%define rtld_name ld-linux.so.3
%endif
%endif
%ifarch ia64
%define rtld_name ld-linux-ia64.so.2
%endif
%ifarch ppc s390 mips hppa m68k
%define rtld_name ld.so.1
%endif
%ifarch ppc64
%define rtld_name ld64.so.1
%endif
%ifarch ppc64le
%define rtld_name ld64.so.2
%endif
%ifarch s390x
%define rtldlib lib
%define rtld_name ld64.so.1
%endif
%ifarch x86_64
%define rtld_name ld-linux-x86-64.so.2
%endif
%ifarch %ix86 %sparc
%define rtld_name ld-linux.so.2
%endif
%ifarch aarch64
%define rtldlib lib
%define rtld_name ld-linux-aarch64.so.1
%endif
%ifarch riscv64
%define rtldlib lib
%define rtld_name ld-linux-riscv64-lp64d.so.1
%endif

%if %{with usrmerged}
%define rootsbindir %{_sbindir}
%define slibdir %{_libdir}
%define rtlddir %{_prefix}/%{rtldlib}
%else
%define rootsbindir /sbin
%define slibdir /%{_lib}
%define rtlddir /%{rtldlib}
%endif

%install
%if !%{build_testsuite}

%if %{with usrmerged}
mkdir -p %{buildroot}%{_libdir}
ln -s %{buildroot}%{_libdir} %{buildroot}/%{_lib}
%if "%{rtldlib}" != "%{_lib}"
mkdir -p %{buildroot}%{rtlddir}
ln -s %{buildroot}%{rtlddir} %{buildroot}/%{rtldlib}
%endif
mkdir -p %{buildroot}%{_sbindir}
ln -s %{buildroot}%{_sbindir} %{buildroot}/sbin
%endif

%if !%{build_cross}
%ifarch riscv64
mkdir -p %{buildroot}%{_libdir}
ln -s . %{buildroot}%{_libdir}/lp64d
%if "%{slibdir}" != "%{_libdir}"
mkdir -p %{buildroot}%{slibdir}
ln -s . %{buildroot}%{slibdir}/lp64d
%endif
%endif
%endif

%if %{build_main}

%if %{with livepatching}
%define tar_basename glibc-livepatch-%{version}-%{release}
%define tar_package_name %{tar_basename}.%{_arch}.tar.xz
%define clones_dest_dir %{tar_basename}/%{_arch}

# Ipa-clones are files generated by gcc which logs changes made across
# functions, and we need to know such changes to build livepatches
# correctly. These files are intended to be used by the livepatch
# developers and may be retrieved by using `osc getbinaries`.
#
# Create ipa-clones destination folder and move clones there.
mkdir -p ipa-clones/%{clones_dest_dir}
find . -name "*.ipa-clones" ! -empty \
       -exec cp -t ipa-clones/%{clones_dest_dir} --parents {} +

# Create tarball with ipa-clones.
tar -cJf %{tar_package_name} -C ipa-clones \
    --owner root --group root --sort name %{tar_basename}

# Copy tarball to the OTHER folder to store it as artifact.
cp %{tar_package_name} %{_topdir}/OTHER

%endif

# We don't want to strip the .symtab from our libraries in find-debuginfo.sh,
# certainly not from libc.so.* because it is used by libthread_db to find
# some non-exported symbols in order to detect if threading support
# should be enabled.  These symbols are _not_ exported, and we can't easily
# export them retroactively without changing the ABI.  So we have to
# continue to "export" them via .symtab, instead of .dynsym :-(
# But we also want to keep .symtab and .strtab of other libraries since some
# debugging tools currently require these sections directly inside the main
# files - specifically valgrind and PurifyPlus.
export STRIP_KEEP_SYMTAB=*.so*

# Install base glibc
make %{?_smp_mflags} %{?make_output_sync} install_root=%{buildroot} install -C cc-base

# Install locales
%if %{build_locales}
	cd cc-base
	# localedef creates hardlinks to other locales if possible
	# this will not work if we generate them in parallel.
	# thus we need to run fdupes on  /usr/lib/locale/
	# Still, on my system this is a speed advantage:
	# non-parallel build for install-locales: 9:34mins
	# parallel build with fdupes: 7:08mins
	make %{?_smp_mflags} %{?make_output_sync} install_root=%{buildroot} localedata/install-locales
	# Avoid hardlinks across subpackages
	mv %{buildroot}/usr/lib/locale/{en_US,C}.utf8 .
	%fdupes %{buildroot}/usr/lib/locale
	mv {en_US,C}.utf8 %{buildroot}/usr/lib/locale/
	cd ..
%endif

%ifnarch i686
# Create file list for glibc-lang package
%{find_lang} libc
%else
# The translations are shared with the base flavour
rm -rf %{buildroot}%{_datadir}/locale/*/
%endif

# Miscelanna:

install -m 644 %{SOURCE7} %{buildroot}/etc
%if %suse_version > 1500
install -D -m 644 %{SOURCE5} %{buildroot}%{_prefix}/etc/nsswitch.conf
%else
install -m 644 %{SOURCE5} %{buildroot}/etc/nsswitch.conf
%endif

%if %{build_html}
mkdir -p %{buildroot}%{_datadir}/doc/glibc
cp -p cc-base/manual/libc/*.html %{buildroot}%{_datadir}/doc/glibc
%endif

cd manpages; make install_root=%{buildroot} install; cd ..

# nscd tools:

%ifnarch i686
cp nscd/nscd.conf %{buildroot}/etc
mkdir -p %{buildroot}/etc/init.d
ln -sf %{rootsbindir}/service %{buildroot}%{_sbindir}/rcnscd
mkdir -p %{buildroot}/run/nscd
mkdir -p %{buildroot}/var/lib/nscd
%endif

#
# Create ld.so.conf
#
cat > %{buildroot}/etc/ld.so.conf <<EOF
%if "%{_lib}" != "lib"
/usr/local/%{_lib}
%endif
%ifarch ppc
/usr/local/lib64
%endif
/usr/local/lib
include /etc/ld.so.conf.d/*.conf
# /lib64, /lib, /usr/lib64 and /usr/lib gets added
# automatically by ldconfig after parsing this file.
# So, they do not need to be listed.
EOF
# Add ldconfig cache directory for directory ownership
mkdir -p %{buildroot}/var/cache/ldconfig
# Empty the ld.so.cache:
rm -f %{buildroot}/etc/ld.so.cache
touch %{buildroot}/etc/ld.so.cache

# Don't look at ldd! We don't wish a /bin/sh requires
chmod 644 %{buildroot}%{_bindir}/ldd

rm -f %{buildroot}%{rootsbindir}/sln

%ifnarch i686
mkdir -p %{buildroot}/usr/lib/tmpfiles.d/
install -m 644 %{SOURCE20} %{buildroot}/usr/lib/tmpfiles.d/
mkdir -p %{buildroot}/usr/lib/systemd/system
install -m 644 %{SOURCE21} %{buildroot}/usr/lib/systemd/system
mkdir -p %{buildroot}/usr/lib/sysusers.d/
install -m 644 %{SOURCE22} %{buildroot}/usr/lib/sysusers.d/nscd.conf
%endif

%if 0%{?rtld_oldname:1}
# Provide compatibility link
ln -s %{rtlddir}/%{rtld_name} %{buildroot}%{rtlddir}/%{rtld_oldname}
%endif

# Move getconf to %{_libexecdir}/getconf/ to avoid cross device link
mv %{buildroot}%{_bindir}/getconf %{buildroot}%{_libexecdir}/getconf/getconf
ln -s %{_libexecdir}/getconf/getconf %{buildroot}%{_bindir}/getconf

%if !%{build_utils}
# Remove unwanted files (packaged in glibc-utils)
rm -f %{buildroot}%{slibdir}/libmemusage*
rm -f %{buildroot}%{slibdir}/libpcprofile*
rm -f %{buildroot}%{_bindir}/memusage*
rm -f %{buildroot}%{_bindir}/mtrace
rm -f %{buildroot}%{_bindir}/pcprofiledump
rm -f %{buildroot}%{_bindir}/sotruss
rm -f %{buildroot}%{_bindir}/xtrace
rm -f %{buildroot}%{_bindir}/pldd
rm -rf %{buildroot}%{_libdir}/audit

%ifarch i686
# Remove files from glibc-{extra,info,i18ndata}, nscd
rm -rf %{buildroot}%{_infodir} %{buildroot}%{_prefix}/share/i18n
rm -f %{buildroot}%{_bindir}/makedb %{buildroot}/var/lib/misc/Makefile
rm -f %{buildroot}%{_sbindir}/nscd
%endif

%ifnarch i686
# /var/lib/misc is incompatible with transactional updates (bsc#1138726)
mkdir %{buildroot}%{_prefix}/share/misc
mv %{buildroot}/var/lib/misc/Makefile %{buildroot}%{_prefix}/share/misc/Makefile.makedb
ln -s %{_prefix}/share/misc/Makefile.makedb %{buildroot}/var/lib/misc/Makefile
%endif

%endif

# LSB
%ifarch %ix86
ln -sf %{rtlddir}/%{rtld_name} $RPM_BUILD_ROOT%{slibdir}/ld-lsb.so.3
%endif
%ifarch x86_64
ln -sf %{rtlddir}/%{rtld_name} $RPM_BUILD_ROOT%{slibdir}/ld-lsb-x86-64.so.3
%endif
%ifarch ppc
ln -sf %{rtlddir}/%{rtld_name} $RPM_BUILD_ROOT%{slibdir}/ld-lsb-ppc32.so.3
%endif
%ifarch ppc64
ln -sf %{rtlddir}/%{rtld_name} $RPM_BUILD_ROOT%{slibdir}/ld-lsb-ppc64.so.3
%endif
%ifarch s390
ln -sf %{rtlddir}/%{rtld_name} $RPM_BUILD_ROOT%{slibdir}/ld-lsb-s390.so.3
%endif
%ifarch s390x
ln -sf %{rtlddir}/%{rtld_name} $RPM_BUILD_ROOT%{slibdir}/ld-lsb-s390x.so.3
%endif

%else

%if %{build_utils}

make %{?_smp_mflags} install_root=%{buildroot} install -C cc-base \
  subdirs='malloc debug elf'
cd manpages; make install_root=%{buildroot} install; cd ..
# Remove unwanted files
rm -f %{buildroot}%{rtlddir}/ld*.so* %{buildroot}%{slibdir}/lib[!mp]*
%if "%{_libdir}" != "%{slibdir}"
rm -f %{buildroot}%{_libdir}/lib*
%else
rm -f %{buildroot}%{_libdir}/lib*.a
%endif
rm -f %{buildroot}%{_bindir}/{ld.so,ldd,sprof}
rm -rf %{buildroot}%{_mandir}/man*
rm -rf %{buildroot}%{rootsbindir} %{buildroot}%{_includedir}
%ifarch riscv64
rm %{buildroot}%{_libdir}/lp64d
%if "%{slibdir}" != "%{_libdir}"
rm %{buildroot}%{slibdir}/lp64d
%endif
%endif

%endif

%if %{build_cross}
# See above
export STRIP_KEEP_SYMTAB=*.so*
export NO_BRP_STRIP_DEBUG=true
make %{?_smp_mflags} install_root=%{buildroot}/%{sysroot} install -C cc-base
rm -rf %{buildroot}/%{sysroot}/%{_libdir}/audit
rm -rf %{buildroot}/%{sysroot}/%{_libdir}/gconv
rm -rf %{buildroot}/%{sysroot}/%{_infodir}
rm -rf %{buildroot}/%{sysroot}/%{_datadir}
rm -rf %{buildroot}/%{sysroot}/%{_libexecdir}
rm -rf %{buildroot}/%{sysroot}/%{_bindir}
rm -rf %{buildroot}/%{sysroot}/%{_sbindir}
rm -rf %{buildroot}/%{sysroot}/etc
rm -rf %{buildroot}/%{sysroot}/var

# Some programs look for <prefix>/lib/../$subdir where subdir is
# for instance "lib64".  For this path lookup to succeed we need the
# ../lib subdir, even if it's empty, so enforce its existence.
mkdir -p %{buildroot}/%{sysroot}/lib
mkdir -p %{buildroot}/%{sysroot}/%{_prefix}/lib

%endif

%endif

%if %{with usrmerged}

rm %{buildroot}/%{_lib}
%if "%{rtldlib}" != "%{_lib}"
rm %{buildroot}/%{rtldlib}
%endif
rm %{buildroot}/sbin

%endif

%endif

%if %{build_main}

%post -p <lua>
function exec(path, ...)
  local pid = posix.fork()
  if pid == 0 then
     posix.exec(path, ...)
     io.write(path, ": exec failed: ", posix.errno(), "\n")
     os.exit(1)
  end
  if not pid then
     error(path .. ": fork failed: " .. posix.errno() .. "\n")
  end
  posix.wait(pid)
end

-- First, get rid of platform-optimized libraries. We remove any we have
-- ever built, since otherwise we might end up using some old leftover
-- libraries when new ones aren't installed in their place anymore.
libraries = { "libc.so.6", "libc.so.6.1", "libm.so.6", "libm.so.6.1",
	      "librt.so.1", "libpthread.so.0", "libthread_db.so.1" }
remove_dirs = {
%ifarch i586
  "%{slibdir}/i686/",
%endif
%ifarch ppc ppc64
  "%{slibdir}/power4/", "%{slibdir}/ppc970/",
  "%{slibdir}/power5/", "%{slibdir}/power5+/",
  "%{slibdir}/power6/", "%{slibdir}/power6x/",
  "%{slibdir}/power7/",
  "%{slibdir}/ppc-cell-be/",
%endif
  "%{slibdir}/tls/"
}
for i, remove_dir in ipairs(remove_dirs) do
  for j, library in ipairs(libraries) do
    local file = remove_dir .. library
    -- This file could be a symlink to library-%{version}.so, so check
    -- this and don't remove only the link, but also the library itself.
    local link = posix.readlink(file)
    if link then
      if link:sub(1, 1) ~= "/" then link = remove_dir .. link end
      os.remove(link)
    end
    os.remove(file)
  end
end
if posix.access("%{rootsbindir}/ldconfig", "x") then
  exec("%{rootsbindir}/ldconfig", "-X")
end
if posix.utime("%{_libdir}/gconv/gconv-modules.cache") then
  exec("/usr/sbin/iconvconfig", "-o", "%{_libdir}/gconv/gconv-modules.cache",
       "--nostdlib", "%{_libdir}/gconv")
end

%postun -p %{rootsbindir}/ldconfig

%post locale-base
/usr/sbin/iconvconfig

%post info
%install_info --info-dir=%{_infodir} %{_infodir}/libc.info.gz

%preun info
%install_info_delete --info-dir=%{_infodir} %{_infodir}/libc.info.gz

%pre -n nscd -f nscd.pre
%service_add_pre nscd.service

%preun -n nscd
%service_del_preun nscd.service

%post -n nscd
%service_add_post nscd.service
%tmpfiles_create /usr/lib/tmpfiles.d/nscd.conf
# Previously we had nscd.socket, remove it
test -x /usr/bin/systemctl && /usr/bin/systemctl stop nscd.socket 2>/dev/null || :
test -x /usr/bin/systemctl && /usr/bin/systemctl disable nscd.socket 2>/dev/null  || :
# Hard removal in case the above did not work
rm -f /etc/systemd/system/sockets.target.wants/nscd.socket
exit 0

%postun -n nscd
%service_del_postun nscd.service
exit 0

%files
# glibc
%defattr(-,root,root)
%license LICENSES
%config(noreplace) /etc/bindresvport.blacklist
%config /etc/ld.so.conf
%attr(0644,root,root) %verify(not md5 size mtime) %ghost %config(missingok,noreplace) /etc/ld.so.cache
%config(noreplace) /etc/rpc
%if %suse_version > 1500
%attr(0644,root,root) %verify(not md5 size mtime) %ghost %config(missingok,noreplace) /etc/nsswitch.conf
%{_prefix}/etc/nsswitch.conf
%else
%verify(not md5 size mtime) %config(noreplace) /etc/nsswitch.conf
%endif
%attr(0644,root,root) %verify(not md5 size mtime) %ghost %config(missingok,noreplace) /etc/gai.conf
%doc posix/gai.conf
%doc %{_mandir}/man1/gencat.1.gz
%doc %{_mandir}/man1/getconf.1.gz
%doc %{_mandir}/man5/*

%{_bindir}/ld.so
%{rtlddir}/%{rtld_name}
%if 0%{?rtld_oldname:1}
%{rtlddir}/%{rtld_oldname}
%endif
%ifarch %ix86 x86_64 ppc ppc64 s390 s390x
# LSB
%{slibdir}/*-lsb*.so.3
%endif

%ifarch riscv64
%{_libdir}/lp64d
%if "%{slibdir}" != "%{_libdir}"
%{slibdir}/lp64d
%endif
%endif

%{slibdir}/libBrokenLocale.so.1
%{slibdir}/libanl.so.1
%{slibdir}/libc.so.6*
%{slibdir}/libc_malloc_debug.so.0
%{slibdir}/libdl.so.2*
%{slibdir}/libm.so.6*
%ifarch x86_64 aarch64
%{slibdir}/libmvec.so.1
%endif
%{slibdir}/libnss_compat.so.2
%{slibdir}/libnss_db.so.2
%{slibdir}/libnss_dns.so.2
%{slibdir}/libnss_files.so.2
%{slibdir}/libnss_hesiod.so.2
%{slibdir}/libpthread.so.0
%{slibdir}/libresolv.so.2
%{slibdir}/librt.so.1
%{slibdir}/libthread_db.so.1
%{slibdir}/libutil.so.1
%dir %attr(0700,root,root) /var/cache/ldconfig
%{rootsbindir}/ldconfig
%{_bindir}/gencat
%{_bindir}/getconf
%{_bindir}/getent
%{_bindir}/iconv
%attr(755,root,root) %{_bindir}/ldd
%{_bindir}/locale
%{_bindir}/localedef
%dir %attr(0755,root,root) %{_libexecdir}/getconf
%{_libexecdir}/getconf/*
%{_sbindir}/iconvconfig

%files locale-base
%defattr(-,root,root)
%{_datadir}/locale/locale.alias
%if %{build_locales}
%dir %{_prefix}/lib/locale
%{_prefix}/lib/locale/C.utf8
%{_prefix}/lib/locale/en_US.utf8
%endif
%dir %{_libdir}/gconv
%{_libdir}/gconv/*.so
%{_libdir}/gconv/gconv-modules
%{_libdir}/gconv/gconv-modules.d
%attr(0644,root,root) %verify(not md5 size mtime) %ghost %{_libdir}/gconv/gconv-modules.cache

%files locale
%defattr(-,root,root)
%if %{build_locales}
%{_prefix}/lib/locale
%exclude %{_prefix}/lib/locale/C.utf8
%exclude %{_prefix}/lib/locale/en_US.utf8
%endif

%files devel
%defattr(-,root,root)
%license COPYING COPYING.LIB
%doc NEWS README
%doc %{_mandir}/man3/*
%{_bindir}/sprof
%{_includedir}/*
%{_libdir}/*.o
%{_libdir}/libBrokenLocale.so
%{_libdir}/libanl.so
%{_libdir}/libc.so
%{_libdir}/libc_malloc_debug.so
%{_libdir}/libm.so
%ifarch x86_64 aarch64
%{_libdir}/libmvec.so
%endif
%{_libdir}/libnss_compat.so
%{_libdir}/libnss_db.so
%{_libdir}/libnss_hesiod.so
%{_libdir}/libresolv.so
%{_libdir}/libthread_db.so
# These static libraries are needed even for shared builds
%{_libdir}/libc_nonshared.a
%{_libdir}/libdl.a
%{_libdir}/libg.a
%ifarch ppc ppc64 ppc64le s390 s390x sparc sparcv8 sparcv9 sparcv9v
# This is not built on sparc64.
	%{_libdir}/libnldbl_nonshared.a
%endif
%{_libdir}/libmcheck.a
%{_libdir}/libpthread.a
%{_libdir}/librt.a
%{_libdir}/libutil.a

%files devel-static
%defattr(-,root,root)
%{_libdir}/libBrokenLocale.a
%{_libdir}/libanl.a
%{_libdir}/libc.a
%{_libdir}/libm.a
%ifarch x86_64 aarch64
%{_libdir}/libm-%{libversion}.a
%{_libdir}/libmvec.a
%endif
%{_libdir}/libresolv.a

%ifnarch i686
%files info
%defattr(-,root,root)
%doc %{_infodir}/libc.info.gz
%doc %{_infodir}/libc.info-?.gz
%doc %{_infodir}/libc.info-??.gz

%if %{build_html}
%files html
%defattr(-,root,root)
%doc %{_prefix}/share/doc/glibc
%endif

%files i18ndata
%defattr(-,root,root)
%{_prefix}/share/i18n

%files -n nscd
%defattr(-,root,root)
%config(noreplace) /etc/nscd.conf
%{_sbindir}/nscd
%{_sbindir}/rcnscd
/usr/lib/systemd/system/nscd.service
%dir /usr/lib/tmpfiles.d
/usr/lib/tmpfiles.d/nscd.conf
%dir /usr/lib/sysusers.d
/usr/lib/sysusers.d/nscd.conf
%dir %attr(0755,root,root) %ghost /run/nscd
%attr(0644,root,root) %verify(not md5 size mtime) %ghost %config(missingok,noreplace) /run/nscd/nscd.pid
%attr(0666,root,root) %verify(not md5 size mtime) %ghost %config(missingok,noreplace) /run/nscd/socket
%dir %attr(0755,root,root) /var/lib/nscd
%attr(0600,root,root) %verify(not md5 size mtime) %ghost %config(missingok,noreplace) /var/lib/nscd/passwd
%attr(0600,root,root) %verify(not md5 size mtime) %ghost %config(missingok,noreplace) /var/lib/nscd/group
%attr(0600,root,root) %verify(not md5 size mtime) %ghost %config(missingok,noreplace) /var/lib/nscd/hosts
%attr(0600,root,root) %verify(not md5 size mtime) %ghost %config(missingok,noreplace) /var/lib/nscd/services
%attr(0600,root,root) %verify(not md5 size mtime) %ghost %config(missingok,noreplace) /var/lib/nscd/netgroup
%endif

%if %{build_profile}
%files profile
%defattr(-,root,root)
%{_libdir}/libc_p.a
%{_libdir}/libBrokenLocale_p.a
%{_libdir}/libanl_p.a
%{_libdir}/libm_p.a
%ifarch x86_64 aarch64
%{_libdir}/libmvec_p.a
%endif
%{_libdir}/libpthread_p.a
%{_libdir}/libresolv_p.a
%{_libdir}/librt_p.a
%{_libdir}/libutil_p.a
%{_libdir}/libdl_p.a
%endif

%ifnarch i686
%files extra
%defattr(-,root,root)
%{_bindir}/makedb
%{_prefix}/share/misc/Makefile.makedb
/var/lib/misc/Makefile

%files lang -f libc.lang
%endif

%ifarch %ix86 %alpha hppa m68k %mips32 %mips64 %sparc ppc ppc64 ppc64le x86_64 s390 s390x %arm aarch64 riscv64
%files -n libnsl1
%{slibdir}/libnsl.so.1
%endif

%endif

%if %{build_cross}
%files -n cross-%{cross_arch}-glibc-devel
%defattr(-,root,root)
%license COPYING COPYING.LIB
%{sysroot}
%endif

%if %{build_utils}
%files -n glibc-utils
%defattr(-,root,root)
%{slibdir}/libmemusage.so
%{slibdir}/libpcprofile.so
%dir %{_libdir}/audit
%{_libdir}/audit/sotruss-lib.so
%{_bindir}/memusage
%{_bindir}/memusagestat
%{_bindir}/mtrace
%{_bindir}/pcprofiledump
%{_bindir}/sotruss
%{_bindir}/xtrace
%{_bindir}/pldd
%endif

%changelog
* Thu Jul 24 2025 schwab@suse.de
- regcomp-double-free.patch: posix: Fix double-free after allocation
  failure in regcomp (CVE-2025-8058, bsc#1246965, BZ #33185)
* Thu Jul 17 2025 schwab@suse.de
- nscd-gethst-race.patch: Reduce chance of crash when using nscd GETFDHST
  (bsc#1240058)
* Mon May 19 2025 schwab@suse.de
- static-setuid-ld-library-path.patch: elf: Ignore LD_LIBRARY_PATH and
  debug env var for setuid for static (CVE-2025-4802, bsc#1243317)
* Thu Apr 17 2025 schwab@suse.de
- Add support for userspace livepatching for ppc64le (jsc#PED-11850)
* Thu Apr  3 2025 schwab@suse.de
- pthread-wakeup.patch: pthreads NPTL: lost wakeup fix 2 (bsc#1234128, BZ
  [#25847])
* Mon Mar 24 2025 schwab@suse.de
- Mark functions in libc_nonshared.a as hidden (bsc#1239883)
* Wed Mar 19 2025 schwab@suse.de
- Bump minimal kernel version to 4.3 to enable use of direct socketcalls
  on x86-32 and s390x (bsc#1234713)
* Thu Jan 23 2025 schwab@suse.de
- assert-message-allocation.patch: Fix underallocation of abort_msg_s
  struct (CVE-2025-0395, bsc#1236282, BZ #32582))
* Thu Jan  9 2025 schwab@suse.de
- prctl-syscall-wrapper.patch: Linux: Switch back to assembly syscall
  wrapper for prctl (bsc#1234665, BZ #29770)
* Tue Jan  7 2025 schwab@suse.de
- Correctly determine livepatching support
* Mon Dec  2 2024 schwab@suse.de
- Remove nss-systemd from default nsswitch.conf (bsc#1233699)
* Wed Oct  9 2024 schwab@suse.de
- Apply libc_nonshared.a workaround also on s390x and ppc64le (bsc#1231051)
* Wed Sep 18 2024 schwab@suse.de
- Use nss-systemd by default also in SLE (bsc#1230638)
* Thu Jul 18 2024 schwab@suse.de
- s390x-wcsncmp.patch: s390x: Fix segfault in wcsncmp (bsc#1228042, BZ
  [#31934])
* Wed May 29 2024 schwab@suse.de
- Fix typo in last change
* Thu May 23 2024 schwab@suse.de
- Also include stat64 in the 32-bit libc_nonshared.a workaround
  (bsc#1221482)
* Wed May  8 2024 giuliano.belinassi@suse.com
- ulp-prologue-into-asm-functions.patch: Avoid creating ULP prologue
  for _start routine (bsc#1221940)
* Tue May  7 2024 schwab@suse.de
- nscd-netgroup-cache-timeout.patch: Use time_t for return type of
  addgetnetgrentX (CVE-2024-33602, bsc#1223425)
- Also add libc_nonshared.a workaround to 32-bit x86 compat package
  (bsc#1221482)
* Wed May  1 2024 giuliano.belinassi@suse.com
- glibc-CVE-2024-33599-nscd-Stack-based-buffer-overflow-in-n.patch:
  nscd: Stack-based buffer overflow in netgroup cache
  (CVE-2024-33599, bsc#1223423, BZ #31677)
- glibc-CVE-2024-33600-nscd-Avoid-null-pointer-crashes-after.patch:
  nscd: Avoid null pointer crashes after notfound response
  (CVE-2024-33600, bsc#1223424, BZ #31678)
- glibc-CVE-2024-33600-nscd-Do-not-send-missing-not-found-re.patch:
  nscd: Do not send missing not-found response in addgetnetgrentX
  (CVE-2024-33600, bsc#1223424, BZ #31678)
- glibc-CVE-2024-33601-CVE-2024-33602-nscd-netgroup-Use-two.patch:
  netgroup: Use two buffers in addgetnetgrentX (CVE-2024-33601,
  CVE-2024-33602, bsc#1223425, BZ #31680)
* Thu Apr 18 2024 schwab@suse.de
- iconv-iso-2022-cn-ext.patch: iconv: ISO-2022-CN-EXT: fix out-of-bound
  writes when writing escape sequence (CVE-2024-2961, bsc#1222992)
* Wed Apr  3 2024 schwab@suse.de
- Add workaround for invalid use of libc_nonshared.a with non-SUSE libc
  (bsc#1221482)
* Thu Mar 21 2024 schwab@suse.de
- malloc-arena-get2.patch: malloc: Use __get_nprocs on arena_get2 (BZ
  [#30945])
- sched-getcpu-rseq-area.patch: linux: Use rseq area unconditionally in
  sched_getcpu (BZ #31479)
* Mon Mar 11 2024 schwab@suse.de
- duplocale-global-locale.patch: duplocale: protect use of global locale
  (bsc#1220441, BZ #23970)
* Mon Feb 26 2024 schwab@suse.de
- s390-clone-error-clobber-r7.patch: S390: Do not clobber r7 in clone (BZ
  [#31402])
* Wed Feb  7 2024 schwab@suse.de
- Add libnsl1 to baselibs.conf (bsc#1219640)
* Wed Jan 31 2024 schwab@suse.de
- syslog-buffer-overflow.patch: syslog: Fix heap buffer overflow in
  __vsyslog_internal (CVE-2023-6246, CVE-2023-6779, CVE-2023-6780,
  bsc#1218863, bsc#1218867, bsc#1218868)
- qsort-invalid-cmp.patch: qsort: handle degenerated compare function
  (bsc#1218866)
* Mon Jan 29 2024 schwab@suse.de
- Change minimum GCC to 13
* Tue Jan 23 2024 schwab@suse.de
- Split off libnsl.so.1 into a separate package
* Tue Jan  2 2024 schwab@suse.de
- sem-open-o-creat.patch: sem_open: Clear O_CREAT when semaphore file is
  expected to exist (BZ #30789)
- ldconfig-process-elf-file.patch: elf: Fix wrong break removal from
  8ee878592c
- tls-modid-reuse.patch: elf: Fix TLS modid reuse generation assignment
  (BZ #29039)
- getaddrinfo-eai-memory.patch: getaddrinfo: translate ENOMEM to
  EAI_MEMORY (bsc#1217589, BZ #31163)
- libio-wdo-write.patch: libio: Check remaining buffer size in
  _IO_wdo_write (BZ #31183)
* Tue Dec  5 2023 schwab@suse.de
- aarch64-rawmemchr-unwind.patch: aarch64: correct CFI in rawmemchr
  (bsc#1217445, BZ #31113)
* Mon Dec  4 2023 schwab@suse.de
- Remove systemd from shadow and gshadow lookups (bsc#1217220)
* Mon Oct 23 2023 schwab@suse.de
- gb18030-2022.patch: add GB18030-2022 charmap (jsc#PED-4908, BZ #30243)
* Wed Oct 18 2023 schwab@suse.de
- dtors-reverse-ctor-order.patch: Remove, has been reverted
* Tue Oct 17 2023 schwab@suse.de
- Avoid use of SSE in i586 build
* Mon Oct  9 2023 schwab@suse.de
- Add systemd also to gshadow lookups (jsc#PED-5188)
- For SLE continue to use nsswitch.conf without systemd
* Wed Oct  4 2023 schwab@suse.de
- setxid-propagate-glibc-tunables.patch: Propagate GLIBC_TUNABLES in
  setxid binaries
- tunables-string-parsing.patch: tunables: Terminate if end of input is
  reached (CVE-2023-4911, bsc#1215501)
* Wed Sep 27 2023 schwab@suse.de
- fstat-implementation.patch: io: Do not implement fstat with fstatat
* Mon Sep 25 2023 schwab@suse.de
- getaddrinfo-memory-leak.patch: Fix leak in getaddrinfo introduced by the
  fix for CVE-2023-4806 (CVE-2023-5156, bsc#1215714, BZ #30884)
* Mon Sep 18 2023 schwab@suse.de
- getcanonname-use-after-free.patch: getaddrinfo: Fix use after free in
  getcanonname (CVE-2023-4806, bsc#1215281, BZ #30843)
- Do not build any cross packages in SLES
* Wed Sep 13 2023 schwab@suse.de
- no-aaaa-read-overflow.patch: Stack read overflow with large TCP
  responses in no-aaaa mode (CVE-2023-4527, bsc#1215280, BZ #30842)
* Tue Sep 12 2023 schwab@suse.de
- Add systemd to passwd, group and shadow lookups (jsc#PED-5188)
* Mon Sep 11 2023 schwab@suse.de
- ppc64-flock-fob64.patch: io: Fix record locking contants for powerpc64
  with __USE_FILE_OFFSET64 (BZ #30804)
- libio-io-vtables.patch: libio: Fix oversized __io_vtables
- call-init-proxy-objects.patch: elf: Do not run constructors for proxy
  objects
- dtors-reverse-ctor-order.patch: elf: Always call destructors in reverse
  constructor order (BZ #30785)
* Tue Sep  5 2023 schwab@suse.de
- intl-c-utf-8-like-c-locale.patch: intl: Treat C.UTF-8 locale like C
  locale (BZ #16621)
- glibc-disable-gettext-for-c-utf8.patch: Removed
* Mon Aug 28 2023 rguenther@suse.com
- Add cross-ppc64le package
* Tue Aug 22 2023 schwab@suse.de
- posix-memalign-fragmentation.patch: malloc: Enable merging of remainders
  in memalign, remove bin scanning from memalign (BZ #30723)
- Limit build counter sync to i686 flavor, to reduce needs for rebuilds
* Tue Aug 22 2023 rguenther@suse.com
- Add cross-s390x package (bsc#1214460)
* Mon Aug 14 2023 schwab@suse.de
- Require that elf/check-localplt does not fail
- glibc-2.3.90-langpackdir.diff: add hidden alias for __strcpy_chk
- cache-amd-legacy.patch: x86: Fix for cache computation on AMD legacy
  cpus
- cache-intel-shared.patch: x86: Fix incorrect scope of setting
  `shared_per_thread` (BZ# 30745)
* Wed Aug  2 2023 schwab@suse.de
- Update to glibc 2.38
  * When C2X features are enabled and the base argument is 0 or 2, the
    following functions support binary integers prefixed by 0b or 0B as
    input
  * PRIb*, PRIB* and SCNb* macros from C2X have been added to
    <inttypes.h>.
  * printf-family functions now support the wN format length modifiers for
    arguments of type intN_t, int_leastN_t, uintN_t or uint_leastN_t
    and the wfN format
    length modifiers for arguments of type int_fastN_t or uint_fastN_t, as
    specified in draft ISO C2X
  * A new tunable, glibc.pthread.stack_hugetlb, can be used to disable
    Transparent Huge Pages (THP) in stack allocation at pthread_create
  * Vector math library libmvec support has been added to AArch64
  * The strlcpy and strlcat functions have been added
  * CVE-2023-25139: When the printf family of functions is called with a
    format specifier that uses an <apostrophe> (enable grouping) and a
    minimum width specifier, the resulting output could be larger than
    reasonably expected by a caller that computed a tight bound on the
    buffer size
- Enable build with _FORTIFY_SOURCE
- glibc-2.3.90-langpackdir.diff: avoid reference to __strcpy_chk
- iconv-error-verbosity.patch: iconv: restore verbosity with unrecognized
  encoding names (BZ #30694)
- printf-grouping.patch, strftime-time64.patch,
  getlogin-no-loginuid.patch, fix-locking-in-_IO_cleanup.patch,
  gshadow-erange-rhandling.patch, system-sigchld-block.patch,
  gmon-buffer-alloc.patch, check-pf-cancel-handler.patch,
  powerpc64-fcntl-lock.patch, realloc-limit-chunk-reuse.patch,
  dl-find-object-return.patch; Removed
- bsc#1211828
- bsc#1212819
* Mon Jul 10 2023 schwab@suse.de
- gshadow-erange-rhandling.patch: gshadow: Matching sgetsgent, sgetsgent_r
  ERANGE handling (BZ #30151)
- system-sigchld-block.patch: posix: Fix system blocks SIGCHLD erroneously
  (BZ #30163)
- gmon-buffer-alloc.patch: gmon: Fix allocated buffer overflow
  (CVE-2023-0687, bsc#1207975, BZ #29444)
- check-pf-cancel-handler.patch: __check_pf: Add a cancellation cleanup
  handler (BZ #20975)
- powerpc64-fcntl-lock.patch: io: Fix F_GETLK, F_SETLK, and F_SETLKW for
  powerpc64
- realloc-limit-chunk-reuse.patch: realloc: Limit chunk reuse to only
  growing requests (BZ #30579)
- dl-find-object-return.patch: elf: _dl_find_object may return 1 during
  early startup (BZ #30515)
* Mon Jul  3 2023 schwab@suse.de
- Need to build with GCC 12 as minimum
* Thu Jun 29 2023 schwab@suse.de
- fix-locking-in-_IO_cleanup.patch: Update to final version
* Fri Apr 28 2023 giuliano.belinassi@suse.com
- ulp-prologue-into-asm-functions.patch: Add support for livepatches in
  ASM written functions (bsc#1210777, bsc#1211726)
* Thu Mar 30 2023 schwab@suse.de
- getlogin-no-loginuid.patch: getlogin_r: fix missing fallback if loginuid
  is unset (bsc#1209229, BZ #30235)
* Thu Mar 16 2023 schwab@suse.de
- Exclude static archives from preparation for live patching (bnc#1208721)
* Wed Feb  1 2023 schwab@suse.de
- Update to glibc 2.37
  * The getent tool now supports the --no-addrconfig option
  * The dynamic linker no longer loads shared objects from the "tls"
    subdirectories on the library search path or the subdirectory that
    corresponds to the AT_PLATFORM system name, or employs the legacy AT_HWCAP
    search mechanism, which was deprecated in version 2.33
- printf-grouping.patch: Account for grouping in printf width (BZ #30068)
- strftime-time64.patch: Use 64-bit time_t interfaces in strftime and
  strptime (BZ #30053)
- glibcextract-compile-c-snippet.patch, sys-mount-kernel-definition.patch,
  sys-mount-usage.patch, nscd-netlink-cache-invalidation.patch,
  syslog-large-messages.patch, dlmopen-libc-early-init.patch,
  ldd-vdso-dependency.patch, syslog-extra-whitespace.patch,
  errlist-edeadlock.patch, makeflags.patch, get-nscd-addresses.patch,
  x86-64-avx2-string-functions.patch, nscd-aicache.patch,
  dl-debug-bindings.patch, floatn.patch: Removed
- bsc#1207957
- bsc#1208358
- bsc#1212910
* Tue Dec 27 2022 lnussel@suse.com
- Remove reference to obsolete %%usrmerged macro (boo#1206798)
* Thu Dec 15 2022 schwab@suse.de
- floatn.patch: Update _FloatN header support for C++ in GCC 13
* Thu Nov 10 2022 gmbr3@opensuse.org
- nscd: Convert to systemd-sysusers
* Wed Oct 26 2022 schwab@suse.de
- dl-debug-bindings.patch: elf: Reinstate on DL_DEBUG_BINDINGS
  _dl_lookup_symbol_x (bsc#1204710)
* Wed Oct  5 2022 schwab@suse.de
- get-nscd-addresses.patch: get_nscd_addresses: Fix subscript typos (BZ
  [#29605])
- x86-64-avx2-string-functions.patch: check for required cpu features in
  AVX2 string functions (BZ #29611)
- nscd-aicache.patch: nscd: Drop local address tuple variable (BZ #29607)
* Wed Sep 21 2022 schwab@suse.de
- makeflags.patch: Makerules: fix MAKEFLAGS assignment for upcoming
  make-4.4 (BZ# 29564)
* Thu Sep  8 2022 schwab@suse.de
- errlist-edeadlock.patch: errlist: add missing entry for EDEADLOCK (BZ
  [#29545])
* Tue Sep  6 2022 schwab@suse.de
- syslog-large-messages.patch: syslog: Fix large messages (CVE-2022-39046,
  bsc#1203011, BZ #29536)
- dlmopen-libc-early-init.patch: elf: Call __libc_early_init for reused
  namespaces (BZ #29528)
- ldd-vdso-dependency.patch: elf: Restore how vDSO dependency is printed
  with LD_TRACE_LOADED_OBJECTS (BZ #29539)
- syslog-extra-whitespace.patch: syslog: Remove extra whitespace between
  timestamp and message (BZ #29544)
* Wed Aug 24 2022 fvogt@suse.com
- nscd-netlink-cache-invalidation.patch: nscd: Fix netlink cache
  invalidation if epoll is used (boo#1199964, BZ #29415)
* Mon Aug 15 2022 schwab@suse.de
- glibcextract-compile-c-snippet.patch: glibcextract.py: Add
  compile_c_snippet
- sys-mount-kernel-definition.patch: linux: Mimic kernel definition for
  BLOCK_SIZE
- sys-mount-usage.patch: linux: Fix sys/mount.h usage with kernel headers
* Tue Aug  2 2022 schwab@suse.de
- Update to glibc 2.36
  Major new features:
  * Support for DT_RELR relative relocation format has been added to
    glibc
  * On Linux, the pidfd_open, pidfd_getfd, and pidfd_send_signal functions
    have been added
  * On Linux, the process_madvise function has been added
  * On Linux, the process_mrelease function has been added
  * The “no-aaaa” DNS stub resolver option has been added
  * On Linux, the fsopen, fsmount, move_mount, fsconfig, fspick, open_tree,
    and mount_setattr have been added
  * localedef now accepts locale definition files encoded in UTF-8
  * Support for the mbrtoc8 and c8rtomb multibyte/UTF-8 character conversion
    functions has been added per the ISO C2X N2653 and C++20 P0482R6 proposals
  * The functions arc4random, arc4random_buf, and arc4random_uniform have been
    added
  Deprecated and removed features, and other changes affecting compatibility:
  * Support for prelink will be removed in the next release
  * The Linux kernel version check has been removed along with the
    LD_ASSUME_KERNEL environment variable
  * On Linux, The LD_LIBRARY_VERSION environment variable has been removed
- get-nprocs-sched-uninit-read.patch, get-nprocs-inaccurate.patch,
  strcmp-rtm-fallback.path, pt-load-invalid-hole.patch,
  localedef-ld-monetary.patch, nptl-spurious-eintr.patch,
  strncpy-power9-vsx.patch, nptl-cleanup-async-restore.patch,
  read-chk-cancel.patch, wcrtomb-fortify.patch,
  nptl-cleanup-async-restore-2.patch: Removed
- CVE-2023-4813, bsc#1215286
- bsc#1198751
- bsc#1200334
* Thu Jul 14 2022 schwab@suse.de
- nptl-cleanup-async-restore-2.patch: nptl: Fix
  ___pthread_unregister_cancel_restore asynchronous restore (bsc#1200093,
  BZ #29214)
* Thu Jun 23 2022 schwab@suse.de
- read-chk-cancel.patch: debug: make __read_chk a cancellation point
  (bsc#1200682, BZ #29274)
- wcrtomb-fortify.patch: wcrtomb: Make behavior POSIX compliant
  (bsc#1200688)
* Thu Jun  9 2022 schwab@suse.de
- Set SUSE_ZNOW=0
* Wed Jun  8 2022 schwab@suse.de
- strncpy-power9-vsx.patch: powerpc: Fix VSX register number on
  __strncpy_power9 (BZ #29197)
- nptl-cleanup-async-restore.patch: nptl: Fix __libc_cleanup_pop_restore
  asynchronous restore (bsc#1200093, BZ #29214)
* Tue May 17 2022 schwab@suse.de
- nptl-spurious-eintr.patch: nptl: Handle spurious EINTR when thread
  cancellation is disabled (BZ #29029)
* Thu May 12 2022 dimstar@opensuse.org
- Follow the distro default gcc version to build the cross
  bootstrap packages.
* Fri Apr 29 2022 meissner@suse.com
- switched to https urls
* Mon Feb 28 2022 schwab@suse.de
- get-nprocs-sched-uninit-read.patch: linux: __get_nprocs_sched: do not
  feed CPU_COUNT_S with garbage (BZ #28850)
- get-nprocs-inaccurate.patch: linux: fix accuracy of get_nprocs and
  get_nprocs_conf (BZ #28865)
- strcmp-rtm-fallback.path: x86: Fallback {str|wcs}cmp RTM in the ncmp
  overflow case (BZ #28896)
- pt-load-invalid-hole.patch: elf: Check invalid hole in PT_LOAD segments
  (BZ #28838)
- localedef-ld-monetary.patch: localedef: Update LC_MONETARY handling (BZ
  [#28845])
* Thu Feb  3 2022 schwab@suse.de
- Update to glibc 2.35
  Major new features:
  * Unicode 14.0.0 Support
  * Bump r_version in the debugger interface to 2
  * Support for the C.UTF-8 locale has been added to glibc
  * <math.h> functions that round their results to a narrower type, and
    corresponding <tgmath.h> macros, are added from TS 18661-1:2014, TS
    18661-3:2015 and draft ISO C2X
  * <math.h> functions for floating-point maximum and minimum,
    corresponding to new operations in IEEE 754-2019, and corresponding
    <tgmath.h> macros, are added from draft ISO C2X
  * <math.h> macros for single-precision float constants are added as a
    GNU extension
  * The __STDC_IEC_60559_BFP__ and __STDC_IEC_60559_COMPLEX__ macros are
    predefined as specified in TS 18661-1:2014
  * The exp10 functions in <math.h> now have a corresponding type-generic
    macro in <tgmath.h>
  * The ISO C2X macro _PRINTF_NAN_LEN_MAX has been added to <stdio.h>
  * printf-family functions now support the %%b format for output of
    integers in binary, as specified in draft ISO C2X, and the %%B variant
    of that format recommended by draft ISO C2X
  * A new DSO sorting algorithm has been added in the dynamic linker that uses
    topological sorting by depth-first search (DFS), solving performance issues
    of the existing sorting algorithm when encountering particular circular
    object dependency cases
  * A new tunable, glibc.rtld.dynamic_sort, can be used to select between
    the two DSO sorting algorithms
  * ABI support for a new function '__memcmpeq'. '__memcmpeq' is meant
    to be used by compilers for optimizing usage of 'memcmp' when its
    return value is only used for its boolean status
  * Support for automatically registering threads with the Linux rseq
    system call has been added
  * A symbolic link to the dynamic linker is now installed under
    /usr/bin/ld.so (or more precisely, '${bindir}/ld.so')
  * All programs and the testsuite in glibc are now built as position independent
    executables (PIE) by default on toolchains and architectures that support it
  * On Linux, a new tunable, glibc.malloc.hugetlb, can be used to
    either make malloc issue madvise plus MADV_HUGEPAGE on mmap and sbrk
    or to use huge pages directly with mmap calls with the MAP_HUGETLB
    flags)
  * The printf family of functions now handles the flagged %%#m conversion
    specifier, printing errno as an error constant (similar to strerrorname_np)
  * The function _dl_find_object has been added
  * On Linux, the epoll_pwait2 function has been added
  * The function posix_spawn_file_actions_addtcsetpgrp_np has been added,
    enabling posix_spawn and posix_spawnp to set the controlling terminal in
    the new process in a race free manner
  * Source fortification (_FORTIFY_SOURCE) level 3 is now available for
    applications compiling with glibc and gcc 12 and later
  Deprecated and removed features, and other changes affecting compatibility:
  * On x86-64, the LD_PREFER_MAP_32BIT_EXEC environment variable support
    has been removed since the first PT_LOAD segment is no longer executable
    due to defaulting to -z separate-code
  * The r_version update in the debugger interface makes the glibc binary
    incompatible with GDB
  * Intel MPX support (lazy PLT, ld.so profile, and LD_AUDIT) has been removed
  * The catchsegv script and associated libSegFault.so shared object have
    been removed
  * Support for prelink will be removed in the next release; this includes
    removal of the LD_TRACE_PRELINKING, and LD_USE_LOAD_BIAS, environment
    variables and their functionality in the dynamic loader
  Changes to build and runtime requirements:
  * The audit module interface version LAV_CURRENT is increased to enable
    proper bind-now support
  * The audit interface on aarch64 is extended to support both the indirect
    result location register (x8) and NEON Q register
  Security related changes:
  * CVE-2022-23219: Passing an overlong file name to the clnt_create
    legacy function could result in a stack-based buffer overflow when
    using the "unix" protocol
  * CVE-2022-23218: Passing an overlong file name to the svcunix_create
    legacy function could result in a stack-based buffer overflow
  * CVE-2021-3998: Passing a path longer than PATH_MAX to the realpath
    function could result in a memory leak and potential access of
    uninitialized memory
  * CVE-2021-3999: Passing a buffer of size exactly 1 byte to the getcwd
    function may result in an off-by-one buffer underflow and overflow
    when the current working directory is longer than PATH_MAX and also
    corresponds to the / directory through an unprivileged mount
    namespace
- copy-and-spawn-sgid-double-close.patch,
  fcntl-time-bits-64-redirect.patch, gaiconf-init-double-free.patch,
  gconv-parseconfdir-memory-leak.patch, getcwd-attribute-access.patch,
  glibc-c-utf8-locale.patch, iconv-charmap-close-output.patch,
  ld-show-auxv-colon.patch, ldconfig-leak-empty-paths.patch,
  librt-null-pointer.patch, pthread-kill-fail-after-exit.patch,
  pthread-kill-race-thread-exit.patch, pthread-kill-return-esrch.patch,
  pthread-kill-send-specific-thread.patch,
  pthread-mutexattr-getrobust-np-type.patch,
  setxid-deadlock-blocked-signals.patch,
  sysconf-nprocessors-affinity.patch, x86-string-control-test.patch:
  Removed.
- bsc#1194640
- bsc#1194768
- bsc#1194770
- bsc#1197718
- bsc#1211829
- bsc#1215891
* Fri Nov 26 2021 matz@suse.com
- Enable building the cross packages in rings.
* Tue Nov 16 2021 giuliano.belinassi@suse.com
- Add ExtraBuildFlags for build flags that cannot be passed to configure.
- Add support for livepatches (jsc#SLE-20049).
- Generate ipa-clones tarball artifact when livepatching is enabled.
* Wed Nov 10 2021 schwab@suse.de
- glibc.rpmlintrc: Update for rpmlint2
* Tue Oct  5 2021 schwab@suse.de
- ld-show-auxv-colon.patch: elf: Fix missing colon in LD_SHOW_AUXV output
  (BZ #282539
- x86-string-control-test.patch: x86-64: Use testl to check
  __x86_string_control
- pthread-kill-fail-after-exit.patch: nptl: pthread_kill, pthread_cancel
  should not fail after exit (BZ #19193)
- pthread-kill-race-thread-exit.patch: nptl: Fix race between pthread_kill
  and thread exit (BZ #12889)
- getcwd-attribute-access.patch: posix: Fix attribute access mode on
  getcwd (BZ #27476)
- pthread-kill-return-esrch.patch: nptl: pthread_kill needs to return
  ESRCH for old programs (BZ #19193)
- pthread-mutexattr-getrobust-np-type.patch: nptl: Fix type of
  pthread_mutexattr_getrobust_np, pthread_mutexattr_setrobust_np (BZ
  [#28036])
- setxid-deadlock-blocked-signals.patch: nptl: Avoid setxid deadlock with
  blocked signals in thread exit (BZ #28361)
- pthread-kill-send-specific-thread.patch: nptl: pthread_kill must send
  signals to a specific thread (BZ #28407)
- sysconf-nprocessors-affinity.patch: linux: Revert the use of
  sched_getaffinity on get_nproc (BZ #28310)
- iconv-charmap-close-output.patch: renamed from
  icon-charmap-close-output.patch
* Wed Aug 11 2021 schwab@suse.de
- Don't create separate debuginfo packages for cross packages
* Wed Aug 11 2021 schwab@suse.de
- ldconfig-leak-empty-paths.patch: ldconfig: avoid leak on empty paths in
  config file
- gconv-parseconfdir-memory-leak.patch: gconv_parseconfdir: Fix memory leak
- gaiconf-init-double-free.patch: gaiconf_init: Avoid double-free in label
  and precedence lists
- copy-and-spawn-sgid-double-close.patch: copy_and_spawn_sgid: Avoid
  double calls to close()
- icon-charmap-close-output.patch: iconv_charmap: Close output file when
  done
- fcntl-time-bits-64-redirect.patch: Linux: Fix fcntl, ioctl, prctl
  redirects for _TIME_BITS=64 (BZ #28182)
- librt-null-pointer.patch: librt: fix NULL pointer dereference (BZ
  [#28213])
* Tue Aug 10 2021 matz@suse.com
- Add cross development packages for aarch64 and riscv64.
* Mon Aug  2 2021 schwab@suse.de
- Update to glibc 2.34
  Major new features:
  * When _DYNAMIC_STACK_SIZE_SOURCE or _GNU_SOURCE are defined,
    PTHREAD_STACK_MIN is no longer constant and is redefined to
    sysconf(_SC_THREAD_STACK_MIN)
  * Add _SC_MINSIGSTKSZ and _SC_SIGSTKSZ
  * The dynamic linker implements the --list-diagnostics option, printing
    a dump of information related to IFUNC resolver operation and
    glibc-hwcaps subdirectory selection
  * On Linux, the function execveat has been added
  * The ISO C2X function timespec_getres has been added
  * The feature test macro __STDC_WANT_IEC_60559_EXT__, from draft ISO
    C2X, is supported to enable declarations of functions defined in Annex F
    of C2X
  * Add support for 64-bit time_t on configurations like x86 where time_t
    is traditionally 32-bit
  * The main gconv-modules file in glibc now contains only a small set of
    essential converter modules and the rest have been moved into a supplementary
    configuration file gconv-modules-extra.conf in the gconv-modules.d directory
    in the same GCONV_PATH
  * On Linux, a new tunable, glibc.pthread.stack_cache_size, can be used
    to configure the size of the thread stack cache
  * The function _Fork has been added as an async-signal-safe fork replacement
    since Austin Group issue 62 droped the async-signal-safe requirement for
    fork (and it will be included in the future POSIX standard)
  * On Linux, the close_range function has been added
  * The function closefrom has been added
  * The posix_spawn_file_actions_closefrom_np function has been added, enabling
    posix_spawn and posix_spawnp to close all file descriptors great than or
    equal to a giver integer
  Deprecated and removed features, and other changes affecting compatibility:
  * The function pthread_mutex_consistent_np has been deprecated
  * The function pthread_mutexattr_getrobust_np has been deprecated
  * The function pthread_mutexattr_setrobust_np has been deprecated
  * The function pthread_yield has been deprecated
  * The function inet_neta declared in <arpa/inet.h> has been deprecated
  * Various rarely-used functions declared in <resolv.h> and
    <arpa/nameser.h> have been deprecated
  * The pthread cancellation handler is now installed with SA_RESTART and
    pthread_cancel will always send the internal SIGCANCEL on a cancellation
    request
  * The symbols mallwatch and tr_break are now deprecated and no longer used in
    mtrace
  * The __morecore and __after_morecore_hook malloc hooks and the default
    implementation __default_morecore have been removed from the API
  * Debugging features in malloc such as the MALLOC_CHECK_ environment variable
    (or the glibc.malloc.check tunable), mtrace() and mcheck() have now been
    disabled by default in the main C library
  * The deprecated functions malloc_get_state and malloc_set_state have been
    moved from the core C library into libc_malloc_debug.so
  * The deprecated memory allocation hooks __malloc_hook, __realloc_hook,
    __memalign_hook and __free_hook are now removed from the API
  Changes to build and runtime requirements:
  * On Linux, the shm_open, sem_open, and related functions now expect the
    file shared memory file system to be mounted at /dev/shm
  Security related changes:
    CVE-2021-27645: The nameserver caching daemon (nscd), when processing
    a request for netgroup lookup, may crash due to a double-free,
    potentially resulting in degraded service or Denial of Service on the
    local system
    CVE-2021-33574: The mq_notify function has a potential use-after-free
    issue when using a notification type of SIGEV_THREAD and a thread
    attribute with a non-default affinity mask
    CVE-2021-35942: The wordexp function may overflow the positional
    parameter number when processing the expansion resulting in a crash
- nss-database-check-reload.patch, nss-load-chroot.patch,
  x86-isa-level.patch, nscd-netgroupcache.patch,
  nss-database-lookup.patch, select-modify-timeout.patch,
  nptl-db-libpthread-load-order.patch, rawmemchr-warning.patch,
  tst-cpu-features-amx.patch, mq-notify-use-after-free.patch: Removed
- bsc#1181403
- bsc#1184035
- bsc#1187911
- jsc#PED-987
* Fri Jun 11 2021 lnussel@suse.de
- Enable usrmerge in Factory always as it's default there
- Add conflict with pre-usrmerge filesystem package
* Thu Jun 10 2021 schwab@suse.de
- mq-notify-use-after-free.patch: Use __pthread_attr_copy in mq_notify
  (CVE-2021-33574, bsc#1186489, BZ #27896)
- Drop glibc-usrmerge-bootstrap-helper package
* Thu May 27 2021 schwab@suse.de
- tst-cpu-features-amx.patch: x86: tst-cpu-features-supports.c: Update AMX
  check
* Wed May 26 2021 schwab@suse.de
- rawmemchr-warning.patch: string: Work around GCC PR 98512 in rawmemchr
* Tue May  4 2021 schwab@suse.de
- nptl-db-libpthread-load-order.patch: nptl_db: Support different
  libpthread/ld.so load orders (bsc#1184214, BZ #27744)
* Tue Apr 13 2021 schwab@suse.de
- Enable support for static PIE (bsc#1184646)
- select-modify-timeout.patch: linux: always update select timeout
  (bsc#1184339, BZ #27706)
* Tue Mar 23 2021 schwab@suse.de
- Don't remove -f[asynchronous-]unwind-tables during configure run, no
  longer needed
* Mon Mar  8 2021 schwab@suse.de
- nss-database-check-reload.patch: nsswitch: return result when nss
  database is locked (BZ #27343)
- nss-load-chroot.patch: nss: Re-enable NSS module loading after chroot
  (bsc#1182323, BZ #27389)
- x86-isa-level.patch: x86: Set minimum x86-64 level marker (bsc#1182522,
  BZ #27318)
- nss-database-lookup.patch: nss: fix nss_database_lookup2's alternate
  handling (bsc#1182247, BZ #27416)
- nss-revert-api.patch: remove
- nscd-netgroupcache.patch: nscd: Fix double free in netgroupcache
  (CVE-2021-27645, bsc#1182733, BZ #27462)
* Tue Mar  2 2021 schwab@suse.de
- Disable x86 ISA level for now (bsc#1182522, BZ #27318)
- nss-revert-api.patch: Workaround for nss-compat brokeness (bsc#1182247,
  BZ #27416)
* Mon Mar  1 2021 schwab@suse.de
- Fix build of utils flavor for usrmerge
* Thu Feb 18 2021 schwab@suse.de
- Prepare for usrmerge (bsc#1029961)
* Tue Feb 16 2021 guillaume.gardet@opensuse.org
- Add --enable-memory-tagging for aarch64
* Tue Feb  2 2021 schwab@suse.de
- Update to glibc 2.33
  * The dynamic linker accepts the --list-tunables argument which prints
    all the supported tunables.
  * The dynamic linker accepts the --argv0 argument and provides opportunity
    to change argv[0] string.
  * The dynamic linker loads optimized implementations of shared objects
    from subdirectories under the glibc-hwcaps directory on the library
    search path if the system's capabilities meet the requirements for
    that subdirectory.
  * The new --help option of the dynamic linker provides usage and
    information and library search path diagnostics.
  * The mallinfo2 function is added to report statistics as per mallinfo,
    but with larger field widths to accurately report values that are
    larger than fit in an integer.
  * Add <sys/platform/x86.h> to provide query macros for x86 CPU features.
  * A new fortification level _FORTIFY_SOURCE=3 is available.
  * The mallinfo function is marked deprecated.
  * When dlopen is used in statically linked programs, alternative library
    implementations from HWCAP subdirectories are no longer loaded.
  * The deprecated <sys/vtimes.h> header and the function vtimes have been
    removed.
  * On s390(x), the type float_t is now derived from the macro
    __FLT_EVAL_METHOD__ that is defined by the compiler, instead of being
    hardcoded to double.
  * A future version of glibc will stop loading shared objects from the
    "tls" subdirectories on the library search path, the subdirectory that
    corresponds to the AT_PLATFORM system name, and also stop employing
    the legacy AT_HWCAP search mechanism.
  * CVE-2021-3326: An assertion failure during conversion from the
    ISO-20220-JP-3 character set using the iconv function has been fixed.
- Remove obsolete, unused /etc/default/nss
- aarch64-static-pie.patch, euc-kr-overrun.patch,
  get-nprocs-cpu-online-parsing.patch, iconv-redundant-shift.patch,
  iconv-ucs4-loop-bounds.patch, ifunc-fma4.patch,
  intl-codeset-suffixes.patch, nscd-gc-cycle.patch,
  printf-long-double-non-normal.patch, strerrorname-np.patch,
  syslog-locking.patch, sysvipc.patch: Removed
- bsc#1180557
- bsc#1181505
- bsc#1191592
- bsc#1201942
* Tue Jan 19 2021 schwab@suse.de
- Remove support for %%optimize_power
- Move to power4 baseline on ppc
* Tue Dec 15 2020 schwab@suse.de
- aarch64-static-pie.patch: fix static PIE start code for BTI
  (bsc#1179450, BZ #27068)
- iconv-redundant-shift.patch: iconv: Accept redundant shift sequences in
  IBM1364 (CVE-2020-27618, bsc#1178386, BZ #26224)
- iconv-ucs4-loop-bounds.patch: iconv: Fix incorrect UCS4 inner loop
  bounds (CVE-2020-29562, bsc#1179694, BZ #26923)
- printf-long-double-non-normal.patch: x86: Harden printf against
  non-normal long double values (CVE-2020-29573, bsc#1179721, BZ #26649)
- get-nprocs-cpu-online-parsing.patch: Fix parsing of
  /sys/devices/system/cpu/online (bsc#1180038, BZ #25859)
* Tue Nov 10 2020 schwab@suse.de
- intl-codeset-suffixes.patch: intl: Handle translation output codesets
  with suffixes (BZ #26383)
- strerrorname-np.patch: string: Fix strerrorname_np return value (BZ
  [#26555])
- sysvipc.patch: sysvipc: Fix SEM_STAT_ANY kernel argument pass (BZ
  [#26637], BZ #26639, BZ #26636)
* Mon Oct 26 2020 rguenther@suse.com
- Use --enable-cet on x86_64 to instrument glibc for indirect branch
  tracking and shadow stack use.  Enable indirect branch tracking
  and shadow stack in the dynamic loader (jsc#PM-2110, bsc#1175154)
* Tue Sep  8 2020 schwab@suse.de
- Keep nsswitch.conf in /etc for SLES15
- syslog-locking.patch: Correct locking and cancellation cleanup in syslog
  functions (bsc#1172085, BZ #26100)
- ifunc-fma4.patch: x86-64: Fix FMA4 detection in ifunc (BZ #26534)
* Thu Aug  6 2020 schwab@suse.de
- Update to glibc 2.32
  * Unicode 13.0.0 Support
  * New locale added: ckb_IQ
  * The GNU C Library now loads audit modules listed in the DT_AUDIT and
    DT_DEPAUDIT dynamic section entries of the main executable
  * powerpc64le supports IEEE128 long double libm/libc redirects when
    using the -mabi=ieeelongdouble to compile C code on supported GCC
    toolchains
  * To help detect buffer overflows and other out-of-bounds accesses
    several APIs have been annotated with GCC 'access' attribute
  * On Linux, functions the pthread_attr_setsigmask_np and
    pthread_attr_getsigmask_np have been added
  * The GNU C Library now provides the header file <sys/single_threaded.h>
    which declares the variable __libc_single_threaded
  * The functions sigabbrev_np and sigdescr_np have been added
  * The functions strerrorname_np and strerrordesc_np have been added
  * AArch64 now supports standard branch protection security hardening
    in glibc when it is built with a GCC that is configured with
  - -enable-standard-branch-protection (or if -mbranch-protection=standard
    flag is passed when building both GCC target libraries and glibc,
    in either case a custom GCC is needed)
  * The deprecated <sys/sysctl.h> header and the sysctl function have been
    removed
  * The sstk function is no longer available to newly linked binaries
  * The legacy signal handling functions siginterrupt, sigpause, sighold,
    sigrelse, sigignore and sigset, and the sigmask macro have been
    deprecated
  * ldconfig now defaults to the new format for ld.so.cache
  * The deprecated arrays sys_siglist, _sys_siglist, and sys_sigabbrev
    are no longer available to newly linked binaries, and their declarations
    have been removed from <string.h>
  * The deprecated symbols sys_errlist, _sys_errlist, sys_nerr, and _sys_nerr
    are no longer available to newly linked binaries, and their declarations
    have been removed from from <stdio.h>
  * Both strerror and strerror_l now share the same internal buffer in the
    calling thread, meaning that the returned string pointer may be invalided
    or contents might be overwritten on subsequent calls in the same thread or
    if the thread is terminated
  * Using weak references to libpthread functions such as pthread_create
    or pthread_key_create to detect the singled-threaded nature of a
    program is an obsolescent feature
  * The "files" NSS module no longer supports the "key" database (used for
    secure RPC)
  * The __morecore and __after_morecore_hook malloc hooks and the default
    implementation __default_morecore have been deprecated
  * The hesiod NSS module has been deprecated and will be removed in a
    future version of glibc
  * CVE-2016-10228: An infinite loop has been fixed in the iconv program when
    invoked with the -c option and when processing invalid multi-byte input
    sequences
  * CVE-2020-10029: Trigonometric functions on x86 targets suffered from stack
    corruption when they were passed a pseudo-zero argument
  * CVE-2020-1752: A use-after-free vulnerability in the glob function when
    expanding ~user has been fixed.
  * CVE-2020-6096: A signed comparison vulnerability in the ARMv7 memcpy and
    memmove functions has been fixed
- riscv-syscall-clobber.patch, ldbl-96-rem-pio2l.patch,
  long-double-alias.patch: Removed
- bsc#1027496
- bsc#1162930
- bsc#1166106
- bsc#1167631
- bsc#1167939
- bsc#1194785, jsc#SLE-18195
- bsc#1200855
- bsc#1201560
- bsc#1201640
- bsc#1207571
- jsc#SLE-13520
* Tue Jun 23 2020 schwab@suse.de
- long-double-alias.patch: Fix build with GCC 10 when long double = double
- nscd-gc-cycle.patch: nscd: bump GC cycle during cache pruning
  (bsc#1171878, BZ #26130)
* Mon May 18 2020 schwab@suse.de
- glibc-nsswitch-usr.diff: read /usr/etc/nsswitch.conf if
  /etc/nsswitch.conf does not exist
- Install default nsswitch.conf in /usr/etc
- Don't install gai.conf in /etc
* Mon Apr 20 2020 schwab@suse.de
- Split off %%lang_package
* Thu Mar 12 2020 schwab@suse.de
- riscv-syscall-clobber.patch: riscv: Avoid clobbering register parameters
  in syscall
- ldbl-96-rem-pio2l.patch: Avoid ldbl-96 stack corruption from range
  reduction of pseudo-zero (CVE-2020-10029, bsc#1165784, BZ #25487)
* Tue Feb 18 2020 schwab@suse.de
- nsswitch.conf: comment out initgroups setting, so that it defaults to
  the group setting (bsc#1164075)
* Thu Feb 13 2020 schwab@suse.de
- fix-locking-in-_IO_cleanup.patch: update to latest version
* Mon Feb  3 2020 schwab@suse.de
- Update to glibc 2.31
  * The GNU C Library now supports a feature test macro _ISOC2X_SOURCE to
    enable features from the draft ISO C2X standard
  * The <math.h> functions that round their results to a narrower type now
    have corresponding type-generic macros in <tgmath.h>
  * The function pthread_clockjoin_np has been added, enabling join with a
    terminated thread with a specific clock
  * New locale added: mnw_MM (Mon language spoken in Myanmar).
  * The DNS stub resolver will optionally send the AD (authenticated data) bit
    in queries if the trust-ad option is set via the options directive in
    /etc/resolv.conf (or if RES_TRUSTAD is set in _res.options)
  * The totalorder and totalordermag functions, and the corresponding
    functions for other floating-point types, now take pointer arguments to
    avoid signaling NaNs possibly being converted to quiet NaNs in argument
    passing
  * The obsolete function stime is no longer available to newly linked
    binaries, and its declaration has been removed from <time.h>
  * The gettimeofday function no longer reports information about a
    system-wide time zone
  * If a lazy binding failure happens during dlopen, during the execution of
    an ELF constructor, the process is now terminated
- malloc-info-whitespace.patch, riscv-vfork.patch,
  prefer-map-32bit-exec.patch, backtrace-powerpc.patch,
  ldconfig-dynstr.patch: Removed.
- bsc#1157893
- bsc#1163184
- fate#325815, fate#325879, fate#325880, fate#325881, fate#325882
- fate#325962
* Tue Jan 21 2020 schwab@suse.de
- backtrace-powerpc.patch: Fix array overflow in backtrace on PowerPC
  (CVE-2020-1751, bsc#1158996, BZ #25423)
- Drop support for pluggable gconv modules (bsc#1159851)
* Mon Dec  9 2019 schwab@suse.de
- prefer-map-32bit-exec.patch: rtld: Check __libc_enable_secure before
  honoring LD_PREFER_MAP_32BIT_EXEC (CVE-2019-19126, bsc#1157292, BZ
  [#25204])
* Tue Nov 26 2019 kukuk@suse.de
- nsswitch.conf: add usrfiles for services, protocols, rpc, ethers
  and aliases for /usr/etc move
* Mon Oct 14 2019 schwab@suse.de
- euc-kr-overrun.patch: Fix buffer overrun in EUC-KR conversion module
  (CVE-2019-25013, BZ #24973)
* Thu Oct 10 2019 schwab@suse.de
- ldconfig-dynstr.patch: ldconfig: handle .dynstr located in separate
  segment (bsc#1153149, BZ #25087)
* Mon Sep 23 2019 schwab@suse.de
- Package gconv-modules.cache as %%ghost
- Regenerate it also in the %%post of glibc-local-base-<targettype>
* Mon Sep 23 2019 lnussel@suse.de
- move mo files to glibc-locale as that's where all the other
  informations for those locales are. glibc-locale-base only has English
  anyways.
* Wed Sep 18 2019 schwab@suse.de
- riscv-vfork.patch: Fix RISC-V vfork build with Linux 5.3 kernel headers
* Mon Aug 19 2019 schwab@suse.de
- Remove NoSource tags (bsc#994835)
* Mon Aug 19 2019 kukuk@suse.de
- pwdutils is long gone and replaced by shadow
* Fri Aug  2 2019 schwab@suse.de
- Update to glibc 2.30
  * Unicode 12.1.0 Support
  * The dynamic linker accepts the --preload argument to preload shared
    objects
  * The twalk_r function has been added
  * On Linux, the getdents64, gettid, and tgkill functions have been added
  * Minguo (Republic of China) calendar support has been added
  * The entry for the new Japanese era has been added
  * Memory allocation functions malloc, calloc, realloc, reallocarray, valloc,
    pvalloc, memalign, and posix_memalign fail now with total object size
    larger than PTRDIFF_MAX
  * The dynamic linker no longer refuses to load objects which reference
    versioned symbols whose implementation has moved to a different soname
    since the object has been linked
  * Add new POSIX-proposed pthread_cond_clockwait, pthread_mutex_clocklock,
    pthread_rwlock_clockrdlock, pthread_rwlock_clockwrlock and sem_clockwait
    functions
  * On AArch64 the GNU IFUNC resolver call ABI changed
  * The copy_file_range function fails with ENOSYS if the kernel does not
    support the system call of the same name
  * The functions clock_gettime, clock_getres, clock_settime,
    clock_getcpuclockid, clock_nanosleep were removed from the librt library
    for new applications (on architectures which had them)
  * The obsolete and never-implemented XSI STREAMS header files <stropts.h>
    and <sys/stropts.h> have been removed
  * Support for the "inet6" option in /etc/resolv.conf and the RES_USE_INET6
    resolver flag (deprecated in glibc 2.25) have been removed
  * The obsolete RES_INSECURE1 and RES_INSECURE2 option flags for the DNS stub
    resolver have been removed from <resolv.h>
  * With --enable-bind-now, installed programs are now linked with the
    BIND_NOW flag.
  * On 32-bit Arm, support for the port-based I/O emulation and the <sys/io.h>
    header have been removed
  * The Linux-specific <sys/sysctl.h> header and the sysctl function have been
    deprecated and will be removed from a future version of glibc
  * CVE-2019-7309: x86-64 memcmp used signed Jcc instructions to check
    size
  * CVE-2019-9169: Attempted case-insensitive regular-expression match
    via proceed_next_node in posix/regexec.c leads to heap-based buffer
    over-read
- pthread-rwlock-trylock-stalls.patch,
  arm-systemtap-probe-constraint.patch, pthread-mutex-barrier.patch,
  fork-handler-lock.patch, pthread-join-probe.patch,
  riscv-clone-unwind.patch, add-new-Fortran-vector-math-header-file.patch,
  regex-read-overrun.patch, japanese-era-name-may-2019.patch,
  dl-show-auxv.patch, s390-vx-vxe-hwcap.patch, taisho-era-string.patch,
  malloc-tracing-hooks.patch, pldd-inf-loop.patch,
  malloc-large-bin-corruption-check.patch, wfile-sync-crash.patch,
  malloc-tests-warnings.patch, fnmatch-collating-elements.patch,
  iconv-reset-input-buffer.patch: Removed
- malloc-info-whitespace.patch: Remove unwanted leading whitespace in
  malloc_info (BZ #24867)
- bsc#1100396
- bsc#1130045
* Wed Jul 17 2019 schwab@suse.de
- Move /var/lib/misc/Makefile to /usr/share/misc/Makefile.makedb (bsc#1138726)
* Wed Jun 26 2019 schwab@suse.de
- malloc-tests-warnings.patch: Fix warnings in malloc tests with GCC 9
* Tue Jun 25 2019 mliska@suse.cz
- Set optflags for i686 after _lto_cflags is set (boo#1138807).
* Thu Jun 20 2019 mliska@suse.cz
- Disable LTO due to a usage of top-level assembler that
  causes LTO issues (boo#1138807).
* Wed Jun 12 2019 schwab@suse.de
- nss-files-long-lines-2.patch: Remove obsolete patch
* Wed May 15 2019 schwab@suse.de
- dl-show-auxv.patch: Fix output of LD_SHOW_AUXV=1
- s390-vx-vxe-hwcap.patch: S390: Mark vx and vxe as important hwcap
- taisho-era-string.patch: ja_JP: Change the offset for Taisho gan-nen
  from 2 to 1 (BZ #24162)
- malloc-tracing-hooks.patch: malloc: Set and reset all hooks for tracing
  (BZ #16573)
- pldd-inf-loop.patch: elf: Fix pldd (BZ#18035)
- malloc-large-bin-corruption-check.patch: malloc: Check for large bin
  list corruption when inserting unsorted chunk (BZ #24216)
- wfile-sync-crash.patch: Fix crash in _IO_wfile_sync (BZ #20568)
* Wed Apr  3 2019 schwab@suse.de
- japanese-era-name-may-2019.patch: ja_JP locale: Add entry for the new
  Japanese era (BZ #22964)
- Replace glibc_post_upgrade with lua script
* Thu Mar  7 2019 schwab@suse.de
- add-new-Fortran-vector-math-header-file.patch: Update from upstream
* Tue Mar  5 2019 schwab@suse.de
- regex-read-overrun.patch: fix read overrun (CVE-2019-9169, bsc#1127308,
  BZ #24114)
- ldconfig-concurrency.patch: Avoid concurrency problem in ldconfig
  (bsc#1117993, BZ #23973)
* Thu Feb 21 2019 mliska@suse.cz
- Add add-new-Fortran-vector-math-header-file.patch.
* Mon Feb 18 2019 schwab@suse.de
- pthread-rwlock-trylock-stalls.patch: nptl: Fix pthread_rwlock_try*lock
  stalls (BZ #23844)
- arm-systemtap-probe-constraint.patch: arm: Use "nr" constraint for
  Systemtap probes (BZ #24164)
- pthread-mutex-barrier.patch: Add compiler barriers around modifications
  of the robust mutex list for pthread_mutex_trylock (BZ #24180)
- fork-handler-lock.patch: nptl: Avoid fork handler lock for
  async-signal-safe fork (BZ #24161)
- pthread-join-probe.patch: nptl: Fix invalid Systemtap probe in
  pthread_join (BZ #24211)
- riscv-clone-unwind.patch: RISC-V: Fix elfutils testsuite unwind failures
  (BZ #24040)
* Fri Feb  1 2019 schwab@suse.de
- Update to glibc 2.29
  * The getcpu wrapper function has been added, which returns the currently
    used CPU and NUMA node
  * Optimized generic exp, exp2, log, log2, pow, sinf, cosf, sincosf and tanf
  * The reallocarray function is now declared under _DEFAULT_SOURCE, not just
    for _GNU_SOURCE, to match BSD environments
  * For powercp64le ABI, Transactional Lock Elision is now enabled iff kernel
    indicates that it will abort the transaction prior to entering the kernel
    (PPC_FEATURE2_HTM_NOSC on hwcap2)
  * The functions posix_spawn_file_actions_addchdir_np and
    posix_spawn_file_actions_addfchdir_np have been added, enabling
    posix_spawn and posix_spawnp to run the new process in a different
    directory
  * The popen and system do not run atfork handlers anymore (BZ#17490)
  * strftime's default formatting of a locale's alternative year (%%Ey)
    has been changed to zero-pad the year to a minimum of two digits,
    like "%%y"
  * As a GNU extension, the '_' and '-' flags can now be applied to
    "%%EY" to control how the year number is formatted
  * The glibc.tune tunable namespace has been renamed to glibc.cpu and the
    tunable glibc.tune.cpu has been renamed to glibc.cpu.name
  * The type of the pr_uid and pr_gid members of struct elf_prpsinfo, defined
    in <sys/procfs.h>, has been corrected to match the type actually used by
    the Linux kernel
  * An archaic GNU extension to scanf, under which '%%as', '%%aS', and '%%a[...]'
    meant to scan a string and allocate space for it with malloc, is now
    restricted to programs compiled in C89 or C++98 mode with _GNU_SOURCE
    defined
- unwind-ctor.patch, old-getdents64.patch, nss-files-leak.patch,
  riscv-feholdexcept-setround.patch,
  pthread-cond-broadcast-waiters-after-spinning.patch,
  regex-uninit-memory-access.patch, spawni-maybe-script-execute.patch,
  gethostid-gethostbyname-failure.patch, strstr-huge-needle.patch,
  pthread-mutex-lock-elision-race.patch, x86-haswell-string-flags.patch,
  if-nametoindex-descr-leak.patch, riscv-flush-icache.patch: Removed
- CVE-2016-10739
- bsc#1114984
- bsc#1114993
- bsc#1122729
- bsc#1131330
- bsc#1149332
- bsc#1151582
- bsc#1164505
* Wed Jan  9 2019 schwab@suse.de
- fnmatch-collating-elements.patch: update
- riscv-flush-icache.patch: fix for compiling against 4.20 headers
* Wed Nov 28 2018 schwab@suse.de
- if-nametoindex-descr-leak.patch: if_nametoindex: Fix descriptor leak for
  overlong name (CVE-2018-19591, BZ #23927, bsc#1117603)
* Wed Nov  7 2018 jengelh@inai.de
- Fix typography for glibc-locale-base.
* Wed Nov  7 2018 schwab@suse.de
- pthread-mutex-lock-elision-race.patch: Fix race in pthread_mutex_lock
  while promoting to PTHREAD_MUTEX_ELISION_NP (BZ #23275)
- x86-haswell-string-flags.patch: x86: Fix Haswell CPU string flags (BZ
  [#23709])
* Wed Sep 26 2018 schwab@suse.de
- unwind-ctor.patch: Add missing unwind information to ld.so on powerpc32
  (BZ #23707)
- old-getdents64.patch: Rewrite __old_getdents64 (BZ #23497)
- nss-files-leak.patch: Fix file stream leak in aliases lookup (BZ #23521)
- riscv-feholdexcept-setround.patch: Fix rounding save/restore bug
- pthread-cond-broadcast-waiters-after-spinning.patch: Fix
  waiters-after-spinning case (BZ #23538)
- regex-uninit-memory-access.patch: fix uninitialized memory access (BZ
  [#23578])
- spawni-maybe-script-execute.patch: Fix segfault in maybe_script_execute
- gethostid-gethostbyname-failure.patch: Check for NULL value from
  gethostbyname_r (BZ #23679)
- strstr-huge-needle.patch: Fix strstr bug with huge needles (BZ #23637)
* Wed Sep 12 2018 schwab@suse.de
- Add libpng-devel and zlib-devel for utils build
* Wed Aug 15 2018 schwab@suse.de
- Add glibc-locale-base subpackage containing only C, C.UTF-8 and
  en_US.UTF-8 locales (fate#326551)
* Thu Aug  2 2018 schwab@suse.de
- Update to glibc 2.28
  * The localization data for ISO 14651 is updated to match the 2016
    Edition 4 release of the standard, this matches data provided by
    Unicode 9.0.0
  * Unicode 11.0.0 Support: Character encoding, character type info, and
    transliteration tables are all updated to Unicode 11.0.0, using
    generator scripts contributed by Mike FABIAN (Red Hat)
  * <math.h> functions that round their results to a narrower type are added
    from TS 18661-1:2014 and TS 18661-3:2015
  * Two grammatical forms of month names are now supported
  * The renameat2 function has been added, a variant of the renameat function
    which has a flags argument
  * The statx function has been added, a variant of the fstatat64
    function with an additional flags argument
  * IDN domain names in getaddrinfo and getnameinfo now use the system libidn2
    library if installed
  * Parsing of dynamic string tokens in DT_RPATH, DT_RUNPATH, DT_NEEDED,
    DT_AUXILIARY, and DT_FILTER has been expanded to support the full
    range of ELF gABI expressions including such constructs as
    '$ORIGIN$ORIGIN' (if valid)
  * Support for ISO C threads (ISO/IEC 9899:2011) has been added.
  * The nonstandard header files <libio.h> and <_G_config.h> are no longer
    installed
  * The stdio functions 'getc' and 'putc' are no longer defined as macros
  * All stdio functions now treat end-of-file as a sticky condition
  * The macros 'major', 'minor', and 'makedev' are now only available from
    the header <sys/sysmacros.h>
  * The obsolete function ustat is no longer available to newly linked
    binaries; the headers <ustat.h> and <sys/ustat.h> have been removed
  * The obsolete function nfsservctl is no longer available to newly linked
    binaries
  * The obsolete function name llseek is no longer available to newly linked
    binaries
  * The AI_IDN_ALLOW_UNASSIGNED and NI_IDN_ALLOW_UNASSIGNED flags for the
    getaddrinfo and getnameinfo functions have been deprecated
  * The AI_IDN_USE_STD3_ASCII_RULES and NI_IDN_USE_STD3_ASCII_RULES flags for
    the getaddrinfo and getnameinfo functions have been deprecated
  * The fcntl function now have a Long File Support variant named fcntl64
  * CVE-2016-6261, CVE-2016-6263, CVE-2017-14062: Various vulnerabilities have
    been fixed by removing the glibc-internal IDNA implementation and using
    the system-provided libidn2 library instead
- Split off all libcrypt related functions into package libxcrypt
- fix-locking-in-_IO_cleanup.patch, fnmatch-collating-elements.patch:
  Rediff
- aarch64-sys-ptrace-update.patch,
  crypt_blowfish-1.2-hack_around_arm.diff, crypt_blowfish-1.2-sha.diff,
  crypt_blowfish-const.patch, crypt_blowfish-gensalt.patch,
  glibc-2.14-crypt.diff, i386-memmove-sse2-unaligned.patch,
  i386-sigaction-sa-restorer.patch, mempcpy-avx512.patch,
  netgroup-cache-keys.patch, nss-database-multiple-dfn.patch,
  pkey-get-reserved-name.patch, powerpc-sys-ptrace-undefine-macros.patch,
  powerpc-sys-ptrace-update.patch, realpath-ssize-max-overflow.patch,
  res-send-enomem.patch, riscv-fmax-fmin-nan.patch,
  riscv-kernel-sigaction.patch, riscv-readelflib.patch,
  riscv-tls-init.patch: Removed
- glibc_post_upgrade.c: Don't reload init (bsc#1103124)
- CVE-2009-5155, CVE-2015-8985
- bsc#1092877
- bsc#1102526
- bsc#1112570
- bsc#1126590
- bsc#1127223
* Tue Jun 19 2018 schwab@suse.de
- Use python3-pexpect instead of python-pexpect
* Thu Jun 14 2018 schwab@suse.de
- riscv-kernel-sigaction.patch: fix struct kernel_sigaction to match the
  kernel version (BZ #23069)
* Mon Jun  4 2018 schwab@suse.de
- glibc-2.3.90-langpackdir.diff: No longer search in /usr/share/locale-bundle
* Wed May 23 2018 schwab@suse.de
- mempcpy-avx512.patch: Don't write beyond destination in
  __mempcpy_avx512_no_vzeroupper (CVE-2018-11237, bsc#1094154)
- realpath-ssize-max-overflow.patch: Fix overflow in path length
  computation (CVE-2018-11236, bsc#1094161, BZ #22786)
- Use %%license also for COPYING, COPYING.LIB
* Thu Apr 12 2018 schwab@suse.de
- Readd nis to netgroup and automount nss config (bsc#1088860)
* Thu Mar 29 2018 schwab@suse.de
- i386-memmove-sse2-unaligned.patch: Fix SSE2 memmove issue when crossing
  2GB boundary (CVE-2017-18269, bnc#1094150, BZ #22644)
* Tue Mar 27 2018 schwab@suse.de
- res-send-enomem.patch: Fix crash in resolver on memory allocation
  failure (bsc#1086690, BZ #23005)
* Thu Mar 22 2018 schwab@suse.de
- pkey-get-reserved-name.patch: Linux: use reserved name __key in pkey_get
  (BZ #22797)
- aarch64-sys-ptrace-update.patch: linux/aarch64: sync sys/ptrace.h with
  Linux 4.15 (BZ #22433)
- powerpc-sys-ptrace-undefine-macros.patch: powerpc: Undefine Linux ptrace
  macros that conflict with __ptrace_request
- powerpc-sys-ptrace-update.patch: linux/powerpc: sync sys/ptrace.h with
  Linux 4.15 (BZ #22433, BZ #22807)
- netgroup-cache-keys.patch: Fix netgroup cache keys (BZ #22342)
- i386-sigaction-sa-restorer.patch: i386: Fix i386 sigaction sa_restorer
  initialization (BZ #21269)
- riscv-tls-init.patch: RISC-V: Do not initialize $gp in TLS macros
- riscv-fmax-fmin-nan.patch: RISC-V: fmax/fmin: Handle signalling NaNs
  correctly (BZ #22884)
* Wed Mar  7 2018 schwab@suse.de
- nss-database-multiple-dfn.patch: Fix multiple definitions of
  __nss_*_database (BZ #22918)
* Thu Feb 22 2018 fvogt@suse.com
- Use %%license (boo#1082318)
* Wed Feb 14 2018 schwab@suse.de
- Add systemtap-headers to BuildRequires
- Add --enable-systemtap to configure arguments (fate#324969, bsc#1073636)
* Tue Feb 13 2018 schwab@suse.de
- riscv-readelflib.patch: Fix parsing flags in ELF64 files on riscv
* Mon Feb  5 2018 schwab@suse.de
- Update to glibc 2.27
  * Optimized x86-64 asin, atan2, exp, expf, log, pow, atan, sin, cosf,
    sinf, sincosf and tan with FMA
  * Optimized x86-64 trunc and truncf for processors with SSE4.1
  * Optimized generic expf, exp2f, logf, log2f, powf, sinf, cosf and
    sincosf
  * In order to support faster and safer process termination the malloc API
    family of functions will no longer print a failure address and stack
    backtrace after detecting heap corruption
  * The abort function terminates the process immediately, without flushing
    stdio streams
  * On platforms where long double has the IEEE binary128 format (aarch64,
    alpha, mips64, riscv, s390 and sparc), the math library now implements
    _Float128 interfaces for that type, as defined by ISO/IEC TS 18661-3:2015
    These are the same interfaces added in version 2.26 for some platforms where
    this format is supported but is not the format of long double
  * On platforms with support for _Float64x (aarch64, alpha, i386, ia64,
    mips64, powerpc64le, riscv, s390, sparc and x86_64), the math library now
    implements interfaces for that type, as defined by ISO/IEC TS
    18661-3:2015
  * The math library now implements interfaces for the _Float32, _Float64 and
    _Float32x types, as defined by ISO/IEC TS 18661-3:2015
  * glibc now implements the memfd_create and mlock2 functions on Linux
  * Support for memory protection keys was added
  * The copy_file_range function was added
  * The ldconfig utility now processes `include' directives using the C/POSIX
    collation ordering
  * Support for two grammatical forms of month names has been added
  * Support for the RISC-V ISA running on Linux has been added
  * Statically compiled applications attempting to load locales compiled for the
    GNU C Library version 2.27 will fail and fall back to the builtin C/POSIX
    locale
  * Support for statically linked applications which call dlopen is deprecated
    and will be removed in a future version of glibc
  * Support for old programs which use internal stdio data structures and
    functions is deprecated
  * On GNU/Linux, the obsolete Linux constant PTRACE_SEIZE_DEVEL is no longer
    defined by <sys/ptrace.h>
  * libm no longer supports SVID error handling (calling a user-provided
    matherr function on error) or the _LIB_VERSION variable to control error
    handling
  * The libm functions pow10, pow10f and pow10l are no longer supported for
    new programs
  * The mcontext_t type is no longer the same as struct sigcontext
  * The add-ons mechanism for building additional packages at the same time as
    glibc has been removed
  * The res_hnok, res_dnok, res_mailok and res_ownok functions now check that
    the specified string can be parsed as a domain name
  * In the malloc_info output, the <heap> element may contain another <aspace>
    element, "subheaps", which contains the number of sub-heaps
  * In the malloc_info output, the <heap> element may contain another <aspace>
    element, "subheaps", which contains the number of sub-heaps
  * The nonstandard header files <libio.h> and <_G_config.h> are deprecated
    and will be removed in a future release
  * CVE-2018-6485, CVE-2018-6551: The posix_memalign and memalign
    functions, when called with an object size near the value of SIZE_MAX,
    would return a pointer to a buffer which is too small, instead of NULL
    (bsc#1079036)
- Support for Sun RPC is no longer available, use libtirpc instead
- glibc-nodate.patch, powerpc-elision-enable-envvar.patch,
  s390-elision-enable-envvar.patch, resolv-context-leak.patch,
  dl-runtime-resolve-opt-avx512f.patch, libpthread-compat-wrappers.patch,
  math-c++-compat.patch, remove-nss-nis-compat.patch,
  eh-frame-zero-terminator.patch, ld-so-hwcap-x86-64.patch,
  assert-pedantic.patch, getaddrinfo-errno.patch, resolv-conf-oom.patch,
  dynarray-allocation.patch, nearbyint-inexact.patch, nss-compat.patch,
  nscd-libnsl.patch, malloc-tcache-leak.patch,
  falkor-memcpy-memmove.patch, aarch64-cpu-features.patch,
  nss-files-large-buffers.patch. sysconf-uio-maxiov.patch,
  glob-tilde-overflow.patch, dl-runtime-resolve-xsave.patch,
  spawni-assert.patch, x86-64-dl-platform.patch, glob64-s390.patch,
  tst-tlsopt-powerpc.patch, powerpc-hwcap-bits.patch,
  malloc-tcache-check-overflow.patch, dl-init-paths-overflow.patch,
  fillin-rpath-empty-tokens.patch, getcwd-absolute.patch,
  ldd-system-interp.patchabort-no-flush.patch: Removed
- All patches refreshed
- bsc#1063675
* Tue Jan 30 2018 schwab@suse.de
- fix-locking-in-_IO_cleanup.patch: Skip locked files during exit
  (bsc#1070491, BZ #15142)
* Tue Jan 30 2018 schwab@suse.de
- Avoid duplicate source rpm
* Mon Jan 22 2018 schwab@suse.de
- getcwd-absolute.patch: make getcwd(3) fail if it cannot obtain an
  absolute path (CVE-2018-1000001, bsc#1074293, BZ #22679)
* Tue Jan  2 2018 schwab@suse.de
- dl-init-paths-overflow.patch: Count components of the expanded path in
  _dl_init_path (CVE-2017-1000408, CVE-2017-1000409, bsc#1071319, BZ
  [#22607], BZ #22627)
- fillin-rpath-empty-tokens.patch: Check for empty tokens before dynamic
  string token expansion (CVE-2017-16997, bsc#1073231, BZ #22625)
* Wed Dec 13 2017 schwab@suse.de
- tst-tlsopt-powerpc.patch: fix tst-tlsopt-powerpc (bcn#1070419)
- powerpc-hwcap-bits.patch: Update HWCAP for powerpc (bnc#1070420)
- malloc-tcache-check-overflow.patch: Fix integer overflow in malloc when
  tcache is enabled (CVE-2017-17426, bnc#1071479, BZ #22375)
* Tue Dec 12 2017 schwab@suse.de
- Add _multibuild
* Mon Nov 27 2017 schwab@suse.de
- glob64-s390.patch: no compat glob64 on s390
- noversion.tar.bz2: remove unused source
* Thu Oct 26 2017 schwab@suse.de
- x86-64-dl-platform.patch: Don't set GLRO(dl_platform) to NULL (BZ #22299)
* Thu Oct 26 2017 schwab@suse.de
- spawni-assert.patch: Fix improper assert in Linux posix_spawn (BZ
  [#22273])
* Mon Oct 23 2017 schwab@suse.de
- math-c++-compat.patch: Add more C++ compatibility (BZ #22296)
- malloc-tcache-leak.patch: Fix tcache leak after thread destruction (BZ
  [#22111])
- falkor-memcpy-memmove.patch: Optimized implementation of memcpy/memmove
  for Qualcomm Falkor
- aarch64-cpu-features.patch: Fix glibc.tune.cpu tunable handling
- nss-files-large-buffers.patch: Avoid large buffers with many host
  addresses (BZ #22078)
- sysconf-uio-maxiov.patch: Fix missing definition of UIO_MAXIOV (BZ
  [#22321])
- glob-tilde-overflow.patch: Fix buffer overflows with GLOB_TILDE
  (CVE-2017-15670, CVE-2017-15671, CVE-2017-15804,
  bsc#1064569. bsc#1064580, bsc#1064583, BZ #22320, BZ #22325, BZ #22332)
- dl-runtime-resolve-xsave.patch: Use fxsave/xsave/xsavec in
  _dl_runtime_resolve (BZ #21265)
* Tue Oct 10 2017 schwab@suse.de
- nscd-libnsl.patch: Remove reference to libnsl from nscd (bsc#1062244)
* Mon Oct  9 2017 schwab@suse.de
- Drop glibc-obsolete
- glibc-2.3.90-noversion.diff: remove
- reinitialize-dl_load_write_lock.patch: remove
* Wed Oct  4 2017 schwab@suse.de
- nss-compat.patch: Move nss_compat from nis to nss subdir and install it
  unconditionally
- nsswitch.conf: switch back to compat for passwd, group, shadow
* Thu Sep 28 2017 schwab@suse.de
- assert-pedantic.patch: Suppress pedantic warning caused by statement
  expression (BZ #21242, BZ #21972)
- math-c++-compat.patch: Add more C++ compatibility (BZ #22235)
- getaddrinfo-errno.patch: Fix errno and h_errno handling in getaddrinfo
  (BZ #21915, BZ #21922)
- resolv-conf-oom.patch: Fix memory handling in OOM situation during
  resolv.conf parsing (BZ #22095, BZ #22096)
- dynarray-allocation.patch: Fix initial size of dynarray allocation and
  set errno on overflow error
- nearbyint-inexact.patch: Avoid spurious inexact in nearbyint (BZ #22225)
* Mon Sep 25 2017 schwab@suse.de
- math-c++-compat.patch: add more C++ compatibility (BZ #22146)
* Tue Sep 12 2017 schwab@suse.de
- Remove rpcsvc/yppasswd.* from glibc-devel
- ld-so-hwcap-x86-64.patch: add x86_64 to hwcap (bsc#1056606, BZ #22093)
* Thu Aug 31 2017 schwab@suse.de
- eh-frame-zero-terminator.patch: Properly terminate .eh_frame (BZ #22051)
* Thu Aug 31 2017 schwab@suse.de
- Disable obsolete libnsl and NIS support
- remove-nss-nis-compat.patch: remove nis and compat from default NSS
  configs
- nsswitch.conf: Likewise
* Tue Aug 29 2017 schwab@suse.de
- math-c++-compat.patch: Do not use __builtin_types_compatible_p in C++
  mode (BZ #21930)
* Thu Aug 17 2017 schwab@suse.de
- Add iconvconfig to baselibs.conf (bsc#1051042)
* Thu Aug 10 2017 schwab@suse.de
- resolv-context-leak.patch: Fix leaks of resolver contexts
- dl-runtime-resolve-opt-avx512f.patch: Use _dl_runtime_resolve_opt only
  with AVX512F (BZ #21871)
- libpthread-compat-wrappers.patch: Don't use IFUNC resolver for longjmp
  or system in libpthread (BZ #21041)
* Wed Aug  2 2017 schwab@suse.de
- Update to glibc 2.26
  * A per-thread cache has been added to malloc
  * Unicode 10.0.0 Support
  * Improvements to the DNS stub resolver
  * New function reallocarray, which resizes an allocated block (like
    realloc) to the product of two sizes, with a guaranteed clean failure
    upon integer overflow in the multiplication
  * New wrappers for the Linux-specific system calls preadv2 and pwritev2
  * posix_spawnattr_setflags now supports the flag POSIX_SPAWN_SETSID, to
    create a new session ID for the spawned process
  * errno.h is now safe to use from C-preprocessed assembly language on all
    supported operating systems
  * On ia64, powerpc64le, x86-32, and x86-64, the math library now implements
    128-bit floating point as defined by ISO/IEC/IEEE 60559:2011 (IEEE
    754-2008) and ISO/IEC TS 18661-3:2015
  * The synchronization that pthread_spin_unlock performs has been changed to
    now be equivalent to a C11 atomic store with release memory order to the
    spin lock's memory location
  * The DNS stub resolver no longer performs EDNS fallback
  * res_mkquery and res_nmkquery no longer support the IQUERY opcode
  * The _res_opcodes variable has been removed from libresolv
  * <string.h> no longer includes inline versions of any string functions,
    as this kind of optimization is better done by the compiler
  * The nonstandard header <xlocale.h> has been removed
  * The obsolete header <sys/ultrasound.h> has been removed
  * The obsolete signal constant SIGUNUSED is no longer defined by <signal.h>
  * The obsolete function cfree has been removed
  * The stack_t type no longer has the name struct sigaltstack
  * The ucontext_t type no longer has the name struct ucontext
  * On S/390 GNU/Linux, the constants defined by <sys/ptrace.h> have been
    synced with the kernel
  * Linux kernel 3.2 or later is required at runtime, on all architectures
    supported by that kernel
  * The DNS stub resolver limits the advertised UDP buffer size to 1200 bytes,
    to avoid fragmentation-based spoofing attacks (CVE-2017-12132)
  * LD_LIBRARY_PATH is now ignored in binaries running in privileged
    AT_SECURE mode to guard against local privilege escalation attacks
    (CVE-2017-1000366)
  * Avoid printing a backtrace from the __stack_chk_fail function since it
    is called on a corrupt stack and a backtrace is unreliable on a
    corrupt stack (CVE-2010-3192)
  * A use-after-free vulnerability in clntudp_call in the Sun RPC system
    has been fixed (CVE-2017-12133)
  * fate#322258, fate#321513, fate#322453
- fts-symbol-redirect.patch, glibc-resolv-reload.diff, glibc-2.2-sunrpc.diff,
  i686-memchr-sse.patch, ld-hwcap-mask-suid.patch, ld-library-path-suid.patch,
  sunrpc-use-after-free.patch, test-math-vector-sincos-aliasing.patch,
  tunables-bigendian.patch: Removed
- bsc#1074208
* Tue Jul 25 2017 schwab@suse.de
- Fix RPM group
* Thu Jul 20 2017 schwab@suse.de
- s390-elision-enable-envvar.patch: enable TLE only if
  GLIBC_ELISION_ENABLE=yes is defined (fate#322271)
* Wed Jun 21 2017 schwab@suse.de
- ld-hwcap-mask-suid.patch: Ignore and remove LD_HWCAP_MASK for AT_SECURE
  programs (BZ #21209)
- ld-library-path-suid.patch: Completely ignore LD_LIBRARY_PATH for
  AT_SECURE=1 programs (CVE-2017-1000366, bsc#1039357, BZ #21624)
* Wed Jun 14 2017 schwab@suse.de
- Remove glibc-cpusetsize.diff, no longer useful
* Wed May 31 2017 schwab@suse.de
- fts-symbol-redirect.patch: Fix symbol redirect for fts_set (bsc#1041123,
  BZ #21289)
* Mon May 29 2017 schwab@suse.de
- test-math-vector-sincos-aliasing.patch: Fix test-math-vector-sincos.h
  aliasing
* Mon May 29 2017 schwab@suse.de
- add-locales.patch: renamed from glibc-2.3.locales.diff.bz2, drop en_BE
  locales (bsc#1039502)
* Tue May 23 2017 schwab@suse.de
- Remove glibc-testsuite.patch, no longer relevant
* Mon May 22 2017 schwab@suse.de
- Use multibuild feature
- Remove obsolete check-build.sh
- glibc.rpmlintrc: remove obsolete entries
- Use %%tmpfiles_create in nscd postin
* Wed Mar 29 2017 schwab@suse.de
- i686-memchr-sse.patch: Fix i686 memchr overflow calculation
  (bsc#1031021, BZ #21182)
- sunrpc-use-after-free.patch: Avoid use-after-free read access in
  clntudp_call (BZ #21115)
- Build testsuite with gdb and python-pexpect to enable more tests
* Wed Feb  8 2017 schwab@suse.de
- tunables-bigendian.patch: Fix getting tunable values on big-endian (BZ
  [#21109])
* Sun Feb  5 2017 schwab@suse.de
- Update to glibc 2.25
  * The feature test macro __STDC_WANT_LIB_EXT2__, from ISO/IEC TR
    24731-2:2010, is supported to enable declarations of functions from that
    TR.
  * The feature test macro __STDC_WANT_IEC_60559_BFP_EXT__, from ISO/IEC TS
    18661-1:2014, is supported to enable declarations of functions and macros
    from that TS.
  * The feature test macro __STDC_WANT_IEC_60559_FUNCS_EXT__, from ISO/IEC TS
    18661-4:2015, is supported to enable declarations of functions and macros
    from that TS.
  * The nonstandard feature selection macros _REENTRANT and _THREAD_SAFE are
    now treated as compatibility synonyms for _POSIX_C_SOURCE=199506L.
  * The inclusion of <sys/sysmacros.h> by <sys/types.h> is deprecated.
  * New <fenv.h> features from TS 18661-1:2014 are added to libm: the
    fesetexcept, fetestexceptflag, fegetmode and fesetmode functions, the
    femode_t type and the FE_DFL_MODE and FE_SNANS_ALWAYS_SIGNAL macros.
  * Integer width macros from TS 18661-1:2014 are added to <limits.h>:
    CHAR_WIDTH, SCHAR_WIDTH, UCHAR_WIDTH, SHRT_WIDTH, USHRT_WIDTH, INT_WIDTH,
    UINT_WIDTH, LONG_WIDTH, ULONG_WIDTH, LLONG_WIDTH, ULLONG_WIDTH; and to
    <stdint.h>: INT8_WIDTH, UINT8_WIDTH, INT16_WIDTH, UINT16_WIDTH,
    INT32_WIDTH, UINT32_WIDTH, INT64_WIDTH, UINT64_WIDTH, INT_LEAST8_WIDTH,
    UINT_LEAST8_WIDTH, INT_LEAST16_WIDTH, UINT_LEAST16_WIDTH,
    INT_LEAST32_WIDTH, UINT_LEAST32_WIDTH, INT_LEAST64_WIDTH,
    UINT_LEAST64_WIDTH, INT_FAST8_WIDTH, UINT_FAST8_WIDTH, INT_FAST16_WIDTH,
    UINT_FAST16_WIDTH, INT_FAST32_WIDTH, UINT_FAST32_WIDTH, INT_FAST64_WIDTH,
    UINT_FAST64_WIDTH, INTPTR_WIDTH, UINTPTR_WIDTH, INTMAX_WIDTH,
    UINTMAX_WIDTH, PTRDIFF_WIDTH, SIG_ATOMIC_WIDTH, SIZE_WIDTH, WCHAR_WIDTH,
    WINT_WIDTH.
  * New <math.h> features are added from TS 18661-1:2014:
  - Signaling NaN macros: SNANF, SNAN, SNANL.
  - Nearest integer functions: roundeven, roundevenf, roundevenl, fromfp,
    fromfpf, fromfpl, ufromfp, ufromfpf, ufromfpl, fromfpx, fromfpxf,
    fromfpxl, ufromfpx, ufromfpxf, ufromfpxl.
  - llogb functions: the llogb, llogbf and llogbl functions, and the
    FP_LLOGB0 and FP_LLOGBNAN macros.
  - Max-min magnitude functions: fmaxmag, fmaxmagf, fmaxmagl, fminmag,
    fminmagf, fminmagl.
  - Comparison macros: iseqsig.
  - Classification macros: iscanonical, issubnormal, iszero.
  - Total order functions: totalorder, totalorderf, totalorderl,
    totalordermag, totalordermagf, totalordermagl.
  - Canonicalize functions: canonicalize, canonicalizef, canonicalizel.
  - NaN functions: getpayload, getpayloadf, getpayloadl, setpayload,
    setpayloadf, setpayloadl, setpayloadsig, setpayloadsigf, setpayloadsigl.
  * The functions strfromd, strfromf, and strfroml, from ISO/IEC TS 18661-1:2014,
    are added to libc.
  * Most of glibc can now be built with the stack smashing protector enabled.
  * The function explicit_bzero, from OpenBSD, has been added to libc.
  * On ColdFire, MicroBlaze, Nios II and SH3, the float_t type is now defined
    to float instead of double.
  * On x86_64, when compiling with -mfpmath=387 or -mfpmath=sse+387, the
    float_t and double_t types are now defined to long double instead of float
    and double.
  * The getentropy and getrandom functions, and the <sys/random.h> header file
    have been added.
  * The buffer size for byte-oriented stdio streams is now limited to 8192
    bytes by default.
  * The <sys/quota.h> header now includes the <linux/quota.h> header.
  * The malloc_get_state and malloc_set_state functions have been removed.
  * The “ip6-dotint” and “no-ip6-dotint” resolver options, and the
    corresponding RES_NOIP6DOTINT flag from <resolv.h> have been removed.
  * The "ip6-bytestring" resolver option and the corresponding RES_USEBSTRING
    flag from <resolv.h> have been removed.
  * The flags RES_AAONLY, RES_PRIMARY, RES_NOCHECKNAME, RES_KEEPTSIG,
    RES_BLAST defined in the <resolv.h> header file have been deprecated.
  * The "inet6" option in /etc/resolv.conf and the RES_USE_INET6 flag for
    _res.flags are deprecated.
  * DNSSEC-related declarations and definitions have been removed from the
    <arpa/nameser.h> header file, and libresolv will no longer attempt to
    decode the data part of DNSSEC record types.
  * The resource record type classification macros ns_t_qt_p, ns_t_mrr_p,
    ns_t_rr_p, ns_t_udp_p, ns_t_xfr_p have been removed from the
    <arpa/nameser.h> header file because the distinction between RR types and
    meta-RR types is not officially standardized, subject to revision, and
    thus not suitable for encoding in a macro.
  * The types res_sendhookact, res_send_qhook, re_send_rhook, and the qhook
    and rhook members of the res_state type in <resolv.h> have been removed.
  * For multi-arch support it is recommended to use a GCC which has
    been built with support for GNU indirect functions.
  * GDB pretty printers have been added for mutex and condition variable
    structures in POSIX Threads.
  * Tunables feature added to allow tweaking of the runtime for an application
    program.
  * A new version of condition variables functions have been implemented in
    the NPTL implementation of POSIX Threads to provide stronger ordering
    guarantees.
  * A new version of pthread_rwlock functions have been implemented to use a more
    scalable algorithm primarily through not using a critical section anymore to
    make state changes.
  * On ARM EABI (32-bit), generating a backtrace for execution contexts which
    have been created with makecontext could fail to terminate due to a
    missing .cantunwind annotation.  (CVE-2016-6323)
  * The DNS stub resolver functions would crash due to a NULL pointer
    dereference when processing a query with a valid DNS question type which
    was used internally in the implementation. (CVE-2015-5180)
- Enable stack protector if part of %%optflags
- startcontext-cantunwind.patch: Removed
- cpuid-assertion.patch: Removed
* Thu Oct 13 2016 schwab@suse.de
- cpuid-assertion.patch: Don't assert on older Intel CPUs (BZ #20647)
* Wed Oct  5 2016 schwab@suse.de
- glibc-2.3.3-nscd-db-path.diff: Move persistent nscd databases to
  /var/lib/nscd
- glibc-2.3.90-langpackdir.diff: simplify
* Tue Aug  2 2016 schwab@suse.de
- Update to glibc 2.24
  * The minimum Linux kernel version that this version of the GNU C Library
    can be used with is 3.2
  * The pap_AN locale has been deleted
  * The readdir_r and readdir64_r functions have been deprecated
  * The type `union wait' has been removed
  * A new NSS action is added to facilitate large distributed system
    administration
  * The deprecated __malloc_initialize_hook variable has been removed from
    the API
  * The long unused localedef --old-style option has been removed
  * nextupl, nextup, nextupf, nextdownl, nextdown and nextdownf are added
    to libm
  * An unnecessary stack copy in _nss_dns_getnetbyname_r was removed
    (CVE-2016-3075)
  * Previously, getaddrinfo copied large amounts of address data to the
    stack, even after the fix for CVE-2013-4458 has been applied,
    potentially resulting in a stack overflow.  getaddrinfo now uses a
    heap allocation instead (CVE-2016-3706)
  * The glob function suffered from a stack-based buffer overflow when it
    was called with the GLOB_ALTDIRFUNC flag and encountered a long file
    name (CVE-2016-1234)
  * The Sun RPC UDP client could exhaust all available stack space when
    flooded with crafted ICMP and UDP messages (CVE-2016-4429)
  * The IPv6 name server management code in libresolv could result in a
    memory leak for each thread which is created, performs a failing
    naming lookup, and exits (CVE-2016-5417)
- startcontext-cantunwind.patch: mark __startcontext as .cantunwind
  (bsc#974800, BZ #20435)
- Removed patches:
  * 0001-Updated-translations-for-2.23.patch
  * 0002-Regenerate-libc.pot-for-2.23.patch
  * 0003-Regenerated-configure-scripts.patch
  * 0004-x86_64-Set-DL_RUNTIME_UNALIGNED_VEC_SIZE-to-8.patch
  * 0005-Add-fts64_-to-sysdeps-arm-nacl-libc.abilist.patch
  * 0006-Don-t-use-long-double-math-functions-if-NO_LONG_DOUB.patch
  * 0007-NEWS-2.23-Fix-typo-in-bug-19048-text.patch
  * 0008-Update-NEWS.patch
  * 0009-sln-use-stat64.patch
  * 0010-Add-sys-auxv.h-wrapper-to-include-sys.patch
  * 0011-mips-terminate-the-FDE-before-the-return-trampoline-.patch
  * 0012-Use-HAS_ARCH_FEATURE-with-Fast_Rep_String.patch
  * 0013-Mention-BZ-19762-in-NEWS.patch
  * 0014-Define-_HAVE_STRING_ARCH_mempcpy-to-1-for-x86.patch
  * 0015-Or-bit_Prefer_MAP_32BIT_EXEC-in-EXTRA_LD_ENVVARS.patch
  * 0016-Fix-resource-leak-in-resolver-bug-19257.patch
  * 0017-math-don-t-clobber-old-libm.so-on-install-BZ-19822.patch
  * 0018-resolv-Always-set-resplen2-out-parameter-in-send_dg-.patch
  * 0019-S390-Save-and-restore-fprs-vrs-while-resolving-symbo.patch
  * 0020-S390-Extend-structs-La_s390_regs-La_s390_retval-with.patch
  * 0021-CVE-2016-3075-Stack-overflow-in-_nss_dns_getnetbynam.patch
  * 0022-configure-fix-test-usage.patch
  * 0023-Suppress-GCC-6-warning-about-ambiguous-else-with-Wpa.patch
  * 0024-nss_db-Propagate-ERANGE-error-if-parse_line-fails-BZ.patch
  * 0025-getnameinfo-Do-not-preserve-errno.patch
  * 0026-getnameinfo-Refactor-and-fix-memory-leak-BZ-19642.patch
  * 0027-getnameinfo-Reduce-line-length-and-add-missing-comme.patch
  * 0028-getnameinfo-Avoid-calling-strnlen-on-uninitialized-b.patch
  * 0029-getnameinfo-Return-EAI_OVERFLOW-in-more-cases-BZ-197.patch
  * 0030-hesiod-Remove-RCS-keywords.patch
  * 0031-hesiod-Always-use-thread-local-resolver-state-BZ-195.patch
  * 0032-hesiod-Avoid-heap-overflow-in-get_txt_records-BZ-200.patch
  * 0033-malloc-Remove-NO_THREADS.patch
  * 0034-Fix-malloc-threaded-tests-link-on-non-Linux.patch
  * 0035-malloc-Run-fork-handler-as-late-as-possible-BZ-19431.patch
  * 0036-malloc-Remove-malloc-hooks-from-fork-handler.patch
  * 0037-malloc-Add-missing-internal_function-attributes-on-f.patch
  * 0038-nss_dns-Fix-assertion-failure-in-_nss_dns_getcanonna.patch
  * 0039-nss_dns-Validate-RDATA-length-against-packet-length-.patch
  * 0040-resolv-nss_dns-Remove-remaining-syslog-logging-BZ-19.patch
  * 0041-nss_dns-Check-address-length-before-creating-addrinf.patch
  * 0042-nss_dns-Skip-over-non-PTR-records-in-the-netent-code.patch
  * 0043-resolv-Always-set-resplen2-out-parameter-in-send_vc-.patch
  * 0044-tst-audit4-tst-audit10-Compile-AVX-AVX-512-code-sepa.patch
  * 0045-Fix-tst-audit10-build-when-mavx512f-is-not-supported.patch
  * 0046-tst-audit10-Fix-compilation-on-compilers-without-bit.patch
  * 0047-strfmon_l-Use-specified-locale-for-number-formatting.patch
  * 0048-glob-Simplify-the-interface-for-the-GLOB_ALTDIRFUNC-.patch
  * 0049-CVE-2016-1234-glob-Do-not-copy-d_name-field-of-struc.patch
  * 0050-ldconfig-Do-not-remove-stale-symbolic-links-with-X-B.patch
  * 0051-Report-dlsym-dlvsym-lookup-errors-using-dlerror-BZ-1.patch
  * 0052-Fix-tst-dlsym-error-build.patch
  * 0053-Remove-trailing-newline-from-date_fmt-in-Serbian-loc.patch
  * 0054-Revert-Report-dlsym-dlvsym-lookup-errors-using-dlerr.patch
  * 0055-CVE-2016-3706-getaddrinfo-stack-overflow-in-hostent-.patch
  * 0056-Fix-strfmon_l-Use-specified-locale-for-number-format.patch
  * clntudp-call-alloca.patch
  * glibc-memset-nontemporal.diff
  * nis-initgroups-status.patch
  * nscd-gc-crash.patch
  * robust-mutex-deadlock.patch
  * strncat-avoid-array-bounds-warning.patch
* Wed Jun 29 2016 schwab@suse.de
- strncat-avoid-array-bounds-warning.patch: Avoid array-bounds warning for
  stncat on i586 (BZ #20260)
- Update glibc.keyring
- Unset MALLOC_CHECK_ during testsuite run
* Thu Jun 23 2016 schwab@suse.de
- nsswitch.conf: Add fallback to files for passwd and group to prepare for
  libnsl removal.
* Wed Jun 22 2016 schwab@suse.de
- nis-initgroups-status.patch: Return proper status from
  _nss_nis_initgroups_dyn (bsc#984269, BZ #20262)
- robust-mutex-deadlock.patch: Fix generic __lll_robust_timedlock_wait to
  check for timeout (bsc#985170, BZ #20263)
* Thu Jun  9 2016 schwab@suse.de
- nscd-gc-crash.patch: Fix nscd assertion failure in gc (bsc#965699, BZ
  [#19755])
* Mon May 30 2016 schwab@suse.de
- clntudp-call-alloca.patch: do not use alloca in clntudp_call
  (CVE-2016-4429, bsc#980854, BZ #20112)
* Wed May 11 2016 schwab@suse.de
- Import patches from 2.23 branch
  0001-Updated-translations-for-2.23.patch
  0002-Regenerate-libc.pot-for-2.23.patch
  0003-Regenerated-configure-scripts.patch
  0004-x86_64-Set-DL_RUNTIME_UNALIGNED_VEC_SIZE-to-8.patch
  0005-Add-fts64_-to-sysdeps-arm-nacl-libc.abilist.patch
  0006-Don-t-use-long-double-math-functions-if-NO_LONG_DOUB.patch
  0007-NEWS-2.23-Fix-typo-in-bug-19048-text.patch
  0008-Update-NEWS.patch
  0009-sln-use-stat64.patch
  0010-Add-sys-auxv.h-wrapper-to-include-sys.patch
  0011-mips-terminate-the-FDE-before-the-return-trampoline-.patch
  0012-Use-HAS_ARCH_FEATURE-with-Fast_Rep_String.patch
  0013-Mention-BZ-19762-in-NEWS.patch
  0014-Define-_HAVE_STRING_ARCH_mempcpy-to-1-for-x86.patch
  0015-Or-bit_Prefer_MAP_32BIT_EXEC-in-EXTRA_LD_ENVVARS.patch
  0016-Fix-resource-leak-in-resolver-bug-19257.patch
  0017-math-don-t-clobber-old-libm.so-on-install-BZ-19822.patch
  0018-resolv-Always-set-resplen2-out-parameter-in-send_dg-.patch
  0019-S390-Save-and-restore-fprs-vrs-while-resolving-symbo.patch
  0020-S390-Extend-structs-La_s390_regs-La_s390_retval-with.patch
  0021-CVE-2016-3075-Stack-overflow-in-_nss_dns_getnetbynam.patch
  0022-configure-fix-test-usage.patch
  0023-Suppress-GCC-6-warning-about-ambiguous-else-with-Wpa.patch
  0024-nss_db-Propagate-ERANGE-error-if-parse_line-fails-BZ.patch
  0025-getnameinfo-Do-not-preserve-errno.patch
  0026-getnameinfo-Refactor-and-fix-memory-leak-BZ-19642.patch
  0027-getnameinfo-Reduce-line-length-and-add-missing-comme.patch
  0028-getnameinfo-Avoid-calling-strnlen-on-uninitialized-b.patch
  0029-getnameinfo-Return-EAI_OVERFLOW-in-more-cases-BZ-197.patch
  0030-hesiod-Remove-RCS-keywords.patch
  0031-hesiod-Always-use-thread-local-resolver-state-BZ-195.patch
  0032-hesiod-Avoid-heap-overflow-in-get_txt_records-BZ-200.patch
  0033-malloc-Remove-NO_THREADS.patch
  0034-Fix-malloc-threaded-tests-link-on-non-Linux.patch
  0035-malloc-Run-fork-handler-as-late-as-possible-BZ-19431.patch
  0036-malloc-Remove-malloc-hooks-from-fork-handler.patch
  0037-malloc-Add-missing-internal_function-attributes-on-f.patch
  0038-nss_dns-Fix-assertion-failure-in-_nss_dns_getcanonna.patch
  0039-nss_dns-Validate-RDATA-length-against-packet-length-.patch
  0040-resolv-nss_dns-Remove-remaining-syslog-logging-BZ-19.patch
  0041-nss_dns-Check-address-length-before-creating-addrinf.patch
  0042-nss_dns-Skip-over-non-PTR-records-in-the-netent-code.patch
  0043-resolv-Always-set-resplen2-out-parameter-in-send_vc-.patch
  0044-tst-audit4-tst-audit10-Compile-AVX-AVX-512-code-sepa.patch
  0045-Fix-tst-audit10-build-when-mavx512f-is-not-supported.patch
  0046-tst-audit10-Fix-compilation-on-compilers-without-bit.patch
  0047-strfmon_l-Use-specified-locale-for-number-formatting.patch
  0048-glob-Simplify-the-interface-for-the-GLOB_ALTDIRFUNC-.patch
  0049-CVE-2016-1234-glob-Do-not-copy-d_name-field-of-struc.patch
  0050-ldconfig-Do-not-remove-stale-symbolic-links-with-X-B.patch
  0051-Report-dlsym-dlvsym-lookup-errors-using-dlerror-BZ-1.patch
  0052-Fix-tst-dlsym-error-build.patch
  0053-Remove-trailing-newline-from-date_fmt-in-Serbian-loc.patch
  0054-Revert-Report-dlsym-dlvsym-lookup-errors-using-dlerr.patch
  0055-CVE-2016-3706-getaddrinfo-stack-overflow-in-hostent-.patch
  0056-Fix-strfmon_l-Use-specified-locale-for-number-format.patch
- CVE-2016-3075 CVE-2016-1234 CVE-2016-3706 bsc#973164 bsc#969727
- resolv-mem-leak.patch: renamed to
  0016-Fix-resource-leak-in-resolver-bug-19257.patch
- no-long-double.patch: renamed to
  0006-Don-t-use-long-double-math-functions-if-NO_LONG_DOUB.patch
- glibc-gcc6.patch: renamed to
  0023-Suppress-GCC-6-warning-about-ambiguous-else-with-Wpa.patch
* Mon May  9 2016 schwab@suse.de
- glibc-c-utf8-locale.patch: fix bad standard in LC_IDENTIFICATION categories
- glibc-2.3.locales.diff.bz2: likewise
* Mon Apr 25 2016 jslaby@suse.com
- glibc-gcc6.patch: Suppress GCC 6 warning about ambiguous 'else' with
  - Wparentheses
* Mon Apr 11 2016 schwab@suse.de
- Add compatibility symlinks for LSB 3.0 (fate#318933)
* Mon Mar 21 2016 schwab@suse.de
- powerpc-elision-enable-envvar.patch: enable TLE only if
  GLIBC_ELISION_ENABLE=yes is defined (bsc#967594, fate#318236)
* Tue Mar 15 2016 schwab@suse.de
- ldd-system-interp.patch: Restore warning about execution permission, it
  is still needed for noexec mounts (bsc#915985)
* Tue Mar  8 2016 idonmez@suse.com
- Add C.UTF-8 locale (see https://sourceware.org/glibc/wiki/Proposals/C.UTF-8)
  and rh#902094 . Added with glibc-c-utf8-locale.patch.
- Add glibc-disable-gettext-for-c-utf8.patch to disable gettext for
  C.UTF-8 same as C locale.
* Mon Mar  7 2016 schwab@suse.de
- Move %%install_info_delete to %%preun
- crypt_blowfish-1.3.tar.gz.sign: Remove, the sign key is no longer
  acceptable
* Tue Feb 23 2016 schwab@suse.de
- no-long-double.patch: Don't use long double functions if NO_LONG_DOUBLE
* Mon Feb 22 2016 schwab@suse.de
- Update to glibc 2.23 release.
  * Unicode 8.0.0 Support
  * sched_setaffinity, pthread_setaffinity_np no longer attempt to guess
    the kernel-internal CPU set size
  * The fts.h header can now be used with -D_FILE_OFFSET_BITS=64
  * getaddrinfo now detects certain invalid responses on an internal
    netlink socket
  * A defect in the malloc implementation, present since glibc 2.15 (2012)
    or glibc 2.10 via --enable-experimental-malloc (2009), could result in
    the unnecessary serialization of memory allocation requests across
    threads
  * The obsolete header <regexp.h> has been removed
  * The obsolete functions bdflush, create_module, get_kernel_syms,
    query_module and uselib are no longer available to newly linked
    binaries
  * Optimized string, wcsmbs and memory functions for IBM z13.
  * Newly linked programs that define a variable called signgam will no
    longer have it set by the lgamma, lgammaf and lgammal functions
- Removed patches:
  * dont-remove-nodelete-flag.patch
  * openat64-readd-o-largefile.patch
  * mntent-blank-line.patch
  * opendir-o-directory-check.patch
  * strcoll-remove-strdiff-opt.patch
  * ld-pointer-guard.patch
  * tls-dtor-list-mangling.patch
  * powerpc-lock-elision-race.patch
  * prelink-elf-rtype-class.patch
  * vector-finite-math-aliases.patch
  * powerpc-elision-adapt-param.patch
  * catopen-unbound-alloca.patch
  * strftime-range-check.patch
  * hcreate-overflow-check.patch
  * errorcheck-mutex-no-elision.patch
  * refactor-nan-parsing.patch
  * send-dg-buffer-overflow.patch
  * isinf-cxx11-conflict.patch
  * ibm93x-redundant-shift-si.patch
  * iconv-reset-input-buffer.patch
  * tzset-tzname.patch
  * static-dlopen.patch
* Thu Feb 18 2016 schwab@suse.de
- isinf-cxx11-conflict.patch: Fix isinf/isnan declaration conflict with
  C++11 (bsc#963700, BZ #19439)
* Wed Feb 17 2016 schwab@suse.de
- tls-dtor-list-mangling.patch: Harden tls_dtor_list with pointer mangling
  (BZ #19018)
- prelink-elf-rtype-class.patch: Keep only ELF_RTYPE_CLASS_{PLT|COPY} bits
  for prelink (BZ #19178)
- vector-finite-math-aliases.patch: Better workaround for aliases of
  * _finite symbols in vector math library (BZ# 19058)
- powerpc-elision-adapt-param.patch: powerpc: Fix usage of elision
  transient failure adapt param (BZ #19174)
- catopen-unbound-alloca.patch: Fix unbound alloca in catopen
  (CVE-2015-8779, bsc#962739, BZ #17905)
- strftime-range-check.patch: Add range check on time fields
  (CVE-2015-8776, bsc#962736, BZ #18985)
- hcreate-overflow-check.patch: Handle overflow in hcreate (CVE-2015-8778,
  bsc#962737, BZ #18240)
- errorcheck-mutex-no-elision.patch: Don't do lock elision on an error
  checking mutex (bsc#956716, BZ #17514)
- refactor-nan-parsing.patch: Refactor strtod parsing of NaN payloads
  (CVE-2014-9761, bsc#962738, BZ #16962)
- send-dg-buffer-overflow.patch: Fix getaddrinfo stack-based buffer
  overflow (CVE-2015-7547, bsc#961721, BZ #18665)
- powerpc-lock-elision-race.patch: renamed from
  0001-powerpc-Fix-a-race-condition-when-eliding-a-lock-20150730.patch
* Mon Feb 15 2016 schwab@suse.de
- Add audit-devel and libcap-devel to BuildRequires, for use by nscd
* Thu Nov 26 2015 schwab@suse.de
- reinitialize-dl_load_write_lock.patch: Reinitialize dl_load_write_lock
  on fork (bsc#958315, BZ #19282)
* Wed Nov 18 2015 schwab@suse.de
- resolv-mem-leak.patch: Fix resource leak in resolver (bsc#955647, BZ #19257)
- tzset-tzname.patch: Force rereading TZDEFRULES after it was used to set
  DST rules only (BZ #19253)
* Mon Oct 26 2015 schwab@suse.de
- glibc-2.3.90-noversion.diff: use stat64
* Mon Oct 19 2015 schwab@suse.de
- ld-pointer-guard.patch: Always enable pointer guard (CVE-2015-8777,
  bsc#950944, BZ #18928)
* Mon Oct 12 2015 schwab@suse.de
- strcoll-remove-strdiff-opt.patch: Remove incorrect STRDIFF-based
  optimization (BZ #18589)
* Thu Oct  8 2015 schwab@suse.de
- iconv-reset-input-buffer.patch: Fix iconv buffer handling with IGNORE
  error handler (BZ #18830)
* Wed Sep 30 2015 normand@linux.vnet.ibm.com
- new patch [BZ #18743] PowerPC: Fix a race condition when eliding a lock
  0001-powerpc-Fix-a-race-condition-when-eliding-a-lock-20150730.patch
* Thu Sep 24 2015 schwab@suse.de
- nss-files-long-lines-2.patch: Properly reread entry after failure in
  nss_files getent function (bsc#945779, BZ #18991)
* Wed Sep 16 2015 schwab@suse.de
- fnmatch-collating-elements.patch: Fix fnmatch handling of collating
  elements (BZ #17396, BZ #16976)
* Mon Sep  7 2015 schwab@suse.de
- opendir-o-directory-check.patch: Fix opendir inverted o_directory_works
  test
- static-dlopen.patch: Static dlopen default library search path fix
  (bsc#937853)
* Mon Aug 31 2015 schwab@suse.de
- mntent-blank-line.patch: Fix memory corruption w/blank lines
* Mon Aug 10 2015 schwab@suse.de
- dont-remove-nodelete-flag.patch: Don't remove DF_1_NODELETE flag from
  all loaded objects on failed dlopen
- openat64-readd-o-largefile.patch: Readd O_LARGEFILE flag for openat64
* Wed Aug  5 2015 schwab@suse.de
- Update to glibc 2.22 release.
  * Cache information can be queried via sysconf() function on s390
  * A buffer overflow in gethostbyname_r and related functions performing DNS
    requests has been fixed.  (CVE-2015-1781)
  * The time zone file parser has been made more robust against crafted time
    zone files
  * A powerpc and powerpc64 optimization for TLS, similar to TLS descriptors
    for LD and GD on x86 and x86-64, has been implemented.
  * Character encoding and ctype tables were updated to Unicode 7.0.0
  * Added vector math library named libmvec
  * A new fmemopen implementation has been added with the goal of POSIX
    compliance.
  * The header <regexp.h> is deprecated, and will be removed in a future
    release.
  * bsc#905313 bsc#920338 bsc#927080 bsc#928723 bsc#931480 bsc#939211
    bsc#940195 bsc#940332 bsc#944494 bsc#968787
- Patches from upstream removed
  * htm-tabort.patch
  * o-tmpfile.patch
  * memcpy-chk-non-SSE2.patch
  * pthread-mutexattr-gettype-kind.patch
  * powerpc-software-sqrt.patch
  * static-tls-dtv-limit.patch
  * threaded-trim-threshold.patch
  * resolv-nameserver-handling.patch
  * nss-separate-state-getXXent.patch
  * aarch64-sigstksz.patch
  * heap-top-corruption.patch
  * pthread-join-deadlock.patch
* Mon Jul 27 2015 schwab@suse.de
- pthread-join-deadlock.patch: Use IE model for static variables in
  libc.so, libpthread.so and rtld (bsc#930015, BZ #18457)
* Wed Jul 22 2015 schwab@suse.de
- glibc-nodate.patch: fix verification of timestamp
* Tue Jul 21 2015 meissner@suse.com
- also filter out -fstack-protector-strong
* Mon Jun 22 2015 schwab@suse.de
- getaddrinfo-ipv6-sanity.diff: Remove.  It breaks services that start
  before IPv6 is up (bsc#931399)
- glibc-2.3.locales.diff.bz2: Remove sh_YU locales, fix currency for en_BE.
* Tue Jun  9 2015 schwab@suse.de
- Add /usr/include/gnu/lib-names-.*.h to baselibs
- pthread-join-deadlock.patch: Don't require rtld lock to store static TLS
  offset in the DTV (bsc#930015, BZ #18457)
- heap-top-corruption.patch: Do not corrupt the top of a threaded heap if
  top chunk is MINSIZE (BZ #18502)
* Wed Apr  8 2015 mgorman@suse.com
- threaded-trim-threshold.patch: Fix regression in threaded application
  malloc performance (bsc#915955, BZ #17195)
* Thu Apr  2 2015 schwab@suse.de
- aarch64-sigstksz.patch: Increase MINSIGSTKSZ and SIGSTKSZ (BZ #16850)
* Mon Mar 30 2015 schwab@suse.de
- powerpc-software-sqrt.patch: Fix powerpc software sqrt (BZ #17964, BZ
  [#17967])
- nss-separate-state-getXXent.patch: Separate internal state between
  getXXent and getXXbyYY NSS calls (CVE-2014-8121, bsc#918187, BZ #18007)
- static-tls-dtv-limit.patch: Fix DTV race, assert, DTV_SURPLUS Static TLS
  limit, and nptl_db garbage (bsc#919678, BZ #17090, BZ #17620, BZ #17621,
  BZ #17628)
* Mon Mar  9 2015 schwab@suse.de
- resolv-nameserver-handling.patch: Replace with simpler version with more
  compatibility
* Mon Mar  2 2015 schwab@suse.de
- memcpy-chk-non-SSE2.patch: Fix __memcpy_chk on non-SSE2 CPUs
  (bsc#920084)
* Tue Feb 24 2015 schwab@suse.de
- resolv-nameserver-handling.patch: Rewrite handling of nameserver
  configuration in resolver
* Thu Feb 12 2015 schwab@suse.de
- htm-tabort.patch: Fix TABORT encoding for little endian
* Mon Feb  9 2015 schwab@suse.de
- Update to glibc 2.21 release.
  * A new semaphore algorithm has been implemented in generic C code for all
    machines
  * Added support for TSX lock elision of pthread mutexes on powerpc32,
    powerpc64 and powerpc64le
  * Optimized strcpy, stpcpy, strchrnul and strrchr implementations for
    AArch64
  * i386 memcpy functions optimized with SSE2 unaligned load/store
  * New locales: tu_IN, bh_IN, raj_IN, ce_RU
  * The obsolete sigvec function has been removed
  * CVE-2015-1472 CVE-2015-1473 CVE-2012-3406 CVE-2014-9402
    CVE-2014-7817 bsc#864081 bsc#906371 bsc#909053 bsc#910599 bsc#916222
- Patches from upstream removed
  * ifunc-x86-slow-sse4.patch
  * pthread-mutex-trylock-elision.patch
- o-tmpfile.patch: Fix value of O_TMPFILE for architectures with
  non-default O_DIRECTORY (BZ #17912)
* Tue Oct 28 2014 schwab@suse.de
- Update to crypt_blowfish 1.3.
  * Add support for the $2b$ prefix.
- ifunc-x86-slow-sse4.patch: Fix misdetected Slow_SSE4_2 cpu feature bit
  (BZ #17501)
* Mon Sep  8 2014 schwab@suse.de
- Update to glibc 2.20 release.
  * Reverted change of ABI data structures for s390 and s390x
  * Support for file description locks is added to systems running the
    Linux kernel
  * Optimized strchr implementation for AArch64
  * The minimum Linux kernel version that this version of the GNU C Library
    can be used with is 2.6.32
  * Running the testsuite no longer terminates as soon as a test fails
  * The am33 port, which had not worked for several years, has been removed
    from ports.
  * The _BSD_SOURCE and _SVID_SOURCE feature test macros are no longer
    supported; they now act the same as _DEFAULT_SOURCE (but generate a
    warning)
  * Optimized strcmp implementation for ARMv7
  * Added support for TX lock elision of pthread mutexes on s390 and s390x
  * All supported architectures now use the main glibc sysdeps directory
    instead of some being in a separate "ports" directory
  * The NPTL implementation of POSIX pthreads is no longer an "add-on"
  * Locale names, including those obtained from environment variables (LANG
    and the LC_* variables), are more tightly checked for proper syntax
  * On x86-64, the dynamic linker's lazy-binding support is now compatible
    with application code using Intel MPX instructions
  * CVE-2013-7423 bsc#915526 bsc#934084 bsc#973010
- Patches from upstream removed
  * nss-dns-memleak.patch
  * sin-sign.patch
  * pldd-wait-ptrace-stop.patch
  * nscd-track-startup-failures.patch
  * powerpc-opt-power8.patch
  * check-pf-alloca.patch
  * getaddrinfo-uninit-result.patch
  * ppc64-copysign.patch
  * nextafterl-ibm-ldouble.patch
  * ibm-long-double-math.patch
  * ibm-long-double-frexpl.patch
  * aarch64-setcontext.patch
  * nss-nis-stack-use.patch
  * psfaa.patch
  * nss-files-long-lines.patch
  * setlocale-directory-traversal.patch
  * dt-ppc64-num.patch
  * ppc64le-profiling.patch
  * s390-revert-abi-change.patch
  * disable-gconv-translit-modules.patch
  * resolv-dont-ignore-second-answer.patch
  * iconv-ibm-sentinel-check.patch
- Remove sprof.1, and mtrace.1, now included in the man-pages package
  (bsc#967190)
* Tue Sep  2 2014 schwab@suse.de
- iconv-ibm-sentinel-check.patch: Fix crashes on invalid input in IBM
  gconv modules (CVE-2014-6040, bnc#894553, BZ #17325)
* Tue Aug 26 2014 schwab@suse.de
- disable-gconv-translit-modules.patch: Disable gconv transliteration
  module loading (CVE-2014-5119, bnc#892073, BZ #17187)
* Mon Aug 11 2014 schwab@suse.de
- pthread-mutexattr-gettype-kind.patch: Filter out
  PTHREAD_MUTEX_NO_ELISION_NP bit in pthread_mutexattr_gettype
  (bnc#888131, BZ #15790)
* Thu Jul 31 2014 schwab@suse.de
- ppc64le-profiling.patch: fix profiling for ppc64le (bnc#889601, BZ
  [#17213])
- s390-revert-abi-change.patch: replace with upstream patch, switches
  default symbol version back to old version (bnc#887228)
* Mon Jul 28 2014 schwab@suse.de
- ibm93x-redundant-shift-si.patch: Avoid redundant shift character in
  iconv output at block boundary (bnc#886416, BZ #17197)
* Thu Jul 24 2014 schwab@suse.de
- s390-revert-abi-change.patch: revert back to pre-2.19 layout of jmp_buf
  and ucontext_t on s390 (bnc#887228)
* Tue Jul 22 2014 schwab@suse.de
- dt-ppc64-num.patch: Correct DT_PPC64_NUM (bnc#887169, BZ #17153)
* Mon Jul 14 2014 schwab@suse.de
- setlocale-directory-traversal.patch: Directory traversal in locale
  environment handling (CVE-2014-0475, bnc#887022, BZ #17137)
- Remove the ppc64le check-abi workaround, no longer needed
* Mon Jun 23 2014 schwab@suse.de
- nss-files-long-lines.patch: Don't ignore too long lines in nss_files
  (bnc#883217, BZ #17079)
- Remove locale.1, localedef.1 and iconvconfig.8 manpages, now included in
  the man-pages package (bnc#880703)
- manpages.patch: update
* Mon Jun 16 2014 schwab@suse.de
- psfaa.patch: copy filename argument in posix_spawn_file_actions_addopen
  (CVE-2014-4043, bnc#882600, BZ #17048)
* Wed May 28 2014 matz@suse.de
- glibc-memset-nontemporal.diff: Speedup memset on x86_64
  (bnc#868622, BZ #16830)
* Thu May 15 2014 schwab@suse.de
- add non-elision-enabled libpthread
* Mon May 12 2014 schwab@suse.de
- nss-nis-stack-use.patch: fix unbound stack use in NIS NSS module (BZ
  [#16932])
* Wed May  7 2014 schwab@suse.de
- getaddrinfo-uninit-result.patch: fix parsing of getai result for
  IPv6-only request (bnc#876521)
* Thu Apr 17 2014 schwab@suse.de
- ibm-long-double-frexpl.patch: Correct IBM long double frexpl
  (bnc#872573, BZ #16740, BZ #16619)
* Tue Apr 15 2014 schwab@suse.de
- ibm-long-double-math.patch: Remove faulty assembler implementations of
  ceil, nearbyintl, roundl for IBM long double (bnc#873457, BZ #16701, BZ
  [#16706], BZ #16707)
* Tue Apr 15 2014 aj@suse.com
- Use /run for nscd instead of /var/run.
* Mon Apr  7 2014 schwab@suse.de
- pthread-mutex-destroy.patch: Remove
- pthread-mutex-trylock-elision.patch: Properly handle forced elision in
  pthread_mutex_trylock (bnc#865968, BZ #16657)
* Thu Apr  3 2014 schwab@suse.de
- nextafterl-ibm-ldouble.patch: Correct IBM long double nextafterl
  (bnc#871637, BZ #16739)
* Tue Apr  1 2014 schwab@suse.de
- Readd dependency on sysvinit.target to nscd.service (bnc#868422)
- Make the nscd service part of the nss-lookup and nss-user-lookup targets
  (bnc#871386)
- Move _constraints back to main package and update it
* Tue Apr  1 2014 schwab@suse.de
- ppc64-copysign.patch: Fix ppc64le copysign overwriting parent stack
  frame (BZ #16786)
- aarch64-setcontext.patch: Fix aarch64 setcontext clobbering alternate
  signal stack (BZ #16629)
* Mon Mar 31 2014 schwab@suse.de
- powerpc-opt-power8.patch: readd POWER8 implementation of isnan,
  isinf, finite with fix for little endian (bnc#871353)
- pthread-mutex-destroy.patch: Skip checks in pthread_mutex_destroy when
  doing elision (bnc#865968)
* Sat Mar 29 2014 schwab@suse.de
- powerpc-opt-power8.patch: remove broken POWER8 implementation of isnan,
  isinf, finite; fix missing static symbols
- glibc-2.14-crypt.diff: update for file renaming
* Thu Mar 27 2014 schwab@suse.de
- check-pf-alloca.patch: remove use of alloca while collecting interface
  addresses
* Tue Mar 25 2014 schwab@suse.de
- abort-no-flush.patch: Don't close or flush stdio streams on abort (BZ
  [#15436])
- fix-locking-in-_IO_cleanup.patch: always do locking when accessing
  streams (bnc#796982, BZ #15142)
- resolv-dont-ignore-second-answer.patch: don't ignore second answer from
  nameserver if the first one was empty (bnc#767266, BZ #13651)
- ldd-system-interp.patch: Never try to execute the file in ldd
  (CVE-2009-5064, bnc#677787, BZ #16750)
* Mon Mar 24 2014 schwab@suse.de
- check-pf-alloca.patch: Account for alloca use when collecting interface
  addresses (bnc#785041, BZ #16002)
* Mon Mar 24 2014 schwab@suse.de
- powerpc-opt-power8.patch: two more POWER8 optimisations (bnc#866711,
  fate#315443)
* Thu Mar 20 2014 schwab@suse.de
- getaddrinfo-uninit-result.patch: Fix use of half-initialized result in
  getaddrinfo when using nscd (bnc#867636, BZ #16743)
* Thu Mar 20 2014 schwab@suse.de
- Remove dependency on sysinit.target from nscd.service again (bnc#86422)
* Wed Mar 19 2014 schwab@suse.de
- powerpc-opt-power8.patch: GLIBC Optimizations For POWER8 Hardware
  (bnc#866711, fate#315443)
* Tue Mar 18 2014 schwab@suse.de
- Make sure nscd is started after sysinit.target (bnc#868422)
- nscd-track-startup-failures.patch: properly track startup failures in
  nscd and mark nscd.service as forking (BZ #16639)
* Thu Mar  6 2014 schwab@suse.de
- Add services and netgroup persistent caches to nscd file list
* Wed Feb 26 2014 schwab@suse.de
- sin-sign.patch: Fix sign of input to bsloww1 (BZ #16623)
- pldd-wait-ptrace-stop.patch: Update from upstream
* Mon Feb 24 2014 schwab@suse.de
- nss-dns-memleak.patch: fix memory leak in nss-dns (bnc#863499)
- Don't build glibc-{extra,info,i18ndata,obsolete} and nscd for i686
* Mon Feb 10 2014 schwab@suse.de
- Bump minimum kernel to 3.0
* Sun Feb  9 2014 schwab@suse.de
- Update to glibc 2.19 release
* Thu Feb  6 2014 schwab@suse.de
- Update to glibc 2.18.90 ee7cc3853761:
  * More bug fixes
  * SystemTap probes for slow multiple precision fallback paths of
    transcendental functions have been introduced.
  * Preliminary documentation about Multi-Thread, Async-Signal and
    Async-Cancel Safety has been added.
  * Change powerpc64le start ABI to 2.17
- Add /usr/local/lib64 to ld.so.conf on all lib64 platforms
- Enable profile build for aarch64
- ppc64le-abilist.patch: Removed
* Wed Jan  8 2014 coolo@suse.com
- make pwdutils a real buildrequire
  Requires(pre) are used during builds, but don't trigger rebuilds.
  But to create clearly defined bootstrap projects, it's essential
  to know what packages are required for others. So move this requires
  to build to a buildrequire
* Tue Jan  7 2014 schwab@suse.de
- Update to glibc 2.18.90 94c8a4bc574c:
  * More bug fixes
  * Slovenian translations for glibc messages have been contributed by the
    Translation Project's Slovenian team of translators.
* Thu Dec 19 2013 schwab@suse.de
- Update to glibc 2.18.90 83f5c32d21f7:
  * Support for STT_GNU_IFUNC symbols added for ppc64
  * A new feature test macro _DEFAULT_SOURCE is available
  * The _BSD_SOURCE feature test macro no longer enables BSD interfaces
    that conflict with POSIX.  The libbsd-compat library (which was a
    dummy library that did nothing) has also been removed.
  * Added support for version-3 tz format files
- i686-no-sse42-strstr.patch: Removed
* Tue Dec 10 2013 schwab@suse.de
- Update to glibc 2.18.90 aeb95445d128:
  * More bug fixes
  * Support for STT_GNU_IFUNC symbols added for ppc32/power4+
  * The public headers no longer use __unused nor __block
  * Support for PowerPC64 ELFv2 ABI
- ppc64le-abilist.patch: ABI lists for PowerLE
* Mon Nov 25 2013 schwab@suse.de
- Update to glibc 2.18.90 bd12ab55c05e:
  * New locales: ak_GH, anp_IN, ar_SS, cmn_TW, hak_TW, lzh_TW, nan_TW, pap_AW,
    pap_CW, quz_PE, the_NP.
  * Substantially revised locales: gd_GB, ht_HT
  * The LC_ADDRESS field was updated to support country_car for almost all
    supported locales.
  * ISO 1427 definitions were updated.
  * ISO 3166 definitions were updated.
  * The localedef utility now supports --big-endian and --little-endian
    command-line options to generate locales for a different system from that
    for which the C library was built.
  * Binary locale files now only depend on the endianness of the system for
    which they are generated and not on other properties of that system.
  * ISO 639 definitions were updated for Chiga (cgg) and Chinese (gan, hak, czh,
    cjy, lzh, cmn, mnp, cdo, czo, cpx, wuu, hsn, yue).
  * SystemTap probes for malloc have been introduced.
  * Support for powerpc64le has been added.
  * The soft-float powerpc port now supports e500 processors.
- Remove upstreamed patches:
  cbrtl-ldbl-96.patch
  fcntl-o-tmpfile.patch
  getaddrinfo-overflow.patch
  glibc-2.18.tar.xz
  glibc-ld-profile.patch
  glibc-nscd-hconf.diff
  i686-strcasecmp-fallback.patch
  m68k-Scrt1.patch
  malloc-overflows.patch
  nonascii-case.patch
  nss-database.patch
  printf-overrun.patch
  readdir_r-overflow.patch
  strcoll-overflow.patch
- Remove obsolete patches:
  glibc-2.4.90-revert-only-euro.diff
* Wed Oct 30 2013 schwab@suse.de
- getaddrinfo-overflow.patch: Fix stack overflow due to large AF_INET6
  requests (CVE-2013-4458, bnc#847227)
* Mon Oct 21 2013 schwab@suse.de
- Remove SSE4.2 implementation of strstr for x86-32 (bnc#843464)
* Mon Oct 14 2013 schwab@suse.de
- Depend on gcc defaults for base powerpc optimisations (bnc#844824)
* Fri Oct 11 2013 schwab@suse.de
- Enable ports addon for m68k and adapt file list
- m68k-Scrt1.patch: Use PIC for Scrt1.o
* Thu Sep 19 2013 schwab@suse.de
- nscd-server-user.patch: run nscd as user nscd (fate#314673)
* Thu Sep 12 2013 schwab@suse.de
- malloc-overflows.patch: Fix integer overflows in malloc (CVE-2013-4332,
  bnc#839870)
* Wed Sep 11 2013 schwab@suse.de
- fcntl-o-tmpfile.patch: Add O_TMPFILE to <fcntl.h> (bnc#837965)
- strcoll-overflow.patch: Update to latest version
* Tue Aug 13 2013 schwab@suse.de
- Update to glibc 2.18 release
  * No source change
- strcoll-overflow.patch: fix buffer overflow in strcoll (CVE-2012-4412,
  bnc#779320)
- readdir_r-overflow.patch: fix readdir_r with long file names
  (CVE-2013-4237, bnc#834594)
* Tue Aug  6 2013 schwab@suse.de
- Update to glibc 2.17.90 85891acadf1b:
  * CVE-2013-2207 Incorrectly granting access to another user's pseudo-terminal
    has been fixed by disabling the use of pt_chown (Bugzilla #15755).
    Distributions can re-enable building and using pt_chown via the new configure
    option `--enable-pt_chown'.  Enabling the use of pt_chown carries with it
    considerable security risks and should only be used if the distribution
    understands and accepts the risks.
  * CVE-2013-0242 Buffer overrun in regexp matcher has been fixed (Bugzilla
    [#15078]).
  * CVE-2013-1914 Stack overflow in getaddrinfo with many results has been
    fixed (Bugzilla #15330).
  * Add support for calling C++11 thread_local object destructors on thread
    and program exit.  This needs compiler support for offloading C++11
    destructor calls to glibc.
  * Improved worst case performance of libm functions with double inputs and
    output.
  * Support for priority inherited mutexes in pthread condition variables on
    non-x86 architectures.
  * Optimized string functions for AArch64.  Implemented by Marcus Shawcroft.
  * Optimized string functions for ARM.  Implemented by Will Newton and
    Richard Henderson.
  * Added a benchmark framework to track performance of functions in glibc.
  * New <math.h> macro named issignaling to check for a signaling NaN (sNaN).
    It is based on draft TS 18661 and currently enabled as a GNU extension.
  * On Linux, the clock function now uses the clock_gettime system call
    for improved precision, rather than old times system call.
  * Added new API functions pthread_getattr_default_np and
    pthread_setattr_default_np to get and set the default pthread
    attributes of a process.
  * Added support for TSX lock elision for pthread mutexes on i386 and x86-64.
    This may improve lock scaling of existing programs on TSX capable systems.
  * Support added for AT_HWCAP2 (to coincide with Linux kernel 3.10
    availability).  Implemented by Ryan S. Arnold.
  * Support added for POWER8 platform.  Implemented by Ryan S. Arnold.
- Remove upstreamed patches:
  pthread-cond-timedwait-i486.patch
  glibc-aarch64-lib64.patch
  arm-ld-so-cache.patch
  rtlddir.patch
  regexp-overrun.patch
  getaddrinfo-overflow.patch
  nscd-netgroup.patch
  disable-pt-chown.patch
  nscd-short-write.patch
  big5-hkscs-update.patch.bz2
  digits-dots.patch
  strptime-W-week.patch
  aarch64-sigsetjmp.patch
- Rediff all remaining patches
- Reenable stackguard randomization on arm
- Recognize ppc64p7 arch to build for power7
- Remove all ABI extensions from libcrypt
- cbrtl-ldbl-96.patch: fix cbrtl for ldbl-96
* Wed Jul 31 2013 schwab@suse.de
- i686-strcasecmp-fallback.patch: fix fallback in i686
  strcasecmp/strncasecmp
- nonascii-case.patch: also test strncasecmp
- pldd-wait-ptrace-stop.patch: fix pldd not to leave process stopped after
  detaching (bnc#819383)
* Mon Jul 22 2013 schwab@suse.de
- disable-pt-chown.patch: disable use of pt_chown (CVE-2013-2207,
  bnc#830257)
- nonascii-case.patch: fix strcasecmp in tr_TR.ISO-8859-9 locale
- glibc-2.14-crypt.diff: fix incomplete dependencies for libowcrypt
* Thu Jul  4 2013 schwab@suse.de
- crypt_blowfish-gensalt.patch: separare crypt_gensalt into its own
  library libowcrypt, remove crypt_rn and crypt_ra (fate#314945)
- glibc-2.14-crypt-versioning.diff, glibc-fix-check-abi.patch,
  glibc-crypt-badsalttest.patch: folded into glibc-2.14-crypt.diff
- crypt_blowfish-const.patch: constify crypt_blowfish
- crypt_blowfish-1.2-sha.diff: refresh
* Fri Jun 28 2013 dmueller@suse.com
- add ld-linux.so.3 compat symlink also for armv6hl
- remove deprecated armv7l
* Wed Jun 19 2013 coolo@suse.com
- rely on systemd-rpm-macros instead of the full thing
* Mon Jun 17 2013 schwab@suse.de
- nscd-netgroup.patch: Fix handling of netgroup cache in nscd (bnc#819524)
* Mon Jun 10 2013 schwab@suse.de
- glibc-bindresvport-blacklist.diff: Renamed from
  glibc-2.3.90-bindresvport.blacklist.diff; fix resource leaks
  (bnc#824046)
- Remove glibc-armhf-compat.patch
* Mon May 20 2013 schwab@suse.de
- aarch64-sigsetjmp.patch: Fix sigsetjmp on aarch64
* Sat May 18 2013 schwab@suse.de
- Fix file list for glibc-utils
* Thu May 16 2013 schwab@suse.de
- big5-hkscs-update.patch.bz2: Update BIG5-HKSCS charmap to HKSCS-2008
- digits-dots.patch: Fix parsing of numeric hosts in gethostbyname_r
- nss-database.patch: Properly cache the result from looking up the nss
  database config
- strptime-W-week.patch: Properly handle %%W in strptime
- printf-overrun.patch: Revert problematic fixes for [BZ #11741]
- glibc-aarch64-lib64.patch: Replace with upstream version
- arm-ld-so-cache.patch: ARM: Support loading unmarked objects from cache
- rtlddir.patch: Add support for rtld directory different from slib
  directory
- regexp-overrun.patch: Fix buffer overrun in regexp matcher
  (CVE-2013-0242, bnc#801246)
- getaddrinfo-overflow.patch: Fix stack overflow in getaddrinfo with many
  results (CVE-2013-1914, bnc#813121)
* Thu May 16 2013 schwab@suse.de
- When testsuite run fails make all test output available
* Tue May  7 2013 schwab@suse.de
- Fix check for XEN build in glibc_post_upgrade
* Sun Apr 14 2013 crrodriguez@opensuse.org
- Do not order nscd after syslog.target
* Mon Mar  4 2013 schwab@suse.de
- Fix crypt ABI for aarch64
* Thu Feb 28 2013 schwab@suse.de
- manpages.patch:
  Remove debianisms from manpages (bnc#805054)
* Mon Feb 18 2013 schwab@suse.de
- Update crypt ABI for aarch64
- glibc-aarch64-lib64.patch:
  Use lib64 for aarch64 and add link for /lib/ld-linux-aarch64.so.1
- Don't build profiling libraries for aarch64
* Tue Feb  5 2013 schwab@suse.de
- Move _constraints file to glibc-testsuite only
* Sun Feb  3 2013 dmueller@suse.com
- build with ports enabled on aarch64
* Thu Jan 31 2013 schwab@suse.de
- Obsolete unscd to migrate back to nscd.
* Thu Jan 31 2013 coolo@suse.com
- looks like testsuite runs only on kvm, so add _constraint
  (actually it does not build on xen, but I can't constraint that)
* Thu Jan 24 2013 schwab@suse.de
- Don't run testsuite in parallel
- Fail if testsuite fails
- Disable crypt/badsalttest test
* Tue Jan  8 2013 coolo@suse.com
- add _constraints file to make sure glibc-testsuite only builds on big
  hosts. Otherwise it's running out of memory and stalls
* Thu Dec 27 2012 aj@suse.de
- Set bugurl to bugs.opensuse.org
* Tue Dec 25 2012 aj@suse.de
- Update to glibc 2.17 release (git id c758a6861537):
  * bump version number
  * update translations
* Tue Dec 18 2012 schwab@suse.de
- Move glibc-utils subpackage to own specfile and enable programs that
  require libgd
- Filter GLIBC_PRIVATE symbols
- Don't stop at first error in testsuite run
* Mon Dec 17 2012 schwab@suse.de
- Split off glibc-testsuite package
* Fri Dec 14 2012 aj@suse.de
- Build profile and locale packages for i686 as well.
* Wed Dec 12 2012 aj@suse.de
- Update to 4641d57e1e00:
  * Updated translations
  * bug fixes
* Mon Dec 10 2012 schwab@suse.de
- pthread-cond-timedwait-i486.patch:
  Extend i486 pthread_cond_timedwait to use futex syscall with absolute
  timeout
- nscd-short-write.patch:
  Properly check for short writes when sending the response in nscd
* Sat Dec  8 2012 aj@suse.de
- Update to 56e7d3ad5c2f:
  * Bugfixes
  * Fix warnings building glibc
- Remove upstreamed patch glibc-revert-fseek-on-fclose.diff
* Mon Dec  3 2012 aj@suse.de
- Remove nosegneg i686 library, it's only used for a 32-bit Xen
  hypervisor (not for a 32-bit guest running under 64-bit hypervisor),
  and since the 32-bit Xen hypervisor is not part of openSUSE anymore,
  it is unneeded (bnc#789607).
* Mon Dec  3 2012 schwab@suse.de
- Suppress error message from systemctl in %%post -n nscd (bnc#792333)
* Sat Dec  1 2012 aj@suse.de
- Update to f638872ab422:
  Fix regression introduced with last update in sunrpc code
  Fix assertion failure in resolver
  Bug fixes
- Remove upstreamed patch glibc-resolv-assert.diff
* Fri Nov 30 2012 aj@suse.de
- Update to c93ec1f091ec:
  Warn about unsupported DT_FLAGS_1 flags
  Add Interlingua locale
  Bug fixes
* Thu Nov 29 2012 schwab@suse.de
- Remove obsolete patches:
  glibc-strict-aliasing.diff
  nscd-avoid-gcc-warning.diff
* Wed Nov 28 2012 aj@suse.de
- Update to e3c6aa3a5848:
  * Fix powerpc64 make abi check failure
  * bugfixes
- Install nscd.socket to new /usr location
* Wed Nov 28 2012 schwab@suse.de
- Reenable multi-arch on ppc ppc64
* Tue Nov 27 2012 aj@suse.de
- Update to de2fd463b1c0:
  Various bugfixes including:
  * Fix fwrite bug causing grep etc testsuite failure
  * sotruss: fix argument parsing
  * Check supported DF_1_XXX bits
  Add new locales niu_NU and niu_NZ
- Refresh patch glibc-revert-fseek-on-fclose.diff
* Sat Nov 24 2012 aj@suse.de
- Remove nscd init script, use systemd files.
* Fri Nov 23 2012 matz@suse.com
- Fix getaddrinfo-ipv6-sanity.diff to apply again.
* Tue Nov 20 2012 aj@suse.de
- Update to f59cba71d848:
  * Various bug fixes
  * Add new locales szl_PL, ayc_PE, nhn_MX, doi_IN, mni_IN, sat_IN.
  * Remove patch glibc-compiled-binaries.diff since getconf can not be
    cross-compiled
  * Refresh patch glibc-2.3.90-bindresvport.blacklist.diff
* Sun Nov 11 2012 aj@suse.de
- Update to current glibc 2.17 development snapshot (git 7e2bd01fcf3c):
  * Upstream changes:
    + ports is now part of main tarball
    + Add port to ARM AArch64
    + New function secure_getenv
    + Systemtap static probes
    + The `clock_*' suite of functions (declared in <time.h>) is now
    available directly in the main C library.
    + The `crypt' function now fails if passed salt bytes that violate the
    specification for those values.  On Linux, the `crypt' function will
    consult /proc/sys/crypto/fips_enabled to determine if "FIPS mode" is
    enabled, and fail on encrypted strings using the MD5 or DES algorithm
    when the mode is enabled.
    + Optimzations and bug fixes
  * remove upstream patches malloc-check-size.patch,
    glibc-2.16-getaddrinfo-buffer.patch, sparc-bind-now.patch,
    pthread-cond-wait-revert.patch
  * Remove now obsolete patch glibc-gai-private4.diff
  * Refresh patches glibc-2.14-crypt.diff, glibc-2.14-crypt-versioning.diff,
    glibc-fix-check-abi.patch, nss-db-path.patch, glibc-2.3.90-noversion.diff,
    glibc-revert-fseek-on-fclose.diff
  * Disable patches glibc-compiled-binaries.diff, getaddrinfo-ipv6-sanity.diff
    and glibc-gai-private4.diff.
  * Use git id instead of ChangeLog date in version output (patch glibc-version.diff)
  * Enable multi-arch on all architectures that support it.
* Wed Nov  7 2012 schwab@suse.de
- Avoid shell comment leaking into %%post.
* Thu Oct 25 2012 aj@suse.de
- Update to glibc 2.16 c3b96f90c9c1:
  - Fix missing _mcount@GLIBC_2.0 on powerpc32 (replaces
    patch glibc-2.16-ppc32-mcount.patch)
- Drop patch glibc-elf-localscope.diff, fixed upstream in 2.16
  in a different way.
* Tue Oct  9 2012 aj@suse.de
- Fix malloc_usable_size (swo#1349, bnc#745247) (patch
  malloc-check-size.patch).
- Update to current glibc 2.16 (git a5cfcf08ffaa):
  * fix fma detection
  * Don't parse %%s format argument as multibyte string
  * Use size_t instead of int for internal variables in glob
  * Properly handle fencepost with MALLOC_ALIGN_MASK
* Sat Oct  6 2012 meissner@suse.com
- make glibc-devel-<baselib> require glibc-<baselib> just
  like glibc-devel
* Fri Sep 28 2012 aj@suse.de
- Fix crash in relocation on sparc with bind-now (swo#14376)
  (patch sparc-bind-now.patch).
* Thu Sep 27 2012 aj@suse.de
- Fix check-abi on more architectures (bnc#780696), patch by Jan
  Engelhardt (updated patch glibc-fix-check-abi.patch).
* Tue Sep  4 2012 schwab@linux-m68k.org
- Move pt_chown to %%{_libexecdir} so that baselibs can find it.
- Do create hardlinks to getconf so that baselibs use the correct binary
  and getconf -v works (obsoletes patch glibc-2.8-getconf.diff).
* Tue Aug 28 2012 aj@suse.de
- Update to da1f43196321, major changes:
  * Fix strtod integer/buffer overflow (bso#14459).
  * Fix segmentation fault in strncasecmp for i686.
  * Translation updates
* Tue Jul 24 2012 aj@suse.de
- Fix _mcount on PowerPC32.
* Sat Jul 21 2012 aj@suse.de
- Enable parallel build for some more cases.
* Fri Jul 20 2012 aj@suse.de
- Update to glibc 2.16 release.
  * update version numbers
  * Documentation updates
- Update manpages from Debian, includes new man pages for
  locale.1, mtrace.1, gencat.1 iconvconfig.8.
- Remove getent.1 man page, use the one from man-pages.
* Fri Jul 20 2012 aj@suse.de
- Update to glibc 2.16 trunk (git id 416bf844227d):
  * Fix check-abi for arm
  * Refresh some patches.
  * remove merged patches
  * mainly lots of bug fixes
  * Support for ISO C11
  * full details in NEWS file
* Fri Jul 20 2012 jengelh@inai.de
- Add commit 10f62770e115d9f (glibc-sparc-fxstatat64.diff) from
  upstream to resolve rpmlint-induced packaging abort due to
  (previously) "no-return-in-nonvoid"
- Add xz as explicit BuildRequires for openSUSE_12.1
* Thu Jul 19 2012 jengelh@inai.de
- Do package ld-linux.so.2 on SPARC as well
* Tue Jul 17 2012 aj@suse.de
- Add makeinfo as build requires to be able to generate the info
  pages.
* Wed Jul  4 2012 aj@suse.de
- Add glibc-2.16-getaddrinfo-buffer.patch to not use a too small
  buffer for dns queries on 64-bit systems (bso#14307,bso#13904).
- Fix getaddrinfo-ipv6-sanity.diff (bnc#767125).
* Mon Jul  2 2012 agraf@suse.com
- Backport *context patch for ARM from 2.16, so we have a working libpth
* Thu Jun 28 2012 aj@suse.de
- Remove not anymore existing nscd.socket.
* Thu Jun 28 2012 aj@suse.de
- Update to glibc 2.15-branch (git id 725b8ee08aff):
  * Fix regression in cosh (bso#14273)
* Wed Jun 27 2012 aj@suse.de
- Testsuite is ok to fail on arm
* Tue Jun 26 2012 aj@suse.de
- Update to glibc 2.15 branch (git id b640404bd8c9):
  * Fix fmod implementation that returned NaN for some values.
  * Fix strcase_cmp_l x86 multiarch implementation.
  * Remove glibc-2.15-avx2.patch, merged now.
* Thu Jun 21 2012 aj@suse.de
- Fix Bogus FP exception on underflow for exp(double)  (bnc#767956)
* Mon Jun 18 2012 aj@suse.de
- Remove nscd.socket since nscd does not support socket
  activation.
* Wed Jun 13 2012 aj@suse.de
- Build power3 libs on 32-bit PowerPC (bnc#765165).
* Wed May 23 2012 aj@suse.de
- Update to current git of glibc 2.15 branch:
  * obsoletes glibc-2.16-avx.patch,
    glibc-2.16-fix-check-localplt.patch
  * Fixes for math functions.
* Wed May 23 2012 aj@suse.de
- Fix first_weekday for tr_TR locale (tr_TR.patch)
* Wed May 16 2012 schwab@linux-m68k.org
- Fix confstr use of local buffer outside its extent
* Wed May  9 2012 aj@suse.de
- Split out glibc-armhf-compat.patch from armhf-ld-so.patch.
- Run nscd in the foreground with systemd (glibc-nscd-foreground-patch)
* Tue May  8 2012 aj@suse.de
- Refresh glibc-2.16-avx.patch and handle FMA4.
* Thu May  3 2012 aj@suse.de
- Fix dynamic linker name for ARMv5.
* Thu Apr 26 2012 aj@suse.de
- Fix mcheck (optimization problem with gcc 4.7)
* Wed Apr 25 2012 aj@suse.de
- Also check for armv7hl.
* Wed Apr 25 2012 adrian@suse.de
- provide old ld-linux.so.3 on armv7, not catched by find.provides
* Sun Apr 22 2012 aj@suse.de
- gcc 4.7 does not build crt files properly on powerpc64, backport
  patch from upstream to handle this
  (glibc-2.16-powerpc-initfini.patch)
* Fri Apr 20 2012 aj@suse.de
- Disable patch x86-cpuid-level2.patch, this should be fixed in
  valgrind now.
* Thu Apr 19 2012 aj@suse.de
- Reorder patches, refresh to apply cleanly with -p1.
* Wed Apr 18 2012 aj@suse.de
- Build profiled libs only for base.
- Cleanup patches.
- Remove glibc-2.14-fix-ctors.patch, we have proper support in
  binutils now.
* Tue Apr 17 2012 aj@suse.de
- The dynamic linker for armv7 hardware float is called
  /lib/ld-linux-armhf.so.3. Provide symlink for old location.
* Tue Apr 17 2012 aj@suse.de
- Build for power4 only, don't support power3 anymore.
- Add option to build glibc a bit more quickly for testing purposes.
- Remove disabled patch glibc-revert-netlink-cache.patch.
- Start renumbering patches.
* Mon Apr 16 2012 aj@suse.de
- Do not build locales and profile for i686 since we only ship
  the base and glibc-devel packages.
* Fri Apr 13 2012 aj@suse.de
- Fix check-abi to succeed.
- Use feof_unlocked in glibc-2.3.90-bindresvport.blacklist.diff.
- Make x86 to be elf/check-localplt clean again
- Refresh some patches to apply without fuzz.
* Fri Apr  6 2012 aj@suse.de
- Disable building with as-needed since it breaks glibc
  (testsuite will fail, even if testsuite does not use as-needed)
* Thu Apr  5 2012 aj@suse.de
- Another fix for dynamic linking.
* Thu Apr  5 2012 aj@suse.de
- Properly fix dl_lookup_x crash (glibc-fix-noload.patch), disable
  glibc-fix-lookup-crash.patch.
* Mon Apr  2 2012 aj@suse.de
- Fix building on Armv5 (glibc-add-arm-dependency-libmemusage.patch)
* Thu Mar 29 2012 aj@suse.de
- Fix crash when nscd is not running (bnc#741021) properly
  and disable patch glibc-revert-netlink-cache.patch
* Fri Mar 23 2012 aj@suse.de
- Fix path for nss_db (bnc#753657).
* Thu Mar 22 2012 aj@suse.de
- Fix LD_PROFILE invocation.
* Wed Mar 21 2012 aj@suse.de
- Fix cycle detection in loading of dynamic objects.
* Wed Mar 14 2012 aj@suse.de
- Update getaddrinfo-ipv6-sanity.diff (bnc#684534).
* Fri Mar  2 2012 aj@suse.de
- vfprintf: validate nargs and maybe allocate from heap (bnc#747768)
* Tue Jan 31 2012 aj@suse.de
- Fix checks for AVX (patch glibc-2.16-avx.patch)
* Tue Jan 31 2012 aj@suse.de
- Use Linux 2.6.32 on x86_64 as oldest supported kernel
* Tue Jan 31 2012 aj@suse.de
- Fix relocation ordering to fix ifunc crash (bnc#740109).
* Sun Jan 29 2012 dmueller@suse.de
- set abi back to gnueabi for ARM (regression from Jan 12)
* Wed Jan 18 2012 aj@suse.de
- Update license strings in spec file.
* Mon Jan 16 2012 aj@suse.de
- Split up new subpackage glibc-extra and move makedb with its
  dependency on libselinux to it. (bnc#741525)
- Use lua script for verifyscript to not have a dependency on bash
* Thu Jan 12 2012 aj@suse.de
- Disable netlink caching (bnc#741021).
* Thu Jan 12 2012 aj@suse.de
- Use _target_cpu instead of _build and _host. This makes the
  i686 library really build for i686.
* Tue Jan 10 2012 aj@suse.de
- Remove all math ifuncs on x86-64.
* Tue Jan 10 2012 aj@suse.de
- Remove math ifuncs that cause crashes in apache2 and
  gdk-pixbuf-query-loaders-64 (bnc#740109) for now
  (glibc-2.15-math64crash.patch).
- nscd needs the same glibc version, add require.
* Mon Jan  9 2012 aj@suse.de
- Fix realloc usage in vfscanf (glibc-2.16-scanf.patch)
- Fix getcontext on powerpc 32-bit (glibc-ppc-getcontext.patch)
* Wed Jan  4 2012 aj@suse.de
- Remove patch glibc-2.3.4-gb18030-big5hkscs.diff.bz2 and
  replace it by gb18030.patch.bz2 (synced with Fedora 16)
  (bnc#743617).
* Tue Jan  3 2012 aj@suse.de
- Compile unwind-dw2-fde-glibc.c with -fno-strict-aliasing
  (patch glibc-strict-aliasing.diff)
- Build on i686 a separate library for usage under Xen.
* Tue Jan  3 2012 aj@suse.de
- Only use -mno-tls-direct-seg-refs on 32-bit x86, it's not needed
  on x86-64.
* Tue Jan  3 2012 aj@suse.de
- Cleanup spec file:
  - experimental malloc is not experimental anymore and only option
    (drop patch glibc-malloc-arena-max.diff)
  - remove obsolete configure flags
* Tue Jan  3 2012 aj@suse.de
- Update to 2ba92745c36e:
  - Update copyright years
  - Fix miscompilation of posix/regex_internal.c with GCC 4.7.
* Tue Jan  3 2012 aj@suse.de
- Revert pthread-cond-wait change for now since it causes hangs
  (patch pthread-cond-wait-revert.patch).
* Mon Dec 26 2011 aj@suse.de
- Update to c0da14cdda1f (glibc-ports: 8a70b2dcabbf):
  - new locales
  - many bugfixes
  - glibc 2.15
  - obsoletes tzfile-corruption-fix.patch,
    glibc-2.14-32args-printf.patch
* Wed Dec 21 2011 meissner@suse.de
- Do not mention the buildtime anymore to make build-compare work.
* Mon Dec 19 2011 aj@suse.de
- Fix timezone loader overflow (bnc#735850,CVE-2009-5029) (patch
  tzfile-corruption-fix.patch)
* Mon Dec  5 2011 aj@suse.de
- Update to 52ad36a21973 (glibc-ports: 4a40b38f6355):
  * fix many warnings
  * bugfixes
  * obsolete patches glibc-strict-aliasing.diff and
    glibc-arm-clone-unwind-fix.diff
* Tue Nov 29 2011 rcoe@wi.rr.com
- fix printf with many args and printf arg specifiers (bnc#733140)
* Thu Nov 24 2011 aj@suse.de
- Updated glibc-ports-2.14.1.tar.bz2 from ftp.gnu.org.
* Thu Nov 24 2011 aj@suse.com
- Create glibc-devel-static baselibs (bnc#732349).
* Thu Nov 17 2011 aj@suse.de
- Remove duplicated locales from glibc-2.3.locales.diff.bz2
* Tue Nov 15 2011 aj@suse.de
- nscd now monitors resolv.conf, so simplify glibc-2.4.90-nscd.diff
  patch and re-enable it
- Adjust and re-enable glibc-2.3.3-nscd-db-path.diff
* Tue Nov 15 2011 aj@suse.de
- Update to 09f93bd3d6b1:
  - Clean up internal fopen uses to always use close-on-exec
  - Optimized strcasecmp and strncasecmp for SSSE3 and SSE4.2 on x86-32
  - bugfixes
* Mon Nov 14 2011 ro@suse.de
- check-build.sh: accept kernel 3.x
* Tue Nov  8 2011 dmueller@suse.de
- make glibc-static provide versioned
* Tue Nov  8 2011 dmueller@suse.de
- fix building with ports again:
  * ifuncs tracing backport requires adjustments
- avoid using asynchronous-unwind-tables only during configure,
  but still built the library with unwind tables enabled
- replace arm build hack with a proper solution
* Mon Nov  7 2011 dmueller@suse.de
- update glibc-ports to 2.14.1:
  * Align the constant pool in start.S to 4 bytes.
  * sysdeps/unix/sysv/linux/arm/eabi/internal_sendmmsg.S: New.
- Fix unwinding crash when using clone()
* Wed Nov  2 2011 aj@suse.de
- Update to d35dce5213b3:
  - caching of network information for getaddrinfo
  - bugfixes
* Sat Oct 29 2011 aj@suse.de
- Update to 21b64b153631:
  - various bugfixes
* Thu Oct 27 2011 aj@suse.de
- Update to 804791474dc1:
  - more libm optimizations
- Remove AMD libm routines due to new upstream libm optimizations
  and interfaces.
- Fix build on x86.
* Mon Oct 24 2011 aj@suse.de
- Update to a201fbcf9cc3:
  - More libm optimizations
  - Optimize wcslen, strnlen
* Thu Oct 20 2011 aj@suse.de
- Ignore failure of chkstat (bnc#725145).
* Wed Oct 19 2011 aj@suse.de
- Provide glibc-static from glibc-devel-static to make packaging
  easier. glibc-static is the Fedora name.
* Tue Oct 18 2011 aj@suse.de
- Update to glibc head 99ce7b04edf1:
  - Optimize many libm functions
  - Optimize access to isXYZ and toXYZ tables
  - Optimized memcmp and wmemcmp for x86-64 and x86-32
  - Add parameter annotation to modf
  - Support optimized isXXX functions in C++ code
  - Optimized memchr, memrchr, rawmemchr for x86-32
  - Add nscd netgroupcache
- Disable libm-x86-64.diff.bz2 since it needs changes following the
  libm optimzations.
* Mon Oct 17 2011 aj@suse.de
- Remove fma, fmaf from libm-x86-64.diff.bz2, they are not accurate
  (bso#13304).
* Tue Oct 11 2011 aj@suse.de
- Update to current git 4f2b767fef50
  * Correctly reparse group line after enlarging the buffer
  * 2.14.1 release
* Mon Oct 10 2011 aj@suse.de
- Fix tracing with ifuncs.
* Tue Oct  4 2011 uli@suse.com
- cross-build: use %%__cc, %%__cxx, %%_build, %%_host
- mipsel: install ports, disable FP
* Fri Sep 30 2011 dmueller@suse.de
- avoid build failure on %%%%arm
* Thu Sep 29 2011 aj@suse.de
- Update to 68577918437e:
  * Obsoleted patches: glibc-gconvcache-s390.diff,
    glibc-2.3.5-nscd-zeronegtimeout.diff,
    glibc-x86-bits-sigcontext.patch,
    glibc-2.15-getsysstats-speedup.patch, glibc-2.13-warnings.fix,
    glibc-2.13-localedef.patch,
    glibc-fix-rwlock-stack-imbalance.patch, glibc-2.15-vsyscall.patch,
    glibc-2.15-avoid-vsyscall.patch
* Thu Sep 29 2011 adrian@suse.de
- just use %%arm for right arm architectures
* Tue Sep 27 2011 aj@suse.de
- Update to 2.14 git version 8bd683657e8a, it contains backports from
  glibc trunk:
  + Avoid race between {,__de}allocate_stack and __reclaim_stacks
    during fork
- Add glibc-2.14-fix-ctors.patch to run ctors (bnc#717671), gcc 4.6
  optimized these away.
* Mon Sep 26 2011 adrian@suse.de
- don't use -fasynchronous-unwind-tables on arm, it breaks the build
* Thu Sep 22 2011 dmueller@suse.de
- fix ports build
* Thu Sep 22 2011 adrian@suse.de
- don't enforce armv5tel for all arm architectures anymore
- armv8l architecture does not exist
* Tue Sep 20 2011 aj@suse.de
- Cleanup spec file: remove some tags
* Mon Sep 19 2011 opensuse@cboltz.de
- move usr.sbin.nscd apparmor profile back to apparmor-profiles package
  (bnc#647718)
* Mon Sep 19 2011 aj@suse.de
- Update to 2.14 git version 69c1dfc1a796, it contains backports from
  glibc trunk:
  + Fix setxid race handling exiting threads
  + Fix cfi directives in x86-64 and i386 code
  + Fix scope handling during dl_close
  + Fix spurious nop at start of __strspn_ia32
  + Correct cycle detection during dependency sorting
  + Fix fopen (non-existing-file, "re") errno (Obsoletes patch
  fopen-close-exec.patch)
* Fri Sep 16 2011 mmarek@suse.cz
- Fix assertion error in res_query.c, triggered by Firefox
  (bso#13013).
* Wed Sep  7 2011 aj@suse.de
- Avoid usage of vsyscalls inside glibc.
* Tue Sep  6 2011 aj@suse.de
- Prefer real syscalls instead of vsyscalls on x86-64 outside
  libc.so
* Mon Sep  5 2011 aj@suse.de
- Do not not trigger an abort when an i586 Intel CPU is running the
  i686 library, as valgrind does. [bnc#681398]
* Mon Sep  5 2011 aj@suse.de
- Readd vdso support.
- Add patch to relocate objects in dependency order, this fixes the
  vdso support.
* Sat Sep  3 2011 aj@suse.de
- Revert last change, it caused some breakage. [bnc#715854]
* Thu Sep  1 2011 aj@suse.de
- Add support for new vdso (3.1 kernel) and only then
  for vsyscall which is not anymore in 3.1 kernel
  (patch glibc-2.15-vsyscall.patch)
* Thu Aug 25 2011 lnussel@suse.de
- use symbol version OW_CRYPT_1.0 for the Owl extensions
  (crypt_r[an], crypt_gensalt.*)
- refactor sha2 gensalt patch
- document sha2 hashes in man page
* Mon Aug 22 2011 aj@suse.de
- Update AppArmor profile of nscd [bnc#647718]
* Fri Aug 19 2011 aj@suse.de
- Fix fopen("non-existing-file", "re") errno [bnc#713146].
* Thu Aug 18 2011 aj@suse.de
- Mark glibc-info, glibc-html and glibc-i18ndata as noarch packages.
* Wed Aug 17 2011 matz@suse.de
- Fix crash (access-after-free) in dl_lookup_x.  [bnc#703140]
* Wed Aug 17 2011 aj@suse.de
- Enhance rpmlintrc to ignore shlib policy violation.
- Remove obsolete patches glibc-fnmatch-multibyte.diff,
  glibc-2.3.1.localedef.diff, glibc-2.10-nscd-nostack.diff.
* Wed Aug 17 2011 aj@suse.de
- Update to current 2.14 branch: Various bugfixes.
  Remove upstreamed patches missing-include-build-fix.diff,
  glibc-2.14-res_send.patch, glibc-dl-fxstatat64.patch)
* Wed Aug 17 2011 aj@suse.de
- Revert removal of sunrpc code (patch glibc2.14-revert-sunrpc-removal.patch).
* Fri Aug 12 2011 matz@suse.de
- Revert seeking on fclose, incomplete implementation of POSIX
  behaviour can confuse current users.  [bnc #711829]
  (patch glibc-revert-fseek-on-fclose.diff)
* Fri Jul 22 2011 lnussel@suse.de
- fix signature of sha2 functions
* Thu Jul 21 2011 rhafer@suse.de
- Disable rewriting ::1 to 127.0.0.1 for /etc/hosts entries.
  (bnc#684534, bnc#706719)
  (patch glibc-fix-double-loopback.diff)
* Wed Jul 20 2011 lnussel@suse.de
- install crypt_blowfish man pages
* Tue Jul 19 2011 aj@suse.de
- Back to old glibc-2.2-sunrpc.diff for now.
* Tue Jul 19 2011 lnussel@suse.de
- update crypt_blowfish to version 1.2 (bnc#700876)
  * due to the signedness bug fix 2a hashes are incompatible with
    previous versions if the password contains 8bit chracters!
  * libcrypt now exports crypt_gensalt
* Fri Jul 15 2011 aj@suse.de
- Update to glibc 2.14:
  * The RPC implementation in libc is obsoleted.  Old programs keep working
  but new programs cannot be linked with the routines in libc anymore.
  Programs in need of RPC functionality must be linked against TI-RPC.
  The TI-RPC implementation is IPv6 enabled and there are other benefits.
  * Various bugfixes, new locales, new linux kernel interfaces.
  * New program sotruss to traces calls through PLTs
  * Removed the following obsoleted patches: glibc-2.6-configure.diff,
    glibc-disable-backward-memcpy.diff, glibc-static-memcpy.diff,
    glibc-zarch-longlong.diff, glibc-bso-12454.diff,
    glibc-vfprintf-positional.diff
- Build without -fno-strict-aliasing.
* Tue Jul 12 2011 aj@suse.de
- Remove ppc-atomic.diff after discussion with glibc PPC experts
  since it does not bring any real benefit.
* Thu Jul  7 2011 aj@suse.de
- Update glibc-2.2-sunrpc.diff with newer patch from sourceware
  bugzilla (bs#bso#5379).
* Wed Jul  6 2011 lnussel@suse.de
- enable glibc-2.8-getconf.diff again to fix build failure
* Tue Jul  5 2011 aj@suse.de
- Add systemd configuration for nscd.
* Tue Jul  5 2011 aj@suse.de
- Remove glibc-statfs64-ia64.diff since current ia32el has the bug
  fixed.
- Remove glibc-ppc64-vdso-time.diff (bnc#590216).
* Mon Jul  4 2011 aj@suse.de
- Further cleanup of ld.so.conf to remove duplicate directories
  and allow override of system directories (bnc#671725)
- Fix futex bug (added patch glibc-fix-rwlock-stack-imbalance.patch
  from sourceware bug #12403).
* Fri Jul  1 2011 aj@suse.de
- Remove glibc-2.10.99-ia64-include.diff, it's not needed anymore.
- Disable obsolete patches glibc-fnmatch-multibyte.diff,
  glibc-2.10-nscd-nostack.diff, glibc-2.3.1.localedef.diff
- Add glibc-2.13-localedef.patch to fix alignment problem.
- Adjust glibc-version.diff, FLOATING_STACKS variable is obsolete.
* Thu Jun 30 2011 aj@suse.de
- Remove old ARM patches as it's unclear why those are needed and arm
  support is anyhow broken according to comments here and in spec
  files.
* Thu Jun 30 2011 aj@suse.de
- Back to kernel 2.6.16 as minimal kernel version.
* Tue Jun 28 2011 aj@suse.de
- Only apply glibc-statfs64-ia64.diff on SLE but not on openSUSE.
* Mon Jun 27 2011 aj@suse.de
- Disable exp_malloc on PPC platforms for now since it caused a
  build failure.
* Mon Jun 27 2011 aj@suse.de
- Update glibc-x86-bits-sigcontext.patch with version accepted
  upstream.
- Add glibc-2.15-getsysstats-speedup.patch to "Fix Linux
  sysconf(_SC_NPROCESSORS_[CONF|ONLN]) performance problem"
- Fix warning about potential array subscript out of bounds
  (glibc-2.13-wranings.fix).
- Fix some rpmlint warnings.
* Tue Jun 21 2011 aj@suse.de
- Cleanup spec file, change PreReqs.
- Remove obsolete patches: glibc-2.3.90-ld.so-madvise.diff,
  glibc-suse-note.diff, glibc-2.3-regcomp.diff.
- Rename glibc-2.3-SuSE.diff to glibc-testsuite.patch, remove obsolete
  parts.
* Mon Jun 20 2011 aj@suse.de
- Enhance rpmlintrc.
- move summary,license and group out of the not-ppc if (by Marcus
  Meissner).
* Mon Jun 20 2011 aj@suse.de
- Remove PreReq on permissions to avoid a cycle that gives leads to
  an uninstallable system (bnc#700925).
- Fix <bits/sigcontext.h> on x86 - fixes scpm compilation
  (bnc#700784).
* Fri Jun 17 2011 aj@suse.de
- Cleanup ld.so.conf list, move kde directories to
  kdelibs3 (bnc#671725).
* Fri Jun 17 2011 aj@suse.de
- Do not package memusage and memusagestat since they require
  lib-gd and that grows the build cycle.
* Thu Jun 16 2011 aj@suse.de
- Fix handling of shared library preloading.
* Thu Jun 16 2011 aj@suse.de
- Fix getaddrinfo IPv6 code (bnc#684534).
- Split up glibc-devel-static subpackage (bnc#655261).
* Thu Jun 16 2011 aj@suse.de
- Reorder ld.so.conf library list (bnc#671725).
* Wed Jun 15 2011 aj@suse.de
- Split up package glibc-utils with some debugging tools, add memusage
  and memusagestat programs
- Fix compile warnings.
- Silence some rpmlint warnings.
- Remove obsolete patches glibc-2.3.2-revert_tcsetattr.diff,
  glibc-2.8-clone.diff, glibc-nptl-2.4-nofixsyscallnr.diff
- Cleanup spec file.
- Use new permission handling for pt_chown.
- Build without -fno-strict-aliasing.
* Mon Jun  6 2011 aj@suse.de
- Fix typos in /etc/gai.conf (bnc#651912).
* Fri Jun  3 2011 aj@suse.de
- Use 2.6.5 as oldest supported kernel on all archs besides x86 and
  x86-64 (reverting partially a change).
* Fri May 27 2011 aj@suse.de
- Remove patches that are not needed anymore:
  * glibc-selinux.diff
  * glibc-check-native-missing-include.diff
  * glibc-2.3.3-execstack.diff
- Disable the following patches, they look obsolete by now but leave them
  in to easy enable them again:
  * glibc-2.3-SuSE.diff
  * glibc-2.3-regcomp.diff
  * glibc-2.8-clone.diff
  * glibc-2.3.2-revert_tcsetattr.diff
- Do not package glibc-obsolete with empty directory on archs that
  don't need it.
* Thu May 26 2011 aj@suse.de
- Add glibc-static-memcpy.diff to fix static compiles.
* Wed May 25 2011 aj@suse.de
- Run on kernel 2.6.32 or newer only - and optimize thus.
- Enable malloc implementation that is be less memory efficient
  but higher performing in multi-threaded programs.
- Use --enable-bind-now to avoid lazy binding in ld.so
  and libc.so
* Wed May 25 2011 aj@suse.de
- Add glibc-bso-12454.diff to not create assert running dynamic linker
  in some cases.
- Disable backwards memcpy for x86-64.
* Wed May 25 2011 aj@suse.de
- Add glibc-dl-signal-error.diff to not crash the dynamic linker
  if dependencies are missing.
* Mon May 23 2011 rguenther@novell.com
- Update longlong.h from GCC sources.  Fixes s390 build with -mzarch.
  [bnc#695102]
* Fri May 20 2011 aj@suse.de
- Fix obsoletes and provides.
- Update to glibc 2.13
- Drop ld-prelink-unique.diff, it is part of 2.13.
* Tue Feb 22 2011 pbaudis@suse.cz
- Fix potential removal of required local scope from shared objects
  marked as NODELETE [bnc#657627]
* Thu Feb  3 2011 pbaudis@suse.cz
- Upgrade to latest release/2.11/master - glibc-2.11.3-b72646ad0c41
  - Random assortion of bugfixes, some #defines for new kernels
- Retired patches:
  - glibc-2.11.3-bnc658509.diff
  - glibc-2.11.3-bso12397.diff
  - glibc-sparc64-fxstat.diff
- Remove the NSS hack of opening modules using RTLD_DEEPBIND.
  This was useful for nss_ldap, since some applications used a different
  LDAP library with clashing symbol names. However, it also created
  many headaches, especially with the NSS modules not respecting
  malloc() overrides. Now, sssd is used by default for LDAP resolutions
  and we can therefore safely get rid of the hack. [bnc#477061]
- Remove the currently disabled AMD string function overrides.
  Benchmarking did not unearth any differences that would make
  convincing case for keeping the functions with all the associated
  maintenance headaches; AMD does not recommend keeping their custom
  versions of the functions either.
- Introduce MALLOC_ARENA_MAX and M_ARENA_MAX support [bnc#659090]
- Fixed stack unwinding past glibc _fini function (proper showing
  of destructor backtraces) [bnc#585879]
- Fix gconv cache generation on s390 [bnc#592944]
- Add missing iconvconfig for refreshing gconv.cache to glibc-locale %%post
- Fixed stack corruption in *printf() with large number of positional
  specifiers [bnc#666179]
* Fri Jan 28 2011 rguenther@novell.com
- Fix ENOSYS detection for mkdirat.  [bso#12397]
* Thu Dec  9 2010 rguenther@novell.com
- Drop restrict qualifiers from memmove and bcopy fortify wrappers.
  [bnc#658509]
* Tue Nov 30 2010 pbaudis@suse.cz
- Upgrade to glibc-2.11.3
  - Just few more bugfixes since the latest upgrade
* Tue Nov 16 2010 pbaudis@suse.cz
- Remove explicit support for nss SPLIT_GROUPS setting; glibc from
  2.11 on should handle this automagically.
* Mon Nov 15 2010 pbaudis@suse.cz
- Upgrade to latest release/2.11/master - glibc-2.11.2-dd2fde461e30
  - Candidate tree for glibc-2.11.3
  - Multitude of various bug fixes
* Mon Nov 15 2010 pbaudis@suse.cz
- Enable multi-arch routines support for ppc, ppc64, i686 and x86_64.
  Temporarily, AMD-optimized string routines are disabled. We will
  selectively re-enable them after some more careful benchmarking
  in the next few days.
  Note: In this glibc version, multi-arch routines DO NOT include
  the controversial backwards-copy memcpy().
* Wed Oct 27 2010 pbaudis@suse.cz
- Fix missing hconf setup in nscd - this meant `multi on` in
  /etc/host.conf was ignored when nscd was used, breaking e.g.
  resolving of multi-AF records in /etc/hosts [bnc#572188]
- Fix two LD_AUDIT security vulnerabilities: spurious $ORIGIN expansion
  and missing setuid library check when running as setuid [bnc#646960]
* Mon Sep 13 2010 aj@suse.de
- Fix build in manual with make 3.82.
* Sun Sep 12 2010 aj@suse.de
- Work around shortest-stem feature in make 3.82+ (patch
  glibc-make-3.82.patch)
* Wed Jul  7 2010 pbaudis@suse.cz
- Fix NIS-compat initgroups routine to work reliably over multiple
  invocations [bnc#607064]
* Thu Jul  1 2010 jengelh@medozas.de
- SPARC: restore fxstat functionality in sparc64 mode
* Sat Jun 26 2010 jengelh@medozas.de
- Use %%_smp_mflags
* Fri Jun 11 2010 pbaudis@suse.cz
- Upgrade to latest release/2.11/master - glibc-2.11.2-21580967038a
  - Fix buggy x86 CPU family and model check
  - Fix sunrpc error condition handling
  - Fix crash on unresolved weak symbol reference
* Tue Jun  1 2010 pbaudis@suse.cz
- Fix ldconfig -r failing on aux cache [bnc#609345]
* Thu May 20 2010 pbaudis@suse.cz
- Upgrade to glibc-2.11.2 (f2020c64596a)
  - Virtually no changes since previous revision, just official tag
* Wed May 12 2010 pbaudis@suse.cz
- Upgrade to latest release/2.11/master - glibc-2.11.1-6d270188ef3f
  - This contains considerable amount of bugfixes cherrypicked from
    the master branch, slated to become glibc-2.11.2 soon with little
    further changes
* Mon May 10 2010 pbaudis@suse.cz
- Add a hint about /etc/host.conf mdns option removal to the parser
  error reporting [bnc#569273]
* Tue May  4 2010 aj@suse.de
- Handle /var/run on tmpfs
* Mon Apr 26 2010 pbaudis@suse.cz
- Remove stale resolv.dynamic.diff patch file
* Thu Apr 22 2010 meissner@suse.de
- for powerpc32 build a generic library and also a power4 library
  to allow older 32bit powerpc to work. bnc#596271
* Tue Apr 20 2010 pbaudis@suse.cz
- Consider private IPv4 addresses as global-scope in gai.conf [bnc#597616]
* Wed Mar 24 2010 pbaudis@suse.cz
- Fix get_nprocs() large CPU count handling [bnc#590665]
* Tue Mar 23 2010 pbaudis@suse.cz
- Fix interference of STB_GNU_UNIQUE and prelink [bnc#586979]
- Make time() use VDSO on PPC64 [bnc#590216]
- Make pthread_cond_timedway() use VDSO on PPC64 [bnc#590517]
* Thu Mar 18 2010 pbaudis@suse.cz
- Add PreReq for cat to glibc-locale [bnc#588997]
* Thu Mar 18 2010 pbaudis@suse.cz
- Fix /etc/resolv.conf change detection for multi-threaded applications
  (by Sebastian Kienzl)
* Fri Mar  5 2010 pbaudis@suse.cz
- Fix spurious mapping from lowercase sharp s to uppercase sharp S
  which is less standard and causes problems for ISO encodings [bnc#541773]
* Mon Mar  1 2010 pbaudis@suse.cz
- Fix spurious overflow messages on some R_X86_64_PC32 relocations,
  patch by Richard Guenther [bnc#583019]
* Wed Feb 24 2010 pbaudis@suse.cz
- Fix assertion failures within valloc(), patch by Michael Matz
  [bnc#581391]
* Wed Feb 17 2010 pbaudis@suse.cz
- S/390: Disable two UTF conversion instructions due to the hardware
  mishandling some error conditions in the input stream [bnc#579871]
* Mon Feb 15 2010 pbaudis@suse.cz
- Further fix previous power5 fix
* Sun Feb 14 2010 rguenther@suse.de
- avoid spurious differences in nscd on rebuilds.
* Thu Feb 11 2010 pbaudis@suse.cz
- Fix another upgrade problem on power5 and older - correctly remove
  the old optimized libraries; found out by matz@suse.de [bnc#563978]
* Fri Feb  5 2010 pbaudis@suse.cz
- Final fix for memchr() overflow on IA64 provided by upstream
  [bnc#567516]
* Thu Jan 28 2010 pbaudis@suse.cz
- Fix another memchr() overflow on IA64; another known one still
  remains a mystery [bnc#567516]
* Wed Jan 27 2010 pbaudis@suse.cz
- Remove now-obsolete powerpc-cpu addon usage [bnc#572819]
- Fix statfs64() on IA64 in 32-bit mode ia32el [bnc#534828]
* Tue Jan 26 2010 pbaudis@suse.cz
- Fix upgrade from previous distributions on power5 and older [bnc#563978]
* Mon Jan 25 2010 meissner@suse.de
- ignore power6, power7 and cell specific Power CPU optimizations
  on openSUSE.
* Tue Jan 19 2010 pbaudis@suse.cz
- Disable AMD-optimized strncmp() on x86_64 that is getting miscompiled
  by the assembler [bnc#540647]
* Mon Jan 18 2010 pbaudis@suse.cz
- Upgrade to latest release/2.11/master - glibc-2.11.1-11c19d374bd4
  - Just trivial changes - version bump to 2.11.1, NIS-related
    resolver fixes, ELF lookup race, .h typo fix and regcomp
    out-of-memory handling miss
  - Few more bugfixes, weekday updated in three locales
- Retired patches:
  glibc-ia64-memchr.diff
* Mon Dec 14 2009 jengelh@medozas.de
- add baselibs.conf as a source
* Fri Dec 11 2009 jengelh@medozas.de
- libnldbl_nonshared is only built for sparc32
* Tue Dec  8 2009 jengelh@medozas.de
- package libnldbl_nonshared for SPARC
* Fri Nov 20 2009 pbaudis@suse.cz
- Upgrade to latest release/2.11/master - glibc-2.11-1bc1954c7357:
  - This snapshot is slated to become 2.11.1 if all goes well
  - Mix of fixes for various recently found out issues
- Retired patches:
  glibc-ifunc-dlsym-fixup.diff
- [IA64] Fix memchr() on mmap(MAP_SHARED) boundary [fix lighttpd crashes]
- Again use 127.0.0.1 as default nameserver in case of empty
  /etc/resolv.conf [bnc#473308]
* Thu Nov 19 2009 pbaudis@suse.cz
- Add 623 and 664 to /etc/bindresvport.blacklist to fix problems with
  port conflict on some IPMI network cards [bnc#513961]
- Allow external additions to /usr/share/locale/locale.alias and
  /usr/lib/gconv/gconv-modules, required for the SAP-locales package.
- Add delimiter in front of external additions to locale/gconv lists
- Remove AMD-provided strlen(), upstream is faster.
- Keep .symtab and .strtab for all glibc-related .so libraries [FATE#307090]
  [bnc#545721]
- [ppc,ppc64] Enable optimized builds for power7; no separate power4, power5
  builds are generated anymore, instead default glibc is built for power4
  at least, tuned for power5 execution [FATE#307021]
- Large clean-up of the specfile
* Fri Nov 13 2009 pbaudis@suse.cz
- Upgrade to Git HEAD as of 20091105 - glibc-2.11-13f6812ffb5b:
  * New interfaces: execvpe, pthread_sigqueue, mkstemps, mkstemps64, mkostemps,
    mkostemps64
    Implemented by Ulrich Drepper.
  * Checking version of longjmp added that fails if an uninitialized stack
    frame would be created.  Implemented by Ulrich Drepper.
  * STT_GNU_IFUNC is now supported in static executables.
    Implemented by H.J. Lu.
  * STT_GNU_IFUNC implemented for PPC by Alan Modra.
  * New optimized string functions for x86-64: strstr, strcasestr, memcmp,
    strcspn, strpbrk, strspn, strcpy, stpcpy, strncpy, strcmp (SSE2, SSE4.2),
    strncmp (SSE2, SSE4.2), strchr (SSE4.2), strrchr (SSE4.2).
    Contributed by H.J. Lu.
    strlen, rawmemchr, strcmp (SSSE3), strncmp (SSSE3).
    Implemented by Ulrich Drepper.
  * New optimized string functions for x86: strlen, strcspn, strspn, strpbrk,
    strstr, strcasestr.
    Contributed by H.J. Lu.
  * Support for fma instruction in AVX on x86-64.
    Implemented by H.J. Lu and Ulrich Drepper.
  * AVX support in x86-64 auditing support in ld.so.
    Implemented by H.J. Lu.
  * STB_GNU_UNIQUE support added.
    Implemented by Ulrich Drepper.
  * Implement second fallback mode for optimized DNS lookup for even more
    broken environments.  If two requests from the same port are not
    handled correctly close the socket and open a new one before sending
    the second request.  The 'single-request-reopen' option in /etc/resolv.conf
    can be used to select this mode right away, instead of rediscovering the
    necessity is every process again.
    Implemented by Ulrich Drepper.
  * New resolver flag RES_USE_DNSSEC to enable use of verified lookup.
    Implemented by Adam Tkac.
  * Optimized iconv conversions for S390x.
    Implemented by Andreas Krebbel.
  * Using condvars with PI mutexes is now more efficient due to kernel
    support for requeueing to PI futexes.  NPTL support added for x86-64.
    Implemented by Ulrich Drepper.
  * New locales: ps_AF, my_MM
- Retired patches:
  glibc-utmp-timeout-raise.diff
  glibc-nscd-assert.diff
  glibc-2.10-mcheck-free-race.diff
  glibc-nscd-prune-ret.diff
- Made ifunc compile on IA64 using patch by H.J.Lu
- Fixed compilation on PPC Cell
- Lowered utmp locking timeout from 30s to 10s to be compatible
  with upstream
- Allow external additions to /usr/share/locale/locale.alias and
  /usr/lib/gconv/gconv-modules, required for the SAP-locales package.
- Remove AMD-provided strlen(), upstream is faster.
- ARM is currently non-functional
* Fri Nov 13 2009 pbaudis@suse.cz
- Rediff patches to apply with fuzz 0
* Sat Sep 19 2009 pbaudis@suse.cz
- Update to glibc-2.10.1-44dbdf3777 of glibc/pb-stable.git glibc-2.10-branch
  (Still the same source is used for glibc-ports.) - random minor fixes
* Fri Sep 18 2009 coolo@novell.com
- add /etc/apparmor.d to the nscd file list
* Mon Sep 14 2009 pbaudis@suse.cz
- define _filter_GLIBC_PRIVATE so that our glibc is not marked to
  provide these symbols [bnc#468748]
- Fix exception behaviour of various libm functions by AMD optimized
  for x86_64 [bnc#487576]
* Tue Sep  8 2009 pbaudis@suse.cz
- Provide the /etc/apparmor.d/usr.sbin.nscd file [bnc#535467]
- Blowfish: Enable BF_SCALE even on x86_64
* Sat Sep  5 2009 pbaudis@suse.cz
- Raise utmp locking timeout from 1s to 30s to ensure logins get recorded
  even on heavily loaded systems [bnc#486631]
- Fix invalid pointer handling in some nscd assertions and the code to send
  data to the client [bnc#513617]
- Radically trim down the Blowfish support patch - keeps only crypt_blowfish
  from the original OWL patch, but does the rest within the current glibc
  crypt infrastructure [bnc#529495]
* Wed Aug 26 2009 mls@suse.de
- make patch0 usage consistent
* Thu Aug 20 2009 pbaudis@suse.cz
- Update to glibc-2.10.1-aa152ec6ad of release/2.10/master branch
  (Still the same source is used for glibc-ports.) - random minor fixes
  * REVERTED invalid fix of broken locking within nscd [bnc#510788]
- Fix nscd handling of zero negative timeout for some databases
  [bnc#523154,bnc#510788]
* Thu Jul 16 2009 pbaudis@suse.cz
- Update to glibc-2.10.1-4253a1e0b2 of glibc/pb-stable.git glibc-2.10-branch
  (Still the same source is used for glibc-ports.) - random minor fixes
  * Fix broken locking within nscd [bnc#510788]
- Fix compilation with 2.6.30 kernel headers on IA64
* Wed Jun 17 2009 pbaudis@suse.cz
- Fix a dead-lock introduced by the bnc#509398 fix
- Update to glibc-2.10.1-d0f6ed789f of glibc/pb-stable.git glibc-2.10-branch
  (Still the same source is used for glibc-ports.) - random minor fixes
* Fri Jun 12 2009 pbaudis@suse.cz
- Update to glibc-2.10.1-9cf557216c of glibc/pb-stable.git glibc-2.10-branch
  (Still the same source is used for glibc-ports.)
  * Fix 100%% CPU usage by Java apps because of bad futex calls [bnc#509096]
* Mon Jun  8 2009 pbaudis@suse.cz
- Fix race condition in the mcheck free() hook [bnc#509398]
* Thu Jun  4 2009 pbaudis@suse.cz
- Remove temporary revert of a namespace leakage cleanup in glibc-2.8
* Fri May 22 2009 pbaudis@suse.cz
- Update to glibc-2.10.1-e38af591a8 of glibc/pb-stable.git glibc-2.10-branch
  maintained by me at repo.or.cz; contains backports of various bugfixes.
  (Still the same source is used for glibc-ports.)
- Stop using separate tarball for nptl.
- Add ARM support, pulling in glibc-ports snapshot when compiling for ARM.
* Fri May 15 2009 pbaudis@suse.cz
- Add support for multiple group records for a single group
  when using nss_compat (configurable in /etc/default/nss) [bnc#480991]
* Sun May 10 2009 pbaudis@suse.cz
- Update to 2009051021, glibc-2.10.1:
  * Add missing cacheinfo entry for a recent Intel processor
  * Fix ABI issue with the gshadow interface
* Sun May 10 2009 pbaudis@suse.cz
- Upgrade to CVS HEAD as of 2009051010 - glibc-2.10:
  * DNS IPv4-IPv6 lookup changed back to parallel version; with broken DNS
    servers (the case e.g. for many ADSL users), you may experience long
    timeouts, once for each process (but eventually your request will be
    resolved) - the solution is to run nscd (default on SUSE) and/or
    put 'options single-request' in /etc/resolv.conf.
  * New interfaces: preadv, preadv64, pwritev, pwritev64, malloc_info
    Implemented by Ulrich Drepper.
  * New Linux interfaces: accept4, fallocate, fallocate64.
    Implemented by Ulrich Drepper.
  * Correct declarations of string function when used in C++ code.  This
    could lead to compile errors for invalid C++ code.
  * XPG7/POSIX 2008 compilation environment.  Many GNU-specific functions are
    now in POSIX.
  * New POSIX 2008 interface: psiginfo
    Implemented by Ulrich Drepper.
  * New ISO C++1x interfaces: quick_exit, at_quick_exit
    Implemented by Ulrich Drepper.
  * Support for selecting between multiple function definitions at runtime
    using STT_GNU_IFUNC symbols.  Implemented by Ulrich Drepper.
  * The libcrypt library can now use the hash function implementations in
    NSS.  Implemented by Ulrich Drepper.
  * The malloc implementation can be compiled to be less memory efficient
    but higher performing in multi-threaded programs.
    Implemented by Ulrich Drepper.
  * New locales: nan_TW@latin, ks_IN
  * Faster strlen, strchr, strchrnul, strrchr, memchr, and rawmemchr for x86-64.
    Implemented by Ulrich Drepper.
  * Extended printf hook support.  It is possible to use user-defined types
    and extend existing format specifiers.
    Implemented by Ulrich Drepper.
  * Handling for group shadow files has been added.
    Implemented by Ulrich Drepper.
- Retired:
  glibc-2.10-dns-fixpack.diff
  glibc-2.10-dns-no-gethostbyname4.diff
  glibc-2.10-locale-tuesday.diff
  glibc-2.10-nscd-gc-dataofs.diff
  glibc-2.10-nscd-prunerace.diff
  glibc-2.10-ppc32-setcontext-fp.diff
  glibc-2.10-unsetenv.diff
  glibc-2.4.90-mdns-resolver.diff
  glibc-2.9-fortify.diff
  glibc-getgroups-fortify.diff
  glibc-ldscript.diff
  glibc-mtfsf.diff
  glibc-2.10-nscd-meminflight.diff
  getaddrinfo-ipv6-sanity.diff
  glibc-2.10-nis-hosts.diff
- Several other modified
* Fri Apr 17 2009 pbaudis@suse.cz
- Fix defaulting to localhost in case of no nameserver entries
  in /etc/resolv.conf [bnc#473308]
* Mon Mar 23 2009 rguenther@suse.de
- Add missing libc_hidden_builtin_def (memcmp) to
  glibc-2.3.3-amd64-string.diff.  Fixes build with GCC 4.4.
* Fri Feb 27 2009 rguenther@suse.de
- Make sure to mark fortified inline routines with
  __attribute__((__artificial__)).
* Wed Feb 18 2009 adrian@suse.de
- Build i686 glibc src rpm as nosrc to get both versions on the
  media [bnc#474338]
* Thu Jan 22 2009 schwab@suse.de
- Fix use of mtfsf insn.
* Wed Jan 21 2009 schwab@suse.de
- Fix ldscript mangling.
* Wed Jan 21 2009 pbaudis@suse.cz
- Re-enable the DNS fixpack, add real fix for bnc#441947
- nscd: Fix gc he_data overflow and turn of alloca usage [bnc#387202]
- Normalize the patch queue
- Increate CPU_SETSIZE from 1024 to 4096 - for large SGI machines
  and compatibility with SLE10SP2 [bnc#464667]
* Fri Jan 16 2009 pbaudis@suse.cz
- Disable the DNS fixpack from CVS, breaks nscd resolving [bnc#441947]
* Fri Jan  9 2009 pbaudis@suse.cz
- Fix unsetenv() crashing after clearenv()
- Backport various DNS-related fixes from CVS
- Disable parallel A-AAAA queries - revert to the glibc-2.9 behaviour
  of sending the queries sequentially [bnc#442572]
- Fix fp register protection on power6 setcontext() [bnc#464738]
* Mon Dec  8 2008 pbaudis@suse.cz
- Fix crash when parsing NIS hosts file [bnc#448702]
* Thu Dec  4 2008 pbaudis@suse.cz
- Fix nscd gc-cacheadd and prune-invalidate races [bnc#446233]
- Remove obsolete and disabled nscd patch
- Fix /etc/bindresvport.blacklist parsing [bnc#439146]
- Fix en_GB and csb_PL locales having weeks start on Tuesdays [bnc#449186]
* Sun Nov 23 2008 pbaudis@suse.cz
- clean up patch filenames
* Fri Nov 21 2008 olh@suse.de
- fix check-build.sh to actually allow building on ppc* again
* Fri Nov 21 2008 olh@suse.de
- do not run compiled binaries with cputuned libraries [bnc#445911]
  the buildhost only generates them, but must not execute them
* Mon Nov 17 2008 pbaudis@suse.cz
- Update to glibc-2.9 final
* Thu Nov  6 2008 jjolly@suse.de
- Added patch to fix s390x 64bit stack frame problem
* Fri Oct 31 2008 matz@suse.de
- Fix atomics on s390/s390x, leading to failures in pthread mutexes.
* Tue Oct 28 2008 schwab@suse.de
- Restore alignment patch.
* Tue Oct 28 2008 olh@suse.de
- symlink power5/power5+ to power4 on ppc32
* Thu Oct 23 2008 ro@suse.de
- only do obsoletes for XXbit packages on ppc, not on x86
* Thu Oct 23 2008 matz@suse.de
- Deactivate unwind tables for initfini.c.
* Thu Oct 23 2008 olh@suse.de
- obsolete glibc{,-devel,-locale,-profile} -XXbit packages [bnc#437293]
- update last change to handle empty rpmspec defines
* Tue Oct 21 2008 olh@suse.de
- update glibc_post_upgrade to handle power7 and ppc-cell-be
- remove libs unconditionally on powerpc to handle libs in
  subdirectories for which we do not ship optimized versions [bnc#437209]
* Thu Oct 16 2008 olh@suse.de
- enable cpu-tuned libraries for power4, power6 and ppc-cell-be
  on ppc32 [fate#304000,bnc#408808]
* Mon Oct 13 2008 matz@suse.de
- Update to trunk from 20081013, no longer export paccept,
  obsoletes some patches (lowlevellock and res_send).
  Fix overflow in nscd patch.
* Sun Oct 12 2008 dmueller@suse.de
- also strip .comment.SUSE.OPTs from the static libs
- add missing-include-build-fix.diff
* Tue Oct  7 2008 schwab@suse.de
- Fix alignment in resolver.
* Tue Sep 30 2008 olh@suse.de
- symlink tuned libs because rpmlint is unable to handle hardlinks
* Mon Sep 29 2008 olh@suse.de
- enable cpu-tuned libraries for power6 and ppc-cell-be [fate#304000,bnc#408808]
- remove -mtune G4/RS64 from CFLAGS on ppc/ppc64
* Thu Sep 18 2008 rguenther@suse.de
- Change .comment.SuSE.OPTs to .comment.SUSE.OPTs.  Also strip from
  crti.o and crtn.o.
- Adjust getgroups fortify check.  [bnc#426958]
* Thu Sep 18 2008 schwab@suse.de
- Fix alignment in resolver.
* Mon Sep 15 2008 adrian@suse.de
- do not run glibc_post_upgrade in XEN builds
* Sat Sep 13 2008 rguenther@suse.de
- strip .note.gnu.build-id from crt1.o.
* Sat Sep 13 2008 dmueller@suse.de
- only strip comment from crt1.o
* Fri Sep 12 2008 mls@suse.de
- block baselibs generation for i586 (use i686 instead)
* Fri Sep 12 2008 dmueller@suse.de
- strip suse opts comments from crt.o (fate #300498)
* Mon Sep  8 2008 rguenther@suse.de
- Add a patch to fix wrong asm constraints in the i386 lowlevel
  futex lock implementation.
* Thu Sep  4 2008 matz@suse.de
- Update to current trunk snapshot (2008-09-04), makes s390 build.
* Wed Aug 27 2008 matz@suse.de
- Add a patch to use RTLD_DEEPBIND to load nss modules.  [bnc #157078]
* Mon Aug 25 2008 matz@suse.de
- Update to current trunk snapshot (2008-08-25).
* Fri Aug 22 2008 prusnak@suse.cz
- enabled SELinux support [Fate#303662]
* Fri Aug 15 2008 aj@suse.de
- Fix nscd init script for new checks.
- package empty ld.so.cache.
* Fri Jul 25 2008 aj@suse.de
- Add directory /var/cache/ldconfig to filelist.
* Wed Jun 25 2008 pbaudis@suse.cz
- Fix strptime() %%EY era support [bnc#355887]
* Wed Jun 25 2008 rguenther@suse.de
- Add patch to fix unwinding through clone for x86_64 and i386.
  [bnc#290807, bnc#403464]
* Fri May 30 2008 schwab@suse.de
- Fix ppc atomic ops.
* Fri Apr 25 2008 pbaudis@suse.cz
- Update to glibc-2.8 CVS branch as of 2008042513 (post-glibc-2.8):
  * Faster sqrt and sqrtf implemention for some PPC variants.
* Thu Apr 10 2008 pbaudis@suse.cz
- Build fix after the revert
* Thu Apr 10 2008 pbaudis@suse.cz
- Temporarily revert a namespace leakage cleanup in glibc-2.8 since
  it causes a lot of build failures
* Thu Apr 10 2008 pbaudis@suse.cz
- Update to CVS HEAD as of 2008041002 (almost glibc-2.8):
  * New locales: bo_CN, bo_IN.
  * New encoding: HP-ROMAN9, HP-GREEK8, HP-THAI8, HP-TURKISH8.
  * Sorting rules for some Indian languages (Devanagari and Gujarati).
    Implemented by Pravin Satpute.
  * IPV6 addresses in /etc/resolv.conf can now have a scope ID
  * nscd caches now all timeouts for DNS entries
    Implemented by Ulrich Drepper.
  * nscd is more efficient and wakes up less often.
    Implemented by Ulrich Drepper.
  * More checking functions: asprintf, dprintf, obstack_printf, vasprintf,
    vdprintf, and obstack_vprintf.
    Implemented by Jakub Jelinek.
  * Faster memset for x86-64.
    Implemented by Harsha Jagasia and H.J. Lu.
  * Faster memcpy on x86.
    Implemented by Ulrich Drepper.
  * ARG_MAX is not anymore constant on Linux.  Use sysconf(_SC_ARG_MAX).
    Implemented by Ulrich Drepper.
- memset() implementation from AMD replaced by new upstream
  implementation
- Enable stackguard randomization
* Thu Apr 10 2008 ro@suse.de
- added baselibs.conf file to build xxbit packages
  for multilib support
* Thu Mar 13 2008 dmueller@suse.de
- don't try to use ipv6 in getaddrinfo() unconditionally (bnc#361697)
* Sat Jan 26 2008 aj@suse.de
- Add fixes from 2.8 branch for string.h and headers.  The string.h
  changes should fix sysvinit compilation.
- Fix some rpmlint warnings:
  * Add versioned provides and obsoletes.
  * Use %%check section.
* Fri Jan 18 2008 schwab@suse.de
- Fix use of fnstsw.
* Wed Nov 21 2007 pbaudis@suse.cz
- Update to CVS branch glibc_2.7 as of today
- Remove the ::1 hack again
- Fix nscd race condition between mempool_alloc() and gc() [#288910]
- Fix nscd stack overflow in gc() when the number of cache entries
  gets too large [#288910]
- Fix sunrpc udp client [#257745]
- Fix initgroups() crash in nss_compat [#310544]
* Thu Sep 27 2007 matz@suse.de
- Remove broken x86_64 bcopy function.  [#328486]
* Sun Sep 16 2007 aj@suse.de
- Install gai.conf on all archs.
* Thu Sep 13 2007 pbaudis@suse.cz
- Install the default gai.conf to /etc instead of docdir
- Do not return ::1 /etc/hosts records for 127.0.0.1 queries [#288879]
* Wed Sep  5 2007 pbaudis@suse.cz
- Fix pthread_mutex_unlock() internal lock handling for incorrectly
  written applications
- Fix nscd assertion failure when opening persistent db failed [#288910]
* Mon Sep  3 2007 mls@suse.de
- fix typos in mdns patch [#206247], [#256690]
* Fri Aug 31 2007 aj@suse.de
- Add /usr/share/locale-bundle/ as fallback directory for usage
  with bundle-lang packages (see bug #302270).
* Tue Aug 21 2007 pbaudis@suse.cz
- Reintroduce errorneously omitted AMD64 optimizations
  (and fix them to compile again)
* Thu Aug 16 2007 aj@suse.de
- Fix static linking (#233835).
- Silence rpmlint for profile package.
- Add Short-Description to nscd init script.
* Mon Aug 13 2007 aj@suse.de
- Backport:
  * Update kernel-features.h.
  * Fix sched_getcpu error path on x86-64.
  * Use upstreamed version of ldconfig cache speed ups.
* Fri Aug  3 2007 pbaudis@suse.cz
- Remove the dynamic linker support for direct bindings (-Bdirect);
  according to Michael Meeks, the performance improvement is too small
- Split timezone package to a separate physical package
- Update glibc to version 2.6.1
* Fri Jul 27 2007 aj@suse.de
- Use fdupes to reduce timezone data size.
* Fri Jul 13 2007 schwab@suse.de
- Update to head of glibc-2.6 branch.
- Fix update on ppc.
* Thu Jun 28 2007 matz@suse.de
- Add a provide for "rtld(GNU_HASH)".
* Sun Jun 17 2007 schwab@suse.de
- Fix section selection in crt objects.
* Thu Jun  7 2007 pbaudis@suse.cz
- Backport fix for crashing printf() of some invalid ldouble values
* Wed May 30 2007 pbaudis@suse.cz
- Added few fixes from 2.6 CVS before 2.6.1 gets released
* Fri May 18 2007 pbaudis@suse.cz
- Update glibc to version 2.6
- Update tzdata to version 2007f
* Fri May  4 2007 schwab@suse.de
- Update build checks.
* Fri Apr 20 2007 dmueller@suse.de
- only keep symtab for libpthread*
* Thu Apr 19 2007 pbaudis@suse.cz
- Fix strtod() exponent limit calculations [#230909]
- Fix random nscd crashes under very heavy passwd/group queries
  load [#192391]
- Add some enums from CVS to sys/personality.h [#253710]
- Fix pthread_atfork()-induced hangs in threaded programs [#256237]
- Fix llrintl() on ppc64 [#241183]
- Fix makecontext() segfault [#249780]
- Fix potential dladdr() breakage [#241464]
- Fix some races in client programs with nscd garbage collection [#252138]
* Fri Mar 30 2007 pbaudis@suse.cz
- Update localtime during timezone update [#239888]
* Sun Mar 25 2007 olh@suse.de
- temporary disable powerpc cputuned libs to reduce turnaround time
* Fri Feb  9 2007 pbaudis@suse.cz
- Update to the latest upstream timezone data [#231833]
* Fri Feb  2 2007 schwab@suse.de
- Remove -ffortify.
* Thu Feb  1 2007 schwab@suse.de
- Remove -fstack-protector.
* Mon Jan 29 2007 sbrabec@suse.cz
- Removed references to /opt/gnome.
* Thu Jan 25 2007 olh@suse.de
- link power4 to ppc970, link power6 to power6x
- Update the powerpc cpu-tuned environment to v0.05
* Mon Jan 22 2007 schwab@suse.de
- Update ppc build check.
* Fri Nov 24 2006 pbaudis@suse.cz
- Fix for Brazilian and Wester Australia timezone DSTs [#213375,#223196]
* Thu Nov 16 2006 pbaudis@suse.cz
- Disable power6 optimization for 10.2, not all pieces are there
  [#219962]
* Wed Oct 25 2006 pbaudis@suse.cz
- Change ld.so madvise() call to posix_fadvise()
- Fix mallopt(M_MXFAST,0) behaviour [#198760]
- Update the powerpc cpu-tuned environment to v0.04 [#215117]
* Sun Oct 22 2006 pbaudis@suse.cz
- Update the powerpc cpu-tuned environment to v0.03 [#212549]
- Improve glibc powerpc optimization [#212548,#212580,#214282]
* Tue Oct 17 2006 mls@suse.de
- add ldconfig-old-cache patch to speed up ldconfig
* Sat Oct 14 2006 olh@suse.de
- dont use uninitialized (and wrong) variable in glibc-2.4.90-bdirect.diff
  [#212470]
* Thu Oct 12 2006 pbaudis@suse.cz
- Update to the latest 2.5 CVS
- More friendly -Bdirect behaviour in case of missing libraries
- Fix 2.4.90-nscd patch wrt. new gcc
* Mon Oct  2 2006 aj@suse.de
- Fix warnings in testsuite (patch from CVS).
* Fri Sep 29 2006 pbaudis@suse.cz
- Update to 2.5 CVS - official release (only minimal changes in CVS
  since the last update)
- Fix a thinko in the -Bdirect patch
* Fri Sep 29 2006 dmueller@suse.de
- fix devel requires
* Fri Sep 29 2006 pbaudis@suse.cz
- Make the dynamic linker support direct bindings (Michael Meeks'
  Solaris-like -Bdirect with minor changes by me)
- Split the kernel headers to a new package (linux-kernel-headers)
* Wed Sep 27 2006 schwab@suse.de
- Fix broken assertion [#208189].
* Tue Sep 26 2006 pbaudis@suse.cz
- Fix mistake when removing some patches
* Mon Sep 25 2006 pbaudis@suse.cz
- Update to current CVS
* Sat Sep 23 2006 pbaudis@suse.cz
- Fix 64bit-cleanliness gcc warnings
* Thu Sep 21 2006 pbaudis@suse.cz
- Add /usr/lib{,64}/Xaw3d to /etc/ld.so.conf (by schwab@suse.de,
  from original STABLE) [#205169]
- Fix chown() instead of lchown() called in fchownat() emulation
  [#201751]
- Fix glob() overflowing stack when producing massive number of
  matches [#190458]
- Update to current CVS
* Wed Sep 20 2006 pbaudis@suse.cz
- Fix cut'n'paste error in a last-minute change
* Wed Sep 20 2006 pbaudis@suse.cz
- Update to current CVS
- Fix powerpc-cpu tarball extension
- Move crypt-blowfish to a patch so that quilt works on the tree
* Sat Sep  2 2006 schwab@suse.de
- Use asm-powerpc for ppc and ppc64.
- Fix chroot check in glibc_post_upgrade.
* Sun Aug 27 2006 pbaudis@suse.cz
- Update to current CVS, should fix false positive heap overflow
  trigger from malloc() causing gcc to hang [#201724]
* Wed Aug 23 2006 pbaudis@suse.cz
- Update the powerpc cpu-tuned environment to v0.02 [#199274]
- Update to current CVS
- Drop pthread_mutexattr_getprioceiling() out of range fix
* Thu Aug 10 2006 pbaudis@suse.cz
  Ported from STABLE:
- Remove libc5 reference from /etc/ld.so.conf, shlibs5 is no longer
  supported [#181947]
- Fix name of a dummy ia64 header from offsets.h to asm-offsets.h
  [#191394]
* Sun Jul 30 2006 pbaudis@suse.cz
- Update to current CVS snapshot (highlight: support for .gnu.hash
  fast linking support)
* Wed Jul 12 2006 pbaudis@suse.cz
- pthread_mutexattr_getprioceiling() was returning prioceiling out
  of range [#182782]
* Wed Jul 12 2006 pbaudis@suse.cz
- Fix the HTML documentation missing an index [#190585]
* Sat Jun 24 2006 kukuk@suse.de
- Update to current CVS snapshot
- Update to kernel-headers 2.6.17
- Remove HZ define (sysconf(_SC_CLK_TCK) instead)
* Tue Jun 13 2006 kukuk@suse.de
- Update to current CVS snapshot
* Tue May  2 2006 pbaudis@suse.cz
- Fix bad memset() size in NTPL code [#159184]
* Tue May  2 2006 kukuk@suse.de
- asm-powerpc/page.h: Don't move #ifdef __KERNEL__ [#171010]
- bits/mman.h: Fix value of MADV_REMOVE [#170734]
* Mon Apr 24 2006 kukuk@suse.de
- Fix china timezone mess [#163674]
- Don't include linux/stddef.h [#167964]
* Tue Apr 18 2006 schwab@suse.de
- Sanitize <asm-powerpc/sigcontext.h>.
* Thu Apr 13 2006 kukuk@suse.de
- Add /usr/include/linux/taskstats.h [#141936]
* Tue Apr 11 2006 kukuk@suse.de
- nptl/init.c (sigcancel_handler): Compare with correct PID even
  if the thread is in the middle of a fork call.
  (sighandler_setxid): Likewise.
* Sun Apr  9 2006 ihno@suse.de
- S390 fix for startupcode. Part of it was not PIC.
* Fri Apr  7 2006 kukuk@suse.de
- Don't hardcode syscall numbers at our own
* Tue Apr  4 2006 schwab@suse.de
- Fix readlink declaration.
* Mon Apr  3 2006 kukuk@suse.de
- Update from CVS:
  - nscd bug fixes
  - Match return value of readlink to what POSIX says
  - Fix NIS+ checks for NULL pointer
* Sun Apr  2 2006 dmueller@suse.de
- Fix clients crash if nscd is unresponsive (glibc-#2501)
- Fix fd leak in nscd daemon (glibc-#2498)
* Sat Apr  1 2006 schwab@suse.de
- Fix <sys/procfs.h> on ppc64.
* Fri Mar 31 2006 schwab@suse.de
- Extend last change also to ppc64.
* Sun Mar 26 2006 schwab@suse.de
- Terminate FDE before clone syscall.
* Fri Mar 24 2006 ro@suse.de
- kernel-headers: asm-powerpc: define PAGE_MASK in page.h
* Mon Mar 20 2006 kukuk@suse.de
- Update to final 2.6.16 kernel headers
- Update to current CVS:
  - Don't use TLS before setting it up.
  - Fix rounding of long doubles on ppc64.
  - Correct usage of cfi_offset on ppc/ppc64.
  - Fix memory leak in dlopen.
* Tue Mar 14 2006 schwab@suse.de
- Update fnmatch patch.
* Fri Mar 10 2006 kukuk@suse.de
- Disable unshare() syscall (request of kernel developer)
* Fri Mar 10 2006 kukuk@suse.de
- Don't include linux/interrupt.h from linux/rtc.h
- Revert last change to linux/input.h, disable struct
- Update to current CVS (nptl/ia64 fix)
* Thu Mar  9 2006 kukuk@suse.de
- Fix linux/input.h for userspace inclusion
* Wed Mar  8 2006 kukuk@suse.de
- Update kernel headers to 2.6.16-rc5
* Mon Mar  6 2006 kukuk@suse.de
- Update to 2.4 CVS
  - official release
  - Fix free on ppc [#155374]
  - Various sysconf() fixes
* Sat Mar  4 2006 kukuk@suse.de
- Update to CVS
  - ldconfig/prelink fixes
* Fri Mar  3 2006 aj@suse.de
- Do not leave hyphens in defines in create_biarch_asm.sh (Bug 154998).
* Fri Mar  3 2006 kukuk@suse.de
- Update to CVS
  - Fix ftw test suite failures
  - Fix alignment of malloc for long double
* Thu Mar  2 2006 kukuk@suse.de
- Update to 2.3.91 CVS
  - Fix 6arg syscall on s390x
  - memccpy fix on IA64
* Wed Mar  1 2006 kukuk@suse.de
- Update to CVS
  - New linkat interface
  - Update from tzdata2006b
  - Define MADV_DONTFORK and MADV_DOFORK.
  - Add robust mutex to NPTL
* Thu Feb 23 2006 kukuk@suse.de
- Reenable power optimized code again [#142839]
* Sun Feb 12 2006 kukuk@suse.de
- Disable power optimized code
* Thu Feb  9 2006 kukuk@suse.de
- Update to CVS
  - Add unshare prototype
  - Add memory barrier on i386/NPTL
* Wed Feb  8 2006 kukuk@suse.de
- Remove glibc-2.4-ppc-dl-procinfo-20060111.diff [#142839]
* Mon Feb  6 2006 kukuk@suse.de
- Update to current CVS
- Fix alternate locale search path patch [#147685]
* Thu Feb  2 2006 kukuk@suse.de
- Update to current CVS
* Mon Jan 30 2006 schwab@suse.de
- Fix the fix.
* Sat Jan 28 2006 kukuk@suse.de
- Fix seg.fault in __atfct_seterrno if called from futimesat()
- Apply patches for CPU-Tuned Environment on Power [#142839]
* Fri Jan 27 2006 kukuk@suse.de
- Cleanup ld.so.conf
* Thu Jan 26 2006 kukuk@suse.de
- Add alternate directory for translations [#144073]
* Wed Jan 25 2006 mls@suse.de
- converted neededforbuild to BuildRequires
* Mon Jan 23 2006 kukuk@suse.de
- Update to CVS
- Fix build_locales
* Sat Jan 21 2006 kukuk@suse.de
- Update to CVS
- Fix inclusion of sys/poll.h with _GNU_SOURCE defined
* Fri Jan 20 2006 kukuk@suse.de
- Update to CVS
* Sun Jan 15 2006 kukuk@suse.de
- Update to CVS (fix long double configure check)
* Sun Jan 15 2006 schwab@suse.de
- Readd <fmtmsg.h>, <ucontext.h>, <sys/ucontext.h>.
* Sun Jan 15 2006 schwab@suse.de
- Readd <alloca.h>.
* Sat Jan 14 2006 kukuk@suse.de
- Update to current CVS (long double support)
* Fri Jan 13 2006 aj@suse.de
- Fix x86-64 w_exp to not use extra plt.
* Thu Jan 12 2006 kukuk@suse.de
- Fix asm-s390/setup.h for userspace inclusion
- nsswitch.conf: Add nis to netgroup and automount entry
- Fix sys/procfs.h for ppc64
* Mon Jan  9 2006 kukuk@suse.de
- Update to current CVS (fix for pthread.h with -std=c99)
- Define PAGE_SIZE on POWER
- Don't include linux/sched.h in asm-power/elf.h
* Sun Jan  8 2006 kukuk@suse.de
- Fix linux/acct.h for userland inclusion
* Sun Jan  8 2006 kukuk@suse.de
- Update to current CVS
- Remove CHILD_MAX from kernel-headers
- Copy subdirectories of asm-power, too.
- Remove da_DK@euro (does not exist)
* Sat Jan  7 2006 kukuk@suse.de
- Fix ext2 kernel headers
- Update to current CVS
* Fri Jan  6 2006 kukuk@suse.de
- Update to kernel headers 2.6.15
* Thu Jan  5 2006 kukuk@suse.de
- Update crypt_blowfish to version 1.0
- Update to current CVS
- Adjust nscd patches
* Tue Jan  3 2006 aj@suse.de
- Update to current CVS to fix pthread.h on 64-bit systems for C++.
* Tue Jan  3 2006 aj@suse.de
- Enable string patch again.
- Update to current CVS.
* Wed Dec 21 2005 kukuk@suse.de
- Update to current CVS
- Fix ldconfig
* Sat Dec 17 2005 kukuk@suse.de
- Remove /usr/lib/zoneinfo
- Add /etc/localtime to filelist
- Update to current CVS
- Fix glibc #1978: statvfs does not know about current filesystems
- Fix wrong error return code of time() on x86
- Remove nscd_nischeck
- Remove audit from nfb
- Remove LinuxThreads
* Wed Dec 14 2005 kukuk@suse.de
- Update to current CVS
- Remove nscd_nischeck
- Remove audit from nfb
* Mon Dec  5 2005 kukuk@suse.de
- Update to current CVS
- Disable SELinux
* Thu Nov 24 2005 kukuk@suse.de
- Remove obsolete patches:
  - glibc-2.3.asprintf-error_handling.diff
  - glibc-2.3.90-missing-string_h.diff
* Wed Nov 23 2005 kukuk@suse.de
- Update to current CVS
- Don't install in parallel
* Fri Nov 18 2005 kukuk@suse.de
- Update to current CVS to fix compiling with new binutils
* Wed Nov 16 2005 kukuk@suse.de
- Update to current CVS
* Sat Nov  5 2005 kukuk@suse.de
- Update to current CVS
- Disable linuxthreads on POWER and Itanium
- Fix kernel-headers for userland inclusion
* Wed Nov  2 2005 kukuk@suse.de
- Update to current CVS
- Update to kernel-headers 2.6.14
* Tue Oct 18 2005 kukuk@suse.de
- Update to current CVS
* Sat Oct  1 2005 schwab@suse.de
- Update libm ULPs.
- Fix limit in getcwd.
* Thu Sep 15 2005 kukuk@suse.de
- Update to current CVS
- Update to final 2.6.13 kernel headers
- Adjust bindresvport.blacklist patch to check already ports >= 512
* Fri Aug 26 2005 kukuk@suse.de
- Update to current CVS
- init.d/nscd: Remove -S support (was removed from nscd)
* Fri Aug 19 2005 schwab@suse.de
- Fix comment.
* Fri Aug 19 2005 matz@suse.de
- Change .note.SuSE format [#105825].
* Fri Aug 19 2005 kukuk@suse.de
- Update to current CVS
- Move all obsolete libraries into own subpackage and document
  them as obsolete
* Fri Aug  5 2005 schwab@suse.de
- -mno-tls-direct-seg-refs is x86-only
* Tue Aug  2 2005 kukuk@suse.de
- Update to 2.3.x CVS branch
- Compile with -mno-tls-direct-seg-refs
* Fri Jul 29 2005 kukuk@suse.de
- Update to current CVS head
* Wed Jul 27 2005 schwab@suse.de
- Update libm ULPs for ia64.
- Add linuxthreads stack guard support for ia64.
* Sat Jul 23 2005 kukuk@suse.de
- Update to current CVS head (adds bits/wchar2.h)
- Move LinuxThreads version to obsolete/linuxthreads
* Thu Jul 21 2005 kukuk@suse.de
- Update to current CVS head
* Fri Jul 15 2005 schwab@suse.de
- Fix file list.
* Fri Jul 15 2005 aj@suse.de
- Fix amd64 string routines and math routines.
* Fri Jul 15 2005 kukuk@suse.de
- Use old LinuxThreads only for runtime and remove static version
  and headers
- Update to current CVS head
- Remove own texi2html
* Mon Jul 11 2005 kukuk@suse.de
- Install generic stdio-lock.h header file
* Sat Jul  9 2005 kukuk@suse.de
- Temporary disable AMD64 string optimization
* Wed Jul  6 2005 kukuk@suse.de
- Build and install ja_JP.SHIFT_JISX0213 locale [Bug #84030]
* Mon Jul  4 2005 kukuk@suse.de
- nss_compat: Preserve original return value [Bug #95033]
- Cleanup old Obsoletes/Requires
* Mon Jun 27 2005 kukuk@suse.de
- Update to current gilbc CVS snapshot
* Thu Jun 23 2005 kukuk@suse.de
- Update to current glibc CVS snapshot
- Adjust nscd patch
- Enable SELinux/audit support for nscd
- Update .note.SuSE version
* Mon Jun 20 2005 kukuk@suse.de
- Update to final 2.6.12 kernel headers
- Update to current glibc CVS snapshot
* Thu Jun 16 2005 kukuk@suse.de
- Update linux/audit.h header
- Update to current CVS snapshot
* Mon Jun 13 2005 kukuk@suse.de
- Don't terminate strings twice in nis/netgroup code.
- netinet/if_tr.h: don't include kernel headers.
* Wed Jun  8 2005 kukuk@suse.de
- Update to kernel-headers 2.6.12-rc6
- Fix build on s390 and s390x
* Wed Jun  8 2005 matz@suse.de
- Don't strip .symtab from libpthread.so.0 (and other libs).
  Fixes debugging of threaded programs (#81253).
* Mon Jun  6 2005 kukuk@suse.de
- Update to current CVS head
- Update to kernel-headers 2.6.12-rc5
* Thu May 26 2005 schwab@suse.de
- No longer build loadlocale.c with -fno-unit-at-a-time.
* Tue May 24 2005 kukuk@suse.de
- Update to current CVS head, obsoletes:
  - glibc-2.3.90-libm.diff
  - glibc-2.3.90-i386-sysdep.diff
  - warn.diff
  - dl-osinfo.diff
  Adjusted:
  - glibc-2.3.90-bindresvport.blacklist.diff
* Sat May 21 2005 schwab@suse.de
- Fix missing include.
* Tue May 17 2005 schwab@suse.de
- Fix warning.
* Fri Apr 29 2005 kukuk@suse.de
- Update to latest CVS snapshost
* Sat Apr 23 2005 kukuk@suse.de
- Fix all the archs using wrong FLAGS
* Sat Apr 23 2005 kukuk@suse.de
- Remove -D_FORTIFY_SOURCE from RPM_OPT_FLAGS
* Fri Apr 22 2005 kukuk@suse.de
- Update to current CVS
* Fri Apr 22 2005 kukuk@suse.de
- Check if nice value does not conflict with test suite
* Tue Apr 19 2005 mls@suse.de
- resolv: trigger re-read of /etc/resolv.conf for all threads if
  a change is detected
- nscd: support a negative timeout of zero, used by hosts cache
* Mon Apr 18 2005 meissner@suse.de
- Enable fortify possibility even for GCC 4.0, we apply
  the necessary patch to the SUSE GCC 4.0.
* Sat Apr 16 2005 aj@suse.de
- Apply amd64 string diff again.
* Tue Apr 12 2005 kukuk@suse.de
- Update kernel-headers to 2.6.10
- Update to current CVS snapshot
* Wed Apr  6 2005 schwab@suse.de
- Cleanup neededforbuild.
* Tue Apr  5 2005 aj@suse.de
- Add gettext-devel to neededforbuild.
* Tue Apr  5 2005 aj@suse.de
- Do not build on xen machines.
- Adjust libm ULPs for PowerPC.
* Fri Apr  1 2005 kukuk@suse.de
- Update to current CVS snapshot
* Tue Mar 29 2005 kukuk@suse.de
- Update to current CVS snapshot
- Fix compiler warnings on ix86
* Mon Mar 28 2005 kukuk@suse.de
- Update to current CVS snapshot
- Disable Noversion Patch on i386 temporary
* Fri Mar 18 2005 ro@suse.de
- check-build.sh: require 2.6.11 on x86,x86_64 for build
* Thu Mar 17 2005 mls@suse.de
- nscd: enable hosts cache
- nscd: also watch /etc/resolv.conf
- nscd: check files every 3 seconds
- nscd: deal correctly with missing files
* Tue Mar 15 2005 kukuk@suse.de
- Update to current CVS snapshot.
* Sat Feb 12 2005 schwab@suse.de
- Remove const from __pthread_internal_tsd_address.
* Fri Feb 11 2005 schwab@suse.de
- Update to current CVS.
- Link glibc_post_upgrade against new libc.
- Fix build error with gcc4.
* Wed Feb  9 2005 kukuk@suse.de
- Update to current CVS
- Fix bindresvport blacklist handling.
- Increase buffer in tst-cancel17 to match new kernel buffer size
  [#50277]
- Enable LinuxThreads again
- Remove ia64-audit patch (is upstream)
* Mon Feb  7 2005 aj@suse.de
- Add patch to allow compilation with gcc4.
* Mon Jan 31 2005 schwab@suse.de
- Readd support for LD_AUDIT on ia64.
* Mon Jan 31 2005 kukuk@suse.de
- Add memory clobber to string inline assemblies on s390 [#50284].
* Mon Jan 31 2005 kukuk@suse.de
- Fix filelist on i686 if we build NPTL only version
* Mon Jan 31 2005 aj@suse.de
- Generate new ULPs file for i386 needed by GCC 4.
* Sun Jan 30 2005 kukuk@suse.de
- Include own copy of texi2html
- Add glibc_pst_upgrade program (based on version from FC3)
- Update to current CVS
* Thu Jan 27 2005 kukuk@suse.de
- Re-add patch for timezone/zic.c (got lost with last merge)
* Wed Jan 26 2005 kukuk@suse.de
- Update timezone data to 2005c release (fixes zdump crash on
  64bit architectures)
* Sat Jan 22 2005 schwab@suse.de
- Add basic (incomplete) support for LD_AUDIT on ia64.
* Fri Jan 21 2005 kukuk@suse.de
- Enable patch for [Bug #49833]
- Allow to build NPTL only glibc
- Update to CVS from Jan 16, 2005, containing:
- Fix execlp argument in SunRPC code [glibc #681]
- Fix errno return values for futimes [glibc #633]
- Update FPU function on PPC/PPC64 [Bug #49764]
* Mon Jan 17 2005 aj@suse.de
- Enable amd64 string patch again after fixing failing hunks.
- Handle missing cpuid better for amd64 string functions. [#49803]
* Sat Jan 15 2005 aj@suse.de
- Fix amd64 string patch to use correct datatype.
* Fri Jan 14 2005 kukuk@suse.de
- Fix memory corruption in getgrouplist function [Bug #49833]
* Tue Jan 11 2005 kukuk@suse.de
- Enable all LinuxThreads tests again
- Finalize getconf -a patch (make it compatible with Solaris)
- Rewrite getconf manual page and mention new option
- Merge GB18030 patches into one.
* Thu Dec 30 2004 kukuk@suse.de
- Merge kernel-headers.remove-SO_BSDCOMPAT.diff with
  kernel-headers.SuSE.diff
- Revert nscd paths on old SuSE Linux distributions
* Wed Dec 29 2004 kukuk@suse.de
- Update to glibc 2.3.90 CVS branch
- Remove alarm-round.patch (merged upstream)
* Thu Dec  9 2004 kukuk@suse.de
- Update to current CVS
- Move nscd persistent database files back to /var/run/nscd
* Mon Dec  6 2004 kukuk@suse.de
- Update to current CVS
- Fix more kernel headers for userland inclusion
* Fri Nov 26 2004 ro@suse.de
- kernel-headers.diff: define __force in compiler.h
* Thu Nov 25 2004 schwab@suse.de
- Add Intel libm update.
* Thu Nov 25 2004 kukuk@suse.de
- Update to current glibc CVS
- Update kernel headers to 2.6.9
* Thu Nov 18 2004 kukuk@suse.de
- Update to current glibc CVS
* Mon Nov 15 2004 kukuk@suse.de
- Update to current glibc CVS
* Mon Nov  8 2004 kukuk@suse.de
- Blacklist port 921 (lwresd) for usage by bindresvport()
- Update to current glibc CVS
- Add /var/run/nscd/* files as ghost entries
* Mon Oct 18 2004 aj@suse.de
- Don't use special fdim functions for x86-64 since those give
  wrong results for fdim (inf,inf).
- Fix ppc64 rebuild issue with ppc32 system [#47325].
* Wed Oct 13 2004 kukuk@suse.de
- Fix symlink librt.so -> tls/librt.so.1
- Backout last glob changes
- Disable nptl as default for linking
* Tue Oct 12 2004 kukuk@suse.de
- Install kernel-headers after merging linuxthreads/NPTL headers
* Tue Oct 12 2004 kukuk@suse.de
- Update to current CVS snapshot
* Mon Oct 11 2004 kukuk@suse.de
- Make NPTL default for linking
* Fri Oct  1 2004 kukuk@suse.de
- Update to current CVS snapshot
- Add workaround for linuxthreads/without-__threads bug
- Remove KDE/ldconfig workaround
- Set SuSE ABI note to 9.3
* Sun Sep 26 2004 kukuk@suse.de
- Add lib/nptl/librt.so symlink to tls/librt.so.1
- Update to current CVS
* Sat Sep 25 2004 kukuk@suse.de
- Implement mq support for rtkaio and enable it again
* Fri Sep 24 2004 kukuk@suse.de
- Add zh_SG.UTF-8 [Bug #46024]
* Thu Sep 23 2004 kukuk@suse.de
- Sync sys/mount.h and linux/fs.h
* Wed Sep 22 2004 kukuk@suse.de
- Make mdns support configurable in /etc/host.conf
* Mon Sep 20 2004 kukuk@suse.de
- Fix invalidating of nscd caches and getaddrinfo()
* Fri Sep 17 2004 kukuk@suse.de
- Update to current CVS (nscd and glob.h fixes)
* Thu Sep 16 2004 kukuk@suse.de
- Fix getaddrinfo/nscd support
* Wed Sep 15 2004 kukuk@suse.de
- Update to current CVS, remove merged patches
* Sun Sep 12 2004 kukuk@suse.de
- Update to CVS: Fix nscd crash if one service is disabled
- glob.h: Add workaround for invalid prototypes
- nss_compat: Check that buffer is larger than 0 bytes
* Fri Sep 10 2004 kukuk@suse.de
- Update to CVS: Fix cdefs.h for C++ usage
* Fri Sep 10 2004 kukuk@suse.de
- Add lwres to hosts search order in nsswitch.conf
- Update to current CVS
- Use new nscd paths for socket/pid file
- Enable NPTL on i586
- Add --print-all option to getconf
* Thu Sep  2 2004 kukuk@suse.de
- Fix NPTL header files on x86-64 for 32bit compilation
- Apply various fixes from CVS
- Remove pre-Install for -devel package (no longer necessary)
* Wed Aug 25 2004 kukuk@suse.de
- Don't compile in eval.c
* Mon Aug 23 2004 kukuk@suse.de
- Update to current CVS:
  - Use CVS version for last fix.
  - Add malloc sanity checks for double free.
* Fri Aug 20 2004 schwab@suse.de
- Fix cancellable syscalls in librt w/ linuxthreads.
* Fri Aug 20 2004 kukuk@suse.de
- Update to current CVS [#43993]
* Thu Aug 19 2004 schwab@suse.de
- Better fix for asm-ia64/gcc_intrin.h.
* Wed Aug 18 2004 kukuk@suse.de
- Update to current CVS
- Workaround linux/ixjuser.h problem not including compiler.h
- Workaround linux/capi.h problem not including compiler.h
* Tue Aug 17 2004 schwab@suse.de
- Fix asm-ia64/gcc_intrin.h.
* Tue Aug 17 2004 kukuk@suse.de
- Workaround broken linux/crc-ccitt.h for usage from glibc [#43884]
* Mon Aug 16 2004 kukuk@suse.de
- Update to current CVS version and update to
  kernel-headers 2.6.8.1 to fix NPTL deadlock problems
- Disable linuxthreads/tst-clock1, does not work on i586 and lower
* Fri Aug 13 2004 kukuk@suse.de
- Update to current CVS version
- Cleanup/adjust all patches
* Thu Jun 17 2004 kukuk@suse.de
- Fix sched_setaffinity return/errno code in error case [#42124]
* Wed Jun 16 2004 meissner@suse.de
- Updated altivec set/get/swapcontext patch to fix
  more problems on altivec capable machines [#42039].
- glibc is also able to build on -pmac64 kernels.
* Mon Jun 14 2004 kukuk@suse.de
- Add fixes from CVS:
  - sysconf return value fixes
  - nscd host caching deadlock
  - backtrace for s390/s390x/ia64 static linked binaries
* Mon Jun 14 2004 kukuk@suse.de
- Increase listen backlog in RPC code [#41955]
* Wed Jun  9 2004 meissner@suse.de
- Fixed typos in powerpc* *context functions to not destroy the r19
  register and save the v19 register correctly. [#41790]
* Sat Jun  5 2004 aj@suse.de
- Fix makecontext with more than 6 arguments on x86-64 [#40546].
* Mon May 24 2004 kukuk@suse.de
- Update to kernel-headers 2.6.6
- Update to current glibc CVS
- Disable rtkaio temporary
* Sun May 23 2004 kukuk@suse.de
- Fix pthread_cond_wait on not ix86 and x86-64 architectures
* Thu May 20 2004 kukuk@suse.de
- Add PPC64 kernel header file fixes [#40831,#40870]
* Wed May 19 2004 kukuk@suse.de
- Add additional NPTL fixes from CVS
* Tue May 18 2004 schwab@suse.de
- Fix mapping of DSOs with holes.
* Fri May 14 2004 schwab@suse.de
- Fix rounding in alarm [#40552].
* Wed May 12 2004 schwab@suse.de
- Fix uninitialized array in regexp compiler [#40009].
* Tue May 11 2004 kukuk@suse.de
- Apply lot of fixes from current CVS
- Fix alignment of stack for makecontext on x86-64 [Bug #39413]
- Make XTABS identical to TABDLY on PPC
* Wed May  5 2004 kukuk@suse.de
- Add some header fixes to match POSIX
* Tue May  4 2004 meissner@suse.de
- Fix INLINE_SYSCALL on ppc and ppc64 (see #38399)
* Mon May  3 2004 kukuk@suse.de
- Port --mlock option for ld.so from UL1 [Bug #39569]
* Tue Apr 20 2004 kukuk@suse.de
- Add execstack fix for s390
* Mon Apr 19 2004 kukuk@suse.de
- Update to current CVS version (fix problems with new binutils
  and gcc)
* Mon Apr 19 2004 kukuk@suse.de
- Add pthread_getattr_np and syslog fixes from CVS
- Update gb18030 and big5hkscs gconv modules [Bug #39080]
* Sat Apr 17 2004 schwab@suse.de
- Pacify autobuild.
* Thu Apr 15 2004 kukuk@suse.de
- Add /etc/ld.so.conf.d/*.conf to /etc/ld.so.conf
- Disable FUTEX_REQUEUE support in NPTL library [Bug #38882]
* Wed Apr 14 2004 schwab@suse.de
- Remove /usr/i386-linux from ld.so.conf,
* Wed Apr 14 2004 kukuk@suse.de
- Fix linux/compiler.h for glibc inclusion
- Really fix ffsl on s390x
* Fri Apr  9 2004 schwab@suse.de
- Fix syntax error in memcmp.
* Fri Apr  9 2004 kukuk@suse.de
- Update from CVS: linuxthread debug fixes
- Fix INLINE_SYSCALL on x86-64 and ia64 (fixes #38399)
- Fix ffsl weak alias on s390x
- Update to 2.6.5 kernel headers
* Wed Apr  7 2004 meissner@suse.de
- forward umount to umount2 on ppc64 because umount syscall
  does not exist
* Mon Apr  5 2004 kukuk@suse.de
- Sync with current CVS (which is nearly identical with most of
  our latest patches)
* Fri Apr  2 2004 kukuk@suse.de
- Make fstatvfs64 working on 32bit architectures
- Add fwrite LFS fix from aj
- Add powerpc fixes from CVS
- Fix wrong errno code for shm_unlink [Bug #38013]
* Wed Mar 31 2004 kukuk@suse.de
- Add three fixes from CVS:
  - Call __nptl_deallocate_tsd for main thread, too
  - setgroups: optimizations for huge number of groups
  - initgroups: Limit the initial allocation to 64 entries
* Tue Mar 30 2004 kukuk@suse.de
- Add IUTF8 to bits/termios.h [Bug #34725]
- *affinitiy.c: Prepend GLIBC_ to version names
* Sat Mar 27 2004 kukuk@suse.de
- Fix wrong return value of getXXbyYY_r in case key was not
  found [Bug #37181]
- Fix typo in dl-open
- Don't set errno in NSS NIS module if group was not found
* Fri Mar 26 2004 aj@suse.de
- Add pow10/pow10f aliases for AMD64.
* Thu Mar 25 2004 kukuk@suse.de
- Update to latest CVS snapshot
* Sat Mar 20 2004 aj@suse.de
- Fix further problems with sched_[sg]etaffinity calls.
* Fri Mar 19 2004 aj@suse.de
- Fix sched_setaffinity compile problem.
* Fri Mar 19 2004 kukuk@suse.de
- Remove conflict with special aaa_base version (rpm will handle
  this with file conflict)
* Fri Mar 19 2004 kukuk@suse.de
- Add SuSE abi note
- Add madvise patch
- Update to current CVS
* Wed Mar 17 2004 kukuk@suse.de
- Use official NPTL version fix
- Add LD_DEBUG=statistic for x86-64
- Fix two JB_SIZE redefinitions on PPC
- Add two fixes for execstack
* Mon Mar 15 2004 kukuk@suse.de
- Require kernel 2.6.4 for glibc/NPTL
- Remove siginfo_t/si_band patch (needs to be fixed in kernel)
- Update linuxthreads_db and nptl_db
- Add libidn fixes from CVS
* Sat Mar 13 2004 kukuk@suse.de
- Adjust filelist (remove libcidn.a and libcidn_p.a)
* Sat Mar 13 2004 kukuk@suse.de
- Update to current CVS (merge of patches)
- Fix siginfo_t/si_band [Bug #34330]
* Thu Mar 11 2004 mls@suse.de
- Add mdns support to resolver library
* Thu Mar 11 2004 kukuk@suse.de
- Update kernel headers to 2.6.4
- Update to current glibc CVS
* Wed Mar 10 2004 bg@suse.de
- Update hppa patches.
* Mon Mar  8 2004 kukuk@suse.de
- Fix vDSO on IA64
* Mon Mar  8 2004 kukuk@suse.de
- Update from glibc CVS:
  - Add libidn
  - Lot of dynamic loader changes
* Thu Mar  4 2004 kukuk@suse.de
- Update from glibc CVS:
  - Fix posix_fadvise vs. posix_fadvise64
* Tue Mar  2 2004 kukuk@suse.de
- Fix PPC kernel header files
- Update from glibc CVS
  - NPTL fixes
  - nscd fixes
* Tue Mar  2 2004 schwab@suse.de
- Fix ppc32 to always use stat64 syscall.
* Sat Feb 28 2004 kukuk@suse.de
- Update from glibc CVS:
  - Fix compiler warnings
  - NPTL: Don't use CLONE_STOPPED
  - Revert vDSO changes
* Fri Feb 27 2004 kukuk@suse.de
- Update from glibc CVS:
  - Real NGROUP_MAX fix
  - Lot of NPTL fixes
  - clock_settime fix
- Add no_NO back (required by OpenI18N spec and [Bug #34745])
* Thu Feb 26 2004 kukuk@suse.de
- Workaround broken NGROUP_MAX function
* Thu Feb 26 2004 kukuk@suse.de
- Add insserv PreRequires for nscd
- Update from CVS: NPTL and getaddrinfo memory leak fixes,
  Get NGROUP_MAX from /proc filesystem
* Sat Feb 21 2004 kukuk@suse.de
- Update from CVS: IA64 relo fix, lot of PPC fixes
- Fix linux/mod_devicetable.h for userland inclusion
- Enable NPTL on Alpha
* Fri Feb 20 2004 schwab@suse.de
- Fix pthread_barrier_wait.
* Thu Feb 19 2004 kukuk@suse.de
- Update from CVS: More NPTL fixes
- nscd.init: Cleanups
* Wed Feb 18 2004 kukuk@suse.de
- Kernel Headers: Fix asm-ppc/unaliged.h, asm-ppc/types.h and
  asm-ppc/bitops.h for userland inclusion.
* Wed Feb 18 2004 bg@suse.de
- Update hppa patches for current glibc
- Add workaround for ICE in hppa
* Tue Feb 17 2004 kukuk@suse.de
- Update from CVS:
  - PPC nptl compatiblity fix
  - NSCD patches merged
  - ld.so.preload: Igreno missing files
  - getaddrinfo: Fix problem with IPv6 addresses
* Tue Feb 17 2004 aj@suse.de
- Fix string optimizations init code on AMD64.
* Sat Feb 14 2004 kukuk@suse.de
- Update from CVS (for NPTL fixes and new PPC longjmp)
- Fix nscd deadlock with kernel 2.6 [Bug #34507]
* Fri Feb 13 2004 aj@suse.de
- Add more string/memory optimizations for AMD64.
* Thu Feb 12 2004 kukuk@suse.de
- Fix regex bug with invalid UTF-8 strings
* Thu Feb 12 2004 aj@suse.de
- Add more string optimizations for AMD64.
* Wed Feb 11 2004 kukuk@suse.de
- Update to current CVS snapshot
- Fix _IOC_TYPECHECK on s390/s390x/parisc
- Compile rtkaio only with NPTL
- Remove glibcbug (was dropped since glibc uses bugzilla now)
- Disable parallel build on s390x
* Tue Feb 10 2004 schwab@suse.de
- Work around invalid use of kernel headers in some packages.
* Fri Feb  6 2004 kukuk@suse.de
- Add kernel stat fix for PPC
- Add fix for off-by-one error in regex code
* Thu Feb  5 2004 kukuk@suse.de
- Update kernel header files to 2.6.2
* Tue Feb  3 2004 bg@suse.de
- Update hppa patch
* Tue Feb  3 2004 kukuk@suse.de
- Fix ypclnt speedup patch
- Update to current CVS snapshot
* Mon Jan 26 2004 kukuk@suse.de
- Update to current CVS snapshot
* Wed Jan 21 2004 aj@suse.de
- Do not run in parallel on s390.
* Mon Jan 19 2004 ro@suse.de
- really fix linux/percpu.h to compile in userland
* Fri Jan 16 2004 kukuk@suse.de
- Update version.h to 2.6.1
- Fix linux/percpu.h to compile in userland
- Update to current CVS snapshot
- Revert nscd path changes
* Wed Jan 14 2004 kukuk@suse.de
- Update to Kernel Headers 2.6.1
- Update to current CVS snapshot
- Don't ignore make check on IA64 any longer
* Sat Jan 10 2004 schwab@suse.de
- Locale no_NO has been renamed to nb_NO.
* Fri Jan  9 2004 kukuk@suse.de
- Temporary ignore make check on IA64 and PPC
  (known kernel/compiler bugs)
* Thu Jan  8 2004 stepan@suse.de
- fix v4l2 headers
* Wed Jan  7 2004 kukuk@suse.de
- Update to glibc CVS from 20040107
- Remove manual pages which are now part of man-pages
* Thu Dec 18 2003 kukuk@suse.de
- Update to glibc CVS from 20031218
- Update ot final 2.6.0 kernel headers
- Fix syntax error in spec file
- Update HPPA patch
* Mon Dec 15 2003 kukuk@suse.de
- Update to glibc CVS from 20031215
* Fri Dec 12 2003 kukuk@suse.de
- Update to glibc CVS from 20031212
- disable rtkaio (does not work with kernel 2.6 yet)
* Fri Dec  5 2003 kukuk@suse.de
- Update to glibc 2.3.3 CVS
- Make an extra sub package for nscd
* Thu Nov 27 2003 kukuk@suse.de
- Update to glibc CVS from 20031127
- Add nsswitch.conf (moved from aaa_base)
- Add ld.so.conf (moved from aaa_base) [Bug #33277]
- Fix ceil on AMD64
* Fri Nov 21 2003 kukuk@suse.de
- Update to glibc CVS from 20031121
- Obsolete epoll (glibc-devel contains now this header files)
* Wed Nov 19 2003 kukuk@suse.de
- Add patch so that ld.so supports linuxthreads and nptl
- PPC64 requires kernel 2.4.21
- Update to glibc CVS from 20031119
* Fri Nov 14 2003 bg@suse.de
- Add hppa patches for current glibc.
* Fri Nov 14 2003 kukuk@suse.de
- Update to glibc CVS from 20031114
- Remove PPC64 symbol version patch
* Thu Nov 13 2003 schwab@suse.de
- Fix last change covering libNoVersion.
* Mon Nov 10 2003 schwab@suse.de
- Specfile cleanup.
* Thu Nov  6 2003 kukuk@suse.de
- Update to glibc CVS from 20031106
* Wed Nov  5 2003 uli@suse.de
- added a number of ARM fixes (glibc-2.3.2-armformat.patch,
  glibc-armisa.patch, glibc-sjlj.patch)
* Mon Oct 27 2003 kukuk@suse.de
- Update to glibc CVS from 20031027
- kernel headers: update to 2.6.0-test9
- nptl: Use -fno-unit-at-a-time on AMD64, too
* Thu Oct 23 2003 kukuk@suse.de
- Update to current CVS version
* Wed Oct 22 2003 kukuk@suse.de
- nptl: Fix prototype in configure script
- nptl: Use -fno-unit-at-a-time
- locale: Add nb_NO
* Fri Oct 17 2003 kukuk@suse.de
- Increase timeout for make check on overloaded architectures
* Thu Oct 16 2003 kukuk@suse.de
- Fix building as normal user
* Wed Oct 15 2003 kukuk@suse.de
- Fix putpwent/putgrent
- Make build as normal user
* Tue Oct 14 2003 kukuk@suse.de
- Update to glibc CVS from 20031014
* Tue Oct 14 2003 aj@suse.de
- Fix w_acosf for AMD64.
* Tue Oct 14 2003 kukuk@suse.de
- Fix asm-sparc/kbio.h to compile kbdrate/X11
* Mon Oct 13 2003 kukuk@suse.de
- Update to kernel header files from 2.6.0-test7
* Fri Oct 10 2003 schwab@suse.de
- Fix misnamed syscalls.
* Tue Oct  7 2003 kukuk@suse.de
- Update to glibc CVS from 20031007
* Fri Oct  3 2003 kukuk@suse.de
- Update to glibc CVS from 20031003
- Fix kernel ioctl header files for PPC/PPC64
- Removed dl-reloc.c fix, merged upstream
* Wed Oct  1 2003 kukuk@suse.de
- Update to glibc cvs from 20031001
- Remove sysmacros.h fix, merged upstream
- Add dl-reloc.c fix for compiler warnings
* Mon Sep 29 2003 kukuk@suse.de
- Update kernel-headers to 2.6.0-test6
- Fix sysmacros.h to compile with -ansi
* Sat Sep 27 2003 kukuk@suse.de
- Update to glibc CVS from 20030927
- Remove SO_BSDCOMPAT (obsoleted by kernel)
- Make _IOC_TYPECHECK useable for userland programs
* Fri Sep 26 2003 kukuk@suse.de
- Remove obsolete patches
- Update to crypt_blowfish 0.4.5
* Fri Sep 26 2003 kukuk@suse.de
- Update to glibc CVS from 20030926
- Update to nptl 0.60
* Sat Sep 20 2003 kukuk@suse.de
- Update to glibc CVS from 20030920
- Update to nptl 0.59
* Wed Sep 17 2003 kukuk@suse.de
- Disable TLS for i586 glibc [Bug #31034]
* Tue Sep 16 2003 kukuk@suse.de
- Remove db1.85
* Mon Sep 15 2003 kukuk@suse.de
- Update to glibc CVS from 20030915
* Mon Sep 15 2003 kukuk@suse.de
- Update to kernel 2.6.0-test5 header files
* Sat Sep 13 2003 kukuk@suse.de
- Add librtkaio, a librt using kernel aio
* Thu Sep 11 2003 kukuk@suse.de
- Update to current CVS
- Remove patches which are merged in CVS
- Update to nptl-0.58
- Remove _POSIX_VERSION hack
* Thu Sep 11 2003 kukuk@suse.de
- version.h: Fix comment how to compile kernel modules
* Wed Sep 10 2003 aj@suse.de
- Install en_US.ISO-8859-15 under this name [#30373].
* Tue Sep  9 2003 aj@suse.de
- Fix last patch.
* Mon Sep  8 2003 aj@suse.de
- Fix setting up of user signal handler in linuxthreads on x86_64.
* Fri Sep  5 2003 kukuk@suse.de
- Add dl-tls.c to fix out of memory with static TLS errors
* Wed Sep  3 2003 kukuk@suse.de
- Update glibc-linuxthreads
- Set _POSIX_VERSION back to 199209
* Mon Sep  1 2003 kukuk@suse.de
- Fix typo in nss_compat patch
* Mon Sep  1 2003 kukuk@suse.de
- Add opendir fix for usage with NPTL
- Fix getspnam/getspent in nss_compat [Bug #29689]
- Set _POSIX2_VERSION back to 199209
* Wed Aug 27 2003 kukuk@suse.de
- Update to current CVS version
* Mon Aug 25 2003 kukuk@suse.de
- Update to current CVS version
* Mon Aug 18 2003 aj@suse.de
- increase stack size for linuxthreads/set-context.
* Mon Aug 18 2003 aj@suse.de
- Fix mathinline.h for i386 to compile under C++ without warnings.
* Sun Aug 17 2003 ro@suse.de
- kernel-headers.dif: don't include device.h from videodev.h
* Fri Aug 15 2003 kukuk@suse.de
- Update to current cvs (fixes assert usage in C++ source code)
* Thu Aug 14 2003 kukuk@suse.de
- Initialize fp->_mode for glibc 2.0 compatibility [Bug #28386]
- On i686, include glibc with and without floating stack enabled
- Update to current cvs
- Update to nptl 0.56
* Thu Jul 31 2003 kukuk@suse.de
- Reenable new quota.h
* Wed Jul 30 2003 kukuk@suse.de
- Update to current cvs
- Update kernel headers to 2.6.0-test2
- Disable sys/quota.h update
- Enable TLS on AMD64 again
* Mon Jul 28 2003 kukuk@suse.de
- Update to current cvs
- Disable TLS on AMD64 temporary
- Update sys/quota.h to match new kernel implementation
* Fri Jul 25 2003 kukuk@suse.de
- Update to current cvs
- Update kernel headers to 2.6.0-test1
- Update to nptl 0.55
- Change minimal symbol version on PPC64 back to 2.2.5
* Wed Jul 23 2003 meissner@suse.de
- Fixed systemcall clobber lists for asm-ppc/unistd.h (by just
  merging over the asm-ppc64 things).
* Wed Jul 23 2003 aj@suse.de
- Handle in ldconfig konqueror.so and other similar KDE hacks that fail without
  rpath.
- Fix bugs exposed by unit-at-a-time option.
* Fri Jul 18 2003 aj@suse.de
- Fix compilation with unit-at-a-time enabled compiler.
- Enlarge stack for tst-setcontext test.
* Mon Jun 30 2003 kukuk@suse.de
- Update to current CVS (includes if_arp.h and nss_compat changes)
- Update to nptl 0.50
- Kernel header fixes for userland inclusion
* Wed Jun 25 2003 kukuk@suse.de
- Update to current CVS
- Add target host and CVS checkout to glibc version printout
- Adjust fnmatch fix
- Update kernel headers to 2.5.73
* Tue Jun 24 2003 kukuk@suse.de
- Fix typos in syscalls.list on Alpha
* Mon Jun 23 2003 kukuk@suse.de
- fix typo in linuxthreads on SPARC
- Update to current CVS
- Update to nptl 0.48
* Thu Jun 19 2003 kukuk@suse.de
- Disable nss_compat patch again
* Wed Jun 18 2003 kukuk@suse.de
- Fix reading of locale.alias file
- Update to current CVS snapshot
- Update to nptl 0.47
- Some kernel header file fixes for PPC64/IA64
- Update nss_compat patch
* Thu Jun 12 2003 kukuk@suse.de
- Update to current CVS snapshot
- Finish patch for printing linker warning
* Wed Jun 11 2003 kukuk@suse.de
- Update to nptl 0.45
- Update to current CVS snapshot
- Add patch to print linker warning, if a static binary calls
  functions using NSS
* Tue Jun 10 2003 kukuk@suse.de
- Use %%find_lang macro and cleanup glibc-locale filelist
* Thu Jun  5 2003 kukuk@suse.de
- Complete mathinline.h fixes
* Thu Jun  5 2003 kukuk@suse.de
- linux/compiler.h: Define all inline variants to __inline__
* Wed Jun  4 2003 kukuk@suse.de
- Update to current glibc CVS snapshot
- Update to nptl 0.43
- More kernel header fixes
- Make --no-archive default for localedef
* Tue May 27 2003 kukuk@suse.de
- Update to current glibc CVS snapshot
- Update to nptl 0.40
- Disable nss_compat patch
- Update to kernel headers from 2.5.70
* Fri May 23 2003 kukuk@suse.de
- Update to current glibc CVS snapshot
- Add patches to make nss_compat work with every service
* Mon May 19 2003 kukuk@suse.de
- Update to current glibc CVS snapshot
- Add patch for better binary compatibility (errno, h_errno)
* Fri May 16 2003 kukuk@suse.de
- Update to current glibc CVS snapshot
- Update to nptl 0.39
- Add URL tag
- asm-i386/byteorder.h: fix asm vs. __asm__
* Wed May 14 2003 schwab@suse.de
- Fix missing syscall numbers on ia64.
* Wed May 14 2003 kukuk@suse.de
- More kernel headers fixes for i386, ia64, ppc and s390
* Mon May 12 2003 kukuk@suse.de
- Update to current glibc CVS snapshot
- Fix lot of more kernel headers
- Adjust netinet/igmp.h patch
- Copy linux/version.h in place before compiling glibc
* Fri May  9 2003 kukuk@suse.de
- Fix bits/stdio.h
- Fix lot of kernel headers to work in userspace
- Fix netinet/igmp.h
- Enable TLS on PPC
* Wed May  7 2003 kukuk@suse.de
- Update CVS snapshot
- Enable TLS on IA64
- Update to nptl 0.37
- Update kernel-headers to 2.5.69
* Thu Apr 24 2003 ro@suse.de
- fix install_info --delete call and move from preun to postun
* Fri Apr 11 2003 aj@suse.de
- Do not build locales in parallel.
* Tue Apr  8 2003 kukuk@suse.de
- Disable TLS for PPC
* Mon Apr  7 2003 kukuk@suse.de
- Update CVS snapshot
* Sat Apr  5 2003 kukuk@suse.de
- Disable TLS for ia64
- Update to nptl 0.34
- Update to CVS snapshot
* Fri Apr  4 2003 kukuk@suse.de
- Remove libNoVersion from filelist on alpha
- Add nptl docu
* Fri Apr  4 2003 kukuk@suse.de
- Update to nptl 0.33
- Run ldconfig after installation
- Update to CVS snapshot
* Sat Mar 29 2003 kukuk@suse.de
- Update to CVS snapshot
- Add USAGI patches for kernel-headers
- Add nptl on i686
- Enable --with-tls
- Implement NI_NUMSERICSCOPE for getnameinfo
- Implement AI_V4MAPPED/AI_ALL for getaddrinfo
- Implement AI_ADDRCONFIG for getaddrinfo
- Add USAGI patch for kernel headers
* Mon Mar 24 2003 schwab@suse.de
- Sanitize <asm-ia64/pal.h>.
* Mon Mar 17 2003 aj@suse.de
- Fix tst-aio7 testcase and enable it again.
- Enable %%fs for x86-64.
- Install en_US.ISO-8859-15 for libstdc++.
- Fix bits/syscalls.h for x86-64.
* Thu Mar 13 2003 kukuk@suse.de
- Revert tcgetattr change [Bug #25173/#25174]
* Tue Mar  4 2003 aj@suse.de
- Fix <tgmath.h> lround/lrint functions.
* Sat Mar  1 2003 aj@suse.de
- Update to 2.3.2 release.
* Wed Feb 26 2003 kukuk@suse.de
- Update to current cvs snapshot (fixes OOo and mysql problems)
* Mon Feb 24 2003 kukuk@suse.de
- Disable aio7 test on s390(x), too.
* Mon Feb 24 2003 kukuk@suse.de
- Update to current cvs snapshot (post 2.3.2-pre1)
- Remove glibc-2.3-ppc32-textrel.patch, is included in CVS
- Remove libc23-ppc64-cvshead20021210.patch
* Thu Feb 20 2003 kukuk@suse.de
- Update to current cvs snapshot
- Remove glibc-2.3-ppc64-vfork-20030214.patch, seems to be
  included already.
* Wed Feb 19 2003 olh@suse.de
- add glibc-2.3-ppc32-textrel.patch
    glibc-2.3-ppc64-vfork-20030214.patch
* Fri Feb 14 2003 kukuk@suse.de
- Extend getent to print all IP addresses of one host
* Thu Feb 13 2003 kukuk@suse.de
- Update to current snapshot
* Wed Feb 12 2003 kukuk@suse.de
- Update to current snapshot (contains official fix for #23513)
* Tue Feb 11 2003 kukuk@suse.de
- Update to current snapshot
- Fix corruption of internal data in gethostbyname2 [Bug #23513]
* Sun Feb  9 2003 aj@suse.de
- Fix cancellation of system calls on x86-64.
* Sat Feb  8 2003 kukuk@suse.de
- Fix wrong logic in dynamic resolv.conf patch
* Fri Feb  7 2003 kukuk@suse.de
- Fix access of _res symbol in multithreaded programs
- Add "dynamic resolv.conf" patch to libnss_dns, too.
- Remove obsolete db2 manual pages
- Update to current snapshot (fixes [Bug #23363])
* Thu Feb  6 2003 kukuk@suse.de
- Update to current snapshot
- Use install-info for info pages
* Tue Feb  4 2003 schwab@suse.de
- Fix regexp parsing.
* Mon Feb  3 2003 meissner@suse.de
- Merged a unistd.h gcc3.3 compliance patch from Franz Sirl for
  ppc and ppc64 kernel headers.
* Fri Jan 31 2003 schwab@suse.de
- Fix building on s390[x].
* Fri Jan 31 2003 kukuk@suse.de
- Update to current glibc cvs
* Thu Jan 30 2003 aj@suse.de
- Fix one build problem on sparc.
- Fix asm-x86_64/mtrr.h.
* Wed Jan 29 2003 kukuk@suse.de
- Update to current glibc cvs
- Fix libm-ulps for x86-64 and ia64
* Thu Jan 23 2003 schwab@suse.de
- Fix use of DT_FINI_ARRAY.
- Temporarily disable tst-aio7 test on ia64.
* Tue Jan 14 2003 schwab@suse.de
- Fix ia64 for non-tls build.
* Tue Jan 14 2003 aj@suse.de
- Package libpthread_nonshared.a.
* Mon Jan 13 2003 kukuk@suse.de
- Add fixed version of x86-64-linuxthreads-disable-fs.diff
- Readd parts of libm-x86-64.diff
- Update to current cvs snapshot
- Add fix for errno compatibility
- Add s390(x) vfork fixes
* Sun Jan 12 2003 kukuk@suse.de
- Update to current cvs snapshot
* Fri Jan  3 2003 bg@suse.de
- removed obsolete patch for hppa
* Fri Dec 20 2002 kukuk@suse.de
- Update to current cvs snapshot
- Obsoletes glibc-2.3-setjmp-ppc64.diff
- Obsoletes build-alpha.diff
- Obsoletes libm-x86-64.diff
- Disable activate-spinlocks.diff temporary
- Disable x86-64-linuxthreads-disable-fs.diff (breaks compilation)
* Tue Dec 17 2002 olh@suse.de
- remove glibc-2.2.5-ppc64-bits-socket_h.diff
  add glibc-2.3-setjmp-ppc64.diff
  add libc23-ppc64-cvshead20021210.patch
  dont run configure in subshell, it can fail and rpm cant catch it
* Fri Dec 13 2002 schwab@suse.de
- Add more ia64 syscalls.
* Fri Dec  6 2002 olh@suse.de
- build also the locals parallel
  do not fail with parallel calculation on lowmem systems
* Tue Dec  3 2002 aj@suse.de
- Build parallel on x86-64.
- Activate fast spinlocks in malloc for x86 and x86-64.
- Integrate new math library from AMD for x86-64.
* Fri Nov 29 2002 bg@suse.de
- Fix build on hppa.
* Thu Nov 28 2002 aj@suse.de
- Fix build on alpha.
* Thu Nov 28 2002 aj@suse.de
- Update to CVS version from 2002-11-28.
- Do not use %%fs for threads for now on x86-64.
- Fix fnmatch bug with multibyte strings.
* Tue Nov 12 2002 bg@suse.de
- Updated hppa patches
* Thu Nov  7 2002 kukuk@suse.de
- getaddrinfo(): get host information for AF_INET and AF_INET6 only
  from the same service [Bug #21237]
* Thu Nov  7 2002 bg@suse.de
- Fix build of linuxthreads for hppa
* Tue Nov  5 2002 bg@suse.de
- Use current kernel-headers for parisc
- Add support for hppa
* Tue Nov  5 2002 kukuk@suse.de
- Update to current CVS snapshot
* Tue Oct 22 2002 schwab@suse.de
- Fix ia64 syscall numbers.
* Mon Oct 21 2002 schwab@suse.de
- Fix alignment in locale-archive.
* Mon Oct 21 2002 kukuk@suse.de
- Update to glibc 2.3.1 cvs 20021021
- Remove nss_dns6 patch
* Tue Oct 15 2002 kukuk@suse.de
- Update to glibc 2.3.1 cvs 20021015
* Wed Oct  2 2002 kukuk@suse.de
- Update to glibc 2.3 cvs 20021002
- Update crypt_blowfish to 0.4.4 (manual page fix, hppa fix)
* Wed Sep 25 2002 mls@suse.de
- build with -finline-limit=2000 on mips/armv4l
* Tue Sep 17 2002 schwab@suse.de
- Add new ia64 syscall numbers.
* Mon Sep 16 2002 meissner@suse.de
- Added AIO syscall numbers for ppc/ppc64, so libaio does not need them.
- Aligned powerpc bits/sem.h to be the same as the other 64bit
  archs, keeping the 32bit layout.
- Added the faster ppc32 memset.S from glibc HEAD on request of IBM.
* Thu Sep 12 2002 meissner@suse.de
- ppc/ppc64: added more biarch things to the SystemV IPC headers
  which are needed to get 64bit ipc / IPC_STAT to work.
* Mon Sep  9 2002 bk@suse.de
- s390x-biarch: use correct wordsize.h(move to main 32/64 directory)
- s390x: requires 64-bit kernel
* Mon Sep  9 2002 kukuk@suse.de
- Increase minimum value of MAXPACKET in libnss_dns, too
* Mon Sep  9 2002 uli@suse.de
- all architectures are created equal, but some are more equal
  than others; increased DB_FILE_ID_LEN in DB2 to 24 on x86-64,
  ia64, s390x and ppc64 to fit 64 bit __ino_t type
* Fri Sep  6 2002 mls@suse.de
- fix squeeze bug in db-1.85: set dirty flag after page modification
* Fri Sep  6 2002 kukuk@suse.de
- Increase minimum value of MAXPACKET
* Mon Sep  2 2002 kukuk@suse.de
- Fix asm header files for sparc/sparc64
* Fri Aug 30 2002 aj@suse.de
- Revert linuxthreads for x86-64 for now.
* Wed Aug 28 2002 aj@suse.de
- Add optimized math routines for x86-64.
* Tue Aug 27 2002 olh@suse.de
- BuildFlags="$(echo $RPM_OPT_FLAGS | sed 's#-mminimal-toc##')"
* Mon Aug 26 2002 olh@suse.de
- add ppc64_glibc_2.2.5_sunrpc-fix.patch
  * sysdeps/unix/sysv/linux/powerpc/bits/socket.h: New file that adds
  __powerpc64__ specific fields and adjust size/alignment for 64-bit.
* Fri Aug 23 2002 aj@suse.de
- Use floating stacks for x86-64.
- Add LICENSE file.
- Fix profiling code on x86-64.
- Add strcspn, strpbrk and strspn optimizations for x86-64.
- Fix makecontext for x86-64.
* Thu Aug 15 2002 aj@suse.de
- Use biarch headers already for building so that bits/syscalls.h is
  build correctly.
- Fix bits/syscalls.h for x86-64.
- Remove *xattr patches.
* Thu Aug  8 2002 kukuk@suse.de
- Update to current cvs (calloc variable overflow fixed)
- Apply fix for calloc fix
- Add *xattr system calls and error number
* Tue Aug  6 2002 kukuk@suse.de
- Update to current cvs (IPv6 fixes)
* Mon Aug  5 2002 aj@suse.de
- Implement *context functions for x86-64.
* Sat Aug  3 2002 kukuk@suse.de
- Update kernel-headers to version 2.4.19
- Add PreRequires "filesystem".
* Thu Aug  1 2002 bk@suse.de
- added s390-may2002.diff with some chunks from may2002 drop
- give make check a second chance on s390(x), but fail if not(race)
* Tue Jul 30 2002 kukuk@suse.de
- sunrpc/xdr_array.c: Check for variable overflow
- Ignore failed make check on s390(x)
* Tue Jul 30 2002 aj@suse.de
- Add optimisations for x86-64.
* Mon Jul 29 2002 kukuk@suse.de
- Don't allocate/free category name in setlocale() unnecessarily
  (for IBM Java [Bug #17065]).
* Tue Jul 23 2002 kukuk@suse.de
- Update to current glibc cvs (pread/pwrite fixes)
- Remove extra de.po, is in official tarball now.
* Sat Jul 20 2002 aj@suse.de
- Fix profiling for x86-64.
* Fri Jul 19 2002 meissner@suse.de
- Fixed the PPC64 patch, several superflous files removed.
- Reenabled make check for ppc64.
* Wed Jul 17 2002 kukuk@suse.de
- Update to current glibc cvs
- Fix kernel headers for s390/s390x
- Move some binaries/shell scripts and manual pages to the
  correct subpackage
* Tue Jul 16 2002 meissner@suse.de
- Merged latest PowerPC patch from IBM.
  * Lots of ppc64 related fixes.
  * Start of biarch support.
  * Changed struct stat in 64bit ABI.
* Fri Jul 12 2002 aj@suse.de
- Add biarch patch for s390 and s390x.
* Mon Jul  8 2002 aj@suse.de
- Add some optimized x86-64 math routines and a fixed lgammal
  implementation.
- Testsuite on x86-64 should pass now.
- Run ldconfig in postinstall.
- Fix memleak in catgets.
* Sat Jul  6 2002 kukuk@suse.de
- Fix typo in create_biarch_asm.sh (asm-sparc path)
* Thu Jul  4 2002 kukuk@suse.de
- Fix typo in manpages/Makefile
- Fix filelist (on some archs ld-linux.so.2 was missing)
* Thu Jul  4 2002 kukuk@suse.de
- Revert mktime patch (check for year < 70)
- Replace ifarch i386 with ix86
- Add patch for arm
- Include pt_chown again
* Wed Jul  3 2002 kukuk@suse.de
- Add more new manual pages
- Fix spec file (renaming of ld-*, creating of include/asm header
  files)
- Update to current CVS version
- Fix kernel-headers for x86-64 (don't overwrite patched version)
- Rename Equador -> Ecuador [Bug #16648]
- Add hint about security problem of host caching with nscd to
  config file.
* Tue Jun 25 2002 aj@suse.de
- Fix dl-machine.h for x86-64 to compile with new binutils.
- Fix mtrr.h header for x86-64.
- Do not package pt_chown.
* Tue Jun 18 2002 sf@suse.de
- reverted changes from Tue Jun 11 10:04:55 CEST 2002
  (took the kernel-headers from before, readded patch9)
- add new archive with kernel-headers for x86_64
- add autofs patch
* Mon Jun 17 2002 bk@suse.de
- remove s390* from the list of archs that ignore check fail
* Thu Jun 13 2002 uli@suse.de
- fixed typos in spec
* Thu Jun 13 2002 schwab@suse.de
- Fix ssize_t and __ipc_pid_t for ppc64.
* Wed Jun 12 2002 uli@suse.de
- fix errlist.c for ARM as well
* Tue Jun 11 2002 sf@suse.de
- made new kernel-header archive from kernel-source
- added asm-ppc64 to kernel-headers
- removed patch9 as it is obsoleted by the new kernel-headers
* Fri Jun  7 2002 olh@suse.de
- update asm-ppc64/ioctls.h, missing TIOCGDEV
* Thu Jun  6 2002 olh@suse.de
- fix glibc-ppc64 patch
* Thu Jun  6 2002 olh@suse.de
- update ppc64_glibc_ldconfig.diff
* Thu Jun  6 2002 ke@suse.de
- Update de.po from
  http://www.iro.umontreal.ca/contrib/po/teams/PO/de/libc-2.2.5.de.po.
- Use only translated entries to make the testsuite happy; call
  msgattrib on de.po and hu.po [# 16438].
* Mon Jun  3 2002 aj@suse.de
- Clean up generation of asm includes for bi-arch systems.
- Enable profiling for x86-64.
* Mon Jun  3 2002 kukuk@suse.de
- Add fix for weak declaration "_old_sys_nerr"
- Remove already disabled alpha patch
- Update to current CVS
* Sat Jun  1 2002 olh@suse.de
- fix stat for real, wrong size for st_nlink
* Wed May 29 2002 olh@suse.de
- add ppc64_glibc_2.2.5-types.diff, fixes stat()
* Wed May 29 2002 olh@suse.de
- add ppc64_glibc_ldconfig.diff for elf32/elf64 coexistance
* Thu May 23 2002 aj@suse.de
- Rename __thread to fix problems with GCC 3.2.
* Mon May 20 2002 olh@suse.de
- add ppc64 kernel headers
  add create_ppc_asm.sh
  update ppc64 patch, use /lib64/ld64.so.1
  move generic syscalls.list to ppc64/32
  set/getrlimit GLIBC2.0 is ppc32 only
  use parallel make on ppc and ppc64
  do not chroot as user
  cleanup nested ifarch for dynamic linker
* Wed May 15 2002 aj@suse.de
- Fix building of linuxthreads with current GCC.
* Mon May 13 2002 olh@suse.de
- fix ppc64 RTLDLIST ld64.so rewrite
* Sat May 11 2002 schwab@suse.de
- Add div/mod compatibility functions for ia64.
* Fri May 10 2002 olh@suse.de
- add ppc64 support
* Thu May  9 2002 aj@suse.de
- Add sys/io.h for x86-64.
* Mon May  6 2002 ihno@suse.de
- corrected memory calculation for parallel build
* Fri May  3 2002 kukuk@suse.de
- Add fix to compile math.h on SPARC with g++
* Thu May  2 2002 kukuk@suse.de
- Update to current CVS 2.2 branch
- Add x86-64 fix for crti.o and /usr/lib64 with libpthread
* Fri Apr 26 2002 kukuk@suse.de
- define sqrtl alias for PowerPC
* Mon Apr 22 2002 aj@suse.de
- Fix vfork for x86-64.
- Fix handling of ld.so.cache for x86-64.
* Thu Apr 18 2002 aj@suse.de
- Allow testsuite to fail for x86-64.
- Add ULPs for x86-64.
* Wed Apr 17 2002 kukuk@suse.de
- Update to current snapshot (mktime and dl fixes)
- Update hu.po
- Do not build a profiled glibc for x86-64
* Fri Apr 12 2002 kukuk@suse.de
- Update to current snapshot (fix SPARC compile)
- Revert do-lookup.h patch on Alpha (does not work here)
* Wed Apr 10 2002 aj@suse.de
- Fix linuxthreads for x86-64.
* Wed Apr 10 2002 aj@suse.de
- Add x86-64.diff to fix glob64.
* Wed Apr 10 2002 kukuk@suse.de
- Update kernel-headers to 2.4.19pre4 (with x86-64 support)
- Update glibc to current cvs snapshot
- Allow old currencies (before EUR)
* Tue Apr  2 2002 aj@suse.de
- Update ULPs.
* Thu Mar 21 2002 kukuk@suse.de
- Create html pages after installation of info pages [Bug #15283]
* Tue Mar 12 2002 kukuk@suse.de
- Add db1 fix if blocksize is not ^2
* Sat Mar  2 2002 kukuk@suse.de
- Add fix for format string bug
* Sat Mar  2 2002 kukuk@suse.de
- Fix return value of nice wrapper
* Fri Mar  1 2002 kukuk@suse.de
- Add fix for corrupt ut_line
- Add fix for current gcc 3.1
- Add patch for nice return values
* Thu Feb 28 2002 kukuk@suse.de
- Add fix for rtime, swscanf and ia64
* Mon Feb 18 2002 kukuk@suse.de
- When a dlopened module references a weak symbol from another
  dlopened module (loaded with RTLD_GLOBAL) no dependency was
  generated for this fact, so the second module was unloaded even
  if the first one was still around.
* Sun Feb 17 2002 kukuk@suse.de
- Add pthread/signal bugfix [Bug #13280]
- Fix directory file list (don't include /usr/include)
* Thu Feb 14 2002 kukuk@suse.de
- Use defattr in spec file to avoid problems with not existing
  UIDs and rpm.
* Thu Feb 14 2002 aj@suse.de
- Update ULPs for GCC 3.1.
* Wed Feb 13 2002 kukuk@suse.de
- Apply db1 patch from mls@suse.de to fix rpm problems
* Tue Feb 12 2002 ro@suse.de
- fix owner/group for kernel headers
* Mon Feb 11 2002 kukuk@suse.de
- Add another solution for the glob problem
* Thu Feb  7 2002 kukuk@suse.de
- Add fix for glob (glob should not call globfree)
- Add fix for innetgr
* Wed Feb  6 2002 kukuk@suse.de
- Use correct BuildRoot
* Wed Feb  6 2002 kukuk@suse.de
- Update hu.po
* Wed Feb  6 2002 kukuk@suse.de
- Set LC_CTYPE for error messages in localedef [Bug #12878]
* Mon Feb  4 2002 kukuk@suse.de
- Add fixes from CVS: dynamic loader, readv and writev seg.fault
  and various architecture fixes for alpha and mips
- Don't compile with -g on Alpha
- Add fix for possible endless loop fix
* Fri Feb  1 2002 bk@suse.de
- merged s390x lib64 patch and spec file changes
* Wed Jan 23 2002 kukuk@suse.de
- Split glibc into glibc and glibc-locale
- Create more UTF8 locale
* Mon Jan 21 2002 kukuk@suse.de
- Update to official glibc 2.2.5
* Wed Jan 16 2002 kukuk@suse.de
- Remove /var/adm/setup/setup.timeconfig
* Wed Jan 16 2002 kukuk@suse.de
- Apply S390 fix
* Wed Jan  9 2002 kukuk@suse.de
- Update to glibc 2.2.5pre1
* Tue Jan  8 2002 egmont@suselinux.hu
- Added partial Hungarian translation
* Tue Jan  8 2002 kukuk@suse.de
- Add patch to pass math tests with gcc 3.x
- Update to current CVS version
* Thu Jan  3 2002 kukuk@suse.de
- Update kernel header files to 2.4.17
* Tue Jan  1 2002 kukuk@suse.de
- Update current CVS version, add final fixes for LSB test suite
* Tue Dec 18 2001 poeml@suse.de
- Install ja_JP.SJIS locale.
* Sat Dec 15 2001 schwab@suse.de
- Fix missing declaration of md5_uintptr.
* Fri Dec 14 2001 kukuk@suse.de
- Update to correct CVS branch
* Thu Dec 13 2001 kukuk@suse.de
- Update to current CVS
- Increase PATH_MAX to 4096 (including the leading zero)
- Clear pointer if asprintf fails
- pthread_key_delete should not contact thread manager before it
  is created.
* Tue Dec 11 2001 kukuk@suse.de
- Fix prelink patch
* Tue Dec 11 2001 kukuk@suse.de
- Add fixes for LSB.os test suite (ftw, grantpt and ftok)
- Update to current CVS
- Add prelink patch
* Fri Dec  7 2001 kukuk@suse.de
- Merge with current CVS
- Add blowfish crypt
* Fri Nov 23 2001 uli@suse.de
- added armv4l arch to spec
- added arm kernel headers
- added trivial fix for dl-machine.h from CVS (see arm.dif)
* Thu Nov 15 2001 adrian@suse.de
- add mips architecture to spec file
- apply further mips fixes for ld
- activate %%clean again
* Sun Nov 11 2001 kukuk@suse.de
- Fix lost permissions of shell script on SPARC
* Thu Nov  8 2001 kukuk@suse.de
- Add 32bit UID fixes
* Thu Nov  8 2001 kukuk@suse.de
- More fixes for asm-ia64 header files
* Thu Nov  8 2001 kukuk@suse.de
- Fix asm-i386/processor.h (don't align struct)
- Fix asm-ia64/bitops.h (define CMPXCHG_BUGCHECK)
- Correct version number in version.h
* Wed Nov  7 2001 uli@suse.de
- fixed sys/io.h, sysmacros.h for icc
* Tue Nov  6 2001 kukuk@suse.de
- Update kernel-headers to 2.4.14
* Thu Nov  1 2001 kukuk@suse.de
- Use again old rules to generate html files
* Sun Oct 21 2001 schwab@suse.de
- Fix inttypes.h for C++.
* Fri Oct 19 2001 aj@suse.de
- Fix typo in inttypes.h that presents compilation by non-GCC compilers.
* Tue Oct 16 2001 aj@suse.de
- Update elf.h to include x86-64 defines since those are needed
  by some other tools.
* Fri Sep 28 2001 schwab@suse.de
- Readd patch from 2001-09-10 with corrections.
- Add compatibility patch for GCC 3.  This allows to build glibc
  with GCC 3.
- Require that make check succeeds on ia64.
* Thu Sep 13 2001 aj@suse.de
- Add a better version of the threads-fork patch that fixes some
  more places where interrupts can occur and does this a bit cleaner.
* Tue Sep 11 2001 aj@suse.de
- Remove patch from 2001-09-10 since it breaks the dynamic linker.
* Tue Sep 11 2001 aj@suse.de
- Fix bug in linuxthreads where manager and threads could
  get out of synch due to an interrupted read call.
* Mon Sep 10 2001 schwab@suse.de
- Fix handling of dependent dynamic objects for dlopen/dlclose.
* Sat Sep  8 2001 kukuk@suse.de
- Don't create gconv cache (else iconv --list seg.faults)
* Tue Aug 28 2001 aj@suse.de
- Improve dynamic linker to relocate dynamic objects faster.  This
  implies a small cache for symbol lookups and handling the ld -z combreloc
  feature if binaries are linked this way.
* Fri Aug 24 2001 kukuk@suse.de
- Add fix for handling of %%l[] in vfscanf
- ldconfig removes stale links now
- Remove susehelp config files, now in susehelp itself
* Wed Aug 22 2001 aj@suse.de
- Update s390 patch from IBM.
* Fri Aug 17 2001 kukuk@suse.de
- Adjust dns6 patch for 2.2.4
- Fix spec file (include lost libnss_dns6.so)
- Fix versionnumber in version.h [Bug #9759]
- Update kernel-header to 2.4.9
* Thu Aug 16 2001 aj@suse.de
- Update to 2.2.4 final.  Add s390-ucontext patch.
* Fri Aug 10 2001 aj@suse.de
- Update to current glibc version.  Do not use the hardlink program
  for compatification since localedef will do this itself now.
  Create gconv cache.
* Wed Aug  1 2001 aj@suse.de
- Add patch for zic to create copy of the timezone instead
  of a symbolic link so that the file exists even if /usr is not
  mounted.
  Use i486 instead of i386 as default architecture for i386.
* Sat Jul 28 2001 kukuk@suse.de
- Fix problem with linux/spinlock.h
* Fri Jul 27 2001 kukuk@suse.de
- Update kernel-header files to 2.4.7
* Thu Jul 26 2001 froh@suse.de
- add fix for failing tst-setcontext on s390
* Wed Jul 25 2001 aj@suse.de
- Add patch to fix loading of dynamic libs in static programs for PPC.
* Fri Jul 20 2001 kukuk@suse.de
- Update to current CVS snapshot
- Disable tst-regex and test-lfs
* Fri Jul  6 2001 kukuk@suse.de
- Add da_DK@euro and da_DK.UTF-8
* Thu Jul  5 2001 kukuk@suse.de
- Update to current CVS snapshot
- Remove obsolete cvs patch
- Fix DNS/IPv6 patch
- Hardlink equal locale files
* Fri Jun 22 2001 olh@suse.de
- add glibc-2.2.3-ppc_dlmachine.diff to fix binutils make check
* Tue Jun 19 2001 bk@suse.de
- added s390x support to spec file
* Tue Jun 19 2001 aj@suse.de
- Fix profiling on PowerPC.
* Fri Jun 15 2001 schwab@suse.de
- Fixup asm-ia64/atomic.h for user-space inclusion.
* Tue Jun 12 2001 aj@suse.de
- Fix testsuite for sparc.
* Mon Jun 11 2001 aj@suse.de
- Fix testsuite for powerpc and S390, build again on alpha.
* Tue May 22 2001 kukuk@suse.de
- Update to current CVS snapshot
- Rmove support for PF_LOCAL from getaddrinfo [Bug #8469]
* Sun May 13 2001 kukuk@suse.de
- Don't use absolute paths in pre-install-section
* Fri May  4 2001 kukuk@suse.de
- Add special version.h which fails on compiling kernel modules
* Sat Apr 28 2001 kukuk@suse.de
- Update to glibc 2.2.3, kernel-headers-2.4.4
* Tue Apr 24 2001 schwab@suse.de
- Fix feenableexcept on ia64.
* Tue Apr 24 2001 aj@suse.de
- Install some more UTF-8 locales, fix tr_TR locale.
* Thu Apr 12 2001 kukuk@suse.de
- Include our own texi2html
* Wed Apr 11 2001 kukuk@suse.de
- Add fixes from SuSE kernel header files
- Add patch to reload /etc/resolv.conf if there was changes
- Add glibc.conf for susehelp (glibc-html pages)
* Mon Apr  9 2001 schwab@suse.de
- Fix ld.so for kernel 2.4.3 on ia64.
* Thu Apr  5 2001 kukuk@suse.de
- Add more fixes from CVS
* Tue Apr  3 2001 kukuk@suse.de
- Fix isdn header files from kernel-headers
* Fri Mar 30 2001 kukuk@suse.de
- Update kernel header files to 2.4.3
* Fri Mar 30 2001 kukuk@suse.de
- Merge s390 patches
- Fix rcmd_af() (allow PF_UNSPEC)
* Fri Mar 30 2001 aj@suse.de
- Add some small fixes, fix spec file for removal of man-pages.
* Thu Mar 29 2001 kukuk@suse.de
- Don't provide kernel_headers any longer
- Remove some man-pages which are now official in the man-pages
  package
* Thu Mar 29 2001 aj@suse.de
- Fix shmfs recognition.
* Wed Mar 28 2001 ro@suse.de
- added db-splitmask fix from mls (hopefully work around bug in db1)
* Wed Mar 28 2001 aj@suse.de
- Fix s390 to not generate wrong relocations, work around compiler
  error.
* Wed Mar 21 2001 kukuk@suse.de
- glibc-devel obsoletes and provides linclude
* Tue Mar 20 2001 kukuk@suse.de
- Add strtok and other fixes from CVS
- Add yp_all fix
* Tue Mar 13 2001 kukuk@suse.de
- Add more s390 string.h fixes
* Mon Mar 12 2001 aj@suse.de
- Add fixes for s390, don't run testsuite on s390 for now.
* Fri Mar  9 2001 kukuk@suse.de
- Fix linux/init.h header file
* Fri Mar  9 2001 aj@suse.de
- Handle new EM_S390 value.
* Fri Mar  9 2001 kukuk@suse.de
- kernel-heaer fixes to build on Alpha
* Thu Mar  8 2001 ro@suse.de
- update kernel-headers to 2.4.2
* Thu Mar  8 2001 ro@suse.de
- kernel-header fixes to build on s390
* Thu Feb 22 2001 schwab@suse.de
- More kernel header fixes for IA64.
* Tue Feb 20 2001 kukuk@suse.de
- Remove optimisation not supported on all plattforms
* Mon Feb 19 2001 kukuk@suse.de
- Fix Optimization of glibc build
- Add Optimization for alphaev6 and sparcv9
* Sat Feb 17 2001 kukuk@suse.de
- Update to glibc 2.2.2 from CVS
* Thu Feb 15 2001 kukuk@suse.de
- kernel header fixes for SPARC and IA64
* Tue Feb 13 2001 kukuk@suse.de
- Make optimization for i686 work
- Add manual page for ldd
* Mon Feb 12 2001 kukuk@suse.de
- Fix more kernel headers
* Thu Feb  8 2001 kukuk@suse.de
- Fix more kernel-headers
* Wed Feb  7 2001 kukuk@suse.de
- Delete links in pre install section for glibc-devel
* Wed Feb  7 2001 kukuk@suse.de
- Fix kernel-header includes
* Tue Feb  6 2001 kukuk@suse.de
- Fix creating of /usr/include/asm on SPARC
- Add more CVS patches
* Mon Feb  5 2001 kukuk@suse.de
- Add some patches from CVS
- Include our own kernel header files
* Mon Jan 22 2001 aj@suse.de
- Add elf patch to fix problems on ia64 and ppc with _dl_pagesize.
* Mon Jan 22 2001 aj@suse.de
- Fix mmap64 on powerpc.
* Tue Jan 16 2001 aj@suse.de
- Fix sunrpc-udp.diff, add mman.h fix for powerpc.
* Wed Jan 10 2001 aj@suse.de
- Add sunrpc-udp.diff to fix UDP timeouts with Linux 2.4 kernel.
* Tue Jan  9 2001 aj@suse.de
- Add glibc-2.2.secure.diff to close some security holes.
* Wed Jan  3 2001 schwab@suse.de
- Fix strtol and friends on 64 bit platforms.
- Use 8192 as default pagesize on ia64.
- Scan AUX vector also in statically linked programs.
* Wed Jan  3 2001 aj@suse.de
- Build some UTF-8 locales using a patch from Markus Kuhn.
* Wed Dec 13 2000 aj@suse.de
- Add compatibility patch for IPv6 and Linux 2.2.
* Wed Dec 13 2000 schwab@suse.de
- Update ia64 patch.
* Sat Dec  9 2000 kukuk@suse.de
- Fix resolver bug
* Fri Dec  1 2000 kukuk@suse.de
- Add bug fixes for setlocale and strncat
- strip gconv modules
* Fri Nov 24 2000 kukuk@suse.de
- Fix typo in spec file
* Thu Nov 23 2000 kukuk@suse.de
- Add hack for POWER3
* Wed Nov 22 2000 kukuk@suse.de
- Add strncat bugfix for S/390
* Tue Nov 21 2000 kukuk@suse.de
- Don't bulid 32bit compat packages
* Mon Nov 20 2000 schwab@suse.de
- Remove use of getpagesize syscall on ia64.
- Follow DT_INIT/DT_FINI change in compiler.
* Sun Nov 19 2000 kukuk@suse.de
- Minor specfile fixes
* Thu Nov 16 2000 kukuk@suse.de
- Add lot of more bug fixes
* Tue Nov 14 2000 kukuk@suse.de
- Add bugfix for static linked binaries/ld.so.cache from aj@suse.de
* Mon Nov 13 2000 aj@suse.de
- Fix noversion.diff and spec file.
* Sat Nov 11 2000 kukuk@suse.de
- Fix nssv1 on PowerPC
- no libNoVersion on SPARC
* Fri Nov 10 2000 kukuk@suse.de
- Disable make check on PowerPC and Alpha
* Fri Nov 10 2000 kukuk@suse.de
- Update to final glibc 2.2
* Fri Nov  3 2000 kukuk@suse.de
- Update to glibc-2.2-20001103 (glibc-2.1.97)
- Don't install libNoVersion on PowerPC
- Rename nssv1 -> glibc-nssv1
- Rename libd -> glibc-profile, move libg.a to libc
- Rename libcinfo -> glibc-info
- Rename libchtml -> glibc-html
- Rename localedb -> glibc-i18ndata
- Rename libc -> glibc-devel
- Rename shlibs -> glibc
* Sat Oct 28 2000 kukuk@suse.de
- Update to glibc-2.2-20001028
- Enable more checks
* Fri Oct 27 2000 aj@suse.de
- Fix NoVersion patch
* Wed Oct 25 2000 kukuk@suse.de
- Update to glibc-2.2-20001025
* Sun Oct 22 2000 kukuk@suse.de
- Update to glibc 2.2-20001021
- Update glibc-db to 2.1.95
- Update ia64 patch
* Fri Oct 20 2000 kukuk@suse.de
- Update to glibc 2.2-20001020
- Add s390 spec file changes
* Tue Oct 10 2000 schwab@suse.de
- Update to glibc 2.2-20001009.
- Fix TRAMPOLINE_TEMPLATE for ia64.
* Sun Oct  1 2000 schwab@suse.de
- Export more ia64 specific symbols.
* Tue Sep 26 2000 kukuk@suse.de
- Update to glibc 2.2-20000926
* Mon Sep 25 2000 kukuk@suse.de
- Update to glibc 2.2-20000925 snapshot
* Thu Sep 14 2000 schwab@suse.de
- Update ia64 patch.
* Fri Sep  8 2000 bk@suse.de
- added glibc-2.1.3-db2-s390.tar.gz from developerworks (db2 fix)
* Wed Sep  6 2000 fober@suse.de
- merge s390-7.0 with STABLE:
  - upgraded to glibc-linuxthreads-2.1.3.1-s390.diff from 2.1.3
  - added new changes from IBM s390 codedrop
  - removed glibc-dlopen-2.1.3-s390.diff which is
    incorporated in glibc-linuxthreads-2.1.3.1-s390.diff now
* Tue Sep  5 2000 kukuk@suse.de
- Add glibc-2.1.security.dif
* Fri Sep  1 2000 olh@suse.de
- add glibc-2.1-ppc_lfs.dif, enables (hopefully) lfs on ppc
* Wed Aug 30 2000 olh@suse.de
- remove sysdeps/powerpc/memset.S on ppc for POWER3
* Mon Aug 28 2000 olh@suse.de
- add glibc-2.1.sgi_fam.dif
- remove sysdeps/rs6000/memcopy.h on ppc for POWER3
* Mon Aug 21 2000 garloff@suse.de
- Fix race on cond_wait WRT owner of mutex (from olh@suse.de)
* Tue Jul 25 2000 kukuk@suse.de
- Add mmap fix for PowerPC
* Tue Jul 11 2000 kukuk@suse.de
- Remove "mutex is owned by current thread" bugfix for IBMs jdk
* Mon Jun 26 2000 schwab@suse.de
- Update ia64 patch.
* Tue Jun 20 2000 kukuk@suse.de
- Move html docu in extra package
- Update ia64 patch
* Wed May 31 2000 kukuk@suse.de
- Remove LICENSE file, it's the same as COPYING.LIB
- Add libc docu as html
* Sun May 28 2000 kukuk@suse.de
- Fix ldconfig on PPC and IA64
* Sat May 27 2000 kukuk@suse.de
- Merge new ldconfig fixes
* Fri May 26 2000 kukuk@suse.de
- Update ia64 patch
* Fri May 26 2000 kukuk@suse.de
- Fix (f)truncate64 and xdr_uint8_t
* Wed May 24 2000 kukuk@suse.de
- Fix ldconfig.8 manual page
* Thu May 18 2000 bk@suse.de
- added s390 dlopen fix
* Tue May 16 2000 bk@suse.de
- updated s390 patches to match IBM_codedrop_2000_05_15
* Fri May 12 2000 kukuk@suse.de
- Fix netinet/in.h IPv6 compare
* Fri May 12 2000 schwab@suse.de
- Update ia64 patches.
* Fri May 12 2000 kukuk@suse.de
- Don't apply LFS patch
* Tue May  9 2000 kukuk@suse.de
- Add LFS patches
* Mon May  8 2000 kukuk@suse.de
- Add lot of bug fixes from CVS
* Tue Apr 25 2000 kukuk@suse.de
- Fix nscd/getgrnam bug
* Thu Apr 20 2000 kukuk@suse.de
- Remove /var/mail -> /var/spool/mail patch
- Update nscd.conf.5 manual page
* Wed Apr 12 2000 kukuk@suse.de
- Add nscd patch from Chris Wing
* Wed Apr 12 2000 kukuk@suse.de
- Add ldconfig fix from aj@suse.de
* Wed Apr 12 2000 schwab@suse.de
- More ia64 patches.
- Use libc.so.0, libm.so.0, ld-linux-ia64.so.1 on ia64.
* Mon Apr 10 2000 kukuk@suse.de
- Support asm-sparc64 and asm-sparc on SPARC
* Mon Apr 10 2000 kukuk@suse.de
- Create /etc/ld.so.cache always with permissions 0644
- Update ia64 patches
* Thu Apr  6 2000 schwab@suse.de
- Fix dynamic linker bug in ia64.
- Add ia64 spinlocks for db2.
* Tue Apr  4 2000 schwab@suse.de
- New IA64 patches.
- Fix ldconfig -p.
* Mon Apr  3 2000 bk@suse.de
- s390 team added s390 patches
* Wed Mar 22 2000 kukuk@suse.de
- Fix last SPARC patch
* Tue Mar 21 2000 kukuk@suse.de
- Add SPARC patches
- Add IA64 patches
* Wed Mar 15 2000 kukuk@suse.de
- Remove personality call (problematic on Alpha)
- Fix typo in localeconv
- alpha/ioperm.c> Add entry for "Nautilus".
* Tue Mar  7 2000 kukuk@suse.de
- Add locale SIGSEGV fix
- Fix getdate bug
* Fri Feb 25 2000 kukuk@suse.de
- Update to final glibc 2.1.3 + fix for bigendian machines
* Thu Feb 24 2000 kukuk@suse.de
- Update to current glibc cvs snapshot
- Add libnss_dns6.so.2, which makes IPv4 and IPv6 lookups.
  Old libnss_dns.so.2 will only make IPv4 lookups.
* Tue Feb 22 2000 kukuk@suse.de
- Update to current glibc cvs snapshot (2.1.3pre4)
* Sat Feb  5 2000 kukuk@suse.de
- Add missing defines for SPARC bits/termios.h
* Thu Feb  3 2000 kukuk@suse.de
- Add regex patch from Andreas Schwab
* Wed Feb  2 2000 kukuk@suse.de
- Add ldconfig fix
- Update to current glibc cvs snapshot
- Fix sys/io.h on Intel (C++)
* Mon Jan 24 2000 kukuk@suse.de
- Fix duplicate setrlimit
* Mon Jan 24 2000 kukuk@suse.de
- Update to current glibc cvs snapshot
* Wed Jan 19 2000 kukuk@suse.de
- Update to current glibc cvs snapshot
* Sat Jan 15 2000 ro@suse.de
-fixed ppc db2-patch
* Fri Jan 14 2000 kukuk@suse.de
- Add patches for Intel and PPC
* Thu Jan 13 2000 kukuk@suse.de
- Add patch for SPARC
* Thu Jan 13 2000 kukuk@suse.de
- Move info pages to /usr/share/info
* Mon Jan 10 2000 kukuk@suse.de
- Add ipv6 patches for getent
* Mon Jan 10 2000 kukuk@suse.de
- Move manual pages for applications and config files into
  shlibs package
- Update to current glibc 2.1.3 snapshot
* Fri Dec 17 1999 kukuk@suse.de
- Add new ldconfig patches
- Add aio patch
- Add fix for bits/string2.h
* Wed Dec 15 1999 kukuk@suse.de
- add ldconfig.8
* Wed Dec 15 1999 kukuk@suse.de
- Update to current glibc 2.1.3 snapshot
- Fix get/setrlimit problems
* Thu Dec  9 1999 kukuk@suse.de
- Update to current glibc 2.1.3 snapshot
- Add new ldconfig
* Sun Dec  5 1999 kukuk@suse.de
- Disable make check for SPARC (kernel bug)
- Add setrlimit patches
- Update to current glibc 2.1.3 snapshot
* Fri Nov 26 1999 kukuk@suse.de
- Update to current glibc 2.1.3 snapshot.
* Tue Oct 26 1999 kukuk@suse.de
- Add fix for correct accounting of needed bytes (gethnamaddr.c)
- Remove not exported, public names from internal md5 functions
* Mon Oct 25 1999 kukuk@suse.de
- Add fix for missing nexttowardl aliase
* Tue Oct 19 1999 kukuk@suse.de
- Add security fix for iruserok
* Sat Oct 16 1999 kukuk@suse.de
- Build libNoVersion.so.1 on every platform
* Mon Oct 11 1999 kukuk@suse.de
- Add linuxthreads/signals.c fix from Andreas Schwab
- Remove dangling symlink (Bug #544)
- Add more bug fixes
* Fri Oct  8 1999 kukuk@suse.de
- Add timezone update
* Fri Oct  8 1999 kukuk@suse.de
- Add NIS+ shadow parser fix
* Thu Oct  7 1999 kukuk@suse.de
- Update to official glibc 2.1.2, add important fixes
- Update nscd, add manual pages for it
* Mon Sep 20 1999 ro@suse.de
- libc: added requires kernel_headers
* Mon Sep 13 1999 bs@suse.de
- ran old prepare_spec on spec file to switch to new prepare_spec.
* Mon Sep  6 1999 kukuk@suse.de
- Update to current glibc 2.1.2 cvs snapshot
- Remove use auf automake in spec file
* Fri Sep  3 1999 kukuk@suse.de
- Update to current glibc 2.1.2 cvs snapshot
- Use RPM macros for configure
* Wed Aug 25 1999 kukuk@suse.de
- Update to current glibc 2.1.2 cvs snapshot
* Thu Aug 19 1999 kukuk@suse.de
- Add PowerPC patches from Uli Hecht
- Cleanup of SPEC file
- Create a profiling version
* Sat Aug 14 1999 kukuk@suse.de
- disable nscd hosts caching by default
- Apply patch from HJL for broken gethostbyname_r in libnss_dns
* Fri Jul 16 1999 kukuk@suse.de
- Add header fix for autoconf/g++
- Move pt_chown and gconv modules from libc to shlibs
* Wed Jul 14 1999 kukuk@suse.de
- Remove /etc/localtime from filelist
* Mon Jul 12 1999 kukuk@suse.de
- Add more bug fixes from cvs
- Add ld.so bug fix from Andreas Schwab
* Thu Jul  8 1999 kukuk@suse.de
- configure for i386-unknown-linux
* Wed Jul  7 1999 kukuk@suse.de
- Remove alpha patch, it's now in glibc 2.1.2 cvs
- Add more bug fixes from cvs
- Remove malloc patch (breaks StarOffice)
* Tue Jul  6 1999 kukuk@suse.de
- Add openpty patch from Andreas Schwab
  (openpty now works if /dev/pts is not mounted)
* Fri Jul  2 1999 kukuk@suse.de
- Remove warning von zic about symlinks.
* Mon Jun 28 1999 kukuk@suse.de
- Remove ndbm links, now in gdbm
* Fri Jun 25 1999 kukuk@suse.de
- Add a lot of more fixes
- Add nscd fixes and enable nscd on alpha
* Mon Jun 14 1999 kukuk@suse.de
- Add nss_dns and fget* fixes.
- Add nscd patches for NIS+
* Thu Jun 10 1999 kukuk@suse.de
- Fix daemon() for MT programs
- Add libio fixes
* Fri May 28 1999 kukuk@suse.de
- Add fix for docu
- Disable nscd for alpha again
* Wed May 26 1999 kukuk@suse.de
- Update to version 2.1.1
* Mon May 17 1999 kukuk@suse.de
- Update to snapshot from 16.5.1999
- Add manpages
- Add __setfpucw to libNoVersion (intel)
- Add COPYING and COPYING.LIB
* Fri May  7 1999 kukuk@suse.de
- Update to snapshot from 6.5.1999
- Add NoVersion patches from RedHat for miscompiled glibc 2.0 apps
- Add patch for Alpha RX164
- Add workaround for nscd on Alpha
* Wed Apr 28 1999 kukuk@suse.de
- Remove latest fnmatch patches from Uli Drepper
* Tue Apr 27 1999 kukuk@suse.de
- update to cvs version of Apr 26 1999
- fix pmap_set/pmap_unset for DHCP clients
- Rename libdb1.so.2[.1] to libdb.so.2[.1] since we don't create
  the symbolic links.
* Tue Apr 20 1999 kukuk@suse.de
- update to cvs version of Apr 20 1999
- remove sunrpc.diff
- fix nssv1 package
- only include nscd on intel
- install /etc/nscd.conf
* Mon Apr 12 1999 kukuk@suse.de
- update to cvs version of Apr 11 1999
- Fix paths in paths.h
- install linuxthreads man pages and documentation
- Add sunrpc patch for Alpha and security fixes
- added links for el_GR and ru_RU.KOI8-R in usr/share/locale
* Wed Mar 31 1999 bs@suse.de
- don't use lx_hack for build
* Tue Mar 16 1999 ro@suse.de
- libc.texinfo: changed to build with stable texinfo version
* Mon Mar 15 1999 ro@suse.de
- update to 2.1.1 (cvs of Mar 15 1999)
- update nssv1 to 2.0.2
* Sat Feb 20 1999 ro@suse.de
- fixed specfile (lddlibc4 not built on alpha)
* Sat Feb 20 1999 ro@suse.de
- fixed specfile ...
* Sat Feb 20 1999 ro@suse.de
- added automake to neededforbuild
* Sat Feb 20 1999 ro@suse.de
- added nss-v1 modules (to keep old rpm happy with file owners)
* Fri Feb 19 1999 ro@suse.de
- update to cvs-version of 1999/02/18
* Fri Sep 25 1998 ro@suse.de
- fixed specfile
* Fri Sep 25 1998 ro@suse.de
- update: use cvs-version of 980925
  edited db/Makefile to ignore messed up target-dependency
* Mon Sep 21 1998 ro@suse.de
- update: use cvs-version of 980921
* Mon Sep 14 1998 ro@suse.de
- update: use today's cvs-version
* Wed Sep  2 1998 ro@suse.de
- build for 586 since egcs generates code for 686 that does NOT run
  on 586 !!! (eg strtok)
* Fri Aug 21 1998 ro@suse.de
- updated to cvs-version 20.8.98
  added gettext as neededforbuild (so configure shuts up)
  glibc-linuxthreads is contained in main archive now
* Tue Jun 16 1998 ro@suse.de
- added symlink usr/include/X11
* Thu May 28 1998 ro@suse.de
- added symlinks to linux include files
* Thu May 28 1998 bs@suse.de
- moved ".so" Links to package libc.
* Wed May 27 1998 bs@suse.de
- changed version do `date`
* Wed May 27 1998 ro@suse.de
- created specfile to build libc, shlibs, libd, libcinfo
    localedb, timezone,
- former libc renamed to libc5.
