#
# spec file for package apache2
#
# Copyright (c) 2024 SUSE LLC
#
# All modifications and additions to the file contributed by third parties
# remain the property of their copyright owners, unless otherwise agreed
# upon. The license for this file, and modifications and additions to the
# file, is the same license as for the pristine package itself (unless the
# license for the pristine package is not an Open Source License, in which
# case the license is the MIT License). An "Open Source License" is a
# license that conforms to the Open Source Definition (Version 1.9)
# published by the Open Source Initiative.

# Please submit bugfixes or comments via https://bugs.opensuse.org/
#


%global upstream_name   httpd
%global testsuite_name  %{upstream_name}-framework
%global tversion        svn1901574
%global flavor          prefork%{nil}
%define mpm             %{nil}
%if "%{flavor}" == "prefork" || "%{flavor}" == "test_prefork"
%define mpm             prefork
%endif
%if "%{flavor}" == "worker" || "%{flavor}" == "test_worker"
%define mpm             worker
%endif
%if "%{flavor}" == "event" || "%{flavor}" == "test_event"
%define mpm             event
%endif
%define test            0
%define unittest        0
%if "%{flavor}" == "test_prefork" || "%{flavor}" == "test_worker" || "%{flavor}" == "test_event" || "%{flavor}" == "test_devel" || "%{flavor}" == "test_main"
%define test            1
%if "%{flavor}" == "test_prefork" || "%{flavor}" == "test_worker" || "%{flavor}" == "test_event"
%define unittest        1
%endif
%endif
%if "%{mpm}" == "prefork"
%define mpm_alt_prio    10
%endif
%if "%{mpm}" == "worker"
%define mpm_alt_prio    20
%endif
%if "%{mpm}" == "event"
%define mpm_alt_prio    30
%endif
%define default_mpm     prefork
%define suse_maintenance_mmn  0
%define apache_mmn      %(test -s %{SOURCE0} && \
                          { echo -n apache_mmn_; bzcat %{SOURCE0} | \
                            awk '/^#define MODULE_MAGIC_NUMBER_MAJOR/ {printf "%d", $3}'; } || \
                          echo apache_mmn_notfound)
%define static_modules  unixd systemd
%define dynamic_modules authz_core access_compat actions alias allowmethods asis auth_basic auth_digest \\\
                        auth_form authn_anon authn_core authn_dbd authn_dbm authn_file authn_socache authnz_ldap \\\
                        authnz_fcgi authz_core authz_dbd authz_dbm authz_groupfile authz_host authz_owner authz_user \\\
                        autoindex bucketeer buffer cache cache_disk cache_socache case_filter case_filter_in cgid cgi \\\
                        charset_lite data dav dav_fs dav_lock dbd deflate brotli dialup dir dumpio echo env expires \\\
                        ext_filter file_cache filter headers heartmonitor http2 imagemap include info lbmethod_bybusyness \\\
                        lbmethod_byrequests lbmethod_bytraffic lbmethod_heartbeat ldap log_config log_debug log_forensic \\\
                        logio lua macro mime mime_magic negotiation optional_fn_export optional_fn_import \\\
                        optional_hook_export optional_hook_import proxy proxy_ajp proxy_balancer proxy_connect \\\
                        proxy_express proxy_fcgi proxy_fdpass proxy_ftp proxy_hcheck proxy_html proxy_http proxy_http2 \\\
                        proxy_scgi proxy_uwsgi proxy_wstunnel ratelimit reflector remoteip reqtimeout request rewrite \\\
                        sed session session_cookie session_crypto session_dbd setenvif slotmem_plain slotmem_shm \\\
                        socache_dbm socache_memcache socache_redis socache_shmcb speling ssl status substitute suexec \\\
                        unique_id userdir usertrack version vhost_alias watchdog xml2enc
%define support_bin     ab check_forensic dbmmanage find_directives gensslcert htdbm htdigest htpasswd httxt2dbm \\\
                        log_server_status logresolve split-logfile
%define support_sbin    apachectl htcacheclean fcgistarter logresolve.pl rotatelogs
# Compatibility symlinks for support tools
%define compat_bin      ab check_forensic dbmmanage htdbm htdigest htpasswd \\\
                        log_server_status logresolve split-logfile
# apache2ctl is handled manually
%define compat_sbin     logresolve.pl rotatelogs

%define platform_string Linux/SUSE
%define httpduser       wwwrun
%define httpdgroup      www

%define datadir         /srv/www
%define htdocsdir       %{datadir}/htdocs
%define manualdir       %{_datadir}/apache2/manual
%define errordir        %{_datadir}/apache2/error
%define iconsdir        %{_datadir}/apache2/icons
%define cgidir          %{datadir}/cgi-bin
%define localstatedir   %{_localstatedir}/lib/apache2
%define runtimedir      /run
%define proxycachedir   %{_localstatedir}/cache/apache2
%define logfiledir      %{_localstatedir}/log/apache2
%define sysconfdir      %{_sysconfdir}/apache2
%define includedir      %{_includedir}/apache2
%if "%{mpm}" != ""
%define libexecdir      %{_libdir}/apache2-%{mpm}
%else
%define libexecdir      %{_libdir}/apache2
%endif
%define installbuilddir %{_datadir}/apache2/build
%define userdir         public_html

%define suexec_safepath %{_prefix}/local/bin:%{_prefix}/bin:/bin

%define psuffix       %{nil}
%if "%{flavor}" != ""
%define psuffix      -%{flavor}
%endif

%define use_firewalld 1
%define build_http2 1

Name:           apache2%{psuffix}
Version:        2.4.58
Release:        150600.5.35.1
Summary:        The Apache HTTPD Server
License:        Apache-2.0
Group:          Productivity/Networking/Web/Servers
URL:            https://httpd.apache.org/
# essential sources
Source0:        https://www.apache.org/dist/httpd/%{upstream_name}-%{version}.tar.bz2
Source1:        https://www.apache.org/dist/httpd/%{upstream_name}-%{version}.tar.bz2.asc
Source2:        apache2.keyring
Source3:        apache2.service
Source4:        apache2@.service
Source5:        apache2.target
# Add file to take mtime from it in prep section
Source6:        apache2.changes
Source10:       apache2-ssl-dirs.tar.bz2
# test
# svn checkout http://svn.apache.org/repos/asf/httpd/test/framework/trunk/ httpd-framework
Source20:       %{testsuite_name}-%{tversion}.tar.bz2
Source21:       apache2-mod_example.c
# integration settings
Source30:       sysconfig.apache2
Source31:       logrotate.apache2
Source32:       permissions.apache2
Source33:       firewalld.apache2
Source34:       firewalld-ssl.apache2
Source35:       susefirewall.apache2
Source36:       susefirewall-ssl.apache2
# scripts
Source100:      apache2-a2enmod
Source101:      apache2-a2enflag
Source102:      apache2-systemd-ask-pass
Source103:      apache2-start_apache2
Source104:      apache2-script-helpers
# additional support
Source130:      apache2-gensslcert
Source131:      apache2-check_forensic
Source132:      apache2-find_directives
# configuration
Source150:      apache2-httpd.conf
Source151:      apache2-errors.conf
Source152:      apache2-default-server.conf
Source153:      apache2-listen.conf
Source154:      apache2-manual.conf
Source155:      apache2-mod_autoindex-defaults.conf
Source156:      apache2-mod_info.conf
Source157:      apache2-mod_log_config.conf
Source158:      apache2-mod_mime-defaults.conf
Source159:      apache2-mod_status.conf
Source160:      apache2-mod_userdir.conf
Source161:      apache2-server-tuning.conf
Source163:      apache2-ssl-global.conf
Source164:      apache2-mod_usertrack.conf
Source165:      apache2-mod_reqtimeout.conf
Source166:      apache2-loadmodule.conf
Source167:      apache2-global.conf
Source168:      apache2-mod_cgid-timeout.conf
Source169:      apache2-protocols.conf
Source190:      apache2-vhost.template
Source191:      apache2-vhost-ssl.template
# READMEs and other documentation
Source200:      apache2-README-access_compat.txt
Source201:      apache2-README-instances.txt
Source202:      apache2-README-configuration.txt
# layout of system dirs configuration, may be upstreamed
Patch0:         apache2-system-dirs-layout.patch
# apachectl is frontend for start_apache2, suse specific
Patch1:         apache2-apachectl.patch
# [bnc#210904] perhaps to be upstreamed
Patch2:         apache2-logresolve-tmp-security.patch
# [bnc#690734] TODO, to be upstreamed
Patch3:         apache2-LimitRequestFieldSize-limits-headers.patch
# [fate317766] backport of an upstream commit
Patch4:         apache2-HttpContentLengthHeadZero-HttpExpectStrict.patch
# [boo#1220681] apache refuses to start, asking for a rebuild with FIPS support
Patch5:         apache2-fips-compatibility-01.patch
Patch6:         apache2-fips-compatibility-02.patch
Patch7:         apache2-fips-compatibility-03.patch
# FIX-UPSTREAM: CVE-2024-39573, bsc#1227271: potential SSRF in mod_rewrite
# - https://svn.apache.org/viewvc?view=revision&revision=1918600
Patch8:         apache2-CVE-2024-39573.patch
# CVE-2023-38709 [bsc#1222330], HTTP response splitting
Patch9:         apache2-CVE-2023-38709.patch
# CVE-2024-27316 [bsc#1221401], HTTP/2 CONTINUATION frames can be utilized for DoS attacks
Patch10:        apache2-CVE-2024-27316.patch
# CVE-2024-24795 [bsc#1222332], HTTP Response Splitting in multiple modules
Patch11:        apache2-CVE-2024-24795.patch
# https://github.com/apache/httpd/pull/444/commits/c2fffd29b0f58bdc9caaaff4fec68e17a676f182
Patch12:        apache2-issue-444.patch
# FIX-UPSTREAM: CVE-2024-38477, bsc#1227270: null pointer dereference in mod_proxy
#  - https://svn.apache.org/viewvc?view=revision&revision=1918607
Patch13:        apache2-CVE-2024-38477.patch
# FIX-UPSTREAM: CVE-2024-38475, bsc#1227268: Improper escaping of output in mod_rewrite
# - https://svn.apache.org/viewvc?view=revision&revision=1918561
Patch14:        apache2-CVE-2024-38475-1.patch
Patch15:        apache2-CVE-2024-38475-2.patch
Patch16:        apache2-CVE-2024-38475-3.patch
# FIX-UPSTREAM: CVE-2024-38476, bsc#1227269: Server may use exploitable/malicious
#   backend application output to run local handlers via internal
#   redirect
# - https://svn.apache.org/viewvc?view=revision&revision=1918560
Patch17:        apache2-CVE-2024-38476-1.patch
Patch18:        apache2-CVE-2024-38476-2.patch
Patch19:        apache2-CVE-2024-38476-3.patch
Patch21:        apache2-CVE-2024-38476-4.patch
Patch22:        apache2-CVE-2024-38476-5.patch
Patch23:        apache2-CVE-2024-38476-6.patch
Patch24:        apache2-CVE-2024-38476-7.patch
Patch25:        apache2-CVE-2024-38476-8.patch
Patch26:        apache2-CVE-2024-38476-9.patch
Patch27:        apache2-CVE-2024-38476-10.patch
Patch28:        apache2-CVE-2024-38476-11.patch
# FIX-UPSTREAM: CVE-2024-36387, bsc#1227272: DoS by null pointer in websocket over HTTP/2
Patch29:        apache2-CVE-2024-36387.patch
# FIX-UPSTREAM: CVE-2024-38474, bsc#1227278: Substitution encoding issue in mod_rewrite
Patch30:        apache2-CVE-2024-38474.patch
# FIX-UPSTREAM: CVE-2024-38473, bsc#1227276: Encoding problem in mod_proxy
Patch31:        apache2-CVE-2024-38473-1.patch
Patch32:        apache2-CVE-2024-38473-2.patch
Patch33:        apache2-CVE-2024-38473-3.patch
Patch34:        apache2-CVE-2024-38473-4.patch
# FIX-UPSTREAM: CVE-2024-39884, bsc#1227353: source code disclosure with handlers configured via AddType
Patch35:        apache2-CVE-2024-39884.patch
# FIX-UPSTREAM: CVE-2024-40725, bsc#1229087: source code disclosure of local content
Patch36:        apache2-CVE-2024-40725.patch
# FIX-UPSTREAM: bsc#1246477 CVE-2024-42516: HTTP response splitting
Patch37:        CVE-2024-42516.patch
# FIX-UPSTREAM: bsc#1246305 CVE-2024-43204: SSRF when mod_proxy is loaded allows an attacker to send outbound proxy requests to a URL controlled by them
Patch38:        CVE-2024-43204.patch
# FIX-UPSTREAM: bsc#1246303 CVE-2024-47252: insufficient escaping of user-supplied data in mod_ssl allows an untrusted SSL/TLS client to insert escape characters into log files
Patch39:        CVE-2024-47252.patch
# FIX-UPSTREAM: bsc#1246302 CVE-2025-23048: access control bypass by trusted clients through TLS 1.3 session resumption in some mod_ssl configurations
Patch40:        CVE-2025-23048.patch
# FIX-UPSTREAM: bsc#1246307 CVE-2025-49630: denial of service can be triggered by untrusted clients causing an assertion in mod_proxy_http2
Patch41:        CVE-2025-49630.patch
# FIX-UPSTREAM: bsc#1246169 CVE-2025-49812: Opossum Attack Application Layer Desynchronization using Opportunistic TLS
Patch42:        CVE-2025-49812.patch
# FIX-UPSTREAM: bsc#1246306 CVE-2025-53020: HTTP/2 denial of service due to late release of memory after effective lifetime
Patch43:        CVE-2025-53020.patch

# PATCH:  https://marc.info/?l=apache-httpd-users&m=147448312531134&w=2
Patch100:       apache-test-application-xml-type.patch
# PATCH:  /test_ssl_var_lookup?SSL_SERVER_SAN_DNS_0 returns <build-host-name>
#         /test_ssl_var_lookup?SSL_SERVER_SAN_OTHER_dnsSRV_0 _https.<build-host-name>
# but Apache::Test::vars()->{servername} returns 'localhost' instead of <build-host-name>
# (see $san_dns and $san_dnssrv variables in t/ssl/varlookup.t)
# even if in live system I do not experience this inconsistency, let's turn off
# these variables from the test
Patch101:       apache-test-turn-off-variables-in-ssl-var-lookup.patch

BuildRequires:  apache-rpm-macros-control
#Since 2.4.7 the event MPM requires apr 1.5.0 or later.
BuildRequires:  apr-devel >= 1.5.0
BuildRequires:  apr-util-devel
BuildRequires:  automake
# for basic testing
BuildRequires:  curl
BuildRequires:  db-devel
%if %{use_firewalld}
BuildRequires:  firewall-macros
%endif
BuildRequires:  libbrotli-devel
%if %{build_http2}
BuildRequires:  pkgconfig(libnghttp2) >= 1.2.1
%endif
BuildRequires:  libcap-devel
BuildRequires:  libxml2-devel
BuildRequires:  lua-devel
BuildRequires:  openldap2-devel
BuildRequires:  openssl-devel >= 0.9.8a
BuildRequires:  pcre2-devel
BuildRequires:  pkgconfig
BuildRequires:  systemd-rpm-macros
BuildRequires:  xz
BuildRequires:  pkgconfig(jansson)
BuildRequires:  pkgconfig(libcurl)
BuildRequires:  pkgconfig(libsystemd)
BuildRequires:  pkgconfig(systemd)
BuildRequires:  pkgconfig(zlib)
# SECTION test requirements
%if %{test}
BuildRequires:  apache-rpm-macros
BuildRequires:  apache2
BuildRequires:  apache2-devel
BuildRequires:  apache2-manual
BuildRequires:  apache2-utils
%endif
%if "%{flavor}" == "test_prefork"
BuildRequires:  apache2-prefork
%endif
%if "%{flavor}" == "test_worker"
BuildRequires:  apache2-worker
%endif
%if "%{flavor}" == "test_event"
BuildRequires:  apache2-event
%endif
%if %{unittest}
# perl-doc is assumed by t/filter/case.t (/usr/lib/perl5/*/pod/perlsub.pod)
BuildRequires:  perl-doc
BuildRequires:  perl(Crypt::SSLeay)
BuildRequires:  perl(DateTime)
BuildRequires:  perl(Devel::CoreStack)
BuildRequires:  perl(Devel::Symdump)
BuildRequires:  perl(Digest::MD5)
BuildRequires:  perl(HTML::HeadParser)
BuildRequires:  perl(HTML::Parser)
BuildRequires:  perl(HTML::Tagset)
BuildRequires:  perl(HTTP::DAV)
BuildRequires:  perl(LWP)
BuildRequires:  perl(LWP::Protocol::https)
BuildRequires:  perl(MIME::Base64)
BuildRequires:  perl(Net::Cmd)
BuildRequires:  perl(URI)
%if %{build_http2}
BuildRequires:  perl(AnyEvent)
BuildRequires:  perl(Protocol::HTTP2::Client)
%endif
BuildRequires:  netcfg
%endif
# /SECTION
%if "%{mpm}" != ""
Provides:       apache2-MPM
%endif
%if "%{flavor}" == ""
Requires:       %{_sysconfdir}/mime.types
Requires:       apache2-MPM
Suggests:       apache2-%{default_mpm}
Recommends:     apache2-utils
Requires:       logrotate
Provides:       %{apache_mmn}
Provides:       http_daemon
Provides:       httpd
Provides:       suse_maintenance_mmn_%{suse_maintenance_mmn}
Obsoletes:      apache2-example-pages
Requires(pre):  group(www)
Requires(pre):  user(wwwrun)
%{?systemd_ordering}
%endif
%if "%{flavor}" == "utils"
Requires:       /usr/bin/which
Recommends:     w3m
%endif
%if "%{flavor}" == "devel"
Requires:       apache2 = %{version}
Requires:       apr-devel
Requires:       apr-util-devel
Requires:       gcc
Provides:       httpd-devel = %{version}
%endif
%if "%{flavor}" == "manual"
Provides:       apache2-doc = %{version}
Obsoletes:      apache2-doc <= %{version}
%endif
%if "%{mpm}" != ""
Requires(pre):  permissions
Requires(post): %fillup_prereq
Requires(post): grep
Requires(post): update-alternatives
Requires(postun): update-alternatives
%endif
%if %{test} || "%{flavor}" == "manual"
BuildArch:      noarch
%endif

%description
The Apache HTTP Server Project is an effort to develop and
maintain an open-source HTTP server for modern operating
systems including UNIX and Windows. The goal of this project
is to provide a secure, efficient and extensible server that
provides HTTP services in sync with the current HTTP standards.

%prep
%autosetup -p1 -n %{upstream_name}-%{version} -a20

#
# BUILD
#

%if ! %{test} && "%{flavor}" != "manual"
%build
echo "================== BUILDING [%{flavor}] flavor"
# configuration (autoreconf, configure) common for all flavors,
# except test and manual

autoreconf --force --install --verbose

# replace PLATFORM string that's seen in the "Server:" header
sed -i -e 's,(" PLATFORM "),(%{platform_string}),' server/core.c
# use mtime of .changes for build time
CHANGES_MTIME=`stat --format="%%y" %{SOURCE6}`
sed -i -e "s/__DATE__ \" \" __TIME__;/\"$CHANGES_MTIME\";/" server/buildmark.c

export CFLAGS="%{optflags} -fPIC -Wall
               -DLDAP_DEPRECATED -DDEFAULT_LISTENBACKLOG=APR_INT32_MAX -DDEFAULT_ERRORLOG='\"%{logfiledir}/error_log\"'"
export CPPFLAGS="%{optflags} -DSSL_EXPERIMENTAL_ENGINE -DMAX_SERVER_LIMIT=200000 -DLDAP_DEPRECATED -DMAXLINE=4096"
cat > config.layout <<-EOF
# SUSE Layout
<Layout SUSE>
Prefix:         %{datadir}
    exec_prefix:   %{_prefix}
    bindir:        %{_bindir}
    sbindir:       %{_sbindir}
    libdir:        %{_libdir}
    libexecdir:    %{libexecdir}
    mandir:        %{_mandir}
    sysconfdir:    %{sysconfdir}
    datadir:       %{datadir}
    installbuilddir: %{installbuilddir}
    errordir:      %{errordir}
    iconsdir:      %{iconsdir}
    htdocsdir:     %{htdocsdir}
    manualdir:     %{manualdir}
    cgidir:        %{cgidir}
    includedir:    %{includedir}
    localstatedir: %{localstatedir}
    runtimedir:    %{runtimedir}
    logfiledir:    %{logfiledir}
    proxycachedir: %{proxycachedir}
</Layout>
EOF
# %%configure (e. g --libexecdir) switches override
# layout paths (use layout because some of variables
# there does not exist as configure switches), so
# override them back
%configure \
        --prefix=%{datadir} \
        --libexecdir=%{libexecdir} \
        --includedir=%{includedir} \
        --sysconfdir=%{sysconfdir} \
        --enable-layout=SUSE \
        --libexecdir=%{libexecdir} \
%if "%{mpm}" != ""
        --with-program-name=httpd-%{mpm} \
%endif
        --with-apr=%{_bindir}/apr-1-config \
        --with-apr-util=%{_bindir}/apu-1-config \
%if "%{mpm}" != ""
        --with-mpm="%{mpm}" \
%endif
%if "%{mpm}" == "worker" || "%{mpm}" == "event"
%ifarch %ix86
%ifnarch i386
                --enable-nonportable-atomics=yes \
%endif
%endif
%endif
        --enable-exception-hook \
        --with-pcre \
        --enable-pie \
        --enable-mods-shared=all \
        --enable-mods-static="%{static_modules}" \
        --enable-ssl=shared \
        --disable-isapi \
        --enable-deflate \
        --enable-brotli \
        --enable-echo \
        --enable-filter \
        --enable-ext-filter \
        --enable-charset-lite \
        --enable-file-cache \
        --enable-logio \
        --enable-dumpio \
        --enable-bucketeer \
        --enable-case_filter \
        --enable-case_filter_in \
        --enable-imagemap \
%if %{build_http2}
        --enable-http2 \
%endif
        --with-ldap \
        --enable-ldap \
        --enable-authnz_ldap \
        --enable-authnz-fcgi \
        --enable-proxy \
        --enable-proxy-connect \
        --enable-proxy-ftp \
        --enable-proxy-http \
%if %{build_http2}
        --enable-proxy-http2 \
%endif
        --enable-proxy-fdpass \
        --enable-cache \
        --enable-disk-cache \
        --enable-mem-cache \
        --enable-version \
        --enable-dav-lock \
        --enable-authn-alias \
        --enable-optional-hook-export \
        --enable-optional-hook-import \
        --enable-optional-fn-import \
        --enable-optional-fn-export \
        --enable-suexec \
        --with-suexec-bin=%{_sbindir}/suexec \
        --with-suexec-caller=%{httpduser} \
        --with-suexec-docroot=%{datadir} \
        --with-suexec-logfile=%{logfiledir}/suexec.log \
        --with-suexec-userdir=%{userdir} \
        --with-suexec-uidmin=96 \
        --with-suexec-gidmin=96 \
        --with-suexec-safepath=%{suexec_safepath} \
        --disable-heartbeat

# MPMs build
%if "%{mpm}" != ""
# adjust SERVER_CONFIG_FILE
sed -i "s:httpd-%{mpm}.conf:httpd.conf:" include/ap_config_auto.h
make %{?_smp_mflags}
%endif

# main package build
%if "%{flavor}" == ""
pushd support
make %{?_smp_flags} suexec
popd
%endif

# utils build
%if "%{flavor}" == "utils"
pushd support
make %{?_smp_mflags}
cp %{SOURCE130} gensslcert
cp %{SOURCE131} check_forensic
cp %{SOURCE132} find_directives
popd
%endif
%endif

#
# INSTALL
#

%if ! %{test}
%install
echo "================== INSTALLING [%{flavor}] flavor"

# MPMs install
%if "%{mpm}" != ""
# install httpd binary
make DESTDIR=%{buildroot} program-install
# install modules
pushd modules
make DESTDIR=%{buildroot} install -j1
popd
# install alternative links (httpd binary, modules)
mkdir -p %{buildroot}%{_sysconfdir}/alternatives
ln -sf %{_sysconfdir}/alternatives/httpd %{buildroot}%{_sbindir}/httpd
mkdir -p %{buildroot}%{_libdir}/apache2/
for module in %{dynamic_modules}; do
  if [ -e %{buildroot}%{libexecdir}/mod_$module.so ]; then
    ln -sf %{_sysconfdir}/alternatives/mod_$module.so %{buildroot}%{_libdir}/apache2/mod_$module.so
  fi
done

# Compat symlinks
ln -s httpd-%{mpm}        %{buildroot}/%{_sbindir}/httpd2-%{mpm}
%endif

# main packge install
%if "%{flavor}" == ""
mkdir -p %{buildroot}%{logfiledir} \
         %{buildroot}%{proxycachedir} \
         %{buildroot}%{localstatedir} \
         %{buildroot}%{libexecdir}

# save MODULE_MAGIC_NUMBER
mkdir -p %{buildroot}/%{_libexecdir}
cat > %{buildroot}/%{_libexecdir}/apache2_MMN <<-EOF
#!/bin/sh
echo %{apache_mmn}
EOF

%if 0%{?suse_version} > 1500
mkdir -p %{buildroot}%{_distconfdir}/logrotate.d
install -m 644 %{SOURCE31} %{buildroot}/%{_distconfdir}/logrotate.d/apache2
%else
mkdir -p %{buildroot}%{_sysconfdir}/logrotate.d
install -m 644 %{SOURCE31} %{buildroot}/%{_sysconfdir}/logrotate.d/apache2
%endif

make DESTDIR=%{buildroot} install-suexec -j1

mkdir -p %{buildroot}%{_unitdir}
install -m 644 %{SOURCE3} %{buildroot}%{_unitdir}/apache2.service
install -m 644 %{SOURCE4} %{buildroot}%{_unitdir}/apache2@.service
install -m 644 %{SOURCE5} %{buildroot}%{_unitdir}/apache2.target
ln -sf service %{buildroot}/%{_sbindir}/rcapache2

install -m 700 %{SOURCE102} %{buildroot}%{_sbindir}/apache2-systemd-ask-pass

install -m 755 %{SOURCE100} %{buildroot}%{_sbindir}/a2enmod
ln -s a2enmod %{buildroot}%{_sbindir}/a2dismod
install -m 755 %{SOURCE101} %{buildroot}%{_sbindir}/a2enflag
ln -s a2enflag %{buildroot}%{_sbindir}/a2disflag
install -m 744 %{SOURCE103} %{buildroot}%{_sbindir}/start_apache2
mkdir -p %{buildroot}/%{_datadir}/apache2/
install -m 644 %{SOURCE104} %{buildroot}/%{_datadir}/apache2/script-helpers

%if %{use_firewalld}
install -D -m 644 %{SOURCE33} %{buildroot}%{_prefix}/lib/firewalld/services/apache2.xml
install -D -m 644 %{SOURCE34} %{buildroot}%{_prefix}/lib/firewalld/services/apache2-ssl.xml
%else
install -d %{buildroot}%{_sysconfdir}/sysconfig/SuSEfirewall2.d/services/
install -m 644 %{SOURCE35} %{buildroot}%{_sysconfdir}/sysconfig/SuSEfirewall2.d/services/apache2
install -m 644 %{SOURCE36} %{buildroot}%{_sysconfdir}/sysconfig/SuSEfirewall2.d/services/apache2-ssl
%endif

mkdir -p %{buildroot}%{_sysconfdir}
mkdir -p %{buildroot}%{_sysconfdir}/apache2/sysconfig.d

mkdir -p %{buildroot}/%{_fillupdir}
install -m 644 %{SOURCE30} %{buildroot}%{_fillupdir}/sysconfig.apache2

mkdir -p %{buildroot}%{sysconfdir}
mkdir -p %{buildroot}%{sysconfdir}/conf.d
for c in default-server.conf \
         errors.conf \
         global.conf \
         httpd.conf \
         listen.conf \
         loadmodule.conf \
         mod_autoindex-defaults.conf \
         mod_info.conf \
         mod_log_config.conf \
         mod_mime-defaults.conf \
         mod_status.conf \
         mod_userdir.conf \
         mod_usertrack.conf \
         server-tuning.conf \
         mod_reqtimeout.conf \
         mod_cgid-timeout.conf \
         ssl-global.conf \
         protocols.conf
do
        install -m 644 %{_sourcedir}/apache2-$c %{buildroot}/%{sysconfdir}/$c
done
cat > %{buildroot}/%{sysconfdir}/uid.conf <<-EOF
        User %{httpduser}
        Group %{httpdgroup}
EOF
tar -xjf %{SOURCE10} -C %{buildroot}/%{sysconfdir}
# fixup libdir
%if "%{_lib}" != "lib64"
sed -e 's/lib64/%{_lib}/' -i \
  %{buildroot}/%{sysconfdir}/loadmodule.conf \
  %{buildroot}/%{_fillupdir}/sysconfig.apache2
%endif
mkdir %{buildroot}/%{sysconfdir}/vhosts.d
install -m 644 %{SOURCE190} %{buildroot}/%{sysconfdir}/vhosts.d/vhost.template
install -m 644 %{SOURCE191} %{buildroot}/%{sysconfdir}/vhosts.d/vhost-ssl.template
install -m 644 docs/conf/charset.conv %{buildroot}/%{sysconfdir}/
install -m 644 docs/conf/magic %{buildroot}/%{sysconfdir}/
ln -sf ../mime.types %{buildroot}/%{sysconfdir}/mime.types

make DESTDIR=%{buildroot} install-icons
make DESTDIR=%{buildroot} install-error
make DESTDIR=%{buildroot} sysconfdir=%{_docdir}/apache2/conf install-conf

cp -r docs/server-status %{buildroot}%{_datadir}/apache2/lua-server-status

mkdir -p %{buildroot}%{_mandir}/man8/
install -D -m 644 docs/man/suexec.8 %{buildroot}%{_mandir}/man8/
install -D -m 644 docs/man/httpd.8 %{buildroot}%{_mandir}/man8/

# Compatibility symlinks for the main flavour
ln -s suexec       %{buildroot}/%{_sbindir}/suexec2
ln -s suexec.8.gz  %{buildroot}/%{_mandir}/man8/suexec2.8.gz
ln -s httpd.8.gz   %{buildroot}/%{_mandir}/man8/httpd2.8.gz

cp %{SOURCE200} README-access_compat.txt
cp %{SOURCE201} README-instances.txt
cp %{SOURCE202} README-configuration.txt
%endif

# utils install
%if "%{flavor}" == "utils"
> utils-filelist
for utility in %{support_bin}; do
  install -D -m 755 support/$utility %{buildroot}%{_bindir}/$utility
  echo %{_bindir}/$utility >> utils-filelist
  if [ -f docs/man/$utility.1 ]; then
    install -D -m 644 docs/man/$utility.1 %{buildroot}%{_mandir}/man1/$utility.1
    echo %{_mandir}/man1/$utility.1.* >> utils-filelist
  fi
done
for utility in %{support_sbin}; do
  install -D -m 755 support/$utility %{buildroot}%{_sbindir}/$utility
  echo %{_sbindir}/$utility >> utils-filelist
  if [ -f docs/man/$utility.8 ]; then
    install -D -m 644 docs/man/$utility.8 %{buildroot}%{_mandir}/man8/$utility.8
    echo %{_mandir}/man8/$utility.8.* >> utils-filelist
  fi
done

# Compatibility symlinks for the utils flavour
for utility in %{compat_bin}; do
  ln -s ${utility} %{buildroot}%{_bindir}/${utility}2
  echo %{_bindir}/${utility}2 >> utils-filelist
  if [ -f docs/man/$utility.1 ]; then
    ln -s ${utility}.1.gz %{buildroot}%{_mandir}/man1/${utility}2.1.gz
    echo %{_mandir}/man1/${utility}2.1.* >> utils-filelist
  fi
done
for utility in %{compat_sbin}; do
  ln -s ${utility} %{buildroot}%{_sbindir}/${utility}2
  echo %{_sbindir}/${utility}2 >> utils-filelist
  if [ -f docs/man/$utility.8 ]; then
    ln -s ${utility}.8.gz %{buildroot}%{_mandir}/man8/${utility}2.8.gz
    echo %{_mandir}/man8/${utility}2.8.* >> utils-filelist
  fi
done
# apache2ctl is the only outlier
ln -s apachectl         %{buildroot}%{_sbindir}/apache2ctl
ln -s apachectl.8.gz    %{buildroot}%{_mandir}/man8/apache2ctl.8.gz
echo %{_sbindir}/apache2ctl         >> utils-filelist
echo %{_mandir}/man8/apache2ctl.8.* >> utils-filelist
%endif

# devel install
%if "%{flavor}" == "devel"
mkdir -p %{buildroot}/%{_bindir}
install -D -m 755 support/apxs %{buildroot}%{_bindir}/
mkdir -p %{buildroot}/%{_mandir}/man1/
install -D -m 644 docs/man/apxs.1 %{buildroot}%{_mandir}/man1/
make DESTDIR=%{buildroot} install-build -j1
make DESTDIR=%{buildroot} install-include -j1

# Compatibility symlinks for the devel flavour
ln -s apxs                %{buildroot}/%{_bindir}/apxs2
mkdir %{buildroot}/%{_sbindir}
ln -s %{_bindir}/apxs     %{buildroot}/%{_sbindir}/apxs
ln -s %{_bindir}/apxs     %{buildroot}/%{_sbindir}/apxs2
ln -s apxs.1.gz           %{buildroot}/%{_mandir}/man1/apxs2.1.gz
%endif

# manual install
%if "%{flavor}" == "manual"
mkdir -p %{buildroot}%{manualdir}
cp -ra docs/manual/* %{buildroot}%{manualdir}
mkdir -p %{buildroot}/%{sysconfdir}/conf.d/
install -m 644 %{SOURCE154} %{buildroot}/%{sysconfdir}/conf.d/manual.conf
%endif

%endif

#
# CHECK
#

%check
# test basic function of just built MPMs
%if ! %{test} && "%{mpm}" != ""
echo "Testing ./httpd-%{mpm}"
test_dir="$PWD/my-test-%{mpm}"
mkdir $test_dir
cat > $test_dir/httpd.conf << EOF
ServerName my_test
ErrorLog $test_dir/error_log
PidFile $test_dir/httpd.pid
User $(id -un)
Group $(id -gn)
Listen 60080
DocumentRoot $test_dir
LoadModule authz_core_module $PWD/modules/aaa/.libs/mod_authz_core.so
EOF
exit_code=0
./httpd-%{mpm} -k start -f $test_dir/httpd.conf
sleep 2
echo 'HTTPD HELLO' > $test_dir/hello.html
curl -s http://localhost:60080/hello.html | grep 'HTTPD HELLO' || exit_code=1
./httpd-%{mpm} -k stop -f $test_dir/httpd.conf
sleep 1
# do not continue %%check phase
exit $exit_code
%endif

# test just built utils
%if "%{flavor}" == "utils"
# htpasswd
echo "Testing htpasswd"
exit_code=0
support/htpasswd -bc htpasswd foo_user foo_password
support/htpasswd -bv htpasswd foo_user foo_password || exit_code=1
# htpasswd
echo "Testing htpasswd"
exit_code=0
support/htdbm -bc htpasswd bar_user bar_password
support/htdbm -bv htpasswd bar_user bar_password || exit_code=2
# do not continue %%check phase
exit $exit_code
%endif

# test _installed_ packages (via test_* flavors)
%if %{test}

%if "%{flavor}" == "test_main"
exit_code=0
# create test configuration, based on default distro one
# with minimum changes to see it is working
mkdir -p $PWD{%{_sysconfdir}/sysconfig,%{localstatedir},%{runtimedir},%{logfiledir}}
# adjust sysconfig file
cp %{_sysconfdir}/sysconfig/apache2 $PWD%{_sysconfdir}/sysconfig/
sed -i -e "s:\(APACHE_HTTPD_CONF=\).*:\1$PWD%{sysconfdir}/httpd.conf:" \
       -e "s:\(%{_localstatedir}\):$PWD\1:" $PWD%{_sysconfdir}/sysconfig/apache2
sed -i 's:\(APACHE_MPM=\).*:\1"prefork":' $PWD%{_sysconfdir}/sysconfig/apache2
# copy and adjust configuration (paths and Listen)
cp -r %{_sysconfdir}/apache2/ %{_sysconfdir}/mime.types etc 2>/dev/null || true
find etc/apache2 -name *.conf | xargs sed -i "s:\(%{_localstatedir}\):$PWD\1:"
find etc/apache2 -name *.conf | xargs sed -i "s:/etc:$PWD/etc:"
sed -i -e 's:80:60080:' -e 's:443:60443:' etc/apache2/listen.conf
# /usr/sbin/start_apache2 is 744
cp %{_sbindir}/start_apache2 .
export START_APACHE_SYSCONFIG_FILE=$PWD/etc/sysconfig/apache2
export START_APACHE_RUN_DIR=$PWD/run
./start_apache2 -k start
sleep 2
curl -s http://localhost:60080/manual/ | grep 'Apache.*HTTP Server.*Documentation' || exit_code=1
curl -s http://localhost:60080/manual/de/ | grep 'Neue Funktionen' || exit_code=2
./start_apache2 -k stop
sleep 1
# do not continue %%check phase
exit $exit_code
%endif

# test of devel package
%if "%{flavor}" == "test_devel"
# apxs test
echo "Testing apxs, compiling example module"
apxs -q CFLAGS | grep "\\%{optflags}"
cp %{SOURCE21} mod_example.c
apxs -c mod_example.c
test_dir="$PWD/my-test-devel"
echo "Try to load example module"
mkdir $test_dir
cat > $test_dir/httpd.conf << EOF
ServerName my_test
ErrorLog $test_dir/error_log
PidFile $test_dir/httpd.pid
User $(id -un)
Group $(id -gn)
Listen 60080
DocumentRoot $test_dir
LoadModule authz_core_module %{libexecdir}-%{default_mpm}/mod_authz_core.so
LoadModule example_module $PWD/.libs/mod_example.so
<Location /hello>
  SetHandler example-handler
</Location>
EOF
exit_code=0
%{_sbindir}/httpd-%{default_mpm} -k start -f $test_dir/httpd.conf
sleep 2
echo "Use example module"
curl -s http://localhost:60080/hello | grep 'Hello, world!' || exit_code=1
%{_sbindir}/httpd-%{default_mpm} -k stop -f $test_dir/httpd.conf
sleep 1
# do not continue %%check phase
exit $exit_code
%endif

# unittest run in test_$MPM flavors
echo "Run httpd-framework unittests"
cd httpd-framework
perl Makefile.PL -apxs %{apache_apxs}
function dep()
{
  dependee="$1"
  requirement="$2"
  # requirement has to be loaded before dependee;
  # there can be duplicite entries in $modules
  # string, will be added only once
  # in load-all-modules.conf
  if echo "$modules" | grep -q "$dependee"; then
    modules="$requirement $modules"
  fi
}
# create a conf loading all MPM's modules
echo >  $PWD/load-all-modules.conf
# hack: sort -u to load mod_proxy before mod_proxy_http, mod_cache before mod_cache_disk, etc.
modules=$(find %{_libdir}/apache2-%{mpm}/ %{_libdir}/apache2/ -name *.so | sed 's:.*/mod_\(.*\).so:\1:' | sort -u)
# fix up dependencies
dep "lbmethod_bybusyness" "proxy"
dep "lbmethod_byrequests" "proxy"
dep "lbmethod_bytraffic"  "proxy"
dep "lbmethod_heartbeat"  "proxy"
for m in $modules; do
  path=$(find %{_libdir}/apache2-%{mpm}/ %{_libdir}/apache2/ -name mod_$m.so | head -n 1)
  if ! grep -q "mod_$m.c" $PWD/load-all-modules.conf; then
    echo "<IfModule !mod_$m.c>"           >> $PWD/load-all-modules.conf
    echo "  LoadModule ${m}_module $path" >> $PWD/load-all-modules.conf
    echo "</IfModule>"                    >> $PWD/load-all-modules.conf
  fi
done
# run the testsuite
echo '#####################################################'
echo "# TESTING %{mpm}"
echo '#'
exit_code=0
t/TEST -clean
t/TEST -httpd /usr/sbin/httpd-%{mpm} -httpd_conf $PWD/load-all-modules.conf -start
t/TEST -run-tests || exit_code=1
t/TEST -stop
exit $exit_code

# end of installed packages test
%endif
exit 0

#
# FILES
#

# MPMs files
%if ! %{test} && "%{mpm}" != ""
%files
%{_sbindir}/httpd
%{_sbindir}/httpd-%{mpm}
%{_sbindir}/httpd2-%{mpm}
%ghost %{_sysconfdir}/alternatives/httpd
# %%ghost %%{_sysconfdir}/alternatives/mod_*.so does not work
%(for module in %{dynamic_modules}; do echo "%ghost %{_sysconfdir}/alternatives/mod_$module.so"; done)
%dir %{_libdir}/apache2-%{mpm}
%dir %{_libdir}/apache2
%{_libdir}/apache2/*.so
%{libexecdir}/mod_*.so
%endif

# main package files
%if "%{flavor}" == ""
%files
%doc INSTALL READM* ABOUT_APACHE CHANGES
%license LICENSE
%attr(750,root,root) %dir %{logfiledir}
%attr(750,%{httpduser},root) %dir %{proxycachedir}
%attr(750,%{httpduser},root) %dir %{localstatedir}
%dir %{libexecdir}
%dir %{_libexecdir}
%attr(755,root,root) %{_libexecdir}/apache2_MMN
%dir %{sysconfdir}
%config %{sysconfdir}/magic
%config %{sysconfdir}/mime.types
%config (noreplace) %{sysconfdir}/*.conf
%config (noreplace) %{sysconfdir}/charset.conv
%{sysconfdir}/vhosts.d/*.template
%dir %{sysconfdir}/ssl.crl
%dir %{sysconfdir}/ssl.crt
%dir %{sysconfdir}/ssl.csr
%dir %attr(700,root,root) %{sysconfdir}/ssl.key
%dir %{sysconfdir}/ssl.prm
%{sysconfdir}/ssl.*/README*
%dir %{sysconfdir}/conf.d
%dir %{sysconfdir}/vhosts.d
%{_fillupdir}/sysconfig.apache2
%if 0%{?suse_version} > 1500
%{_distconfdir}/logrotate.d/apache2
%else
%config(noreplace) %{_sysconfdir}/logrotate.d/apache2
%endif
%{_unitdir}/apache2.service
%{_unitdir}/apache2@.service
%{_unitdir}/apache2.target
%{_sbindir}/apache2-systemd-ask-pass
%{_sbindir}/a2enflag
%{_sbindir}/a2enmod
%{_sbindir}/a2disflag
%{_sbindir}/a2dismod
%{_sbindir}/start_apache2
%{_sbindir}/rcapache2
%{_datadir}/apache2/script-helpers
%verify(not mode) %attr(0755,root,root) %{_sbindir}/suexec
%{_sbindir}/suexec2
%if %{use_firewalld}
%dir %{_prefix}/lib/firewalld
%dir %{_prefix}/lib/firewalld/services
%{_prefix}/lib/firewalld/services/apache2.xml
%{_prefix}/lib/firewalld/services/apache2-ssl.xml
%else
%config %{_sysconfdir}/sysconfig/SuSEfirewall2.d/services/apache2
%config %{_sysconfdir}/sysconfig/SuSEfirewall2.d/services/apache2-ssl
%endif
%{_datadir}/apache2
%{iconsdir}
%{errordir}
%{_mandir}/man8/httpd.8.*
%{_mandir}/man8/httpd2.8.*
%{_mandir}/man8/suexec.8.*
%{_mandir}/man8/suexec2.8.*
%doc support/SHA1
%{_docdir}/apache2/conf
%endif

# utils files
%if "%{flavor}" == "utils"
%files -f utils-filelist
%endif

# devel files
%if "%{flavor}" == "devel"
%files
%{_bindir}/apxs
%{_bindir}/apxs2
%{_sbindir}/apxs
%{_sbindir}/apxs2
%{_mandir}/man1/apxs.1.*
%{_mandir}/man1/apxs2.1.*
%{_datadir}/apache2
%{installbuilddir}
%{includedir}
%endif

# manual files
%if "%{flavor}" == "manual"
%files
%dir %{_datadir}/apache2
%{manualdir}
%dir %{sysconfdir}
%dir %{sysconfdir}/conf.d/
%config %{sysconfdir}/conf.d/manual.conf
%endif

#
# SCRIPTLETS
#

# MPMs scriptlets
%if ! %{test} && "%{mpm}" != ""
%post
%{_sbindir}/update-alternatives --quiet --force \
    --install %{_sbindir}/httpd httpd %{_sbindir}/httpd-%{mpm} %{mpm_alt_prio}
for module in %{dynamic_modules}; do
  if [ -e %{libexecdir}/mod_$module.so ]; then
    %{_sbindir}/update-alternatives --quiet --force \
        --install %{_libdir}/apache2/mod_$module.so mod_$module.so %{libexecdir}/mod_$module.so %{mpm_alt_prio}
  fi
done
exit 0

%postun
if [ "$1" = 1 ]; then
  %apache_request_restart
fi
if [ "$1" = 0 ]; then
  %{_sbindir}/update-alternatives --quiet --force --remove httpd %{_sbindir}/httpd
  for module in %{dynamic_modules}; do
    %{_sbindir}/update-alternatives --quiet --force --remove mod_$module.so %{_libdir}/apache2/mod_$module.so
  done
fi
exit 0

%posttrans
%{_sbindir}/update-alternatives --quiet --force \
    --install %{_sbindir}/httpd httpd %{_sbindir}/httpd-%{mpm} %{mpm_alt_prio}
for module in %{dynamic_modules}; do
  if [ -e %{libexecdir}/mod_$module.so ]; then
    %{_sbindir}/update-alternatives --quiet --force \
        --install %{_libdir}/apache2/mod_$module.so mod_$module.so %{libexecdir}/mod_$module.so %{mpm_alt_prio}
  fi
done

%apache_restart_if_needed
exit 0
%endif

# main package scriptlets
%if "%{flavor}" == ""
%pre
%service_add_pre apache2.service apache2.target
%if 0%{?suse_version} > 1500
# Prepare for migration to /usr/etc; save any old .rpmsave
for i in logrotate.d/apache2 ; do
   test -f %{_sysconfdir}/${i}.rpmsave && mv -v %{_sysconfdir}/${i}.rpmsave %{_sysconfdir}/${i}.rpmsave.old ||:
done
%endif
exit 0

%post
# wwwadmin group existed in past remove after openSUSE-13.2 out of support scope
if grep -q "^wwwadmin:" %{_sysconfdir}/group; then
        groupmod -n www wwwadmin 2>/dev/null ||:
        usermod -g %{httpdgroup} %{httpduser} 2>/dev/null ||:
        usermod -s /bin/false %{httpduser} 2>/dev/null ||:
fi
%service_add_post apache2.service apache2.target
%set_permissions %{_sbindir}/suexec || \
  echo "Please check %{_sysconfdir}/permissions.local for settings of %{_sbindir}/suexec ."
%{fillup_only apache2}
%if %{use_firewalld}
%firewalld_reload
%endif
exit 0

%posttrans
%apache_restart_if_needed
%if 0%{?suse_version} > 1500
# Migration to /usr/etc, restore just created .rpmsave
for i in logrotate.d/apache2 ; do
   test -f %{_sysconfdir}/${i}.rpmsave && mv -v %{_sysconfdir}/${i}.rpmsave %{_sysconfdir}/${i} ||:
done
%endif

%verifyscript
%verify_permissions -e %{_sbindir}/suexec

%preun
%service_del_preun apache2.service apache2.target
exit 0

%postun
%if %{defined service_del_postun_without_restart}
%service_del_postun_without_restart apache2.service apache2.target
%else
DISABLE_RESTART_ON_UPDATE='yes'
%service_del_postun apache2.service apache2.target
%endif
if [ "$1" = 1 ]; then
  %apache_request_restart
fi
%endif

%changelog
* Mon Jul 21 2025 martin.schreiner@suse.com
- Fix the following CVEs / bugs:
  * CVE-2024-42516 bsc#1246477
  * CVE-2024-43204 bsc#1246305
  * CVE-2024-47252 bsc#1246303
  * CVE-2025-23048 bsc#1246302
  * CVE-2025-49630 bsc#1246307
  * CVE-2025-49812 bsc#1246169
  * CVE-2025-53020 bsc#1246306
- Add the following patches:
  * CVE-2024-42516.patch
  * CVE-2024-43204.patch
  * CVE-2024-47252.patch
  * CVE-2025-23048.patch
  * CVE-2025-49630.patch
  * CVE-2025-49812.patch
  * CVE-2025-53020.patch
* Tue Dec 10 2024 martin.schreiner@suse.com
- Fix for bsc#1233433.
  MPM: Copy %%post section triggering update-alternatives to %%posttrans
  as 'zypper dup' never triggers %%post, but it does trigger %%posttrans.
* Fri Oct  4 2024 pgajdos@suse.com
- provide /usr/sbin/apxs and /usr/sbin/apxs2 symlinks to
  /usr/bin/aspx [bsc#1227261]
* Thu Oct  3 2024 martin.schreiner@suse.com
- Apply fix for CVE-2024-40725, bsc#1228097.
  Patch file added:
  * apache2-CVE-2024-40725.patch
* Thu Aug 22 2024 martin.schreiner@suse.com
- Apply fix for CVE-2024-39884, bsc#1227353.
  Patch file added:
  * apache2-CVE-2024-39884.patch
* Wed Aug 14 2024 martin.schreiner@suse.com
- Apply fix for CVE-2024-38474, bsc#1227278.
  Patch file added:
  * apache2-CVE-2024-38474.patch
- Apply fix for CVE-2024-38473, bsc#1227276.
  Patch files added:
  * apache2-CVE-2024-38473-1.patch
  * apache2-CVE-2024-38473-2.patch
  * apache2-CVE-2024-38473-3.patch
  * apache2-CVE-2024-38473-4.patch
* Wed Jul 17 2024 martin.schreiner@suse.com
- Security fix:
  - CVE-2024-36387, bsc#1227272: DoS by null pointer in websocket over HTTP/2
  * Added apache2-CVE-2024-36387.patch
* Mon Jul  8 2024 david.anes@suse.com
- Security fix:
  - CVE-2024-39573, bsc#1227271: potential SSRF in mod_rewrite
  * Added apache2-CVE-2024-39573.patch
  - CVE-2024-38477, bsc#1227270: null pointer dereference in mod_proxy
  * Added apache2-CVE-2024-38477.patch
  - CVE-2024-38475, bsc#1227268: Improper escaping of output in mod_rewrite
  * Added apache2-CVE-2024-38475-1.patch
  * Added apache2-CVE-2024-38475-2.patch
  * Added apache2-CVE-2024-38475-3.patch
  - CVE-2024-38476, bsc#1227269: Server may use exploitable/malicious
  backend application output to run local handlers via internal
  redirect
  * Added apache2-CVE-2024-38476-1.patch
  * Added apache2-CVE-2024-38476-2.patch
  * Added apache2-CVE-2024-38476-3.patch
  * Added apache2-CVE-2024-38476-4.patch
  * Added apache2-CVE-2024-38476-5.patch
  * Added apache2-CVE-2024-38476-6.patch
  * Added apache2-CVE-2024-38476-7.patch
  * Added apache2-CVE-2024-38476-8.patch
  * Added apache2-CVE-2024-38476-9.patch
  * Added apache2-CVE-2024-38476-10.patch
  * Added apache2-CVE-2024-38476-11.patch
* Thu Jun 13 2024 pgajdos@suse.com
- added patches [bsc#1226217]
  https://github.com/apache/httpd/pull/444/commits/c2fffd29b0f58bdc9caaaff4fec68e17a676f182
  + apache2-issue-444.patch
* Tue May 28 2024 pgajdos@suse.com
- security update
- added patches
  fix CVE-2023-38709 [bsc#1222330], HTTP response splitting
  + apache2-CVE-2023-38709.patch
  fix CVE-2024-24795 [bsc#1222332], HTTP Response Splitting in multiple modules
  + apache2-CVE-2024-24795.patch
  fix CVE-2024-27316 [bsc#1221401], HTTP/2 CONTINUATION frames can be utilized for DoS attacks
  + apache2-CVE-2024-27316.patch
* Thu Mar 28 2024 david.anes@suse.com
- Add patches to improve FIPS compatibility (bsc#1220681):
  * apache2-fips-compatibility-01.patch
  * apache2-fips-compatibility-02.patch
  * apache2-fips-compatibility-03.patch
* Fri Mar 22 2024 eugenio.paolantonio@suse.com
- SLE-only: forward-port compatibility symlinks (e.g. httpd2-prefork,
  apache2ctl, htpasswd2) change, including the relative
  manpages (bsc#1221880)
* Mon Mar 18 2024 eugenio.paolantonio@suse.com
- SLE-only: forward-port gensslcert change to generate dhparams certificate
  using a valid FIPS method (bsc#1198913)
* Tue Feb 20 2024 dimstar@opensuse.org
- Use %%autosetup macro. Allows to eliminate the usage of deprecated
  %%patchN.
* Mon Jan 29 2024 dmueller@suse.com
- use grep -E for egrep
* Thu Oct 19 2023 david.anes@suse.com
- Update to 2.4.58:
  * ) SECURITY: CVE-2023-45802: Apache HTTP Server: HTTP/2 stream
    memory not reclaimed right away on RST (cve.mitre.org)
    When a HTTP/2 stream was reset (RST frame) by a client, there
    was a time window were the request's memory resources were not
    reclaimed immediately. Instead, de-allocation was deferred to
    connection close. A client could send new requests and resets,
    keeping the connection busy and open and causing the memory
    footprint to keep on growing. On connection close, all resources
    were reclaimed, but the process might run out of memory before
    that.
    This was found by the reporter during testing of CVE-2023-44487
    (HTTP/2 Rapid Reset Exploit) with their own test client. During
    "normal" HTTP/2 use, the probability to hit this bug is very
    low. The kept memory would not become noticeable before the
    connection closes or times out.
    Users are recommended to upgrade to version 2.4.58, which fixes
    the issue.
    Credits: Will Dormann of Vul Labs
  * ) SECURITY: CVE-2023-43622: Apache HTTP Server: DoS in HTTP/2 with
    initial windows size 0 (cve.mitre.org)
    An attacker, opening a HTTP/2 connection with an initial window
    size of 0, was able to block handling of that connection
    indefinitely in Apache HTTP Server. This could be used to
    exhaust worker resources in the server, similar to the well
    known "slow loris" attack pattern.
    This has been fixed in version 2.4.58, so that such connection
    are terminated properly after the configured connection timeout.
    This issue affects Apache HTTP Server: from 2.4.55 through
    2.4.57.
    Users are recommended to upgrade to version 2.4.58, which fixes
    the issue.
    Credits: Prof. Sven Dietrich (City University of New York)
  * ) SECURITY: CVE-2023-31122: mod_macro buffer over-read
    (cve.mitre.org)
    Out-of-bounds Read vulnerability in mod_macro of Apache HTTP
    Server.This issue affects Apache HTTP Server: through 2.4.57.
    Credits: David Shoon (github/davidshoon)
  * ) mod_ssl: Silence info log message "SSL Library Error: error:0A000126:
    SSL routines::unexpected eof while reading" when using
    OpenSSL 3 by setting SSL_OP_IGNORE_UNEXPECTED_EOF if
    available. [Rainer Jung]
  * ) mod_http2: improved early cleanup of streams.
    [Stefan Eissing]
  * ) mod_proxy_http2: improved error handling on connection errors while
    response is already underway.
    [Stefan Eissing]
  * ) mod_http2: fixed a bug that could lead to a crash in main connection
    output handling. This occured only when the last request on a HTTP/2
    connection had been processed and the session decided to shut down.
    This could lead to an attempt to send a final GOAWAY while the previous
    write was still in progress. See PR 66646.
    [Stefan Eissing]
  * ) mod_proxy_http2: fix `X-Forward-Host` header to carry the correct value.
    Fixes PR66752.
    [Stefan Eissing]
  * ) mod_http2: added support for bootstrapping WebSockets via HTTP/2, as
    described in RFC 8441. A new directive 'H2WebSockets on|off' has been
    added. The feature is by default not enabled.
    As also discussed in the manual, this feature should work for setups
    using "ProxyPass backend-url upgrade=websocket" without further changes.
    Special server modules for WebSockets will have to be adapted,
    most likely, as the handling if IO events is different with HTTP/2.
    HTTP/2 WebSockets are supported on platforms with native pipes. This
    excludes Windows.
    [Stefan Eissing]
  * ) mod_rewrite: Fix a regression with both a trailing ? and [QSA].
    in OCSP stapling. PR 66672. [Frank Meier <frank.meier ergon.ch>, covener]
  * ) mod_http2: fixed a bug in flushing pending data on an already closed
    connection that could lead to a busy loop, preventing the HTTP/2 session
    to close down successfully. Fixed PR 66624.
    [Stefan Eissing]
  * ) mod_http2: v2.0.15 with the following fixes and improvements
  - New directive 'H2EarlyHint name value' to add headers to a response,
    picked up already when a "103 Early Hints" response is sent. 'name' and
    'value' must comply to the HTTP field restrictions.
    This directive can be repeated several times and header fields of the
    same names add. Sending a 'Link' header with 'preload' relation will
    also cause a HTTP/2 PUSH if enabled and supported by the client.
  - Fixed an issue where requests were not logged and accounted in a timely
    fashion when the connection returns to "keepalive" handling, e.g. when
    the request served was the last outstanding one.
    This led to late appearance in access logs with wrong duration times
    reported.
  - Accurately report the bytes sent for a request in the '%%O' Log format.
    This addresses #203, a long outstanding issue where mod_h2 has reported
    numbers over-eagerly from internal buffering and not what has actually
    been placed on the connection.
    The numbers are now the same with and without H2CopyFiles enabled.
    [Stefan Eissing]
  * ) mod_proxy_http2: fix retry handling to not leak temporary errors.
    On detecting that that an existing connection was shutdown by the other
    side, a 503 response leaked even though the request was retried on a
    fresh connection.
    [Stefan Eissing]
  * ) mod_rewrite: Add server directory to include path as mod_rewrite requires
    test_char.h. PR 66571 [Valeria Petrov <valeria.petrov@spinetix.com>]
  * ) mod_http2: new directive `H2ProxyRequests on|off` to enable handling
    of HTTP/2 requests in a forward proxy configuration.
    General forward proxying is enabled via `ProxyRequests`. If the
    HTTP/2 protocol is also enabled for such a server/host, this new
    directive is needed in addition.
    [Stefan Eissing]
  * ) core: Updated conf/mime.types:
  - .js moved from 'application/javascript' to 'text/javascript'
  - .mjs was added as 'text/javascript'
  - add .opus ('audio/ogg')
  - add 'application/vnd.geogebra.slides'
  - add WebAssembly MIME types and extension
    [Mathias Bynens <@mathiasbynens> via PR 318,
    Richard de Boer <richard tubul.net>, Dave Hodder <dmh dmh.org.uk>,
    Zbynek Konecny <zbynek1729 gmail.com>]
  * ) mod_proxy_http2: fixed using the wrong "bucket_alloc" from the backend
    connection when sending data on the frontend one. This caused crashes
    or infinite loops in rare situations.
  * ) mod_proxy_http2: fixed a bug in retry/response handling that could lead
    to wrong status codes or HTTP messages send at the end of response bodies
    exceeding the announced content-length.
  * ) mod_proxy_http2: fix retry handling to not leak temporary errors.
    On detecting that that an existing connection was shutdown by the other
    side, a 503 response leaked even though the request was retried on a
    fresh connection.
  * ) mod_http2: fixed a bug that did cleanup of consumed and pending buckets in
    the wrong order when a bucket_beam was destroyed.
    [Stefan Eissing]
  * ) mod_http2: avoid double chunked-encoding on internal redirects.
    PR 66597 [Yann Ylavic, Stefan Eissing]
  * ) mod_http2: Fix reporting of `Total Accesses` in server-status to not count
    HTTP/2 requests twice. Fixes PR 66801.
    [Stefan Eissing]
  * ) mod_ssl: Fix handling of Certificate Revoked messages
    in OCSP stapling. PR 66626. [<gmoniker gmail.com>]
  * ) mod_http2: fixed a bug in handling of stream timeouts.
    [Stefan Eissing]
  * ) mod_tls: updating to rustls-ffi version 0.9.2 or higher.
    Checking in configure for proper version installed. Code
    fixes for changed clienthello member name.
    [Stefan Eissing]
  * ) mod_md:
  - New directive `MDMatchNames all|servernames` to allow more control over how
    MDomains are matched to VirtualHosts.
  - New directive `MDChallengeDns01Version`. Setting this to `2` will provide
    the command also with the challenge value on `teardown` invocation. In version
    1, the default, only the `setup` invocation gets this parameter.
    Refs #312. Thanks to @domrim for the idea.
  - For Managed Domain in "manual" mode, the checks if all used ServerName and
    ServerAlias are part of the MDomain now reports a warning instead of an error
    (AH10040) when not all names are present.
  - MDChallengeDns01 can now be configured for individual domains.
    Using PR from JÃ©rÃ´me Billiras (@bilhackmac) and adding test case and fixing proper working
  - Fixed a bug found by JÃ©rÃ´me Billiras (@bilhackmac) that caused the challenge
    teardown not being invoked as it should.
  * ) mod_ldap: Avoid performance overhead of APR-util rebind cache for
    OpenLDAP 2.2+.  PR 64414.  [Joe Orton]
  * ) mod_http2: new directive 'H2MaxDataFrameLen n' to limit the maximum
    amount of response body bytes put into a single HTTP/2 DATA frame.
    Setting this to 0 places no limit (but the max size allowed by the
    protocol is observed).
    The module, by default, tries to use the maximum size possible, which is
    somewhat around 16KB. This sets the maximum. When less response data is
    available, smaller frames will be sent.
  * ) mod_md: fixed passing of the server environment variables to programs
    started via MDMessageCmd and MDChallengeDns01 on *nix system.
    See <https://github.com/icing/mod_md/issues/319>.
    [Stefan Eissing]
  * ) mod_dav: Add DavBasePath directive to configure the repository root
    path.  PR 35077.  [Joe Orton]
  * ) mod_alias: Add AliasPreservePath directive to map the full
    path after the alias in a location. [Graham Leggett]
  * ) mod_alias: Add RedirectRelative to allow relative redirect targets to be
    issued as-is. [Eric Covener, Graham Leggett]
  * ) core: Add formats %%{z} and %%{strftime-format} to ErrorLogFormat, and make
    sure that if the format is configured early enough it applies to every log
    line.  PR 62161.  [Yann Ylavic]
  * ) mod_deflate: Add DeflateAlterETag to control how the ETag
    is modified. The 'NoChange' parameter mimics 2.2.x behavior.
    PR 45023, PR 39727. [Eric Covener]
  * ) core: Optimize send_brigade_nonblocking(). [Yann Ylavic, Christophe Jaillet]
  * ) mod_status: Remove duplicate keys "BusyWorkers" and "IdleWorkers".
    Resolve inconsistency between the previous two occurrences by
    counting workers in state SERVER_GRACEFUL no longer as busy,
    but instead in a new counter "GracefulWorkers" (or on HTML
    view as "workers gracefully restarting"). Also add the graceful
    counter as a new column to the existing HTML per process table
    for async MPMs. PR 63300. [Rainer Jung]
* Sat Aug  5 2023 opensuse@dstoecker.de
- Enable building of mod_md
* Fri Apr  7 2023 suse+build@de-korte.org
- Update to 2.4.57:
  * ) mod_proxy: Check before forwarding that a nocanon path has not been
    rewritten with spaces during processing.  [Yann Ylavic]
  * ) mod_proxy: In case that AllowEncodedSlashes is set to NoDecode do not
    double encode encoded slashes in the URL sent by the reverse proxy to the
    backend. [Ruediger Pluem]
  * ) mod_http2: fixed a crash during connection termination. See PR 66539.
    [Stefan Eissing]
  * ) mod_rewrite: Fix a 2.4.56 regression for substitutions ending
    in a question mark. PR66547. [Eric Covener]
  * ) mod_rewrite: Add "BCTLS" and "BNE" RewriteRule flags. Re-allow encoded
    characters on redirections without the "NE" flag.
    [Yann Ylavic, Eric Covener]
  * ) mod_proxy: Fix double encoding of the uri-path of the request forwarded
    to the origin server, when using mapping=encoded|servlet.  [Yann Ylavic]
  * ) mod_mime: Do not match the extention against possible query string
    parameters in case ProxyPass was used with the nocanon option.
    [Ruediger Pluem]
* Wed Mar  8 2023 david.anes@suse.com
- This update fixes the following security issues:
  * CVE-2023-27522 [bsc#1209049]: mod_proxy_uwsgi HTTP response splitting
  * CVE-2023-25690 [bsc#1209047]: HTTP request splitting with mod_rewrite and mod_proxy
- Update to 2.4.56:
  * ) rotatelogs: Add -T flag to allow subsequent rotated logfiles to be
    truncated without the initial logfile being truncated.  [Eric Covener]
  * ) mod_ldap: LDAPConnectionPoolTTL should accept negative values in order to
    allow connections of any age to be reused. Up to now, a negative value
    was handled as an error when parsing the configuration file.  PR 66421.
    [nailyk <bzapache nailyk.fr>, Christophe Jaillet]
  * ) mod_proxy_ajp: Report an error if the AJP backend sends an invalid number
    of headers. [Ruediger Pluem]
  * ) mod_md:
  - Enabling ED25519 support and certificate transparency information when
    building with libressl v3.5.0 and newer. Thanks to Giovanni Bechis.
  - MDChallengeDns01 can now be configured for individual domains.
    Thanks to Jérôme Billiras (@bilhackmac) for the initial PR.
  - Fixed a bug found by Jérôme Billiras (@bilhackmac) that caused the challenge
    teardown not being invoked as it should.
    [Stefan Eissing]
  * ) mod_http2: client resets of HTTP/2 streams led to unwanted 500 errors
    reported in access logs and error documents. The processing of the
    reset was correct, only unneccesary reporting was caused.
    [Stefan Eissing]
  * ) mod_proxy_uwsgi: Stricter backend HTTP response parsing/validation.
    [Yann Ylavic]
* Wed Jan 18 2023 david.anes@suse.com
- This update fixes the following security issues:
  * CVE-2022-37436 [bsc#1207251], mod_proxy backend HTTP response splitting
  * CVE-2022-36760 [bsc#1207250], mod_proxy_ajp Possible request smuggling
  * CVE-2006-20001 [bsc#1207247], mod_dav out of bounds read, or write of zero byte
- Update to 2.4.55:
  * ) SECURITY: CVE-2022-37436: Apache HTTP Server: mod_proxy prior to
    2.4.55 allows a backend to trigger HTTP response splitting
    (cve.mitre.org)
    Prior to Apache HTTP Server 2.4.55, a malicious backend can
    cause the response headers to be truncated early, resulting in
    some headers being incorporated into the response body. If the
    later headers have any security purpose, they will not be
    interpreted by the client.
    Credits: Dimas Fariski Setyawan Putra (@nyxsorcerer)
  * ) SECURITY: CVE-2022-36760: Apache HTTP Server: mod_proxy_ajp
    Possible request smuggling (cve.mitre.org)
    Inconsistent Interpretation of HTTP Requests ('HTTP Request
    Smuggling') vulnerability in mod_proxy_ajp of Apache HTTP Server
    allows an attacker to smuggle requests to the AJP server it
    forwards requests to.  This issue affects Apache HTTP Server
    Apache HTTP Server 2.4 version 2.4.54 and prior versions.
    Credits: ZeddYu_Lu from Qi'anxin Research Institute of Legendsec
    at Qi'anxin Group
  * ) SECURITY: CVE-2006-20001: mod_dav out of  bounds read, or write
    of zero byte (cve.mitre.org)
    A carefully crafted If: request header can cause a memory read,
    or write of a single zero byte, in a pool (heap) memory location
    beyond the header value sent. This could cause the process to
    crash.
    This issue affects Apache HTTP Server 2.4.54 and earlier.
  * ) mod_dav: Open the lock database read-only when possible.
    PR 36636 [Wilson Felipe <wfelipe gmail.com>, manu]
  * ) mod_proxy_http2: apply the standard httpd content type handling
    to responses from the backend, as other proxy modules do. Fixes PR 66391.
    Thanks to Jérôme Billiras for providing the patch.
    [Stefan Eissing]
  * ) mod_dav: mod_dav overrides dav_fs response on PUT failure. PR 35981
    [Basant Kumar Kukreja <basant.kukreja sun.com>, Alejandro Alvarez
    <alejandro.alvarez.ayllon cern.ch>]
  * ) mod_proxy_hcheck: Honor worker timeout settings.  [Yann Ylavic]
  * ) mod_http2: version 2.0.10 of the module, synchronizing changes
    with the gitgub version. This is a partial rewrite of how connections
    and streams are handled.
  - an APR pollset and pipes (where supported) are used to monitor
    the main connection and react to IO for request/response handling.
    This replaces the stuttered timed waits of earlier versions.
  - H2SerializeHeaders directive still exists, but has no longer an effect.
  - Clients that seemingly misbehave still get less resources allocated,
    but ongoing requests are no longer disrupted.
  - Fixed an issue since 1.15.24 that "Server" headers in proxied requests
    were overwritten instead of preserved. [PR by @daum3ns]
  - A regression in v1.15.24 was fixed that could lead to httpd child
    processes not being terminated on a graceful reload or when reaching
    MaxConnectionsPerChild. When unprocessed h2 requests were queued at
    the time, these could stall. See #212.
  - Improved information displayed in 'server-status' for H2 connections when
    Extended Status is enabled. Now one can see the last request that IO
    operations happened on and transferred IO stats are updated as well.
  - When reaching server limits, such as MaxRequestsPerChild, the HTTP/2 connection
    send a GOAWAY frame much too early on new connections, leading to invalid
    protocol state and a client failing the request. See PR65731 at
    <https://bz.apache.org/bugzilla/show_bug.cgi?id=65731>.
    The module now initializes the HTTP/2 protocol correctly and allows the
    client to submit one request before the shutdown via a GOAWAY frame
    is being announced.
  - :scheme pseudo-header values, not matching the
    connection scheme, are forwarded via absolute uris to the
    http protocol processing to preserve semantics of the request.
    Checks on combinations of pseudo-headers values/absence
    have been added as described in RFC 7540. Fixes #230.
  - A bug that prevented trailers (e.g. HEADER frame at the end) to be
    generated in certain cases was fixed. See #233 where it prevented
    gRPC responses to be properly generated.
  - Request and response header values are automatically stripped of leading
    and trialing space/tab characters. This is equivalent behaviour to what
    Apache httpd's http/1.1 parser does.
    The checks for this in nghttp2 v1.50.0+ are disabled.
  - Extensive testing in production done by Alessandro Bianchi (@alexskynet)
    on the v2.0.x versions for stability. Many thanks!
  * ) mod_proxy_http2: fixed #235 by no longer forwarding 'Host:' header when
    request ':authority' is known. Improved test case that did not catch that
    the previous 'fix' was incorrect.
  * ) mod_proxy_hcheck: hcmethod now allows for HTTP/1.1 requests
    using GET11, HEAD11 and/or OPTIONS11. [Jim Jagielski]
  * ) mod_proxy: The AH03408 warning for a forcibly closed backend
    connection is now logged at INFO level.  [Yann Ylavic]
  * ) mod_ssl: When dumping the configuration, the existence of
    certificate/key files is no longer tested.  [Joe Orton]
  * ) mod_authn_core: Add expression support to AuthName and AuthType.
    [Graham Leggett]
  * ) mod_ssl: when a proxy connection had handled a request using SSL, an
    error was logged when "SSLProxyEngine" was only configured in the
    location/proxy section and not the overall server. The connection
    continued to work, the error log was in error. Fixed PR66190.
    [Stefan Eissing]
  * ) mod_proxy_hcheck: Re-enable workers in standard ERROR state. PR 66302.
    [Alessandro Cavaliere <alessandro.cavalier7 unibo.it>]
  * ) mod_proxy_hcheck: Detect AJP/CPING support correctly. PR 66300.
    [Alessandro Cavaliere <alessandro.cavalier7 unibo.it>]
  * ) mod_http2: Export mod_http2.h as public header. [Stefan Eissing]
  * ) mod_md: a new directive `MDStoreLocks` can be used on cluster
    setups with a shared file system for `MDStoreDir` to order
    activation of renewed certificates when several cluster nodes are
    restarted at the same time. Store locks are not enabled by default.
    Restored curl_easy cleanup behaviour from v2.4.14 and refactored
    the use of curl_multi for OCSP requests to work with that.
    Fixes <https://github.com/icing/mod_md/issues/293>.
  * ) core: Avoid an overflow on large inputs in ap_is_matchexp.  PR 66033
    [Ruediger Pluem]
  * ) mod_heartmonitor: Allow "HeartbeatMaxServers 0" to use file based
    storage instead of slotmem. Needed after setting
    HeartbeatMaxServers default to the documented value 10 in 2.4.54.
    PR 66131.  [Jérôme Billiras]
  * ) mod_dav: DAVlockDiscovery option to disable WebDAV lock discovery
    This is a game changer for performances if client use PROPFIND a lot,
    PR 66313. [Emmanuel Dreyfus]
* Mon Dec 12 2022 dmueller@suse.com
- switch to pkgconfig(zlib) so that alternative providers can be
  used
* Fri Sep 23 2022 coolo@suse.com
- The 2.4.54 release brought support for PCRE2, but for that we also
  need to change buildrequires to pcre2-devel
* Tue Sep 20 2022 david.anes@suse.com
- Remove references to README.QUICKSTART and point them to
  https://en.opensuse.org/SDB:Apache_installation (bsc#1203573)
* Thu Sep  1 2022 schubi@suse.com
- Migration to /usr/etc: Saving user changed configuration files
  in /etc and restoring them while an RPM update.
* Tue Jun 28 2022 schubi@intern
- Moved logrotate files from user specific directory /etc/logrotate.d
  to vendor specific directory /usr/etc/logrotate.d.
* Wed Jun  8 2022 pgajdos@suse.com
- update httpd-framework to svn revision 1898917
* Wed Jun  8 2022 pgajdos@suse.com
- version update to 2.4.54
  Changes with Apache 2.4.54
  * ) SECURITY: CVE-2022-31813: mod_proxy X-Forwarded-For dropped by
    hop-by-hop mechanism (cve.mitre.org)
    Apache HTTP Server 2.4.53 and earlier may not send the
    X-Forwarded-* headers to the origin server based on client side
    Connection header hop-by-hop mechanism.
    This may be used to bypass IP based authentication on the origin
    server/application.
    Credits: The Apache HTTP Server project would like to thank
    Gaetan Ferry (Synacktiv) for reporting this issue
  * ) SECURITY: CVE-2022-30556: Information Disclosure in mod_lua with
    websockets (cve.mitre.org)
    Apache HTTP Server 2.4.53 and earlier may return lengths to
    applications calling r:wsread() that point past the end of the
    storage allocated for the buffer.
    Credits: The Apache HTTP Server project would like to thank
    Ronald Crane (Zippenhop LLC) for reporting this issue
  * ) SECURITY: CVE-2022-30522: mod_sed denial of service
    (cve.mitre.org)
    If Apache HTTP Server 2.4.53 is configured to do transformations
    with mod_sed in contexts where the input to mod_sed may be very
    large, mod_sed may make excessively large memory allocations and
    trigger an abort.
    Credits: This issue was found by Brian Moussalli from the JFrog
    Security Research team
  * ) SECURITY: CVE-2022-29404: Denial of service in mod_lua
    r:parsebody (cve.mitre.org)
    In Apache HTTP Server 2.4.53 and earlier, a malicious request to
    a lua script that calls r:parsebody(0) may cause a denial of
    service due to no default limit on possible input size.
    Credits: The Apache HTTP Server project would like to thank
    Ronald Crane (Zippenhop LLC) for reporting this issue
  * ) SECURITY: CVE-2022-28615: Read beyond bounds in
    ap_strcmp_match() (cve.mitre.org)
    Apache HTTP Server 2.4.53 and earlier may crash or disclose
    information due to a read beyond bounds in ap_strcmp_match()
    when provided with an extremely large input buffer.  While no
    code distributed with the server can be coerced into such a
    call, third-party modules or lua scripts that use
    ap_strcmp_match() may hypothetically be affected.
    Credits: The Apache HTTP Server project would like to thank
    Ronald Crane (Zippenhop LLC) for reporting this issue
  * ) SECURITY: CVE-2022-28614: read beyond bounds via ap_rwrite()
    (cve.mitre.org)
    The ap_rwrite() function in Apache HTTP Server 2.4.53 and
    earlier may read unintended memory if an attacker can cause the
    server to reflect very large input using ap_rwrite() or
    ap_rputs(), such as with mod_luas r:puts() function.
    Credits: The Apache HTTP Server project would like to thank
    Ronald Crane (Zippenhop LLC) for reporting this issue
  * ) SECURITY: CVE-2022-28330: read beyond bounds in mod_isapi
    (cve.mitre.org)
    Apache HTTP Server 2.4.53 and earlier on Windows may read beyond
    bounds when configured to process requests with the mod_isapi
    module.
    Credits: The Apache HTTP Server project would like to thank
    Ronald Crane (Zippenhop LLC) for reporting this issue
  * ) SECURITY: CVE-2022-26377: mod_proxy_ajp: Possible request
    smuggling (cve.mitre.org)
    Inconsistent Interpretation of HTTP Requests ('HTTP Request
    Smuggling') vulnerability in mod_proxy_ajp of Apache HTTP Server
    allows an attacker to smuggle requests to the AJP server it
    forwards requests to.  This issue affects Apache HTTP Server
    Apache HTTP Server 2.4 version 2.4.53 and prior versions.
    Credits: Ricter Z @ 360 Noah Lab
  * ) mod_ssl: SSLFIPS compatible with OpenSSL 3.0.  PR 66063.
    [Petr Sumbera <petr.sumbera oracle.com>, Yann Ylavic]
  * ) mod_proxy_http: Avoid 417 responses for non forwardable 100-continue.
    PR 65666.  [Yann Ylavic]
  * ) mod_md:  a bug was fixed that caused very large MDomains
    with the combined DNS names exceeding ~7k to fail, as
    request bodies would contain partially wrong data from
    uninitialized memory. This would have appeared as failure
    in signing-up/renewing such configurations.
    [Stefan Eissing, Ronald Crane (Zippenhop LLC)]
  * ) mod_proxy_http: Avoid 417 responses for non forwardable 100-continue.
    PR 65666.  [Yann Ylavic]
  * ) MPM event: Restart children processes killed before idle maintenance.
    PR 65769.  [Yann Ylavic, Ruediger Pluem]
  * ) ab: Allow for TLSv1.3 when the SSL library supports it.
    [abhilash1232 gmail.com, xiaolongx.jiang intel.com, Yann Ylavic]
  * ) core: Disable TCP_NOPUSH optimization on OSX since it might introduce
    transmission delays.  PR 66019.  [Yann Ylavic]
  * ) MPM event: Fix accounting of active/total processes on ungraceful restart,
    PR 66004 (follow up to PR 65626 from 2.4.52).  [Yann Ylavic]
  * ) core: make ap_escape_quotes() work correctly on strings
    with more than MAX_INT/2 characters, counting quotes double.
    Credit to <generalbugs@zippenhop.com> for finding this.
    [Stefan Eissing]
  * ) mod_md: the `MDCertificateAuthority` directive can take more than one URL/name of
    an ACME CA. This gives a failover for renewals when several consecutive attempts
    to get a certificate failed.
    A new directive was added: `MDRetryDelay` sets the delay of retries.
    A new directive was added: `MDRetryFailover` sets the number of errored
    attempts before an alternate CA is selected for certificate renewals.
    [Stefan Eissing]
  * ) mod_http2: remove unused and insecure code. Fixes PR66037.
    Thanks to Ronald Crane (Zippenhop LLC) for reporting this.
    [Stefan Eissing]
  * ) mod_proxy: Add backend port to log messages to
    ease identification of involved service.  [Rainer Jung]
  * ) mod_http2: removing unscheduling of ongoing tasks when
    connection shows potential abuse by a client. This proved
    counter-productive and the abuse detection can false flag
    requests using server-side-events.
    Fixes <https://github.com/icing/mod_h2/issues/231>.
    [Stefan Eissing]
  * ) mod_md: Implement full auto status ("key: value" type status output).
    Especially not only status summary counts for certificates and
    OCSP stapling but also lists. Auto status format is similar to
    what was used for mod_proxy_balancer.
    [Rainer Jung]
  * ) mod_md: fixed a bug leading to failed transfers for OCSP
    stapling information when more than 6 certificates needed
    updates in the same run.  [Stefan Eissing]
  * ) mod_proxy: Set a status code of 502 in case the backend just closed the
    connection in reply to our forwarded request.  [Ruediger Pluem]
  * ) mod_md: a possible NULL pointer deref was fixed in
    the JSON code for persisting time periods (start+end).
    Fixes #282 on mod_md's github.
    Thanks to @marcstern for finding this.  [Stefan Eissing]
  * ) mod_heartmonitor: Set the documented default value
    "10" for HeartbeatMaxServers instead of "0". With "0"
    no shared memory slotmem was initialized.  [Rainer Jung]
  * ) mod_md: added support for managing certificates via a
    local tailscale daemon for users of that secure networking.
    This gives trusted certificates for tailscale assigned
    domain names in the *.ts.net space.
    [Stefan Eissing]
- modified patches
  %% apache-test-application-xml-type.patch (refreshed)
  %% apache-test-turn-off-variables-in-ssl-var-lookup.patch (refreshed)
  %% apache2-HttpContentLengthHeadZero-HttpExpectStrict.patch (refreshed)
* Mon Mar 14 2022 pgajdos@suse.com
- httpd-framework updated to svn1898917
- deleted patches
  - apache-test-DirectorySlash-NotFound-logic.patch (upstreamed)
  - apache2-perl-io-socket.patch (upstreamed)
* Mon Mar 14 2022 pgajdos@suse.com
- version update to 2.4.53
  * ) SECURITY: CVE-2022-23943: mod_sed: Read/write beyond bounds
    (cve.mitre.org)
    Out-of-bounds Write vulnerability in mod_sed of Apache HTTP
    Server allows an attacker to overwrite heap memory with possibly
    attacker provided data.
    This issue affects Apache HTTP Server 2.4 version 2.4.52 and
    prior versions.
    Credits: Ronald Crane (Zippenhop LLC)
  * ) SECURITY: CVE-2022-22721: core: Possible buffer overflow with
    very large or unlimited LimitXMLRequestBody (cve.mitre.org)
    If LimitXMLRequestBody is set to allow request bodies larger
    than 350MB (defaults to 1M) on 32 bit systems an integer
    overflow happens which later causes out of bounds writes.
    This issue affects Apache HTTP Server 2.4.52 and earlier.
    Credits: Anonymous working with Trend Micro Zero Day Initiative
  * ) SECURITY: CVE-2022-22720: HTTP request smuggling vulnerability
    in Apache HTTP Server 2.4.52 and earlier (cve.mitre.org)
    Apache HTTP Server 2.4.52 and earlier fails to close inbound
    connection when errors are encountered discarding the request
    body, exposing the server to HTTP Request Smuggling
    Credits: James Kettle <james.kettle portswigger.net>
  * ) SECURITY: CVE-2022-22719: mod_lua Use of uninitialized value of
    in r:parsebody (cve.mitre.org)
    A carefully crafted request body can cause a read to a random
    memory area which could cause the process to crash.
    This issue affects Apache HTTP Server 2.4.52 and earlier.
    Credits: Chamal De Silva
  * ) core: Make sure and check that LimitXMLRequestBody fits in system memory.
    [Ruediger Pluem, Yann Ylavic]
  * ) core: Simpler connection close logic if discarding the request body fails.
    [Yann Ylavic, Ruediger Pluem]
  * ) mod_http2: preserve the port number given in a HTTP/1.1
    request that was Upgraded to HTTP/2. Fixes PR65881.
    [Stefan Eissing]
  * ) mod_proxy: Allow for larger worker name.  PR 53218.  [Yann Ylavic]
  * ) dbm: Split the loading of a dbm driver from the opening of a dbm file. When
    an attempt to load a dbm driver fails, log clearly which driver triggered
    the error (not "default"), and what the error was. [Graham Leggett]
  * ) mod_proxy: Use the maxium of front end and backend timeouts instead of the
    minimum when tunneling requests (websockets, CONNECT requests).
    Backend timeouts can be configured more selectively (per worker if needed)
    as front end timeouts and typically the backend timeouts reflect the
    application requirements better.  PR 65886 [Ruediger Pluem]
  * ) ap_regex: Use Thread Local Storage (TLS) to recycle ap_regexec() buffers
    when an efficient TLS implementation is available. [Yann Ylavic]
  * ) core, mod_info: Add compiled and loaded PCRE versions to version
    number display.  [Rainer Jung]
  * ) mod_md: do not interfere with requests to /.well-known/acme-challenge/
    resources if challenge type 'http-01' is not configured for a domain.
    Fixes <https://github.com/icing/mod_md/issues/279>.
    [Stefan Eissing]
  * ) mod_dav: Fix regression when gathering properties which could lead to huge
    memory consumption proportional to the number of resources.
    [Evgeny Kotkov, Ruediger Pluem]
  * ) Support pcre2 (10.x) library in place of the now end-of-life pcre (8.x)
    for regular expression evaluation. This depends on locating pcre2-config.
    [William Rowe, Petr Pisar <ppisar redhat.com>, Rainer Jung]
  * ) Add the ldap function to the expression API, allowing LDAP filters and
    distinguished names based on expressions to be escaped correctly to
    guard against LDAP injection. [Graham Leggett]
  * ) mod_md: the status description in MDomain's JSON, exposed in the
    md-status handler (if configured) did sometimes not carry the correct
    message when certificates needed renew.
    [Stefan Eissing]
  * ) mpm_event: Fix a possible listener deadlock on heavy load when restarting
    and/or reaching MaxConnectionsPerChild.  PR 65769.  [Yann Ylavic]
* Thu Jan 27 2022 pgajdos@suse.com
- ssl-global.conf: set SSLCipherSuite to PROFILE=SYSTEM instead of
  DEFAULT_SUSE [jsc#SLE-22561]
- set also SSLProxyCipherSuite to PROFILE=SYSTEM
- modified sources
  %% apache2-ssl-global.conf
* Tue Jan 11 2022 david.anes@suse.com
- Align some defaults in apache2-server-tuning.conf to upstream
  defaults:
  * Updated MaxRequestWorkers and ServerLimit to 256. [bsc#1194062]
- The old name MaxRequestsPerChild is changed to MaxConnectionsPerChild.
  * See https://httpd.apache.org/docs/2.4/mod/mpm_common.html#maxconnectionsperchild
* Mon Jan 10 2022 dimstar@opensuse.org
- Add apache2-perl-io-socket.patch: t/ssl/ocsp.t: Handle new error
  message raised by IO-Socket-SSL 2.073.
* Mon Jan 10 2022 olaf@aepfle.de
- remove instance units from post scripts, they can not be reloaded
* Mon Dec 20 2021 david.anes@suse.com
- version update to 2.4.52:
  * fix CVE-2021-44224: NULL dereference or SSRF in forward proxy
    configurations [boo#1193943]
  * fix CVE-2021-44790: buffer overflow when parsing multipart
    content in mod_lua [boo#1193942]
  * ) http: Enforce that fully qualified uri-paths not to be forward-proxied
    have an http(s) scheme, and that the ones to be forward proxied have a
    hostname, per HTTP specifications.
  * ) OpenSSL autoconf detection improvement: pick up openssl.pc in the
    already sent it to the client.
  * ) mod_http: Correctly sent a 100 Continue status code when sending an interim
    response as result of an Expect: 100-Continue in the request and not the
    current status code of the request
  * ) mod_dav: Some DAV extensions, like CalDAV, specify both document
    elements and property elements that need to be taken into account
    when generating a property. The document element and property element
    are made available in the dav_liveprop_elem structure by calling
    dav_get_liveprop_element()
  * ) mod_dav: Add utility functions dav_validate_root_ns(),
    dav_find_child_ns(), dav_find_next_ns(), dav_find_attr_ns() and
    dav_find_attr() so that other modules get to play too.
  * ) mpm_event: Restart stopping of idle children after a load peak
  * ) mod_http2: fixes 2 regressions in server limit handling.
    1. When reaching server limits, such as MaxRequestsPerChild, the
    HTTP/2 connection send a GOAWAY frame much too early on new
    connections, leading to invalid protocol state and a client
    failing the request
    The module now initializes the HTTP/2 protocol correctly and
    allows the client to submit one request before the shutdown
    via a GOAWAY frame is being announced.
    2. A regression in v1.15.24 was fixed that could lead to httpd
    child processes not being terminated on a graceful reload or
    when reaching MaxConnectionsPerChild. When unprocessed h2
    requests were queued at the time, these could stall.
    See <https://github.com/icing/mod_h2/issues/212>.
  * ) mod_ssl: Add build support for OpenSSL v3
  * ) mod_proxy_connect: Honor the smallest of the backend or client timeout
    while tunneling
  * ) mod_proxy: SetEnv proxy-nohalfclose (or alike) allows to disable TCP
    half-close forwarding when tunneling protocols
  * ) core: Be safe with ap_lingering_close() called with a socket NULL-ed by
    a third-party module.  PR 65627.
  * ) mod_md: Fix memory leak in case of failures to load the private key.
  * ) mod_md: adding v2.4.8 with the following changes
  - Added support for ACME External Account Binding (EAB).
    Use the new directive `MDExternalAccountBinding` to provide the
    server with the value for key identifier and hmac as provided by
    your CA.
    While working on some servers, EAB handling is not uniform
    across CAs. First tests with a Sectigo Certificate Manager in
    demo mode are successful. But ZeroSSL, for example, seems to
    regard EAB values as a one-time-use-only thing, which makes them
    fail if you create a seconde account or retry the creation of the
    first account with the same EAB.
  - The directive 'MDCertificateAuthority' now checks if its parameter
    is a http/https url or one of a set of known names. Those are
    'LetsEncrypt', 'LetsEncrypt-Test', 'Buypass' and 'Buypass-Test'
    for now and they are not case-sensitive.
    The default of LetsEncrypt is unchanged.
  - `MDContactEmail` can now be specified inside a `<MDomain dnsname>`
    section.
  - Treating 401 HTTP status codes for orders like 403, since some ACME
    servers seem to prefer that for accessing oders from other accounts.
  - When retrieving certificate chains, try to read the repsonse even
    if the HTTP Content-Type is unrecognized.
  - Fixed a bug that reset the error counter of a certificate renewal
    and prevented the increasing delays in further attempts.
  - Fixed the renewal process giving up every time on an already existing
    order with some invalid domains. Now, if such are seen in a previous
    order, a new order is created for a clean start over again.
    See <https://github.com/icing/mod_md/issues/268>
  - Fixed a mixup in md-status handler when static certificate files
    and renewal was configured at the same time.
  * ) mod_md: values for External Account Binding (EAB) can
    now also be configured to be read from a separate JSON
    file. This allows to keep server configuration permissions
    world readable without exposing secrets.
  * ) mod_proxy_uwsgi: Remove duplicate slashes at the beginning of PATH_INFO.
* Wed Nov 24 2021 pgajdos@suse.com
- httpd-framework updated to svn1894461
- added patches
  fix reverted logic, DirectorySlash NotFound is available in trunk only
  + apache-test-DirectorySlash-NotFound-logic.patch
- do not consider php tests, they do not run anyway
* Sun Nov 14 2021 suse+build@de-korte.org
- add section for the 'event' MPM to apache2-server-tuning.conf
* Thu Oct  7 2021 michael@stroeder.com
- version update to 2.4.51
  * ) SECURITY: CVE-2021-42013: Path Traversal and Remote Code
    Execution in Apache HTTP Server 2.4.49 and 2.4.50 (incomplete
    fix of CVE-2021-41773) (cve.mitre.org)
  * ) core: Add ap_unescape_url_ex() for better decoding control, and deprecate
    unused AP_NORMALIZE_DROP_PARAMETERS flag.
* Mon Oct  4 2021 michael@stroeder.com
- version update to 2.4.50
  * ) core: AP_NORMALIZE_DECODE_UNRESERVED should normalize the second dot in
    the uri-path when it's preceded by a dot.  [Yann Ylavic]
  * ) mod_md: when MDMessageCmd for a 'challenge-setup:<type>:<dnsname>'
    fails (!= 0 exit), the renewal process is aborted and an error is
    reported for the MDomain. This provides scripts that distribute
    information in a cluster to abort early with bothering an ACME
    server to validate a dns name that will not work. The common
    retry logic will make another attempt in the future, as with
    other failures.
    Fixed a bug when adding private key specs to an already working
    MDomain, see <https://github.com/icing/mod_md/issues/260>.
    [Stefan Eissing]
  * ) mod_proxy: Handle UDS URIs with empty hostname ("unix:///...") as if they
    had no hostname ("unix:/...").  [Yann Ylavic]
  * ) mod_md: fixed a bug in handling multiple parallel OCSP requests. These could
    run into an assertion which terminated (and restarted) the child process where
    the task was running. Eventually, all OCSP responses were collected, but not
    in the way that things are supposed to work.
    See also <https://bz.apache.org/bugzilla/show_bug.cgi?id=65567>.
    The bug was possibly triggered when more than one OCSP status needed updating
    at the same time. For example for several renewed certificates after a server
    reload.
  * ) mod_rewrite: Fix UDS ("unix:") scheme for [P] rules.  PR 57691 + 65590.
    [Janne Peltonen <janne.peltonen sange.fi>]
  * ) event mpm: Correctly count active child processes in parent process if
    child process dies due to MaxConnectionsPerChild.
    PR 65592 [Ruediger Pluem]
  * ) mod_http2: when a server is restarted gracefully, any idle h2 worker
    threads are shut down immediately.
    Also, change OpenSSL API use for deprecations in OpenSSL 3.0.
    Adds all other, never proposed code changes to make a clean
    sync of http2 sources. [Stefan Eissing]
  * ) mod_dav: Correctly handle errors returned by dav providers on REPORT
    requests. [Ruediger Pluem]
  * ) core: do not install core input/output filters on secondary
    connections. [Stefan Eissing]
  * ) core: Add ap_pre_connection() as a wrapper to ap_run_pre_connection()
    and use it to prevent that failures in running the pre_connection
    hook cause crashes afterwards. [Ruediger Pluem]
  * ) mod_speling: Add CheckBasenameMatch PR 44221.  [Christophe Jaillet]
* Fri Sep 17 2021 pgajdos@suse.com
- version update to 2.4.49
  * ) core/mod_proxy/mod_ssl:
    Adding `outgoing` flag to conn_rec, indicating a connection is
    initiated by the server to somewhere, in contrast to incoming
    connections from clients.
    Adding 'ap_ssl_bind_outgoing()` function that marks a connection
    as outgoing and is used by mod_proxy instead of the previous
    optional function `ssl_engine_set`. This enables other SSL
    module to secure proxy connections.
    The optional functions `ssl_engine_set`, `ssl_engine_disable` and
    `ssl_proxy_enable` are now provided by the core to have backward
    compatibility with non-httpd modules that might use them. mod_ssl
    itself no longer registers these functions, but keeps them in its
    header for backward compatibility.
    The core provided optional function wrap any registered function
    like it was done for `ssl_is_ssl`.
    [Stefan Eissing]
  * ) mod_ssl: Support logging private key material for use with
    wireshark via log file given by SSLKEYLOGFILE environment
    variable.  Requires OpenSSL 1.1.1.  PR 63391.  [Joe Orton]
  * ) mod_proxy: Do not canonicalize the proxied URL when both "nocanon" and
    "ProxyPassInterpolateEnv On" are configured.  PR 65549.
    [Joel Self <joelself gmail.com>]
  * ) mpm_event: Fix children processes possibly not stopped on graceful
    restart.  PR 63169.  [Joel Self <joelself gmail.com>]
  * ) mod_proxy: Fix a potential infinite loop when tunneling Upgrade(d)
    protocols from mod_proxy_http, and a timeout triggering falsely when
    using mod_proxy_wstunnel, mod_proxy_connect or mod_proxy_http with
    upgrade= setting.  PRs 65521 and 65519.  [Yann Ylavic]
  * ) mod_unique_id: Reduce the time window where duplicates may be generated
    PR 65159
    [Christophe Jaillet]
  * ) mpm_prefork: Block signals for child_init hooks to prevent potential
    threads created from there to catch MPM's signals.
    [Ruediger Pluem, Yann Ylavic]
  * ) Revert "mod_unique_id: Fix potential duplicated ID generation under heavy load.
    PR 65159" added in 2.4.47.
    This causes issue on Windows.
    [Christophe Jaillet]
  * ) mod_proxy_uwsgi: Fix PATH_INFO setting for generic worker.  [Yann Ylavic]
  * ) mod_md: Certificate/keys pairs are verified as matching before a renewal is accepted
    as successful or a staged renewal is replacing the existing certificates.
    This avoid potential mess ups in the md store file system to render the active
    certificates non-working. [@mkauf]
  * ) mod_proxy: Faster unix socket path parsing in the "proxy:" URL.
    [Yann Ylavic]
  * ) mod_ssl: tighten the handling of ALPN for outgoing (proxy)
    connections. If ALPN protocols are provided and sent to the
    remote server, the received protocol selected is inspected
    and checked for a match. Without match, the peer handshake
    fails.
    An exception is the proposal of "http/1.1" where it is
    accepted if the remote server did not answer ALPN with
    a selected protocol. This accomodates for hosts that do
    not observe/support ALPN and speak http/1.x be default.
  * ) mod_proxy: Fix possible reuse/merging of Proxy(Pass)Match worker instances
    with others when their URLs contain a '$' substitution.  PR 65419 + 65429.
    [Yann Ylavic]
  * ) mod_dav: Add method_precondition hook. WebDAV extensions define
    conditions that must exist before a WebDAV method can be executed.
    This hook allows a WebDAV extension to verify these preconditions.
    [Graham Leggett]
  * ) Add hooks deliver_report and gather_reports to mod_dav.h. Allows other
    modules apart from versioning implementations to handle the REPORT method.
    [Graham Leggett]
  * ) Add dav_get_provider(), dav_open_lockdb(), dav_close_lockdb() and
    dav_get_resource() to mod_dav.h. [Graham Leggett]
  * ) core: fix ap_escape_quotes substitution logic. [Eric Covener]
  * ) Easy patches: synch 2.4.x and trunk
  - mod_auth_basic: Use ap_cstr_casecmp instead of strcasecmp.
  - mod_ldap: log and abort locking errors.
  - mod_ldap: style fix for r1831165
  - mod_ldap: build break fix for r1831165
  - mod_deflate: Avoid hard-coded "%%ld" format strings in mod_deflate's logging statements
  - mod_deflate: Use apr_uint64_t instead of uint64_t (follow up to r1849590)
  - mod_forensic: Follow up to r1856490: missing one mod_log_forensic test_char_table case.
  - mod_rewrite: Save a few cycles.
  - mod_request: Fix a comment (missing '_' in 'keep_body') and some style issues
  - core: remove extra whitespace in HTTP_NOT_IMPLEMENTED
    [Christophe Jaillet]
  * ) core/mpm: add hook 'child_stopping` that gets called when the MPM is
    stopping a child process. The additional `graceful` parameter allows
    registered hooks to free resources early during a graceful shutdown.
    [Yann Ylavic, Stefan Eissing]
  * ) mod_proxy: Fix icomplete initialization of BalancerMember(s) from the
    balancer-manager, which can lead to a crash.  [Yann Ylavic]
  * ) mpm_event: Fix graceful stop/restart of children processes if connections
    are in lingering close for too long.  [Yann Ylavic]
  * ) mod_md: fixed a potential null pointer dereference if ACME/OCSP
    server returned 2xx responses without content type. Reported by chuangwen.
    [chuangwen, Stefan Eissing]
  * ) mod_md:
  - Domain names in `<MDomain ...>` can now appear in quoted form.
  - Fixed a failure in ACME challenge selection that aborted further searches
    when the tls-alpn-01 method did not seem to be suitable.
  - Changed the tls-alpn-01 setup to only become unsuitable when none of the
    dns names showed support for a configured 'Protocols ... acme-tls/1'. This
    allows use of tls-alpn-01 for dns names that are not mapped to a VirtualHost.
  * ) Add CPING to health check logic. [Jean-Frederic Clere]
  * ) core: Split ap_create_request() from ap_read_request(). [Graham Leggett]
  * ) core, h2: common ap_parse_request_line() and ap_check_request_header()
    code. [Yann Ylavic]
  * ) core: Add StrictHostCheck to allow unconfigured hostnames to be
    rejected. [Eric Covener]
  * ) htcacheclean: Improve help messages.  [Christophe Jaillet]
- modified patches
  %% apache2-HttpContentLengthHeadZero-HttpExpectStrict.patch (refreshed)
- modified sources
  %% apache2.keyring
* Mon Aug  2 2021 pgajdos@suse.com
- introduce APACHE_TACEENABLE sysconfig variable, which translates
  to /etc/apache2/sysconfig.d/global.conf:TraceEnable on/off
- modified sources
  %% apache2-global.conf
  %% apache2-start_apache2
  %% sysconfig.apache2
* Wed Jun  2 2021 pgajdos@suse.com
- version update to 2.4.48
  Changes with Apache 2.4.48
  * ) mod_proxy_wstunnel: Add ProxyWebsocketFallbackToProxyHttp to opt-out the
    fallback to mod_proxy_http for WebSocket upgrade and tunneling.
    [Yann Ylavic]
  * ) mod_proxy: Fix flushing of THRESHOLD_MIN_WRITE data while tunneling.
    BZ 65294.  [Yann Ylavic]
  * ) core: Fix a regression that stripped the ETag header from 304 responses.
    PR 61820 [Ruediger Pluem, Roy T. Fielding]
  * ) core: Adding SSL related inquiry functions to the server API.
    These function are always available, even when no module providing
    SSL is loaded. They provide their own "shadowing" implementation for
    the optional functions of similar name that mod_ssl and impersonators
    of mod_ssl provide.
    This enables loading of several SSL providing modules when all but
    one of them registers itself into the new hooks. Two old-style SSL
    modules will not work, as they replace the others optional functions
    with their own.
    Modules using the old-style optional functions will continue to work
    as core supplies its own versions of those.
    The following has been added so far:
  - ap_ssl_conn_is_ssl() to query if a connection is using SSL.
  - ap_ssl_var_lookup() to query SSL related variables for a
    server/connection/request.
  - Hooks for 'ssl_conn_is_ssl' and 'ssl_var_lookup' where modules
    providing SSL can install their own value supplying functions.
  - ap_ssl_add_cert_files() to enable other modules like mod_md to provide
    certificate and keys for an SSL module like mod_ssl.
  - ap_ssl_add_fallback_cert_files() to enable other modules like mod_md to
    provide a fallback certificate in case no 'proper' certificate is
    available for an SSL module like mod_ssl.
  - ap_ssl_answer_challenge() to enable other modules like mod_md to
    provide a certificate as used in the RFC 8555 'tls-alpn-01' challenge
    for the ACME protocol for an SSL module like mod_ssl. The function
    and its hook provide PEM encoded data instead of file names.
  - Hooks for 'ssl_add_cert_files', 'ssl_add_fallback_cert_files' and
    'ssl_answer_challenge' where modules like mod_md can provide providers
    to the above mentioned functions.
  - These functions reside in the new 'http_ssl.h' header file.
    [Stefan Eissing]
  * ) core/mod_ssl/mod_md: adding OCSP response provisioning as core feature. This
    allows modules to access and provide OCSP response data without being tied
    of each other. The data is exchanged in standard, portable formats (PEM encoded
    certificates and DER encoded responses), so that the actual SSL/crypto
    implementations used by the modules are independant of each other.
    Registration and retrieval happen in the context of a server (server_rec)
    which modules may use to decide if they are configured for this or not.
    The area of changes:
    1. core: defines 2 functions in include/http_ssl.h, so that modules may
    register a certificate, together with its issuer certificate for OCSP
    response provisioning and ask for current response data (DER bytes) later.
    Also, 2 hooks are defined that allow modules to implement this OCSP
    provisioning.
    2. mod_ssl uses the new functions, in addition to what it did already, to
    register its certificates this way. If no one is interested in providing
    OCSP, it falls back to its own (if configured) stapling implementation.
    3. mod_md registers itself at the core hooks for OCSP provisioning. Depending
    on configuration, it will accept registrations of its own certificates only,
    all certificates or none.
    [Stefan Eissing]
  * ) mod_md: v2.4.0 with improvements and bugfixes
  - MDPrivateKeys allows the specification of several types. Beside "RSA" plus
    optional key lengths elliptic curves can be configured. This means you can
    have multiple certificates for a Managed Domain with different key types.
    With ```MDPrivateKeys secp384r1 rsa2048``` you get one ECDSA  and one RSA
    certificate and all modern client will use the shorter ECDSA, while older
    client will get the RSA certificate.
    Many thanks to @tlhackque who pushed and helped on this.
  - Support added for MDomains consisting of a wildcard. Configuring
    ```MDomain *.host.net``` will match all virtual hosts matching that pattern
    and obtain one certificate for it (assuming you have 'dns-01' challenge
    support configured). Addresses #239.
  - Removed support for ACMEv1 servers. The only known installation used to
    be Let's Encrypt which has disabled that version more than a year ago for
    new accounts.
  - Andreas Ulm (<https://github.com/root360-AndreasUlm>) implemented the
    ```renewing``` call to ```MDMessageCmd``` that can deny a certificate
    renewal attempt. This is useful in clustered installations, as
    discussed in #233).
  - New event ```challenge-setup:<type>:<domain>```, triggered when the
    challenge data for a domain has been created. This is invoked before the
    ACME server is told to check for it. The type is one of the ACME challenge
    types. This is invoked for every DNS name in a MDomain.
  - The max delay for retries has been raised to daily (this is like all
    retries jittered somewhat to avoid repeats at fixed time of day).
  - Certain error codes reported by the ACME server that indicate a problem
    with the configured data now immediately switch to daily retries. For
    example: if the ACME server rejects a contact email or a domain name,
    frequent retries will most likely not solve the problem. But daily retries
    still make sense as there might be an error at the server and un-supervised
    certificate renewal is the goal. Refs #222.
  - Test case and work around for domain names > 64 octets. Fixes #227.
    When the first DNS name of an MD is longer than 63 octets, the certificate
    request will not contain a CN field, but leave it up to the CA to choose one.
    Currently, Lets Encrypt looks for a shorter name in the SAN list given and
    fails the request if none is found. But it is really up to the CA (and what
    browsers/libs accept here) and may change over the years. That is why
    the decision is best made at the CA.
  - Retry delays now have a random +/-[0-50]%% modification applied to let
    retries from several servers spread out more, should they have been
    restarted at the same time of day.
  - Fixed several places where the 'badNonce' return code from an ACME server
    was not handled correctly. The test server 'pebble' simulates this behaviour
    by default and helps nicely in verifying this behaviour. Thanks, pebble!
  - Set the default `MDActivationDelay` to 0. This was confusing to users that
    new certificates were deemed not usably before a day of delay. When clocks are
    correct, using a new certificate right away should not pose a problem.
  - When handling ACME authorization resources, the module no longer requires
    the server to return a "Location" header, as was necessary in ACMEv1.
    Fixes #216.
  - Fixed a theoretical uninitialized read when testing for JSON error responses
    from the ACME CA. Reported at <https://bz.apache.org/bugzilla/show_bug.cgi?id=64297>.
  - ACME problem reports from CAs that include parameters in the Content-Type
    header are handled correctly. (Previously, the problem text would not be
    reported and retries could exceed CA limits.)
  - Account Update transactions to V2 CAs now use the correct POST-AS-GET method.
    Previously, an empty JSON object was sent - which apparently LE accepted,
    but others reject.
    [Stefan Eissing, @tlhackque, Andreas Ulm]
  Changes with Apache 2.4.47
  * ) mod_dav_fs: Improve logging output when failing to open files for
    writing.  PR 64413.  [Bingyu Shen <ahshenbingyu gmail.com>]
  * ) mod_http2: Fixed a race condition that could lead to streams being
    aborted (RST to the client), although a response had been produced.
    [Stefan Eissing]
  * ) mod_lua: Add support to Lua 5.4  [Joe Orton, Giovanni Bechis, Ruediger Pluem]
  * ) MPM event/worker: Fix possible crash in child process on early signal
    delivery.  PR 64533.  [Ruediger Pluem]
  * ) mod_http2: sync with github standalone version 1.15.17
  - Log requests and sent the configured error response in case of early detected
    errors like too many or too long headers. [Ruediger Pluem]
  - new option 'H2OutputBuffering on/off' which controls the buffering of stream output.
    The default is on, which is the behaviour of older mod-h2 versions. When off, all
    bytes are made available immediately to the main connection for sending them
    out to the client. This fixes interop issues with certain flavours of gRPC, see
    also <https://github.com/icing/mod_h2/issues/207>.
    [Stefan Eissing]
  * ) mod_unique_id: Fix potential duplicated ID generation under heavy load.
    PR 65159
    [Jonas Müntener <jonas.muentener ergon.ch>, Christophe Jaillet]
  * ) "[mod_dav_fs etag handling] should really honor the FileETag setting".
  - It now does.
  - Add "Digest" to FileETag directive, allowing a strong ETag to be
    generated using a file digest.
  - Add ap_make_etag_ex() and ap_set_etag_fd() to allow full control over
    ETag generation.
  - Add concept of "binary notes" to request_rec, allowing packed bit flags
    to be added to a request.
  - First binary note - AP_REQUEST_STRONG_ETAG - allows modules to force
    the ETag to a strong ETag to comply with RFC requirements, such as those
    mandated by various WebDAV extensions.
    [Graham Leggett]
  * ) mod_proxy_http: Fix a possibly crash when the origin connection gets
    interrupted before completion.  PR 64234.
    [Barnim Dzwillo <dzwillo strato.de>, Ruediger Pluem]
  * ) mod_ssl: Do not keep connections to OCSP responders alive when doing
    OCSP requests.  PR 64135.  [Ruediger Pluem]
  * ) mod_ssl: Improve the coalescing filter to buffer into larger TLS
    records, and avoid revealing the HTTP header size via TLS record
    boundaries (for common response generators).
    [Joe Orton, Ruediger Pluem]
  * ) mod_proxy_hcheck: Don't pile up health checks if the previous one did
    not finish before hcinterval.  PR 63010.  [Yann Ylavic]
  * ) mod_session: Improve session parsing.  [Yann Yalvic]
  * ) mod_authnz_ldap: Prevent authentications with empty passwords for the
    initial bind to fail with status 500. [Ruediger Pluem]
  * ) mod_auth_digest: Fast validation of the nonce's base64 to fail early if
    the format can't match anyway.  [Yann Ylavic]
  * ) mod_proxy_fcgi: Honor "SetEnv proxy-sendcl" to forward a chunked
    Transfer-Encoding from the client, spooling the request body when needed
    to provide a Content-Length to the backend.  PR 57087.  [Yann Ylavic]
  * ) mod_proxy: Put mod_proxy_{connect,wstunnel} tunneling code in common in
    proxy_util.  [Yann Ylavic]
  * ) mod_proxy: Improve tunneling loop to support half closed connections and
    pending data draining (for protocols like rsync). PR 61616. [Yann Ylavic]
  * ) mod_proxy_http: handle Upgrade request, 101 (Switching Protocol) response
    and switched protocol forwarding.  [Yann Ylavic]
  * ) mod_proxy_wstunnel: Leave Upgrade requests handling to mod_proxy_http,
    allowing for (non-)Upgrade negotiation with the origin server.
    [Yann Ylavic]
  * ) mod_proxy: Allow ProxyErrorOverride to be restricted to specific status
    codes.  PR63628. [Martin Drößler <mail martindroessler.de>]
  * ) core: Add ReadBufferSize, FlushMaxThreshold and FlushMaxPipelined
    directives.  [Yann Ylavic]
  * ) core: Ensure that aborted connections are logged as such. PR 62823
    [Arnaud Grandville <contact@grandville.net>]
  * ) http: Allow unknown response status' lines returned in the form of
    "HTTP/x.x xxx Status xxx".  [Yann Ylavic]
  * ) mod_proxy_http: Fix 100-continue deadlock for spooled request bodies,
    leading to Request Timeout (408).  PR 63855.  [Yann Ylavic]
  * ) core: Remove headers on 304 Not Modified as specified by RFC7234, as
    opposed to passing an explicit subset of headers. PR 61820.
    [Giovanni Bechis]
  * ) mpm_event: Don't reset connections after lingering close, restoring prior
    to 2.4.28 behaviour.  [Yann Ylavic]
  * ) mpm_event: Kill connections in keepalive state only when there is no more
    workers available, not when the maximum number of connections is reached,
    restoring prior to 2.4.30 behaviour.  [Yann Ylavic]
  * ) mod_unique_id: Use base64url encoding for UNIQUE_ID variable,
    avoiding the use of '@'.  PR 57044.
    [Michael Kaufmann <apache-bugzilla michael-kaufmann.ch>]
  * ) mod_rewrite: Extend the [CO] (cookie) flag of RewriteRule to accept a
    SameSite attribute. [Eric Covener]
  * ) mod_proxy: Add proxy check_trans hook.  This allows proxy
    modules to decline request handling at early stage.
  * ) mod_proxy_wstunnel: Decline requests without an Upgrade
    header so ws/wss can be enabled overlapping with later
    http/https.
  * ) mod_http2: Log requests and sent the configured error response in case of
    early detected errors like too many or too long headers.
    [Ruediger Pluem, Stefan Eissing]
  * ) mod_md: Lowered the required minimal libcurl version from 7.50 to 7.29
    as proposed by <alexander.gerasimov codeit.pro>. [Stefan Eissing]
  * ) mod_ssl: Fix request body buffering with PHA in TLSv1.3.  [Joe Orton]
  * ) mod_proxy_uwsgi: Fix a crash when sending environment variables with no
    value. PR 64598 [Ruediger Pluem]
  * ) mod_proxy: Recognize parameters from ProxyPassMatch workers with dollar
    substitution, such that they apply to the backend connection.  Note that
    connection reuse is disabled by default to avoid compatibility issues.
    [Takashi Sato, Jan Kaluza, Eric Covener, Yann Ylavic, Jean-Frederic Clere]
- modified sources
  %% apache2.keyring
- deleted patches
  - apache2-mod_proxy_uwsgi-fix-crash.patch (upstreamed)
  - lua54.patch (upstreamed)
* Wed Jan 27 2021 gmbr3@opensuse.org
- Add lua54.patch to fix building with lua54
* Fri Jan 22 2021 michael@stroeder.com
- added Provides: http_daemon to fix bsc#1180132
* Fri Jan 22 2021 pgajdos@suse.com
- remove suexec2 from premissions as well
- modified sources
  %% permissions.apache2
* Wed Jan 13 2021 suse+build@de-korte.org
- suexec2 was renamed to suexec in 2015 and the symlink to suexec
  no longer exists
* Tue Jan 12 2021 suse-beta@cboltz.de
- re-add rcapache2 symlink (was lost about 6 weeks ago)
* Tue Jan 12 2021 pgajdos@suse.com
- gensslcert sets CA:TRUE in basic constrains of CA cert [bsc#1180530]
- modified sources
  %% apache2-gensslcert
* Fri Jan  8 2021 pgajdos@suse.com
- mod_php8 provides php_module
- modified sources
  %% apache2-script-helpers
* Thu Dec 17 2020 pgajdos@suse.com
- obsolete apache2-example-page and obsolete/provide apache2-doc
  [bsc#180075]
* Mon Dec 14 2020 suse+build@de-korte.org
- only include BuildRequires: apache2-{prefork,worker,event} used in
  test target
* Fri Dec 11 2020 pgajdos@suse.com
- added https://httpd.apache.org/server-status like server status
  (configured by APACHE_EXTENDED_STATUS="lua")
- modified sources
  %% apache2-mod_status.conf
  %% apache2-start_apache2
  %% sysconfig.apache2
* Thu Dec 10 2020 pgajdos@suse.com
- actually install suse provided READMEs
- install upstream configuration example
- added sources
  + apache2-README-configuration.txt
* Wed Dec  9 2020 pgajdos@suse.com
- apxs -q PREFIX now returns /srv/www again
* Wed Dec  9 2020 dleuenberger@suse.com
- Require /usr/bin/which instead of which: allow usage of busybox
  variant for containers.
* Mon Dec  7 2020 pgajdos@suse.com
- apache2 main package recommends apache2-utils
* Thu Dec  3 2020 pgajdos@suse.com
- break some long lines in the spec as requested by review team
* Tue Dec  1 2020 pgajdos@suse.com
- maybe make spec acceptable for factory bot
- modified sources
  %% _multibuild
* Tue Dec  1 2020 pgajdos@suse.com
- buildrequire netcfg for tests
* Sun Nov 29 2020 suse+build@de-korte.org
- remove Requires(post): firewalld hard dependency (this is already
  handled by the %%firewalld_reload macro)
* Thu Nov 26 2020 pgajdos@suse.com
- package reworked trough [bsc#1178478]
- modified patches
  %% apache2-mod_proxy_uwsgi-fix-crash.patch (refreshed)
- modified sources
  %% apache2-loadmodule.conf
  %% apache2-manual.conf
  %% apache2-script-helpers
  %% apache2@.service
  %% sysconfig.apache2
- deleted patches
  - deprecated-scripts-arch.patch (not needed)
  - httpd-2.0.54-envvars.dif (not needed)
  - httpd-2.1.3alpha-layout.dif
    (renamed to apache2-system-dirs-layout.patch)
  - httpd-2.2.0-apxs-a2enmod.dif (not needed)
  - httpd-2.4.9-bnc690734.patch
    (renamed to apache2-LimitRequestFieldSize-limits-headers.patch)
  - httpd-2.4.x-fate317766-config-control-two-protocol-options.diff
    (renamed to apache2-HttpContentLengthHeadZero-HttpExpectStrict.patch)
  - httpd-2.x.x-logresolve.patch
    (renamed to apache2-logresolve-tmp-security.patch)
  - httpd-apachectl.patch
    (renamed to apache2-apachectl.patch)
  - httpd-implicit-pointer-decl.patch (not needed)
  - httpd-visibility.patch (not needed)
- deleted sources
  - SUSE-NOTICE (outdated)
  - a2enflag (renamed to apache2-a2enflag)
  - a2enmod (renamed to apache2-a2enmod)
  - apache-22-24-upgrade (outdated)
  - apache2-README (outdated)
  - apache2-README.QUICKSTART (outedated)
  - apache2-find-directives (renamed to apache2-find_directives)
  - apache2-init.logrotate (not needed)
  - apache2.firewalld (renamed to firewalld.apache2)
  - apache2.logrotate (renamed to logrotate.apache2)
  - apache2.ssl.firewalld (renamed to firewalld-ssl.apache2)
  - apache2.ssl.susefirewall (renamed to susefirewall.apache2)
  - apache2.susefirewall (renamed to susefirewall-ssl.apache2)
  - favicon.ico (not needed)
  - rc.apache2 (not needed)
  - robots.txt (not needed)
  - sysconf_addword (not needed, in aaa_base)
- added patches
  fix PATCH:  https://marc.info/?l=apache-httpd-users&m=147448312531134&w=2
  + apache-test-application-xml-type.patch
  fix these variables from the test
  + apache-test-turn-off-variables-in-ssl-var-lookup.patch
  fix [fate317766] backport of an upstream commit
  + apache2-HttpContentLengthHeadZero-HttpExpectStrict.patch
  fix [bnc#690734] TODO, to be upstreamed
  + apache2-LimitRequestFieldSize-limits-headers.patch
  fix apachectl is frontend for start_apache2, suse specific
  + apache2-apachectl.patch
  fix [bnc#210904] perhaps to be upstreamed
  + apache2-logresolve-tmp-security.patch
  fix layout of system dirs configuration, may be upstreamed
  + apache2-system-dirs-layout.patch
- added sources
  + _multibuild
  + apache2-a2enflag
  + apache2-a2enmod
  + apache2-find_directives
  + apache2-gensslcert
  + apache2-mod_example.c
  + apache2-start_apache2
  + firewalld-ssl.apache2
  + firewalld.apache2
  + logrotate.apache2
  + susefirewall-ssl.apache2
  + susefirewall.apache2
* Wed Oct 14 2020 fbui@suse.com
- /var/run is depracted, /run should be used instead
* Wed Oct 14 2020 fbui@suse.com
- systemd-ask-password is located in /usr/bin
* Tue Sep 22 2020 jtomasiak@suse.com
- gensslcert: fix SAN in final cert generation
* Fri Sep 11 2020 fbui@suse.com
- Make use of %%service_del_postun_without_restart
  And stop using DISABLE_RESTART_ON_UPDATE as this interface is
  obsolete.
* Mon Aug 31 2020 jtomasiak@suse.com
- gensslcert: add -a argument to override default SAN value
* Mon Aug 31 2020 alarrosa@suse.com
- Provide mod_ssl to keep compatibility with other distributions.
  Now obsoletes mod_ssl < %%{version} instead of mod_ssl < 2.8.16.
* Thu Aug 27 2020 dimstar@opensuse.org
- Install firewalld service files to %%{_prefix}/lib/firewalld, NOT
  %%{_libexecdir}/firewalld (boo#1174075).
* Fri Aug  7 2020 pgajdos@suse.com
- version update to 2.4.46
  Changes with Apache 2.4.46
  * ) mod_proxy_fcgi: Fix build warnings for Windows platform
    [Eric Covener, Christophe Jaillet]
  Changes with Apache 2.4.45
  * ) mod_http2: remove support for abandoned http-wg draft
    <https://datatracker.ietf.org/doc/draft-kazuho-h2-cache-digest/>.
    [Stefan Eissing]
  Changes with Apache 2.4.44
  * ) mod_proxy_uwsgi: Error out on HTTP header larger than 16K (hard
    protocol limit).  [Yann Ylavic]
  * ) mod_http2:
    Fixes <https://github.com/icing/mod_h2/issues/200>:
    "LimitRequestFields 0" now disables the limit, as documented.
    Fixes <https://github.com/icing/mod_h2/issues/201>:
    Do not count repeated headers with same name against the field
    count limit. The are merged internally, as if sent in a single HTTP/1 line.
    [Stefan Eissing]
  * ) mod_http2: Avoid segfaults in case of handling certain responses for
    already aborted connections.  [Stefan Eissing, Ruediger Pluem]
  * ) mod_http2: The module now handles master/secondary connections and has marked
    methods according to use. [Stefan Eissing]
  * ) core: Drop an invalid Last-Modified header value coming
    from a FCGI/CGI script instead of replacing it with Unix epoch.
    [Yann Ylavic, Luca Toscano]
  * ) Add support for strict content-length parsing through addition of
    ap_parse_strict_length() [Yann Ylavic]
  * ) mod_proxy_fcgi: ProxyFCGISetEnvIf unsets variables when expression
    evaluates to false.  PR64365. [Michael König <mail ikoenig.net>]
  * ) mod_proxy_http: flush spooled request body in one go to avoid
    leaking (or long lived) temporary file. PR 64452. [Yann Ylavic]
  * ) mod_ssl: Fix a race condition and possible crash when using a proxy client
    certificate (SSLProxyMachineCertificateFile).
    [Armin Abfalterer <a.abfalterer gmail.com>]
  * ) mod_ssl: Fix memory leak in stapling code. PR63687. [Stefan Eissing]
  * ) mod_http2: Fixed regression that no longer set H2_STREAM_ID and H2_STREAM_TAG.
    PR64330 [Stefan Eissing]
  * ) mod_http2: Fixed regression that caused connections to close when mod_reqtimeout
    was configured with a handshake timeout. Fixes gitub issue #196.
    [Stefan Eissing]
  * ) mod_proxy_http2: the "ping" proxy parameter
    (see <https://httpd.apache.org/docs/2.4/mod/mod_proxy.html>) is now used
    when checking the liveliness of a new or reused h2 connection to the backend.
    With short durations, this makes load-balancing more responsive. The module
    will hold back requests until ping conditions are met, using features of the
    HTTP/2 protocol alone. [Ruediger Pluem, Stefan Eissing]
  * ) core: httpd is no longer linked against -lsystemd if mod_systemd
    is enabled (and built as a DSO).  [Rainer Jung]
  * ) mod_proxy_http2: respect ProxyTimeout settings on backend connections
    while waiting on incoming data. [Ruediger Pluem, Stefan Eissing]
- modified patches
  %% apache2-mod_proxy_uwsgi-fix-crash.patch (refreshed)
- modified sources
  %% apache2.keyring
* Wed Jul 15 2020 pgajdos@suse.com
- fix crash in mod_proxy_uwsgi for empty values of environment
  variables [bsc#1174052]
- added patches
  fix https://svn.apache.org/viewvc?view=revision
  + apache2-mod_proxy_uwsgi-fix-crash.patch
* Fri Apr  3 2020 pgajdos@suse.com
- declare ap_sock_disable_nagle to fix loading mod_proxy_http2
  (thanks to mliska@suse.com)
- modified patches
  %% httpd-visibility.patch (refreshed)
* Thu Apr  2 2020 pgajdos@suse.com
- version update to 2.4.43
  * ) mod_ssl: Fix memory leak of OCSP stapling response. [Yann Ylavic]
  * ) mod_proxy_http: Fix the forwarding of requests with content body when a
    balancer member is unavailable; the retry on the next member was issued
    with an empty body (regression introduced in 2.4.41). PR63891.
    [Yann Ylavic]
  * ) mod_http2: Fixes issue where mod_unique_id would generate non-unique request
    identifier under load, see <https://github.com/icing/mod_h2/issues/195>.
    [Michael Kaufmann, Stefan Eissing]
  * ) mod_proxy_hcheck: Allow healthcheck expressions to use %%{Content-Type}.
    PR64140. [Renier Velazco <renier.velazco upr.edu>]
  * ) mod_authz_groupfile: Drop AH01666 from loglevel "error" to "info".
    PR64172.
  * ) mod_usertrack: Add CookieSameSite, CookieHTTPOnly, and CookieSecure
    to allow customization of the usertrack cookie. PR64077.
    [Prashant Keshvani <prashant2400 gmail.com>, Eric Covener]
  * ) mod_proxy_ajp: Add "secret" parameter to proxy workers to implement legacy
    AJP13 authentication.  PR 53098. [Dmitry A. Bakshaev <dab1818 gmail com>]
  * ) mpm_event: avoid possible KeepAliveTimeout off by -100 ms.
    [Eric Covener, Yann Ylavic]
  * ) Add a config layout for OpenWRT. [Graham Leggett]
  * ) Add support for cross compiling to apxs. If apxs is being executed from
    somewhere other than its target location, add that prefix to includes and
    library directories. Without this, apxs would fail to find config_vars.mk
    and exit. [Graham Leggett]
  * ) mod_ssl: Disable client verification on ACME ALPN challenges. Fixes github
    issue mod_md#172 (https://github.com/icing/mod_md/issues/172).
    [Michael Kaufmann <mail michael-kaufmann.ch>, Stefan Eissing]
  * ) mod_ssl: use OPENSSL_init_ssl() to initialise OpenSSL on versions 1.1+.
    [Graham Leggett]
  * ) mod_ssl: Support use of private keys and certificates from an
    OpenSSL ENGINE via PKCS#11 URIs in SSLCertificateFile/KeyFile.
    [Anderson Sasaki <ansasaki redhat.com>, Joe Orton]
  * ) mod_md:
  - Prefer MDContactEmail directive to ServerAdmin for registration. New directive
    thanks to Timothe Litt (@tlhackque).
  - protocol check for pre-configured "tls-alpn-01" challenge has been improved. It will now
    check all matching virtual hosts for protocol support. Thanks to @mkauf.
  - Corrected a check when OCSP stapling was configured for hosts
    where the responsible MDomain is not clear, by Michal Karm Babacek (@Karm).
  - Softening the restrictions where mod_md configuration directives may appear. This should
    allow for use in <If> and <Macro> sections. If all possible variations lead to the configuration
    you wanted in the first place, is another matter.
    [Michael Kaufmann <mail michael-kaufmann.ch>, Timothe Litt (@tlhackque),
    Michal Karm Babacek (@Karm), Stefan Eissing (@icing)]
  * ) test: Added continuous testing with Travis CI.
    This tests various scenarios on Ubuntu with the full test suite.
    Architectures tested: amd64, s390x, ppc64le, arm64
    The tests pass successfully.
    [Luca Toscano, Joe Orton, Mike Rumph, and others]
  * ) core: Be stricter in parsing of Transfer-Encoding headers.
    [ZeddYu <zeddyu.lu gmail.com>, Eric Covener]
  * ) mod_ssl: negotiate the TLS protocol version per name based vhost
    configuration, when linked with OpenSSL-1.1.1 or later. The base vhost's
    SSLProtocol (from the first vhost declared on the IP:port) is now only
    relevant if no SSLProtocol is declared for the vhost or globally,
    otherwise the vhost or global value apply.  [Yann Ylavic]
  * ) mod_cgi, mod_cgid: Fix a memory leak in some error cases with large script
    output.  PR 64096.  [Joe Orton]
  * ) config: Speed up graceful restarts by using pre-hashed command table. PR 64066.
    [Giovanni Bechis <giovanni paclan.it>, Jim Jagielski]
  * ) mod_systemd: New module providing integration with systemd.  [Jan Kaluza]
  * ) mod_lua: Add r:headers_in_table, r:headers_out_table, r:err_headers_out_table,
    r:notes_table, r:subprocess_env_table as read-only native table alternatives
    that can be iterated over. [Eric Covener]
  * ) mod_http2: Fixed rare cases where a h2 worker could deadlock the main connection.
    [Yann Ylavic, Stefan Eissing]
  * ) mod_lua: Accept nil assignments to the exposed tables (r.subprocess_env,
    r.headers_out, etc) to remove the key from the table. PR63971.
    [Eric Covener]
  * ) mod_http2: Fixed interaction with mod_reqtimeout. A loaded mod_http2 was disabling the
    ssl handshake timeouts. Also, fixed a mistake of the last version that made `H2Direct`
    always `on`, regardless of configuration. Found and reported by
    <Armin.Abfalterer@united-security-providers.ch> and
    <Marcial.Rion@united-security-providers.ch>. [Stefan Eissing]
  * ) mod_http2: Multiple field length violations in the same request no longer cause
    several log entries to be written. [@mkauf]
  * ) mod_ssl: OCSP does not apply to proxy mode.  PR 63679.
    [Lubos Uhliarik <luhliari redhat.com>, Yann Ylavic]
  * ) mod_proxy_html, mod_xml2enc: Fix build issues with macOS due to r1864469
    [Jim Jagielski]
  * ) mod_authn_socache: Increase the maximum length of strings that can be cached by
    the module from 100 to 256.  PR 62149 [<thorsten.meinl knime.com>]
  * ) mod_proxy: Fix crash by resolving pool concurrency problems. PR 63503
    [Ruediger Pluem, Eric Covener]
  * ) core: On Windows, fix a start-up crash if <IfFile ...> is used with a path that is not
    valid (For example, testing for a file on a flash drive that is not mounted)
    [Christophe Jaillet]
  * ) mod_deflate, mod_brotli: honor "Accept-Encoding: foo;q=0" as per RFC 7231; which
    means 'foo' is "not acceptable".  PR 58158 [Chistophe Jaillet]
  * ) mod_md v2.2.3:
  - Configuring MDCAChallenges replaces any previous existing challenge configuration. It
    had been additive before which was not the intended behaviour. [@mkauf]
  - Fixing order of ACME challenges used when nothing else configured. Code now behaves as
    documented for `MDCAChallenges`. Fixes #156. Thanks again to @mkauf for finding this.
  - Fixing a potential, low memory null pointer dereference [thanks to @uhliarik].
  - Fixing an incompatibility with a change in libcurl v7.66.0 that added unwanted
    "transfer-encoding" to POST requests. This failed in directy communication with
    Let's Encrypt boulder server. Thanks to @mkauf for finding and fixing. [Stefan Eissing]
  * ) mod_md: Adding the several new features.
    The module offers an implementation of OCSP Stapling that can replace fully or
    for a limited set of domains the existing one from mod_ssl. OCSP handling
    is part of mod_md's monitoring and message notifications. If can be used
    for sites that do not have ACME certificates.
    The url for a CTLog Monitor can be configured. It is used in the server-status
    to link to the external status page of a certicate.
    The MDMessageCmd is called with argument "installed" when a new certificate
    has been activated on server restart/reload. This allows for processing of
    the new certificate, for example to applications that require it in different
    locations or formats.
    [Stefan Eissing]
  * ) mod_proxy_balancer: Fix case-sensitive referer check related to CSRF/XSS
    protection. PR 63688. [Armin Abfalterer <a.abfalterer gmail.com>]
- deleted patches
  - apache2-load-private-keys-from-pkcs11.patch (upstreamed)
  - httpd-2.4.3-mod_systemd.patch (upstreamed)
* Wed Feb 26 2020 pgajdos@suse.com
- use r1874196 [SLE-7653]
- modified patches
  %% apache2-load-private-keys-from-pkcs11.patch (upstream 2.4.x port)
- deleted patches
  - apache2-load-certificates-from-pkcs11.patch (merged to above)
* Tue Feb 18 2020 pgajdos@suse.com
- require just libbrotli-devel
* Thu Feb 13 2020 pgajdos@suse.com
- build mod_proxy_http2 extension
* Wed Feb 12 2020 pgajdos@suse.com
- fix build for older distributions
* Fri Jan 31 2020 crrodriguez@opensuse.org
- define DEFAULT_LISTENBACKLOG=APR_INT32_MAX. We want apache
  to honour net.core.somaxconn sysctl as the mandatory limit.
  the old value of 511 was never used as until v5.4-rc6 it was
  clamped to 128, in current kernels the default limit is 4096.
  Cannot use the apr_socket_listen(.., -1) idiom because the function
  expects a positive integer argument.
* Mon Jan 20 2020 pgajdos@suse.com
- apache2-devel now provides httpd-devel [bsc#1160100]
* Wed Dec 18 2019 pgajdos@suse.com
- add openssl call to DEFAULT_SUSE comment [bsc#1159480]
- modified sources
  %% apache2-ssl-global.conf
* Fri Nov  8 2019 pgajdos@suse.com
- use %%license [bsc#1156171]
* Tue Oct 22 2019 pgajdos@suse.com
- load private keys and certificates from pkcs11 token [SLE-7653]
- added patches
  load certificates from openssl engine
  + apache2-load-certificates-from-pkcs11.patch
  load private keys from openssl engine
  + apache2-load-private-keys-from-pkcs11.patch
* Sat Oct  5 2019 i@guoyunhe.me
- Add custom log files to logrotate according to apache2-vhost.template
* Fri Aug 23 2019 jengelh@inai.de
- Remove redundant metadata from summary.
* Thu Aug 15 2019 michael@stroeder.com
- version update to 2.4.41 with security fixes:
  * low: Limited cross-site scripting in mod_proxy
    error page (CVE-2019-10092)
  * low: mod_rewrite potential open redirect (CVE-2019-10098)
  * moderate: mod_http2, read-after-free in h2
    connection shutdown (CVE-2019-10082)
  * moderate: mod_http2, memory corruption on
    early pushes (CVE-2019-10081)
  * moderate: mod_http2, DoS attack by exhausting
    h2 workers. (CVE-2019-9517)
  * moderate: mod_remoteip: Stack buffer overflow and
    NULL pointer dereference (CVE-2019-10097)
* Wed Jun 26 2019 pgajdos@suse.com
- fix typo
- modified sources
  %% apache2-README-instances.txt
* Mon May 13 2019 pgajdos@suse.com
- revive apache-22-24-upgrade [bsc#1134294] (internal)
- added sources
  + apache-22-24-upgrade
* Tue Apr  2 2019 pgajdos@suse.com
- version update to 2.4.39
  * mod_proxy/ssl: Cleanup per-request SSL configuration anytime a
    backend connection is recycled/reused to avoid a possible crash
    with some SSLProxy configurations in <Location> or <Proxy>
    context. PR 63256. [Yann Ylavic]
  * mod_ssl: Correctly restore SSL verify state after TLSv1.3 PHA
    failure. [Michael Kaufmann <mail michael-kaufmann.ch>]
  * mod_log_config: Support %%{c}h for conn-hostname, %%h for
    useragent_host PR 55348
  * mod_socache_redis: Support for Redis as socache storage
    provider.
  * core: new configuration option 'MergeSlashes on|off' that
    controls handling of multiple, consecutive slash ('/')
    characters in the path component of the request URL. [Eric
    Covener]
  * mod_http2: when SSL renegotiation is inhibited and a 403
    ErrorDocument is in play, the proper HTTP/2 stream reset did
    not trigger with H2_ERR_HTTP_1_1_REQUIRED. Fixed. [Michael
    Kaufmann]
  * mod_http2: new configuration directive: `H2Padding numbits` to
    control padding of HTTP/2 payload frames. 'numbits' is a number
    from 0-8, controlling the range of padding bytes added to a
    frame. The actual number added is chosen randomly per frame.
    This applies to HEADERS, DATA and PUSH_PROMISE frames equally.
    The default continues to be 0, e.g. no padding. [Stefan
    Eissing]
  * mod_http2: ripping out all the h2_req_engine internal features
    now that mod_proxy_http2 has no more need for it. Optional
    functions are still declared but no longer implemented. While
    previous mod_proxy_http2 will work with this, it is
    recommeneded to run the matching versions of both modules.
    [Stefan Eissing]
  * mod_proxy_http2: changed mod_proxy_http2 implementation and
    fixed several bugs which resolve PR63170. The proxy module does
    now a single h2 request on the (reused) connection and returns.
    [Stefan Eissing]
  * mod_http2/mod_proxy_http2: proxy_http2 checks correct master
    connection aborted status to trigger immediate shutdown of
    backend connections. This is now always signalled by mod_http2
    when the the session is being released. proxy_http2 now only
    sends a PING frame to the backend when there is not already one
    in flight. [Stefan Eissing]
  * mod_proxy_http2: fixed an issue where a proxy_http2 handler
    entered an infinite loop when encountering certain errors on
    the backend connection. See
    <https://bz.apache.org/bugzilla/show_bug.cgi?id=63170>. [Stefan
    Eissing]
  * mod_http2: Configuration directives H2Push and H2Upgrade can
    now be specified per Location/Directory, e.g. disabling PUSH
    for a specific set of resources. [Stefan Eissing]
  * mod_http2: HEAD requests to some module such as mod_cgid caused
    the stream to terminate improperly and cause a HTTP/2
    PROTOCOL_ERROR. Fixes
    <https://github.com/icing/mod_h2/issues/167>. [Michael
    Kaufmann]
  * http: Fix possible empty response with mod_ratelimit for HEAD
    requests. PR 63192. [Yann Ylavic]
  * mod_cache_socache: Avoid reallocations and be safe with
    outgoing data lifetime. [Yann Ylavic]
  * MPMs unix: bind the bucket number of each child to its slot
    number, for a more efficient per bucket maintenance. [Yann
    Ylavic]
  * mod_auth_digest: Fix a race condition. Authentication with
    valid credentials could be refused in case of concurrent
    accesses from different users. PR 63124. [Simon Kappel
    <simon.kappel axis.com>]
  * mod_http2: enable re-use of slave connections again. Fixed
    slave connection keepalives counter. [Stefan Eissing]
  * mod_reqtimeout: Allow to configure (TLS-)handshake timeouts. PR
    61310. [Yann Ylavic]
  * mod_proxy_wstunnel: Fix websocket proxy over UDS. PR 62932
    <pavel dcmsys.com>
  * mod_ssl: Don't unset FIPS mode on restart unless it's forced by
    configuration (SSLFIPS on) and not active by default in
    OpenSSL. PR 63136. [Yann Ylavic]
- deleted patches
  - apache2-mod_http2-issue-167.patch (upstreamed)
* Mon Mar 18 2019 jengelh@inai.de
- Reduce scriptlets' hard dependency on systemd.
* Wed Feb 27 2019 pgajdos@suse.com
- added patches
  fix https://github.com/icing/mod_h2/issues/167 [bsc#1125965]
  + apache2-mod_http2-issue-167.patch
* Fri Feb  8 2019 jengelh@inai.de
- Replace old $RPM_* shell vars. Avoid old tar syntax.
- Tag scriptlets as explicitly requiring bash.
* Fri Jan 18 2019 mmanu84@outlook.de
- updated to 2.4.38
  * mod_ssl: Clear retry flag before aborting client-initiated renegotiation.
    PR 63052 [Joe Orton]
  * mod_negotiation: Treat LanguagePriority as case-insensitive to match
    AddLanguage behavior and HTTP specification. PR 39730 [Christophe Jaillet]
  * mod_md: incorrect behaviour when synchronizing ongoing ACME challenges
    have been fixed. [Michael Kaufmann, Stefan Eissing]
  * mod_setenvif: We can have expressions that become true if a regex pattern
    in the expression does NOT match. In this case val is NULL
    and we should just set the value for the environment variable
    like in the pattern case. [Ruediger Pluem]
  * mod_session: Always decode session attributes early. [Hank Ibell]
  * core: Incorrect values for environment variables are substituted when
    multiple environment variables are specified in a directive. [Hank Ibell]
  * mod_rewrite: Only create the global mutex used by "RewriteMap prg:" when
    this type of map is present in the configuration.  PR62311.
    [Hank Ibell <hwibell gmail.com>]
  * mod_dav: Fix invalid Location header when a resource is created by
    passing an absolute URI on the request line [Jim Jagielski]
  * mod_session_cookie: avoid duplicate Set-Cookie header in the response.
    [Emmanuel Dreyfus <manu@netbsd.org>, Luca Toscano]
  * mod_ssl: clear *SSL errors before loading certificates and checking
    afterwards. Otherwise errors are reported when other SSL using modules
    are in play. Fixes PR 62880. [Michael Kaufmann]
  * mod_ssl: Fix the error code returned in an error path of
    'ssl_io_filter_handshake()'. This messes-up error handling performed
    in 'ssl_io_filter_error()' [Yann Ylavic]
  * mod_ssl: Fix $HTTPS definition for "SSLEngine optional" case, and fix
    authz provider so "Require ssl" works correctly in HTTP/2.
    PR 61519, 62654.  [Joe Orton, Stefan Eissing]
  * mod_proxy: If ProxyPassReverse is used for reverse mapping of relative
    redirects, subsequent ProxyPassReverse statements, whether they are
    relative or absolute, may fail.  PR 60408.  [Peter Haworth <pmh1wheel gmail.com>]
  * mod_lua: Now marked as a stable module [https://s.apache.org/Xnh1]
* Wed Jan 16 2019 suse+build@de-korte.org
- SSLProtocol use TLSv1.2 or higher
* Wed Jan  9 2019 pgajdos@suse.com
- do not create sysconfig.d when already exists [bsc#1121086]
* Sun Jan  6 2019 dmueller@suse.com
- use secure http sites by default in configs
- Switch to DEFAULT_SUSE Cipher suite
* Thu Oct 25 2018 suse+build@de-korte.org
- the "event" MPM is fully supported since 2.4
- configure an OCSP stapling cache by default (still requires enabling
  SSLUseStapling in vhost)
* Thu Oct 18 2018 mmanu84@outlook.de
- updated to 2.4.37
  * mod_ssl: Fix HTTP/2 failures when using OpenSSL 1.1.1. [Rainer Jung]
  * mod_ssl: Fix crash during SSL renegotiation with OptRenegotiate set,
    when client certificates are available from the original handshake
    but were originally not verified and should get verified now.
    This is a regression in 2.4.36 (unreleased). [Ruediger Pluem]
  * mod_ssl: Correctly merge configurations that have client certificates set
    by SSLProxyMachineCertificate{File|Path}. [Ruediger Pluem]
- updated to 2.4.36
  * mod_brotli, mod_deflate: Restore the separate handling of 304 Not Modified
    responses. Regression introduced in 2.4.35.
  * mod_proxy_scgi, mod_proxy_uwsgi: improve error handling when sending the
    body of the response. [Jim Jagielski]
  * mod_http2: adding defensive code for stream EOS handling, in case the request handler
    missed to signal it the normal way (eos buckets). Addresses github issues
    https://github.com/icing/mod_h2/issues/164, https://github.com/icing/mod_h2/issues/167
    and https://github.com/icing/mod_h2/issues/170. [Stefan Eissing]
  * ab: Add client certificate support. [Graham Leggett]
  * ab: Disable printing temp key for OpenSSL before
    version 1.0.2. SSL_get_server_tmp_key is not available
    there. [Rainer Jung]
  * mod_ssl: Fix a regression that the configuration settings for verify mode
    and verify depth were taken from the frontend connection in case of
    connections by the proxy to the backend. PR 62769. [Ruediger Pluem]
  * MPMs: Initialize all runtime/asynchronous objects on a dedicated pool and
    before signals handling to avoid lifetime issues on restart or shutdown.
    PR 62658. [Yann Ylavic]
  * mod_ssl: Add support for OpenSSL 1.1.1 and TLSv1.3.  TLSv1.3 has
    behavioural changes compared to v1.2 and earlier; client and
    configuration changes should be expected.  SSLCipherSuite is
    enhanced for TLSv1.3 ciphers, but applies at vhost level only.
    [Stefan Eissing, Yann Ylavic, Ruediger Pluem, Joe Orton]
  * mod_auth_basic: Be less tolerant when parsing the credencial. Only spaces
    should be accepted after the authorization scheme. \t are also tolerated.
    [Christophe Jaillet]
  * mod_proxy_hcheck: Fix issues with interval determination. PR 62318
    [Jim Jagielski]
  * mod_proxy_hcheck: Fix issues with TCP health checks. PR 61499
    [Dominik Stillhard <dominik.stillhard united-security-providers.ch>]
  * mod_proxy_hcheck: take balancer's SSLProxy* directives into account.
    [Jim Jagielski]
  * mod_status, mod_echo: Fix the display of client addresses.
    They were truncated to 31 characters which is not enough for IPv6 addresses.
    This is done by deprecating the use of the 'client' field and using
    the new 'client64' field in worker_score.
    PR 54848 [Bernhard Schmidt <berni birkenwald de>, Jim Jagielski]
* Mon Oct  1 2018 pgajdos@suse.com
- consider also patterns in APACHE_CONF_INCLUDE_DIRS as documentation
  says (patch Juergen Gleiss)
* Thu Sep 27 2018 pgajdos@suse.com
- relink /usr/sbin/httpd after apache2-MPM uninstall [bsc#1107930c#1]
- simplify find_mpm function from script-helpers
- /usr/sbin/httpd is now created depending on preference hardcoded
  in find_mpm (script-helpers), not depending on alphabetical
  order of MPMs
- simplify spec file a bit
* Mon Sep 24 2018 michael@stroeder.com
- updated to 2.4.35:
  * http: Enforce consistently no response body with both 204 and 304
    statuses.
  * mod_status: Cumulate CPU time of exited child processes in the
    "cu" and "cs" values. Add CPU time of the parent process to the
    "c" and "s" values.
  * mod_proxy: Improve the balancer member data shown in mod_status when
    "ProxyStatus" is "On": add "busy" count and show byte counts in
    auto mode always in units of kilobytes.
  * mod_status: Add cumulated response duration time in milliseconds.
  * mod_status: Complete the data shown for async MPMs in "auto" mode.
    Added number of processes, number of stopping processes and number
    of busy and idle workers.
  * mod_ratelimit: Don't interfere with "chunked" encoding, fixing regression
    introduced in 2.4.34.  PR 62568.
  * mod_proxy: Remove load order and link dependency between mod_lbmethod_*
    modules and mod_proxy. PR 62557.
  * Allow the argument to <IfFile>, <IfDefine>, <IfSection>, <IfDirective>,
    and <IfModule> to be quoted.  This is primarily for the benefit of
    <IfFile>.
  * mod_watchdog: Correct some log messages.
  * mod_md: When the last domain name from an MD is moved to another one,
    that now empty MD gets moved to the store archive. PR 62572.
  * mod_ssl: Fix merging of SSLOCSPOverrideResponder.
  * mod_proxy_balancer: Restore compatibility with APR 1.4.
* Mon Jul 16 2018 pgajdos@suse.com
- updated to 2.4.34:
  * ) Introduce zh-cn and zh-tw (simplified and traditional Chinese) error
    document translations. [CodeingBoy, popcorner]
  * ) event: avoid possible race conditions with modules on the child pool.
    [Stefan Fritsch]
  * ) mod_proxy: Fix a corner case where the ProxyPassReverseCookieDomain or
    ProxyPassReverseCookiePath directive could fail to update correctly
    'domain=' or 'path=' in the 'Set-Cookie' header.  PR 61560.
    [Christophe Jaillet]
  * ) mod_ratelimit: fix behavior when proxing content. PR 62362.
    [Luca Toscano, Yann Ylavic]
  * ) core: Re-allow '_' (underscore) in hostnames.
    [Eric Covener]
  * ) mod_authz_core: If several parameters are used in a AuthzProviderAlias
    directive, if these parameters are not enclosed in quotation mark, only
    the first one is handled. The other ones are silently ignored.
    Add a message to warn about such a spurious configuration.
    PR 62469 [Hank Ibell <hwibell gmail.com>, Christophe Jaillet]
  * ) mod_md: improvements and bugfixes
  - MDNotifyCmd now takes additional parameter that are passed on to the called command.
  - ACME challenges have better checks for interference with other modules
  - ACME challenges are only handled for domains managed by the module, allowing
    other ACME clients to operate for other domains in the server.
  - better libressl integration
  * ) mod_proxy_wstunnel: Add default schema ports for 'ws' and 'wss'.
    PR 62480. [Lubos Uhliarik <luhliari redhat.com>}
  * ) logging: Some early logging-related startup messages could be lost
    when using syslog for the global ErrorLog. [Eric Covener]
  * ) mod_cache: Handle case of an invalid Expires header value RFC compliant
    like the case of an Expires time in the past: allow to overwrite the
    non-caching decision using CacheStoreExpired and respect Cache-Control
    "max-age" and "s-maxage".  [Rainer Jung]
  * ) mod_xml2enc: Fix forwarding of error metadata/responses. PR 62180.
    [Micha Lenk <micha lenk.info>, Yann Ylavic]
  * ) mod_proxy_http: Fix response header thrown away after the previous one
    was considered too large and truncated. PR 62196. [Yann Ylavic]
  * ) core: Add and handle AP_GETLINE_NOSPC_EOL flag for ap_getline() family
    of functions to consume the end of line when the buffer is exhausted.
    PR 62198. [Yann Ylavic]
  * ) mod_proxy_http: Add new worker parameter 'responsefieldsize' to
    allow maximum HTTP response header size to be increased past 8192
    bytes.  PR 62199.  [Hank Ibell <hwibell gmail.com>]
  * ) mod_ssl: Extend SSLOCSPEnable with mode 'leaf' that only checks the leaf
    of a certificate chain.  PR62112.
    [Ricardo Martin Camarero <rickyepoderi yahoo.es>]
  * ) http: Fix small memory leak per request when handling persistent
    connections.  [Ruediger Pluem, Joe Orton]
  * ) mod_proxy_html: Fix variable interpolation and memory allocation failure
    in ProxyHTMLURLMap.  [Ewald Dieterich <ewald mailbox.org>]
  * ) mod_remoteip: Fix RemoteIP{Trusted,Internal}ProxyList loading broken by 2.4.30.
    PR 62220.  [Chritophe Jaillet, Yann Ylavic]
  * ) mod_remoteip: When overriding the useragent address from X-Forwarded-For,
    zero out what had been initialized as the connection-level port.  PR59931.
    [Hank Ibell <hwibell gmail.com>]
  * ) core: In ONE_PROCESS/debug mode, cleanup everything when exiting.
    [Yann Ylavic]
  * ) mod_proxy_balancer: Add hot spare member type and corresponding flag (R).
    Hot spare members are used as drop-in replacements for unusable workers
    in the same load balancer set. This differs from hot standbys which are
    only used when all workers in a set are unusable. PR 61140. [Jim Riggs]
  * ) suexec: Add --enable-suexec-capabilites support on Linux, to use
    setuid/setgid capability bits rather than a setuid root binary.
    [Joe Orton]
  * ) suexec: Add support for logging to syslog as an alternative to
    logging to a file; use --without-suexec-logfile --with-suexec-syslog.
    [Joe Orton]
  * ) mod_ssl: Restore 2.4.29 behaviour in SSL vhost merging/enabling
    which broke some rare but previously-working configs.  [Joe Orton]
  * ) core, log: improve sanity checks for the ErrorLog's syslog config, and
    explicitly allow only lowercase 'syslog' settings. PR 62102
    [Luca Toscano, Jim Riggs, Christophe Jaillet]
  * ) mod_http2: accurate reporting of h2 data input/output per request via
    mod_logio. Fixes an issue where output sizes where counted n-times on
    reused slave connections.  [Stefan Eissing]
    See github issue: https://github.com/icing/mod_h2/issues/158
  * ) mod_http2: Fix unnecessary timeout waits in case streams are aborted.
    [Stefan Eissing]
  * ) mod_http2: restoring the v1.10.16 keepalive timeout behaviour of mod_http2.
    [Stefan Eissing]
  * ) mod_proxy: Do not restrict the maximum pool size for backend connections
    any longer by the maximum number of threads per process and use a better
    default if mod_http2 is loaded.
    [Yann Ylavic, Ruediger Pluem, Stefan Eissing, Gregg Smith]
  * ) mod_slotmem_shm: Add generation number to shm filename to fix races
    with graceful restarts. PRs 62044 and 62308.  [Jim Jagielski, Yann Ylavic]
  * ) core: Preserve the original HTTP request method in the '%%<m' LogFormat
    when an path-based ErrorDocument is used.  PR 62186.
    [Micha Lenk <micha lenk.info>]
  * ) mod_remoteip: make proxy-protocol work on slave connections, e.g. in
    HTTP/2 requests.  [Stefan Eissing]
    See also https://github.com/roadrunner2/mod-proxy-protocol/issues/6
  * ) mod_ssl: Fix merging of proxy SSL context outside <Proxy> sections,
    regression introduced in 2.4.30. PR 62232. [Rainer Jung, Yann Ylavic]
  * ) mod_md: Fix compilation with OpenSSL before version 1.0.2.  [Rainer Jung]
  * ) mod_dumpio: do nothing below log level TRACE7.  [Yann Ylavic]
  * ) mod_remoteip: Restore compatibility with APR 1.4 (apr_sockaddr_is_wildcard).
    [Eric Covener]
  * ) core: On ECBDIC platforms, some errors related to oversized headers
    may be misreported or be logged as ASCII escapes.  PR 62200
    [Hank Ibell <hwibell gmail.com>]
  * ) mod_ssl: Fix cmake-based build.  PR 62266.  [Rainer Jung]
  * ) core: Add <IfFile>, <IfDirective> and <IfSection> conditional
    section containers.  [Eric Covener, Joe Orton]
  * %%check: do not load all modules, just use default loadmodule.conf; some
  modules require to load another ones in advance
  * %%install: parallel install is broken
* Tue Mar 27 2018 mikhail.kasimov@gmail.com
- Updated description for SSLProtocol option. [bsc#1086854]
* Tue Mar 27 2018 mikhail.kasimov@gmail.com
- Updated description (PCI DSS) for SSLProtocol option. [bsc#1086854]
* Mon Mar 26 2018 pgajdos@suse.com
- SSLProtocol TLSv1.2 [bsc#1086854]
* Mon Mar 19 2018 pgajdos@suse.com
- updated to 2.4.33:
  * ) core: Fix request timeout logging and possible crash for error_log hooks.
    [Yann Ylavic]
  * ) mod_slomem_shm: Fix failure to create balancers's slotmems in Windows MPM,
    where children processes need to attach them instead since they are owned
    by the parent process already.  [Yann Ylavic]
  * ) ab: try all destination socket addresses returned by
    apr_sockaddr_info_get instead of failing on first one when not available.
    Needed for instance if localhost resolves to both ::1 and 127.0.0.1
    e.g. if both are in /etc/hosts.  [Jan Kaluza]
  * ) ab: Use only one connection to determine working destination socket
    address.  [Jan Kaluza]
  * ) ab: LibreSSL doesn't have or require Windows applink.c.  [Gregg L. Smith]
  * ) htpasswd/htdigest: Disable support for bcrypt on EBCDIC platforms.
    apr-util's bcrypt implementation doesn't tolerate EBCDIC.  [Eric Covener]
  * ) htpasswd/htdbm: report the right limit when get_password() overflows.
    [Yann Ylavic]
  * ) htpasswd: Don't fail in -v mode if password file is unwritable.
    PR 61631.  [Joe Orton]
  * ) htpasswd: don't point to (unused) stack memory on output
    to make static analysers happy.  PR 60634.
    [Yann Ylavic, reported by shqking and Zhenwei Zou]
  * ) mod_access_compat: Fail if a comment is found in an Allow or Deny
    directive.  [Jan Kaluza]
  * ) mod_authz_host: Ignore comments after "Require host", logging a
    warning, or logging an error if the line is otherwise empty.
    [Jan Kaluza, Joe Orton]
  * ) rotatelogs: Fix expansion of %%Z in localtime (-l) mode, and fix
    Y2K38 bug.  [Joe Orton]
  * ) mod_ssl: Support SSL DN raw variable extraction without conversion
    to UTF-8, using _RAW suffix on variable names.  [Joe Orton]
  * ) ab: Fix https:// connection failures (regression in 2.4.30); fix
    crash generating CSV output for large -n.  [Joe Orton, Jan Kaluza]
  * ) mod_proxy_fcgi: Add the support for mod_proxy's flushpackets and flushwait
    parameters. [Luca Toscano, Ruediger Pluem, Yann Ylavic]
  * ) mod_ldap: Avoid possible crashes, hangs, and busy loops due to
    improper merging of the cache lock in vhost config.
    PR 43164 [Eric Covener]
  * ) mpm_event: Do lingering close in worker(s).  [Yann Ylavic]
  * ) mpm_queue: Put fdqueue code in common for MPMs event and worker.
    [Yann Ylavic]
  * ) mod_session: Strip Session header when SessionEnv is on.  [Yann Ylavic]
  * ) mod_cache_socache: Fix caching of empty headers up to carriage return.
    [Yann Ylavic]
  * ) core: For consistency, ensure that read lines are NUL terminated on any
    error, not only on buffer full.  [Yann Ylavic]
  * ) mod_authnz_ldap: Fix language long names detection as short name.
    [Yann Ylavic]
  * ) mod_proxy: Worker schemes and hostnames which are too large are no
    longer fatal errors; it is logged and the truncated values are stored.
    [Jim Jagielski]
  * ) regex: Allow to configure global/default options for regexes, like
    caseless matching or extended format.  [Yann Ylavic]
  * ) mod_auth_digest: Actually use the secret when generating nonces. This change
    may cause problems if used with round robin load balancers. PR 54637
    [Stefan Fritsch]
  * ) mod_proxy: Allow setting options to globally defined balancer from
    ProxyPass used in VirtualHost. Balancers are now merged using the new
    merge_balancers method which merges the balancers options.  [Jan Kaluza]
  * ) logresolve: Fix incorrect behavior or segfault if -c flag is used
    Fixes: https://bugs.debian.org/cgi-bin/bugreport.cgi?bug=823259
    [Stefan Fritsch]
  * ) mod_remoteip: Add support for PROXY protocol (code donated by Cloudzilla).
    Add ability for PROXY protocol processing to be optional to donated code.
    See also: http://www.haproxy.org/download/1.5/doc/proxy-protocol.txt
    [Cloudzilla/roadrunner2@GitHub, Jim Jagielski, Daniel Ruggeri]
  * ) mod_proxy, mod_ssl: Handle SSLProxy* directives in <Proxy> sections,
    allowing per backend TLS configuration.  [Yann Ylavic]
  * ) mod_proxy_uwsgi: Add in UWSGI proxy (sub)module. [Roberto De Ioris,
    Jim Jagielski]
  * ) mod_proxy_balancer,mod_slotmem_shm: Rework SHM reuse/deletion to not
    depend on the number of restarts (non-Unix systems) and preserve shared
    names as much as possible on configuration changes for SHMs and persisted
    files.  PR 62044.  [Yann Ylavic, Jim Jagielski]
  * ) mod_http2: obsolete code removed, no more events on beam pool destruction,
    discourage content encoders on http2-status response (where they do not work).
    [Stefan Eissing]
  * ) mpm_event: Let the listener thread do its maintenance job on resources
    shortage.  PR 61979.  [Yann Ylavic]
  * ) mpm_event: Wakeup the listener to re-enable listening sockets.
    [Yann Ylavic]
  * ) mod_ssl: The SSLCompression directive will now give an error if used
    with an OpenSSL build which does not support any compression methods.
    [Joe Orton]
  * ) mpm_event,worker: Mask signals for threads created by modules in child
    init, so that they don't receive (implicitely) the ones meant for the MPM.
    PR 62009. [Armin Abfalterer <a.abfalterer gmail com>, Yann Ylavic]
  * ) mod_md: new experimental, module for managing domains across virtual hosts,
    implementing the Let's Encrypt ACMEv1 protocol to signup and renew
    certificates. Please read the modules documentation for further instructions
    on how to use it. [Stefan Eissing]
  * ) mod_proxy_html: skip documents shorter than 4 bytes
    PR 56286 [Micha Lenk <micha lenk info>]
  * ) core, mpm_event: Avoid a small memory leak of the scoreboard handle, for
    the lifetime of the connection, each time it is processed by MPM event.
    [Yann Ylavic]
  * ) mpm_event: Update scoreboard status for KeepAlive state.  [Yann Ylavic]
  * ) mod_ldap: Fix a case where a full LDAP cache would continually fail to
    purge old entries and log AH01323. PR61891.
    [Hendrik Harms <hendrik.harms gmail.com>]
  * ) mpm_event: close connections not reported as handled by any module to
    avoid losing track of them and leaking scoreboard entries.  PR 61551.
    [Yann Ylavic]
  * ) core: A signal received while stopping could have crashed the main
    process.  PR 61558.  [Yann Ylavic]
  * ) mod_ssl: support for mod_md added. [Stefan Eissing]
  * ) mod_proxy_html: process parsed comments immediately.
    Fixes bug (seen in the wild when used with IBM's HTTPD bundle)
    where parsed comments may be lost. [Nick Kew]
  * ) mod_proxy_html: introduce doctype for HTML 5 [Nick Kew]
  * ) mod_proxy_html: fix typo-bug processing "strict" vs "transitional"
    HTML/XHTML.  PR 56457  [Nick Kew]
  * ) mpm_event: avoid a very unlikely race condition between the listener and
    the workers when the latter fails to add a connection to the pollset.
    [Yann Ylavic]
  * ) core: silently ignore a not existent file path when IncludeOptional
    is used. PR 57585. [Alberto Murillo Silva <powerbsd yahoo.com>, Luca Toscano]
  * ) mod_macro: fix usability of globally defined macros in .htaccess files.
    PR 57525.  [Jose Kahan <jose w3.org>, Yann Ylavic]
  * ) mod_rewrite, core: add the Vary header when a condition evaluates to true
    and the related RewriteRule is used in a Directory context
    (triggering an internal redirect). [Luca Toscano]
  * ) ab: Make the TLS layer aware that the underlying socket is nonblocking,
    and use/handle POLLOUT where needed to avoid busy IOs and recover write
    errors when appropriate.  [Yann Ylavic]
  * ) ab: Keep reading nonblocking to exhaust TCP or SSL buffers when previous
    read was incomplete (the SSL case can cause the next poll() to timeout
    since data are buffered already).  PR 61301 [Luca Toscano, Yann Ylavic]
  * ) mod_http2: avoid unnecessary data retrieval for a trace log. Allow certain
    information retrievals on null bucket beams where it makes sense. [Stefan Eissing]
* Mon Mar 19 2018 pgajdos@suse.com
- Replace SuSEFirewall2 by firewalld II (fate#323460) [bsc#1083492]
* Mon Feb  5 2018 pgajdos@suse.com
- build mod_http2 also for 42.3
* Fri Feb  2 2018 pgajdos@suse.com
- remove NameVirtualHost from documentation [bsc#1078557]
* Tue Jan 30 2018 pgajdos@suse.com
- for older distros, still use SuSEFirewall2 [bsc#1071548c#7]
* Mon Jan 29 2018 knut.anderssen@suse.com
- Replace SuSEFirewall2 by firewalld (fate#323460)
* Wed Dec 20 2017 pgajdos@suse.com
- build brotli module
* Wed Dec 20 2017 fcrozat@suse.com
- Do not requires w3m, only recommends it.
* Tue Dec 19 2017 pgajdos@suse.com
- do not build with nghttp2 for SLE-12-SP3
* Fri Dec 15 2017 pgajdos@suse.com
- which was split after SLE11
* Sun Nov 26 2017 sergiolindo.empresa@gmail.com
- Add which and w3m as dependencies. poo#28406
* Thu Nov 23 2017 rbrown@suse.com
- Replace references to /var/adm/fillup-templates with new
  %%_fillupdir macro (boo#1069468)
* Mon Nov 20 2017 pgajdos@suse.com
- APACHE_MODULES now contains authn_core in default configuration
  [bsc#1066661]
* Wed Oct 18 2017 pgajdos@suse.com
- updated to 2.4.29:
  * ) mod_unique_id: Use output of the PRNG rather than IP address and
    pid, avoiding sleep() call and possible DNS issues at startup,
    plus improving randomness for IPv6-only hosts.  [Jan Kaluza]
  * ) mod_rewrite, core: Avoid the 'Vary: Host' response header when HTTP_HOST
    is used in a condition that evaluates to true. PR 58231 [Luca Toscano]
  * ) mod_http2: v0.10.12, removed optimization for mutex handling in bucket
    beams that could lead to assertion failure in edge cases.
    [Stefan Eissing]
  * ) mod_proxy: Fix regression for non decimal loadfactor parameter introduced
    in 2.4.28.  [Jim Jagielski]
  * ) mod_authz_dbd: fix a segmentation fault if AuthzDBDQuery is not set.
    PR 61546.  [Lubos Uhliarik <luhliari redhat.com>]
  * ) mod_rewrite: Add support for starting External Rewriting Programs
    as non-root user on UNIX systems by specifying username and group
    name as third argument of RewriteMap directive.  [Jan Kaluza]
  * ) core: Rewrite the Content-Length filter to avoid excessive memory
    consumption. Chunked responses will be generated in more cases
    than in previous releases.  PR 61222.  [Joe Orton, Ruediger Pluem]
  * ) mod_ssl: Fix SessionTicket callback return value, which does seem to
    matter with OpenSSL 1.1. [Yann Ylavic]
* Tue Oct 17 2017 pgajdos@suse.com
- gensslcert:
  * set also SAN [bsc#1045159]
  * drop -C argument, it was not mapped to CN actually
  * consider also case when hostname does return empty string or
    does not exist [bsc#1057406]
  * do not consider environment ROOT variable
* Fri Oct  6 2017 pgajdos@suse.com
- updated to 2.4.28:
  * ) SECURITY: CVE-2017-9798 (cve.mitre.org)
    Corrupted or freed memory access. <Limit[Except]> must now be used in the
    main configuration file (httpd.conf) to register HTTP methods before the
    .htaccess files.  [Yann Ylavic]
  * ) event: Avoid possible blocking in the listener thread when shutting down
    connections. PR 60956.  [Yann Ylavic]
  * ) mod_speling: Don't embed referer data in a link in error page.
    PR 38923 [Nick Kew]
  * ) htdigest: prevent a buffer overflow when a string exceeds the allowed max
    length in a password file.
    [Luca Toscano, Hanno Böck <hanno hboeck de>]
  * ) mod_proxy: loadfactor parameter can now be a decimal number (eg: 1.25).
    [Jim Jagielski]
  * ) mod_proxy_wstunnel: Allow upgrade to any protocol dynamically.
    PR 61142.
  * ) mod_watchdog/mod_proxy_hcheck: Time intervals can now be spefified
    down to the millisecond. Supports 'mi' (minute), 'ms' (millisecond),
    's' (second) and 'hr' (hour!) time suffixes. [Jim Jagielski]
  * ) mod_http2: Fix for stalling when more than 32KB are written to a
    suspended stream.  [Stefan Eissing]
  * ) build: allow configuration without APR sources.  [Jacob Champion]
  * ) mod_ssl, ab: Fix compatibility with LibreSSL.  PR 61184.
    [Bernard Spil <brnrd freebsd.org>, Michael Schlenker <msc contact.de>,
    Yann Ylavic]
  * ) core/log: Support use of optional "tag" in syslog entries.
    PR 60525. [Ben Rubson <ben.rubson gmail.com>, Jim Jagielski]
  * ) mod_proxy: Fix ProxyAddHeaders merging.  [Joe Orton]
  * ) core: Disallow multiple Listen on the same IP:port when listener buckets
    are configured (ListenCoresBucketsRatio > 0), consistently with the single
    bucket case (default), thus avoiding the leak of the corresponding socket
    descriptors on graceful restart.  [Yann Ylavic]
  * ) event: Avoid listener periodic wake ups by using the pollset wake-ability
    when available.  PR 57399.  [Yann Ylavic, Luca Toscano]
  * ) mod_proxy_wstunnel: Fix detection of unresponded request which could have
    led to spurious HTTP 502 error messages sent on upgrade connections.
    PR 61283.  [Yann Ylavic]
* Thu Oct  5 2017 pgajdos@suse.com
- suexec binary moved to main package [bsc#1054741]
* Tue Oct  3 2017 pgajdos@suse.com
- do not call and do not ship apache-22-24-upgrade [bsc#1042037]
* Mon Jul 24 2017 schneemann@b1-systems.de
- make the package runable on non systemd systems
  + deprecated-scripts-arch.patch
* Mon Jul 17 2017 pgajdos@suse.com
- updated to 2.4.27:
  * ) COMPATIBILITY: mod_lua: Remove the undocumented exported 'apr_table'
    global variable when using Lua 5.2 or later. This was exported as a
    side effect from luaL_register, which is no longer supported as of
    Lua 5.2 which deprecates pollution of the global namespace.
    [Rainer Jung]
  * ) COMPATIBILITY: mod_http2: Disable and give warning when using Prefork.
    The server will continue to run, but HTTP/2 will no longer be negotiated.
    [Stefan Eissing]
  * ) COMPATIBILITY: mod_proxy_fcgi: Revert to 2.4.20 FCGI behavior for the
    default ProxyFCGIBackendType, fixing a regression with PHP-FPM. PR 61202.
    [Jacob Champion, Jim Jagielski]
  * ) mod_lua: Improve compatibility with Lua 5.1, 5.2 and 5.3.
    PR58188, PR60831, PR61245. [Rainer Jung]
  * ) mod_http2: Simplify ready queue, less memory and better performance. Update
    mod_http2 version to 1.10.7. [Stefan Eissing]
  * ) Allow single-char field names inadvertantly disallowed in 2.4.25.
    PR 61220. [Yann Ylavic]
  * ) htpasswd / htdigest: Do not apply the strict permissions of the temporary
    passwd file to a possibly existing passwd file. PR 61240. [Ruediger Pluem]
  * ) core: Avoid duplicate HEAD in Allow header.
    This is a regression in 2.4.24 (unreleased), 2.4.25 and 2.4.26.
    PR 61207. [Christophe Jaillet]
- drop upstreamed patch:
  * httpd-2.4.12-lua-5.2.patch (see upstream's PR#58188 for details)
* Wed Jul 12 2017 mpluskal@suse.com
- Adjust dependencies for rename apr/apr-util packages
* Tue Jun 20 2017 pgajdos@suse.com
-  remove /usr/bin/http2 symlink only during apache2 package
  uninstall, not upgrade [bsc#1041830]
* Mon Jun 19 2017 pgajdos@suse.com
- updated to 2.4.26: This release of Apache is a security, feature,
  and bug fix release. For details, see
  http://httpd.apache.org/dev/dist/CHANGES_2.4.26
- refreshed patches:
  . httpd-2.4.12-lua-5.2.patch
  . httpd-2.4.x-fate317766-config-control-two-protocol-options.diff
- removed patches (upstreamed)
  . httpd-cache-forward-http-proxy.patch
  . httpd-cache-revert-svn1773397.patch
* Fri Jun  2 2017 pgajdos@suse.com
- server-tunning.conf: MaxClients was renamed to MaxRequestWorkers
  [bsc#1037731]
- gensslcert: use hostname when fqdn is too long [bsc#1035829]
* Fri May 19 2017 pgajdos@suse.com
- remove apache-doc and apache-example-pages obsoletes/provides
* Thu Mar 16 2017 kukuk@suse.com
- PreRequire user wwwrun and group www
* Tue Mar  7 2017 pgajdos@suse.com
- start_apache2: include individual sysconfig.d files instead of
  sysconfig.d dir, include sysconfig.d/include.conf after httpd.conf
  is processed [bsc#1023616]
* Thu Feb 23 2017 pgajdos@suse.com
- revert an attempt to fix PR 60458
  + httpd-cache-revert-svn1773397.patch
* Tue Feb 21 2017 pgajdos@suse.com
- fix caching of forward proxy
  + httpd-cache-forward-http-proxy.patch
* Sat Feb 18 2017 kukuk@suse.com
- Don't require insserv if we don't need it.
* Tue Jan 10 2017 jweberhofer@weberhofer.at
- Added new HTTP2 option to sysconfig to ease enabling http2
- Added new protocols.conf which is included globally
- Enable http2 also for Leap 42.2+ and SLE12_SP2+
* Mon Jan  2 2017 pgajdos@suse.com
- update to 2.4.25: fixed several security issues (CVE-2016-8740,
  CVE-2016-5387, CVE-2016-2161, CVE-2016-0736, CVE-2016-8743), many
  fixes and improvements of mod_http2 and other modules; see CHANGES
  for full change log
- verify tarball: added httpd*.bz2.asc, apache2.keyring and remove
  60C5442D.key
* Fri Dec  9 2016 pgajdos@suse.com
- fix build with new systemd
  + amended httpd-2.4.3-mod_systemd.patch
* Fri Dec  2 2016 jimmy@boombatower.com
- Replace mixed indentation with predominant style.
* Tue Sep 13 2016 pgajdos@suse.com
- add NotifyAccess=all to service file [bsc#980663]
* Fri Aug  5 2016 tchvatal@suse.com
- Remove the omc xml config. It is useless nowdays
* Fri Aug  5 2016 pgajdos@suse.com
- readd the support of multiple entries in APACHE_ACCESS_LOG
  [bsc#991032]
* Tue Jul 12 2016 kstreitova@suse.com
- add httpd-2.4.x-fate317766-config-control-two-protocol-options.diff
  Introduces directives to control two protocol options:
  * HttpContentLengthHeadZero - allow Content-Length of 0 to be
    returned on HEAD
  * HttpExpectStrict - allow admin to control whether we must
    see "100-continue"
  [bsc#894225], [fate#317766]
* Wed Jul  6 2016 crrodriguez@opensuse.org
- version 2.4.23
  * Fixes CVE-2016-4979 [bsc#987365]
  * mod_proxy_hcheck was missing due to upstream bug.
  * mod_proxy_fdpass needs explicit configure line now.
  * Full list of changes:
  http://www-eu.apache.org/dist//httpd/CHANGES_2.4.23
* Wed Jul  6 2016 fbui@suse.com
- Remove pkgconfig(libsystemd-daemon). Nowadays pkgconfig(libsystemd)
  is enough and replaces all libsystemd-* libs which are obsolete.
* Thu May 26 2016 pgajdos@suse.com
- remove Alias= from [Install] of the template service
  [bsc#981541c#10]
* Wed May 18 2016 pgajdos@suse.com
- remove unneded httpd-2.4.17-debug-crash.patch
* Mon May  9 2016 pgajdos@suse.com
- start apache services after remote-fs [bsc#978543]
* Thu May  5 2016 pgajdos@suse.com
- removed note about ulimits in sysconfig file [bsc#976711]
* Mon May  2 2016 pgajdos@suse.com
- do not build mod_http2 for 13.2
* Mon Apr 11 2016 crrodriguez@opensuse.org
- Update to version 2.4.20 (2.4.19 was never released)
  * Drop httpd-2.4.18-missing-semicolon.patch now upstream
- Big changelog available, see:
  http://www.apache.org/dist/httpd/CHANGES_2.4.20 for details.
* Mon Apr  4 2016 pgajdos@suse.com
- enable authnz_fcgi module
* Mon Dec 14 2015 pgajdos@suse.com
- fix build for SLE_11_SP4:
  + httpd-2.4.18-missing-semicolon.patch
* Sat Dec 12 2015 crrodriguez@opensuse.org
- Update to version 2.4.18
  * drop 2.4.17-protocols.patch in upstream.
- Change list too long to mention here see:
  http://www.apache.org/dist/httpd/CHANGES_2.4.18 for details.
* Mon Dec  7 2015 crrodriguez@opensuse.org
- systemd: Set TasksMax=infinity for current systemd releases.
  The default limit of 512 is too small and prevents the creation of
  new server processes. Apache has its own runtime/harcoded limits.
* Thu Dec  3 2015 pgajdos@suse.com
- fix crash when for -X
  + httpd-2.4.17-debug-crash.patch
* Mon Nov 23 2015 pgajdos@suse.com
- add a note: FollowSymLinks or SymLinksIfOwnerMatch is neccessary
  for RewriteRule in given dir [bnc#955701]
* Fri Nov  6 2015 pgajdos@suse.com
- restart apache once after the rpm or zypper transaction
  [bnc#893659]
- drop some old compat code from %%post
* Thu Nov  5 2015 crrodriguez@opensuse.org
- 2.4.17-protocols.patch from upstream http2 module:
  * master conn_rec* addition to conn_rec
  * improved ALPN and Upgrade handling
  * allowing requests for servers whose TLS configuration is compatible
  to the SNI server ones
  * disabling TLS renegotiation for slave connections
* Wed Nov  4 2015 pgajdos@suse.com
- LogLevel directive into correct config file, thanks Michael Calmer
  for the fix [bsc#953329]
* Mon Oct 26 2015 pgajdos@suse.com
- do not build mod_http2 for older distros than 13.2 for now (nghttp2
  does not build there)
* Mon Oct 26 2015 pgajdos@suse.com
- Include directives really into /etc/apache2/sysconfig.d/include.conf,
  fix from Erik Wegner [bsc#951901]
* Wed Oct 21 2015 pgajdos@suse.com
- gensslcert: CN now defaults to `hostname -f` [bnc#949766]
  (internal), fix help [bnc#949771] (internal)
* Sun Oct 18 2015 crrodriguez@opensuse.org
-  Update to 2.4.17
- Enable mod_http2/ BuildRequire nghttp2
- MPMs: Support SO_REUSEPORT to create multiple duplicated listener
  records for scalability
- mod_ssl: Support compilation against libssl built with OPENSSL_NO_SSL3
- For more changes see: http://www.apache.org/dist/httpd/CHANGES_2.4.17
* Mon Oct 12 2015 pgajdos@suse.com
- start_apache2: reintroduce sysconfig.d, include it on
  command line (not in httpd.conf) instead of individual directives
  [bnc#949434] (internal), [bnc#941331]
* Thu Aug 13 2015 schwab@suse.de
- Fixup libdir in installed files
* Tue Aug 11 2015 kstreitova@suse.com
- fix Logjam vulnerability: change SSLCipherSuite cipherstring to
  disable export cipher suites and deploy Ephemeral Elliptic-Curve
  Diffie-Hellman (ECDHE) ciphers. Adjust 'gensslcert' script to
  generate a strong and unique Diffie Hellman Group and append it
  to the server certificate file [bnc#931723], [CVE-2015-4000]
* Wed Jul 29 2015 pgajdos@suse.com
- add reference upstream bug#58188 along httpd-2.4.12-lua-5.2.patch
* Mon Jul 20 2015 kstreitova@suse.com
- update to 2.4.16
  * changes http://www.apache.org/dist/httpd/CHANGES_2.4.16
  * remove the following patches (fixed in 2.4.16)
  * httpd-2.4.x-mod_lua_websocket_DoS.patch
  * httpd-2.4.12-CVE-2015-0253.patch
  * update httpd-2.4.12-lua-5.2.patch
* Sat Jul 18 2015 i@marguerite.su
- add patch: httpd-2.4.12-lua-5.2.patch
  * lua_dump introduced a new strip option in 5.3, set it to 0
    to get the old behavior
  * luaL_register was deprecated in 5.2, use luaL_setfuncs and
    luaL_newlib instead
  * luaL_optint was deprecated in 5.3, use luaL_optinteger instead
  * lua_strlen and lua_objlen wad deprecated in 5.2, use lua_rawlen
    instead
* Thu Jul 16 2015 pgajdos@suse.com
- change Provides: from suse_maintenance_mmn = # to
  suse_maintenance_mmn_#
* Wed Jul 15 2015 pgajdos@suse.com
- apache2 Suggests:, not Recommends: apache2-prefork; that means
  for example, that `zypper in apache2-worker` will not pull
  apache2-prefork also
- installing /usr/sbin/httpd link:
  * do not try to install it in '%%post <MPM>' when apache2 (which
    includes /usr/share/apache2/script-helpers) is not installed
    yet (fixes installation on 11sp3)
  * install it in '%%post' if apache2 is installed after
    apache2-<MPM> to be sure it is there
* Tue Jul 14 2015 pgajdos@suse.com
- access_compat shared also for 11sp3
* Mon Jul 13 2015 pgajdos@suse.com
- apache2-implicit-pointer-decl.patch renamed to
  httpd-implicit-pointer-decl.patch to align with other
  patches names
* Mon Jul 13 2015 pgajdos@suse.com
- apachectl is now wrapper to start_apache2; therefore, it honors
  HTTPD_INSTANCE variable, see README-instances.txt for details
  + httpd-apachectl.patch
  - httpd-2.4.10-apachectl.patch
* Mon Jul 13 2015 pgajdos@suse.com
- a2enmod/a2dismod and a2enflag/a2disflag now respect
  HTTPD_INSTANCE=<instance_name> environment variable, which can be
  used to specify apache instance name; sysconfig file is expected
  at /etc/sysconfig/apache2@<instance_name>
  (see README-instances.txt for details)
* Mon Jul 13 2015 pgajdos@suse.com
- provides suse_maintenance_mmn symbol [bnc#915666] (internal)
* Mon Jul 13 2015 pgajdos@suse.com
- credits to Roman Drahtmueller:
  * add reference to /etc/permissions.local to output of %%post if
    setting the permissions of suexec2 fails
  * do not enable mod_php5 by default any longer
  * httpd-2.0.49-log_server_status.dif obsoleted
  * apache2-mod_ssl_npn.patch removed because not used
  * include mod_reqtimeout.conf in httpd.conf
  * added cgid-timeout.conf, include
    it in httpd.conf
- fix default value APACHE_MODULES in sysconfig file
- %%service_* macros for apache2@.service
* Mon Jul 13 2015 pgajdos@suse.com
- reenable 690734.patch, it should be upstreamed by the author
  (Adrian Schroeter) though
  + httpd-2.4.9-bnc690734.patch
  - httpd-2.2.x-bnc690734.patch
* Mon Jul 13 2015 pgajdos@suse.com
- drop startssl from start_apache2
* Wed Jul  1 2015 pgajdos@suse.com
- allow to run multiple instances of Apache on one system
  [fate#317786] (internal)
  * distributed httpd.conf no longer includes sysconfig.d, nor this
    directory is shipped. httpd.conf includes loadmodule.conf and
    global.conf which are former sysconfig.d/loadmodule.conf and
    sysconfig.d/global.conf for default /etc/sysconfig/apache2
    global.conf and loadmodule.conf are not included when
    sysconfig variables could have been read by start_apache2
    startup script (run with systemd services). Therefore, when
    starting server via /usr/sbin/httpd, sysconfig variables
    are not taken into account.
  * some not-maintained scripts are moved from
    /usr/share/apache2 to /usr/share/apache2/deprecated-scripts
  * all modules comment in sysconfig file is not generated
    anymore
  * added README-instances.txt
  * removed Sources:
    load_configuration
    find_mpm
    get_module_list
    get_includes
    find_httpd_includes
    apache-find-directives
  * added Sources:
    deprecated-scripts.tar.xz
    apache2-README-instances.txt
    apache2-loadmodule.conf
    apache2-global.conf
    apache2-find-directives
    apache2@.service
    apache2-script-helpers
* Thu Jun 25 2015 kstreitova@suse.com
- add SSLHonorCipherOrder directive to apache2-ssl-global.conf
- adopt SSLCipherSuite directive value from SLE12
- remove default-vhost-ssl.conf and default-vhost.conf from
  /etc/apache2. These two files are not (!) read by the
  configuration framework, but are named *.conf, which is
  misleading. The files are almost identical with the vhost
  templates in /etc/apache2/vhosts.d/. The two templates there do
  it right because they are not named *.conf and are not sourced
  either. apache's response with no explicit (eg. default, vanilla)
  configuration is contained in /etc/apache2/default-server.conf.
  * remove apache2-README.default-vhost as there are no
    default-vhost* files anymore.
* Thu Jun 25 2015 crrodriguez@opensuse.org
- apache2.service: We have to use KillMode=mixed for the
  graceful stop, restart to work properly.
* Thu Jun 11 2015 pgajdos@suse.com
- dropped 2.0 -> 2.2 modules transition during upgrade
  * apache-20-22-upgrade renamed to apache-22-24-upgrade
- apache-*-upgrade script is called in %%posttrans now [bnc#927223]
* Tue Jun  9 2015 pgajdos@suse.com
- fix find_mpm to echo mpm binary
* Tue Jun  2 2015 crrodriguez@opensuse.org
- apache2.service: Only order us after network.target and
  nss-lookup.target but not pull the units in.
- apache2.service: SSL requires correct system time to
  work properly, order after time-sync.target
* Tue May 26 2015 pgajdos@suse.com
- align filenames with upstream names (and add compat symlinks)
- find_httpd2_includes renamed to find_httpd_includes
* Mon May 25 2015 pgajdos@suse.com
- access_compat now built as shared and disabled by default
- amend config to use also old syntax when access_compat is
  loaded
- added apache2-README-access_compat.txt
- added apache-find-directive script
- see [bnc#896083] and its duplicates
* Mon May 11 2015 hguo@suse.com
- add httpd-2.4.12-CVE-2015-0253.patch to fix SECURITY: CVE-2015-0253
  (cve.mitre.org) core: Fix a crash introduced in with ErrorDocument
  400 pointing to a local URL-path with the INCLUDES filter active,
  introduced in 2.4.11. PR 57531. [Yann Ylavic]
* Tue May  5 2015 kstreitova@suse.com
- simplify apache2.logrotate, use sharedscripts [bnc#713581]
* Tue May  5 2015 kstreitova@suse.com
- remove curly brackets around format sequence "%%y" in
  `stat --format="%%{y}" %%{SOURCE1}` that caused an incorrect
  evaluation. Add escaping to proper spec-cleaner processing in
  the future
* Thu Apr  9 2015 kstreitova@suse.com
- remove 'exit 0' from the %%post section in the specfile that was
  placed here incorrectly and caused that the rest of the %%post
  section couldn't be executed.
* Thu Apr  9 2015 pgajdos@suse.com
- /etc/init.d/apache2 reload -> systemctl reload apache2.service
  in apache2.logrotate [bnc#926523]
* Mon Mar 30 2015 pgajdos@suse.com
- authz_default -> authz_core in sysconfig.apache2/APACHE_MODULES
  [bnc#922236]
* Tue Mar 10 2015 dimstar@opensuse.org
- Add Requires(post) apache2 to the subpackage -worker, -event and
  - prefork: their respective post scriptlets execute
  /usr/share/apache2/get_module_list, which is shipped as part of
  the main package. This script has the side-effect to call
  find_mpm, which in turn creates the corresponding /usr/sbin/httpd2
  symlink.
* Thu Feb 26 2015 jsegitz@novell.com
- Patched get_module_list to ensure proper SELinux context for
  sysconfig.d/loadmodule.conf
* Wed Feb 25 2015 tchvatal@suse.com
- Pname -> name variable reduction
- Try to fix sle11 build
* Wed Feb 25 2015 tchvatal@suse.com
- Version bumpt o 2.4.12:
  * ) mpm_winnt: Accept utf-8 (Unicode) service names and descriptions for
    internationalization.  [William Rowe]
  * ) mpm_winnt: Normalize the error and status messages emitted by service.c,
    the service control interface for Windows.  [William Rowe]
  * ) configure: Fix --enable-v4-mapped configuration on *BSD. PR 53824.
    [ olli hauer <ohauer gmx.de>, Yann Ylavic ]
* Wed Feb 25 2015 tchvatal@suse.com
- Exit cleanly on end of the post and cleanup the update detection
- Remove Apache.xpm as it ain't used
* Wed Feb 25 2015 tchvatal@suse.com
- Cleanup init/unit decision making and provide just systemd service
  on systemd systems
* Wed Feb 25 2015 tchvatal@suse.com
- Deprecate realver define as it is equal to version.
- Explicitely state MPM mods to ensure we don't lose some bnc#444878
* Wed Feb 25 2015 tchvatal@suse.com
- Pass over spec-cleaner, there should be no actual technical
  change in this just reduction of lines in the spec
* Mon Feb 23 2015 kstreitova@suse.com
- add httpd-2.4.x-mod_lua_websocket_DoS.patch to fix mod_lua bug
  where a maliciously crafted websockets PING after a script calls
  r:wsupgrade() can cause a child process crash
  [CVE-2015-0228], [bnc#918352].
* Tue Feb  3 2015 pgajdos@suse.com
- httpd2.pid in rc.apache2 was wrong [bnc#898193]
* Mon Jan 19 2015 crrodriguez@opensuse.org
- httpd-2.4.3-mod_systemd.patch find libsystemd-daemon
  with pkg-config, this is the only correct way, in current
  versions sd_notify is in libsystemd and in old products
  in libsystemd-daemon.
* Fri Jan 16 2015 crrodriguez@opensuse.org
- remove obsolete patches
  * httpd-2.4.10-check_null_pointer_dereference.patch
  * httpd-event-deadlock.patch
  * httpd-2.4.x-bnc871310-CVE-2013-5704-mod_headers_chunked_requests.patch
  * httpd-2.4.x-bnc909715-CVE-2014-8109-mod_lua_handling_of_Require_line.patch
* Fri Jan 16 2015 crrodriguez@opensuse.org
- Apache 2.4.11
  * ) SECURITY: CVE-2014-3583 (cve.mitre.org)
    mod_proxy_fcgi: Fix a potential crash due to buffer over-read, with
    response headers' size above 8K.  [Yann Ylavic, Jeff Trawick]
  * ) SECURITY: CVE-2014-3581 (cve.mitre.org)
    mod_cache: Avoid a crash when Content-Type has an empty value.
    PR 56924.  [Mark Montague <mark catseye.org>, Jan Kaluza]
  * ) SECURITY: CVE-2014-8109 (cve.mitre.org)
    mod_lua: Fix handling of the Require line when a LuaAuthzProvider is
    used in multiple Require directives with different arguments.
    PR57204 [Edward Lu <Chaosed0 gmail.com>]
  * ) SECURITY: CVE-2013-5704 (cve.mitre.org)
    core: HTTP trailers could be used to replace HTTP headers
    late during request processing, potentially undoing or
    otherwise confusing modules that examined or modified
    request headers earlier.  Adds "MergeTrailers" directive to restore
    legacy behavior.  [Edward Lu, Yann Ylavic, Joe Orton, Eric Covener]
  * ) mod_ssl: New directive SSLSessionTickets (On|Off).
    The directive controls the use of TLS session tickets (RFC 5077),
    default value is "On" (unchanged behavior).
    Session ticket creation uses a random key created during web
    server startup and recreated during restarts. No other key
    recreation mechanism is available currently. Therefore using session
    tickets without restarting the web server with an appropriate frequency
    (e.g. daily) compromises perfect forward secrecy. [Rainer Jung]
  * ) mod_proxy_fcgi: Provide some basic alternate options for specifying
    how PATH_INFO is passed to FastCGI backends by adding significance to
    the value of proxy-fcgi-pathinfo. PR 55329. [Eric Covener]
  * ) mod_proxy_fcgi: Enable UDS backends configured with SetHandler/RewriteRule
    to opt-in to connection reuse and other Proxy options via explicitly
    declared "proxy workers" (<Proxy unix:... enablereuse=on max=...)
    [Eric Covener]
  * ) mod_proxy: Add "enablereuse" option as the inverse of "disablereuse".
    [Eric Covener]
  * ) mod_proxy_fcgi: Enable opt-in to TCP connection reuse by explicitly
    setting proxy option disablereuse=off. [Eric Covener] PR 57378.
  * ) event: Update the internal "connection id" when requests
    move from thread to thread. Reuse can confuse modules like
    mod_cgid. PR 57435. [Michael Thorpe <mike gistnet.com>]
  * ) mod_proxy_fcgi: Remove proxy:balancer:// prefix from SCRIPT_FILENAME
    passed to fastcgi backends. [Eric Covener]
  * ) core: Configuration files with long lines and continuation characters
    are not read properly. PR 55910. [Manuel Mausz <manuel-as mausz.at>]
  * ) mod_include: the 'env' function was incorrectly handled as 'getenv' if the
    leading 'e' was written in upper case in <!--#if expr="..." -->
    statements. [Christophe Jaillet]
  * ) split-logfile: Fix perl error:  'Can't use string ("example.org:80")
    as a symbol ref while "strict refs"'. PR 56329.
    [Holger Mauermann <mauermann gmail.com>]
  * ) mod_proxy: Prevent ProxyPassReverse from doing a substitution when
    the URL parameter interpolates to an empty string. PR 56603.
    [<ajprout hotmail.com>]
  * ) core: Fix -D[efined] or <Define>[d] variables lifetime accross restarts.
    PR 57328.  [Armin Abfalterer <a.abfalterer gmail.com>, Yann Ylavic].
  * ) mod_proxy: Preserve original request headers even if they differ
    from the ones to be forwarded to the backend. PR 45387.
    [Yann Ylavic]
  * ) mod_ssl: dump SSL IO/state for the write side of the connection(s),
    like reads (level TRACE4). [Yann Ylavic]
  * ) mod_proxy_fcgi: Ignore body data from backend for 304 responses. PR 57198.
    [Jan Kaluza]
  * ) mod_ssl: Do not crash when looking up SSL related variables during
    expression evaluation on non SSL connections. PR 57070  [Ruediger Pluem]
  * ) mod_proxy_ajp: Fix handling of the default port (8009) in the
    ProxyPass and <Proxy> configurations.  PR 57259.  [Yann Ylavic]
  * ) mpm_event: Avoid a possible use after free when notifying the end of
    connection during lingering close.  PR 57268.  [Eric Covener, Yann Ylavic]
  * ) mod_ssl: Fix recognition of OCSP stapling responses that are encoded
    improperly or too large.  [Jeff Trawick]
  * ) core: Add ap_log_data(), ap_log_rdata(), etc. for logging buffers.
    [Jeff Trawick]
  * ) mod_proxy_fcgi, mod_authnz_fcgi: stop reading the response and issue an
    error when parsing or forwarding the response fails. [Yann Ylavic]
  * ) mod_ssl: Fix a memory leak in case of graceful restarts with OpenSSL >= 0.9.8e
    PR 53435 [tadanori <tadanori2007 yahoo.com>, Sebastian Wiedenroth <wiedi frubar.net>]
  * ) mod_proxy_connect: Don't issue AH02447 on sockets hangups, let the read
    determine whether it is a normal close or a real error. PR 57168. [Yann
    Ylavic]
  * ) mod_proxy_wstunnel: abort backend connection on polling error to avoid
    further processing.  [Yann Ylavic]
  * ) core: Support custom ErrorDocuments for HTTP 501 and 414 status codes.
    PR 57167 [Edward Lu <Chaosed0 gmail.com>]
  * ) mod_proxy_connect: Fix ProxyRemote to https:// backends on EBCDIC
    systems. PR 57092 [Edward Lu <Chaosed0 gmail.com>]
  * ) mod_cache: Avoid a 304 response to an unconditional requst when an AH00752
    CacheLock error occurs during cache revalidation. [Eric Covener]
  * ) mod_ssl: Move OCSP stapling information from a per-certificate store to
    a per-server hash. PR 54357, PR 56919. [Alex Bligh <alex alex.org.uk>,
    Yann Ylavic, Kaspar Brand]
  * ) mod_cache_socache: Change average object size hint from 32 bytes to
    2048 bytes.  [Rainer Jung]
  * ) mod_cache_socache: Add cache status to server-status.  [Rainer Jung]
  * ) event: Fix worker-listener deadlock in graceful restart.
    PR 56960.
  * ) Concat strings at compile time when possible. PR 53741.
  * ) mod_substitute: Restrict configuration in .htaccess to
    FileInfo as documented.  [Rainer Jung]
  * ) mod_substitute: Make maximum line length configurable.  [Rainer Jung]
  * ) mod_substitute: Fix line length limitation in case of regexp plus flatten.
    [Rainer Jung]
  * ) mod_proxy: Truncated character worker names are no longer fatal
    errors. PR53218. [Jim Jagielski]
  * ) mod_dav: Set r->status_line in dav_error_response. PR 55426.
  * ) mod_proxy_http, mod_cache: Avoid (unlikely) accesses to freed memory.
    [Yann Ylavic, Christophe Jaillet]
  * ) http_protocol: fix logic in ap_method_list_(add|remove) in order:
  - to correctly reset bits
  - not to modify the 'method_mask' bitfield unnecessarily
    [Christophe Jaillet]
  * ) mod_slotmem_shm: Increase log level for some originally debug messages.
    [Jim Jagielski]
  * ) mod_ldap: In 2.4.10, some LDAP searches or comparisons might be done with
    the wrong credentials when a backend connection is reused.
    [Eric Covener]
  * ) mod_macro: Add missing APLOGNO for some Warning log messages.
    [Christophe Jaillet]
  * ) mod_cache: Avoid sending 304 responses during failed revalidations
    PR56881. [Eric Covener]
  * ) mod_status: Honor client IP address using mod_remoteip. PR 55886.
    [Jim Jagielski]
  * ) cmake-based build for Windows: Fix incompatibility with cmake 2.8.12
    and later.  PR 56615.  [Chuck Liu <cliu81 gmail.com>, Jeff Trawick]
  * ) mod_ratelimit: Drop severity of AH01455 and AH01457 (ap_pass_brigade
    failed) messages from ERROR to TRACE1.  Other filters do not bother
    re-reporting failures from lower level filters.  PR56832.  [Eric Covener]
  * ) core: Avoid useless warning message when parsing a section guarded by
    <IfDefine foo> if $(foo) is used within the section.
    PR 56503 [Christophe Jaillet]
  * ) mod_proxy_fcgi: Fix faulty logging of large amounts of stderr from the
    application.  PR 56858.  [Manuel Mausz <manuel-asf mausz.at>]
  * ) mod_proxy_http: Proxy responses with error status and
    "ProxyErrorOverride On" hang until proxy timeout.
    PR53420 [Rainer Jung]
  * ) mod_log_config: Allow three character log formats to be registered. For
    backwards compatibility, the first character of a three-character format
    must be the '^' (caret) character.  [Eric Covener]
  * ) mod_lua: Don't quote Expires and Path values. PR 56734.
    [Keith Mashinter, <kmashint yahoo com>]
  * ) mod_authz_core: Allow <AuthzProviderAlias>'es to be seen from auth
    stanzas under virtual hosts. PR 56870. [Eric Covener]
* Mon Jan 12 2015 bruno@ioda-net.ch
- Redone lost patch to fix boo#859439
  + service reload can cause log data to be lost with logrotate
  under some circumstances: remove "-t" from service reload.
  [bnc#859439]
* Thu Jan  8 2015 schwab@linux-m68k.org
- Fix URL syntax in various files
* Mon Dec 29 2014 pgajdos@suse.com
- fix IfModule directive around SSLSessionCache [bnc#842377c#11]
* Mon Dec 15 2014 kstreitova@suse.com
- added httpd-2.4.x-bnc871310-CVE-2013-5704-mod_headers_chunked_requests.patch
  to fix flaw in the way mod_headers handled chunked requests. Adds
  "MergeTrailers" directive to restore legacy behavior
  [bnc#871310], [CVE-2013-5704].
* Fri Dec 12 2014 kstreitova@suse.com
- added httpd-2.4.x-bnc909715-CVE-2014-8109-mod_lua_handling_of_Require_line.patch
  that fixes handling of the Require line when a LuaAuthzProvider is
  used in multiple Require directives with different arguments
  [bnc#909715], [CVE-2014-8109].
* Fri Dec  5 2014 pgajdos@suse.com
- fixed start at boot for ssl and encrypted key [bnc#792309]
* Sat Nov 29 2014 ledest@gmail.com
- fix shebang in start_apache2 script that contains bash-specific
  constructions
* Thu Nov 27 2014 pgajdos@suse.com
- small improvement of ssl instructions [bnc#891813]
* Sun Nov  9 2014 ledest@gmail.com
- fix bashisms in post scripts
* Fri Nov  7 2014 kstreitova@suse.com
- added httpd-2.4.10-check_null_pointer_dereference.patch to avoid
  a crash when Content-Type has an empty value [bnc#899836],
  CVE-2014-3581
* Fri Oct 31 2014 crrodriguez@opensuse.org
- httpd-event-deadlock.patch:  Fix worker-listener
  deadlock in graceful restart.
* Sat Oct 18 2014 ledest@gmail.com
- httpd-2.1.9-apachectl.dif renamed to httpd-2.4.10-apachectl.patch
  and updated (fixed bashism).
* Thu Oct 16 2014 pgajdos@suse.com
- drop (turned off) itk mpm spec file code as mpm-itk is now
  provided as a separate module, not via patch
  (see http://mpm-itk.sesse.net/ and [bnc#851229])
* Mon Oct 13 2014 pgajdos@suse.com
- enable mod_imagemap [bnc#866366]
* Fri Oct 10 2014 pgajdos@suse.com
- fixed link to Apache quickstart [bnc#624681], [bnc#789806]
* Mon Oct  6 2014 kstreitova@suse.com
- the following unused patches were removed from the package:
  * apache2-mod_ssl_npn.patch
  * httpd-2.0.49-log_server_status.dif
* Mon Sep 29 2014 pgajdos@suse.com
- 700 permissions for /usr/sbin/apache2-systemd-ask-pass and
  /usr/sbin/start_apache2 [bnc#851627]
* Fri Sep 26 2014 oholecek@suse.com
- allow only TCP ports in Yast2 firewall files
* Fri Sep 26 2014 pgajdos@suse.com
- more 2.2 -> 2.4 [bnc#862058]
* Thu Sep 25 2014 pgajdos@suse.com
- ServerSignature=Off and ServerTokens=Prod by request from
  security team [bnc#716495]
* Wed Sep 24 2014 pgajdos@suse.com
- fix documentation links 2.2 -> 2.4 [bnc#888163] (internal)
* Mon Jul 21 2014 crrodriguez@opensuse.org
- Update package Summary and Description.
- version 2.4.10
  * SECURITY: CVE-2014-0117 (cve.mitre.org)
  * SECURITY: CVE-2014-3523 (cve.mitre.org)
  * SECURITY: CVE-2014-0226 (cve.mitre.org)
  * SECURITY: CVE-2014-0118 (cve.mitre.org)
  * SECURITY: CVE-2014-0231 (cve.mitre.org)
  * Multiple bugfixes to mod_ssl, mod_cache, mod_deflate, mod_lua
  * mod_proxy_fcgi supports unix sockets.
* Mon Jul 21 2014 mc@suse.com
- provide httpd.service as alias for apache2.service for
  compatibility reasons (bnc#888093)
* Mon Apr 14 2014 lnussel@suse.de
- move most ssl options to ssl-global.conf. There is usually no need
  for every vhost to re-define the ciphers for example (bnc#865582).
  Drop some commented entries that only lead to confusion.
* Thu Mar 27 2014 crrodriguez@opensuse.org
- version 2.4.9
  * SECURITY: CVE-2014-0098
  * SECURITY: CVE-2013-6438
  * multiple bugfixes and improvements to mod_ssl, mod_lua,
  mod_session and core, see CHANGES for details.
* Mon Feb 17 2014 draht@suse.de
- /etc/sysconfig/apache2: add socache_shmcb to the list of modules
  that are enabled.
  /etc/apache2/ssl-global.conf: make SSLSessionCache shmcb...
  conditional on IfModule socache_shmcb.
  The same applies to SSLSessionCache dmb:* via module socache_dbm
  in commented section of same file. [bnc#864185]
- /etc/sysconfig/apache2: remove reference to non-existing script
  /usr/share/doc/packages/apache2/certificate.sh, which was only a
  wrapper to mkcert.sh anyways. [bnc#864185]
* Mon Nov 25 2013 crrodriguez@opensuse.org
- update to apache 2.4.7, important changes:
  * This release requires both apr and apr-util 1.5.x series
  and therefore will no longer build in older released products
  * mod_ssl: Improve handling of ephemeral DH and ECDH keys
  (obsoletes httpd-mod_ssl_ephemeralkeyhandling.patch)
  * event MPM: Fix possible crashes
  * mod_deflate: Improve error detection
  * core: Add open_htaccess hook  in conjunction with dirwalk_stat.
  * mod_rewrite: Make rewrite websocket-aware to allow proxying.
  * mod_ssl: drop support for export-grade ciphers with ephemeral RSA
  keys, and unconditionally disable aNULL, eNULL and EXP ciphers
  (not overridable via SSLCipherSuite)
  * core, mod_ssl: Enable the ability for a module to reverse the
  sense of a poll event from a read to a write or vice versa
  (obsoletes httpd-event-ssl.patch)
  * see CHANGES for more details
* Sat Nov 16 2013 crrodriguez@opensuse.org
- httpd-mod_ssl_ephemeralkeyhandling.patch obsoletes
  mod_ssl-2.4.x-ekh.diff this new patch is the final
  form of the rework, merged for 2.4.7.
* Mon Nov  4 2013 freek@opensuse.org
- Removed obsolete directive DefaultType
- Changed all access control to new Require directive
* Fri Oct 25 2013 crrodriguez@opensuse.org
- reenable mod_ssl-2.4.x-ekh.diff
* Tue Oct 22 2013 crrodriguez@opensuse.org
- Correct build in old distros.
* Tue Oct 22 2013 crrodriguez@opensuse.org
- disable (revert) mod_ssl changes in the previous
  commit so it does not end in factory or 13.1 yet.
* Tue Oct 22 2013 crrodriguez@opensuse.org
- make mod_systemd static so scenarios described in
  [bnc#846897] do not happen again.
* Mon Oct 21 2013 crrodriguez@opensuse.org
- mod_ssl: improve ephemeral key handling in particular, support DH params
  with more than 1024 bits, and allow custom configuration.
  This patch adjust DH parameters according to the relevant RFC
  recommendations and permanently disables the usage of "export"
  and "NULL" ciphers no matter what the user configuration is
  (mod_ssl-2.4.x-ekh.diff, to be in 2.4.7)
* Mon Oct 21 2013 crrodriguez@opensuse.org
- fix [bnc#846897] problems building kiwi images due to
  systemd not being running in chroot. (submit to 13.1 ASAP)
* Mon Oct 14 2013 aj@suse.com
- Fix SUSE spelling.
* Tue Sep  3 2013 crrodriguez@opensuse.org
- Also fix subtle non-obvious systemd unit confusion
  we really mean -DFOREGROUND not -DNO_DETACH the latter only
  inhibits the parent from forking, not quite the same as
  running in well.. the foreground as required.
* Tue Sep  3 2013 crrodriguez@opensuse.org
- Ensure we only use /run and not /var/run
* Fri Aug 30 2013 crrodriguez@opensuse.org
- Really use %%requires_ge for libapr1 and libapr-util1
  mentioned but not implemented in the previous commit.
* Wed Aug 28 2013 crrodriguez@opensuse.org
- Use %%requires_ge for libapr1 and libapr-util1
- apache2-default-server.conf: Need to use IncludeOptional
- apache-20-22-upgrade: also load authz_core
- httpd-visibility.patch: Use compiler symbol visibility.
* Fri Aug  2 2013 meissner@suse.com
- Make the default keysize in the sample gensslcerts 2048 bits to match
  government recommendations.
* Thu Aug  1 2013 crrodriguez@opensuse.org
- Enable mod_proxy_html, mod_xml2enc and mod_lua (missed BuildRequires)
* Mon Jul 29 2013 crrodriguez@opensuse.org
- provide and obsolete mod_macro
- upgrade: some people complain that log_config module
  is not enabled by default sometimes, fix that.
- upgrade : "SSLMutex" no longer exists.
- Toogle EnableSendfile on because now apache defaults to off
  due to kernel bugs. that's a silly thing to do here
  as kernel bugs have to be fixed at their source, not worked around
  in applications.
* Mon Jul 22 2013 crrodriguez@opensuse.org
- httpd-event-ssl.patch: from upstream
  Lift the restriction that prevents mod_ssl taking
  full advantage of the event MPM.
* Mon Jul 22 2013 crrodriguez@opensuse.org
- Update to version 2.4.6
  * SECURITY: CVE-2013-1896 (cve.mitre.org)
  * SECURITY: CVE-2013-2249 (cve.mitre.org)
  * Major updates to mod_lua
  * Support for proxying websocket requests
  * Higher performant shm-based cache implementation
  * Addition of mod_macro for easier configuration management
  * As well as several exciting fixes, especially those related to RFC edge
  cases in mod_cache and mod_proxy.
- IMPORTANT : With the current packaging scheme, we can no longer
  Include the ITK MPM, therefore it has been disabled. This is because
  this MPM can now only be provided as a loadable module but we do
  not currently build MPMs as shared modules but as independant
  binaries and all helpers/startup scripts depend on that behaviour.
  It will be fixed in the upcoming weeks/months.
* Tue Jun 18 2013 crrodriguez@opensuse.org
- apache-20-22-upgrade: still no cookie, module authn_file
  is ok and must not be disabled on update.
  authn_core must however be enabled too.
* Tue Jun 18 2013 crrodriguez@opensuse.org
- fix apache_mmn spec macro, otherwise all modules down
  the chain will have broken dependencies
* Tue Jun 18 2013 crrodriguez@opensuse.org
- remove After=mysql.service php-fpm.service postgresql.service
  which were added in the previous change, those must be added
  as Before=apache2.service in the respective services.
* Fri Jun 14 2013 crrodriguez@opensuse.org
- Include mod_systemd for more complete integration with
  systemd, turn the service to Typé=notify as required
- Disable SSL NPN patch for now, it is required for mod_spdy
  but mod_spdy does not support apache 2.4
* Sat Jun  1 2013 crrodriguez@opensuse.org
- apache 2.4.4
  * fix for CVE-2012-3499
  * fix for the CRIME attack (disable ssl compression by default)
  * many other bugfies
  * build access_compat amd unixd as static modules and solve
  some other upgrade quirks (bnc#813705)
* Mon Feb 25 2013 mlin@suse.com
- Install apache2.service accordingly (/usr/lib/systemd for 12.3
  and up or /lib/systemd for older versions).
* Sat Jan 26 2013 crrodriguez@opensuse.org
-  Apache 2.4.3
  * SECURITY: CVE-2012-3502
  * SECURITY: CVE-2012-2687
  * mod_cache: Set content type in case we return stale content.
  * lots of bugfixes see http://www.apache.org/dist/httpd/CHANGES_2.4.3
* Sat Jan 26 2013 crrodriguez@opensuse.org
- Improve systemd unit file (tested for months)
* Fri Jan 18 2013 mhrusecky@suse.cz
- use %%set_permissions instead %%run_permissions (bnc#764097)
* Wed Aug  1 2012 crrodriguez@opensuse.org
- Fix factory-auto (aka r2dbag) complains about URL.
- Provide a symlink for apxs2 new location otherwise
  all buggy spec files of external modules will break.
* Wed Aug  1 2012 crrodriguez@opensuse.org
- BuildRequire xz explicitly, fix build in !Factory
- Drop more old, unused patches
* Wed Aug  1 2012 crrodriguez@opensuse.org
- Upgrade to apache 2.4.2
  * * ATTENTION, before installing this update YOU MUST
  READ https://httpd.apache.org/docs/2.4/upgrading.html
  CAREFULLY otherwise your server will most likely
  fail to start due to backward incompatible changes.
  * You can read the huge complete list of changes
  at https://httpd.apache.org/docs/2.4/new_features_2_4.html
* Wed Jul 25 2012 saschpe@suse.de
- gensslcert: Use 0400 permissions for generated SSL certificate files
  instead of 0644
* Fri Jul  6 2012 meissner@suse.com
- modified apache2.2-mpm-itk-20090414-00.patch to fix
  itk running as root. bnc#681176 / CVE-2011-1176
* Fri Jul  6 2012 meissner@suse.com
- remove the insecure LD_LIBRARY_PATH line. bnc#757710
* Sun Apr 22 2012 dimstar@opensuse.org
- Add apache2-mod_ssl_npn.patch: Add npn support to mod_ssl, which
  is needed by spdy.
- Provide apache2(mod_ssl+npn), indicating that our mod_ssl does
  have the npn patch. This can be used by mod_spdy to ensure a
  compatible apache/mod_ssl is installed.
* Tue Mar 20 2012 adrian@suse.de
- fix truncating and resulting paniking of answer headers (bnc#690734)
* Sat Feb 18 2012 poeml@cmdline.net
- update to 2.2.22
  * ) SECURITY: CVE-2011-3368 (cve.mitre.org)
    Reject requests where the request-URI does not match the HTTP
    specification, preventing unexpected expansion of target URLs in
    some reverse proxy configurations.
  * ) SECURITY: CVE-2011-3607 (cve.mitre.org)
    Fix integer overflow in ap_pregsub() which, when the mod_setenvif module
    is enabled, could allow local users to gain privileges via a .htaccess
    file.
  * ) SECURITY: CVE-2011-4317 (cve.mitre.org)
    Resolve additional cases of URL rewriting with ProxyPassMatch or
    RewriteRule, where particular request-URIs could result in undesired
    backend network exposure in some configurations.
  * ) SECURITY: CVE-2012-0021 (cve.mitre.org)
    mod_log_config: Fix segfault (crash) when the '%%{cookiename}C' log format
    string is in use and a client sends a nameless, valueless cookie, causing
    a denial of service. The issue existed since version 2.2.17. PR 52256.
  * ) SECURITY: CVE-2012-0031 (cve.mitre.org)
    Fix scoreboard issue which could allow an unprivileged child process
    could cause the parent to crash at shutdown rather than terminate
    cleanly.
  * ) SECURITY: CVE-2012-0053 (cve.mitre.org)
    Fix an issue in error responses that could expose "httpOnly" cookies
    when no custom ErrorDocument is specified for status code 400.
  * ) mod_proxy_ajp: Try to prevent a single long request from marking a worker
    in error.
  * ) config: Update the default mod_ssl configuration: Disable SSLv2, only
    allow >= 128bit ciphers, add commented example for speed optimized cipher
    list, limit MSIE workaround to MSIE <= 5.
  * ) core: Fix segfault in ap_send_interim_response(). PR 52315.
  * ) mod_log_config: Prevent segfault. PR 50861.
  * ) mod_win32: Invert logic for env var UTF-8 fixing.
    Now we exclude a list of vars which we know for sure they dont hold UTF-8
    chars; all other vars will be fixed. This has the benefit that now also
    all vars from 3rd-party modules will be fixed. PR 13029 / 34985.
  * ) core: Fix hook sorting for Perl modules, a regression introduced in
    2.2.21. PR: 45076.
  * ) Fix a regression introduced by the CVE-2011-3192 byterange fix in 2.2.20:
    A range of '0-' will now return 206 instead of 200. PR 51878.
  * ) Example configuration: Fix entry for MaxRanges (use "unlimited" instead
    of "0").
  * ) mod_substitute: Fix buffer overrun.
- adjusted SSL template/default config for upstream changes, and added
  MaxRanges example to apache2-server-tuning.conf
- fixed installation of (moved) man pages
* Sat Feb 11 2012 coolo@suse.com
- compile with pcre 8.30 - patch taken from apache bugzilla
* Sat Jan 21 2012 draht@suse.de
- enable mod_reqtimeout by default via APACHE_MODULES in
  /etc/sysconfig/apache2, configuration
  /etc/apache2/mod_reqtimeout.conf .
  Of course, the existing configuration remains unchanged.
* Fri Dec 16 2011 chris@computersalat.de
- add default vhost configs
  * default-vhost.conf, default-vhost-ssl.conf, README.default-vhost
* Sat Dec 10 2011 meissner@suse.de
- openldap2 is not necessary, just openldap2-devel as buildrequires
* Fri Dec  2 2011 coolo@suse.com
- add automake as buildrequire to avoid implicit dependency
* Fri Nov 18 2011 draht@suse.de
- update to /etc/init.d/apache2: handle reload with deleted
  binaries after package update more thoughtfully: If the binaries
  have been replaced, then a dlopen(3) on the apache modules is
  prone to fail. => Don't reload then, but complain and fail.
  Especially important for logrotate!
* Fri Oct  7 2011 draht@suse.de
- httpd-2.2.x-CVE-2011-3368-server_protocl_c.diff fixes mod_proxy
  reverse exposure via RewriteRule or ProxyPassMatch directives.
  This is CVE-2011-3368.
* Fri Oct  7 2011 fcrozat@suse.com
- Ensure service_add_pre macro is correctly called for
  openSUSE 12.1 or later.
* Tue Sep 27 2011 fcrozat@suse.com
- Fix systemd files packaging, %%ghost is not a good idea.
- Use systemd rpm macros for openSUSE 12.1 and later.
* Thu Sep 15 2011 draht@suse.de
- don't create $RPM_BUILD_ROOT/etc/init.d twice in %%install.
* Tue Sep 13 2011 draht@suse.de
- Update to 2.2.21. News therein:
  * re-worked CVE-2011-3192 (byterange_filter.c) with a regression
    fix. New config option: MaxRanges (PR 51748)
  * multi fixes in mod_filter, mod_proxy_ajp, mod_dav_fs,
    mod_alias, mod_rewrite. As always, see CHANGES file.
- added httpd-%%{realver}.tar.bz2.asc to source, along with
  60C5442D.key which the tarball was signed with.
* Tue Sep 13 2011 draht@suse.de
- need to add %%ghost /lib/systemd to satisfy distributions that
  have no systemd yet.
* Thu Sep  1 2011 fcrozat@suse.com
- Add apache2-systemd-ask-pass / apache2.service / start_apache2
  and modify apache2-ssl-global.conf for systemd support
  (bnc#697137).
* Wed Aug 31 2011 crrodriguez@opensuse.org
- Update to version 2.2.20, fix CVE-2011-3192
  mod_deflate D.o.S.
* Fri Aug  5 2011 crrodriguez@opensuse.org
-  Fix apache PR 45076
* Sun Jul 17 2011 crrodriguez@opensuse.org
- Use SSL_MODE_RELEASE_BUFFERS to reduce mod_ssl memory usage
* Wed Jun 22 2011 crrodriguez@opensuse.org
- Add 2 patches from the "low hanging fruit" warnings in apache
  STATUS page.
  * mod_deflate: Stop compressing HEAD requests
  if there is not Content-Length header
  * mod_reqtimeout: Disable keep-alive after read timeout
* Fri Jun 10 2011 crrodriguez@opensuse.org
- Remove -fno-strict-aliasing from CFLAGS, no longer needed.
* Wed Jun  8 2011 crrodriguez@opensuse.org
- Allow KeepAliveTimeout to be expressed in miliseconds
  sometimes one second is too long, upstream r733557.
* Mon Jun  6 2011 crrodriguez@opensuse.org
- When linux changes to version 3.x configure tests are gonna break.
  remove version check, assuming kernel 2.2 or later.
* Thu May 26 2011 crrodriguez@opensuse.org
- Update to 2.2.19, only one bugfix.
  * ) Revert ABI breakage in 2.2.18 caused by the function signature change
    of ap_unescape_url_keep2f().  This release restores the signature from
    2.2.17 and prior, and introduces ap_unescape_url_keep2f_ex().
    [Eric Covener]
* Fri May 20 2011 crrodriguez@opensuse.org
- Remove SSLv2 disabled patch, already in upstream.
- Update to version 2.2.18
  * mod_ssl, ab: Support OpenSSL compiled without SSLv2 support.
  * core: Treat timeout reading request as 408 error, not 400.
  * core: Only log a 408 if it is no keepalive timeout.
  * mod_rewrite: Allow to unset environment variables.
  * prefork: Update MPM state in children during a graceful restart.
  * Other fixes in mod_cache,mod_dav,mod_proxy se NEWS for detail.
* Wed Apr 20 2011 crrodriguez@opensuse.org
- Fix regular expression in vhost ssl template IE workaround
  it is obsolete see https://issues.apache.org/bugzilla/show_bug.cgi?id=49484
  You should apply this update to fix painfully slow SSL
  connections when using IE.
* Mon Apr 11 2011 crrodriguez@opensuse.org
- Allow usage of an openSSL library compiled without SSlv2
* Fri Apr  8 2011 lnussel@suse.de
- set sane default cipher string in apache2-vhost-ssl.template
- remove useless example snakeoil certs
- remove broken mkcert script
* Thu Feb 17 2011 werner@suse.de
- Tag boot script as interactive as systemd uses it
* Mon Feb  7 2011 lnussel@suse.de
- recommend the default mpm package (bnc#670027)
* Tue Oct 19 2010 poeml@cmdline.net
- update to 2.2.17:
  SECURITY: CVE-2010-1623 (cve.mitre.org)
    Fix a denial of service attack against apr_brigade_split_line().
    [Actual fix is in the libapr 1.3 line, which we don't use // poeml]
  SECURITY: CVE-2009-3560, CVE-2009-3720 (cve.mitre.org)
    Fix two buffer over-read flaws in the bundled copy of expat which could
    cause applications to crash while parsing specially-crafted XML documents.
    [We build with system expat library // poeml]
  prefork MPM: Run cleanups for final request when process exits gracefully
    to work around a flaw in apr-util.  PR 43857
  core:
  - check symlink ownership if both FollowSymlinks and
    SymlinksIfOwnerMatch are set
  - fix origin checking in SymlinksIfOwnerMatch PR 36783
  - (re)-introduce -T commandline option to suppress documentroot
    check at startup.  PR 41887
  vhost:
  - A purely-numeric Host: header should not be treated as a port.  PR 44979
  rotatelogs:
  - Fix possible buffer overflow if admin configures a
    mongo log file path.
  Proxy balancer: support setting error status according to HTTP response
    code from a backend.  PR 48939.
  mod_authnz_ldap:
  - If AuthLDAPCharsetConfig is set, also convert the
    password to UTF-8. PR 45318.
  mod_dir, mod_negotiation:
  - Pass the output filter information to newly created sub requests; as these
    are later on used as true requests with an internal redirect. This allows
    for mod_cache et.al. to trap the results of the redirect.  PR 17629, 43939
  mod_headers:
  - Enable multi-match-and-replace edit option PR 46594
  mod_log_config:
  - Make ${cookie}C correctly match whole cookie names
    instead of substrings. PR 28037.
  mod_reqtimeout:
  - Do not wrongly enforce timeouts for mod_proxy's backend
    connections and other protocol handlers (like mod_ftp). Enforce the
    timeout for AP_MODE_GETLINE. If there is a timeout, shorten the lingering
    close time from 30 to 2 seconds.
  mod_ssl:
  - Do not do overlapping memcpy. PR 45444
* Tue Oct  5 2010 cristian.rodriguez@opensuse.org
- Add missing libcap-devel to BuildRequires, wanted by "itk" MPM.
* Thu Jul 29 2010 poeml@cmdline.net
- update to 2.2.16:
  SECURITY: CVE-2010-1452 (cve.mitre.org)
    mod_dav, mod_cache: Fix Handling of requests without a path segment.
    PR: 49246
  SECURITY: CVE-2010-2068 (cve.mitre.org)
    mod_proxy_ajp, mod_proxy_http, mod_reqtimeout: Fix timeout detection
    for platforms Windows, Netware and OS2.  PR: 49417.
  core:
  - Filter init functions are now run strictly once per request
    before handler invocation.  The init functions are no longer run
    for connection filters.  PR 49328.
  mod_filter:
  - enable it to act on non-200 responses.  PR 48377
  mod_ldap:
  - LDAP caching was suppressed (and ldap-status handler returns
    title page only) when any mod_ldap directives were used in VirtualHost
    context.
  mod_ssl:
  - Fix segfault at startup if proxy client certs are shared
    across multiple vhosts.  PR 39915.
  mod_proxy_http:
  - Log the port of the remote server in various messages.
    PR 48812.
  apxs:
  - Fix -A and -a options to ignore whitespace in httpd.conf
  mod_dir:
  - add FallbackResource directive, to enable admin to specify
    an action to happen when a URL maps to no file, without resorting
    to ErrorDocument or mod_rewrite.  PR 47184
  mod_rewrite:
  - Allow to set environment variables without explicitely
    giving a value.
- add Requires and BuildRequires on libapr1 >= 1.4.2. In the past, libapr1 >=
  1.0 was sufficient.  But since 2.2.16, a failure to create listen sockets can
  occur, unless newer libapr1 is used. See
  https://bugzilla.redhat.com/show_bug.cgi?id=516331
- remove obsolete httpd-2.2.15-deprecated_use_of_build_in_variable.patch
* Mon May 17 2010 poeml@cmdline.net
- add type and encoding for zipped SVG images (.svgz)
  Thanks to Sebastian Siebert (via Submit Request #40059)
* Tue May 11 2010 lars@linux-schulserver.de
- fix deprecated usage of $[ in apxs2
  (httpd-2.2.15-deprecated_use_of_build_in_variable.patch)
* Fri May  7 2010 aj@suse.de
- Do not compile in build time but use mtime of changes file instead.
  This allows build-compare to identify that no changes have happened.
* Tue Mar 30 2010 poeml@cmdline.net
- add apache2-prefork to the Requires of apache2-devel, because apxs2 will
  build for prefork, if not called as apxs2-worker (which should rarely be the
  case). Also added gcc to the Requires.
* Mon Mar  8 2010 poeml@cmdline.net
- update to 2.2.15:
  SECURITY: CVE-2009-3555 (cve.mitre.org)
    mod_ssl: Comprehensive fix of the TLS renegotiation prefix injection
    attack when compiled against OpenSSL version 0.9.8m or later. Introduces
    the 'SSLInsecureRenegotiation' directive to reopen this vulnerability and
    offer unsafe legacy renegotiation with clients which do not yet support
    the new secure renegotiation protocol, RFC 5746.
  SECURITY: CVE-2009-3555 (cve.mitre.org)
    mod_ssl: A partial fix for the TLS renegotiation prefix injection attack
    by rejecting any client-initiated renegotiations. Forcibly disable
    keepalive for the connection if there is any buffered data readable. Any
    configuration which requires renegotiation for per-directory/location
    access control is still vulnerable, unless using OpenSSL >= 0.9.8l.
  SECURITY: CVE-2010-0408 (cve.mitre.org)
    mod_proxy_ajp: Respond with HTTP_BAD_REQUEST when the body is not sent
    when request headers indicate a request body is incoming; not a case of
    HTTP_INTERNAL_SERVER_ERROR.
  SECURITY: CVE-2010-0425 (cve.mitre.org)
    mod_isapi: Do not unload an isapi .dll module until the request processing
    is completed, avoiding orphaned callback pointers.
  SECURITY: CVE-2010-0434 (cve.mitre.org)
    Ensure each subrequest has a shallow copy of headers_in so that the parent
    request headers are not corrupted.  Elimiates a problematic optimization
    in the case of no request body.  PR 48359
  mod_reqtimeout:
  - New module to set timeouts and minimum data rates for receiving requests
    from the client.
  core:
  - Fix potential memory leaks by making sure to not destroy bucket brigades
    that have been created by earlier filters.
  - Return APR_EOF if request body is shorter than the length announced by the
    client. PR 33098
  - Preserve Port information over internal redirects PR 35999
  - Build: fix --with-module to work as documented PR 43881
  worker:
  - Don't report server has reached MaxClients until it has.  Add message when
    server gets within MinSpareThreads of MaxClients.  PR 46996.
  ab, mod_ssl:
  - Restore compatibility with OpenSSL < 0.9.7g.
  mod_authnz_ldap:
  - Add AuthLDAPBindAuthoritative to allow Authentication to try other
    providers in the case of an LDAP bind failure.  PR 46608
  - Failures to map a username to a DN, or to check a user password now result
    in an informational level log entry instead of warning level.
  mod_cache:
  - Introduce the thundering herd lock, a mechanism to keep the flood of
    requests at bay that strike a backend webserver as a cached entity goes
    stale.
  - correctly consider s-maxage in cacheability decisions.
  mod_disk_cache, mod_mem_cache:
  - don't cache incomplete responses, per RFC 2616, 13.8.  PR15866.
  mod_charset_lite:
  - Honor 'CharsetOptions NoImplicitAdd'.
  mod_filter:
  - fix FilterProvider matching where "dispatch" string doesn't exist.  PR 48054
  mod_include:
  - Allow fine control over the removal of Last-Modified and ETag headers
    within the INCLUDES filter, making it possible to cache responses if
    desired. Fix the default value of the SSIAccessEnable directive.
  mod_ldap:
  - If LDAPSharedCacheSize is too small, try harder to purge some cache
    entries and log a warning. Also increase the default LDAPSharedCacheSize
    to 500000. This is a more realistic size suitable for the default values
    of 1024 for LdapCacheEntries/LdapOpCacheEntries.  PR 46749.
  mod_log_config:
  - Add the R option to log the handler used within the request.
  mod_mime:
  - Make RemoveType override the info from TypesConfig.  PR 38330.
  - Detect invalid use of MultiviewsMatch inside Location and LocationMatch
    sections.  PR 47754.
  mod_negotiation:
  - Preserve query string over multiviews negotiation.  This buglet was fixed
    for type maps in 2.2.6, but the same issue affected multiviews and was
    overlooked.  PR 33112
  mod_proxy:
  - unable to connect to a backend is SERVICE_UNAVAILABLE, rather than
    BAD_GATEWAY or (especially) NOT_FOUND.  PR 46971
  mod_proxy, mod_proxy_http:
  - Support remote https proxies by using HTTP CONNECT.  PR 19188.
  mod_proxy_http:
  - Make sure that when an ErrorDocument is served from a reverse proxied URL,
    that the subrequest respects the status of the original request. This
    brings the behaviour of proxy_handler in line with default_handler. PR
    47106.
  mod_proxy_ajp:
  - Really regard the operation a success, when the client aborted the
    connection. In addition adjust the log message if the client aborted the
    connection.
  mod_rewrite:
  - Make sure that a hostname:port isn't fully qualified if the request is a
    CONNECT request. PR 47928
  - Add scgi scheme detection.
  mod_ssl:
  - Fix a potential I/O hang if a long list of trusted CAs is configured for
    client cert auth. PR 46952.
  - When extracting certificate subject/issuer names to the SSL_*_DN_*
    variables, handle RDNs with duplicate tags by exporting multiple
    varialables with an "_n" integer suffix.  PR 45875.
- obsolete patch CVE-2009-3555-2.2.patch removed
* Fri Mar  5 2010 coolo@novell.com
- readd whitespace removed by autobuild
* Wed Dec 16 2009 jengelh@medozas.de
- package documentation as noarch
* Sat Nov  7 2009 poeml@cmdline.net
- add patch for CVE-2009-3555 (cve.mitre.org)
  http://www.apache.org/dist/httpd/patches/apply_to_2.2.14/CVE-2009-3555-2.2.patch
  http://mail-archives.apache.org/mod_mbox/httpd-announce/200911.mbox/%%3c20091107013220.31376.qmail@minotaur.apache.org%%3e
  A partial fix for the TLS renegotiation prefix injection attack by rejecting
  any client-initiated renegotiations. Any configuration which requires
  renegotiation for per-directory/location access control is still vulnerable,
  unless using OpenSSL >= 0.9.8l.
* Mon Oct 26 2009 poeml@cmdline.net
- update to 2.2.14:
  * ) SECURITY: CVE-2009-2699 (cve.mitre.org)
    Fixed in APR 1.3.9.  Faulty error handling in the Solaris pollset support
    (Event Port backend) which could trigger hangs in the prefork and event
    MPMs on that platform.  PR 47645.  [Jeff Trawick]
  * ) SECURITY: CVE-2009-3095 (cve.mitre.org)
    mod_proxy_ftp: sanity check authn credentials.
    [Stefan Fritsch <sf fritsch.de>, Joe Orton]
  * ) SECURITY: CVE-2009-3094 (cve.mitre.org)
    mod_proxy_ftp: NULL pointer dereference on error paths.
    [Stefan Fritsch <sf fritsch.de>, Joe Orton]
  * ) mod_proxy_scgi: Backport from trunk. [André Malo]
  * ) mod_ldap: Don't try to resolve file-based user ids to a DN when AuthLDAPURL
    has been defined at a very high level.  PR 45946.  [Eric Covener]
  * ) htcacheclean: 19 ways to fail, 1 error message. Fixed. [Graham Leggett]
  * ) mod_ldap: Bring the LDAPCacheEntries and LDAPOpCacheEntries
    usage() in synch with the manual and the implementation (0 and -1
    both disable the cache). [Eric Covener]
  * ) mod_ssl: The error message when SSLCertificateFile is missing should
    at least give the name or position of the problematic virtual host
    definition. [Stefan Fritsch sf sfritsch.de]
  * ) htdbm: Fix possible buffer overflow if dbm database has very
    long values.  PR 30586 [Dan Poirier]
  * ) Add support for HTTP PUT to ab. [Jeff Barnes <jbarnesweb yahoo.com>]
  * ) mod_ssl: Fix SSL_*_DN_UID variables to use the 'userID' attribute
    type.  PR 45107.  [Michael Ströder <michael stroeder.com>,
    Peter Sylvester <peter.sylvester edelweb.fr>]
  * ) mod_cache: Add CacheIgnoreURLSessionIdentifiers directive to ignore
    defined session identifiers encoded in the URL when caching.
    [Ruediger Pluem]
  * ) mod_mem_cache: fix seg fault under load due to pool concurrency problem
    PR: 47672 [Dan Poirier <poirier pobox.com>]
  * ) mod_autoindex: Correctly create an empty cell if the description
    for a file is missing. PR 47682 [Peter Poeml <poeml suse.de>]
* Mon Aug 10 2009 poeml@suse.de
- update to 2.2.13:
  * ) SECURITY: CVE-2009-2412 (cve.mitre.org)
    Distributed with APR 1.3.8 and APR-util 1.3.9 to fix potential overflow
    in pools and rmm, where size alignment was taking place.
  * ) mod_ssl, ab: improve compatibility with OpenSSL 1.0.0 betas.  Report
    warnings compiling mod_ssl against OpenSSL to the httpd developers.
  * ) mod_cgid: Do not add an empty argument when calling the CGI script.
    PR 46380
  * ) Fix potential segfaults with use of the legacy ap_rputs() etc
    interfaces, in cases where an output filter fails.  PR 36780.
* Mon Jul 27 2009 poeml@suse.de
- update to 2.2.12:
  SECURITY: CVE-2009-1891 (cve.mitre.org)
    Fix a potential Denial-of-Service attack against mod_deflate or other
    modules, by forcing the server to consume CPU time in compressing a
    large file after a client disconnects.  PR 39605.
  SECURITY: CVE-2009-1195 (cve.mitre.org)
    Prevent the "Includes" Option from being enabled in an .htaccess
    file if the AllowOverride restrictions do not permit it.
  SECURITY: CVE-2009-1890 (cve.mitre.org)
    Fix a potential Denial-of-Service attack against mod_proxy in a
    reverse proxy configuration, where a remote attacker can force a
    proxy process to consume CPU time indefinitely.
  SECURITY: CVE-2009-1191 (cve.mitre.org)
    mod_proxy_ajp: Avoid delivering content from a previous request which
    failed to send a request body. PR 46949
  SECURITY: CVE-2009-0023, CVE-2009-1955, CVE-2009-1956 (cve.mitre.org)
    The bundled copy of the APR-util library has been updated, fixing three
    different security issues which may affect particular configurations
    and third-party modules.
  core:
  - New piped log syntax: Use "||process args" to launch the given process
    without invoking the shell/command interpreter.  Use "|$command line"
    (the default behavior of "|command line" in 2.2) to invoke using shell,
    consuming an additional shell process for the lifetime of the logging
    pipe program but granting additional process invocation flexibility.
  - prefork: Fix child process hang during graceful restart/stop in
    configurations with multiple listening sockets.  PR 42829.
  - Translate the status line to ASCII on EBCDIC platforms in
    ap_send_interim_response() and for locally generated "100
    Continue" responses.
  - CGI: return 504 (Gateway timeout) rather than 500 when a
    script times out before returning status line/headers. PR 42190
  - prefork: Log an error instead of segfaulting when child startup fails
    due to pollset creation failures.  PR 46467.
  - core/utils: Enhance ap_escape_html API to support escaping non-ASCII chars
  - Set Listen protocol to "https" if port is set to 443 and no proto is specified
    (as documented but not implemented). PR 46066
  - Output -M and -S dumps (modules and vhosts) to stdout instead of stderr.
    PR 42571 and PR 44266 (dup).
  mod_alias:
  - check sanity in Redirect arguments.  PR 44729
  - Ensure Redirect emits HTTP-compliant URLs.  PR 44020
  mod_authnz_ldap:
  - Reduce number of initialization debug messages and make
    information more clear. PR 46342
  mod_cache:
  - Introduce 'no-cache' per-request environment variable to
    prevent the saving of an otherwise cacheable response.
  - Correctly save Content-Encoding of cachable entity. PR 46401
  - When an explicit Expires or Cache-Control header is set, cache
    normally non-cacheable response statuses. PR 46346.
  mod_cgid:
  - fix segfault problem on solaris.  PR 39332
  mod_disk_cache:
  - The module now turns off sendfile support if 'EnableSendfile
    off' is defined globally. PR 41218.
  mod_disk_cache/mod_mem_cache:
  - Fix handling of CacheIgnoreHeaders directive to correctly
    remove headers before storing them.
  mod_deflate:
  - revert changes in 2.2.8 that caused an invalid etag to be
    emitted for on-the-fly gzip content-encoding.  PR 39727 will
    require larger fixes and this fix was far more harmful than
    the original code. PR 45023.
  mod_ext_filter:
  - fix error handling when the filter prog fails to start, and
    introduce an onfail configuration option to abort the request
    or to remove the broken filter and continue.  PR 41120
  mod_include:
  - fix potential segfault when handling back references on an
    empty SSI variable.
  - Prevent a case of SSI timefmt-smashing with filter chains
    including multiple INCLUDES filters. PR 39369
  - support generating non-ASCII characters as entities in SSI PR
    25202
  mod_ldap:
  - Avoid a segfault when result->rc is checked in
    uldap_connection_init when result is NULL. This could happen
    if LDAP initialization failed.  PR 45994.
  mod_negotiation:
  - Escape pathes of filenames in 406 responses to avoid HTML
    injections and HTTP response splitting.  PR 46837.
  mod_proxy:
  - Complete ProxyPassReverse to handle balancer URL's.  Given;
    BalancerMember balancer://alias http://example.com/foo
    ProxyPassReverse /bash balancer://alias/bar backend url
    http://example.com/foo/bar/that is now translated /bash/that
  mod_proxy_ajp:
  - Check more strictly that the backend follows the AJP protocol.
  - Forward remote port information by default.
  mod_proxy_http:
  - fix Host: header for literal IPv6 addresses.  PR 47177
  - fix case sensitivity checking transfer encoding PR 47383
  mod_rewrite:
  - Remove locking for writing to the rewritelog.  PR 46942
  - Fix the error string returned by RewriteRule.  RewriteRule
    returned "RewriteCond: bad flag delimiters" when the 3rd
    argument of RewriteRule was not started with "[" or not ended
    with "]".  PR 45082
  - When evaluating a proxy rule in directory context, do escape
    the filename by default. PR 46428
  - Introduce DiscardPathInfo|DPI flag to stop the troublesome way
    that per-directory rewrites append the previous notion of
    PATH_INFO to each substitution before evaluating subsequent
    rules.  PR38642
  - fix "B" flag breakage by reverting r589343 PR 45529
  mod_ssl:
  - Add server name indication support (RFC 4366) and better
    support for name based virtual hosts with SSL. PR 34607
  - Add SSLProxyCheckPeerExpire and SSLProxyCheckPeerCN directives
    to enable stricter checking of remote server certificates.
  - Add SSLRenegBufferSize directive to allow changing the size of
    the buffer used for the request-body where necessary during a
    per-dir renegotiation.  PR 39243.
  mod_substitute:
  - Fix a memory leak. PR 44948
* Tue Jul 14 2009 hvogel@suse.de
- Fix missing -Y option in gensslcert [bnc#416888]
* Tue Jun  9 2009 poeml@suse.de
- merge changes from openSUSE:Factory:
  - trailing spaces removed from robots.txt
  - moved Snakeoil certificates to separate subpackage
    example-certificates [bnc#419601]
  - removed outdated ca-bundle.crt
  - NOT merging the change from [bnc#301380] (setting TraceEnable
    Off), since there is no reason to deviate from upstream
* Tue Jun  9 2009 poeml@suse.de
- avoid useless (and potentially irritating) messages from usermod
  called in %%post when updating the package - this should probably
  only be run when updating from very old installs anyway.
- likewise, avoid similar useless messages about creation of the
  httpd user when installing on Fedora.
* Tue May  5 2009 poeml@suse.de
- fix hyperref to the quickstart howto in the installed httpd.conf
  [bnc#500938] Thanks, Frank!
* Mon Apr 27 2009 poeml@suse.de
- add ITK MPM (apache2.2-mpm-itk-20090414-00.patch)
  see http://mpm-itk.sesse.net/
* Mon Apr 27 2009 poeml@suse.de
- buildfix (from Factory): replace "shadow" by "pwdutils" in requires
* Thu Mar 12 2009 crrodriguez@suse.de
- update apache2-vhost.template mod_php4 references [bnc#444205]
* Mon Mar  9 2009 poeml@suse.de
- fixed the ed script which turns apxs into
  apxs-{prefork,worker,event) to work on Fedora, by using - instead
  of ^ to go "up" one line. Thereby fixing Fedora build. (Package
  probably needs further tuning to fit into a Fedora environment.)
* Fri Dec 19 2008 poeml@suse.de
- update to 2.2.11:
  core:
  - Worker MPM: Crosscheck that idle workers are still available
    before using them and thus preventing an overflow of the
    worker queue which causes a SegFault. PR 45605
  - Add ap_timeout_parameter_parse to public API.
  - When the ap_http_header_filter processes an error bucket,
    cleanup the passed brigade before returning AP_FILTER_ERROR
    down the filter chain. This unambiguously ensures the same
    error bucket isn't revisited
  - Error responses set by filters were being coerced into 500
    errors, sometimes appended to the original error response. Log
    entry of: 'Handler for (null) returned invalid result code -3'
  - configure: Don't reject libtool 2.x PR 44817
  - Build: Correctly set SSL_LIBS during openssl detection if
    pkgconfig is not available. PR 46018
  mod_autoindex:
  - add configuration option to insert string in HTML HEAD (IndexHeadInsert).
  mod_cache:
  - Convert age of cached object to seconds before comparing it to
    age supplied by the request when checking whether to send a
    Warning header for a stale response. PR 39713.
  mod_expires:
  - Do not sets negative max-age / Expires header in the past.  PR 39774
  mod_info:
  - Was displaying the wrong value for the KeepAliveTimeout value.
  mod_log_config:
  - Add new LogFormat parameter, %%k, which logs the number of
    keepalive requests on this connection for this request.  PR 45762
  mod_proxy:
  - Add the possibility to set the worker parameters
    connectiontimeout and ping in milliseconds.
  - Prevent segmentation faults by correctly adjusting the
    lifetime of the buckets read from the proxy backend. PR 45792
  mod_proxy_ajp:
  - Do not fail if response data is sent before all request
    data is read. PR 45911
  - Fix wrongly formatted requests where client sets
    Content-Length header, but doesn't provide a body.  Servlet
    container always expects that next packet is body whenever C-L
    is present in the headers. This can lead to wrong
    interpretation of the packets. In this case send the empty
    body packet, so container can deal with that.
  mod_proxy_balancer:
  - Add in forced recovery for balancer members if
    all are in error state.
  mod_rewrite:
  - Export and install the mod_rewrite.h header to ensure the optional
    rewrite_mapfunc_t and ap_register_rewrite_mapfunc functions are
    available to third party modules.
* Wed Dec 17 2008 poeml@suse.de
- remove mod_authn_dbm and mod_suexec from the default module list.
* Fri Nov 21 2008 skh@suse.de
- apache2-server-tuning.conf:
  Enclose module-specific configuration in IfModule tags [bnc#440584]
* Fri Nov 14 2008 poeml@suse.de
- apply Dirks fix for [bnc#444878], making the packaging of per-mpm
  modules more deterministic. They'll reliably put into the
  subpackage or main package now, which varied in a ping-pong way
  from build to build in the past.
* Wed Oct 29 2008 poeml@suse.de
- update year of copyright in rc.apache2
* Tue Oct 28 2008 poeml@suse.de
- update to 2.2.10:
  SECURITY: CVE-2008-2939 (cve.mitre.org)
    mod_proxy_ftp: Prevent XSS attacks when using wildcards in the path of
    the FTP URL. Discovered by Marc Bevand of Rapid7.
  core:
  - Support chroot on Unix-family platforms. PR 43596
  mod_authn_alias:
  - Detect during startup when AuthDigestProvider is configured to
    use an incompatible provider via AuthnProviderAlias.  PR 45196
  mod_cgid:
  - Pass along empty command line arguments from an ISINDEX query
    that has consecutive '+' characters in the QUERY_STRING,
    matching the behavior of mod_cgi.
  mod_charset_lite:
  - Avoid dropping error responses by handling meta buckets
    correctly. PR 45687
  mod_dav_fs:
  - Retrieve minimal system information about directory entries
    when walking a DAV fs, resolving a performance degradation on
    Windows.  PR 45464.
  mod_headers:
  - Prevent Header edit from processing only the first header of
    possibly multiple headers with the same name and deleting the
    remaining ones. PR 45333.
  mod_proxy:
  - Allow for smax to be 0 for balancer members so that all idle
    connections are able to be dropped should they exceed ttl. PR 43371
  - Add 'scolonpathdelim' parameter to allow for ';' to also be
    used as a session path separator/delim  PR 45158.
  - Add connectiontimeout parameter for proxy workers in order to
    be able to set the timeout for connecting to the backend separately.
    PR 45445.
  mod_proxy_http:
  - Don't trigger a retry by the client if a failure to
    read the response line was the result of a timeout.
  - Introduce environment variable proxy-initial-not-pooled to
    avoid reusing pooled connections if the client connection is an initial
    connection. PR 37770.
  - Do not forward requests with 'Expect: 100-continue' to
    known HTTP/1.0 servers. Return 'Expectation failed' (417) instead.
  mod_proxy_balancer:
  - Move nonce field in the balancer manager page inside
    the html form where it belongs. PR 45578.
  - Add 'bybusyness' load balance method.
  mod_rewrite:
  - Allow Cookie option to set secure and HttpOnly flags.  PR 44799
  - Preserve the query string when [proxy,noescape]. PR 45247.
  mod_ssl:
  - implement dynamic mutex callbacks for the benefit of OpenSSL.
  - Rewrite shmcb to avoid memory alignment issues.  PR 42101.
- drop obsolete patch httpd-2.2.x-CVE-2008-2939.patch
* Fri Oct 24 2008 skh@suse.de
- apache2.firewall, apache2.ssl-firewall
  Use unique name tags "HTTP Server" and "HTTPS Server" in for
  SuSEFirewall2 configuration [bnc#414962]
* Fri Sep 19 2008 skh@suse.de
- add httpd-2.x.x-logresolve.patch again [bnc#210904]
- add httpd-2.2.x-CVE-2008-2939.patch [bnc#415061]:
  mod_proxy_ftp: Prevent XSS attacks when using wildcards in
  the path of the FTP URL. Discovered by Marc Bevand of Rapid7.
  [Ruediger Pluem]
* Tue Aug 26 2008 poeml@suse.de
- drop rc.config handling (was removed in or after SuSE Linux 8.0)
- don't use fillup_insserv options which have been removed lately
* Fri Aug 15 2008 poeml@suse.de
- fix init script LSB headers
* Wed Jun 25 2008 poeml@suse.de
- add note to /etc/sysconfig/apache2 and /etc/init.d/apache2 about
  how to set ulimits when starting the server
- undocument APACHE_BUFFERED_LOGS and APACHE_TIMEOUT in the
  sysconfig template. They still work but I think it is good to
  keep this stuff out of the beginner's config, first because both
  features are sophisticated enough to not being tweaked in most
  cases, second because it only confuses people I guess, and makes
  the sysconfig file larger than necessary.
* Sun Jun 15 2008 poeml@suse.de
- update to 2.2.9:
  SECURITY: CVE-2008-2364 (cve.mitre.org)
    mod_proxy_http: Better handling of excessive interim responses
    from origin server to prevent potential denial of service and
    high memory usage. Reported by Ryujiro Shibuya.
  SECURITY: CVE-2007-6420 (cve.mitre.org)
    mod_proxy_balancer: Prevent CSRF attacks against the
    balancer-manager interface.
  - htpasswd: Fix salt generation weakness. PR 31440
  worker/event MPM:
  - Fix race condition in pool recycling that leads to
    segmentation faults under load.  PR 44402
  core:
  - Fix address-in-use startup failure on some platforms caused by
    creating an IPv4 listener which overlaps with an existing IPv6
    listener.
  - Add the filename of the configuration file to the warning
    message about the useless use of AllowOverride. PR 39992.
  - Do not allow Options ALL if not all options are allowed to be
    overwritten. PR 44262
  - reinstate location walk to fix config for subrequests PR 41960
  - Fix garbled TRACE response on EBCDIC platforms.
  - gen_test_char: add double-quote to the list of
    T_HTTP_TOKEN_STOP.  PR 9727
  http_filters:
  - Don't return 100-continue on redirects. PR 43711
  - Don't return 100-continue on client error PR 43711
  - Don't spin if get an error when reading the next chunk. PR 44381
  - Don't add bogus duplicate Content-Language entries
  suexec:
  - When group is given as a numeric gid, validate it by looking up
    the actual group name such that the name can be used in log entries.
    PR 7862
  mod_authn_dbd:
  - Disambiguate and tidy database authentication error messages.  PR 43210.
  mod_cache:
  - Handle If-Range correctly if the cached resource was stale.  PR 44579
  - Revalidate cache entities which have Cache-Control: no-cache
    set in their response headers. PR 44511
  mod_cgid:
  - Explicitly set permissions of the socket (ScriptSock) shared
    by mod_cgid and request processing threads, for OS'es such as
    HPUX and AIX that do not use umask for AF_UNIX socket permissions.
  - Don't try to restart the daemon if it fails to initialize the socket.
  mod_charset_lite:
  - Add TranslateAllMimeTypes sub-option to CharsetOptions,
    allowing the administrator to skip the mimetype checking that
    precedes translation.
  mod_dav:
  - Return "method not allowed" if the destination URI of a WebDAV
    copy / move operation is no DAV resource. PR 44734
  mod_headers:
  - Add 'merge' option to avoid duplicate values within the same header.
  mod_include:
  - Correctly handle SSI directives split over multiple filter
  mod_log_config:
  - Add format options for %%p so that the actual local or remote
    port can be logged.  PR 43415.
  mod_logio:
  - Provide optional function to allow modules to adjust the
    bytes_in count
  mod_proxy:
  - Make all proxy modules nocanon aware and do not add the
    query string again in this case. PR 44803.
  - scoreboard: Remove unused proxy load balancer elements from scoreboard
    image (not scoreboard memory itself).
  - Support environment variable interpolation in reverse
    proxying directives.
  - Do not try a direct connection if the connection via a
    remote proxy failed before and the request has a request body.
  - ProxyPassReverse is now balancer aware.
  - Lower memory consumption for short lived connections.
    PR 44026.
  - Keep connections to the backend persistent in the HTTPS case.
  mod_proxy_ajp:
  - Do not retry request in the case that we either failed to
    sent a part of the request body or if the request is not idempotent.
    PR 44334
  mod_proxy_ftp:
  - Fix base for directory listings.  PR 27834
  mod_proxy_http:
  - Fix processing of chunked responses if Connection:
    Transfer-Encoding is set in the response of the proxied
    system. PR 44311
  - Return HTTP status codes instead of apr_status_t values for
    errors encountered while forwarding the request body PR 44165
  mod_rewrite:
  - Initialize hash needed by ap_register_rewrite_mapfunc early
    enough. PR 44641
  - Check all files used by DBM maps for freshness, mod_rewrite
    didn't pick up on updated sdbm maps due to this.  PR41190
  - Don't canonicalise URLs with [P,NE] PR 43319
  mod_speling:
  - remove regression from 1.3/2.0 behavior and drop dependency
    between mod_speling and AcceptPathInfo.
  mod_ssl:
  - Fix a memory leak with connections that have zlib compression
    turned on. PR 44975
  mod_substitute:
  - The default is now flattening the buckets after each
    substitution. The newly added 'q' flag allows for the quicker,
    more efficient bucket-splitting if the user so
  mod_unique_id:
  - Fix timestamp value in UNIQUE_ID.  PR 37064
  ab (apache benchmark):
  - Include <limits.h> earlier if available since we may need
    INT_MAX (defined there on Windows) for the definition of MAX_REQUESTS.
  - Improve client performance by clearing connection pool instead
  - Don't stop sending a request if EAGAIN is returned, which
    will only happen if both the write and subsequent wait are
    returning EAGAIN, and count posted bytes correctly when the initial
    write of a request is not complete. PR 10038, 38861, 39679
  - Overhaul stats collection and reporting to avoid integer
    truncation and time divisions within the test loop, retain
    native time resolution until output, remove unused data,
    consistently round milliseconds, and generally avoid losing
    accuracy of calculation due to type casts. PR 44878, 44931.
  - Add -r option to continue after socket receive errors.
  - Do not try to read non existing response bodies of HEAD requests.
  - Use a 64 bit unsigned int instead of a signed long to count the
  rotatelogs:
  - Log the current file size and error code/description when
    failing to write to the log file.
  - Added '-f' option to force rotatelogs to create the logfile as
    soon as started, and not wait until it reads the first entry.
  - Don't leak memory when reopening the logfile.  PR 40183
  - Improve atomicity when using -l and cleaup code.  PR 44004
- drop obsolete patches httpd-2.1.3alpha-autoconf-2.59.dif
    httpd-2.2.x-CVE-2008-1678.patch
- don't run autoreconf on SLES9
- remove the addition of -g to the CFLAGS, since the build service
  handles debuginfo packages now
* Mon Jun  9 2008 poeml@suse.de
- build service supports the debuginfo flag in metadata now; remove
  debug_package macro from the specfile therefore.
* Mon May 26 2008 skh@suse.de
- CVE-2008-1678: modules/ssl/mod_ssl.c (ssl_cleanup_pre_config):
  Remove the call to CRYPTO_cleanup_all_ex_data here, fixing a
  per-connection memory leak which occurs if the client indicates
  support for a compression algorithm in the initial handshake, and
  mod_ssl is linked against OpenSSL >= 0.9.8f.  [bnc#392096]
  httpd-2.2.x-CVE-2008-1678.patch
* Wed May 14 2008 poeml@suse.de
- fix build on Mandriva 2007, by escaping commented %%build macro
- make filelist of man pages independant of the compression method
  (gz, bz2, lzma)
* Fri Apr 18 2008 poeml@suse.de
- fix from Factory:
  - remove dir /usr/share/omc/svcinfo.d as it is provided now
    by filesystem
- remove obsolete httpd-2.2.x.doublefree.patch file, which isn't
  used since quite some time since the issue is resolved.
* Thu Apr 17 2008 poeml@suse.de
- new implementation of sysconf_addword, using sed instead of ed.
  Moving it from the -utils subpackage into the parent package,
  where it's actually needed. If sysconf_addword is already present
  in the system, it is preferred (by PATH). That's because the tool
  has been integrated into aaa_base.rpm with openSUSE 11.0.
  Removing the requires on the ed package. [bnc#377131]
* Wed Mar 12 2008 poeml@suse.de
- require ed package, since ed is needed by sysconf_addword, which
  in turn is used by a2enmod/a2enflag
* Fri Feb 29 2008 poeml@suse.de
- better documentation how to enable SSL in /etc/sysconfig/apache2
- quickstart readme: the link to the openSUSE wiki is about to move
* Tue Feb 19 2008 poeml@suse.de
- add "127.0.0.1" to the local access list in mod_status.conf,
  because on some systems "localhost" seems to resolve only to IPv6
  localhost
* Sat Feb  2 2008 crrodriguez@suse.de
- upstream 2.2.8
  SECURITY: CVE-2007-6421 (cve.mitre.org)
    mod_proxy_balancer: Correctly escape the worker route and the worker
    redirect string in the HTML output of the balancer manager.
    Reported by SecurityReason.
  SECURITY: CVE-2007-6422 (cve.mitre.org)
    Prevent crash in balancer manager if invalid balancer name is passed
    as parameter. Reported by SecurityReason.
  SECURITY: CVE-2007-6388 (cve.mitre.org)
    mod_status: Ensure refresh parameter is numeric to prevent
    a possible XSS attack caused by redirecting to other URLs.
    Reported by SecurityReason.
  SECURITY: CVE-2007-5000 (cve.mitre.org)
    mod_imagemap: Fix a cross-site scripting issue.  Reported by JPCERT.
  SECURITY: CVE-2008-0005 (cve.mitre.org)
    Introduce the ProxyFtpDirCharset directive, allowing the administrator
    to identify a default, or specific servers or paths which list their
    contents in other-than ISO-8859-1 charset (e.g. utf-8).
  mod_autoindex:
  - Generate valid XHTML output by adding the xhtml namespace. PR 43649
  mod_charset_lite:
  - Don't crash when the request has no associated filename.
  mod_dav:
  - Fix evaluation of If-Match * and If-None-Match * conditionals.  PR 38034
  - Adjust etag generation to produce identical results on 32-bit
    and 64-bit platforms and avoid a regression with conditional PUT's on lock
    and etag. PR 44152.
  mod_deflate:
  - initialise inflate-out filter correctly when the first brigade
    contains no data buckets.  PR 43512
  mod_disk_cache:
  - Delete temporary files if they cannot be renamed to their final
    name.
  mod_filter:
  - Don't segfault on (unsupported) chained FilterProvider usage.  PR 43956
  mod_include:
  - Add an "if" directive syntax to test whether an URL is
    accessible, and if so, conditionally display content. This
    allows a webmaster to hide a link to a private page when the
    user has no access to that page.
  mod_ldap:
  - Try to establish a new backend LDAP connection when the
    Microsoft LDAP client library returns LDAP_UNAVAILABLE, e.g.
    after the LDAP server has closed the connection due to a
    timeout.  PR 39095
  - Give callers a reference to data copied into the request pool
    instead of references directly into the cache PR 43786
  - Stop passing a reference to pconf around for (limited) use
    during request processing, avoiding possible memory corruption
    and crashes.
  mod_proxy:
  - Canonicalisation improvements. Add "nocanon" keyword to
    ProxyPass, to suppress URI-canonicalisation in a reverse proxy. Also,
    don't escape/unescape forward-proxied URLs.  PR 41798, 42592
  - Don't by default violate RFC2616 by setting Max-Forwards when
    the client didn't send it to us.  Leave that as a
    configuration option.  PR 16137
  - Fix persistent backend connections.  PR 43472
  - escape error-notes correctly PR 40952
  - check ProxyBlock for all blocked addresses PR 36987
  - Don't lose bytes when a response line arrives in small chunks.
    PR 40894
  mod_proxy_ajp:
  - Use 64K as maximum AJP packet size. This is the maximum length
    we can squeeze inside the AJP message packet.
  - Ignore any ajp13 flush packets received before we send the
    response headers. See Tomcat PR 43478.
  - Differentiate within AJP between GET and HEAD requests. PR 43060
  mod_proxy_balancer:
  - Do not reset lbstatus, lbfactor and lbset when starting a new
    child.  PR 39907
  mod_proxy_http:
  - Remove Warning headers with wrong date PR 16138
  - Correctly parse all Connection headers in proxy.  PR 43509
  - add Via header correctly (if enabled) to response, even where
    other Via headers exist.  PR 19439
  - Correctly forward unexpected interim (HTTP 1xx) responses from
    the backend according to RFC2616.  But make it configurable in
    case something breaks on it.  PR 16518
  - strip hop-by-hop response headers PR 43455
  - Propagate Proxy-Authorization header correctly.  PR 25947
  - Don't segfault on bad line in FTP listing PR 40733
  mod_rewrite:
  - Add option to suppress URL unescaping PR 34602
  - Add the novary flag to RewriteCond.
  mod_substitute:
  - Added a new output filter, which performs inline response
    content pattern matching (including regex) and substitution.
  mod_ssl:
  - Fix handling of the buffered request body during a per-location
    renegotiation, when an internal redirect occurs.  PR 43738.
  - Fix SSL client certificate extensions parsing bug. PR 44073.
  - Prevent memory corruption of version string.  PR 43865, 43334
  mod_status:
  - Add SeeRequestTail directive, which determines if
    ExtendedStatus displays the 1st 63 characters of the request
    or the last 63. Useful for those requests with large string
    lengths and which only vary with the last several characters.
  event MPM:
  - Add support for running under mod_ssl, by reverting to the
    Worker MPM behaviors, when run under an input filter that buffers
    its own data.
  core:
  - Fix regression in 2.2.7 in chunk filtering with massively
    chunked requests.
  - Lower memory consumption of ap_r* functions by reusing the
    brigade instead of recreating it during each filter pass.
  - Lower memory consumption in case that flush buckets are passed
    thru the chunk filter as last bucket of a brigade. PR 23567.
  - Fix broken chunk filtering that causes all non blocking reads
    to be converted into blocking reads.  PR 19954, 41056.
  - Change etag generation to produce identical results on 32-bit
    and 64-bit platforms.  PR 40064.
  - Handle unrecognised transfer-encodings.  PR 43882
  - Avoid some unexpected connection closes by telling the client
    that the connection is not persistent if the MPM process
    handling the request is already exiting when the response
    header is built.
  - fix possible crash at startup in case of nonexistent
    DocumentRoot.  PR 39722
  - http_core: OPTIONS * no longer maps to local storage or URI
    space. Note that unlike previous versions, OPTIONS * no longer
    returns an Allow: header. PR 43519
  - scoreboard: improve error message on apr_shm_create failure PR
    40037
  - Don't send spurious "100 Continue" response lines.  PR 38014
  - http_protocol:
  - Escape request method in 413 error reporting.  Determined to
    be not generally exploitable, but a flaw in any case.  PR
    44014
  - Add "DefaultType none" option.  PR 13986 and PR 16139
  - Escape request method in 405 error reporting.  This has no
    security impact since the browser cannot be tricked into
    sending arbitrary method strings.
  - Various code cleanups. PR 38699, 39518, 42005, 42006, 42007, 42008, 42009
  - Add explicit charset to the output of various modules to work
    around possible cross-site scripting flaws affecting web
    browsers that do not derive the response character set as
    required by  RFC2616.  One of these reported by SecurityReason
  - rotatelogs: Change command-line parsing to report more types
    of errors.  Allow local timestamps to be used when rotating based
    on file size.
* Wed Sep 12 2007 poeml@suse.de
- fix graceful-restart. Wait until the pidfile is gone, but don't
  wait for the parent to disappear. It stays there, after closing
  the listen ports.
* Wed Sep 12 2007 poeml@suse.de
- use debug_package macro only on suse, because it breaks the build
  on Mandriva
* Wed Sep 12 2007 poeml@suse.de
- don't configure in maintainer-mode. It not only enables compile
  time warnings, but also adds AP_DEBUG into the mix which causes
  enablement of debug code which is not wanted in production
  builds.
* Mon Sep 10 2007 poeml@suse.de
- upstream 2.2.6
  SECURITY: CVE-2007-3847 (cve.mitre.org)
    mod_proxy: Prevent reading past the end of a buffer when parsing
    date-related headers.  PR 41144.
  SECURITY: CVE-2007-1863 (cve.mitre.org)
    mod_cache: Prevent a segmentation fault if attributes are listed in a
    Cache-Control header without any value.
  SECURITY: CVE-2007-3304 (cve.mitre.org)
    prefork, worker, event MPMs: Ensure that the parent process cannot
    be forced to kill processes outside its process group.
  SECURITY: CVE-2006-5752 (cve.mitre.org)
    mod_status: Fix a possible XSS attack against a site with a public
    server-status page and ExtendedStatus enabled, for browsers which
    perform charset "detection".  Reported by Stefan Esser.
  SECURITY: CVE-2007-1862 (cve.mitre.org)
    mod_mem_cache: Copy headers into longer lived storage; header names and
    values could previously point to cleaned up storage.  PR 41551.
  mod_alias:
  - Accept path components (URL part) in Redirects. PR 35314.
  mod_authnz_ldap:
  - Don't return HTTP_UNAUTHORIZED during authorization when
    LDAP authentication is configured but we haven't seen any
    'Require ldap-*' directives, allowing authorization to be passed to lower
    level modules (e.g. Require valid-user) PR 43281
  mod_autoindex:
  - Add in Type and Charset options to IndexOptions
    directive. This allows the admin to explicitly set the
    content-type and charset of the generated page and is therefore
    a viable workaround for buggy browsers affected by CVE-2007-4465
  mod_cache:
  - Remove expired content from cache that cannot be revalidated.
    PR 30370.
  - Do not set Date or Expires when they are missing from the
    original response or are invalid.
  - Correctly handle HEAD requests on expired cache content.  PR
    41230.
  - Let Cache-Control max-age set the expiration of the cached
    representation if Expires is not set.
  - Allow caching of requests with query arguments when
    Cache-Control max-age is explicitly specified.
  - Use the same cache key throughout the whole request processing
    to handle escaped URLs correctly.  PR 41475.
  - Add CacheIgnoreQueryString directive. PR 41484.
  - While serving a cached entity ensure that filters that have
    been applied to this cached entity before saving it to the
    cache are not applied again. PR 40090.
  - Correctly cache objects whose URL query string has been
    modified by mod_rewrite. PR 40805.
  mod_cgi, mod_cgid:
  - Fix use of CGI scripts as ErrorDocuments.  PR 39710.
  mod_dbd:
  - Introduce configuration groups to allow inheritance by virtual
    hosts of database configurations from the main server.
    Determine the minimal set of distinct configurations and share
    connection pools whenever possible.  Allow virtual hosts to
    override inherited SQL statements.  PR 41302.
  - Create memory sub-pools for each DB connection and close DB
    connections in a pool cleanup function.  Ensure prepared
    statements are destroyed before DB connection is closed.  When
    using reslists, prevent segfaults when child processes exit,
    and stop memory leakage of ap_dbd_t structures.  Avoid use of
    global s->process->pool, which isn't destroyed by exiting
    child processes in most multi-process MPMs.  PR 39985.
  - Handle error conditions in dbd_construct() properly.  Simplify
    ap_dbd_open() and use correct arguments to apr_dbd_error()
    when non-threaded.  Register correct cleanup data in
    non-threaded ap_dbd_acquire() and ap_dbd_cacquire().  Clean up
    configuration data and merge function.  Use ap_log_error()
    wherever possible.
  - Stash DBD connections in request_config of initial request
    only, or else sub-requests and internal redirections may cause
    entire DBD pool to be stashed in a single HTTP request.
  mod_deflate:
  - don't try to process metadata buckets as data.  what should
    have been a 413 error was logged as a 500 and a blank screen
    appeared at the browser.
  - fix protocol handling in deflate input filter PR 23287
  mod_disk_cache:
  - Allow Vary'd responses to be refreshed properly.
  mod_dumpio:
  - Fix for correct dumping of traffic on EBCDIC hosts Data had
    been incorrectly converted twice, resulting in garbled log
    output.
  mod_expires:
  - don't crash on bad configuration data PR 43213
  mod_filter:
  - fix integer comparisons in dispatch rules PR 41835
  - fix merging of ! and = in FilterChain PR 42186
  mod_headers:
  - Allow %% at the end of a Header value. PR 36609.
  mod_info:
  - mod_info outputs invalid XHTML 1.0 transitional.  PR 42847
  mod_ldap:
  - Avoid possible crashes, hangs, and busy loops due to improper
    merging of the cache lock in vhost config PR 43164
  mod_ldap:
  - Remove the hardcoded size limit parameter for
    ldap_search_ext_s and replace it with an APR_ defined value
    that is set according to the LDAP SDK being used.
  mod_mem_cache:
  - Increase the minimum and default value for MCacheMinObjectSize
    from 0 to 1, as a MCacheMinObjectSize of 0 does not make sense
    and leads to a division by zero.  PR 40576.
  mod_negotiation:
  - preserve Query String in resolving a type map PR 33112
  mod_proxy:
  - mod_proxy_http: accept proxy-sendchunked/proxy-sendchunks as
    synonymous.  PR 43183
  - Ensure that at least scheme://hostname[:port] matches between
    worker and URL when searching for the best fitting worker for
    a given URL.  PR 40910
  - Improve network performance by setting APR_TCP_NODELAY
    (disable Nagle algorithm) on sockets if implemented.  PR 42871
  - Add a missing assignment in an error checking code path.  PR 40865
  - don't URLencode tilde in path component PR 38448
  - enable Ignore Errors option on ProxyPass Status.  PR 43167
  - Allow to use different values for sessionid in url encoded id
    and cookies. PR 41897.
  - Fix the 503 returned when session route does not match any of
    the balancer members.
  - Added ProxyPassMatch directive, which is similar to ProxyPass
    but takes a regex local path prefix.
  - Print the correct error message for erroneous configured
    ProxyPass directives. PR 40439.
  - Fix some proxy setting inheritance problems (eg:
    ProxyTimeout). PR 11540.
  - proxy/ajp_header.c: Fixed header token string comparisons
    Matching of header tokens failed to include the trailing NIL
    byte and could misinterpret a longer header token for a
    shorter.  Additionally, a "Content-Type" comparison was made
    case insensitive.
  - proxy/ajp_header.c: Backport of an AJP protocol fix for EBCDIC
    On EBCDIC machines, the status_line string was incorrectly
    converted twice.
  mod_proxy_connect:
  - avoid segfault on DNS lookup failure.  PR 40756
  mod_proxy_http:
  - HTTP proxy ProxyErrorOverride: Leave 1xx and 3xx responses
    alone.  Only processing of error responses (4xx, 5xx) will be
    altered. PR 39245.
  - Don't try to read body of a HEAD request before responding. PR 41644
  - Handle request bodies larger than 2 GB by converting the
    Content-Length header of the request correctly. PR 40883.
  mod_ssl:
  - Fix spurious hostname mismatch warning for valid wildcard
    certificates.  PR 37911.
  - Version reporting update; displays 'compiled against' Apache
    and build-time SSL Library versions at loglevel [info], while
    reporting the run-time SSL Library version in the server info
    tags.  Helps to identify a mod_ssl built against one flavor of
    OpenSSL but running against another (also adds SSL-C version
    number reporting.)
  - initialize thread locks before initializing the hardware
    acceleration library, so the latter can make use of the
    former.  PR 20951.
  core:
  - Do not replace a Date header set by a proxied backend server. PR 40232
  - log core: ensure we use a special pool for stderr logging, so that
    the stderr channel remains valid from the time plog is destroyed,
    until the time the open_logs hook is called again.
  - main core: Emit errors during the initial apr_app_initialize()
    or apr_pool_create() (when apr-based error reporting is not ready).
  - log core: fix the new piped logger case where we couldn't connect
    the replacement stderr logger's stderr to the NULL stdout stream.
    Continue in this case, since the previous alternative of no error
    logging at all (/dev/null) is far worse.
  - Correct a regression since 2.0.x in the handling of AllowOverride
    Options. PR 41829.
  - Unix MPMs: Catch SIGFPE so that exception hooks and CoreDumpDirectory
    can work after that terminating signal.
  - mod_so: Provide more helpful LoadModule feedback when an error occurs.
  misc:
  - mime.types: Many updates to sync with IANA registry and common
    unregistered types that the owners refuse to register.  Admins
    are encouraged to update their installed mime.types file.  PR:
    35550, 37798, 39317, 31483
  - mime.types: add Registered Javascript/ECMAScript MIME types
    (RFC4329) PR 40299
  - htdbm: Enable crypt support on platforms with crypt() but not
    <crypt.h>, such as z/OS.
  - ab.c: Correct behavior of HTTP request headers sent by ab in
    presence of -H command-line overrides. PR 31268, 26554.
  - ab.c: The apr_port_t type is unsigned, but ab was using a
    signed format code in its reports. PR 42070.
- drop obsolete patches apache2-mod_cache-CVE-2007-1863.patch
    apache2-mod_status-CVE-2006-5752.patch
    httpd-2.2.4-mod_autoindex-charset-r570962.patch
    mod_dbd.c-issue18989-autoconnect.dif
    mod_dbd.c-r571441
* Mon Sep  3 2007 skh@suse.de
- get_module_list: replace loadmodule.conf atomically [bnc #214863]
* Fri Aug 31 2007 poeml@suse.de
- /etc/init.d/apache2: implement restart-graceful, stop-graceful
* Fri Aug 31 2007 poeml@suse.de
- update mod_dbd to trunk version (r571441)
  * apr_dbd_check_conn() just returns APR_SUCCESS or
    APR_EGENERAL, so we don't actually have a driver-specific value
    to pass to apr_dbd_error(), but that's OK because most/all
    drivers just ignore this value anyway
* Fri Aug 31 2007 poeml@suse.de
- replace httpd-2.2.3-AddDirectoryIndexCharset.patch with the upstream
  solution, httpd-2.2.4-mod_autoindex-charset-r570962.patch [#153557]
  (backport from 2.2.6)
  * Merge r570532, r570535, r570558 from trunk:
    IndexOptions ContentType=text/html Charset=UTF-8 magic.
    http://svn.apache.org/viewvc?rev=570962&view=rev
    http://issues.apache.org/bugzilla/show_bug.cgi?id=42105
  This means that the AddDirectoryIndexCharset is no longer
  available. Instead, IndexOptions Charset=xyz can be used.
* Fri Aug 31 2007 poeml@suse.de
- remove libexpat-devel in the build service version of the package
- apply apache2-mod_cache-CVE-2007-1863.patch (patch 152) in the
  buildservice package
- don't apply mod_dbd.c-issue18989-autoconnect.dif, since it
  patches only modules/database/mod_dbd.c which is replaced with
  trunk version anyway
* Thu Aug 23 2007 mskibbe@suse.de
- Bug 289996 - VUL-0: mod_status XSS in public server status page
- Bug 289997 - VUL-0: apache2: mod_cache remote denial of service
* Wed Jul 18 2007 skh@suse.de
- split off apache2-utils subpackage, containing all helper tools that
  are useful for system administrators in general (b.n.c. #272292 and
  FATE #302059)
* Thu Mar 29 2007 dmueller@suse.de
- add zlib-devel to BuildRequires
* Fri Mar 23 2007 poeml@suse.de
- add mod_dbd.c from trunk (r512038), the version we run ourselves
  http://svn.apache.org/viewvc/httpd/httpd/trunk/modules/database/mod_dbd.c?view=log
- add mod_dbd.c-issue18989-autoconnect.dif, but disabled. It
  applies to 2.2.4 mod_dbd.c but not to the trunk version
- build mod_version
- fix documentation link in apache2-httpd.conf
* Tue Mar 20 2007 mskibbe@suse.de
- add firewall file for ssl (#246929)
* Mon Mar 19 2007 mskibbe@suse.de
- Apache - Support for FATE #300687: Ports for SuSEfirewall added
  via packages (#246929)
* Fri Jan 26 2007 poeml@suse.de
- the QUICKSTART Readmes have been moved to
  http://www.opensuse.org/Apache
* Mon Jan 22 2007 poeml@suse.de
- point out better in README.QUICKSTART.SSL that a vhost needs to
  be created
- fixes to README.QUICKSTART.WebDAV
- updated email addresses (now there is apache@suse.de)
* Sat Jan 20 2007 poeml@suse.de
- add httpd-2.2.x.doublefree.patch, backport of
  http://svn.apache.org/viewvc?diff_format=h&view=rev&revision=496831
  See http://issues.apache.org/bugzilla/show_bug.cgi?id=39985
* Thu Jan 18 2007 poeml@suse.de
- create debuginfo package in the buildservice
* Fri Jan 12 2007 mskibbe@suse.de
- change path to service cml document (fate #301708)
* Tue Jan  9 2007 poeml@suse.de
- upstream 2.2.4
  mod_authnz_ldap:
  - Add an AuthLDAPRemoteUserAttribute directive. If set,
    REMOTE_USER will be set to this attribute, rather than the
    username supplied by the user. Useful for example when you
    want users to log in using an email address, but need to
    supply a userid instead to the backend.
  mod_cache:
  - From RFC3986 (section 6.2.3.) if a URI contains an authority
    component and an empty path, the empty path is to be
    equivalent to "/". It explicitly cites the following four URIs
    as equivalents:
    http://example.com
    http://example.com/
    http://example.com:/
    http://example.com:80/
  - Eliminate a bogus error in the log when a filter returns
    AP_FILTER_ERROR.
  - Don't cache requests with a expires date in the past;
    otherwise mod_cache will always try to cache the URL. This bug
    might lead to numerous rename() errors on win32 if the URL was
    previously cached.
  mod_cgi and mod_cgid:
  - Don't use apr_status_t error return from input filters as HTTP
    return value from the handler.  PR 31579.
  mod_dbd:
  - share per-request database handles across subrequests and
    internal redirects
  - key connection pools to virtual hosts correctly even when
    ServerName is unset/unavailable
  mod_deflate:
  - Rework inflate output and deflate output filter to fix several
    issues: Incorrect handling of flush buckets, potential memory
    leaks, excessive memory usage in inflate output filter for
    large compressed content. PR 39854.
  mod_disk_cache:
  - Make sure that only positive integers are accepted for the
    CacheMaxFileSize and CacheMinFileSize parameters in the config
    file. PR39380.
  mod_dumpio:
  - Allow mod_dumpio to log at other than DEBUG levels via the new
    DumpIOLogLevel directive.
  mod_echo:
  - Fix precedence problem in if statement. PR 40658.
  mod_ext_filter:
  - Handle filter names which include capital letters.  PR 40323.
  mod_headers:
  - Support regexp-based editing of HTTP headers.
  mod_mime_magic:
  - Fix precedence problem in if statement. PR 40656.
  mod_mem_cache:
  - Memory leak fix: Unconditionally free the buffer.
  - Convert mod_mem_cache to use APR memory pool functions by
    creating a root pool for object persistence across requests.
    This also eliminates the need for custom serialization code.
  mod_proxy:
  - Don't try to use dead backend connection. PR 37770.
  - Add explicit flushing feature. When Servlet container sends
    AJP body message with size 0, this means that Servlet
    container has asked for an explicit flush. Create flush bucket
    in that case. This feature has been added to the recent Tomcat
    versions without breaking the AJP protocol.
  mod_proxy_ajp:
  - Close connection to backend if reading of request body fails.
    PR 40310.
  - Added cping/cpong support for the AJP protocol.  A new worker
    directive ping=timeout will cause CPING packet to be send
    expecting CPONG packet within defined timeout.  In case the
    backend is too busy this will fail instead sending the full
    header.
  mod_proxy_balancer:
  - Workers can now be defined as part of a balancer cluster "set"
    in which members of a lower-numbered set are preferred over
    higher numbered ones.
  - Workers can now be defined as "hot standby" which will only be
    used if all other workers are unusable (eg: in error or
    disabled). Also, the balancer-manager displays the election
    count and I/O counts of all workers.
  - Retry worker chosen by route / redirect worker if it is in
    error state before sending "Service Temporarily Unavailable".
    PR 38962.
  - Extract stickysession routing information contained as
    parameter in the URL correctly. PR 40400.
  - Set the new environment variable BALANCER_ROUTE_CHANGED if a
    worker with a route different from the one supplied by the
    client had been chosen or if the client supplied no routing
    information for a balancer with sticky sessions.
  - Add information about the route, the sticky session and the
    worker used during a request as environment variables. PR
    39806.
  core:
  - Fix issue which could cause piped loggers to be orphaned and
    never terminate after a graceful restart.  PR 40651.
  - Fix address-in-use startup failure caused by corruption of the
    list of listen sockets in some configurations with multiple
    generic Listen directives.
  - Fix NONBLOCK status of listening sockets on restart/graceful
    PR 37680.
  - Deal with the widespread use of apr_status_t return values as
    HTTP status codes, as documented in PR#31759 (a bug shared by
    the default handler, mod_cgi, mod_cgid, mod_proxy, and
    probably others). PR31759.
  - The full server version information is now included in the
    error log at startup as well as server status reports,
    irrespective of the setting of the ServerTokens directive.
    ap_get_server_version() is now deprecated, and is replaced by
    ap_get_server_banner() and ap_get_server_description().
  misc:
  - Allow htcacheclean, httxt2dbm, and fcgistarter to link
    apr/apr-util statically like the older support programs.
  - Better detection and clean up of ldap connection that has been
    terminated by the ldap server.  PR 40878.
  - rotatelogs: Improve error message for open failures.  PR
    39487.
* Mon Jan  8 2007 mskibbe@suse.de
- Apache XML Service Description Document (fate #301708)
* Thu Dec 21 2006 poeml@suse.de
- add patch to add charset=utf-8 to directory listings generated by
  mod_autoindex, and add a directive to allow overriding the
  charset (testing, needs to be discussed with upstream) [#153557]
  httpd-2.2.3-AddDirectoryIndexCharset.patch
* Wed Dec 20 2006 poeml@suse.de
- set a proper HOME (/var/lib/apache2), otherwise the server might
  end up HOME=/root and some script might try to use that [#132769]
- add two notes to the QUICKSTART readmes
- don't install /etc/apache2/extra configuration since this is only
  serving as an example and installed with the documentation anyway
* Tue Sep 26 2006 poeml@suse.de
- add rpm macro for suexec_safepath
- use _bindir/_sbindir in a few places [#202355]
- remove unused /sbin/conf.d directory from build root
* Thu Aug 31 2006 poeml@suse.de
- Enable fatal exception hook for use by diagnostic modules.
* Tue Aug 29 2006 poeml@suse.de
- move some binaries, where calling by users makes sense (dbmmanage
  htdbm htdigest htpasswd), from /usr/sbin to /usr/bin [#140133]
* Wed Aug  9 2006 poeml@suse.de
- upstream 2.2.3
  |SECURITY: CVE-2006-3747 (cve.mitre.org)
  |  mod_rewrite: Fix an off-by-one security problem in the ldap scheme
  |  handling.  For some RewriteRules this could lead to a pointer being
  |  written out of bounds.  Reported by Mark Dowd of McAfee.
  | mod_authn_alias: Add a check to make sure that the base provider and the
  |  alias names are different and also that the alias has not been registered
  |  before. PR 40051.
  | mod_authnz_ldap: Fix a problem with invalid auth error detection for LDAP
  |  client SDKs that don't support the LDAP_SECURITY_ERROR macro. PR 39529.
  | mod_autoindex: Fix filename escaping with FancyIndexing disabled.
  |  PR 38910.
  | mod_cache:
  | - Make caching of reverse SSL proxies possible again. PR 39593.
  | - Do not overwrite the Content-Type in the cache, for
  |   successfully revalidated cached objects. PR 39647.
  | mod_charset_lite: Bypass translation when the source and dest charsets
  |  are the same.
  | mod_dbd: Fix dependence on virtualhost configuration in
  |  defining prepared statements (possible segfault at startup
  |  in user modules such as mod_authn_dbd).
  | mod_mem_cache: Set content type correctly when delivering data from
  |  cache. PR 39266.
  | mod_speling: Add directive to deal with case corrections only
  |  and ignore other misspellings
  | miscellaneous:
  |  - Add optional 'scheme://' prefix to ServerName directive,
  |    allowing correct determination of the canonical server URL
  |    for use behind a proxy or offload device handling SSL;
  |    fixing redirect generation in those cases. PR 33398.
  |  - Added server_scheme field to server_rec for above. Minor MMN bump.
  |  - Worker MPM: On graceless shutdown or restart, send signals
  |    to each worker thread to wake them up if they're polling on
  |    a Keep-Alive connection.  PR 38737.
  |  - worker and event MPMs: fix excessive forking if fork() or
  |    child_init take a long time.  PR 39275.
  |  - Respect GracefulShutdownTimeout in the worker and event MPMs.
  |  - configure: Add "--with-included-apr" flag to force use of
  |    the bundled version of APR at build time.
* Tue Jul  4 2006 poeml@suse.de
- a2enmod, a2enflag: add /usr/sbin to PATH so sysconf_addword is
  found
* Fri Jun 23 2006 poeml@suse.de
- fix typo in apache-20-22-upgrade script: mod_image_map ->
  mod_imagemap
* Mon Jun 12 2006 poeml@suse.de
- enable logresolve processing of lines longer than 1024 characters
  by compiling with MAXLINE=4096 [#162806]
* Fri Jun  9 2006 poeml@suse.de
- upstream 2.2.2
  | SECURITY: CVE-2005-3357 (cve.mitre.org)
  |   mod_ssl: Fix a possible crash during access control checks
  |   if a non-SSL request is processed for an SSL vhost (such as
  |   the "HTTP request received on SSL port" error message when
  |   an 400 ErrorDocument is configured, or if using "SSLEngine
  |   optional").  PR 37791.
  | SECURITY: CVE-2005-3352 (cve.mitre.org)
  |   mod_imagemap: Escape untrusted referer header before
  |   outputting in HTML to avoid potential cross-site scripting.
  |   Change also made to ap_escape_html so we escape quotes.
  |   Reported by JPCERT.
  | mod_cache:
  |  - Make caching of reverse proxies possible again. PR 38017.
  | mod_disk_cache:
  |  - Return the correct error codes from bucket read failures,
  |    instead of APR_EGENERAL.
  | mod_dbd:
  |  - Update defaults, improve error reporting.
  |  - Create own pool and mutex to avoid problem use of process
  |    pool in request processing.
  | mod_deflate:
  |  - work correctly in an internal redirect
  | mod_proxy:
  |  - don't reuse a connection that may be to the wrong backend PR 39253
  |  - Do not release connections from connection pool twice.  PR 38793.
  |  - Fix KeepAlives not being allowed and set to backend servers.  PR 38602.
  |  - Fix incorrect usage of local and shared worker init.  PR 38403.
  |  - If we get an error reading the upstream response, close the
  |    connection.
  | mod_proxy_balancer:
  |  - Initialize members of a balancer correctly.  PR 38227.
  | mod_proxy_ajp:
  |  - Flushing of the output after each AJP chunk is now
  |    configurable at runtime via the 'flushpackets' and 'flushwait'
  |    worker params. Minor MMN bump.
  |  - Crosscheck the length of the body chunk with the length of the
  |    ajp message to prevent mod_proxy_ajp from reading beyond the
  |    buffer boundaries and thus revealing possibly sensitive memory
  |    contents to the client.
  |  - Support common headers of the AJP protocol in responses.  PR 38340.
  | mod_proxy_http:
  |  - Do send keep-alive header if the client sent connection:
  |    keep-alive and do not close backend connection if the client
  |    sent connection: close. PR 38524.
  | mod_proxy_balancer:
  |  - Do not overwrite the status of initialized workers and respect
  |    the configured status of uninitilized workers when creating a
  |    new child process.
  |  - Fix off-by-one error in proxy_balancer.  PR 37753.
  | mod_speling:
  |  - Stop crashing with certain non-file requests.
  | mod_ssl:
  |  - Fix possible crashes in shmcb with gcc 4 on platforms
  |    requiring word-aligned pointers.  PR 38838.
  | miscellaneous:
  |  - core: Prevent reading uninitialized memory while reading a line of
  |    protocol input.  PR 39282.
  |  - core: Reject invalid Expect header immediately. PR 38123.
  |  - Default handler: Don't return output filter apr_status_t values.
  |    PR 31759.
  |  - Add APR/APR-Util Compiled and Runtime Version numbers to the
  |    output of 'httpd -V'.
  |  - http: If a connection is aborted while waiting for a chunked line,
  |    flag the connection as errored out.
  |  - Don't hang on error return from post_read_request.  PR 37790.
  |  - Fix mis-shifted 32 bit scope, masked to 64 bits as a method.
  |  - Fix recursive ErrorDocument handling.  PR 36090.
  |  - Ensure that the proper status line is written to the client, fixing
  |    incorrect status lines caused by filters which modify r->status without
  |    resetting r->status_line, such as the built-in byterange filter.
  |  - HTML-escape the Expect error message.  Not classed as security as
  |    an attacker has no way to influence the Expect header a victim will
  |    send to a target site.
  |  - Chunk filter: Fix chunk filter to create correct chunks in the case that
  |    a flush bucket is surrounded by data buckets.
  |  - Avoid Server-driven negotiation when a script has emitted an
  |    explicit Status: header.  PR 38070.
  |  - htdbm: Fix crash processing -d option in 64-bit mode on HP-UX.
  |  - htdbm: Warn the user when adding a plaintext password on a platform
  |    where it wouldn't work with the server (i.e., anywhere that has
  |    crypt()).
- adapted httpd-2.1.3alpha-autoconf-2.59.dif
- other user visible changes:
  * use a2enmod, a2enflag in apache2-README.QUICKSTART.*
  * add README.QUICKSTART link to httpd.conf
- when installing/updating, avoid irritating message in
  /var/log/messages ("group is unknown - group=wwwadmin") [#183071]
- build system changes:
  * clean up old cruft tight to suse_version macros
  * don't run buildconf, and thus don't need python.
  * don't ship uid.conf as source file, but create it dynamically
    instead, according to user/group defined via rpm macro
  * create wwwrun:www user on non-SUSE builds
  * work around missimg macros insserv_prereq and fillup_prereq on non-SUSE builds
  * add openssl-devel and expat-devel to Buildrequires for non-SUSE builds
  * make sure that the rpm macro sles_version is defined
  * remove obsolete VENDOR UnitedLinux macro
* Tue Apr 25 2006 poeml@suse.de
- obsolete 'apache' package on SLES10 (obsolete it on all platforms
  except SLES9 and old SL releases)
* Wed Mar 29 2006 poeml@suse.de
- remove php4 from default modules [#155333]
- fix comment in /etc/init.d/apache2 [#148559]
* Mon Feb 20 2006 poeml@suse.de
- fixed comment in init script which indicated wrong version [#148559]
* Mon Jan 30 2006 poeml@suse.de
- added Requires: libapr-util1-devel to apache2-devel package [#146496]
* Fri Jan 27 2006 poeml@suse.de
- add a note about NameVirtualHost statements to the vhost template
  files [#145000]
* Wed Jan 25 2006 mls@suse.de
- converted neededforbuild to BuildRequires
* Fri Jan 20 2006 poeml@suse.de
- cleanup: remove obsolete metuxmpm patch
- improve informational text in apache-20-22-upgrade
* Wed Jan 18 2006 poeml@suse.de
- the new DYNAMIC_MODULE_LIMIT default in 2.2 is 128, so no need to
  increase it anymore (fixes [#143536])
* Mon Dec 19 2005 poeml@suse.de
- update to 2.2.0
- enable all new modules
- replaced modules "auth auth_dbm access" in default configuration
  by "auth_basic authn_file authn_dbm authz_host authz_default
  authz_user""
- /usr/share/apache2/apache-20-22-upgrade will fix the module list
  on upgrade
- fix bug in sysconf_addword (used by a2enmod) to respect word
  boundaries when removing a word (but don't count slashes as word
  boundary)
- remove perchild mpm subpackage, add experimemtal event mpm
- remove obsolete tool apache2-reconfigure-mpm
- remove obsolete perchild config from apache2-server-tuning.conf
- remove libapr0 subpackage; add libapr1 and libapr-util1 to #neededforbuild
- build against system pcre
- build with --enable-pie
- don't modify which libraries are linked in
- adjust IndexIgnore setting to upstream default. Previously, the
  parent directory (..) was being ignored
- package the symlinks in ssl.crt
* Wed Dec  7 2005 poeml@suse.de
- patch apxs to use the new a2enmod tool, when called with -a
- add -l option to a2enmod, which gives a list of active modules
- adjust feedback address in the readmes
- update README.QUICKSTART.SSL (mention TinyCA)
- add more documentation in server-tuning.conf, and adjust defaults
- do not document the restart-hup action of the init script. It
  should not be used
- don't install the tool checkgid -- it is only usable during
  installation
* Fri Nov 18 2005 poeml@suse.de
- fix duplicated Source45 tag
* Mon Oct 24 2005 poeml@suse.de
- update to 2.0.55. Relevant changes:
  | SECURITY: CAN-2005-2700 (cve.mitre.org)
  |    mod_ssl: Fix a security issue where "SSLVerifyClient" was
  |    not enforced in per-location context if "SSLVerifyClient
  |    optional" was configured in the vhost configuration.
  | SECURITY: CAN-2005-2491 (cve.mitre.org):
  |    Fix integer overflows in PCRE in quantifier parsing which
  |    could be triggered by a local user through use of a
  |    carefully-crafted regex in an .htaccess file.
  | SECURITY: CAN-2005-2088 (cve.mitre.org)
  |    proxy: Correctly handle the Transfer-Encoding and
  |    Content-Length headers.  Discard the request Content-Length
  |    whenever T-E: chunked is used, always passing one of either
  |    C-L or T-E: chunked whenever the request includes a request
  |    body.  Resolves an entire class of proxy HTTP Request
  |    Splitting/Spoofing attacks.
  | SECURITY: CAN-2005-2728 (cve.mitre.org)
  |    Fix cases where the byterange filter would buffer responses
  |    into memory.  PR 29962.
  | SECURITY: CAN-2005-2088 (cve.mitre.org)
  |    core: If a request contains both Transfer-Encoding and
  |    Content-Length headers, remove the Content-Length,
  |    mitigating some HTTP Request Splitting/Spoofing attacks.
  | SECURITY: CAN-2005-1268 (cve.mitre.org)
  |    mod_ssl: Fix off-by-one overflow whilst printing CRL
  |    information at "LogLevel debug" which could be triggered if
  |    configured to use a "malicious" CRL.  PR 35081.
  | miscellaneous:
  |  - worker MPM: Fix a memory leak which can occur after an
  |    aborted connection in some limited circumstances.
  |  - worker mpm: don't take down the whole server for a transient
  |    thread creation failure. PR 34514
  |  - Added TraceEnable [on|off|extended] per-server directive to
  |    alter the behavior of the TRACE method.  This addresses a
  |    flaw in proxy conformance to RFC 2616 - previously the proxy
  |    server would accept a TRACE request body although the RFC
  |    prohibited it.  The default remains 'TraceEnable on'.
  |  - Add ap_log_cerror() for logging messages associated with
  |    particular client connections.
  |  - Support the suppress-error-charset setting, as with Apache
  |    1.3.x.  PR 31274.
  |  - Fix bad globbing comparison which could result in getting a
  |    directory listing when a file was requested. PR 34512.
  |  - Fix a file descriptor leak when starting piped loggers.  PR
  |    33748.
  |  - Prevent hangs of child processes when writing to piped
  |    loggers at the time of graceful restart.  PR 26467.
  | mod_cgid:
  |  - Correct mod_cgid's argv[0] so that the full path can be
  |    delved by the invoked cgi application, to conform to the
  |    behavior of mod_cgi.
  | mod_include:
  |  - Fix possible environment variable corruption when using
  |    nested includes.  PR 12655.
  | mod_ldap:
  |  - Fix PR 36563. Keep track of the number of attributes
  |    retrieved from LDAP so that all of the values can be
  |    properly cached even if the value is NULL.
  |  - Fix core dump if mod_auth_ldap's
  |    mod_auth_ldap_auth_checker() was called even if
  |    mod_auth_ldap_check_user_id() was not (or if it didn't
  |    succeed) for non-authoritative cases.
  |  - Avoid segfaults when opening connections if using a version
  |    of OpenLDAP older than 2.2.21.  PR 34618.
  |  - Fix various shared memory cache handling bugs.  PR 34209.
  | mod_proxy:
  |  - Fix over-eager handling of '%%' for reverse proxies.  PR
  |    15207.
  |  - proxy HTTP: If a response contains both Transfer-Encoding
  |    and a Content-Length, remove the Content-Length and don't
  |    reuse the connection, mitigating some HTTP Response
  |    Splitting attacks.
  |  - proxy HTTP: Rework the handling of request bodies to handle
  |    chunked input and input filters which modify content length,
  |    and avoid spooling arbitrary-sized request bodies in memory.
  |    PR 15859.
  | mod_ssl:
  |  - Fix build with OpenSSL 0.9.8.  PR 35757.
  | mod_rewrite:
  |  - use buffered I/O to improve performance with large
  |    RewriteMap txt: files.
  | mod_userdir:
  |  - Fix possible memory corruption issue.  PR 34588.
- drop obsolete patches httpd-2.0.54-openssl-0.9.8.dif
    httpd-2.0.54-CAN-2005-1268-mod_ssl-crl.dif
    apache2-bundled-pcre-5.0-CAN-2005-2491.dif
    httpd-2.0.54-SSLVerifyClient-CAN-2005-2700.diff
    httpd-2.0.54-ap_byterange-CAN-2005-2728.diff
- add httpd-2.0.55-37145_2.0.x.diff (broken mod_proxy in 2.0.55)
* Thu Oct 20 2005 poeml@suse.de
- rc.apache2: when stopping the server, wait for the actual binary
  of the parent process to disappear. Waiting for the pid file to
  disappear is not sufficient, because not all cleanup might be
  finished at the time of its removal. [#96492], [#85539]
* Wed Oct 12 2005 poeml@suse.de
- fix security hole by wrongly initializing LD_LIBRARY_PATH in
  /usr/sbin/envvars (used by apache2ctl only) [#118188]
* Fri Sep 30 2005 poeml@suse.de
- accomodate API changes to OpenSSL 0.9.8 (r209468 from 2.0.x branch)
* Sun Sep 25 2005 ro@suse.de
- define LDAP_DEPRECATED in CFLAGS
* Fri Sep  2 2005 poeml@suse.de
- security fix [CAN-2005-2728 (cve.mitre.org)]:
  fix memory consumption bug in byterange handling
- security fix [CAN-2005-2700 (cve.mitre.org)]: [#114701]
  if "SSLVerifyClient optional" has been configured at the vhost
  context then "SSLVerifyClient require" is not enforced in a
  location context within that vhost; effectively allowing clients
  to bypass client-cert authentication checks. [#114701]
* Wed Aug 31 2005 poeml@suse.de
- Security fix: fix integer overflows in PCRE in quantifier parsing which
  could be triggered by a local user through use of a carefully-crafted
  regex in an .htaccess file. CAN-2005-2491 [#112651] [#106209]
* Tue Aug 30 2005 lmuelle@suse.de
- Escape also any forward slash while removing a word with sysconf_addword.
* Fri Aug 26 2005 lmuelle@suse.de
- Escape any forward slash in the word argument of sysconf_addword.
* Sat Aug 13 2005 ro@suse.de
- alingn suexec2 permissions with permissions.secure
* Thu Aug 11 2005 poeml@suse.de
- the permissions files are now maintained centrally and packaged
  in the permissions package. Package suexec2 with mode 0750. [#66304]
* Fri Aug  5 2005 poeml@suse.de
- change SSLMutex "default" so APR always picks the best on the
  platform
- fix Source42 tag which was present twice
- add a2enmod/a2enflag to add/remove modules/flags conveniently
- add charset.conv table for mod_auth_ldap
- make sure that suse_version is defined (it might be unset by e.g.
  ISPs preinstallations)
* Tue Jul 12 2005 poeml@suse.de
- security fix [CAN-2005-2088 (cve.mitre.org)]: core: If a request
  contains both Transfer-Encoding and a Content-Length, remove the
  Content-Length, stopping some HTTP Request smuggling attacks.
  mod_proxy: Reject chunked requests. [#95709]
- security fix [CAN-2005-1268 (cve.mitre.org)]: mod_ssl: fix
  off-by-one overflow whilst printing CRL information at "LogLevel
  debug" which could be triggered if configured to use a
  "malicious" CRL.  PR 35081. [#95709]
* Mon Jun 20 2005 poeml@suse.de
- add httpd-2.0.47-pie.patch from from 2.1.3-dev to compile with
  - fpie and link with -pie
* Wed May 18 2005 poeml@suse.de
- update to 2.0.54. Relevant changes:
  | mod_cache:
  |  - Add CacheIgnoreHeaders directive.  PR 30399.
  | mod_dav:
  | - Correctly export all public functions.
  | mod_ldap:
  | - Added the directive LDAPConnectionTimeout to configure the
  |   ldap socket connection timeout value.
  | mod_ssl:
  | - If SSLUsername is used, set r->user earlier.  PR 31418.
  | miscellaneous:
  | - Unix MPMs: Shut down the server more quickly when child
  |   processes are slow to exit.
  | - worker MPM: Fix a problem which could cause httpd processes
  |   to remain active after shutdown.
  | - Remove formatting characters from ap_log_error() calls.
  |   These were escaped as fallout from CAN-2003-0020.
  | - core_input_filter: Move buckets to a persistent brigade
  |   instead of creating a new brigade. This stop a memory leak
  |   when proxying a Streaming Media Server. PR 33382.
  | - htdigest: Fix permissions of created files.  PR 33765.
* Mon Mar 14 2005 poeml@suse.de
- revise README
* Mon Mar  7 2005 poeml@suse.de
- when building the suexec binary, set the "docroot" compile time
  option to the datadir (/srv/www) instead of the htdocsdir
  (/srv/www/htdocs), so it can be used with virtual hosts placed
  e.g. in /srv/www/vhosts [#63845] Suggested by Winfried Kuiper.
- add php5 to APACHE_MODULES by default, so it can be used simply
  by installing the package. Suppress warning about not-found
  module in the php4/php5 case. [#66729]
- remove a redundant get_module_list call from the init script
- add hints about vhost setup to README.QUICKSTART
- after a change of APACHE_MPM, apache2-reconfigure-mpm is no
  longer needed since SuSEconfig.apache2 is gone. Leave it for
  compatibility, because /etc/sysconfig/apache2 is probably not
  updated and yast may still use it.
- move the 4 most important variables in sysconfig.apache2 to the
  top of the file
- add note about the old monolithic configuration file and how to
  use it
- drop patch httpd-2.0.40-openssl-version.dif (we don't even have
  openssl-0.9.6e anywhere, any longer)
* Wed Mar  2 2005 poeml@suse.de
- fix TLS upgrade patch: with SSLEngine set to Optional, an
  additional token in an Upgrade: header before "TLS/1.0" could
  result into an infinite loop [#67126]
* Tue Feb 22 2005 poeml@suse.de
- run /usr/share/apache2/get_module_list post install, which will
  also create the symlink to the httpd2 binary, which might be
  necessary during package building when apache has been installed
  but never been run.
* Mon Feb 21 2005 poeml@suse.de
- remove SuSEconfig.apache2
* Fri Feb 11 2005 poeml@suse.de
- raise DYNAMIC_MODULE_LIMIT to 80. The test suite loading all
  available modules plus 9 perl modules was beginning to fail
* Wed Feb  9 2005 poeml@suse.de
- update to 2.0.53. Relevant changes:
  | SECURITY: CAN-2004-0942 (cve.mitre.org)
  |   Fix for memory consumption DoS in handling of MIME folded request
  |   headers.
  | SECURITY: CAN-2004-0885 (cve.mitre.org)
  |   mod_ssl: Fix a bug which allowed an SSLCipherSuite setting to be
  |   bypassed during an SSL renegotiation.  PR 31505.
  | mod_dumpio:
  |  - new I/O logging/dumping module, added to the
  |    modules/expermimental subdirectory.
  | mod_ssl:
  |  - fail quickly if SSL connection is aborted rather than making
  |    many doomed ap_pass_brigade calls.  PR 32699.
  |  - Fail at startup rather than segfault at runtime if a client cert
  |    is configured with an encrypted private key.  PR 24030.
  | mod_include:
  |  - Fix bug which could truncate variable expansions of N*64
  |    characters by one byte.  PR 32985.
  | mod_status:
  |  - Start keeping track of time-taken-to-process-request again if
  |    ExtendedStatus is enabled.
  | util_ldap:
  |  - Util_ldap: Implemented the util_ldap_cache_getuserdn() API so
  |    that ldap authorization only modules have access to the
  |    util_ldap user cache without having to require ldap
  |    authentication as well.  PR 31898.
  | mod_ldap:
  |  - Fix format strings to use %%APR_PID_T_FMT instead of %%d.
  |  - prevent the possiblity of an infinite loop in the LDAP
  |    statistics display. PR 29216.
  |  - fix a bogus error message to tell the user which file is causing
  |    a potential problem with the LDAP shared memory cache.  PR 31431
  |  - Fix the re-linking issue when purging elements from the LDAP
  |    cache PR 24801.
  | mod_auth_ldap:
  |  - Added the directive "Requires ldap-attribute" that allows the
  |    module to only authorize a user if the attribute value specified
  |    matches the value of the user object. PR 31913
  |  - Handle the inconsistent way in which the MS LDAP library handles
  |    special characters.  PR 24437.
  | mod_proxy:
  |  - Fix ProxyRemoteMatch directive.  PR 33170.
  |  - Respect errors reported by pre_connection hooks.
  |  - Handle client-aborted connections correctly.  PR 32443.
  | mod_cache:
  |  - CacheDisable will only disable the URLs it was meant to disable,
  |    not all caching. PR 31128.
  |  - Try to correctly follow RFC 2616 13.3 on validating stale cache
  |    responses.
  |  - Fix Expires handling.
  | mod_disk_cache:
  |  - Do not store aborted content.  PR 21492.
  |  - Correctly store cached content type.  PR 30278.
  |  - Do not store hop-by-hop headers.
  |  - Fix races in saving responses.
  | mod_expires:
  |  - Alter mod_expires to run at a different filter priority to allow
  |    proper Expires storage by mod_cache.
  | mod_rewrite:
  |  - Handle per-location rules when r->filename is unset.  Previously
  |    this would segfault or simply not match as expected, depending
  |    on the platform.
  |  - Fix 0 bytes write into random memory position.  PR 31036.
  | miscellaneous:
  |  - Fix --with-apr=/usr and/or --with-apr-util=/usr.  PR 29740.
  |  - apxs: fix handling of -Wc/-Wl and "-o mod_foo.so". PR 31448
  |  - Allow for the use of --with-module=foo:bar where the ./modules/foo
  |    directory is local only. Assumes, of course, that the required
  |    files are in ./modules/foo, but makes it easier to statically
  |    build/log "external" modules.
  |  - --with-module can now take more than one module to be statically
  |    linked: --with-module=<modtype>:<modfile>,<modtype>:<modfile>,...
  |    If the <modtype>-subdirectory doesn't exist it will be created and
  |    populated with a standard Makefile.in.
  |  - Fix handling of files >2Gb on all platforms (or builds) where
  |    apr_off_t is larger than apr_size_t.  PR 28898.
  |  - Remove compiled-in upper limit on LimitRequestFieldSize.
  |  - Correct handling of certain bucket types in ap_save_brigade, fixing
  |    possible segfaults in mod_cgi with #include virtual.  PR 31247.
  |  - conf: Remove AddDefaultCharset from the default configuration
  |    because setting a site-wide default does more harm than good. PR
  |    23421.
  |  - Add charset to example CGI scripts.
- merge tls-upgrade.patch
- remove obsolete httpd-2.0.47-headtail.dif
  httpd-2.0.52-util_ldap_cache_mgr.c.dif
  httpd-2.0.52-SSLCipherSuite-bypass-CAN-2004-0885.dif
  httpd-2.0.52-ssl-incomplete-keypair.dif
  httpd-2.0.52-memory-consumption-DoS-CAN-2004-0942.dif
  httpd-2.0.52.21492.diff
  httpd-2.0.52.30278.diff
  httpd-2.0.52.30399.diff
  httpd-2.0.52.30419.diff
  httpd-2.0.52.31385.diff
- sync configuration with upstream changes
  * Remove AddDefaultCharset (see upstream changelog above)
  * LanguagePriority for error documents updated
* Sat Jan 15 2005 schwab@suse.de
- Use <owner>:<group> in permissions file.
* Tue Jan 11 2005 schwab@suse.de
- Fix /etc/init.d/apache2 to use readlink instead of linkto or file.
* Mon Nov 29 2004 hvogel@suse.de
- fix permission handling
* Thu Nov 11 2004 poeml@suse.de
- fix /etc/init.d/apache2 to correctly handle the start of multiple
  instances of the same binary (using startproc -f plus prior check
  for running instance) [#48153]
- fix helper scripts to allow overriding of $sysconfig_file and
  other useful values
- remove unused 'rundir' variable from /etc/init.d/apache2
- removed backward compatibility code for pre-8.0
- add documentation to the vhost template files and
  README.QUICKSTART
* Mon Nov  8 2004 poeml@suse.de
- security fix [CAN-2004-0942 (cve.mitre.org)]: Fix for memory
  consumption DoS [#47967]
* Thu Nov  4 2004 poeml@suse.de
- remove heimdal-devel from #neededforbuild, it is not needed
* Fri Oct 15 2004 poeml@suse.de
- fix SSLCipherSuite bypass CAN-2004-0885 (cve.mitre.org) [#47117]
- update the TLS upgrade patch [#47207]
  - mod_ssl returned invalid method on TLS upgraded connections
  - additional checks for httpd_method and default_port hooks
  - fixed typo in upgrade header
- add patches from Ruediger Pluem for the experimental modules
  mod_disk_cache, mod_cache
  PR 21492: mod_disk_cache: Do not store aborted content.
  PR 30278: mod_disk_cache: Correctly store cached content type.
  PR 30399: make storing of Set-Cookie headers optional
  PR 30419: weird caching behaviour of mod_cache and old Cookies
  PR 31385: skipping start of file if recaching already cached file
- patch from 2.0.53: Fail to configure when an SSL proxy is
  configured with incomplete client cert keypair, rather than
  segfaulting at runtime. PR 24030
  http://cvs.apache.org/viewcvs/httpd-2.0/modules/ssl/ssl_engine_init.c.diff?r1=1.118&r2=1.119
* Mon Oct 11 2004 poeml@suse.de
- add patch fixing re-linking issue when purging elements from the
  LDAP cache. http://nagoya.apache.org/bugzilla/show_bug.cgi?id=24801
  http://www.apache.org/dist/httpd/patches/apply_to_2.0.52/util_ldap_cache_mgr.c.patch
* Mon Oct 11 2004 poeml@suse.de
- sync update configuration with upstream changes (2.0.52)
  (mostly comments; configuration for spanish manual added)
- add mime type for shortcut icons (favicon.ico)
* Fri Oct  8 2004 poeml@suse.de
- update to 2.0.52. Relevant changes:
  | SECURITY: CAN-2004-0811 (cve.mitre.org)
  |   Fix merging of the Satisfy directive, which was applied to
  |   the surrounding context and could allow access despite configured
  |   authentication.  PR 31315.
  | util_ldap:
  |   Fix a segfault in the LDAP cache when it is configured switched off.
  | mod_mem_cache:
  |   Fixed race condition causing segfault because of memory being
  |   freed twice, or reused after being freed.
  | mod_log_config:
  |   Fix a bug which prevented request completion time from being
  |   logged for I_INSIST_ON_EXTRA_CYCLES_FOR_CLF_COMPLIANCE
  |   processing.  PR 29696.
  | miscellaneous:
  |   - Use HTML 2.0 <hr> for error pages. PR 30732
  |   - Fix the handling of URIs containing %%2F when
  |     AllowEncodedSlashes is enabled.  Previously, such urls would
  |     still be rejected.
  |   - Fix the global mutex crash when the global mutex is never
  |     allocated due to disabled/empty caches.
  |   - Add -l option to rotatelogs to let it use local time rather
  |     than UTC.  PR 24417.
- changes from 2.0.51:
  | SECURITY: CAN-2004-0786 (cve.mitre.org)
  |   Fix an input validation issue in apr-util which could be
  |   triggered by malformed IPv6 literal addresses.
  | SECURITY: CAN-2004-0747 (cve.mitre.org)
  |   Fix buffer overflow in expansion of environment variables in
  |   configuration file parsing.
  | SECURITY: CAN-2004-0809 (cve.mitre.org)
  |   mod_dav_fs: Fix a segfault in the handling of an indirect lock
  |   refresh.  PR 31183.
  | SECURITY: CAN-2004-0751 (cve.mitre.org)
  |   mod_ssl: Fix a segfault in the SSL input filter which could be
  |   triggered if using "speculative" mode, for instance by a proxy
  |   request to an SSL server.  PR 30134.
  | SECURITY: CAN-2004-0748 (cve.mitre.org)
  |   mod_ssl: Fix a potential infinite loop.  PR 29964.
  | mod_include:
  |   no longer checks for recursion, because that's done in the core.
  |   This allows for careful usage of recursive SSI.
  | mod_rewrite:
  |  - Fix memory leak in the cache handlingof mod_rewrite. PR 27862.
  |  - Add %%{SSL:...} and %%{HTTPS} variable lookups.  PR 30464.
  |  - mod_rewrite now officially supports RewriteRules in <Proxy>
  |    sections.  PR 27985.
  |  - no longer confuse the RewriteMap caches if different maps
  |    defined in different virtual hosts use the same map name. PR 26462.
  | mod_ssl:
  |  - Add new 'ssl_is_https' optional function.
  |  - Add "SSLUserName" directive to set r->user based on a chosen SSL
  |    environment variable.  PR 20957.
  |  - Avoid startup failure after unclean shutdown if using shmcb.  PR 18989.
  | mod_autoindex:
  |  - Don't truncate the directory listing if a stat() call fails (for
  |    instance on a >2Gb file).  PR 17357.
  | mod_cache, mod_disk_cache, mod_mem_cache:
  |  - Refactor cache modules, and switch to the provider API instead
  |    of hooks.
  | mod_disk_cache:
  |  - Implement binary format for on-disk header files.
  |  - Optimize network performance of disk cache subsystem by allowing
  |    zero-copy (sendfile) writes and other miscellaneous fixes.
  | mod_userdir:
  |  - Ensure that the userdir identity is used for suexec userdir
  |    access in a virtual host which has suexec configured.  PR 18156.
  | mod_setenvif:
  |  - Remove "support" for Remote_User variable which never worked at
  |    all. PR 25725.
  |  - Extend the SetEnvIf directive to capture subexpressions of the
  |    matched value.
  | mod_headers:
  |  - Backport from 2.1 / Regression from 1.3: mod_headers now knows
  |    again the functionality of the ErrorHeader directive. But
  |    instead using this misnomer additional flags to the Header
  |    directive were introduced ("always" and "onsuccess", defaulting
  |    to the latter).  PR 28657.
  | mod_usertrack:
  |  - Escape the cookie name before pasting into the regexp.
  | mod_dir:
  |  - the trailing-slash behaviour is now configurable using the
  |    DirectorySlash directive.
  | util_ldap:
  |  - Switched the lock types on the shared memory cache from thread
  |    reader/writer locks to global mutexes in order to provide cross
  |    process cache protection.
  |  - Reworked the cache locking scheme to eliminate duplicate cache
  |    entries in the credentials cache due to race conditions.
  |  - Enhanced the util_ldap cache-info display to show more detail
  |    about the contents and current state of the cache.
  | mod_ldap:
  |  - Enable the option to support anonymous shared memory in
  |    mod_ldap.  This makes the cache work on Linux again.
  | miscellaneous:
  |  - Include directives no longer refuse to process symlinks on
  |    directories. Instead there's now a maximum nesting level of
  |    included directories (128 as distributed). This is configurable
  |    at compile time using the -DAP_MAX_INCLUDE_DIR_DEPTH switch.  PR
  |    28492, PR 28370.
  |  - Prevent CGI script output which includes a Content-Range header
  |    from being passed through the byterange filter.
  |  - Satisfy directives now can be influenced by a surrounding
  |    <Limit> container.  PR 14726.
  |  - Makefile fix: httpd is linked against LIBS given to the 'make'
  |    invocation.  PR 7882.
  |  - suexec: Pass the SERVER_SIGNATURE envvar through to CGIs.
  |  - apachectl: Fix a problem finding envvars if sbindir != bindir.
  |    PR 30723.
  |  - Use the higher performing 'httpready' Accept Filter on all
  |    platforms except FreeBSD < 4.1.1.
  |  - Allow proxying of resources that are invoked via DirectoryIndex.
  |    PR 14648, 15112, 29961.
  |  - Small fix to allow reverse proxying to an ftp server. Previously
  |    an attempt to do this would try and connect to 0.0.0.0,
  |    regardless of the server specified. PR 24922
  |  - Enable special ErrorDocument value 'default' which restores the
  |    canned server response for the scope of the directive.
  |  - work around MSIE Digest auth bug - if
  |    AuthDigestEnableQueryStringHack is set in r->subprocess_env
  |    allow mismatched query strings to pass.  PR 27758.
  |  - Accept URLs for the ServerAdmin directive. If the supplied
  |    argument is not recognized as an URL, assume it's a mail
  |    address.  PR 28174.
  |  - initialize server arrays prior to calling
  |    ap_setup_prelinked_modules so that static modules can push
  |    Defines values when registering hooks just like DSO modules can
- drop obsolete security fixes
    httpd-2.0.50-CAN-2004-0751-mod_ssl-proxied-request-segfault.dif
    httpd-2.0.50-CAN-2004-0748-mod_ssl-input-filter-infinite-loop.dif
    httpd-2.0.50-CAN-2004-0747-ENVVAR.dif
    httpd-2.0.50-CAN-2004-0786-apr_uri_parse-IPv6-address-validation.dif
    httpd-2.0.50-CAN-2004-0809-mod_dav-crash.dif
- httpd-2.0.45-anon-mmap.dif included upstream
* Tue Sep 14 2004 poeml@suse.de
- security fix [CAN-2004-0809 (cve.mitre.org)]: fix possible DoS in
  mod_dav by remotely triggerable null-pointer dereference
  http://nagoya.apache.org/bugzilla/show_bug.cgi?id=31183 [#45231]
- fix hint about vhost checking in the SSL readme
* Wed Sep  8 2004 poeml@suse.de
- security fix [CAN-2004-0786 (cve.mitre.org)]: fix a vulnerability
  in the apr-util library (lacking input validation on IPv6 literal
  addresses in the apr_uri_parse function [#44736]
- security fix [CAN-2004-0747 (cve.mitre.org)]: fix a buffer
  overflow that can occur when expanding ${ENVVAR} constructs in
  .htaccess or httpd.conf files. [#44736]
* Mon Sep  6 2004 poeml@suse.de
- rename check_forensic script to avoid clash with apache 1.3.x
  package
* Fri Aug 27 2004 poeml@suse.de
- implement action "startssl" in the init script. [#42365]
- add /usr/bin/check_forensic script to evaluate mod_log_forensic logs.
- disable building of leader and metuxmpm MPMs.
* Wed Aug 25 2004 poeml@suse.de
- security fix [CAN-2004-0748 (cve.mitre.org)]: fix a potential
  infinite loop in the SSL input filter which can be triggered by
  an aborted connection
  http://nagoya.apache.org/bugzilla/show_bug.cgi?id=29964 [#44103]
- security fix [CAN-2004-0751 (cve.mitre.org)]: fix a potential
  segfault in the SSL input filter which can be triggered by the
  response to request which is proxied to a remote SSL server
  http://nagoya.apache.org/bugzilla/show_bug.cgi?id=30134 [#44103]
- remove the obsolete notify message on package update
* Thu Jul  8 2004 poeml@suse.de
- update to 2.0.50. Relevant changes:
  | SECURITY: CAN-2004-0493 (cve.mitre.org)
  |   Close a denial of service vulnerability identified by Georgi
  |   Guninski which could lead to memory exhaustion with certain
  |   input data.
  | SECURITY: CAN-2004-0488 (cve.mitre.org)
  |   mod_ssl: Fix a buffer overflow in the FakeBasicAuth code for
  |   a (trusted) client certificate subject DN which exceeds 6K in
  |   length.
  | mod_alias:
  |   now emits a warning if it detects overlapping *Alias* directives.
  | mod_cgi: Handle output on stderr during script execution on Unix
  |   platforms; preventing deadlock when stderr output fills pipe
  |   buffer.  Also fixes case where stderr from nph- scripts could be
  |   lost.  PR 22030, 18348.
  | mod_dav:
  |  - Fix a problem that could cause crashes when manipulating locks
  |    on some platforms.
  | mod_dav_fs:
  |  - Fix MKCOL response for missing parent collections, which caused
  |    issues for the Eclipse WebDAV extension.  PR 29034.
  | mod_deflate:
  |  - Fix memory consumption (which was proportional to the response
  |    size).  PR 29318.
  | mod_expires:
  |  - Fix segfault which occured under certain circumstances. PR 28047.
  | mod_headers:
  |  - no longer crashes if an empty header value should be added.
  | mod_log_forensic:
  |  - new module.
  | mod_logio:
  |  - no longer removes the EOS bucket. PR 27928.
  | mod_proxy:
  |  - Fix handling of IPv6 numeric strings.
  | mod_rewrite:
  |   no longer turns forward proxy requests into reverse proxy
  |   requests. PR 28125
  | mod_ssl:
  |  - Log the errors returned on failure to load or initialize a
  |    crypto accelerator engine.
  |  - Fix a potential segfault in the 'shmcb' session cache for small
  |    cache sizes.  PR 27751.
  |  - Fix memory leak in session cache handling.  PR 26562
  |  - Fix potential segfaults when performing SSL shutdown from a pool
  |    cleanup.  PR 27945.
  | mod_auth_ldap/util_ldap:
  |  - allow relative paths for LDAPTrustedCA to be resolved against
  |    ServerRoot PR#26602
  |  - Throw an error message if an attempt is made to use the
  |    LDAPTrustedCA or LDAPTrustedCAType directives in a VirtualHost.
  |    PR 26390
  |  - Fix a potential segfault if the bind password in the LDAP cache
  |    is NULL.  PR 28250.
  |  - Overhaul handling of LDAP error conditions, so that the
  |    util_ldap_* functions leave the connections in a sane state
  |    after errors have occurred. PR 27748, 17274, 17599, 18661,
  |    21787, 24595, 24683, 27134, 27271
  |  - mod_ldap calls ldap_simple_bind_s() to validate the user
  |    credentials.  If the bind fails, the connection is left in an
  |    unbound state.  Make sure that the ldap connection record is
  |    updated to show that the connection is no longer bound.
  |  - Update the bind credentials for the cached LDAP connection to
  |    reflect the last bind.  This prevents util_ldap from creating
  |    unnecessary connections rather than reusing cached connections.
  |  - Quotes cannot be used around require group and require dn
  |    directives, update the documentation to reflect this. Also add
  |    quotes around the dn and group within debug messages, to make it
  |    more obvious why authentication is failing if quotes are used in
  |    error.  PR 19304.
  | miscellaneous:
  |  - Allow RequestHeader directives to be conditional. PR 27951.
  |  - Allow LimitRequestBody to be reset to unlimited. PR 29106
  |  - <VirtualHost myhost> now applies to all IP addresses for myhost
  |    instead of just the first one reported by the resolver.  This
  |    corrects a regression since 1.3.
  |  - Fix a bunch of cases where the return code of the regex compiler
  |    was not checked properly. This affects: mod_setenvif,
  |    mod_usertrack, mod_proxy, mod_proxy_ftp and core. PR 28218.
  |  - Remove 2Gb log file size restriction on some 32-bit platforms.
  |    PR 13511.
  |  - htpasswd no longer refuses to process files that contain empty
  |    lines.
  |  - Regression from 1.3: At startup, suexec now will be checked for
  |    availability, the setuid bit and user root. The works only if
  |    httpd is compiled with the shipped APR version (0.9.5).  PR
  |    28287.
  |  - Unix MPMs: Stop dropping connections when the file descriptor is
  |    at least FD_SETSIZE.
  |  - Fix a segfault when requests for shared memory fails and returns
  |    NULL. Fix a segfault caused by a lack of bounds checking on the
  |    cache.  PR 24801.
  |  - Ensure that lines in the request which are too long are properly
  |    terminated before logging.
  |  - htpasswd: use apr_temp_dir_get() and general cleanup
  |  - logresolve: Allow size of log line buffer to be overridden at
  |    build time (MAXLINE).  PR 27793.
  |  - Fix the comment delimiter in htdbm so that it correctly parses
  |    the username comment.  Also add a terminate function to allow
  |    NetWare to pause the output before the screen is destroyed.
  |  - Fix crash when Apache was started with no Listen directives.
  |  - core_output_filter: Fix bug that could result in sending garbage
  |    over the network when module handlers construct bucket brigades
  |    containing multiple file buckets all referencing the same open
  |    file descriptor.
  |  - Fix memory corruption problem with ap_custom_response()
  |    function.  The core per-dir config would later point to request
  |    pool data that would be reused for different purposes on
  |    different requests.
- drop obsolete patches
- change vendor string SuSE -> SUSE
* Tue Jun 29 2004 poeml@suse.de
- security fix [CAN-2004-0493 (cve.mitre.org)]: fix Denial of
  Service vulnaribility which could lead to memory exhaustion with
  certain input data. [#42566]
* Fri Jun 18 2004 poeml@suse.de
- package forgotten CHANGES file
- package apr and apr-util documentation files
- fix log_server_status2 to use perl's Socket module
* Wed May 19 2004 poeml@suse.de
- security fix for mod_ssl: fix buffer overflow in
  ssl_util_uuencode() [#40791]
* Wed Apr 28 2004 poeml@suse.de
- add TLS upgrade patch [#39449]
- add patch to allow writing log files larger than 2>GB [#39453]
- obsolete apache and mod_ssl versions only when older than what is
  shipped with 9.1
- don't provide mod_ssl
* Fri Apr  2 2004 cschum@suse.de
- Add "suse_help_viewer" provides [#37932]
* Mon Mar 29 2004 poeml@suse.de
- provide and obsolete packages apache, mod_ssl, apache-doc and
  apache-example-pages [#37084]
* Mon Mar 22 2004 poeml@suse.de
- disable large file support by not building with _FILE_OFFSET_BITS=64,
  in favour of retaining a binary compatible module API.
  Therefore, do not change the module magic number. LFS can be
  enabled by building via rpmbuild --define 'build_with_LFS 1'
* Thu Mar 18 2004 poeml@suse.de
- update to proposed 2.0.49 tarball
  - mod_cgid: Fix storage corruption caused by use of incorrect pool.
  - docs update
- remove APACHE_DOCUMENT_ROOT from sysconfig.apache2 [#32635]
- fix a comment in default-server.conf
- remove obsolete ssl_scache_cleanup support script and ftok helper
* Mon Mar 15 2004 poeml@suse.de
- change mmn in header file as well, for modules that include it
  from there
* Mon Mar 15 2004 poeml@suse.de
- update to 2.0.49-rc2. Relevant changes:
  | The whole codebase was relicensed and is now available under the
  |   Apache License, Version 2.0 (http://www.apache.org/licenses).
  |   [Apache Software Foundation]
  | Security [CAN-2004-0113 (cve.mitre.org)]: mod_ssl: Fix a memory
  |   leak in plain-HTTP-on-SSL-port handling.  PR 27106.
  | Security [CAN-2003-0020 (cve.mitre.org)]: Escape arbitrary data
  |   before writing into the errorlog. Unescaped errorlogs are still
  |   possible using the compile time switch
  |   "-DAP_UNSAFE_ERROR_LOG_UNESCAPED".
  | mod_ssl:
  |  - Send the Close Alert message to the peer before closing the
  |    SSL session.  PR 27428.
  |  - Fix bug in passphrase handling which could cause spurious
  |    failures in SSL functions later.  PR 21160.
  |  - Fix potential segfault on lookup of SSL_SESSION_ID.  PR 15057.
  |  - Fix streaming output from an nph- CGI script. PR 21944
  |  - Advertise SSL library version as determined at run-time rather
  |    than at compile-time.  PR 23956.
  |  - Fix segfault on a non-SSL request if the 'c' log format code
  |    is used.  PR 22741.
  |  - Fix segfaults at startup if other modules which use OpenSSL
  |    are also loaded.
  |  - Use human-readable OpenSSL error strings in logs; use
  |    thread-safe interface for retrieving error strings.
  | mod_cache:
  |  - Fixed cache-removal order in mod_mem_cache.
  |  - Fix segfault in mod_mem_cache cache_insert() due to cache size
  |    becoming negative.  PR: 21285, 21287
  |  - Modified the cache code to be header-location agnostic. Also
  |    fixed a number of other cache code bugs related to PR 15852.
  |    Includes a patch submitted by Sushma Rai <rsushma novell.com>.
  |    This fixes mod_mem_cache but not mod_disk_cache yet so I'm not
  |    closing the PR since that is what they are using.
  | mod_dav:
  |  - Reject requests which include an unescaped fragment in the
  |    Request-URI.  PR 21779.
  |  - Use bucket brigades when reading PUT data. This avoids
  |    problems if the data stream is modified by an input filter. PR
  |    22104.
  |  - Return a WWW-auth header for MOVE/COPY requests where the
  |    destination resource gives a 401.  PR 15571.
  |  - Fix a problem with namespace mappings being dropped in
  |    mod_dav_fs; if any property values were set which defined
  |    namespaces these came out mangled in the PROPFIND response.
  |    PR 11637.
  | mod_expires:
  |  - Initialize ExpiresDefault to NULL instead of "" to avoid
  |    reporting an Internal Server error if it is used without
  |    having been set in the httpd.conf file. PR: 23748, 24459
  |  - Add support for IMT minor-type wildcards (e.g., text/*) to
  |    ExpiresByType.  PR#7991
  | mod_log_config / logging:
  |  - Fix some piped log problems: bogus "piped log program '(null)'
  |    failed" messages during restart and problem with the logger
  |    respawning again after Apache is stopped.  PR 21648, PR 24805.
  |  - mod_log_config: Fix corruption of buffered logs with threaded
  |    MPMs.  PR 25520.
  |  - mod_log_config: Log the minutes component of the timezone correctly.
  |    PR 23642.
  | mod_proxy*:
  |  - proxy_http fix: mod_proxy hangs when both KeepAlive and
  |    ProxyErrorOverride are enabled, and a non-200 response without a
  |    body is generated by the backend server. (e.g.: a client makes a
  |    request containing the "If-Modified-Since" and "If-None-Match"
  |    headers, to which the backend server respond with status 304.)
  |  - Fix memory leak in handling of request bodies during reverse
  |    proxy operations.  PR 24991.
  |  - mod_proxy: Fix cases where an invalid status-line could be sent
  |    to the client.  PR 23998.
  | mod_rewrite:
  |  - Catch an edge case, where strange subsequent RewriteRules
  |    could lead to a 400 (Bad Request) response.
  |  - Make REMOTE_PORT variable available in mod_rewrite.  PR 25772.
  |  - In external rewrite maps lookup keys containing
  |    a newline now cause a lookup failure. PR 14453.
  |  - Fix RewriteBase directive to not add double slashes.
  | mod_usertrack:
  |  - Fix bug in mod_usertrack when no CookieName is set.
  |  - mod_usertrack no longer inspects the Cookie2 header for
  |    the cookie name. PR 11475.
  |  - mod_usertrack no longer overwrites other cookies.
  |    PR 26002.
  | mod_include, filters:
  |  - Backport major overhaul of mod_include's filter parser from 2.1.
  |    The new parser code is expected to be more robust and should
  |    catch all of the edge cases that were not handled by the previous one.
  |    The 2.1 external API changes were hidden by a wrapper which is
  |    expected to keep the API backwards compatible.
  |  - Add a hook (insert_error_filter) to allow filters to re-insert
  |    themselves during processing of error responses. Enable mod_expires
  |    to use the new hook to include Expires headers in valid error
  |    responses. This addresses an RFC violation. It fixes PRs 19794,
  |    24884, and 25123.
  |  - complain via error_log when mod_include's INCLUDES filter is
  |    enabled, but the relevant Options flag allowing the filter to run
  |    for the specific resource wasn't set, so that the filter won't
  |    silently get skipped. next remove itself, so the warning will be
  |    logged only once
  |  - Fix mod_include's expression parser to recognize strings correctly
  |    even if they start with an escaped token.
  |  - Fix a problem with the display of empty variables ("SetEnv foo") in
  |    mod_include.  PR 24734
  |  - mod_include no longer allows an ETag header on 304 responses.
  |    PR 19355.
  | mod_autoindex:
  |  - Don't omit the <tr> start tag if the SuppressIcon option is
  |    set. PR 21668.
  |  - Restore the ability to add a description for directories that
  |    don't contain an index file.  (Broken in 2.0.48)
  |  - mod_autoindex / core: Don't fail to show filenames containing
  |    special characters like '%%'. PR 13598.
  |  - Add 'XHTML' option in order to allow switching between HTML
  |    3.2 and XHTML 1.0 output. PR 23747.
  | mod_status:
  |  - Add mod_status hook to allow modules to add to the mod_status
  |    report.
  |  - Report total CPU time accurately when using a threaded MPM.
  |    PR 23795.
  | mod_info:
  |  - Fix mod_info to use the real config file name, not the default
  |    config file name.
  |  - HTML escape configuration information so it displays
  |    correctly. PR 24232.
  | mod_auth_digest:
  |  - Allow mod_auth_digest to work with sub-requests with different
  |    methods than the original request.  PR 25040.
  | mod_auth_ldap:
  |  - Fix some segfaults in the cache logic.  PR 18756.
  | mod_cgid:
  |  - Restart the cgid daemon if it crashes.  PR 19849
  | mod_setenvif:
  |  - Fix the regex optimizer, which under circumstances
  |    treated the supplied regex as literal string. PR 24219.
  | miscellaneous:
  |  - core.c: If large file support is enabled, allow any file that is
  |    greater than AP_MAX_SENDFILE to be split into multiple buckets.
  |    This allows Apache to send files that are greater than 2gig.
  |    Otherwise we run into 32/64 bit type mismatches in the file size.
  |  - Fixed file extensions for real media files and removed rpm extension
  |    from mime.types. PR 26079.
  |  - Remove compile-time length limit on request strings. Length is
  |    now enforced solely with the LimitRequestLine config directive.
  |  - Set the scoreboard state to indicate logging prior to running
  |    logging hooks so that server-status will show 'L' for hung loggers
  |    instead of 'W'.
  |  - Fix the inability to log errors like exec failure in
  |    mod_ext_filter/mod_cgi script children.  This was broken after
  |    such children stopped inheriting the error log handle.
  |  - fix "Expected </Foo>> but saw </Foo>" errors in nested,
  |    argumentless containers.
  |  - ap_mpm.h: Fix include guard of ap_mpm.h to reference mpm
  |    instead of mmn.
  |  - Add Polish translation of error messages.  PR 25101.
  |  - Add AP_MPMQ_MPM_STATE function code for ap_mpm_query.
  |  - Fix htdbm to generate comment fields in DBM files correctly.
  |  - Correct UseCanonicalName Off to properly check incoming port number.
  |  - Fix slow graceful restarts with prefork MPM.
  |  - Keep focus of ITERATE and ITERATE2 on the current module when
  |    the module chooses to return DECLINE_CMD for the directive.
  |    PR 22299.
  |  - Build array of allowed methods with proper dimensions, fixing
  |    possible memory corruption.
  |  - worker MPM: fix stack overlay bug that could cause the parent
  |    process to crash.
  |  - Add XHTML Document Type Definitions to httpd.h (minor MMN bump).
  |  - Fix build with parallel make.  PR 24643.
  |  - Add fatal exception hook for use by diagnostic modules.  The hook
  |    is only available if the --enable-exception-hook configure parm
  |    is used and the EnableExceptionHook directive has been set to
  |    "on".
  |  - Improve 'configure --help' output for some modules.
- drop two hunks from httpd-2.0.47-headtail.dif (buildcheck.sh is
  fixed)
- disable automatic restarts, because they do not work properly
  [#35408]
- change MMN to prevent loading of incompatible modules (modules
  that are not built with `apxs -q CFLAGS` and therefore miss
  _FILE_OFFSET_BITS=64). Provide our old apache_mmn_20020903 in
  addition.
- use CPPFLAGS for passing preprocessor flags because they are
  removed from CFLAGS
- Stop dropping connections when the file descriptor
  is at least FD_SETSIZE. This isn't a problem on Linux because
  poll() is used instead of select() by APR. Assert HAVE_POLL.
  [#34178]
- add modifications to the code to the NOTICE file as required by
  the new license
* Fri Feb 27 2004 poeml@suse.de
- compile with -DSSL_EXPERIMENTAL_ENGINE to allow usage of hardware
  crypto accelerators
- compile with -DMAX_SERVER_LIMIT=200000
- if an SSL passphrase is not entered within the timeout, fall back
  to start apache without SSL (with -D NOSSL). This could/should be
  made configurable.
- clean up output of SuSEconfig.apache2
- add pre-defined LogFormat "vhost_combined"
- configure /var/lib/apache2 for WebDAV locks
- add a readme about configuring WebDAV with digest authentication
- add default configuration for mod_usertrack (this is the current
  workaround for the problem in the 1.3.29/2.0.48 release that
  occurs if no CookieName is configured)
- in vhost.template, enclose all virtual host configuration in the
  VirtualHost container
- update metuxmpm patch to r7
- fix test run as non-root
* Tue Jan 13 2004 schwab@suse.de
- Fix quoting in autoconf macros.
* Sat Dec 13 2003 poeml@suse.de
- add changes to gensslcert from Volker Kuhlmann [#31803]
- revert default character set from UTF-8 to ISO-8859-1, and revert
  the misleading comment that talked about filenames while it is
  all about content of the files
* Tue Nov 18 2003 poeml@suse.de
- add a ServerLimit directive to server-tuning.conf, so it's
  already in the right place if someone needs to tweak it [#32852]
* Fri Nov  7 2003 poeml@suse.de
- mark apache2-manual.conf in %%files doc as %%config
- wrap directives specific to the mod_negotiation module into an
  <IfModule> block [#32848]
* Thu Oct 30 2003 poeml@suse.de
- update to 2.0.48. Relevant / user visible changes are:
  Security [CAN-2003-0789]: Resolve some mishandling of the AF_UNIX
    socket used to communicate with the cgid daemon and the CGI
    script.
  Security [CAN-2003-0542]: Fix buffer overflows in mod_alias and
    mod_rewrite which occurred if one configured a regular
    expression with more than 9 captures.
  mod_rewrite:
  - Don't die silently when failing to open RewriteLogs. PR 23416
  - Fix support of the [P] option to send rewritten request using
    "proxy:". The code was adding multiple "proxy:" fields in the
    rewritten URI. PR: 13946.
  - Ignore RewriteRules in .htaccess files if the directory
    containing the .htaccess file is requested without a trailing
    slash.  PR 20195.
  mod_include:
  - Fix a trio of bugs that would cause various unusual sequences
    of parsed bytes to omit portions of the output stream. PR 21095
  - fix segfault which occured if the filename was not set, for
    example, when processing some error conditions.
  mod_cgid: fix a hash table corruption problem which could
    result in the wrong script being cleaned up at the end of a
    request.
  mod_ssl: Fix segfaults after renegotiation failure. PR 21370
  - Fix a problem setting variables that represent the client
    certificate chain.  PR 21371
  - Fix FakeBasicAuth for subrequest.  Log an error when an
    identity spoof is encountered.
  - Assure that we block properly when reading input bodies with
    SSL.  PR 19242.
  mod_autoindex: If a directory contains a file listed in the
    DirectoryIndex directive, the folder icon is no longer replaced
    by the icon of that file. PR 9587.
  mod_usertrack: do not get false positive matches on the
    user-tracking cookie's name. PR 16661.
  mod_cache:
  - Fix the cache code so that responses can be cached if they
    have an Expires header but no Etag or Last-Modified headers.
    PR 23130.  cache_util: Fix ap_check_cache_freshness to check
    max_age, smax_age, and expires as directed in RFC 2616.
  mod_deflate:
  - fix to not call deflate() without checking first whether it
    has something to deflate. (Currently this causes deflate to
    generate a fatal error according to the zlib spec.) PR 22259.
  - Don't attempt to hold all of the response until we're done.
  - Fix a bug, where mod_deflate sometimes unconditionally
    compressed the content if the Accept-Encoding header
    contained only other tokens than "gzip" (such as "deflate").
    PR 21523.
  mod_proxy: Don't respect the Server header field as set by
    modules and CGIs.  As with 1.3, for proxy requests any such
    field is from the origin server; otherwise it will have our
    server info as controlled by the ServerTokens directive.
  mod_log_config: Fix %%b log format to write really "-" when 0
    bytes were sent (e.g. with 304 or 204 response codes).
  mod_ext_filter: Set additional environment variables for use by
    the external filter.  PR 20944.
  core:
  - allow <Foo>..</Foo> containers (no arguments in the opening
    tag), as in 1.3. Needed by mod_perl <Perl> sections
  - Fix a misleading message from the some of the threaded MPMs
    when MaxClients has to be lowered due to the setting of
    ServerLimit.
  - Avoid an infinite recursion, which occured if the name of an
    included config file or directory contained a wildcard
    character. PR 22194.
  - MPMs: The bucket brigades subsystem now honors the MaxMemFree
    setting.
  - Lower the severity of the "listener thread didn't exit"
    message to debug, as it is of interest only to developers.
  miscellaneous:
  - Update the header token parsing code to allow LWS between the
    token word and the ':' seperator.  [PR 16520]
  - Remember an authenticated user during internal redirects if
    the redirection target is not access protected and pass it to
    scripts using the REDIRECT_REMOTE_USER environment variable.
    PR 10678, 11602.
  - Update mime.types to include latest IANA and W3C types.
  - Modify ap_get_client_block() to note if it has seen EOS.
  ab:
  - Overlong credentials given via command line no longer clobber
    the buffer.
  - Work over non-loopback on Unix again. PR 21495.
  - Fix NULL-pointer issue in ab when parsing an incomplete or
    non-HTTP response. PR 21085.
- add another example to apache2-listen.conf
- update apache2-mod_mime-defaults.conf according to 2.0.48 changes
  (be clearer in describing the connection between AddType and
  AddEncoding for defining the meaning of compressed file
  extensions.)
- use a better example domain name in apache2-vhost-ssl.template
- the "define version_perl" was nowhere needed
* Mon Sep 22 2003 mls@suse.de
- don't provide httpddoc in apache2-doc
* Thu Sep 18 2003 poeml@suse.de
- add mod_php4 to the default list of APACHE_MODULES, and change
  get_module_list to ignore non-existant modules (warnings will
  be issued when it is run from SuSEconfig, but not from the init
  script). How to enable the PHP4 module has been the most
  frequently asked questions in user feedback [cf to #29735].
  This bug is tracked in [#31306]
- include conf.d/*.conf by default, as it was the case until
  recently. User feedback showed that for many people the
  separation of configuration includes into individual virtual
  hosts is overkill, and it complicates the setup too much. More
  finegrained control can be achieved by commenting out the
  respective line in the default server config. [#30866], [#29735]
- remove the FIXME at the end of httpd.conf (obsoleted by the above
  change), and place a strategical comment there about .local files
- add <IfDefine SSL> container around configuration in ssl template
* Tue Sep  9 2003 poeml@suse.de
- change comment in sysconfig template to work around a fillup bug
  [#30279]
* Mon Sep  8 2003 poeml@suse.de
- fix wrong variable name in a comment of the sysconfig template
- update README.QUICKSTART
- add README.QUICKSTART.SSL
* Mon Sep  8 2003 poeml@suse.de
- remove unused ENABLE_SUSECONFIG_APACHE from sysconfig template
* Fri Sep  5 2003 poeml@suse.de
- disallow UserDir for user root
- cope with "no" or "yes" as values for APACHE_SERVERSIGNATURE, as
  they were set on SuSE Linux 8.1
- add more documentation to README.QUICKSTART, also mentioning what
  might be too obvious: the document root [#29674]
- in %%post, diff to httpd.conf.default only when .rpmnew is present
- improve message sent on update
* Fri Aug 29 2003 poeml@suse.de
- improve documentation on configuration
- compile with -Wall
- do not obsolete httpddoc, which is provided by apache-doc package
  from apache1
- add conflict apache2-example-pages <-> apache-example-pages
- fix building on older distros
* Tue Aug 19 2003 poeml@suse.de
- use httpd-2.0.47-metuxmpm-r6.diff, previous one was broken by me
- don't force setting of a DocumentRoot, because the configuration
  of the default vhost already contains it
- when testing on SL 8.0, the www group has to be created as well
- when testing on even older systems, don't add buildroot to
  DocumentRoot in default-server.conf
* Fri Aug 15 2003 poeml@suse.de
- revamped configuration
  - add some CustomLog formats
  - AddDefaultCharset UTF-8 [#22427]
  - add activation metadata to sysconfig template [#28834]
  - default APACHE_MODULES: add mod_ssl, remove mod_status
  - new sysconfig variables: APACHE_USE_CANONICAL_NAME,
    APACHE_DOCUMENT_ROOT
  - get rid of the "suse_" prefix in generated config snippets, and
    place them below /etc/apache2/sysconfig.d/. On update, convert
    the Include statements in httpd.conf for the new locations
  - add /etc/apache2/vhosts.d and virtual host templates
  - the configuration for the manual is now seperate and installed
    together with apache2-doc (conf.d/apache2-manual.conf)
- add distilled wisdom in form of README.QUICKSTART
- change group of wwwrun user: nogroup -> www [#21782]
- proxycachedir and localstatedir should not be world readable
- set DEFAULT_PIDLOG to /var/run/httpd2.pid, so we don't need to
  configure the PidFile directive
- add -fno-strict-aliasing, due to warnings about code where
  dereferencing type-punned pointers will break strict aliasing
- clean the RPM_BUILD_ROOT, but not in the build system
- new macros for stop/restart of services on rpm update/removal,
  and improved try-restart section in rc.apache2
- get rid of "modules" subdir, and remove dead code from
  SuSEconfig.apache2
- add some tools: get_includes, find_httpd2_includes,
  apache2-reconfigure-mpm
- rename README.SuSE to README.{SuSE,UnitedLinux}
- include directories in filelists of MPM subpackages
- enclose package descriptions of MPMs in %%ifdef
- add a dependency of the MPM subpackages on the version of the
  main package
- build a new MPM: metuxmpm (httpd-2.0.47-metuxmpm.diff)
* Mon Jul 28 2003 poeml@suse.de
- add new sysconfig variables: APACHE_LOGLEVEL, APACHE_ACCESS_LOG,
  and remove the respective directives from httpd.conf.dist
- merge the ssl.conf.dif and httpd.conf.dif into one patch
* Sun Jul 27 2003 poeml@suse.de
- build with -D_FILE_OFFSET_BITS=64 when presumably the kernel
  supports sendfile64 [#22191, #22018]. Define APR_HAS_LARGE_FILES
  (which is unconditionally off, otherwise). Keep
  - D_LARGEFILE_SOURCE since some modules might need it.
- make sure the package can be built as ordinary user
- special case mod_auth_mysql since its module_id is reversed
- don't increase DYNAMIC_MODULE_LIMIT (64 should be copious)
- don't explicitely strip binaries since RPM handles it, and may
  keep the stripped information somewhere
- reformat the header of the spec file
- allow to pass a number-of-jobs parameter into spec file via rpm
  - -define 'jobs N'
* Thu Jul 10 2003 poeml@suse.de
- update to 2.0.47. relevant / user visible changes:
  Security [CAN-2003-0192]: Fixed a bug whereby certain sequences
    of per-directory renegotiations and the SSLCipherSuite
    directive being used to upgrade from a weak ciphersuite to a
    strong one could result in the weak ciphersuite being used in
    place of the strong one.
  Security [CAN-2003-0253]: Fixed a bug in prefork MPM causing
    temporary denial of service when accept() on a rarely accessed
    port returns certain errors.
  Security [CAN-2003-0254]: Fixed a bug in ftp proxy causing denial
    of service when target host is IPv6 but proxy server can't
    create IPv6 socket.  Fixed by the reporter.
  Security [VU#379828]: Prevent the server from crashing when entering
    infinite loops. The new LimitInternalRecursion directive
    configures limits of subsequent internal redirects and nested
    subrequests, after which the request will be aborted.  PR 19753+
  core:
    core_output_filter: don't split the brigade after a FLUSH
    bucket if it's the last bucket.  This prevents creating
    unneccessary empty brigades which may not be destroyed until
    the end of a keepalive connection.
  mod_cgid:
    Eliminate a double-close of a socket.  This resolves various
    operational problems in a threaded MPM, since on the second
    attempt to close the socket, the same descriptor was often
    already in use by another thread for another purpose.
  mod_negotiation:
    Introduce "prefer-language" environment variable, which allows
    to influence the negotiation process on request basis to prefer
    a certain language.
  mod_expire:
    Make ExpiresByType directive work properly, including for
    dynamically-generated documents.
- apr bugfixes
- more fixes of deprecated head/tail -1 calls
* Wed May 28 2003 poeml@suse.de
- update to 2.0.46. relevant / user visible changes:
  Security [CAN-2003-0245]: Fixed a bug that could be triggered
    remotely through mod_dav
  Security [CAN-2003-0189]: Fixed a denial-of-service
    vulnerability affecting basic authentication
  Security: forward port of buffer overflow fixes for htdigest.
  mod_ssl:
  - SSL session caching(shmht) : Fix a SEGV problem with SHMHT
    session caching.
  mod_deflate:
  - Add another check for already compressed content
  - Check also err_headers_out for an already set
    Content-Encoding: gzip header. This prevents gzip compressed
    content from a CGI script from being compressed once more.
  mod_mime_magic:
  - If mod_mime_magic does not know the content-type, do not
    attempt to guess.
  mod_rewrite:
  - Fix handling of absolute URIs.
  mod_log_config:
  - Add the ability to log the id of the thread processing the
    request via new %%P formats.
  mod_auth_ldap:
  - Use generic whitespace character class when parsing "require"
    directives, instead of literal spaces only.
  mod_proxy:
  - Fixed a segfault when multiple ProxyBlock directives were used.
  - Added AllowEncodedSlashes directive to permit control of
    whether the server will accept encoded slashes ('%%2f') in the
    URI path.  Default condition is off (the historical behaviour).
  - If Apache is started as root and you code CoreDumpDirectory,
    coredumps are enabled via the prctl() syscall.
  - htpasswd: Check the processed file on validity; add a delete flag.
- httpd-2.0.45-libtool-1.5.dif is obsolete
- mark suse_include.conf as %%ghost
- note the rebirth of the httpd and apachectl man pages (thanks to
  RPMv4 :)
- let the module RPM packages only depend on the _major_ module
  magic number, not on the minor
- fix some paths in config_vars.mk, which facilitates building of
  certain modules
* Wed May 14 2003 poeml@suse.de
- use mmap() via MAP_ANON as shared memory allocation method, to
  prevent restart problems with stale (or in use) files that are
  associated with shared memory
- package forgotten files, and remove hack in %%clean
- remove files from the build root that are not packaged
- remove suse_include.conf from filelist
* Fri May  9 2003 poeml@suse.de
- update to 2.0.45. relevant / user visible changes:
  Security:  Eliminated leaks of several file descriptors to
    child processes, such as CGI scripts.  This fix depends on the
    latest APR library release 0.9.2, which is distributed with the
    httpd source tarball for Apache 2.0.45.  PR 17206
  Security [CAN-2003-0132]: Close a Denial of Service
    vulnerability identified by David Endler <DEndler@iDefense.com>
    on all platforms.
  General:
  - Fix segfault which occurred when a section in an included
    configuration file was not closed. PR 17093.
  - Fix a nasty segfault in mmap_bucket_setaside() caused by
    passing an incompatible pointer type to mmap_bucket_destroy(void*).
  - prevent filters (such as mod_deflate) from adding garbage to
    the response. PR 14451.
  - Simpler, faster code path for request header scanning
  - Try to log an error if a piped log program fails.  Try to
    restart a piped log program in more failure situations.
  - Fix bug where 'Satisfy Any' without an AuthType lost all MIME
    information (and more). Related to PR 9076.
  - Fix If header parsing when a non-mod_dav lock token is passed to it.
  - Fix apxs to insert LoadModule directives only outside of
    sections.
  - apxs: Include any special APR ld flags when linking the DSO.
  suexec: Be more pedantic when cleaning environment. Clean it
    immediately after startup. PR 2790, 10449. Use saner default
    config values for suexec. PR 15713.
  mod_cgi, mod_cgid, mod_ext_filter: Log errors when scripts cannot
    be started on Unix because of such problems as bad permissions,
    bad shebang line, etc. Fix possible segfaults under obscure
    error conditions within the cgid daemon.
  mod_deflate:
  - you can now specify the compression level.
  - Extend the DeflateFilterNote directive to allow accurate
    logging of the filter's in- and outstream.
  - Fix potential memory leaks in mod_deflate on malformed data.  PR 16046.
  mod_ssl:
    Allow SSLMutex to select/use the full range of APR locking
    mechanisms available to it. Also, fix the bug that SSLMutex
    uses APR_LOCK_DEFAULT no matter what.  PR 8122
  mod_autoindex no longer forgets output format and enabled version
    sort in linked column headers.
  mod_rewrite:
  - Prevent endless loops of internal redirects in mod_rewrite by
    aborting after exceeding a limit of internal redirects. The
    limit defaults to 10 and can be changed using the
    RewriteOptions directive. PR 17462.
  - Allow "RewriteEngine Off" even if no "Options FollowSymlinks"
    (or SymlinksIfOwnermatch) is set. PR 12395.
  mod_ldap:
  - Updated mod_ldap and mod_auth_ldap to support the Novell LDAP
    SDK SSL and standardized the LDAP SSL support across the
    various LDAP SDKs.  Isolated the SSL functionality to
    mod_ldap rather than speading it across mod_auth_ldap and
    mod_ldap.  Also added LDAPTrustedCA and LDAPTrustedCAType
    directives to mod_ldap to allow for a more common method of
    specifying the SSL certificate.
  - fix fault when caching was disabled, and some memory leaks
  - Fix mod_ldap to open an existing shared memory file should
    one already exist. PR 12757.
  - Added character set support to mod_auth_LDAP to allow it to
    convert extended characters used in the user ID to UTF-8
    before authenticating against the LDAP directory. The new
    directive AuthLDAPCharsetConfig is used to specify the config
    file that contains the character set conversion table.
  mod_ssl:
  - Fixed mod_ssl's SSLCertificateChain initialization to no
    longer skip the first cert of the chain by default.  This
    misbehavior was introduced in 2.0.34.  PR 14560
  - Fix 64-bit problem in mod_ssl input logic.
  mod_proxy:
  - Hook mod_proxy's fixup before mod_rewrite's fixup, so that by
    mod_rewrite proxied URLs will not be escaped accidentally by
    mod_proxy's fixup. PR 16368
  - Don't remove the Content-Length from responses in mod_proxy PR: 8677
  mod_auth_digest no longer tries to guess AuthDigestDomain, if it's
    not specified. Now it assumes "/" as already documented. PR 16937.
  mod_file_cache: fix segfaults
- improve the start/restart section of the init script, and add a
  ssl_scache_cleanup script
- understand a syntax like -DSTATUS, as described in the sysconfig
  file help text (bug noted in #25404]
- don't package the *.exp files, as they are needed only on AIX
- fix filelist for usage of %%dir for files
- fix the cosmetical but irritating "Inappropriate ioctl for
  device" error message, when rcapache2 is called from within YaST
- remove the unused /etc/apache2/modules directory from the package
- remove the now unused --enable-experimental-libtool
- fix to build with libtool-1.5
* Wed Apr  9 2003 ro@suse.de
- fix deprecated head/tail call syntax "-1"
* Mon Mar 17 2003 kukuk@suse.de
- Remove suse_help_viewer from provides [Bug #25436]
* Thu Mar 13 2003 poeml@suse.de
- security fix: do not write the startup log file to a world
  writable directory, reversing the change of Jan 23 (wasn't in any
  released package) [#25239]
* Mon Mar 10 2003 poeml@suse.de
- change permissions of /var/log/apache2 from wwwrun:root mode 770
  to root:root mode 750 [#24951]
- fix wrong list() in sysconfig.apache2 [#24719], and add a missing
  default value
* Mon Mar  3 2003 kukuk@suse.de
- Remove ghost entry for pid file [Bug #24566]
* Thu Feb 27 2003 poeml@suse.de
- use the official MIME types, which are more complete [#23988]
* Mon Feb 24 2003 poeml@suse.de
- don't include log files into the package, and don't touch them in
  %%post; it's not needed
- fix comment in httpd.conf talking about SuSEconfig
- adjust some variable types in the sysconfig template
* Tue Feb 18 2003 poeml@suse.de
- apache2 Makefiles do support DESTDIR now, so let's use that
  instead of the explicit paths (fixes a wrong path in
  config_vars.mk [#23699]).  Some files (*.exp, libapr*) are
  automatically installed in the right location now.
* Fri Feb 14 2003 poeml@suse.de
- fix configuration script to find apache modules on 64 bit archs
- mark ssl.conf (noreplace)
* Mon Feb 10 2003 poeml@suse.de
- add mod_ldap, mod_auth_ldap, but link only them against the LDAP
  libs. Likewise, do not link everything against ssl libs. This way
  we can avoid RPM package (and build) requirements on a lot of
  libs for subversion and other packages that build on apache.
- move more code from SuSEconfig into rcapache2 (actually into
  support scripts below /usr/share/apache2/, so apache2 can be
  configured without starting it)
- improve full-server-status once again
- remove suse_loadmodule.conf from filelist
- remove obsolete README.modules
- rename LOADMODULES -> APACHE_MODULES
- add APACHE_BUFFERED_LOGS
- update README.SuSE
* Tue Jan 28 2003 poeml@suse.de
- rc.apache2
  - add extreme-configtest (trying to run server as nobody, which
    detects _all_ config errors)
  - evaluate LOADMODULES from sysconfig.apache2 on-the-fly from
    rcapache2 instead of SuSEconfig
  - when restarting, do something useful instead of 'sleep 3': wait
    just as long until the server has terminated all children
* Sun Jan 26 2003 poeml@suse.de
- build mod_logio, mod_case_filter, mod_case_filter_in
- rename apr subpackage to libapr0 (the library is called libapr-0
  meanwhile). add compatibility links named (libapr{,util}.so.0)
- configure SSL session caching with shm circular buffer
    SSLSessionCache         shm:/var/lib/httpd/ssl_scache
    SSLSessionCacheTimeout  600
    SSLMutex  sem
- SuSEconfig.apache2: prefer prefork MPM over worker, if guessing
- strip objects
- rename gensslcert2 to gensslcert
- show a list all available modules in /etc/sysconfig/apache2
- nicer output of apache2ctl
- reorder Requires
* Thu Jan 23 2003 poeml@suse.de
- update to 2.0.44
- obsoletes patch httpd-2.0.43-mod_ssl-memory-leak.dif
- the apachectl and httpd man pages have been dropped upstreams
- add robots.txt to the example-pages subpackage that blocks spiders
- disable the perchild MPM
- disable httpd-2.0.36-64bit.dif
- rename apachectl2 to apache2ctl
- write the startup log to /var/tmp instead of /var/log/apache2
* Sun Jan 12 2003 poeml@suse.de
- fix last fix (rpm macro before hash wasn't expanded)
* Fri Jan 10 2003 poeml@suse.de
- fix lib64 path in SuSEconfig
* Fri Jan  3 2003 poeml@suse.de
- fix typo in spec file, preventing replacement of @userdir@ in
  httpd.conf-std.in
* Wed Dec 18 2002 poeml@suse.de
- sysconfig.apache2:
  - add APACHE_SERVER_FLAGS variable
  - change default: APACHE_SERVERSIGNATURE=on to match apache deflt
  - add APACHE_CONF_INCLUDE_DIRS
  - drop bogus APACHE_ACCESS_SERVERINFO variable
  - adapt to our new sysconfig template
- SuSEconfig.apache2:
  - understand LOADMODULES also if it is not an array [#21816]
  - be very flexible with regard to LOADMODULE input (e.g., say
    mod_php4 and it will find libphp4.so with ID php4_module)
  - also ignore *,v files
  - include APACHE_CONF_INCLUDE_DIRS
  - dump some files: suse_define.conf (not needed) & suse_text.conf
    (too much overhead)
- rc.apache2:
  - implement most of apachectl's commands (graceful, configtest)
  - use server_flags from sysconfig.apache2
  - pass server flags like -DSTATUS from the command line through
    to httpd2
  - add commmands to show the server status
  - don't quit silently when no apache MPM is installed
  - handle ServerSignature and other stuff on the command line
    (save modifications to httpd.conf)
- fix the /manual Alias that points to the documentation
- configure /cgi-bin for cgi execution
- configure /home/*/public_html for mod_userdir -- if it is loaded
- configure internationalized error responses
- fix apachectl2
- add /etc/apache2/{,modules} to the filelist
- add /etc/apache2/conf.d as drop-in directory for packages
- hard code some more default paths into the executable
- finally, run a test!
* Thu Dec  5 2002 poeml@suse.de
- move ap{r,u}-config* into the apr package, as well
- add generic ap{r,u}-config
- add %%includedir to filelist
* Wed Dec  4 2002 poeml@suse.de
- more checks and warnings to SuSEconfig.apache2
- shift APR files into the the apr package
- try 1.136 revision of perchild.c
* Tue Dec  3 2002 poeml@suse.de
- add forgotten ssl.conf to the filelist (thanks, Robert)
- add httpd-2.0.43-mod_ssl-memory-leak.dif
* Mon Oct 14 2002 poeml@suse.de
- update to 2.0.43, that fixes a Cross-Site Scripting bug (CVE:
  CAN-2002-0840)
* Mon Oct  7 2002 poeml@suse.de
- do not append a '2' suffix to the scripts included with the
  documentation
- move error, icons and manual dir to /usr/share/apache2
- fix nested array in SuSEconfig.apache2
- let SuSEconfig pick one MPM that is installed. Do not default to
  "worker". [#20724]
* Thu Oct  3 2002 poeml@suse.de
- update to 2.0.42 (primarily a bug-fix release, including updates
  to the experimental caching module, the removal of several memory
  leaks, and fixes for several segfaults, one of which could have
  been used as a denial-of-service against mod_dav (VU#406121).)
- increase flexibility of the spec file: build any set of MPMs,
  depending on RPM %%defines. Improve the mechanism that merges the
  modules so it works with any number of MPMs.
- use a "Server:" header that fits the product apache is built for
- add an RPM dependency on the module magic number to the MPM
  subpackages
- build the "leader/follower" MPM. On i686, enable nonportable but
  faster atomics for it.
- use filelists for more flexibility. APRVARS ceased to exist.
  Don't add README* twice.
- perchild: use AcceptMutex fcntl to prevent permission conflict as
  suggested in Apache Bugzilla #7921
- remove mod_rewrite and mod_proxy from the default modules
- build the mod_auth_digest module
* Mon Sep  9 2002 poeml@suse.de
- add patch that changes PLATFORM (as seen in the HTTP Server
  header) from "Unix" to "SuSE/Linux" [#18543]
- add README.SuSE, explaining how to build modules with apxs2
- fixed some paths in README.modules, put it into docdir and mark
  it as %%doc
* Wed Aug 28 2002 poeml@suse.de
- new package, now building all three MPMs and putting all specific
  modules in specific directories. Branch a subpackage for each
  MPM, containing the server and MPM-specific modules.
- branch apr package off, so apache2 doesn't need to be installed
  to have the libs. (apr is not released yet, that's why we build
  it here)
- allow coexistence of apache1 by using directories named apache2
  or suffixed with "2"
- allow building modules via apxs2 (for all server MPMs) --- or via
  apxs2-{worker,perchild,prefork} for a specific server MPM
- add permissions.apache2 setting /usr/sbin/suexec2 to 4755
- rewrite SuSEconfig.apache2 for apache 2.
- add httpd-2.0.40-cache_util.c.diff that prevents a segfault in
  mod_proxy when given an invalid URL
- branch apache2-example-pages off (docroot contents)
* Mon Aug 19 2002 poeml@suse.de
- actually use the new SuSE81 layout, and add SuSE81_64 layout
- cleaned up httpd-2.0.36-conf.dif
- fixed comment in SuSEconfig.apache
- drop SuSEconfig subpackage
- split main package and -devel package in three packages, one for
  each MPM...
  apache2       -> apache2-{worker,perchild,prefork}
  apache2-devel -> apache2-{worker,perchild,prefork}-devel
* Mon Aug 12 2002 poeml@suse.de
- bugfix update to 2.0.40
- fix Requires of -devel subpackage
- add variable to sysconfig.apache to switch off SuSEconfig.apache
- add new layout SUSE81 to config.layout due to the moved server
  root (so the old SuSE6.1 can be kept for building on older
  distributions)
- one of the lib64 path fixes could be removed, now included
  upstream
* Wed Aug  7 2002 poeml@suse.de
- put PreReq in an if-statement to allow building on older distris
- relax the Requires
- the apache_mmn macro had to be moved down in the spec file to be
  evaluated
- libmm is not needed for building (and it is not threadsafe)
- fix config.layout for the moved server root
* Fri Aug  2 2002 poeml@suse.de
- fix libdir in config.layout for lib64
* Fri Aug  2 2002 poeml@suse.de
- fix RPM Requires
* Thu Aug  1 2002 poeml@suse.de
- move datadir (i.e., ServerRoot) from /usr/local/httpd to /srv/www
- drop obsolete README.SuSE
* Wed Jul 31 2002 poeml@suse.de
- spec file: use PreReq
- don't delete SuSEconfig's md5 files in %%post, that's no good
- add apache.logrotate
- provide the magic module number as executable script
  (/usr/lib/apache/MMN) and as RPM Provides, indicating API changes
- mark httpd.conf noreplace
- fix installbuilddir in config.layout, needed for apxs
* Sun Jul 14 2002 poeml@suse.de
- update to 2.0.39
- drop obsolete moduledir and apxs patches
- rc.apache INIT section: use X-UnitedLinux-Should-Start
* Tue Jul  2 2002 ro@suse.de
- rename to "apache2" again
* Tue Jun 11 2002 ro@suse.de
- get apxs to work:
    include needed files in devel package
    adapt some pathes in apxs
* Wed May 29 2002 poeml@suse.de
- update to 2.0.36
- drop mod_ssl subpackage; mod_ssl is part of the apache bsae
  distribution now
- RPM can be built as user now
- SuSEconfig.apache: understand relative and absolute pathnames
- disable experimental auth_digest_module
