(function ContextMenu($angular, $document) {

    "use strict";

    /**
     * @module ngContextMenu
     * @author Adam Timberlake
     * @link https://github.com/Wildhoney/ngContextMenu
     */
    var module = $angular.module('ngContextMenu', []);

    /**
     * @module ngContextMenu
     * @service ContextMenu
     * @author Adam Timberlake
     * @link https://github.com/Wildhoney/ngContextMenu
     */
    module.factory('contextMenu', ['$rootScope', function contextMenuService($rootScope) {

        /**
         * @method cancelAll
         * @return {void}
         */
        function cancelAll() {
            $rootScope.$broadcast('context-menu/close');
        }

        return { cancelAll: cancelAll, eventBound: false };

    }]);

    /**
     * @module ngContextMenu
     * @directive contextMenu
     * @author Adam Timberlake
     * @link https://github.com/Wildhoney/ngContextMenu
     */
    module.directive('contextMenu', ['$http', '$timeout', '$interpolate', '$compile', 'contextMenu',

        function contextMenuDirective($http, $timeout, $interpolate, $compile, contextMenu) {

            return {

                /**
                 * @property restrict
                 * @type {String}
                 */
                restrict: 'EA',

                /**
                 * @property scope
                 * @type {Boolean}
                 */
                scope: true,

                /**
                 * @property require
                 * @type {String}
                 */
                require: '?ngModel',

                /**
                 * @method link
                 * @param {Object} scope
                 * @param {angular.element} element
                 * @param {Object} attributes
                 * @param {Object} model
                 * @return {void}
                 */
                link: function link(scope, element, attributes, model) {

                    if (!contextMenu.eventBound) {

                        // Bind to the `document` if we haven't already.
                        $document.addEventListener('click', function click() {
                            contextMenu.cancelAll();
                            scope.$applyAsync();
                        });

                        contextMenu.eventBound = true;

                    }

                    /*****************************************************/
                    // *** Dell change ***
                    // Check if there are any modal parents (find first using 'parents').
                    // Modal parents have an offset that affects the positioning of the menu and can cause the context menu to appear in an unexpected position.
                    var modalParent = element.parents('div[class^="modal-dialog"]');
                    _.set(scope, 'offset.x', _.get(modalParent, '[0].offsetLeft', 0));
                    _.set(scope, 'offset.y', _.get(modalParent, '[0].offsetTop', 0));
                    /*****************************************************/

                    /**
                     * @method closeMenu
                     * default and for click event
                     * @return {void}
                     */
                    function closeMenu($event) {
                        var a = $event;

                        if (scope.menu) {
                            scope.menu.remove();
                            scope.menu = null;
                            scope.position = null;
                        }
                        /*****************************************************/
                        // *** Dell change ***
                        // We added a click handler to open the menu. Stop propagation when we close it. Otherwise
                        // the context menu will reappear.
                        if (a.stopPropagation) {
                            a.stopPropagation();
                        }
                        /*****************************************************/
                    }

                    /*****************************************************/
                    // *** Dell change ***
                    //for mouseleave event
                    //must check position of event vs menu
                    function closeMenuMouseLeave($event) {
                        var e = $event;

                        if (scope.menu) {
                            if (scope.menu[0])
                                if (e.offsetX > 0
                                    && e.offsetY > 0
                                    && e.offsetX < scope.menu[0].clientWidth
                                    && e.offsetY < scope.menu[0].clientHeight)
                                    return;

                            scope.menu.remove();
                            scope.menu = null;
                            scope.position = null;
                        }
                    }
                    /*****************************************************/

                    scope.$on('context-menu/close', closeMenu);
                    scope.$on('$destroy', closeMenu);

                    /**
                     * @method getModel
                     * @return {Object}
                     */
                    function getModel() {
                        return model ? $angular.extend(scope, model.$modelValue) : scope;
                    }

                    /**
                     * @method render
                     * @param {Object} event
                     * @param {String} [strategy="append"]
                     * @return {void}
                     */
                    function render(event, strategy) {
                        strategy = strategy || 'append';

                        /*****************************************************/
                        // *** Dell change ***
                        // do not re-render the menu if it already exists - several bugs on this - JIT121550
                        if (scope.menu) {
                            event.stopPropagation();
                            event.preventDefault();
                            return;
                        }
                        /*****************************************************/

                        if ('preventDefault' in event) {

                            contextMenu.cancelAll();
                            event.stopPropagation();
                            event.preventDefault();
                            scope.position = {x: event.clientX, y: event.clientY};
                            /*****************************************************/
                            // *** Dell change ***
                            // Offset based on any modal offsets.
                            scope.position.x -= _.get(scope, 'offset.x', 0);
                            scope.position.y -= _.get(scope, 'offset.y', 0);
                            /*****************************************************/
                        } else {

                            if (!scope.menu) {
                                return;
                            }

                        }

                        /*****************************************************/
                        // *** Dell change ***
                        //do not try to get the menu template if the template was not specified!
                        //placing this check here results in getting no template on right click
                        //if we place this check at the beginning of this function or even before binding render to the click handler
                        //then we instead get the default browser context menu
                        //we do not want that for consistency within controls using the context menu
                        if (!(attributes.contextMenu && attributes.contextMenu.length > 0))
                            return;
                        /*****************************************************/

                        $http.get(attributes.contextMenu, {cache: true}).then(function then(response) {
                            var compiled = $compile(response.data)($angular.extend(getModel())),
                                menu = $angular.element(compiled);

                            // Determine whether to append new, or replace an existing.
                            switch (strategy) {
                                case ('append'):
                                    element.append(menu);
                                    break;
                                default:
                                    scope.menu.replaceWith(menu);
                                    break;
                            }

                            menu.css({

                                position: 'fixed',
                                'z-index': 1000,
                                top: 0,
                                left: 0,
                                transform: $interpolate('translate({{x}}px, {{y}}px)')({
                                    x: scope.position.x, y: scope.position.y
                                })

                            });

                            scope.menu = menu;
                            scope.menu.bind('click', closeMenu);
                            scope.menu.bind('mouseleave', closeMenuMouseLeave);

                        });

                    }

                    /*****************************************************/
                    // *** Dell change ***
                    //determine if the given html element id has child items
                    //returns true if the html element at id has child items
                    //this can be used with ng-show to hide an empty list
                    scope.hasChildItems = function (id) {
                        var elemArray = $('#' + id);
                        if (elemArray == null || elemArray == undefined || elemArray.length == 0) return false;
                        var elem = elemArray[0];
                        if (elem == null || elem == undefined) return false;
                        if (elem.childElementCount == 0) {
                            return false;
                        }
                        return true;
                    }
                    /*****************************************************/

                    if (model) {

                        var listener = function listener() {
                            return model.$modelValue;
                        };

                        // Listen for updates to the model...
                        scope.$watch(listener, function modelChanged() {
                            render({}, 'replaceWith');
                        }, true);

                    }

                    element.bind(attributes.contextEvent || 'contextmenu', render);
                }
            }
        }]);

})(window.angular, window.document);
