/*
 * Copyright (c) 2006 JMockit developers
 * This file is subject to the terms of the MIT license (see LICENSE.txt).
 */
package mockit.internal.faking;

import static mockit.internal.util.Utilities.NO_ARGS;

import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;

import java.lang.reflect.Constructor;
import java.lang.reflect.Member;

import mockit.internal.BaseInvocation;
import mockit.internal.state.TestRun;

import org.checkerframework.checker.index.qual.NonNegative;

/**
 * An invocation to a {@linkplain mockit.Mock fake} method.
 */
public final class FakeInvocation extends BaseInvocation {
    @NonNull
    private final FakeState fakeState;
    @NonNull
    private final String fakedClassDesc;
    @NonNull
    private final String fakedMethodName;
    @NonNull
    private final String fakedMethodDesc;
    boolean proceeding;

    @NonNull // called by generated bytecode
    public static FakeInvocation create(@Nullable Object invokedInstance, @Nullable Object[] invokedArguments,
            @NonNull String fakeClassDesc, @NonNegative int fakeStateIndex, @NonNull String fakedClassDesc,
            @NonNull String fakedMethodName, @NonNull String fakedMethodDesc) {
        Object fake = TestRun.getFake(fakeClassDesc, invokedInstance);
        assert fake != null;
        FakeState fakeState = TestRun.getFakeStates().getFakeState(fake, fakeStateIndex);
        Object[] args = invokedArguments == null ? NO_ARGS : invokedArguments;
        return new FakeInvocation(invokedInstance, args, fakeState, fakedClassDesc, fakedMethodName, fakedMethodDesc);
    }

    FakeInvocation(@Nullable Object invokedInstance, @NonNull Object[] invokedArguments, @NonNull FakeState fakeState,
            @NonNull String fakedClassDesc, @NonNull String fakedMethodName, @NonNull String fakedMethodDesc) {
        super(invokedInstance, invokedArguments, fakeState.getTimesInvoked());
        this.fakeState = fakeState;
        this.fakedClassDesc = fakedClassDesc;
        this.fakedMethodName = fakedMethodName;
        this.fakedMethodDesc = fakedMethodDesc;
    }

    @NonNull
    @Override
    protected Member findRealMember() {
        Object invokedInstance = getInvokedInstance();

        if (invokedInstance != null) {
            Class<?> mockedClass = invokedInstance.getClass();
            return fakeState.getRealMethodOrConstructor(mockedClass, fakedMethodName, fakedMethodDesc);
        }

        return fakeState.getRealMethodOrConstructor(fakedClassDesc, fakedMethodName, fakedMethodDesc);
    }

    @SuppressWarnings("WeakerAccess") // called from generated bytecode
    public boolean shouldProceedIntoConstructor() {
        if (proceeding && getInvokedMember() instanceof Constructor) {
            fakeState.clearProceedIndicator();
            return true;
        }

        return false;
    }

    @Override
    public void prepareToProceed() {
        fakeState.prepareToProceed(this);
        proceeding = true;
    }

    public void prepareToProceedFromNonRecursiveMock() {
        fakeState.prepareToProceedFromNonRecursiveFake(this);
        proceeding = true;
    }

    @Override
    public void cleanUpAfterProceed() {
        fakeState.clearProceedIndicator();
    }
}
