#
# spec file for package postgresql17
#
# Copyright (c) 2025 SUSE LLC and contributors
#
# All modifications and additions to the file contributed by third parties
# remain the property of their copyright owners, unless otherwise agreed
# upon. The license for this file, and modifications and additions to the
# file, is the same license as for the pristine package itself (unless the
# license for the pristine package is not an Open Source License, in which
# case the license is the MIT License). An "Open Source License" is a
# license that conforms to the Open Source Definition (Version 1.9)
# published by the Open Source Initiative.

# Please submit bugfixes or comments via https://bugs.opensuse.org/
#


%define pgversion 17.6
%define pgmajor 17
%define buildlibs 1
%define tarversion %{pgversion}
%define oldest_supported_llvm_ver 10
# To be able to use cmake(LLVM) < ...
%define latest_supported_llvm_ver_plus_one 19

### CUT HERE ###
%define pgname postgresql%pgmajor
%define libpq libpq5
%define libecpg libecpg6
%define libpq_so libpq.so.5
%define libecpg_so libecpg.so.6
%define pgbasedir %_prefix/lib/%pgname
%define pgtestdir %pgbasedir/test
%define pgbindir %pgbasedir/bin
%define pglibdir %pgbasedir/%_lib
%define pgincludedir %_includedir/pgsql
%define pgdatadir %_datadir/%pgname
%define pgdocdir %_docdir/%pgname
%define pgextensiondir %pgdatadir/extension
%define pgcontribdir %pgdatadir/contrib
%define pgmandir %_mandir

%define requires_file() %( readlink -f '%*' | LC_ALL=C xargs -r rpm -q --qf 'Requires: %%{name} >= %%{epoch}:%%{version}\\n' -f | sed -e 's/ (none):/ /' -e 's/ 0:/ /' | grep -v "is not")

%if "" == "mini"
%define devel devel-mini
%define mini 1
%define buildlibs 1
Name:           %pgname-mini
%else
%define devel devel
%define mini 0
Name:           %pgname
%endif

# Use Python 2 for for PostgreSQL 10 on SLE12.
# Use Python 3 for everything else.
%if 0%{?is_opensuse} || 0%{?sle_version} >= 150000 || %pgmajor > 10
%define python python3
%else
%define python python
%endif

%if %pgmajor >= 17
%bcond_with derived
%else
%bcond_without derived
%endif

%if 0%{?suse_version} >= 1500
%bcond_without liblz4
%endif

%if 0%{?sle_version} >= 150100 || 0%{?suse_version} >= 1550
%bcond_without libzstd
%endif

%if %{without derived}
BuildRequires:  bison
BuildRequires:  docbook-xsl-stylesheets
BuildRequires:  flex
BuildRequires:  perl
%endif
%if %mini
%bcond_with  selinux
%if %pgmajor >= 16
%bcond_without icu
%else
%bcond_with  icu
%endif
%else
BuildRequires:  %{python}-devel
BuildRequires:  docbook_4
BuildRequires:  gettext-devel
BuildRequires:  libuuid-devel
BuildRequires:  ncurses-devel
BuildRequires:  pam-devel
BuildRequires:  readline-devel
BuildRequires:  tcl-devel
BuildRequires:  timezone
BuildRequires:  zlib-devel
%if %{with liblz4}
BuildRequires:  pkgconfig(liblz4)
%endif

%if %{with libzstd}
BuildRequires:  pkgconfig(libzstd)
%endif

%bcond_without  selinux
%bcond_without  icu
%if  !%buildlibs
BuildRequires:  %libecpg
BuildRequires:  %libpq
%endif

%if 0%{?suse_version} >= 1500 && %pgmajor >= 11
%ifarch riscv64 loongarch64
%bcond_with     llvm
%else
%bcond_without  llvm
%endif
%else
%bcond_with     llvm
%endif
%endif

%ifnarch %arm
%bcond_without  check
%else
%bcond_with     check
%endif

%if %pgmajor >= 11 || %mini
%bcond_without server_devel
%else
%bcond_with server_devel
%endif

BuildRequires:  fdupes
%if %{with icu}
BuildRequires:  libicu-devel
%endif
%if %{with selinux}
BuildRequires:  libselinux-devel
%endif
%if %{with llvm}
BuildRequires:  gcc-c++
BuildRequires:  (cmake(Clang) >= %{oldest_supported_llvm_ver} with cmake(Clang) < %{latest_supported_llvm_ver_plus_one})
BuildRequires:  (cmake(LLVM)  >= %{oldest_supported_llvm_ver} with cmake(LLVM)  < %{latest_supported_llvm_ver_plus_one})
%endif
BuildRequires:  libxslt-devel
BuildRequires:  openldap2-devel
BuildRequires:  openssl-devel
BuildRequires:  pkg-config
BuildRequires:  pkgconfig(krb5)
BuildRequires:  pkgconfig(libsystemd)
BuildRequires:  pkgconfig(systemd)
#!BuildIgnore:  %pgname
#!BuildIgnore:  %pgname-server
#!BuildIgnore:  %pgname-devel
#!BuildIgnore:  %pgname-server-devel
#!BuildIgnore:  %pgname-llvmjit
#!BuildIgnore:  %pgname-llvmjit-devel
#!BuildIgnore:  %pgname-contrib
#!BuildIgnore:  %pgname-docs
#!BuildIgnore:  %pgname-test
#!BuildIgnore:  %pgname-pltcl
#!BuildIgnore:  %pgname-plperl
#!BuildIgnore:  %pgname-plpython
#!BuildIgnore:  postgresql-implementation
#!BuildIgnore:  postgresql-server-implementation
#!BuildIgnore:  postgresql-server-devel-implementation
#!BuildIgnore:  postgresql-llvmjit-devel-implementation
Summary:        Basic Clients and Utilities for PostgreSQL
License:        PostgreSQL
Group:          Productivity/Databases/Tools
Version:        %pgversion
Release:        150600.13.16.1
Source0:        https://ftp.postgresql.org/pub/source/v%{tarversion}/postgresql-%{tarversion}.tar.bz2
Source1:        https://ftp.postgresql.org/pub/source/v%{tarversion}/postgresql-%{tarversion}.tar.bz2.sha256
Source2:        baselibs.conf
Source17:       postgresql-rpmlintrc
Patch1:         postgresql-conf.patch
Patch2:         postresql-pg_config_paths.patch
# PL/Perl needs to be linked with rpath (bsc#578053)
Patch4:         postgresql-plperl-keep-rpath.patch
Patch8:         postgresql-testsuite-keep-results-file.patch
Patch9:         postgresql-var-run-socket.patch
%if %{with llvm}
Patch10:        postgresql-llvm-optional.patch
Patch11:        0001-jit-Workaround-potential-datalayout-mismatch-on-s390.patch
%endif
URL:            https://www.postgresql.org/
BuildRoot:      %{_tmppath}/%{name}-%{version}-build
Provides:       postgresql = %version-%release
Provides:       postgresql-implementation = %version-%release
Requires:       %libpq >= %version
Requires(post): postgresql-noarch >= %pgmajor
Requires(postun): postgresql-noarch >= %pgmajor
# At this point we changed the package layout on SLE and conflict with
# older releases to get a clean cut.
Conflicts:      postgresql-noarch < 12.0.1

%description
PostgreSQL is an advanced object-relational database management system
that supports an extended subset of the SQL standard, including
transactions, foreign keys, subqueries, triggers, and user-defined
types and functions.

This package contains the basic utility and client programs necessary
to maintain and work with local or remote PostgreSQL databases as well
as manual pages for the SQL commands that PostgreSQL supports. Full
HTML documentation for PostgreSQL can be found in the postgresql-docs
package.

%package -n %libpq
Summary:        Shared Libraries Required for PostgreSQL Clients
Group:          Productivity/Databases/Clients
Provides:       postgresql-libs:%_libdir/libpq.so.5
Obsoletes:      postgresql-libs < %version
# bug437293
%if "%_lib" == "lib64"
Conflicts:      %libpq-32bit < %version
%endif
%ifarch ia64
Conflicts:      %libpq-x86 < %version
%endif
%ifarch ppc64
Obsoletes:      postgresql-libs-64bit
%endif

%description -n %libpq

PostgreSQL is an advanced object-relational database management system
that supports an extended subset of the SQL standard, including
transactions, foreign keys, subqueries, triggers, user-defined types
and functions.

This package provides the client library that most PostgreSQL client
program or language bindings are using.

%package -n %libecpg

Summary:        Shared Libraries Required for PostgreSQL Clients
Group:          Productivity/Databases/Clients
Provides:       postgresql-libs:%_libdir/libecpg.so.6

%description -n %libecpg
PostgreSQL is an advanced object-relational database management system
that supports an extended subset of the SQL standard, including
transactions, foreign keys, subqueries, triggers, user-defined types
and functions.

This package provides the runtime library of the embedded SQL C
preprocessor for PostgreSQL.

%package -n %pgname-%devel
Summary:        PostgreSQL client development header files and libraries
Group:          Development/Libraries/C and C++
Provides:       postgresql-devel = %version-%release
Provides:       postgresql-devel-implementation = %version-%release
%if %mini
Requires:       this-is-only-for-build-envs
Provides:       %libecpg = %version-%release
Provides:       %libpq = %version-%release
Provides:       %pgname-devel = %version-%release
Conflicts:      %libecpg
Conflicts:      %libpq
Conflicts:      %pgname-devel
%else
Requires:       %libecpg >= %version
Requires:       %libpq >= %version
Requires:       postgresql-devel-noarch >= %pgmajor
%endif
# Installation of postgresql??-devel is exclusive
Provides:       postgresql-devel-exclusive = %pgmajor
Conflicts:      postgresql-devel-exclusive < %pgmajor

%if %{with server_devel}
%package server-devel
Summary:        PostgreSQL server development header files and utilities
Group:          Development/Libraries/C and C++
%else
Provides:       %pgname-server-devel = %version-%release
%endif
Provides:       postgresql-server-devel = %version-%release
Provides:       postgresql-server-devel-implementation = %version-%release
Requires(post): postgresql-server-devel-noarch >= %pgmajor
Requires(postun): postgresql-server-devel-noarch >= %pgmajor
Requires:       %pgname-devel = %version
Requires:       %pgname-server = %version-%release
# Installation of postgresql??-devel is exclusive
Provides:       postgresql-server-devel-exclusive = %pgmajor
Conflicts:      postgresql-server-devel-exclusive < %pgmajor
Requires:       libxslt-devel
Requires:       openssl-devel
Requires:       pam-devel
Requires:       readline-devel
Requires:       zlib-devel
Requires:       pkgconfig(krb5)
%if %{with selinux}
Requires:       libselinux-devel
%endif
%if %{with llvm}
Recommends:     %pgname-llvmjit-devel = %version-%release
%endif

%if %{with server_devel}
%description server-devel
PostgreSQL is an advanced object-relational database management system
that supports an extended subset of the SQL standard, including
transactions, foreign keys, subqueries, triggers, and user-defined
types and functions.

This package contains the header files and libraries needed to compile
C extensions that link into the PostgreSQL server. For building client
applications, see the postgresql%pgmajor-devel package.
%endif

%description -n %pgname-%devel
PostgreSQL is an advanced object-relational database management system
that supports an extended subset of the SQL standard, including
transactions, foreign keys, subqueries, triggers, and user-defined
types and functions.

This package contains the header files and libraries needed to compile
C applications which will directly interact with a PostgreSQL database
management server and the ECPG Embedded C Postgres preprocessor. You
need to install this package if you want to develop applications in C
which will interact with a PostgreSQL server.

For building PostgreSQL server extensions, see the
postgresql%pgmajor-server-devel package.

%package server
Summary:        The Programs Needed to Create and Run a PostgreSQL Server
Group:          Productivity/Databases/Servers
PreReq:         postgresql = %version
Requires:       glibc-locale
Requires:       timezone
%if %{with llvm}
Recommends:     %{name}-llvmjit
%endif
Provides:       postgresql-server-implementation = %version-%release
Requires:       %libpq >= %version
Requires(pre):  postgresql-server-noarch >= %pgmajor
Requires(preun): postgresql-server-noarch >= %pgmajor
Requires(postun): postgresql-server-noarch >= %pgmajor
Requires(post): postgresql-noarch >= %pgmajor
Requires(postun): postgresql-noarch >= %pgmajor

%description server
PostgreSQL is an advanced object-relational database management system
that supports an extended subset of the SQL standard, including
transactions, foreign keys, sub-queries, triggers, and user-defined
types and functions.

This package includes the programs needed to create and run a
PostgreSQL server, which will in turn allow you to create and maintain
PostgreSQL databases.

%package llvmjit
Summary:        Just-in-time compilation support for PostgreSQL
Group:          Productivity/Databases/Servers
Provides:       postgresql-llvmjit-implementation = %version-%release
Requires:       %pgname-server = %version-%release
Requires:       postgresql-llvmjit-noarch >= %pgmajor

%description llvmjit
PostgreSQL is an advanced object-relational database management system
that supports an extended subset of the SQL standard, including
transactions, foreign keys, sub-queries, triggers, and user-defined
types and functions.

This package contains support for just-in-time compiling parts of
PostgreSQL queries. Using LLVM it compiles e.g. expressions and tuple
deforming into native code, with the goal of accelerating analytics
queries.

%package llvmjit-devel
Summary:        PostgreSQL development files for extensions with LLVM support
Group:          Development/Libraries/C and C++
Provides:       postgresql-llvmjit-devel = %version-%release
Provides:       postgresql-llvmjit-devel-implementation = %version-%release
Requires:       %pgname-server-devel = %version
%if %{with llvm}
Requires:       %pgname-llvmjit = %version
Requires(post): postgresql-llvmjit-devel-noarch >= %pgmajor
Requires(postun): postgresql-llvmjit-devel-noarch >= %pgmajor
%requires_file	%_bindir/llc
%requires_file	%_bindir/clang
%endif

%description llvmjit-devel
PostgreSQL is an advanced object-relational database management system
that supports an extended subset of the SQL standard, including
transactions, foreign keys, sub-queries, triggers, and user-defined
types and functions.

This package pulls in the right versions of llvm and clang to compile
PostgreSQL extensions that support just-in-time compilation with LLVM,
if llvm is supported. Otherwise it will just pull the
%{pgname}-server-devel package.

%package test
Summary:        The test suite for PostgreSQL
Group:          Productivity/Databases/Servers
Provides:       postgresql-test-implementation = %version-%release
Requires:       %pgname-server = %version
Requires:       postgresql-test-noarch >= %pgmajor

%description test
This package contains the sources and pre-built binaries of various
tests for the PostgreSQL database management system, including
regression tests and benchmarks.

%package docs
Summary:        HTML Documentation for PostgreSQL
Group:          Productivity/Databases/Tools
Provides:       postgresql-docs-implementation = %version-%release
Requires:       postgresql-docs-noarch >= %pgmajor
BuildArch:      noarch

%description docs
PostgreSQL is an advanced object-relational database management system
that supports an extended subset of the SQL standard, including
transactions, foreign keys, subqueries, triggers, and user-defined
types and functions.

This package contains the HTML documentation for PostgreSQL. The start
page is: file:///usr/share/doc/packages/%pgname/html/index.html .
Manual pages for the PostgreSQL SQL statements can be found in the
postgresql package.

%package contrib
Summary:        Contributed Extensions and Additions to PostgreSQL
Group:          Productivity/Databases/Tools
Provides:       postgresql-contrib-implementation = %version-%release
Requires:       postgresql-contrib-noarch >= %pgmajor
Requires(post): %pgname >= %{version}
Requires:       %pgname >= %{version}
PreReq:         %pgname-server = %version-%release

%description contrib
PostgreSQL is an advanced object-relational database management system
that supports an extended subset of the SQL standard, including
transactions, foreign keys, subqueries, triggers, and user-defined
types and functions.

The postgresql-contrib package includes extensions and additions that
are distributed along with the PostgreSQL sources, but are not (yet)
officially part of the PostgreSQL core.

Documentation for the modules contained in this package can be found in
/usr/share/doc/packages/postgresql/contrib.

%package plperl
Summary:        The PL/Tcl, PL/Perl, and  PL/Python procedural languages for PostgreSQL
Group:          Productivity/Databases/Servers
Provides:       postgresql-plperl-implementation = %version-%release
Requires:       %pgname-server = %version-%release
Requires:       perl = %perl_version
Requires:       postgresql-plperl-noarch >= %pgmajor

%description plperl
This package contains the the PL/Tcl, PL/Perl, and PL/Python procedural
languages for the back-end.  With these modules one can use Perl,
Python, and Tcl to write stored procedures, functions and triggers.

PostgreSQL also offers the builtin procedural language PL/SQL.

%package plpython
Summary:        The PL/Python Procedural Languages for PostgreSQL
Group:          Productivity/Databases/Servers
Provides:       postgresql-plpython-implementation = %version-%release
Requires:       %pgname-server = %version-%release
Requires:       %python
Requires:       postgresql-plpython-noarch >= %pgmajor

%description plpython
PostgreSQL is an advanced object-relational database management system
that supports an extended subset of the SQL standard, including
transactions, foreign keys, subqueries, triggers, and user-defined
types and functions.

This package contains the PL/Python procedural language for PostgreSQL.
With this module one can use Python to write stored procedures,
functions, and triggers.

PostgreSQL also offers the built-in procedural language PL/SQL which is
included in the postgresql-server package.

%package pltcl
Summary:        PL/Tcl Procedural Language for PostgreSQL
Group:          Productivity/Databases/Tools
Provides:       postgresql-pltcl-implementation = %version-%release
Requires:       %pgname-server = %version
Requires:       postgresql-pltcl-noarch >= %pgmajor
Requires:       tcl

%description pltcl
PostgreSQL is an advanced object-relational database management system
that supports an extended subset of the SQL standard, including
transactions, foreign keys, subqueries, triggers, and user-defined
types and functions.

This package contains the PL/Tcl procedural language for PostgreSQL.
With thie module one can use Tcl to write stored procedures, functions,
and triggers.

PostgreSQL also offers the built-in procedural language PL/SQL which is
included in the postgresql-server package.

%prep
%setup -q -n postgresql-%tarversion
# Keep the timestamp of configure, because patching it would otherwise
# confuse PostgreSQL's build system
touch -r configure tmp
%patch -P 1
%patch -P 2
%patch -P 4
%patch -P 8
%patch -P 9
%if %{with llvm}
%patch -P 10
%patch -P 11
%endif
touch -r tmp configure
rm tmp
find src/test/ -name '*.orig' -delete
find -name .gitignore -delete

%build
%global _lto_cflags %{_lto_cflags} -ffat-lto-objects
export PYTHON=%python
%ifarch %arm
export USE_ARMV8_CRC32C=0
%endif
PACKAGE_TARNAME=%pgname %configure \
        --bindir=%pgbindir \
        --libdir=%pglibdir \
        --includedir=%pgincludedir \
        --datadir=%pgdatadir \
        --docdir=%pgdocdir \
        --mandir=%pgmandir \
        --disable-rpath \
        --enable-nls \
        --enable-thread-safety \
        --enable-integer-datetimes \
%if !%mini
        --with-python \
        --with-perl \
        --with-tcl \
        --with-tclconfig=%_libdir \
        --with-pam \
        --with-uuid=e2fs \
        --with-libxml \
        --with-libxslt \
%if %{with liblz4}
        --with-lz4 \
%endif
%if %{with libzstd}
        --with-zstd \
%endif
        --with-systemd \
%if %{with selinux}
        --with-selinux \
%endif
%if %{with icu}
        --with-icu \
%endif
%if %{with llvm}
	--with-llvm \
%endif
%else
        --without-readline \
%endif
        --with-openssl \
        --with-ldap \
        --with-gssapi \
        --with-krb5 \
        --with-system-tzdata=/usr/share/zoneinfo
%if %mini
make -C src/interfaces %{?_smp_mflags} PACKAGE_TARNAME=%pgname
%else
make %{?_smp_mflags} PACKAGE_TARNAME=%pgname world

%if %{with check}

%check
#
# Run the regression tests.
#
make check || {
  for f in src/test/regress/log/* {,src/test/regress/}regression.diffs; do
    if test -f $f; then
	cat $f
    fi
  done
  exit 1
}
%endif
%endif

%install
VLANG=%pgmajor
%if %mini
make DESTDIR=%buildroot PACKAGE_TARNAME=%pgname -C src/include install
make DESTDIR=%buildroot PACKAGE_TARNAME=%pgname -C src/interfaces install
rm -rf %buildroot%pgincludedir/server
rm -rf %buildroot%pgdatadir
%else
make DESTDIR=%buildroot PACKAGE_TARNAME=%pgname install install-docs
%if 0
mv %buildroot%pgincludedir/{server,..}
make DESTDIR=%buildroot PACKAGE_TARNAME=%pgname -C src/interfaces uninstall
rm -rf %buildroot%pgincludedir/*
mv %buildroot%pgincludedir{/../server,}
%endif

# {{{ the test package
mkdir -p %buildroot%pgtestdir/regress
install -sm 0755 contrib/spi/{refint.so,autoinc.so} %buildroot%pgtestdir/regress
install -sm 0755 src/test/regress/{pg_regress,regress.so} %buildroot%pgtestdir/regress
for i in  src/test/regress/{data,expected,input,output,sql}; do
	test -d $i && cp -r $i %buildroot%pgtestdir/regress/
done
install -m 0644 src/test/regress/*_schedule %buildroot%pgtestdir/regress
# }}}
%endif

# The client libraries go to libdir
mkdir -p %buildroot/%_libdir
ls %buildroot%pglibdir/lib* |
   grep -v walreceiver |
   xargs mv -t %buildroot/%_libdir
mv %buildroot%pglibdir/pkgconfig %buildroot%_libdir
find %buildroot%_libdir/pkgconfig -type f -exec sed -i 's, -L%pglibdir,,' '{}' +

# Don't ship static libraries,
# libpgport.a and libpgcommon.a are needed, though.
rm -f $(ls %buildroot/%_libdir/*.a %buildroot%pglibdir/*.a | grep -F -v -e libpgport.a -e libpgcommon.a)

%if !%mini
#
# Install and collect the contrib stuff
#
touch flag; sleep 1 # otherwise we have installed files that are not newer than flag
make DESTDIR=%buildroot -C contrib install
find %buildroot -type f -cnewer flag -printf "/%%P\n" |
     grep -v -e %_docdir -e %pgbindir -e %pgincludedir -e %pglibdir/bitcode \
     > contrib.files
rm flag
install -d -m 750 %buildroot/var/lib/pgsql
install -d -m 755 %buildroot%pgdocdir
cp doc/KNOWN_BUGS doc/MISSING_FEATURES COPYRIGHT \
   README* HISTORY  %buildroot%pgdocdir
# Use versioned names for the man pages:
for f in %buildroot%pgmandir/man*/*; do
        mv $f ${f}pg%pgmajor
done
%endif

mkdir -p %buildroot{%_bindir,%_mandir/man1}
mkdir -p %buildroot/etc/alternatives
genlists ()
{
    # usage: genlists packagename basenames
    PKG=$1
    shift
    for f in $@
    do
        BIN=%_bindir/$f
        ALTBIN=/etc/alternatives/$f
        PGBIN=%pgbindir/$f
        MAN=%pgmandir/man1/$f.1*

	# Package only binaries that exist in this version
        test -e %buildroot$PGBIN || continue

        touch %buildroot$ALTBIN
        ln -s $ALTBIN %buildroot$BIN

        echo "$PGBIN" >> $PKG.files
        echo "$BIN" >> $PKG.files
        echo "%ghost $ALTBIN" >> $PKG.files
        test -e %buildroot$MAN &&
            echo "%doc $MAN" >> $PKG.files
        %find_lang $f-$VLANG $PKG.files ||:
    done
}
%if !%mini
genlists main \
	createdb \
	clusterdb \
	createuser \
	dropdb \
	dropuser \
	pg_dump \
	pg_dumpall \
	pg_restore \
	pg_rewind \
	psql \
	vacuumdb \
	reindexdb \
	pg_basebackup \
	pg_isready \
	pg_recvlogical \
	createlang \
	droplang \
	pg_receivexlog \
	pg_receivewal \
	pg_verify_checksums \
	pg_checksums \
	pg_combinebackup \
	pg_verifybackup

%find_lang plpgsql-$VLANG main.files
%find_lang pgscripts-$VLANG main.files

genlists server \
	initdb \
	pg_ctl \
	pg_controldata \
	pg_resetwal \
	pg_createsubscriber \
	pg_walsummary \
	pg_waldump \
	pg_resetxlog \
%if %pgmajor >= 15
	pg_upgrade \
%endif
	postgres \
	postmaster

genlists contrib \
	pg_xlogdump \
	oid2name \
	pg_archivecleanup \
	pg_amcheck \
	pg_standby \
	pg_test_fsync \
%if %pgmajor < 15
	pg_upgrade \
%endif
        pgbench \
	vacuumlo \
	pg_test_timing
for pl in plperl plpython pltcl; do
    %find_lang $pl-$VLANG $pl.lang
done
ln -s /etc/alternatives/postgresql %buildroot/usr/lib/postgresql
touch %buildroot/etc/alternatives/postgresql

# Remove mostly unneeded buildtime requirements for server extensions
sed -i '/^LIBS = /s/= .*/=/' %buildroot/%pglibdir/pgxs/src/Makefile.global
%endif

# Make sure we can also link agaist newer versions
pushd %buildroot%_libdir
for f in *.so; do
    ln -sf $f.? $f
done
%if 0
for long in *.so.*.*; do
    short=${long%%.*}
    so=${short%%.*}
    ln -sf $long $short
    ln -sf $short $so
done
%endif
popd

%find_lang ecpg-$VLANG devel.files
# The devel subpackage is exclusive across versions
# and not handled by update-alternatives.
mv %buildroot%pgbindir/ecpg %buildroot%_bindir/ecpg

%if !%mini
%find_lang pg_config-$VLANG server-devel.files
ln -s %pgbindir/pg_config %buildroot%_bindir/pg_config
%endif

%if %{without server_devel}
cat server-devel.files >> devel.files
%endif

# Build up the file lists for the libpq and libecpg packages
cat > libpq.files <<EOF
%defattr(-,root,root)
%if !%mini
%dir %pgdatadir
%pgdatadir/pg_service.conf.sample
%endif
EOF
find %buildroot -name 'libpq*.so.*' -printf '/%%P\n' >> libpq.files
%find_lang libpq5-$VLANG libpq.files

cat > libecpg.files <<EOF
%defattr(-,root,root)
EOF
find %buildroot \( -name 'libecpg*.so.*' -o -name 'libpgtypes.so.*' \) -printf '/%%P\n' >> libecpg.files
%find_lang ecpglib6-$VLANG libecpg.files

%if !%buildlibs
# Delete the contents of the library packages, if we don't want to build them
awk -v P=%buildroot '/^(%lang|[^%])/{print P $NF}' libpq.files libecpg.files | xargs rm
%endif

%fdupes %buildroot

%post -n %pgname-%devel
/sbin/ldconfig

%postun -n %pgname-%devel
/sbin/ldconfig

%if %{with server_devel}
%post server-devel
/usr/share/postgresql/install-alternatives %pgmajor

%postun server-devel
/usr/share/postgresql/install-alternatives %pgmajor
%endif

%if !%mini

%postun
/usr/share/postgresql/install-alternatives %pgmajor

%post
/usr/share/postgresql/install-alternatives %pgmajor

%post server
/usr/share/postgresql/install-alternatives %pgmajor

%preun server
# Stop only when we are uninstalling the currently running version
test -x /usr/bin/systemctl &&
  MAINPID=$(/usr/bin/systemctl show postgresql.service --property=MainPID --value) ||:
if test -n "$MAINPID" && test "$MAINPID" -ne 0; then
  BIN=$(readlink -n /proc/$MAINPID/exe)
  DIR=$(dirname ${BIN% *})
  if test "$DIR" = "%pgbindir" -o "$DIR" = "%_bindir"; then
      %service_del_preun postgresql.service
  fi
fi

%postun server
/usr/share/postgresql/install-alternatives %pgmajor
# Restart only when we are updating the currently running version
test -x /usr/bin/systemctl &&
  MAINPID=$(/usr/bin/systemctl show postgresql.service --property=MainPID --value) ||:
if test -n "$MAINPID" && test "$MAINPID" -ne 0; then
  BIN=$(readlink -n /proc/$MAINPID/exe)
  DIR=$(dirname ${BIN% *})
  if test "$DIR" = "%pgbindir" -o "$DIR" = "%_bindir"; then
      %service_del_postun postgresql.service
  fi
fi

%post contrib
/usr/share/postgresql/install-alternatives %pgmajor

%postun contrib
/usr/share/postgresql/install-alternatives %pgmajor

%if %buildlibs
%post -n %libpq -p /sbin/ldconfig

%postun -n %libpq -p /sbin/ldconfig

%post -n %libecpg -p /sbin/ldconfig

%postun -n %libecpg -p /sbin/ldconfig
%endif

%files -f main.files
%defattr(-,root,root)
%dir %pgbindir
%doc %pgmandir/man7/*
%docdir %pgdocdir
%dir %pgdocdir
%pgdocdir/[[:upper:]]*
%dir %pglibdir
/usr/lib/postgresql
%ghost /etc/alternatives/postgresql

%files test
%defattr(-,root,root,-)
%pgtestdir

%files docs
%defattr(-,root,root)
%doc %pgmandir/man3/*
%docdir %pgdocdir
%dir %pgdocdir
%pgdocdir/html

%files contrib -f contrib.files
%defattr(-,root,root)
%docdir %pgdocdir
%dir %pgdocdir
%pgdocdir/extension
%dir %pgdatadir
%dir %pgcontribdir

%files server -f server.files
%defattr(-,root,root)
%dir %pgbasedir
%dir %pgextensiondir
%dir %pglibdir
%pglibdir/pgoutput.so
%pglibdir/plpgsql.so
%pglibdir/dict_snowball.so
%pgdatadir/tsearch_data
%exclude %pgdatadir/tsearch_data/*.rules
%dir %pgdatadir
%pgdatadir/timezone*
%pgdatadir/*.*
%if %buildlibs
%exclude %pgdatadir/pg_service.conf.sample
%endif
%pglibdir/*_and_*.so
%pglibdir/euc2004_sjis2004.so
%pglibdir/libpqwalreceiver.so
%pgextensiondir/plpgsql*
%attr(750,postgres,postgres) %dir /var/lib/pgsql

%if %{with llvm}
%dir %pglibdir/bitcode

%files llvmjit
%defattr(-,root,root)
%pglibdir/llvm*
%pglibdir/bitcode/*
%endif

%files llvmjit-devel
%defattr(-,root,root)

%files pltcl -f pltcl.lang
%defattr(-,root,root)
%pgextensiondir/pltcl*
%pglibdir/pltcl.so

%files plperl -f plperl.lang
%defattr(-,root,root)
%pgextensiondir/plperl*
%pglibdir/plperl.so

%files plpython -f plpython.lang
%defattr(-,root,root)
%pgextensiondir/plpython*
%pglibdir/plpython*.so

%endif

%if %buildlibs && !%mini
%files -n %libpq -f libpq.files

%files -n %libecpg -f libecpg.files
%endif

%if %buildlibs && %mini
%files -n %pgname-%devel -f devel.files -f libpq.files -f libecpg.files
%defattr(-,root,root)
%else

%files -n %pgname-%devel -f devel.files
%defattr(-,root,root)
%endif

%dir %pgbasedir
%dir %pgbindir
%_bindir/ecpg
%_libdir/pkgconfig/*
%_libdir/lib*.so
%pgincludedir

%if %{with server_devel}
%exclude %pgincludedir/server
%endif

%if !%mini
%doc %pgmandir/man1/ecpg.1*
%if %{with server_devel}
%files server-devel -f server-devel.files
%endif
%defattr(-,root,root)
%ghost %_bindir/pg_config
%pgbindir/pg_config
%pgincludedir/server
%pglibdir/pgxs
%_libdir/lib*.a
%doc %pgmandir/man1/pg_config.1*
%endif

%changelog
* Thu Aug 14 2025 mrueckert@suse.de
- Update to 17.6
  * https://www.postgresql.org/about/news/postgresql-176-1610-1514-1419-1322-and-18-beta-3-released-3118/
  * https://www.postgresql.org/docs/release/17.6/
  Security Issues
  * CVE-2025-8713: PostgreSQL optimizer statistics can expose
    sampled data within a view, partition, or child table
    (boo#1248120)
  * CVE-2025-8714: PostgreSQL pg_dump lets superuser of origin
    server execute arbitrary code in psql client (boo#1248122)
  * CVE-2025-8715: PostgreSQL pg_dump newline in object name
    executes arbitrary code in psql client and in restore target
    server (boo#1248119)
* Fri May  9 2025 max@suse.com
- Upgrade to 17.5:
  * bsc#1242931, CVE-2025-4207: postgresql: PostgreSQL GB18030
    encoding validation can read one byte past end of allocation
    for text that fails validation.
  * https://www.postgresql.org/docs/release/17.5/
  * https://www.postgresql.org/about/news/p-3072/
- Add postresql-pg_config_paths.patch to Fix a race condition
  while generating pg_config_paths.h.
* Tue Feb 18 2025 max@suse.com
- Upgrade to 17.4:
  * Improve behavior of libpq's quoting functions:
    The changes made for CVE-2025-1094 had one serious oversight:
    PQescapeLiteral() and PQescapeIdentifier() failed to honor
    their string length parameter, instead always reading to the
    input string's trailing null. This resulted in including
    unwanted text in the output, if the caller intended to
    truncate the string via the length parameter. With very bad
    luck it could cause a crash due to reading off the end of
    memory.
    In addition, modify all these quoting functions so that when
    invalid encoding is detected, an invalid sequence is
    substituted for just the first byte of the presumed
    character, not all of it. This reduces the risk of problems
    if a calling application performs additional processing on
    the quoted string.
  * Fix small memory leak in pg_createsubscriber.
  * https://www.postgresql.org/docs/release/17.4/
  * https://www.postgresql.org/about/news/p-3018/
* Tue Feb 11 2025 max@suse.com
- Upgrade to 17.3:
  * bsc#1237093, CVE-2025-1094: Harden PQescapeString and allied
    functions against invalidly-encoded input strings.
  * obsoletes postgresql-tzdata2025a.patch
  * https://www.postgresql.org/docs/release/17.3/
  * https://www.postgresql.org/about/news/-3015/
* Wed Jan 29 2025 obs.coke518@passinbox.com
- Apply postgresql-tzdata2025a.patch regardless of
  whether LLVM JIT is enabled
* Tue Jan 28 2025 max@suse.com
- Fix build, add postgresql-tzdata2025a.patch
* Thu Dec 26 2024 obs.coke518@passinbox.com
- Disable LLVM JIT on loongarch64
* Tue Nov 19 2024 max@suse.com
- Upgrade to 17.2:
  * Repair ABI break for extensions that work with struct
    ResultRelInfo.
  * Restore functionality of ALTER {ROLE|DATABASE} SET role.
  * Fix cases where a logical replication slot's restart_lsn could
    go backwards.
  * Avoid deleting still-needed WAL files during pg_rewind.
  * Fix race conditions associated with dropping shared statistics
    entries.
  * Count index scans in contrib/bloom indexes in the statistics
    views, such as the pg_stat_user_indexes.idx_scan counter.
  * Fix crash when checking to see if an index's opclass options
    have changed.
  * Avoid assertion failure caused by disconnected NFA sub-graphs
    in regular expression parsing.
  * https://www.postgresql.org/about/news/p-2965/
  * https://www.postgresql.org/docs/release/17.2/
* Wed Nov 13 2024 max@suse.com
- Upgrade to 17.1:
  * CVE-2024-10976, bsc#1233323: Ensure cached plans are marked as
    dependent on the calling role when RLS applies to a
    non-top-level table reference.
  * CVE-2024-10977, bsc#1233325: Make libpq discard error messages
    received during SSL or GSS protocol negotiation.
  * CVE-2024-10978, bsc#1233326: Fix unintended interactions
    between SET SESSION AUTHORIZATION and SET ROLE
  * CVE-2024-10979, bsc#1233327: Prevent trusted PL/Perl code from
    changing environment variables.
  * obsoletes postgresql17-jsonb_jsonpath.patch
  * https://www.postgresql.org/about/news/p-2955/
  * https://www.postgresql.org/docs/release/17.1/
* Mon Nov  4 2024 max@suse.com
- Add postgresql17-jsonb_jsonpath.patch as a temporary build fix.
* Mon Oct  7 2024 fvogt@suse.com
- Use cmake(LLVM) < instead of <= to also allow building against
  LLVM 18.1 and newer 18.x versions
* Thu Sep 26 2024 max@suse.com
- Upgrade to 17.0:
  * New memory management system for VACUUM, which reduces memory
    consumption and can improve overall vacuuming performance.
  * New SQL/JSON capabilities, including constructors, identity
    functions, and the JSON_TABLE() function, which converts JSON
    data into a table representation.
  * Various query performance improvements, including for
    sequential reads using streaming I/O, write throughput under
    high concurrency, and searches over multiple values in a btree
    index.
  * Logical replication enhancements, including:
    + Failover control
    + pg_createsubscriber, a utility that creates logical replicas
    from physical standbys
    + pg_upgrade now preserves replication slots on both publishers
    and subscribers
  * New client-side connection option, sslnegotiation=direct, that
    performs a direct TLS handshake to avoid a round-trip
    negotiation.
  * pg_basebackup now supports incremental backup.
  * COPY adds a new option, ON_ERROR ignore, that allows a copy
    operation to continue in the event of an error.
  * https://www.postgresql.org/about/news/p-2936/
  * https://www.postgresql.org/docs/17/release-17.html
* Sat Sep  7 2024 mrueckert@suse.de
- Upgrade to 17rc1
  https://www.postgresql.org/about/news/postgresql-17-rc1-released-2926/
* Sat Aug 10 2024 mrueckert@suse.de
- Upgrade to 17beta3 (bsc#1229013):
  * bsc#1229013, CVE-2024-7348 PostgreSQL relation replacement
    during pg_dump executes arbitrary SQL
  * https://www.postgresql.org/about/news/p-2910/
* Wed Jun 26 2024 max@suse.com
- Upgrade to 17beta2.
* Mon May 27 2024 max@suse.com
- Upgrade to 17beta1.
- Allow LLVM <= 18
* Mon May  6 2024 max@suse.com
- Prepare for PostgreSQL 17.
- Make sure all compilation and doc generation happens in %%build.
* Tue Mar 12 2024 aaronpuchert@alice-dsl.net
- Require LLVM <= 17 for now, because LLVM 18 doesn't seem to work.
* Thu Mar  7 2024 sarah.kriesch@opensuse.org
- Remove constraints file because improved memory usage for s390x
* Thu Feb 29 2024 dimstar@opensuse.org
- Use %%patch -P N instead of deprecated %%patchN.
* Thu Feb  8 2024 max@suse.com
- Upgrade to 16.2:
  * bsc#1219679, CVE-2024-0985: Tighten security restrictions
    within REFRESH MATERIALIZED VIEW CONCURRENTLY.
    One step of a concurrent refresh command was run under weak
    security restrictions. If a materialized view's owner could
    persuade a superuser or other high-privileged user to perform a
    concurrent refresh on that view, the view's owner could control
    code executed with the privileges of the user running REFRESH.
    Fix things so that all user-determined code is run as the
    view's owner, as expected
  * If you use GIN indexes, you may need to reindex after updating
    to this release.
  * LLVM 18 is now supported.
  * https://www.postgresql.org/docs/release/16.2/
* Wed Nov  8 2023 max@suse.com
- Upgrade to 16.1:
  * bsc#1216962, CVE-2023-5868: Fix handling of unknown-type
    arguments in DISTINCT "any" aggregate functions. This error led
    to a text-type value being interpreted as an unknown-type value
    (that is, a zero-terminated string) at runtime. This could
    result in disclosure of server memory following the text value.
  * bsc#1216961, CVE-2023-5869: Detect integer overflow while
    computing new array dimensions. When assigning new elements to
    array subscripts that are outside the current array bounds, an
    undetected integer overflow could occur in edge cases. Memory
    stomps that are potentially exploitable for arbitrary code
    execution are possible, and so is disclosure of server memory.
  * bsc#1216960, CVE-2023-5870: Prevent the pg_signal_backend role
    from signalling background workers and autovacuum processes.
    The documentation says that pg_signal_backend cannot issue
    signals to superuser-owned processes. It was able to signal
    these background  processes, though, because they advertise a
    role OID of zero. Treat that as indicating superuser ownership.
    The security implications of cancelling one of these process
    types are fairly small so far as the core code goes (we'll just
    start another one), but extensions might add background workers
    that are more vulnerable.
    Also ensure that the is_superuser parameter is set correctly in
    such processes. No specific security consequences are known for
    that oversight, but it might be significant for some extensions.
  * Add support for LLVM 16 and 17
  * https://www.postgresql.org/docs/16/release-16-1.html
* Tue Oct 31 2023 max@suse.com
- boo#1216734: Revert the last change and make the devel package
  independend of all other subpackages except for the libs.
* Tue Oct 10 2023 max@suse.com
- boo#1216022: Call install-alternatives from the devel subpackage
  as well, otherwise the symlink for ecpg might be missing.
* Mon Sep 18 2023 dimstar@opensuse.org
- Also buildignore the postgresql*-implementation symbols: this is
  needed in order to bootstrap when no postgresql version currently
  has valid symbols provided. Once the packages are built, OBS
  could translate this to the pgname-* packages and accept the
  ignores; during bootstrap though, there is nothing providing the
  symbol and the existing buildignores do not suffice.
* Thu Sep 14 2023 max@suse.com
- Upgrade to 16.0:
  * https://www.postgresql.org/about/news/2715
  * https://www.postgresql.org/docs/16/release-16.html
* Mon Sep  4 2023 mrueckert@suse.de
- Upgrade to 16rc1:
  * https://www.postgresql.org/about/news/2702/
* Thu Aug 10 2023 max@suse.com
- Upgrade too v16beta3.
* Mon Aug  7 2023 max@suse.com
- Copy postgresql15 to postgresql16 and upgrade to v16beta2.
- Don't create a unix domain socket under /tmp anymore.
* Fri May 26 2023 max@suse.com
- Restore the independence of mini builds from the main build after
  the -mini name change from April 4, 2023.
- Adjust icu handling to prepare for PostgreSQL 16.
* Mon May 15 2023 max@suse.com
- Overhaul postgresql-README.SUSE and move it from the binary
  package to the noarch wrapper package.
- Change the unix domain socket location from /var/run to /run.
* Tue May  9 2023 max@suse.com
- Update to 15.3:
  * bsc#1211228, CVE-2023-2454:
    Prevent CREATE SCHEMA from defeating changes in search_path
  * bsc#1211229, CVE-2023-2455: Enforce row-level security
    policies correctly after inlining a set-returning function
  * https://www.postgresql.org/about/news/2637/
  * https://www.postgresql.org/docs/15/release-15-3.html
* Tue Apr 18 2023 max@suse.com
- bsc#1210303: Stop using the obsolete internal %%_restart_on_update
  macro and drop support for sysv init to simplify the scriptlets.
* Tue Apr  4 2023 fvogt@suse.com
- Include -mini in Name: to avoid conflicts in the source package
  name and OBS internal dependency tracking.
* Thu Feb  9 2023 max@suse.com
- Update to 15.2:
  * CVE-2022-41862, bsc#1208102: memory leak in libpq
  * https://www.postgresql.org/about/news/2592/
  * https://www.postgresql.org/docs/15/release-15-2.html
- Bump latest_supported_llvm_ver to 15.
* Thu Nov 10 2022 max@suse.com
- Update to 15.1:
  * https://www.postgresql.org/about/news/2543/
  * https://www.postgresql.org/docs/15/release-15-1.html
* Thu Oct 13 2022 max@suse.com
- Update to 15.0:
  * https://www.postgresql.org/about/news/p-2526/
  * https://www.postgresql.org/docs/15/release-15.html
- Move pg_upgrade from *-contrib to *-server.
- Drop support for the 9.x versioning scheme.
* Thu Oct  6 2022 max@suse.com
- Update to 15~rc2
  * https://www.postgresql.org/about/news/p-2521/
  * Reverting the "optimized order of GROUP BY keys" feature.
* Fri Sep 30 2022 fvogt@suse.com
- Fix source URLs
* Thu Sep 29 2022 max@suse.com
- Update to 15~rc1
  https://www.postgresql.org/about/news/p-2516/
* Thu Sep 22 2022 aaronpuchert@alice-dsl.net
- Create mechanism to specify the latest supported LLVM version.
  Automatically pin to that version if the distribution has a newer
  unsupported default version.
* Mon Sep 12 2022 schwab@suse.de
- Disable LLVM JIT on riscv64
* Thu Sep  8 2022 max@suse.com
- Update to 15~beta4
  https://www.postgresql.org/about/news/p-2507/
* Mon Sep  5 2022 max@suse.com
- Update to 15~beta3
  https://www.postgresql.org/about/news/p-2496/
* Sat May 21 2022 mrueckert@suse.de
- use %%version requires for the contrib package for now as
  15~beta1 is actually smaller than 15.
* Sat May 21 2022 mrueckert@suse.de
- Add proper conditionals for lz4 and zstd
* Sat May 21 2022 mrueckert@suse.de
- Upgrade to 15~beta1
  https://www.postgresql.org/about/news/postgresql-15-beta-1-released-2453/
  https://www.postgresql.org/docs/15/release-15.html
- Refreshed patches to apply cleanly again:
  0001-jit-Workaround-potential-datalayout-mismatch-on-s390.patch
  postgresql-conf.patch
  postgresql-llvm-optional.patch
  postgresql-plperl-keep-rpath.patch
  postgresql-testsuite-keep-results-file.patch
  postgresql-var-run-socket.patch
- Add buildrequires for lz4 and zstd support
* Sat May 21 2022 mrueckert@suse.de
- fork package for postgresql 15
* Thu May 12 2022 max@suse.com
- Upgrade to 14.3:
  * bsc#1199475, CVE-2022-1552: Confine additional operations
    within "security restricted operation" sandboxes.
  * https://www.postgresql.org/docs/14/release-14-3.html
* Wed Apr 13 2022 max@suse.com
- bsc#1198166: Pin to llvm13 until the next patchlevel update.
* Tue Feb  8 2022 max@suse.com
- bsc#1195680: Upgrade to 14.2:
  * https://www.postgresql.org/docs/14/release-14-2.html
  * Reindexing might be needed after applying this upgrade, so
    please read the release notes carefully.
* Sat Dec 11 2021 ada.lovelace@gmx.de
- boo#1190740: Add constraints file with 12GB of memory for s390x
  as a workaround
* Thu Nov 25 2021 max@suse.com
- Add a llvmjit-devel subpackage to pull in the right versions
  of clang and llvm for building extensions.
- Fix some mistakes in the interdependencies between the
  implementation packages and their noarch counterpart.
- Update the BuildIgnore section.
* Wed Nov 10 2021 max@suse.com
- bsc#1192516: Upgrade to 14.1
  * Make the server reject extraneous data after an SSL or GSS
    encryption handshake (CVE-2021-23214).
  * Make libpq reject extraneous data after an SSL or GSS
    encryption handshake (CVE-2021-23222).
  * https://www.postgresql.org/docs/14/release-14-1.html
* Wed Oct 20 2021 max@suse.com
- boo#1191782: Let rpmlint ignore shlib-policy-name-error.
* Tue Oct  5 2021 max@suse.com
- Remove postgresql-testsuite-int8.sql.patch, because its purpose
  is unclear. This affects only the test subpackage.
* Thu Sep 30 2021 mrueckert@suse.de
- Upgrade to 14.0
  https://www.postgresql.org/about/news/postgresql-14-released-2318/
  https://www.postgresql.org/docs/14/release-14.html
* Mon Sep 27 2021 max@suse.com
- Let genlists skip non-existing binaries to avoid lots of version
  conditionals in the file lists.
* Sat Sep 25 2021 mrueckert@suse.de
- Upgrade to 14~rc1
  https://www.postgresql.org/about/news/postgresql-14-rc-1-released-2309/
  https://www.postgresql.org/docs/14/release-14.html
  https://wiki.postgresql.org/wiki/PostgreSQL_14_Open_Items
* Fri Jun 25 2021 mrueckert@suse.de
- Upgrade to 14~beta2
  https://www.postgresql.org/about/news/postgresql-14-beta-2-released-2249/
  https://www.postgresql.org/docs/14/release-14.html
  https://wiki.postgresql.org/wiki/PostgreSQL_14_Open_Items
* Fri May 21 2021 mrueckert@suse.de
- Upgrade to 14~beta1
  https://www.postgresql.org/about/news/postgresql-14-beta-1-released-2213/
  https://www.postgresql.org/docs/14/release-14.html
  https://wiki.postgresql.org/wiki/PostgreSQL_14_Open_Items
- disable postgresql-testsuite-int8.sql.patch:
  it seems it is not needed anymore, need to be double checked.
* Wed May 19 2021 max@suse.com
- bsc#1185952: llvm12 breaks PostgreSQL 11 and 12 on s390x.
  Use llvm11 as a workaround.
* Tue May 11 2021 max@suse.com
- Upgrade to version 13.3:
  * https://www.postgresql.org/docs/13/release-13-3.html
  * CVE-2021-32027, bsc#1185924:
    Prevent integer overflows in array subscripting calculations.
  * CVE-2021-32028, bsc#1185925: Fix mishandling of “junk”
    columns in INSERT ... ON CONFLICT ... UPDATE target lists.
  * CVE-2021-32029, bsc#1185926: Fix possibly-incorrect
    computation of UPDATE ... RETURNING
    "pg_psql_temporary_savepoint" does not exist”.
- Don't use %%_stop_on_removal, because it was meant to be private
  and got removed from openSUSE. %%_restart_on_update is also
  private, but still supported and needed for now (bsc#1183168).
* Mon Mar 15 2021 max@suse.com
- Re-enable build of the llvmjit subpackage on SLE, but it will
  only be delivered on PackageHub for now (boo#1183118).
* Tue Mar  9 2021 max@suse.com
- Remove leftover PreReq on chkconfig, we stopped using it long
  time ago.
* Fri Feb 19 2021 max@suse.com
- boo#1179945: Disable icu for PostgreSQL 10 (and older) on TW.
* Wed Feb 10 2021 max@suse.com
- Upgrade to version 13.2:
  * https://www.postgresql.org/docs/13/release-13-2.html
  * Updating stored views and reindexing might be needed after
    applying this update.
  * CVE-2021-3393, bsc#1182040: Fix information leakage in
    constraint-violation error messages.
  * CVE-2021-20229, bsc#1182039: Fix failure to check per-column
    SELECT privileges in some join queries.
  * Obsoletes postgresql-icu68.patch.
* Mon Dec 14 2020 gmbr3@opensuse.org
- Add postgresql-icu68.patch: fix build with ICU 68
* Fri Nov 20 2020 max@suse.com
- bsc#1178961: %%ghost the symlinks to pg_config and ecpg.
- boo#1179765: BuildRequire libpq5 and libecpg6 when not building
  them to avoid dangling symlinks in the devel package.
* Wed Nov 11 2020 max@suse.com
- Upgrade to version 13.1:
  * CVE-2020-25695, bsc#1178666: Block DECLARE CURSOR ... WITH HOLD
    and firing of deferred triggers within index expressions and
    materialized view queries.
  * CVE-2020-25694, bsc#1178667:
    a) Fix usage of complex connection-string parameters in pg_dump,
    pg_restore, clusterdb, reindexdb, and vacuumdb.
    b) When psql's \connect command re-uses connection parameters,
    ensure that all non-overridden parameters from a previous
    connection string are re-used.
  * CVE-2020-25696, bsc#1178668: Prevent psql's \gset command from
    modifying specially-treated variables.
  * Fix recently-added timetz test case so it works when the USA
    is not observing daylight savings time.
    (obsoletes postgresql-timetz.patch)
  * https://www.postgresql.org/about/news/2111/
  * https://www.postgresql.org/docs/13/release-13-1.html
* Tue Nov  3 2020 max@suse.com
- Fix a DST problem in the test suite: postgresql-timetz.patch
  https://postgr.es/m/16689-57701daa23b377bf@postgresql.org
* Fri Sep 25 2020 max@suse.com
- Initial packaging of PostgreSQL 13:
  * https://www.postgresql.org/about/news/2077/
  * https://www.postgresql.org/docs/13/release-13.html
