/*
 * Copyright 2014 Ben Manes. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.github.benmanes.caffeine.cache;

import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentMap;

import org.checkerframework.checker.nullness.qual.NonNull;

/**
 * A semi-persistent mapping from keys to values. Values are automatically loaded by the cache
 * asynchronously, and are stored in the cache until either evicted or manually invalidated.
 * <p>
 * Implementations of this interface are expected to be thread-safe, and can be safely accessed
 * by multiple concurrent threads.
 *
 * @author ben.manes@gmail.com (Ben Manes)
 * @param <K> the type of keys maintained by this cache
 * @param <V> the type of mapped values
 */
public interface AsyncLoadingCache<K, V> extends AsyncCache<K, V> {

  /**
   * Returns the future associated with {@code key} in this cache, obtaining that value from
   * {@link CacheLoader#asyncLoad} if necessary. If the asynchronous computation fails, the entry
   * will be automatically removed from this cache.
   * <p>
   * If the specified key is not already associated with a value, attempts to compute its value
   * asynchronously and enters it into this cache unless {@code null}. The entire method invocation
   * is performed atomically, so the function is applied at most once per key.
   *
   * @param key key with which the specified value is to be associated
   * @return the current (existing or computed) future value associated with the specified key
   * @throws NullPointerException if the specified key is null or if the future returned by the
   *         {@link AsyncCacheLoader} is null
   * @throws RuntimeException or Error if the {@link CacheLoader} does when constructing the future,
   *         in which case the mapping is left unestablished
   */
  @NonNull
  CompletableFuture<V> get(@NonNull K key);

  /**
   * Returns the future of a map of the values associated with {@code keys}, creating or retrieving
   * those values if necessary. The returned map contains entries that were already cached, combined
   * with newly loaded entries; it will never contain null keys or values. If the any of the
   * asynchronous computations fail, those entries will be automatically removed from this cache.
   * <p>
   * Caches loaded by a {@link CacheLoader} supporting bulk loading will issue a single request to
   * {@link CacheLoader#asyncLoadAll} for all keys which are not already present in the cache. If
   * another call to {@link #get} tries to load the value for a key in {@code keys}, that thread
   * simply waits for this computation to finish and returns the loaded value. Caches that do not
   * use a {@link CacheLoader} with an optimized bulk load implementation will sequentially load
   * each key by making individual {@link CacheLoader#asyncLoad} calls. Note that multiple threads
   * can concurrently load values for distinct keys.
   * <p>
   * Note that duplicate elements in {@code keys}, as determined by {@link Object#equals}, will be
   * ignored.
   *
   * @param keys the keys whose associated values are to be returned
   * @return the future containing an unmodifiable mapping of keys to values for the specified keys
   *         in this cache
   * @throws NullPointerException if the specified collection is null or contains a null element, or
   *         if the future returned by the {@link AsyncCacheLoader} is null
   * @throws RuntimeException or Error if the {@link CacheLoader} does so, if
   *         {@link CacheLoader#asyncLoadAll} returns {@code null}, or fails when constructing the
   *         future, in which case the mapping is left unestablished
   */
  @NonNull
  CompletableFuture<Map<K, V>> getAll(@NonNull Iterable<? extends @NonNull K> keys);

  /**
   * Returns a view of the entries stored in this cache as a thread-safe map. Modifications made to
   * the map directly affect the cache.
   * <p>
   * Iterators from the returned map are at least <i>weakly consistent</i>: they are safe for
   * concurrent use, but if the cache is modified (including by eviction) after the iterator is
   * created, it is undefined which of the changes (if any) will be reflected in that iterator.
   *
   * @return a thread-safe view of this cache supporting all of the optional {@link Map} operations
   */
  @Override
  default @NonNull ConcurrentMap<@NonNull K, @NonNull CompletableFuture<V>> asMap() {
    // This method was added & implemented in version 2.7.0
    throw new UnsupportedOperationException();
  }

  /**
   * Returns a view of the entries stored in this cache as a synchronous {@link LoadingCache}. A
   * mapping is not present if the value is currently being loaded. Modifications made to the
   * synchronous cache directly affect the asynchronous cache. If a modification is made to a
   * mapping that is currently loading, the operation blocks until the computation completes.
   *
   * @return a thread-safe synchronous view of this cache
   */
  @NonNull
  @Override
  LoadingCache<K, V> synchronous();
}
