#
# spec file for package shim
#
# Copyright (c) 2021 SUSE LLC
#
# All modifications and additions to the file contributed by third parties
# remain the property of their copyright owners, unless otherwise agreed
# upon. The license for this file, and modifications and additions to the
# file, is the same license as for the pristine package itself (unless the
# license for the pristine package is not an Open Source License, in which
# case the license is the MIT License). An "Open Source License" is a
# license that conforms to the Open Source Definition (Version 1.9)
# published by the Open Source Initiative.

# Please submit bugfixes or comments via https://bugs.opensuse.org/
#
# needssslcertforbuild


%undefine _debuginfo_subpackages
%undefine _build_create_debug
%ifarch aarch64
%define grubplatform arm64-efi
%else
%define grubplatform %{_target_cpu}-efi
%endif
%if %{defined sle_version} && 0%{?sle_version} <= 150000
%define sysefidir      /usr/lib64/efi
%else
%define sysefibasedir  %{_datadir}/efi
%define sysefidir      %{sysefibasedir}/%{_target_cpu}
%if "%{grubplatform}" == "x86_64-efi" && 0%{?suse_version} < 1600
# provide compatibility sym-link for residual kiwi, etc.
%define shim_lib64_share_compat 1
%endif
%endif

Name:           shim
Version:        15.8
Release:        150300.4.23.1
Summary:        UEFI shim loader
License:        BSD-2-Clause
Group:          System/Boot
URL:            https://github.com/rhboot/shim
Source:         %{name}-%{version}.tar.bz2
# run "extract_signature.sh shim.efi" where shim.efi is the binary
# with the signature from the UEFI signing service.
# Note: For signature requesting, check SIGNATURE_UPDATE.txt
Source1:        signature-opensuse.x86_64.asc
Source2:        openSUSE-UEFI-CA-Certificate.crt
Source3:        shim-install
Source4:        SLES-UEFI-CA-Certificate.crt
Source5:        extract_signature.sh
Source6:        attach_signature.sh
Source7:        show_hash.sh
Source8:        show_signatures.sh
Source9:        timestamp.pl
Source10:       strip_signature.sh
Source11:       signature-sles.x86_64.asc
Source12:       signature-opensuse.aarch64.asc
Source13:       signature-sles.aarch64.asc
Source14:       generate-vendor-dbx.sh
# revoked certificates for dbx
Source50:       revoked-openSUSE-UEFI-SIGN-Certificate-2013-01.crt
Source51:       revoked-openSUSE-UEFI-SIGN-Certificate-2013-08.crt
Source52:       revoked-openSUSE-UEFI-SIGN-Certificate-2020-01.crt
Source53:       revoked-openSUSE-UEFI-SIGN-Certificate-2020-07.crt
Source54:       revoked-openSUSE-UEFI-SIGN-Certificate-2021-05.crt
Source55:       revoked-SLES-UEFI-SIGN-Certificate-2013-01.crt
Source56:       revoked-SLES-UEFI-SIGN-Certificate-2013-04.crt
Source57:       revoked-SLES-UEFI-SIGN-Certificate-2016-02.crt
Source58:       revoked-SLES-UEFI-SIGN-Certificate-2020-07.crt
Source59:       revoked-SLES-UEFI-SIGN-Certificate-2021-05.crt
###
Source99:       SIGNATURE_UPDATE.txt
# PATCH-FIX-SUSE shim-arch-independent-names.patch glin@suse.com -- Use the Arch-independent names
Patch1:         shim-arch-independent-names.patch
# PATCH-FIX-OPENSUSE shim-change-debug-file-path.patch glin@suse.com -- Change the default debug file path
Patch2:         shim-change-debug-file-path.patch
# PATCH-FIX-SUSE shim-bsc1177315-verify-eku-codesign.patch bsc#1177315 glin@suse.com -- Verify CodeSign in the signer's EKU
Patch3:         shim-bsc1177315-verify-eku-codesign.patch
# PATCH-FIX-SUSE remove_build_id.patch -- Remove the build ID to make the binary reproducible when building with AArch64 container
Patch4:         remove_build_id.patch
# PATCH-FIX-SUSE shim-disable-export-vendor-dbx.patch bsc#1185261 glin@suse.com -- Disable exporting vendor-dbx to MokListXRT
Patch5:         shim-disable-export-vendor-dbx.patch
BuildRequires:  dos2unix
BuildRequires:  efitools
BuildRequires:  mozilla-nss-tools
BuildRequires:  openssl >= 0.9.8
BuildRequires:  pesign
BuildRequires:  pesign-obs-integration
%if 0%{?suse_version} >= 1600
BuildRequires:  fde-tpm-helper-rpm-macros
%endif
%if 0%{?suse_version} > 1320
BuildRequires:  update-bootloader-rpm-macros
%endif
%if 0%{?update_bootloader_requires:1}
%update_bootloader_requires
%else
Requires:       perl-Bootloader
%endif
%if 0%{?fde_tpm_update_requires:1}
%fde_tpm_update_requires
%endif
BuildRoot:      %{_tmppath}/%{name}-%{version}-build
# For shim-install script grub is needed but we also want to use
# shim for systemd-boot where shim-install is not actually used.
# Requires:       grub2-%{grubplatform}
Requires:       mokutil
ExclusiveArch:  x86_64 aarch64

%description
shim is a trivial EFI application that, when run, attempts to open and
execute another application.

%package -n shim-debuginfo
Summary:        UEFI shim loader - debug symbols
Group:          Development/Debug

%description -n shim-debuginfo
The debug symbols of UEFI shim loader

%package -n shim-debugsource
Summary:        UEFI shim loader - debug source
Group:          Development/Debug

%description -n shim-debugsource
The source code of UEFI shim loader

%prep
%setup -q
%patch1 -p1
%patch2 -p1
%patch3 -p1
%patch4 -p1
%patch5 -p1

%build
# generate the vendor SBAT metadata
%if 0%{?is_opensuse} == 1 || 0%{?sle_version} == 0
distro_id="opensuse"
distro_name="The openSUSE project"
%else
distro_id="sle"
distro_name="SUSE Linux Enterprise"
%endif
distro_sbat=1
sbat="shim.${distro_id},${distro_sbat},${distro_name},%{name},%{version},mail:security@suse.de"
echo "${sbat}" > data/sbat.vendor.csv

# generate dbx files based on revoked certs
bash %{_sourcedir}/generate-vendor-dbx.sh %{_sourcedir}/revoked-*.crt
ls -al *.esl

# first, build MokManager and fallback as they don't depend on a
# specific certificate
make RELEASE=0 \
     MMSTEM=MokManager FBSTEM=fallback \
     MokManager.efi.debug fallback.efi.debug \
     MokManager.efi fallback.efi
# make sure all object files gets rebuilt
rm -f *.o

# now build variants of shim that embed different certificates
default=''
suffixes=(opensuse sles)
# check whether the project cert is a known one. If it is we build
# just one shim that embeds this specific cert. If it's a devel
# project we build all variants to simplify testing.
if test -e %{_sourcedir}/_projectcert.crt ; then
    prjsubject=$(openssl x509 -in %{_sourcedir}/_projectcert.crt -noout -subject_hash)
    prjissuer=$(openssl x509 -in %{_sourcedir}/_projectcert.crt -noout -issuer_hash)
    opensusesubject=$(openssl x509 -in %{SOURCE2} -noout -subject_hash)
    slessubject=$(openssl x509 -in %{SOURCE4} -noout -subject_hash)
    if test "$prjissuer" = "$opensusesubject" ; then
	suffixes=(opensuse)
    elif test "$prjissuer" = "$slessubject" ; then
	suffixes=(sles)
    elif test "$prjsubject" = "$prjissuer" ; then
	suffixes=(devel opensuse sles)
    fi
fi

for suffix in "${suffixes[@]}"; do
    if test "$suffix" = "opensuse"; then
	cert=%{SOURCE2}
	verify='openSUSE Secure Boot CA1'
	vendor_dbx='vendor-dbx-opensuse.esl'
%ifarch x86_64
	signature=%{SOURCE1}
%else
	# AArch64 signature
	# Disable AArch64 signature attachment temporarily
	# until we get a real one.
	#signature=%{SOURCE12}
%endif
    elif test "$suffix" = "sles"; then
	cert=%{SOURCE4}
	verify='SUSE Linux Enterprise Secure Boot CA1'
	vendor_dbx='vendor-dbx-opensuse.esl'
%ifarch x86_64
	signature=%{SOURCE11}
%else
	# AArch64 signature
	signature=%{SOURCE13}
%endif
    elif test "$suffix" = "devel"; then
	cert=%{_sourcedir}/_projectcert.crt
	verify=`openssl x509 -in "$cert" -noout -email`
	vendor_dbx='vendor-dbx.esl'
	signature=''
	test -e "$cert" || continue
    else
	echo "invalid suffix"
	false
    fi

    openssl x509 -in $cert -outform DER -out shim-$suffix.der
    make RELEASE=0 SHIMSTEM=shim \
         VENDOR_CERT_FILE=shim-$suffix.der ENABLE_HTTPBOOT=1 \
         DEFAULT_LOADER="\\\\\\\\grub.efi" \
         VENDOR_DBX_FILE=$vendor_dbx \
         shim.efi.debug shim.efi
    #
    # assert correct certificate embedded
    grep -q "$verify" shim.efi
    # make VENDOR_CERT_FILE=cert.der VENDOR_DBX_FILE=dbx
    chmod 755 %{SOURCE9}
    # alternative: verify signature
    #sbverify --cert MicCorThiParMarRoo_2010-10-05.pem shim-signed.efi
    if test -n "$signature"; then
	head -1 "$signature" > hash1
	cp shim.efi shim.efi.bak
	# pe header contains timestamp and checksum. we need to
	# restore that
	%{SOURCE9} --set-from-file "$signature" shim.efi
	pesign -h -P -i shim.efi > hash2
	cat hash1 hash2
	if ! cmp -s hash1 hash2; then
		echo "ERROR: $suffix binary changed, need to request new signature!"
%if %{defined shim_enforce_ms_signature} && 0%{?shim_enforce_ms_signature} > 0
		# compare suffix (sles, opensuse) with distro_id (sle, opensuse)
		# when hash mismatch and distro_id match with suffix, stop building 
		if test "$suffix" = "$distro_id" || test "$suffix" = "${distro_id}s"; then
			false
		fi
%endif
		mv shim.efi.bak shim-$suffix.efi
		rm shim.efi
	else
		# attach signature
		pesign -m "$signature" -i shim.efi -o shim-$suffix.efi
		rm -f shim.efi
	fi
    else
        mv shim.efi shim-$suffix.efi
    fi
    mv shim.efi.debug shim-$suffix.debug
    # remove the build cert if exists
    rm -f shim_cert.h shim.cer shim.crt
    # make sure all object files gets rebuilt
    rm -f *.o
done

ln -s shim-${suffixes[0]}.efi shim.efi
mv shim-${suffixes[0]}.debug shim.debug

# Collect the source for debugsource
mkdir ../source
find . \( -name "*.c" -o -name "*.h" \) -type f -exec cp --parents -a {} ../source/ \;
mv ../source .

%install
export BRP_PESIGN_FILES='%{sysefidir}/shim*.efi %{sysefidir}/MokManager.efi %{sysefidir}/fallback.efi'
install -d %{buildroot}/%{sysefidir}
cp -a shim*.efi %{buildroot}/%{sysefidir}
install -m 444 shim-*.der %{buildroot}/%{sysefidir}
install -m 644 MokManager.efi %{buildroot}/%{sysefidir}/MokManager.efi
install -m 644 fallback.efi %{buildroot}/%{sysefidir}/fallback.efi
install -d %{buildroot}/%{_sbindir}
install -m 755 %{SOURCE3} %{buildroot}/%{_sbindir}/
# install SUSE certificate
install -d %{buildroot}/%{_sysconfdir}/uefi/certs/
for file in shim-*.der; do
    fpr=$(openssl x509 -sha1 -fingerprint -inform DER -noout -in $file | cut -c 18- | cut -d ":" -f 1,2,3,4 | sed 's/://g')
    install -m 644 $file %{buildroot}/%{_sysconfdir}/uefi/certs/${fpr}-shim.crt
done
%if %{defined shim_lib64_share_compat}
    [ "%{sysefidir}" != "/usr/lib64/efi" ] || exit 1
    # provide compatibility sym-link for residual "consumers"
    install -d %{buildroot}/usr/lib64/efi
    ln -srf %{buildroot}/%{sysefidir}/*.efi %{buildroot}/usr/lib64/efi/
%endif

# install the debug symbols
install -d %{buildroot}/usr/lib/debug/%{sysefidir}
install -m 644 shim.debug %{buildroot}/usr/lib/debug/%{sysefidir}
install -m 644 MokManager.efi.debug %{buildroot}/usr/lib/debug/%{sysefidir}/MokManager.debug
install -m 644 fallback.efi.debug %{buildroot}/usr/lib/debug/%{sysefidir}/fallback.debug

# install the debug source
install -d %{buildroot}/usr/src/debug/%{name}-%{version}
cp -r source/* %{buildroot}/usr/src/debug/%{name}-%{version}

%clean
%{?buildroot:%__rm -rf "%{buildroot}"}

%post
%if 0%{?fde_tpm_update_post:1}
%fde_tpm_update_post shim
%endif

%if 0%{?update_bootloader_check_type_reinit_post:1}
%update_bootloader_check_type_reinit_post grub2-efi
%else
/sbin/update-bootloader --reinit || true
%endif

# copy from kernel-scriptlets/cert-script
is_efi () {
    local msg rc=0
# The below statement fails if mokutil isn't installed or UEFI is unsupported.
# It doesn't fail if UEFI is available but secure boot is off.
    msg="$(mokutil --sb-state 2>&1)" || rc=$?
    return $rc
}
# run mokutil for setting sbat policy to latest mode
EFIVARFS=/sys/firmware/efi/efivars
SBAT_POLICY="$EFIVARFS/SbatPolicy-605dab50-e046-4300-abb6-3dd810dd8b23"
if is_efi; then
        if [ -w $EFIVARFS ] && \
           [ ! -f "$SBAT_POLICY" ] && \
           mokutil -h | grep -q "set-sbat-policy"; \
        then
        # Only apply CA check on the kernel package certs (bsc#1173115)
                mokutil --set-sbat-policy latest
        fi
fi

%if %{defined update_bootloader_posttrans}
%posttrans
%{?update_bootloader_posttrans}
%{?fde_tpm_update_posttrans}
%endif

%files
%defattr(-,root,root)
%doc COPYRIGHT
%dir %{?sysefibasedir}
%dir %{sysefidir}
%{sysefidir}/shim.efi
%{sysefidir}/shim-*.efi
%{sysefidir}/shim-*.der
%{sysefidir}/MokManager.efi
%{sysefidir}/fallback.efi
%{_sbindir}/shim-install
%dir %{_sysconfdir}/uefi/
%dir %{_sysconfdir}/uefi/certs/
%{_sysconfdir}/uefi/certs/*.crt
%if %{defined shim_lib64_share_compat}
# provide compatibility sym-link for previous kiwi, etc.
%dir /usr/lib64/efi
/usr/lib64/efi/*.efi
%endif

%files -n shim-debuginfo
%defattr(-,root,root,-)
/usr/lib/debug%{sysefidir}/shim.debug
/usr/lib/debug%{sysefidir}/MokManager.debug
/usr/lib/debug%{sysefidir}/fallback.debug

%files -n shim-debugsource
%defattr(-,root,root,-)
%dir /usr/src/debug/%{name}-%{version}
/usr/src/debug/%{name}-%{version}/*

%changelog
* Fri Oct  4 2024 dennis.tseng@suse.com
- Update shim-install to apply the missing fix for openSUSE Leap
  (bsc#1210382) fixed by Gary.
  * 86b73d1 Fix that bootx64.efi is not updated on Leap
- Update shim-install to use the 'removable' way for SL-Micro
  (bsc#1230316) fixed by Gary.
  * 433cc4e Always use the removable way for SL-Micro
* Thu Apr 18 2024 dennis.tseng@suse.com
- update public keys of shim-15.8 after it has been signed back
  from Microsoft.
* Thu Feb 15 2024 jlee@suse.com
- Sometimes SLE shim signature be Microsoft updated before openSUSE shim
  signature. When submit request on IBS for updating SLE shim, the submitreq
  project be generated, but it always be blocked by checking the signature
  of openSUSE shim.
  It doesn't make sense checking openSUSE shim signature when building
  SLE shim on SLE platform, and vice versa. So the following change adds the
  logic to compare suffix (sles, opensuse) with distro_id (sle, opensuse).
  When and only when hash mismatch and distro_id match with suffix, stop
  building.
    [#] compare suffix (sles, opensuse) with distro_id (sle, opensuse)
    [#] when hash mismatch and distro_id match with suffix, stop building
- Sync the changelog between openSUSE:Factory/shim with SLE-15-SP3/shim
  - Add CVE-2022-28737 number to "Mon Mar 27 09:26:02 UTC 2023" record
  - Add "Thu Apr 13 05:28:10 UTC 2023" record for updating shim-install
    for bsc#1210382.
  - Add "Thu Apr 13 09:13:22 UTC 2023" record for changing the logic of
    checking shim signature.
* Wed Feb  7 2024 glin@suse.com
- Update shim-install to set the TPM2 SRK algorithm (bsc#1213945)
  92d0f4305df73 Set the SRK algorithm for the TPM2 protector
* Fri Feb  2 2024 glin@suse.com
- Limit the requirement of fde-tpm-helper-macros to the distro with
  suse_version 1600 and above (bsc#1219460)
* Sun Jan 28 2024 dennis.tseng@suse.com
-- Update to version 15.8
  - Various CVE fixes are already merged into this version
    mok: fix LogError() invocation (bsc#1215099,CVE-2023-40546)
    avoid incorrectly trusting HTTP headers (bsc#1215098,CVE-2023-40547)
    Fix integer overflow on SBAT section size on 32-bit system (bsc#1215100,CVE-2023-40548)
    Authenticode: verify that the signature header is in bounds (bsc#1215101,CVE-2023-40549)
    pe: Fix an out-of-bound read in verify_buffer_sbat() (bsc#1215102,CVE-2023-40550)
    pe-relocate: Fix bounds check for MZ binaries (bsc#1215103,CVE-2023-40551)
  - remove shim-Enable-the-NX-compatibility-flag-by-default.patch
    The codes in this patch are already existing in shim-15.8
    The NX flag is disable which is same as the default value of shim-15.8,
    hence, not need to enable it by this patch now.
  - Patches (git log --oneline --reverse 15.7..15.8)
    657b248 Make sbat_var.S parse right with buggy gcc/binutils
    7c76425 Enable the NX compatibility flag by default.
    89972ae CryptoPkg/BaseCryptLib: Fix buffer overflow issue in realloc wrapper
    c7b3051 pe: Align section size up to page size for mem attrs
    e4f40ae pe: Add IS_PAGE_ALIGNED macro
    f23883c Don't loop forever in load_certs() with buggy firmware
    1f38cb3 Optionally allow to keep shim protocol installed
    102a658 Drop invalid calls to `CRYPTO_set_mem_functions`
    aae3df0 test-sbat: Fix exit code
    cca3933 Block Debian grub binaries with SBAT < 4
    cf59f34 Further improve load_certs() for non-compliant drivers/firmwares
    0601f44 SBAT-related documents formatting and spelling
    0640e13 Add a security contact email address in README.md
    0bfc397 Work around malformed path delimiters in file paths from DHCP
    a8b0b60 pe: only process RelocDir->Size of reloc section
    f7a4338 Skip testing msleep()
    549d346 Rename 'msecs' to 'usecs' to avoid potential confusion
    908c388 Change type of fallback_verbose_wait from int to unsigned long
    05eae92 Add SbatLevel_Variable.txt to document the various revocations
    243f125 Use -Wno-unused-but-set-variable for Cryptlib and OpenSSL
    89d25a1 Add a make rule for compile_commands.json
    118ff87 Add gnu-stack notes
    f132655 test: Make our fake dprintf be a statement.
    be00279 Remove CentOS 7 test builds.
    9964960 Split pe.c up even more.
    569270d Test (and fix) ImageAddress()
    61e9894 Verify signature before verifying sbat levels
    1578b55 Add libFuzzer support for csv.c
    a0673e3 Fix a 1-byte memory leak in .sbat parsing.
    e246812 Add libFuzzer support to the .sbat parser.
    fd43eda Work around ImageAddress() usage mistake
    1e985a3 Correctly free memory allocated in handle_image()
    dbbe3c8 mok: Avoid underflow in maximum variable size calculation
    04111d4 Make some of the static analysis tools a little easier to run
    7ba7440 compile_commands.json: remove stuff clang doesn't like
    66e6579 CVE-2023-40546 mok: fix LogError() invocation
    f271826 Add primitives for overflow-checked arithmetic operations.
    8372147 pe-relocate: Add a fuzzer for read_header()
    5a5147d CVE-2023-40551: pe-relocate: Fix bounds check for MZ binaries
    e912071 pe-relocate: make read_header() use checked arithmetic operations.
    93ce255 CVE-2023-40550 pe: Fix an out-of-bound read in verify_buffer_sbat()
    e7f5fdf pe-relocate: Ensure nothing else implements CVE-2023-40550
    afdc503 CVE-2023-40549 Authenticode: verify that the signature header is in bounds.
    96dccc2 CVE-2023-40548 Fix integer overflow on SBAT section size on 32-bit system
    dae82f6 Further mitigations against CVE-2023-40546 as a class
    ea0f9df Allow SbatLevel data from external binary
    b078ef2 Always clear SbatLevel when Secure Boot is disabled
    7dfb687 BS Variables for bootmgr revocations
    a967c0e shim should not self revoke
    577cedd Print message when refusing to apply SbatLevel
    e801b0d sbat revocations: check the full section name
    0226b56 CVE-2023-40547 - avoid incorrectly trusting HTTP headers
    6f0c8d2 Print errors when setting/clearing memory attrs
    57c0eed Updated Revocations for January 2024 CVEs
    49c6d95 Fix some minor ia32 build issues.
    be8ff7c post-process-pe: Don't set the NX_COMPAT flag by default after all.
    13abd9f pe-relocate: Avoid __builtin_add_overflow() on GCC < 5
    c46c975 Suppress "Failed to open <..>\revocations.efi" when file does not exist
    30a4f37 Rename "previous" revocations to "automatic"
    6f395c2 Build time selectable automatic SBATLevel revocations
    a23e2f0 netboot read_image() should not hardcode DEFAULT_LOADER
    993a345 Try to load revocations.efi even if directory read fails
    1770a03 gitmodules: use shim-15.8 for gnu-efi branch
    5914984 (HEAD -> main, tag: latest-release, tag: 15.8, origin/main, origin/HEAD) Bump version to 15.8
* Wed Jan 24 2024 lnussel@suse.com
- Generate dbx during build so we don't include binary files in sources
* Thu Oct  5 2023 lnussel@suse.com
- Don't require grub so shim can still be used with systemd-boot
* Wed Sep 20 2023 mchang@suse.com
- Update shim-install to fix boot failure of ext4 root file system
  on RAID10 (bsc#1205855)
  226c94ca5cfca  Use hint in looking for root if possible
* Tue Sep 19 2023 glin@suse.com
- Adopt the macros from fde-tpm-helper-macros to update the
  signature in the sealed key after a bootloader upgrade
* Mon May 15 2023 glin@suse.com
- Update shim-install to amend full disk encryption support
    b540061e041b  Adopt TPM 2.0 Key File for grub2 TPM 2.0 protector
    f2e8143ce831  Use the long name to specify the grub2 key protector
    72830120e5ea  cryptodisk: support TPM authorized policies
    49e7a0d307f3  Do not use tpm_record_pcrs unless the command is in command.lst
* Thu Apr 13 2023 jlee@suse.com
- Sometimes SLE shim signature be Microsoft updated before openSUSE shim
  signature. When submit request on IBS for updating SLE shim, the submitreq
  project be generated, but it always be blocked by checking the signature
  of openSUSE shim.
  It doesn't make sense checking openSUSE shim signature when building
  SLE shim on SLE platform, and vice versa. So the following change adds the
  logic to compare suffix (sles, opensuse) with distro_id (sle, opensuse).
  When and only when hash mismatch and distro_id match with suffix, stop
  building.
    [#] compare suffix (sles, opensuse) with distro_id (sle, opensuse)
    [#] when hash mismatch and distro_id match with suffix, stop building
* Thu Apr 13 2023 jlee@suse.com
- Upgrade shim-install for bsc#1210382
  After closing Leap-gap project since Leap 15.3, openSUSE Leap direct
  uses shim from SLE. So the ca_string is 'SUSE Linux Enterprise Secure Boot
  CA1', not 'openSUSE Secure Boot CA1'. It causes that the update_boot=no,
  so all files in /boot/efi/EFI/boot are not updated.
  The 86b73d1 patch added the logic that using ID field in os-release for
  checking Leap distro and set ca_string to 'SUSE Linux Enterprise Secure
  Boot CA1'. Then /boot/efi/EFI/boot/* can also be updated.
- https://github.com/SUSE/shim-resources (git log --oneline)
  86b73d1 Fix that bootx64.efi is not updated on Leap
  f2e8143 Use the long name to specify the grub2 key protector
  7283012 cryptodisk: support TPM authorized policies
  49e7a0d Do not use tpm_record_pcrs unless the command is in command.lst
  26c6bd5 Have grub take a snapshot of "relevant" TPM PCRs
  5c2c3ad Handle different cases of controlling cryptomount volumes during first stage boot
  a5c5734 Introduce --no-grub-install option
* Mon Apr 10 2023 jlee@suse.com
- Removed POST_PROCESS_PE_FLAGS=-N from the build command in shim.spec to
  enable the NX compatibility flag when using post-process-pe after
  discussed with grub2 experts in mail. It's useful for further development
  and testing. (bsc#1205588)
* Mon Mar 27 2023 jlee@suse.com
- Updated shim signature after shim 15.7 of SLE be signed back:
  signature-sles.x86_64.asc, signature-sles.aarch64.asc (bsc#1198458, CVE-2022-28737)
* Thu Jan 12 2023 jlee@suse.com
- Removed shim-bsc1198101-opensuse-cert-prompt.patch (bsc#1198101)
  - Detail discussion is in bugzilla:
  https://bugzilla.suse.com/show_bug.cgi?id=1198101
  - The shim community review and challenge this prompt. No other
    distro shows prompt (Have checked Fedora 37, CentOS 9 and Ubuntu 22.10).
    Currently, it blocked the review process of openSUSE shim.
  - Other distros lock-down kernel when secure boot is enabled. Some of
    them used different key for signing kernel binary with In-tree kernel
    module. And their build service does not provide signed Out-off-tree
    module.
* Fri Dec  9 2022 jlee@suse.com
- Modified shim-install, add the following Olaf Kirch's patches to support
  full disk encryption: (jsc#PED-922)
    a5c57340740c	Introduce --no-grub-install option
    5c2c3addc51f	Handle different cases of controlling cryptomount volumes during first stage boot
    26c6bd5df7ae	Have grub take a snapshot of "relevant" TPM PCRs
* Wed Nov 23 2022 jlee@suse.com
- Add POST_PROCESS_PE_FLAGS=-N to the build command in shim.spec to
  disable the NX compatibility flag when using post-process-pe because
  grub2 is not ready. (bsc#1205588)
  - Kernel can boot with the NX compatibility flag since 82e0d6d76a2a7
    be merged to v5.19. On the other hand, upstream is working on
    improve compressed kernel stage for NX:
    [PATCH v3 00/24] x86_64: Improvements at compressed kernel stage
    https://www.spinics.net/lists/kernel/msg4599636.html
* Fri Nov 18 2022 jlee@suse.com
- Add shim-Enable-the-NX-compatibility-flag-by-default.patch to
  enable the NX compatibility flag by default. (jsc#PED-127)
* Fri Nov 18 2022 jlee@suse.com
- Drop upstreamed patch:
  - shim-Enable-TDX-measurement-to-RTMR-register.patch
  - Enable TDX measurement to RTMR register (jsc#PED-1273)
  - 4fd484e4c2	15.7
* Thu Nov 17 2022 jlee@suse.com
- Update to 15.7 (bsc#1198458)(jsc#PED-127)
  - Patches (git log --oneline --reverse 15.6..15.7)
  0eb07e1 Make SBAT variable payload introspectable
  092c2b2 Reference MokListRT instead of MokList
  8b59b69 Add a link to the test plan in the readme.
  4fd484e Enable TDX measurement to RTMR register
  14d6339 Discard load-options that start with a NUL
  5c537b3 shim: Flush the memory region from i-cache before execution
  2d4ebb5 load_cert_file: Fix stack issue
  ea4911c load_cert_file: Use EFI RT memory function
  0cf43ac Add -malign-double to IA32 compiler flags
  17f0233 pe: Fix image section entry-point validation
  5169769 make-archive: Build reproducible tarball
  aa1b289 mok: remove MokListTrusted from PCR 7
  53509ea CryptoPkg/BaseCryptLib: fix NULL dereference
  616c566 More coverity modeling
  ea0d0a5 Update shim's .sbat to sbat,3
  dd8be98 Bump grub's sbat requirement to grub,3
  1149161 (HEAD -> main, tag: 15.7, origin/main, origin/HEAD) Update version to 15.7
  - 15.7 release note https://github.com/rhboot/shim/releases
  Make SBAT variable payload introspectable by @chrisccoulson in #483
  Reference MokListRT instead of MokList by @esnowberg in #488
  Add a link to the test plan in the readme. by @vathpela in #494
  [V3] Enable TDX measurement to RTMR register by @kenplusplus in #485
  Discard load-options that start with a NUL by @frozencemetery in #505
  load_cert_file bugs by @esnowberg in #523
  Add -malign-double to IA32 compiler flags by @nicholasbishop in #516
  pe: Fix image section entry-point validation by @iokomin in #518
  make-archive: Build reproducible tarball by @julian-klode in #527
  mok: remove MokListTrusted from PCR 7 by @baloo in #519
  - Drop upstreamed patch:
  - shim-bsc1177789-fix-null-pointer-deref-AuthenticodeVerify.patch
  - Cryptlib/CryptAuthenticode: fix NULL pointer dereference in  AuthenticodeVerify()
  - 53509eaf22	15.7
  - shim-jscPED-127-upgrade-shim-in-SLE15-SP5.patch
  - For backporting the following patches between 15.6 with aa1b289a1a (jsc#PED-127)
  - The following patches are merged to 15.7
  aa1b289a1a mok: remove MokListTrusted from PCR 7
  0cf43ac6d7 Add -malign-double to IA32 compiler flags
  ea4911c2f3 load_cert_file: Use EFI RT memory function
  2d4ebb5a79 load_cert_file: Fix stack issue
  5c537b3d0c shim: Flush the memory region from i-cache before execution
  14d6339829 Discard load-options that start with a NUL
  092c2b2bbe Reference MokListRT instead of MokList
  0eb07e11b2 Make SBAT variable payload introspectable
* Thu Nov 17 2022 jlee@suse.com
- Update shim.changes, added missed shim 15.6-rc1 and 15.6 changelog to
  the item in Update to 15.6. (bsc#1198458)
* Tue Nov 15 2022 jlee@suse.com
- Add shim-jscPED-127-upgrade-shim-in-SLE15-SP5.patch for backporting the following
  patches between 15.6 with aa1b289a1a (jsc#PED-127):
    aa1b289a1a16774afc3143b8948d97261f0872d0 mok: remove MokListTrusted from PCR 7
    0cf43ac6d78c6f47f8b91210639ac1aa63665f0b Add -malign-double to IA32 compiler flags
    ea4911c2f3ce8f8f703a1476febac86bb16b00fd load_cert_file: Use EFI RT memory function
    2d4ebb5a798aafd3b06d2c3cb9c9840c1caa41ef load_cert_file: Fix stack issue
    5c537b3d0cf8c393dad2e61d49aade68f3af1401 shim: Flush the memory region from i-cache before execution
    14d63398298c8de23036a4cf61594108b7345863 Discard load-options that start with a NUL
    092c2b2bbed950727e41cf450b61c794881c33e7 Reference MokListRT instead of MokList
    0eb07e11b20680200d3ce9c5bc59299121a75388 Make SBAT variable payload introspectable
* Tue Nov 15 2022 jlee@suse.com
- Add shim-Enable-TDX-measurement-to-RTMR-register.patch to support
  enhance shim measurement to TD RTMR. (jsc#PED-1273)
* Tue Nov 15 2022 jlee@suse.com
- For pushing openSUSE:Factory/shim to SLE15-SP5, sync the shim.spec
  and shim.changes: (jsc#PED-127)
  - Add some change log from SLE shim.changes to Factory shim.changes
    Those messages are added "(sync shim.changes from SLE)" tag.
  - Add the following changes to shim.spec
  - only apply Patch100, the shim-bsc1198101-opensuse-cert-prompt.patch
    on openSUSE.
  - Enable the AArch64 signature check for SLE:
  [#] AArch64 signature
  signature=%%{SOURCE13}
* Thu Sep 29 2022 mchang@suse.com
- shim-install: ensure grub.cfg created is not overwritten after
  installing grub related files
* Mon Sep 12 2022 khanich.opensource@gmx.de
- Add logic to shim.spec to only set sbat policy when efivarfs is writeable.
  (bsc#1201066)
* Fri Aug  5 2022 jlee@suse.com
- Add logic to shim.spec for detecting --set-sbat-policy option before
  using mokutil to set sbat policy. (bsc#1202120)
* Fri Jul 29 2022 jlee@suse.com
- Change the URL in SBAT section to mail:security@suse.de. (bsc#1193282)
* Mon Jul 25 2022 jlee@suse.com
- Revoked the change in shim.spec for "use common SBAT values (boo#1193282)"
  - we need to build openSUSE Tumbleweed's shim on Leap 15.4 because Factory
    is unstable for building out a stable shim binary for signing. (bsc#1198458)
  - But the rpm-config-suse package in Leap 15.4 is direct copied from SLE 15.4
    because closing-the-leap-gap. So sbat_distro_* variables are SLE version,
    not for openSUSE. (bsc#1198458)
* Tue Jun 28 2022 jlee@suse.com
- Update to 15.6 (bsc#1198458)
  - shim-15.6.tar.bz2 is downloaded from bsc#1198458#c76
    which is from upstream grub2.cve_2021_3695.ms keybase channel.
  - For building 15.6~rc1 aarch64 image (d6eb9c6 Modernize aarch64), objcopy needs to
    support efi-app-aarch64 target. So we need the following patches in bintuils:
  - binutils-AArch64-Add-support-for-AArch64-EFI-efi-aarch64.patch
    b69c9d41e8 AArch64: Add support for AArch64 EFI (efi-*-aarch64).
  - binutils-Re-AArch64-Add-support-for-AArch64-EFI-efi-aarch64.patch
    32384aa396 Re: AArch64: Add support for AArch64 EFI (efi-*-aarch64)
  - binutils-Re-Add-support-for-AArch64-EFI-efi-aarch64.patch
    d91c67e873 Re: Add support for AArch64 EFI (efi-*-aarch64)
  - Patches (git log --oneline --reverse 15.5~..77144e5a4)
    448f096 MokManager: removed Locate graphic output protocol fail error message (bsc#1193315, bsc#1198458)
    a2da05f shim: implement SBAT verification for the shim_lock protocol
    bda03b8 post-process-pe: Fix a missing return code check
    af18810 CI: don't cancel testing when one fails
    ba580f9 CI: remove EOL Fedoras from github actions
    bfeb4b3 Remove aarch64 build tests before f35
    38cc646 CI: Add f36 and centos9 CI build tests.
    b5185cb post-process-pe: Fix format string warnings on 32-bit platforms
    31094e5 tests: also look for system headers in multi-arch directories
    4df989a mock-variables.c: fix gcc warning
    6aac595 test-str.c: fix gcc warnings with FORTIFY_SOURCE enabled
    2670c6a Allow MokListTrusted to be enabled by default
    5c44aaf Add code of conduct
    d6eb9c6 Modernize aarch64
    9af50c1 Use ASCII as fallback if Unicode Box Drawing characters fail
    de87985 make: don't treat cert.S specially
    803dc5c shim: use SHIM_DEVEL_VERBOSE when built in devel mode
    6402f1f SBAT matching: Break out of the inner sbat loop if we find the entry.
    bb4b60e Add verify_image
    acfd48f Abstract out image reading
    35d7378 Load additional certs from a signed binary
    8ce2832 post-process-pe: there is no 's' argument.
    465663e Add some missing PE image flag definitions
    226fee2 PE Loader: support and require NX
    df96f48 Add MokPolicy variable and MOK_POLICY_REQUIRE_NX
    b104fc4 post-process-pe: set EFI_IMAGE_DLLCHARACTERISTICS_NX_COMPAT
    f81a7cc SBAT revocation management
    abe41ab make: unbreak scan-build again for gnu-efi
    610a1ac sbat.h: minor reformatting for legibility
    f28833f peimage.h: make our signature macros force the type
    5d789ca Always initialize data/datasize before calling read_image()
    a50d364 sbat policy: make our policy change actions symbolic
    5868789 load_certs: trust dir->Read() slightly less.
    a78673b mok.c: fix a trivial dead assignment
    759f061 Fix preserve_sbat_uefi_variable() logic
    aa61fdf Give the Coverity scanner some more GCC blinders...
    0214cd9 load_cert_file(): don't defererence NULL
    1eca363 mok import: handle OOM case
    75449bc sbat: Make nth_sbat_field() honor the size limit
    c0bcd04 shim-15.6~rc1
    77144e5 SBAT Policy latest should be a one-shot
  - 15.5 release note https://github.com/rhboot/shim/releases
  Broken ia32 relocs and an unimportant submodule change. by @vathpela in #357
  mok: allocate MOK config table as BootServicesData by @lcp in #361
  Don't call QueryVariableInfo() on EFI 1.10 machines by @vathpela in #364
  Relax the check for import_mok_state() by @lcp in #372
  SBAT.md: trivial changes by @hallyn in #389
  shim: another attempt to fix load options handling by @chrisccoulson in #379
  Add tests for our load options parsing. by @vathpela in #390
  arm/aa64: fix the size of .rela* sections by @lcp in #383
  mok: fix potential buffer overrun in import_mok_state by @jyong2 in #365
  mok: relax the maximum variable size check by @lcp in #369
  Don't unhook ExitBootServices when EBS protection is disabled by @sforshee in #378
  fallback: find_boot_option() needs to return the index for the boot entry in optnum by @jsetje in #396
  httpboot: Ignore case when checking HTTP headers by @frozencemetery in #403
  Fallback allocation errors by @vathpela in #402
  shim: avoid BOOTx64.EFI in message on other architectures by @xypron in #406
  str: remove duplicate parameter check by @xypron in #408
  fallback: add compile option FALLBACK_NONINTERACTIVE by @xnox in #359
  Test mok mirror by @vathpela in #394
  Modify sbat.md to help with readability. by @eshiman in #398
  csv: detect end of csv file correctly by @xypron in #404
  Specify that the .sbat section is ASCII not UTF-8 by @daxtens in #413
  tests: add "include-fixed" GCC directory to include directories by @diabonas in #415
  pe: simplify generate_hash() by @xypron in #411
  Don't make shim abort when TPM log event fails (RHBZ #2002265) by @rmetrich in #414
  Fallback to default loader if parsed one does not exist by @julian-klode in #393
  fallback: Fix for BootOrder crash when index returned by find_boot_option() is not in current BootOrder list by @rmetrich in #422
  Better console checks by @vathpela in #416
  docs: update SBAT UEFI variable name by @nicholasbishop in #421
  Don't parse load options if invoked from removable media path by @julian-klode in #399
  fallback: fix fallback not passing arguments of the first boot option by @martinezjavier in #433
  shim: Don't stop forever at "Secure Boot not enabled" notification by @rmetrich in #438
  Shim 15.5 coverity by @vathpela in #439
  Allocate mokvar table in runtime memory. by @vathpela in #447
  Remove post-process-pe on 'make clean' by @vathpela in #448
  pe: missing perror argument by @xypron in #443
  - 15.6-rc1 release note https://github.com/rhboot/shim/releases
  MokManager: removed Locate graphic output protocol fail error message by @joeyli in #441
  shim: implement SBAT verification for the shim_lock protocol by @chrisccoulson in #456
  post-process-pe: Fix a missing return code check by @vathpela in #462
  Update github actions matrix to be more useful by @frozencemetery in #469
  Add f36 and centos9 CI builds by @vathpela in #470
  post-process-pe: Fix format string warnings on 32-bit platforms by @steve-mcintyre in #464
  tests: also look for system headers in multi-arch directories by @steve-mcintyre in #466
  tests: fix gcc warnings by @akodanev in #463
  Allow MokListTrusted to be enabled by default by @esnowberg in #455
  Add code of conduct by @frozencemetery in #427
  Re-add ARM AArch64 support by @vathpela in #468
  Use ASCII as fallback if Unicode Box Drawing characters fail by @vathpela in #428
  make: don't treat cert.S specially by @vathpela in #475
  shim: use SHIM_DEVEL_VERBOSE when built in devel mode by @vathpela in #474
  Break out of the inner sbat loop if we find the entry. by @vathpela in #476
  Support loading additional certificates by @esnowberg in #446
  Add support for NX (W^X) mitigations. by @vathpela in #459
  Misc fixups from scan-build. by @vathpela in #477
  Fix preserve_sbat_uefi_variable() logic by @jsetje in #478
  - 15.6 release note https://github.com/rhboot/shim/releases
  MokManager: removed Locate graphic output protocol fail error message by @joeyli in #441
  shim: implement SBAT verification for the shim_lock protocol by @chrisccoulson in #456
  post-process-pe: Fix a missing return code check by @vathpela in #462
  Update github actions matrix to be more useful by @frozencemetery in #469
  Add f36 and centos9 CI builds by @vathpela in #470
  post-process-pe: Fix format string warnings on 32-bit platforms by @steve-mcintyre in #464
  tests: also look for system headers in multi-arch directories by @steve-mcintyre in #466
  tests: fix gcc warnings by @akodanev in #463
  Allow MokListTrusted to be enabled by default by @esnowberg in #455
  Add code of conduct by @frozencemetery in #427
  Re-add ARM AArch64 support by @vathpela in #468
  Use ASCII as fallback if Unicode Box Drawing characters fail by @vathpela in #428
  make: don't treat cert.S specially by @vathpela in #475
  shim: use SHIM_DEVEL_VERBOSE when built in devel mode by @vathpela in #474
  Break out of the inner sbat loop if we find the entry. by @vathpela in #476
  Support loading additional certificates by @esnowberg in #446
  Add support for NX (W^X) mitigations. by @vathpela in #459
  Misc fixups from scan-build. by @vathpela in #477
  Fix preserve_sbat_uefi_variable() logic by @jsetje in #478
  SBAT Policy latest should be a one-shot by @jsetje in #481
  pe: Fix a buffer overflow when SizeOfRawData > VirtualSize by @chriscoulson
  pe: Perform image verification earlier when loading grub by @chriscoulson
  Update advertised sbat generation number for shim by @jsetje
  Update SBAT generation requirements for 05/24/22 by @jsetje
  Also avoid CVE-2022-28737 in verify_image() by @vathpela
  - Drop upstreamed patch:
  - shim-bsc1184454-allocate-mok-config-table-BS.patch
  - Allocate MOK config table as BootServicesData to avoid the error message
  from linux kernel
  - 4068fd42c8		15.5-rc1~70
  - shim-bsc1185441-fix-handling-of-ignore_db-and-user_insecure_mode.patch
  - Handle ignore_db and user_insecure_mode correctly
  - 822d07ad4f07		15.5-rc1~73
  - shim-bsc1185621-relax-max-var-sz-check.patch
  - Relax the maximum variable size check for u-boot
  - 3f327f546c219634b2	15.5-rc1~49
  - shim-bsc1185261-relax-import_mok_state-check.patch
  - Relax the check for import_mok_state() when Secure Boot is off
  - 9f973e4e95b113	15.5-rc1~67
  - shim-bsc1185232-relax-loadoptions-length-check.patch
  - Relax the check for the LoadOptions length
  - ada7ff69bd8a95	15.5-rc1~52
  - shim-fix-aa64-relsz.patch
  - Fix the size of rela* sections for AArch64
  - 34e3ef205c5d65	15.5-rc1~51
  - shim-bsc1187260-fix-efi-1.10-machines.patch
  - Don't call QueryVariableInfo() on EFI 1.10 machines
  - 493bd940e5		15.5-rc1~69
  - shim-bsc1185232-fix-config-table-copying.patch
  - Avoid buffer overflow when copying the MOK config table
  - 7501b6bb44		15.5-rc1~50
  - shim-bsc1187696-avoid-deleting-rt-variables.patch
  - Avoid deleting the mirrored RT variables
  - b1fead0f7c9		15.5-rc1~37
  - Add "rm -f *.o" after building MokManager/fallback in shim.spec
    to make sure all object files gets rebuilt
  - reference: https://github.com/rhboot/shim/pull/461
- The following fix-CVE-2022-28737-v6 patches against bsc#1198458 are included
  in shim-15.6.tar.bz2
  - shim-bsc1198458-pe-Fix-a-buffer-overflow-when-SizeOfRawData-VirtualS.patch
    pe: Fix a buffer overflow when SizeOfRawData VirtualSize
  - shim-bsc1198458-pe-Perform-image-verification-earlier-when-loading-g.patch
    pe: Perform image verification earlier when loading grub
  - shim-bsc1198458-Update-advertised-sbat-generation-number-for-shim.patch
    Update advertised sbat generation number for shim
  - shim-bsc1198458-Update-SBAT-generation-requirements-for-05-24-22.patch
    Update SBAT generation requirements for 05/24/22
  - shim-bsc1198458-Also-avoid-CVE-2022-28737-in-verify_image.patch
    Also avoid CVE-2022-28737 in verify_image()
  - 0006-shim-15.6-rc2.patch
  - 0007-sbat-add-the-parsed-SBAT-variable-entries-to-the-deb.patch
    sbat: add the parsed SBAT variable entries to the debug log
  - 0008-bump-version-to-shim-15.6.patch
- Add mokutil command to post script for setting sbat policy to latest mode
  when the SbatPolicy-605dab50-e046-4300-abb6-3dd810dd8b23 is not created.
  (bsc#1198458)
- Add shim-bsc1198101-opensuse-cert-prompt.patch back to openSUSE shim to
  show the prompt to ask whether the user trusts openSUSE certificate or not
  (bsc#1198101)
- Updated vendor dbx binary and script (bsc#1198458)
  - Updated dbx-cert.tar.xz and vendor-dbx-sles.bin for adding
    SLES-UEFI-SIGN-Certificate-2021-05.crt to vendor dbx list.
  - Updated dbx-cert.tar.xz and vendor-dbx-opensuse.bin for adding
    openSUSE-UEFI-SIGN-Certificate-2021-05.crt to vendor dbx list.
  - Updated vendor-dbx.bin for adding SLES-UEFI-SIGN-Certificate-2021-05.crt
    and openSUSE-UEFI-SIGN-Certificate-2021-05.crt for testing environment.
  - Updated generate-vendor-dbx.sh script for generating a vendor-dbx.bin
    file which includes all .der for testing environment.
* Tue Apr 12 2022 lnussel@suse.de
- use common SBAT values (boo#1193282)
* Thu Jul 15 2021 jsegitz@suse.com
- Update the SLE signatures (sync shim.changes from SLE)
* Thu Jul  1 2021 glin@suse.com
- Add shim-bsc1187696-avoid-deleting-rt-variables.patch to avoid
  deleting the mirrored RT variables (bsc#1187696)
* Mon Jun 21 2021 glin@suse.com
  (sync shim.changes from SLE)
- Split the keys in vendor-dbx.bin to vendor-dbx-sles and
  vendor-dbx-opensuse for shim-sles and shim-opensuse to reduce
  the size of MokListXRT (bsc#1185261)
  + Also update generate-vendor-dbx.sh in dbx-cert.tar.xz
- Add shim-bsc1185441-fix-handling-of-ignore_db-and-user_insecure_mode.patch
  to handle ignore_db and user_insecure_mode correctly
  (bsc#1185441, bsc#1187071)
- Add shim-bsc1185621-relax-max-var-sz-check.patch to relax the
  maximum variable size check for u-boot (bsc#1185621)
  + Also drop AArch64 suse-signed shim since we merged this patch
- Add shim-bsc1185261-relax-import_mok_state-check.patch to relax
  the check for import_mok_state() when Secure Boot is off.
  (bsc#1185261)
- Add shim-bsc1185232-relax-loadoptions-length-check.patch to
  ignore the odd LoadOptions length (bsc#1185232)
- shim-install: reset def_shim_efi to "shim.efi" if the given
  file doesn't exist
- Add shim-fix-aa64-relsz.patch to fix the size of rela sections
  for AArch64
  Fix: https://github.com/rhboot/shim/issues/371
- Add shim-disable-export-vendor-dbx.patch to disable exporting
  vendor-dbx to MokListXRT since writing a large RT variable
  could crash some machines (bsc#1185261)
- Add shim-bsc1187260-fix-efi-1.10-machines.patch to avoid the
  potential crash when calling QueryVariableInfo in EFI 1.10
  machines (bsc#1187260)
- Add shim-bsc1185232-fix-config-table-copying.patch to avoid
  buffer overflow when copying data to the MOK config table
  (bsc#1185232)
* Mon Jun 21 2021 glin@suse.com
- Add shim-bsc1185232-fix-config-table-copying.patch to avoid
  buffer overflow when copying data to the MOK config table
  (bsc#1185232)
* Mon Jun 21 2021 glin@suse.com
- Add shim-disable-export-vendor-dbx.patch to disable exporting
  vendor-dbx to MokListXRT since writing a large RT variable
  could crash some machines (bsc#1185261)
- Add shim-bsc1187260-fix-efi-1.10-machines.patch to avoid the
  potential crash when calling QueryVariableInfo in EFI 1.10
  machines (bsc#1187260)
* Thu Jun 17 2021 glin@suse.com
- Add shim-fix-aa64-relsz.patch to fix the size of rela sections
  for AArch64
  Fix: https://github.com/rhboot/shim/issues/371
* Fri Jun  4 2021 glin@suse.com
- Add shim-bsc1185232-relax-loadoptions-length-check.patch to
  ignore the odd LoadOptions length (bsc#1185232)
* Fri Jun  4 2021 glin@suse.com
- shim-install: reset def_shim_efi to "shim.efi" if the given
  file doesn't exist
* Wed May 19 2021 glin@suse.com
- shim-install: instead of assuming "removable" for Azure, remove
  fallback.efi from \EFI\Boot and copy grub.efi/cfg to \EFI\Boot
  to make \EFI\Boot bootable and keep the boot option created by
  efibootmgr (bsc#1185464, bsc#1185961)
* Tue May 11 2021 glin@suse.com
- Add shim-bsc1185261-relax-import_mok_state-check.patch to relax
  the check for import_mok_state() when Secure Boot is off.
  (bsc#1185261)
* Fri May  7 2021 glin@suse.com
- shim-install: always assume "removable" for Azure to avoid the
  endless reset loop (bsc#1185464)
* Thu May  6 2021 glin@suse.com
- Include suse-signed shim for AArch64 (bsc#1185621)
  (sync shim.changes from SLE)
* Thu May  6 2021 glin@suse.com
- Add shim-bsc1185621-relax-max-var-sz-check.patch to relax the
  maximum variable size check for u-boot (bsc#1185621)
* Mon May  3 2021 glin@suse.com
- Add shim-bsc1185441-fix-handling-of-ignore_db-and-user_insecure_mode.patch
  to handle ignore_db and user_insecure_mode correctly
  (bsc#1185441, bsc#1187071)
* Wed Apr 28 2021 glin@suse.com
- Split the keys in vendor-dbx.bin to vendor-dbx-sles and
  vendor-dbx-opensuse for shim-sles and shim-opensuse to reduce
  the size of MokListXRT (bsc#1185261)
  + Also update generate-vendor-dbx.sh in dbx-cert.tar.xz
* Thu Apr 22 2021 glin@suse.com
- Enable the AArch64 signature check for SLE (sync shim.changes from SLE)
* Wed Apr 21 2021 jsegitz@suse.com
- Update the SLE signatures (sync shim.changes from SLE)
* Thu Apr  8 2021 glin@suse.com
- Add shim-bsc1184454-allocate-mok-config-table-BS.patch to avoid
  the error message during linux system boot (bsc#1184454)
* Wed Apr  7 2021 jsegitz@suse.com
- Add remove_build_id.patch to prevent the build id being added to
  the binary. That can cause issues with the signature
* Wed Mar 31 2021 glin@suse.com
- Update to 15.4 (bsc#1182057)
  + Rename the SBAT variable and fix the self-check of SBAT
  + sbat: add more dprint()
  + arm/aa64: Swizzle some sections to make old sbsign happier
  + arm/aa64 targets: put .rel* and .dyn* in .rodata
- Drop upstreamed patch:
  + shim-bsc1182057-sbat-variable-enhancement.patch
* Mon Mar 29 2021 glin@suse.com
- Add shim-bsc1182057-sbat-variable-enhancement.patch to change
  the SBAT variable name and enhance the handling of SBAT
  (bsc#1182057)
* Wed Mar 24 2021 glin@suse.com
- Update to 15.3 for SBAT support (bsc#1182057)
  + Drop gnu-efi from BuildRequires since upstream pull it into the
    tar ball.
- Generate vender-specific SBAT metadata
  + Add dos2unix to BuildRequires since Makefile requires it for
    vendor SBAT
- Update dbx-cert.tar.xz and vendor-dbx.bin to block the following
  sign keys:
  + SLES-UEFI-SIGN-Certificate-2020-07.crt
  + openSUSE-UEFI-SIGN-Certificate-2020-07.crt
- Refresh patches
  + shim-arch-independent-names.patch
  + shim-change-debug-file-path.patch
  + shim-bsc1177315-verify-eku-codesign.patch
  - Unified with shim-bsc1177315-fix-buffer-use-after-free.patch
- Drop upstreamed fixes
  + shim-correct-license-in-headers.patch
  + shim-always-mirror-mok-variables.patch
  + shim-bsc1175509-more-tpm-fixes.patch
  + shim-bsc1173411-only-check-efi-var-on-sb.patch
  + shim-fix-verify-eku.patch
  + gcc9-fix-warnings.patch
  + shim-fix-gnu-efi-3.0.11.patch
  + shim-bsc1177404-fix-a-use-of-strlen.patch
  + shim-do-not-write-string-literals.patch
  + shim-VLogError-Avoid-Null-pointer-dereferences.patch
  + shim-bsc1092000-fallback-menu.patch
  + shim-bsc1175509-tpm2-fixes.patch
  + shim-bsc1174512-correct-license-in-headers.patch
  + shim-bsc1182776-fix-crash-at-exit.patch
- Drop shim-opensuse-cert-prompt.patch
  + All newly released openSUSE kernels enable kernel lockdown
    and signature verification, so there is no need to add the
    prompt anymore.
* Thu Mar 11 2021 glin@suse.com
- Refresh shim-bsc1182776-fix-crash-at-exit.patch to do the cleanup
  also when Secure Boot is disabled (bsc#1183213, bsc#1182776)
- Merged linker-version.pl into timestamp.pl and add the linker
  version to signature files accordingly
* Mon Mar  8 2021 glin@suse.com
- Add shim-bsc1182776-fix-crash-at-exit.patch to fix the potential
  crash at Exit() (bsc#1182776)
* Fri Jan 22 2021 glin@suse.com
- Update the SLE signature
- Exclude some patches from x86_64 to avoid breaking the signature
- Add shim-correct-license-in-headers.patch back for x86_64 to
  match the SLE signature
- Add linker-version.pl to modify the EFI/PE header to match the
  SLE signature
* Wed Nov  4 2020 glin@suse.com
- Disable the signature attachment for AArch64 temporarily until
  we get a real one.
* Mon Nov  2 2020 glin@suse.com
- Add shim-bsc1177315-verify-eku-codesign.patch to check CodeSign
  in the signer's EKU (bsc#1177315)
- Add shim-bsc1177789-fix-null-pointer-deref-AuthenticodeVerify.patch
  to fix NULL pointer dereference in AuthenticodeVerify()
  (bsc#1177789, CVE-2019-14584)
- shim-install: Support changing default shim efi binary in
  /usr/etc/default/shim and /etc/default/shim (bsc#1177315)
- Add shim-bsc1177315-fix-buffer-use-after-free.patch to fix buffer
  use-after-free at the end of the EKU verification (bsc#1177315)
* Wed Oct 14 2020 glin@suse.com
- Add shim-bsc1177404-fix-a-use-of-strlen.patch to fix the length
  of the option data string to launch the program correctly
  (bsc#1177404)
- Add shim-bsc1175509-more-tpm-fixes.patch to fix the file path
  in the tpm even log (bsc#1175509)
* Mon Sep 14 2020 glin@suse.com
- Add shim-VLogError-Avoid-Null-pointer-dereferences.patch to fix
  VLogError crash in AArch64 (jsc#SLE-15824)
- Add shim-fix-verify-eku.patch to fix the potential crash at
  verify_eku() (jsc#SLE-15824)
- Add shim-do-not-write-string-literals.patch to fix the potential
  crash when accessing the DEFAULT_LOADER string (jsc#SLE-15824)
* Fri Sep  4 2020 guillaume.gardet@opensuse.org
- Enable build on aarch64
* Mon Aug 24 2020 glin@suse.com
- shim-install: install MokManager to \EFI\boot to process the
  pending MOK request (bsc#1175626, bsc#1175656)
* Fri Aug 21 2020 glin@suse.com
- Add shim-bsc1175509-tpm2-fixes.patch to fix the TPM2 measurement
  (bsc#1175509)
* Thu Aug  6 2020 glin@suse.com
- Amend the check of %%shim_enforce_ms_signature
* Fri Jul 31 2020 jsegitz@suse.com
- Updated openSUSE signature
* Mon Jul 27 2020 glin@suse.com
- Replace shim-correct-license-in-headers.patch with the upstream
  commit: shim-bsc1174512-correct-license-in-headers.patch
  (bsc#1174512)
* Wed Jul 22 2020 glin@suse.com
- Update the path to grub-tpm.efi in shim-install (bsc#1174320)
* Fri Jul 10 2020 glin@suse.com
- Use vendor-dbx to block old SUSE/openSUSE signkeys (bsc#1168994)
  + Add dbx-cert.tar.xz which contains the certificates to block
    and a script, generate-vendor-dbx.sh, to generate
    vendor-dbx.bin
  + Add vendor-dbx.bin as the vendor dbx to block unwanted keys
- Drop shim-opensuse-signed.efi
  + We don't need it anymore
* Fri Jul 10 2020 glin@suse.com
- Add shim-bsc1173411-only-check-efi-var-on-sb.patch to only check
  EFI variable copying when Secure Boot is enabled (bsc#1173411)
* Tue Mar 31 2020 glin@suse.com
- Use the full path of efibootmgr to avoid errors when invoking
  shim-install from packagekitd (bsc#1168104)
* Mon Mar 30 2020 glin@suse.com
- Use "suse_version" instead of "sle_version" to avoid
  shim_lib64_share_compat being set in Tumbleweed forever.
* Mon Mar 16 2020 glin@suse.com
- Add shim-fix-gnu-efi-3.0.11.patch to fix the build error caused
  by the upgrade of gnu-efi
* Wed Nov 27 2019 mchang@suse.com
- shim-install: add check for btrfs is used as root file system to enable
  relative path lookup for file. (bsc#1153953)
* Fri Aug 16 2019 glin@suse.com
- Fix a typo in shim-install (bsc#1145802)
* Fri Apr 19 2019 mliska@suse.cz
- Add gcc9-fix-warnings.patch (bsc#1121268).
* Mon Apr 15 2019 glin@suse.com
- Add shim-opensuse-signed.efi, the openSUSE shim-15+git47 binary
  (bsc#1113225)
* Fri Apr 12 2019 glin@suse.com
- Disable AArch64 build (FATE#325971)
  + AArch64 machines don't use UEFI CA, at least for now.
* Thu Apr 11 2019 jsegitz@suse.com
- Updated shim signature: signature-sles.x86_64.asc (bsc#1120026)
* Thu Feb 14 2019 rw@suse.com
- Fix conditions for '/usr/share/efi'-move  (FATE#326960)
* Mon Jan 28 2019 glin@suse.com
- Amend shim.spec to remove $RPM_BUILD_ROOT
* Thu Jan 17 2019 rw@suse.com
- Move 'efi'-executables to '/usr/share/efi'  (FATE#326960)
  (preparing the move to 'noarch' for this package)
* Mon Jan 14 2019 glin@suse.com
- Update shim-install to handle the partitioned MD devices
  (bsc#1119762, bsc#1119763)
* Thu Dec 20 2018 glin@suse.com
- Update to 15+git47 (bsc#1120026, FATE#325971)
  + git commit: b3e4d1f7555aabbf5d54de5ea7cd7e839e7bd83d
- Retire the old openSUSE 4096 bit certificate
  + Those programs are already out of maintenance.
- Add shim-always-mirror-mok-variables.patch to mirror MOK
  variables correctly
- Add shim-correct-license-in-headers.patch to correct the license
  declaration
- Refresh patches:
  + shim-arch-independent-names.patch
  + shim-change-debug-file-path.patch
  + shim-bsc1092000-fallback-menu.patch
  + shim-opensuse-cert-prompt.patch
- Drop upstreamed patches:
  + shim-bsc1088585-handle-mok-allocations-better.patch
  + shim-httpboot-amend-device-path.patch
  + shim-httpboot-include-console.h.patch
  + shim-only-os-name.patch
  + shim-remove-cryptpem.patch
* Wed Dec  5 2018 glin@suse.com
- Update shim-install to specify the target for grub2-install and
  change the boot efi file name according to the architecture
  (bsc#1118363, FATE#325971)
* Tue Aug 21 2018 glin@suse.com
- Enable AArch64 build (FATE#325971)
  + Also add the aarch64 signature files and rename the x86_64
    signature files
* Tue May 29 2018 glin@suse.com
- Add shim-bsc1092000-fallback-menu.patch to show a menu before
  system reset ((bsc#1092000))
* Tue Apr 10 2018 glin@suse.com
- Add shim-bsc1088585-handle-mok-allocations-better.patch to avoid
  double-freeing after enrolling a key from the disk (bsc#1088585)
  + Also refresh shim-opensuse-cert-prompt.patch due to the change
    in MokManager.c
* Tue Apr  3 2018 glin@suse.com
- Install the certificates with a shim suffix to avoid conflicting
  with other packages (bsc#1087847)
* Fri Mar 23 2018 glin@suse.com
- Add the missing leading backlash to the DEFAULT_LOADER
  (bsc#1086589)
* Fri Jan  5 2018 glin@suse.com
- Add shim-httpboot-amend-device-path.patch to amend the device
  path matching rule for httpboot (bsc#1065370)
* Thu Jan  4 2018 glin@suse.com
- Update to 14 (bsc#1054712)
- Adjust make commands in spec
- Drop upstreamed fixes
  + shim-add-fallback-verbose-print.patch
  + shim-back-to-openssl-1.0.2e.patch
  + shim-fallback-workaround-masked-ami-variables.patch
  + shim-fix-fallback-double-free.patch
  + shim-fix-httpboot-crash.patch
  + shim-fix-openssl-flags.patch
  + shim-more-tpm-measurement.patch
- Add shim-httpboot-include-console.h.patch to include console.h
  in httpboot.c to avoid build failure
- Add shim-remove-cryptpem.patch to replace functions in CryptPem.c
  with the null function
- Update SUSE/openSUSE specific patches
  + shim-only-os-name.patch
  + shim-arch-independent-names.patch
  + shim-change-debug-file-path.patch
  + shim-opensuse-cert-prompt.patch
* Fri Dec 29 2017 ngompa13@gmail.com
- Fix debuginfo + debugsource subpackage generation for RPM 4.14
- Set the RPM groups correctly for debug{info,source} subpackages
- Drop deprecated and out of date Authors information in description
* Wed Sep 13 2017 glin@suse.com
- Add shim-back-to-openssl-1.0.2e.patch to avoid rejecting some
  legit certificates (bsc#1054712)
- Add the stderr mask back while compiling MokManager.efi since the
  warnings in Cryptlib is back after reverting the openssl commits.
* Tue Aug 29 2017 glin@suse.com
- Add shim-add-fallback-verbose-print.patch to print the debug
  messages in fallback.efi dynamically
- Refresh shim-fallback-workaround-masked-ami-variables.patch
- Add shim-more-tpm-measurement.patch to measure more components
  and support TPM better
* Wed Aug 23 2017 glin@suse.com
- Add upstream fixes
  + shim-fix-httpboot-crash.patch
  + shim-fix-openssl-flags.patch
  + shim-fix-fallback-double-free.patch
  + shim-fallback-workaround-masked-ami-variables.patch
- Remove the stderr mask while compiling MokManager.efi since the
  warnings in Cryptlib were fixed.
* Tue Aug 22 2017 glin@suse.com
- Add shim-arch-independent-names.patch to use the Arch-independent
  names. (bsc#1054712)
- Refresh shim-change-debug-file-path.patch
- Disable shim-opensuse-cert-prompt.patch automatically in SLE
- Diable AArch64 until we have a real user and aarch64 signature
* Fri Jul 14 2017 bwiedemann@suse.com
- Make build reproducible by avoiding race between find and cp
* Thu Jun 22 2017 glin@suse.com
- Update to 12
- Rename the result EFI images due to the upstream name change
  + shimx64 -> shim
  + mmx64 -> MokManager
  + fbx64 -> fallback
- Refresh patches:
  + shim-only-os-name.patch
  + shim-change-debug-file-path.patch
  + shim-opensuse-cert-prompt.patch
- Drop upstreamed patches:
  + shim-httpboot-support.patch
  + shim-bsc973496-mokmanager-no-append-write.patch
  + shim-bsc991885-fix-sig-length.patch
  + shim-update-openssl-1.0.2g.patch
  + shim-update-openssl-1.0.2h.patch
* Tue May 23 2017 glin@suse.com
- Add the build flag to enable HTTPBoot
* Wed Mar 22 2017 mchang@suse.com
- shim-install: add option --suse-enable-tpm (fate#315831)
* Fri Jan 13 2017 mchang@suse.com
- Support %%posttrans with marcos provided by update-bootloader-rpm-macros
  package (bsc#997317)
* Fri Nov 18 2016 glin@suse.com
- Add SIGNATURE_UPDATE.txt to state the steps to update
  signature-*.asc
- Update the comment of strip_signature.sh
* Wed Sep 21 2016 mchang@suse.com
- shim-install :
  * add option --no-nvram (bsc#999818)
  * improve removable media and fallback mode handling
* Fri Aug 19 2016 mchang@suse.com
- shim-install : fix regression of password prompt (bsc#993764)
* Fri Aug  5 2016 glin@suse.com
- Add shim-bsc991885-fix-sig-length.patch to fix the signature
  length passed to Authenticode (bsc#991885)
* Wed Aug  3 2016 glin@suse.com
- Update shim-bsc973496-mokmanager-no-append-write.patch to try
  append write first
* Tue Aug  2 2016 glin@suse.com
- Add shim-update-openssl-1.0.2h.patch to update openssl to 1.0.2h
- Bump the requirement of gnu-efi due to the HTTPBoot support
* Mon Aug  1 2016 glin@suse.com
- Add shim-httpboot-support.patch to support HTTPBoot
- Add shim-update-openssl-1.0.2g.patch to update openssl to 1.0.2g
  and Cryptlib to 5e2318dd37a51948aaf845c7d920b11f47cdcfe6
- Drop patches since they are merged into
  shim-update-openssl-1.0.2g.patch
  + shim-update-openssl-1.0.2d.patch
  + shim-gcc5.patch
  + shim-bsc950569-fix-cryptlib-va-functions.patch
  + shim-fix-aarch64.patch
- Refresh shim-change-debug-file-path.patch
- Add shim-bsc973496-mokmanager-no-append-write.patch to work
  around the firmware that doesn't support APPEND_WRITE (bsc973496)
- shim-install : remove '\n' from the help message (bsc#991188)
- shim-install : print a message if there is no valid EFI partition
  (bsc#991187)
* Mon May  9 2016 rw@suse.com
- shim-install : support simple MD RAID1 target devices (FATE#314829)
* Wed May  4 2016 agraf@suse.com
- Add shim-fix-aarch64.patch to fix compilation on AArch64 (bsc#978438)
* Wed Mar  9 2016 mchang@suse.com
- shim-install : fix typing ESC can escape to parent config which is
  in command mode and cannot return back (bsc#966701)
- shim-install : fix no which command for JeOS (bsc#968264)
* Thu Dec  3 2015 jsegitz@novell.com
- acquired updated signature from Microsoft
* Mon Nov  9 2015 glin@suse.com
- Add shim-bsc950569-fix-cryptlib-va-functions.patch to fix the
  definition of va functions to avoid the potential crash
  (bsc#950569)
- Update shim-opensuse-cert-prompt.patch to avoid setting NULL to
  MokListRT (bsc#950801)
- Drop shim-fix-mokmanager-sections.patch as we are using the
  newer binutils now
- Refresh shim-change-debug-file-path.patch
* Thu Oct  8 2015 jsegitz@novell.com
- acquired updated signature from Microsoft
* Tue Sep 15 2015 mchang@suse.com
- shim-install : set default GRUB_DISTRIBUTOR from /etc/os-release
  if it is empty or not set by user (bsc#942519)
* Thu Jul 16 2015 glin@suse.com
- Add shim-update-openssl-1.0.2d.patch to update openssl to 1.0.2d
- Refresh shim-gcc5.patch and add it back since we really need it
- Add shim-change-debug-file-path.patch to change the debug file
  path in shim.efi
  + also add the debuginfo and debugsource subpackages
- Drop shim-fix-gnu-efi-30w.patch which is not necessary anymore
* Mon Jul  6 2015 glin@suse.com
- Update to 0.9
- Refresh patches
  + shim-fix-gnu-efi-30w.patch
  + shim-fix-mokmanager-sections.patch
  + shim-opensuse-cert-prompt.patch
- Drop upstreamed patches
  + shim-bsc920515-fix-fallback-buffer-length.patch
  + shim-mokx-support.patch
  + shim-update-cryptlib.patch
- Drop shim-bsc919675-uninstall-shim-protocols.patch since
  upstream fixed the bug in another way.
- Drop shim-gcc5.patch which was fixed in another way
* Wed Apr  8 2015 glin@suse.com
- Fix tags in the spec file
* Tue Apr  7 2015 glin@suse.com
- Add shim-update-cryptlib.patch to update Cryptlib to r16559 and
  openssl to 0.9.8zf
- Add shim-bsc919675-uninstall-shim-protocols.patch to uninstall
  the shim protocols at Exit (bsc#919675)
- Add shim-bsc920515-fix-fallback-buffer-length.patch to adjust
  the buffer size for the boot options (bsc#920515)
- Refresh shim-opensuse-cert-prompt.patch
* Thu Apr  2 2015 crrodriguez@opensuse.org
- shim-gcc5.patch: shim needs -std=gnu89 to build with GCC5
* Tue Feb 17 2015 mchang@suse.com
- shim-install : fix cryptodisk installation (boo#917427)
* Tue Nov 11 2014 glin@suse.com
- Add shim-fix-mokmanager-sections.patch to fix the objcopy
  parameters for the EFI files
* Tue Oct 28 2014 glin@suse.com
- Update to 0.8
- Add shim-fix-gnu-efi-30w.patch to adapt the change in
  gnu-efi-3.0w
- Merge shim-signed-unsigned-compares.patch,
  shim-mokmanager-support-sha-family.patch and
  shim-bnc863205-mokmanager-fix-hash-delete.patch into
  shim-mokx-support.patch
- Refresh shim-opensuse-cert-prompt.patch
- Drop upstreamed patches: shim-update-openssl-0.9.8zb.patch,
  bug-889332_shim-overflow.patch, and bug-889332_shim-mok-oob.patch
- Enable aarch64
* Mon Oct 13 2014 jsegitz@novell.com
- Fixed buffer overflow and OOB access in shim trusted code path
  (bnc#889332, CVE-2014-3675, CVE-2014-3676, CVE-2014-3677)
  * added bug-889332_shim-mok-oob.patch, bug-889332_shim-overflow.patch
- Added new certificate by Microsoft
* Wed Sep  3 2014 lnussel@suse.de
- re-introduce build failure if shim_enforce_ms_signature is defined. That way
  a project like openSUSE:Factory can decide whether or not shim needs a valid
  MS signature.
* Tue Aug 19 2014 glin@suse.com
- Add shim-update-openssl-0.9.8zb.patch to update openssl to
  0.9.8zb
* Tue Aug 12 2014 jsegitz@suse.com
- updated shim to new version (OpenSSL 0.9.8za) and requested a new
  certificate from Microsoft. Removed
  * shim-allow-fallback-use-system-loadimage.patch
  * shim-bnc872503-check-key-encoding.patch
  * shim-bnc877003-fetch-from-the-same-device.patch
  * shim-correct-user_insecure-usage.patch
  * shim-fallback-avoid-duplicate-bootorder.patch
  * shim-fallback-improve-entries-creation.patch
  * shim-fix-dhcpv4-path-generation.patch
  * shim-fix-uninitialized-variable.patch
  * shim-fix-verify-mok.patch
  * shim-get-variable-check.patch
  * shim-improve-error-messages.patch
  * shim-mokmanager-delete-bs-var-right.patch
  * shim-mokmanager-handle-keystroke-error.patch
  * shim-remove-unused-variables.patch
  since they're included in upstream and rebased the remaining onces.
  Added shim-signed-unsigned-compares.patch to fix some compiler
  warnings
* Tue Aug 12 2014 glin@suse.com
- Keep shim-devel.efi for the devel project
* Fri Aug  8 2014 lnussel@suse.de
- don't fail the build if the UEFI signing service signature can't
  be attached anymore. This way shim can still pass through staging
  projects. We will verify the correct signature for release builds
  using openQA instead.
* Mon Aug  4 2014 mchang@suse.com
- shim-install: fix GRUB shows broken letters at boot by calling
  grub2-install to initialize /boot/grub2 directory with files
  needed by grub.cfg (bnc#889765)
* Wed May 28 2014 glin@suse.com
- Add shim-remove-unused-variables.patch to remove the unused
  variables
- Add shim-bnc872503-check-key-encoding.patch to check the encoding
  of the keys (bnc#872503)
- Add shim-bnc877003-fetch-from-the-same-device.patch to fetch the
  netboot image from the same device (bnc#877003)
- Refresh shim-opensuse-cert-prompt.patch
* Wed May 14 2014 glin@suse.com
- Use --reinit instead of --refresh in %%post to update the files
  in /boot
* Tue Apr 29 2014 mchang@suse.com
- shim-install: fix boot partition and rollback support kluge
  (bnc#875385)
* Thu Apr 10 2014 glin@suse.com
- Replace shim-mokmanager-support-sha1.patch with
  shim-mokmanager-support-sha-family.patch to support the SHA
  family
* Mon Apr  7 2014 glin@suse.com
- Add shim-mokmanager-support-sha1.patch to support SHA1 hashes in
  MOK
* Mon Mar 31 2014 mchang@suse.com
- snapper rollback support (fate#317062)
  - refresh shim-install
* Thu Mar 13 2014 glin@suse.com
- Insert the right signature (bnc#867974)
* Mon Mar 10 2014 glin@suse.com
- Add shim-fix-uninitialized-variable.patch to fix the use of
  uninitialzed variables in lib
* Fri Mar  7 2014 glin@suse.com
- Add shim-mokmanager-delete-bs-var-right.patch to delete the BS+NV
  variables the right way
- Update shim-opensuse-cert-prompt.patch to delete openSUSE_Verify
  correctly
* Thu Mar  6 2014 glin@suse.com
- Add shim-fallback-avoid-duplicate-bootorder.patch to fix the
  duplicate entries in BootOrder
- Add shim-allow-fallback-use-system-loadimage.patch to handle the
  shim protocol properly to keep only one protocol entity
- Refresh shim-opensuse-cert-prompt.patch
* Thu Mar  6 2014 mchang@suse.com
- shim-install: fix the $prefix to use grub2-mkrelpath for paths
  on btrfs subvolume (bnc#866690).
* Tue Mar  4 2014 glin@suse.com
- FATE#315002: Update shim-install to install shim.efi as the EFI
  default bootloader when none exists in \EFI\boot.
* Thu Feb 27 2014 fcrozat@suse.com
- Update signature-sles.asc: shim signed by UEFI signing service,
  based on code from "Thu Feb 20 11:57:01 UTC 2014"
* Fri Feb 21 2014 glin@suse.com
- Add shim-opensuse-cert-prompt.patch to show the prompt to ask
  whether the user trusts the openSUSE certificate or not
* Thu Feb 20 2014 lnussel@suse.de
- allow package to carry multiple signatures
- check correct certificate is embedded
* Thu Feb 20 2014 lnussel@suse.de
- always clean up generated files that embed certificates
  (shim_cert.h shim.cer shim.crt) to make sure next build loop
  rebuilds them properly
* Mon Feb 17 2014 glin@suse.com
- Add shim-bnc863205-mokmanager-fix-hash-delete.patch to fix the
  hash deletion operation to avoid ruining the whole list
  (bnc#863205)
* Tue Feb 11 2014 glin@suse.com
- Update shim-mokx-support.patch to support the resetting of MOK
  blacklist
- Add shim-get-variable-check.patch to fix the variable checking
  in get_variable_attr
- Add shim-fallback-improve-entries-creation.patch to improve the
  boot entry pathes and avoid generating the boot entries that
  are already there
- Update SUSE certificate
- Update attach_signature.sh, show_hash.sh, strip_signature.sh,
  extract_signature.sh and show_signatures.sh to remove the
  creation of the temporary nss database
- Add shim-only-os-name.patch: remove the kernel version of the
  build server
- Match the the prefix of the project name properly by escaping the
  percent sign.
* Wed Jan 22 2014 lnussel@suse.de
- enable signature assertion also in SUSE: hierarchy
* Fri Dec  6 2013 glin@suse.com
- Add shim-mokmanager-handle-keystroke-error.patch to handle the
  error status from ReadKeyStroke to avoid unexpected keys
* Thu Dec  5 2013 glin@suse.com
- Update to 0.7
- Add upstream patches:
  + shim-fix-verify-mok.patch
  + shim-improve-error-messages.patch
  + shim-correct-user_insecure-usage.patch
  + shim-fix-dhcpv4-path-generation.patch
- Add shim-mokx-support.patch to support the MOK blacklist
  (Fate#316531)
- Drop upstreamed patches
  + shim-fix-pointer-casting.patch
  + shim-merge-lf-loader-code.patch
  + shim-fix-simple-file-selector.patch
  + shim-mokmanager-support-crypt-hash-method.patch
  + shim-bnc804631-fix-broken-bootpath.patch
  + shim-bnc798043-no-doulbe-separators.patch
  + shim-bnc807760-change-pxe-2nd-loader-name.patch
  + shim-bnc808106-correct-certcount.patch
  + shim-mokmanager-ui-revamp.patch
  + shim-netboot-fixes.patch
  + shim-mokmanager-disable-gfx-console.patch
- Drop shim-suse-build.patch: it's not necessary anymore
- Drop shim-bnc841426-silence-shim-protocols.patch: shim is not
  verbose by default
* Thu Oct 31 2013 fcrozat@suse.com
- Update microsoft.asc: shim signed by UEFI signing service, based
  on code from "Tue Oct  1 04:29:29 UTC 2013".
* Tue Oct  1 2013 glin@suse.com
- Add shim-netboot-fixes.patch to include upstream netboot fixes
- Add shim-mokmanager-disable-gfx-console.patch to disable the
  graphics console to avoid system hang on some machines
- Add shim-bnc841426-silence-shim-protocols.patch to silence the
  shim protocols (bnc#841426)
* Wed Sep 25 2013 glin@suse.com
- Create boot.csv in ESP for fallback.efi to restore the boot entry
* Tue Sep 17 2013 fcrozat@suse.com
- Update microsoft.asc: shim signed by UEFI signing service, based
  on code from "Fri Sep  6 13:57:36 UTC 2013".
- Improve extract_signature.sh to work on current path.
* Fri Sep  6 2013 lnussel@suse.de
- set timestamp of PE file to time of the binary the signature was
  made for.
- make sure cert.o get's rebuilt for each target
* Fri Sep  6 2013 fcrozat@suse.com
- Update microsoft.asc: shim signed by UEFI signing service, based
  on code from "Wed Aug 28 15:54:38 UTC 2013"
* Wed Aug 28 2013 lnussel@suse.de
- always build a shim that embeds the distro's certificate (e.g.
  shim-opensuse.efi). If the package is built in the devel project
  additionally shim-devel.efi is created. That allows us to either
  load grub2/kernel signed by the distro or signed by the devel
  project, depending on use case. Also shim-$distro.efi from the
  devel project can be used to request additional signatures.
* Wed Aug 28 2013 lnussel@suse.de
- also include old openSUSE 4096 bit certificate to be able to still
  boot kernels signed with that key.
- add show_signatures script
* Tue Aug 27 2013 lnussel@suse.de
- replace the 4096 bit openSUSE UEFI CA certificate with new a
  standard compliant 2048 bit one.
* Tue Aug 20 2013 lnussel@suse.de
- fix shell syntax error
* Wed Aug  7 2013 lnussel@suse.de
- don't include binary in the sources. Instead package the raw
  signature and attach it during build (bnc#813448).
* Tue Jul 30 2013 glin@suse.com
- Update shim-mokmanager-ui-revamp.patch to include fixes for
  MokManager
  + reboot the system after clearing MOK password
  + fetch more info from X509 name
  + check the suffix of the key file
* Tue Jul 23 2013 glin@suse.com
- Update to 0.4
- Rebase patches
  + shim-suse-build.patch
  + shim-mokmanager-support-crypt-hash-method.patch
  + shim-bnc804631-fix-broken-bootpath.patch
  + shim-bnc798043-no-doulbe-separators.patch
  + shim-bnc807760-change-pxe-2nd-loader-name.patch
  + shim-bnc808106-correct-certcount.patch
  + shim-mokmanager-ui-revamp.patch
- Add patches
  + shim-merge-lf-loader-code.patch: merge the Linux Foundation
    loader UI code
  + shim-fix-pointer-casting.patch: fix a casting issue and the
    size of an empty vendor cert
  + shim-fix-simple-file-selector.patch: fix the buffer allocation
    in the simple file selector
- Remove upstreamed patches
  + shim-support-mok-delete.patch
  + shim-reboot-after-changes.patch
  + shim-clear-queued-key.patch
  + shim-local-key-sign-mokmanager.patch
  + shim-get-2nd-stage-loader.patch
  + shim-fix-loadoptions.patch
- Remove unused patch: shim-mokmanager-new-pw-hash.patch and
  shim-keep-unsigned-mokmanager.patch
- Install the vendor certificate to /etc/uefi/certs
* Wed May  8 2013 glin@suse.com
- Add shim-mokmanager-ui-revamp.patch to update the MokManager UI
* Wed Apr  3 2013 glin@suse.com
- Call update-bootloader in %%post to update *.efi in \efi\opensuse
  (bnc#813079)
* Fri Mar  8 2013 glin@suse.com
- Add shim-bnc807760-change-pxe-2nd-loader-name.patch to change the
  PXE 2nd stage loader name (bnc#807760)
- Add shim-bnc808106-correct-certcount.patch to correct the
  certificate count of the signature list (bnc#808106)
* Fri Mar  1 2013 glin@suse.com
- Add shim-bnc798043-no-doulbe-separators.patch to remove double
  seperators from the bootpath (bnc#798043#c4)
* Thu Feb 28 2013 lnussel@suse.de
- sign shim also with openSUSE certificate
* Wed Feb 27 2013 mls@suse.de
- identify project, export certificate as DER file
- don't create an unused extra keypair
* Thu Feb 21 2013 glin@suse.com
- Add shim-bnc804631-fix-broken-bootpath.patch to fix the broken
  bootpath generated in generate_path(). (bnc#804631)
* Mon Feb 11 2013 fcrozat@suse.com
- Update with shim signed by UEFI signing service, based on code
  from "Thu Feb  7 06:56:19 UTC 2013".
* Thu Feb  7 2013 lnussel@suse.de
- prepare for having a signed shim from the UEFI signing service
* Thu Feb  7 2013 glin@suse.com
- Sign shim-opensuse.efi and MokManager.efi with the openSUSE cert
- Add shim-keep-unsigned-mokmanager.patch to keep the unsigned
  MokManager and sign it later.
* Wed Feb  6 2013 mchang@suse.com
- Add shim-install utility
- Add Recommends to grub2-efi
* Wed Jan 30 2013 glin@suse.com
- Add shim-mokmanager-support-crypt-hash-method.patch to support
  password hash from /etc/shadow (FATE#314506)
* Tue Jan 29 2013 glin@suse.com
- Embed openSUSE-UEFI-CA-Certificate.crt in shim
- Rename shim-unsigned.efi to shim-opensuse.efi.
* Fri Jan 18 2013 glin@suse.com
- Update shim-mokmanager-new-pw-hash.patch to extend the password
  hash format
- Rename shim.efi as shim-unsigned.efi
* Wed Jan 16 2013 glin@suse.com
- Merge patches for FATE#314506
  + Add shim-support-mok-delete.patch to add support for deleting
    specific keys
  + Add shim-mokmanager-new-pw-hash.patch to support the new
    password hash.
- Drop shim-correct-mok-size.patch which is included in
  shim-support-mok-delete.patch
- Merge shim-remove-debug-code.patch and
  shim-local-sign-mokmanager.patch into
  shim-local-key-sign-mokmanager.patch
- Install COPYRIGHT
* Tue Jan 15 2013 glin@suse.com
- Add shim-fix-loadoptions.patch to adopt the UEFI shell style
  LoadOptions (bnc#798043)
- Drop shim-check-pk-kek.patch since upstream rejected the patch
  due to violation of SPEC.
- Install EFI binaries to /usr/lib64/efi
* Wed Dec 26 2012 glin@suse.com
- Update shim-reboot-after-changes.patch to avoid rebooting the
  system after enrolling keys/hashes from the file system
- Add shim-correct-mok-size.patch to correct the size of MOK
- Add shim-clear-queued-key.patch to clear the queued key and show
  the menu properly
* Wed Dec 12 2012 fcrozat@suse.com
- Remove shim-rpmlintrc, it wasn't fixing the error, hide error
  stdout to prevent post build check to get triggered by cast
  warnings in openSSL code
- Add shim-remove-debug-code.patch: remove debug code
* Wed Dec 12 2012 glin@suse.com
- Add shim-rpmlintrc to filter 64bit portability errors
* Tue Dec 11 2012 glin@suse.com
- Add shim-local-sign-mokmanager.patch to create a local certicate
  to sign MokManager
- Add shim-get-2nd-stage-loader.patch to get the second stage
  loader path from the load options
- Add shim-check-pk-kek.patch to verify EFI images with PK and KEK
- Add shim-reboot-after-changes.patch to reboot the system after
  enrolling or erasing keys
- Install the EFI images to /usr/lib64/shim instead of the EFI
  partition
- Update the mail address of the author
* Fri Nov  2 2012 glin@suse.com
- Add new package shim 0.2 (FATE#314484)
  + It's in fact git 2fd180a92 since there is no tag for 0.2
