#
# spec file for package qemu
#
# Copyright (c) 2024 SUSE LLC
#
# All modifications and additions to the file contributed by third parties
# remain the property of their copyright owners, unless otherwise agreed
# upon. The license for this file, and modifications and additions to the
# file, is the same license as for the pristine package itself (unless the
# license for the pristine package is not an Open Source License, in which
# case the license is the MIT License). An "Open Source License" is a
# license that conforms to the Open Source Definition (Version 1.9)
# published by the Open Source Initiative.

# Please submit bugfixes or comments via https://bugs.opensuse.org/
#


%include %{_sourcedir}/common.inc

# So, we have openSUSE:Factory, and we have "ports". In openSUSE:Factory, we
# have i586 and x86_64. In the :ARM port, we have aarch64, armv6l and armv7l.
# In the :PowerPC port, we have ppc64, ppc and ppc64le. In the :zSystems port
# we have s390x. And in the :RISCV we have riscv.
#
# Ideally, we'd want to build the firmwares at least once per port, and then
# share the resulting packages among the arch-es within each port (check the
# `ExportFilter` directives in the project config).
#
# Of course, we always build the "native fimrwares" (e.g., x86 firmwares on
# x86_64, PPC firmwares on ppc64le, etc). But we also cross compile as much
# firmwares as we can (e.g., both x86 and PPC firmwares on aarch64) so they'll
# be available in as many ports as possible (as noarch packages).

%ifarch x86_64 aarch64
%define build_ppc_firmware 1
# Currently, opensbi does not cross build cleanly on 15.3 and 15.4
%if ! 0%{?sle_version}
%define build_opensbi_firmware 1
%endif
%define build_x86_firmware 1
%endif
%ifarch ppc64 ppc64le
%define build_ppc_firmware 1
%if ! 0%{?sle_version}
%define build_opensbi_firmware 1
%endif
# FIXME: Try to enable cross building of x86 firmwares here on PPC
%endif
%ifarch riscv64
%define build_opensbi_firmware 1
%endif

%ifarch %ix86 x86_64 ppc ppc64 ppc64le s390x armv7hl aarch64 riscv64
%define kvm_available 1
%define with_uring 1
%define liburing_min_version 1.0
%endif

%ifarch %ix86 x86_64 s390x
%define legacy_qemu_kvm 1
%endif

%ifarch x86_64 aarch64 ppc64le s390x
%define with_rbd 1
%endif

%ifarch x86_64 ppc64le
%define with_daxctl 1
%endif

# enforce pxe rom sizes for migration compatability from SLE 11 SP3 forward
# the following need to be > 64K
%define supported_nics_large {e1000 rtl8139}
# the following need to be <= 64K
%define supported_nics_small {virtio}
# Though not required, make unsupported pxe roms migration compatable as well
%define unsupported_nics {eepro100 ne2k_pci pcnet}

Name:           qemu
URL:            https://www.qemu.org/
Summary:        Machine emulator and virtualizer
License:        BSD-2-Clause AND BSD-3-Clause AND GPL-2.0-only AND GPL-2.0-or-later AND LGPL-2.1-or-later AND MIT
Group:          System/Emulators/PC
Version:        8.2.10
Release:        150600.3.40.1
Source0:        qemu-%{version}.tar.xz
Source1:        common.inc
Source303:      README.PACKAGING
Source1000:     qemu-rpmlintrc
BuildRoot:      %{_tmppath}/%{name}-%{version}-build
## Packages we REQUIRE during build
%if %{build_x86_firmware}
%ifnarch %ix86 x86_64
# We must cross-compile on non-x86*
BuildRequires:  cross-x86_64-binutils
BuildRequires:  cross-x86_64-gcc%gcc_version
%endif
BuildRequires:  acpica
BuildRequires:  binutils-devel
BuildRequires:  dos2unix
BuildRequires:  glibc-devel-32bit glibc-32bit
BuildRequires:  pkgconfig(liblzma)
%endif
%if %{build_opensbi_firmware}
%ifnarch riscv64
BuildRequires:  cross-riscv64-binutils
BuildRequires:  cross-riscv64-gcc%gcc_version
%endif
%endif
%if %{build_ppc_firmware}
%ifnarch ppc64 ppc64le
BuildRequires:  cross-ppc64-binutils
BuildRequires:  cross-ppc64-gcc%gcc_version
%endif
%endif
%ifarch x86_64
BuildRequires:  gcc-32bit
%if %{with_xen}
BuildRequires:  xen-devel >= 4.2
%endif
BuildRequires:  pkgconfig(libpmem)
%endif
%ifnarch %arm s390x
BuildRequires:  libnuma-devel
%endif
%if 0%{with canokey}
BuildRequires:  canokey-qemu-devel
%endif
%if 0%{?with_daxctl}
BuildRequires:  pkgconfig(libndctl)
%endif
%if %{kvm_available}
BuildRequires:  pkgconfig(udev)
%endif
%if 0%{?with_rbd}
BuildRequires:  librbd-devel
%endif
%if 0%{with spice}
BuildRequires:  pkgconfig(spice-protocol) >= 0.12.3
BuildRequires:  pkgconfig(spice-server) >= 0.12.5
%endif
%if 0%{?with_uring}
BuildRequires:  pkgconfig(liburing) >= %liburing_min_version
%endif
%if 0%{with xdp}
BuildRequires:  libxdp-devel
%endif
%if 0%{?suse_version} >= 1600
BuildRequires:  python3-Sphinx
BuildRequires:  python3-base >= 3.8
%else
BuildRequires:  python311-Sphinx
BuildRequires:  python311-base
%endif
BuildRequires:  Mesa-devel
BuildRequires:  bison
BuildRequires:  brlapi-devel
BuildRequires:  discount
BuildRequires:  fdupes
BuildRequires:  flex
BuildRequires:  gcc-c++
BuildRequires:  keyutils-devel
BuildRequires:  libaio-devel
BuildRequires:  libattr-devel
BuildRequires:  libbpf-devel
BuildRequires:  libbz2-devel
BuildRequires:  libcapstone-devel
BuildRequires:  libfdt-devel >= 1.4.2
BuildRequires:  libgcrypt-devel >= 1.8.0
BuildRequires:  lzfse-devel
BuildRequires:  meson
BuildRequires:  multipath-tools-devel
BuildRequires:  ninja >= 1.7
BuildRequires:  pam-devel
BuildRequires:  pkgconfig
BuildRequires:  pkgconfig(alsa)
BuildRequires:  pkgconfig(epoxy)
BuildRequires:  pkgconfig(gbm)
BuildRequires:  pkgconfig(glib-2.0) >= 2.56
BuildRequires:  pkgconfig(glusterfs-api) >= 3
BuildRequires:  pkgconfig(gnutls) >= 3.5.18
BuildRequires:  pkgconfig(gtk+-3.0) >= 3.22
BuildRequires:  pkgconfig(jack)
BuildRequires:  pkgconfig(libcacard) >= 2.5.1
BuildRequires:  pkgconfig(libcap-ng)
BuildRequires:  pkgconfig(libcurl) >= 7.29
BuildRequires:  pkgconfig(libdrm)
BuildRequires:  pkgconfig(libiscsi) >= 1.9.0
BuildRequires:  pkgconfig(libjpeg)
BuildRequires:  pkgconfig(libnfs) >= 1.9.3
BuildRequires:  pkgconfig(libpipewire-0.3)
BuildRequires:  pkgconfig(libpng)
BuildRequires:  pkgconfig(libpulse)
BuildRequires:  pkgconfig(libsasl2)
BuildRequires:  pkgconfig(libseccomp) >= 2.3.0
BuildRequires:  pkgconfig(libssh) >= 0.8.7
BuildRequires:  pkgconfig(libudev)
BuildRequires:  pkgconfig(libusb-1.0) >= 1.0.13
BuildRequires:  pkgconfig(libzstd)
BuildRequires:  pkgconfig(lzo2)
BuildRequires:  pkgconfig(ncurses)
BuildRequires:  pkgconfig(openssl) >= 1.0.0
BuildRequires:  pkgconfig(pixman-1) >= 0.21.8
BuildRequires:  pkgconfig(slirp) >= 4.2.0
BuildRequires:  pkgconfig(systemd)
BuildRequires:  pkgconfig(vdeplug)
BuildRequires:  pkgconfig(virglrenderer) >= 0.4.1
BuildRequires:  pkgconfig(vte-2.91)
BuildRequires:  pkgconfig(xkbcommon)
BuildRequires:  pkgconfig(zlib)
BuildRequires:  rdma-core-devel
BuildRequires:  snappy-devel
BuildRequires:  update-desktop-files
BuildRequires:  usbredir-devel >= 0.6
BuildRequires:  xfsprogs-devel
%{?systemd_ordering}
## Packages we will REQUIRE
%if %{kvm_available}
Requires(post): acl
Requires(post): udev
%endif
Requires(post): coreutils
Requires:       group(kvm)
Requires:       group(qemu)
Requires:       user(qemu)
# Due to change in where some documentation files are, if qemu-guest-agent
# is installed, we need to make sure we update it to our version.
Requires:       (qemu-guest-agent = %{version} if qemu-guest-agent)
## Packages we will RECOMMEND
%ifarch s390x
Recommends:     qemu-hw-s390x-virtio-gpu-ccw
%else
%ifarch arm
Requires:       qemu-hw-display-virtio-gpu-pci
%else
Recommends:     qemu-hw-display-virtio-gpu-pci
%endif
Recommends:     qemu-hw-display-qxl
Recommends:     qemu-hw-display-virtio-gpu
Recommends:     qemu-hw-display-virtio-gpu-pci
Recommends:     qemu-hw-display-virtio-vga
Recommends:     qemu-hw-usb-host
Recommends:     qemu-hw-usb-redirect
Recommends:     qemu-hw-usb-smartcard
%if 0%{with spice}
Recommends:     qemu-ui-spice-app
%endif
# End of "ifarch s390x"
%endif
%if %{kvm_available}
Recommends:     kvm_stat
%endif
Recommends:     qemu-block-curl
Recommends:     qemu-block-nfs
Recommends:     qemu-ksm = %{version}
Recommends:     qemu-tools
Recommends:     qemu-ui-curses
## Packages we will SUGGEST
%if 0%{?with_rbd}
Suggests:       qemu-block-rbd
%endif
Suggests:       qemu-accel-qtest
Suggests:       qemu-block-dmg
Suggests:       qemu-block-gluster
Suggests:       qemu-block-iscsi
Suggests:       qemu-block-ssh
Suggests:       qemu-chardev-baum
Suggests:       qemu-extra
Suggests:       qemu-microvm
Suggests:       qemu-skiboot
Suggests:       qemu-vhost-user-gpu
Suggests:       qemu-ui-gtk
Suggests:       qemu-doc
## Packages we PROVIDE
Provides:       kvm = %{version}
Provides:       qemu-kvm = %{version}
## Pacakges we OBSOLETE (and CONFLICT)
Obsoletes:      kvm <= %{version}
Obsoletes:      qemu-audio-oss < %{version}
Obsoletes:      qemu-audio-sdl < %{version}
Obsoletes:      qemu-kvm <= %{version}
Obsoletes:      qemu-sgabios <= 8
Obsoletes:      qemu-ui-sdl < %{version}
## What we do with the main emulator depends on the architecture we're on
%if %{kvm_available}
%ifarch %ix86 x86_64
Requires:       qemu-x86
%else
Suggests:       qemu-x86
%endif
%ifarch ppc ppc64 ppc64le
Requires:       qemu-ppc
%else
Suggests:       qemu-ppc
%endif
%ifarch s390x
Requires:       qemu-s390x
Requires(post): procps
%else
Suggests:       qemu-s390x
%endif
%ifarch %arm aarch64
Requires:       qemu-arm
%else
Suggests:       qemu-arm
%endif
%ifarch riscv64
Requires:       qemu-extra
%else
Suggests:       qemu-extra
%endif
# End of "if kvm_available"
%endif

%description
%{generic_qemu_description}

This package acts as an umbrella package to the other QEMU sub-packages.

%lang_package

%files
%if %{kvm_available}
%ifarch s390x
%{_prefix}/lib/modules-load.d/kvm.conf
%endif
/usr/lib/udev/rules.d/80-kvm.rules
# End of "if kvm_available"
%endif
%if %{legacy_qemu_kvm}
%doc %_docdir/qemu-kvm
%_mandir/man1/qemu-kvm.1.gz
%endif
%_bindir/qemu-kvm
%dir %_datadir/icons/hicolor
%dir %_datadir/icons/hicolor/*/
%dir %_datadir/icons/hicolor/*/apps
%dir %_datadir/%name
%dir %_datadir/%name/firmware
%dir %_datadir/%name/vhost-user
%dir %_sysconfdir/%name
%dir %_sysconfdir/%name/firmware
%dir /usr/lib/supportconfig
%dir /usr/lib/supportconfig/plugins
%_datadir/applications/qemu.desktop
%_datadir/icons/hicolor/16x16/apps/qemu.png
%_datadir/icons/hicolor/24x24/apps/qemu.png
%_datadir/icons/hicolor/32x32/apps/qemu.bmp
%_datadir/icons/hicolor/32x32/apps/qemu.png
%_datadir/icons/hicolor/48x48/apps/qemu.png
%_datadir/icons/hicolor/64x64/apps/qemu.png
%_datadir/icons/hicolor/128x128/apps/qemu.png
%_datadir/icons/hicolor/256x256/apps/qemu.png
%_datadir/icons/hicolor/512x512/apps/qemu.png
%_datadir/icons/hicolor/scalable/apps/qemu.svg
%_datadir/%name/keymaps
%_datadir/%name/qemu-ifup
%_datadir/%name/qemu-nsis.bmp
%_datadir/%name/trace-events-all
%_mandir/man1/%name.1.gz
%_mandir/man7/qemu-block-drivers.7.gz
%_mandir/man7/qemu-cpu-models.7.gz
%_mandir/man7/qemu-qmp-ref.7.gz
%_mandir/man7/qemu-ga-ref.7.gz
/usr/lib/supportconfig/plugins/%name
%license COPYING COPYING.LIB LICENSE

%if %{kvm_available}
%post
# Do not execute operations affecting host devices while running in a chroot
if [ $(stat -L -c "%i" /proc/1/root/) = $(stat -L -c "%i" /) ]; then
  setfacl --remove-all /dev/kvm &> /dev/null || :
%ifarch s390x
  if [ -c /dev/kvm ]; then
    %_bindir/chmod 0666 /dev/kvm
    %_bindir/chgrp kvm /dev/kvm
  fi
%endif
  %udev_rules_update
  %_bindir/udevadm trigger -y kvm || :
%ifarch s390x
  sysctl vm.allocate_pgste=1 || :
%endif
fi
# End of "if {kvm_available}"
%endif

# Modules need to match {qemu-system-*,qemu-img} version.
# We cannot have qemu and qemu-tools require them in the right version,
# as that would drag in the dependencies the modules are supposed to avoid.
# Nor can we have modules require the right version of qemu and qemu-tools
# as Xen reuses our qemu-tools but does not want our qemu and qemu-x86.
%define qemu_module_conflicts \
Conflicts:      %name < %{version}-%{release} \
Conflicts:      %name > %{version}-%{release} \
Conflicts:      qemu-tools < %{version}-%{release} \
Conflicts:      qemu-tools > %{version}-%{release}

%prep
%autosetup -n qemu-%{version} -p1

# We have the meson subprojects there, but as submodules (because OBS
# SCM bridge can handle the latter, but not the former) so we need to
# apply the layering of the packagefiles manually
meson subprojects packagefiles --apply berkeley-testfloat-3
meson subprojects packagefiles --apply berkeley-softfloat-3

# for the record, this set of firmware files is installed, but we don't
# build (yet): bamboo.dtb canyonlands.dtb hppa-firmware.img openbios-ppc
# openbios-sparc32 openbios-sparc64 palcode-clipper petalogix-ml605.dtb
# petalogix-s3adsp1800.dtb QEMU,cgthree.bin QEMU,tcx.bin qemu_vga.ndrv
# u-boot.e500 u-boot-sam460-20100605.bin opensbi-riscv32-generic-fw_dynamic.bin
# opensbi-riscv32-generic-fw_dynamic.elfnpcm7xx_bootrom.bin vof.bin
# vof-nvram.bin

# Note that:
# - default firmwares are built "by default", i.e., they're built automatically
#   during the process of building QEMU (on each specific arch)
# - extra firmwares are built "manually" (see below)  from their own sources
#   (which, typically, are submodules checked out in the {srcdi}r/roms directory)
%define ppc_default_firmware %{nil}
#{vof.bin vof-nvram.bin}
%define ppc_extra_firmware {skiboot.lid slof.bin}
%define riscv64_default_firmware %{nil}
%define riscv64_extra_firmware {opensbi-riscv64-generic-fw_dynamic.bin}
%define s390x_default_firmware {s390-ccw.img s390-netboot.img}
%define s390x_extra_firmware %{nil}
%define x86_default_firmware {linuxboot.bin linuxboot_dma.bin multiboot.bin \
multiboot_dma.bin kvmvapic.bin pvh.bin}
%define x86_extra_firmware {bios.bin bios-256k.bin bios-microvm.bin qboot.rom \
pxe-e1000.rom pxe-eepro100.rom pxe-ne2k_pci.rom pxe-pcnet.rom pxe-rtl8139.rom \
pxe-virtio.rom vgabios-ati.bin vgabios-bochs-display.bin \
vgabios.bin vgabios-cirrus.bin vgabios-qxl.bin vgabios-ramfb.bin \
vgabios-stdvga.bin vgabios-virtio.bin vgabios-vmware.bin \
efi-e1000.rom efi-e1000e.rom efi-eepro100.rom efi-ne2k_pci.rom efi-pcnet.rom \
efi-rtl8139.rom efi-virtio.rom efi-vmxnet3.rom}

# Complete list of all the firmwares that we build, if we consider
# all the builds, on all the arches.
%define firmware { \
%{ppc_default_firmware} %{ppc_extra_firmware} \
%{riscv64_default_firmware} %{riscv64_extra_firmware} \
%{s390x_default_firmware} %{s390x_extra_firmware} \
%{x86_default_firmware} %{x86_extra_firmware} }

# Note that:
# - {arch}_default_built_firmware are the firmwares that we will be built by
#   default in this particular build, on the arch where we currently are on
# - {arch}_extra_built_fimrware, likewise, but for extra firmwares, built manually
%ifarch ppc64 ppc64le
%define ppc_default_built_firmware %{ppc_default_firmware}
%endif
%ifarch riscv64
%define riscv64_default_built_firmware %{riscv64_default_firmware}
%endif
%ifarch s390x
%define s390x_default_built_firmware %{s390x_default_firmware}
%endif
%ifarch %ix86 x86_64
%define x86_default_built_firmware %{x86_default_firmware}
%endif

%if %{build_opensbi_firmware}
%define riscv64_extra_built_firmware %{riscv64_extra_firmware}
%endif
%if %{build_ppc_firmware}
%define ppc_extra_built_firmware %{ppc_extra_firmware}
%endif
%if %{build_x86_firmware}
%define x86_extra_built_firmware %{x86_extra_firmware}
%endif

# List of only firmwares that will actually be built, in this instance
%define built_firmware { \
%{?ppc_default_built_firmware} %{?ppc_extra_built_firmware} \
%{?riscv64_default_built_firmware} %{?riscv64_extra_built_firmware} \
%{?s390x_default_built_firmware} %{?s390x_extra_built_firmware} \
%{?x86_default_built_firmware} %{?x86_extra_built_firmware} }

%build

%define rpmfilesdir %{_builddir}/qemu-%{version}/rpm

%if %{legacy_qemu_kvm}
# FIXME: Why are we copying the s390 specific one?
cp %{rpmfilesdir}/supported.s390.txt docs/supported.rst
sed -i '/^\ \ \ about\/index.*/i \ \ \ supported.rst' docs/index.rst
%endif

# When generating an upstream release tarball, the following commands
# are run (see scripts/make-release):
#  (cd roms/seabios && git describe --tags --long --dirty > .version)
#  (cd roms/skiboot && ./make_version.sh > .version)
# This has not happened for the archive we're using, since it's cloned
# from a git branch. We, therefore, assumed that the following commands
# have been run, and the result committed to the repository (with seabios
# and skiboot at the proper commit/tag/...):
#  git -C roms/seabios describe --tags --long --dirty > rpm/seabios_version
#  (cd roms/skiboot && ./make_version.sh > ../../rpm/skiboot_version)
cp %{rpmfilesdir}/seabios_version roms/seabios/.version
cp %{rpmfilesdir}/skiboot_version roms/skiboot/.version
find . -iname ".git" -exec rm -rf {} +

mkdir -p %blddir
cd %blddir
export USER=abuild
export HOSTNAME=OBS # is used in roms/SLOF/Makefile.gen (boo#1084909)

# We define a few general and common options and then we disable
# pretty much everything. Afterwards, there is a section for each
# of the flavors where we explicitly enable all the feature we want
# for them.

# TODO: Check whether we want to enable the followings:
# * avx512f
# * debug-info
# * fuse
# * malloc-trim
# * multiprocess
# * qom-cast-debug
# * trace-backends=dtrace
#
# Fedora has avx2 enabled for ix86, while we can't (I tried). Guess it's
# because, for them, ix86 == i686 (while for us it's i586).

# Let's try to stick to _FORTIFY_SOURCE=2 for now
EXTRA_CFLAGS="$(echo %{optflags} | sed -E 's/-[A-Z]?_FORTIFY_SOURCE[=]?[0-9]*//g') -U_FORTIFY_SOURCE -D_FORTIFY_SOURCE=2 -Wno-error"

%srcdir/configure \
%if 0%{?suse_version} >= 1600
	--python=%_bindir/python3 \
%else
	--python=%_bindir/python3.11 \
%endif
	--docdir=%_docdir \
	--datadir=%_datadir \
	--extra-cflags="${EXTRA_CFLAGS}" \
	--firmwarepath=%_datadir/%name \
	--libdir=%_libdir \
	--libexecdir=%_libexecdir \
	--localstatedir=%_localstatedir \
	--prefix=%_prefix \
	--sysconfdir=%_sysconfdir \
	--with-pkgversion="%(echo '%{distro}' | sed 's/ (.*)//')" \
	--disable-af-xdp \
	--disable-alsa \
	--disable-attr \
	--disable-auth-pam \
	--disable-avx2 \
	--disable-avx512f \
	--disable-block-drv-whitelist-in-tools \
	--disable-bochs \
	--disable-bpf \
	--disable-brlapi \
	--disable-bsd-user \
	--disable-bzip2 \
	--disable-cap-ng \
	--disable-capstone \
	--disable-cfi \
	--disable-cfi-debug \
	--disable-cloop \
	--disable-cocoa \
	--disable-coreaudio \
	--disable-coroutine-pool \
	--disable-crypto-afalg \
	--disable-curl \
	--disable-curses \
	--disable-dbus-display \
	--disable-debug-info \
	--disable-debug-mutex \
	--disable-debug-tcg \
	--disable-dmg \
	--disable-docs \
	--disable-download \
	--disable-dsound \
	--disable-fdt \
	--disable-fuse \
	--disable-fuse-lseek \
	--disable-gcrypt \
	--disable-gettext \
	--disable-gio \
	--disable-glusterfs \
	--disable-gnutls \
	--disable-gtk \
	--disable-guest-agent \
	--disable-guest-agent-msi \
	--disable-hv-balloon \
	--disable-hvf \
	--disable-iconv \
	--disable-jack \
	--disable-kvm \
	--disable-l2tpv3 \
	--disable-libdaxctl \
	--disable-libiscsi \
	--disable-libkeyutils \
	--disable-libnfs \
	--disable-libpmem \
	--disable-libssh \
	--disable-libudev \
	--disable-libusb \
	--disable-linux-aio \
	--disable-linux-io-uring \
	--disable-linux-user \
	--disable-live-block-migration \
	--disable-lto \
	--disable-lzfse \
	--disable-lzo \
	--disable-malloc-trim \
	--disable-membarrier \
	--disable-module-upgrades \
	--disable-modules \
	--disable-mpath \
	--disable-multiprocess \
	--disable-netmap \
	--disable-nettle \
	--disable-numa \
	--disable-nvmm \
	--disable-opengl \
	--disable-oss \
	--disable-pa \
	--disable-parallels \
	--disable-pie \
	--disable-pipewire \
	--disable-pixman \
	--disable-plugins \
	--disable-png \
	--disable-pvrdma \
	--disable-qcow1 \
	--disable-qed \
	--disable-qom-cast-debug \
	--disable-rbd \
	--disable-rdma \
	--disable-relocatable \
	--disable-replication \
	--disable-rng-none \
	--disable-rutabaga-gfx \
	--disable-safe-stack \
	--disable-sanitizers \
	--disable-sdl \
	--disable-sdl-image \
	--disable-seccomp \
	--disable-selinux \
	--disable-slirp \
	--disable-slirp-smbd \
	--disable-smartcard \
	--disable-snappy \
	--disable-sparse \
	--disable-spice \
	--disable-spice-protocol \
	--disable-strip \
	--disable-system \
	--disable-tcg \
	--disable-tcg-interpreter \
	--disable-tools \
	--disable-tpm \
	--disable-u2f \
	--disable-usb-redir \
	--disable-user \
	--disable-vde \
	--disable-vdi \
	--disable-vhost-crypto \
	--disable-vhost-kernel \
	--disable-vhost-net \
	--disable-vhost-user \
	--disable-vhost-user-blk-server \
	--disable-vhost-vdpa \
	--disable-virglrenderer \
	--disable-virtfs \
	--disable-vnc \
	--disable-vnc-jpeg \
	--disable-vnc-sasl \
	--disable-vte \
	--disable-vvfat \
	--disable-werror \
	--disable-whpx \
	--disable-xen \
	--disable-xen-pci-passthrough \
	--disable-xkbcommon \
	--disable-zstd \
	--without-default-devices \
%if 0%{?suse_version} >= 1600
	--audio-drv-list=pipewire,pa,alsa,jack,oss \
%else
	--audio-drv-list=pa,pipewire,alsa,jack,oss \
%endif
%ifarch x86_64
	--enable-avx2 \
	--enable-libpmem \
%if %{with_xen}
	--enable-xen \
	--enable-xen-pci-passthrough \
%endif
%endif
%if 0%{with xdp}
	--enable-af-xdp \
%endif
%if 0%{with canokey}
	--enable-canokey \
%endif
%if %{kvm_available}
	--enable-kvm \
%endif
%if 0%{?with_daxctl}
	--enable-libdaxctl \
%endif
%if 0%{?with_uring}
        --enable-linux-io-uring \
%endif
%if "%{_lto_cflags}" != "%{nil}"
	--enable-lto \
%endif
%if %{with malloc_trim}
	--enable-malloc-trim \
%endif
%if %{with system_membarrier}
	--enable-membarrier \
%endif
%ifnarch %arm s390x
	--enable-numa \
%endif
%if 0%{?with_rbd}
	--enable-rbd \
%endif
%if %{has_rutabaga_gfx}
	--enable-rutabaga-gfx \
%endif
	--enable-alsa \
	--enable-attr \
	--enable-auth-pam \
	--enable-bochs \
	--enable-bpf \
	--enable-brlapi \
	--enable-bzip2 \
	--enable-cap-ng \
	--enable-capstone \
	--enable-cloop \
	--enable-coroutine-pool \
	--enable-curl \
	--enable-curses \
	--enable-dbus-display \
	--enable-dmg \
	--enable-docs \
	--enable-fdt=system \
	--enable-gcrypt \
	--enable-gettext \
	--enable-gio \
	--enable-glusterfs \
	--enable-gnutls \
	--enable-gtk \
	--enable-guest-agent \
	--enable-hv-balloon \
	--enable-iconv \
	--enable-jack \
	--enable-l2tpv3 \
	--enable-libiscsi \
	--enable-libkeyutils \
	--enable-libnfs \
	--enable-libssh \
	--enable-libudev \
	--enable-libusb \
	--enable-linux-aio \
	--enable-live-block-migration \
	--enable-lzfse \
	--enable-lzo \
	--enable-modules \
	--enable-mpath \
	--enable-opengl \
	--enable-oss \
	--enable-pa \
	--enable-parallels \
	--enable-pie \
	--enable-pipewire \
	--enable-pixman \
	--enable-png \
	--enable-pvrdma \
	--enable-qcow1 \
	--enable-qed \
	--enable-rdma \
	--enable-relocatable \
	--enable-replication \
	--enable-seccomp \
	--enable-selinux \
	--enable-slirp \
	--enable-slirp-smbd \
	--enable-smartcard \
	--enable-snappy \
%if 0%{with spice}
	--enable-spice \
	--enable-spice-protocol \
%endif
	--enable-system \
	--enable-tcg \
	--enable-tools \
	--enable-tpm \
	--enable-usb-redir \
	--enable-vde \
	--enable-vdi \
	--enable-vhost-crypto \
	--enable-vhost-kernel \
	--enable-vhost-net \
	--enable-vhost-user \
	--enable-vhost-user-blk-server \
	--enable-vhost-vdpa \
	--enable-virglrenderer \
	--enable-virtfs \
	--enable-vnc \
	--enable-vnc-jpeg \
	--enable-vnc-sasl \
	--enable-vte \
	--enable-vvfat \
	--enable-werror \
	--enable-xkbcommon \
	--enable-zstd \
	--with-coroutine=ucontext \
	--with-default-devices

echo "=== Content of config-host.mak: ==="
cat config-host.mak
echo "=== ==="

# For building QEMU and all the "default" firmwares, for each arch,
# for the package qemu, we first need to delete the firmware files that
# we intend to build...
#
# TODO: check if this can be common to qemu and qemu-linux-user
for i in %built_firmware
do
  unlink %srcdir/pc-bios/$i
done

%make_build

# ... Then, we need to reinstate the firmwares that have been built already
for i in %{?s390x_default_built_firmware}
do
  cp pc-bios/s390-ccw/$i %srcdir/pc-bios/
done

for i in %{?x86_default_built_firmware}
do
  cp pc-bios/optionrom/$i %srcdir/pc-bios/
done

# Build the "extra" firmwares. Note that the QEMU Makefile in {srcdir}/roms
# does some cross-compiler auto detection. So we often don't need to define
# or pass CROSS= and CROSS_COMPILE ourselves.

%if %{build_ppc_firmware}
# FIXME: check if we can upstream: Makefile-define-endianess-for-cross-buil.patch
%make_build -C %srcdir/roms skiboot

%make_build -C %srcdir/roms slof
%endif

%if %{build_opensbi_firmware}
%make_build -C %srcdir/roms opensbi64-generic
# End of "if build_ppc_firmware"
%endif

%if %{build_x86_firmware}
%make_build %{?_smp_mflags} -C %srcdir/roms bios \
  SEABIOS_EXTRAVERSION="-rebuilt.opensuse.org" \

pushd %srcdir/roms/seabios/docs
for f in *.md
do
  b="${f%.md}"
  # Ensure the correct media type
  markdown "${f}" >"${b}.html"
  # Links to b.md will be rendered as to b
  ln -Ts "${b}.html" "${b}"
done
popd

# FIXME: check if we can upstream: roms-Makefile-add-cross-file-to-qboot-me.patch
# and qboot-add-cross.ini-file-to-handle-aarch.patch
%make_build -C %srcdir/roms qboot

%make_build -C %srcdir/roms seavgabios \

%make_build -C %srcdir/roms seavgabios-ati \

%make_build -C %srcdir/roms pxerom

%make_build -C %srcdir/roms edk2-basetools EXTRA_OPTFLAGS='-fPIE'
%make_build -C %srcdir/roms efirom

%if %{force_fit_virtio_pxe_rom}
pushd %srcdir
patch -p1 < %{rpmfilesdir}/openSUSE-pcbios-stub-out-the-SAN-req-s-i.patch
popd
%make_build -C %srcdir/roms pxerom_variants=virtio pxerom_targets=1af41000 pxerom
%endif

for i in %supported_nics_large %unsupported_nics
  do
    if test "`stat -c '%s' %srcdir/pc-bios/pxe-$i.rom`" -gt "131072" ; then
    echo "pxe rom is too large"
    exit 1
  fi
  if test "`stat -c '%s' %srcdir/pc-bios/pxe-$i.rom`" -le "65536" ; then
    %srcdir/roms/ipxe/src/util/padimg.pl %srcdir/pc-bios/pxe-$i.rom -s 65536 -b 255
    echo -ne "SEGMENT OVERAGE\0" >> %srcdir/pc-bios/pxe-$i.rom
  fi
done
for i in %supported_nics_small
  do
    if test "`stat -c '%s' %srcdir/pc-bios/pxe-$i.rom`" -gt "65536" ; then
    echo "pxe rom is too large"
    exit 1
  fi
done
# End of "if build_x86_firmware"
%endif

%install
cd %blddir

%make_build install DESTDIR=%{buildroot}

%find_lang %name
install -d -m 0755 %{buildroot}%_datadir/%name/firmware
install -d -m 0755 %{buildroot}/usr/lib/supportconfig/plugins
install -d -m 0755 %{buildroot}%_sysconfdir/%name/firmware
install -D -m 0644 %{rpmfilesdir}/bridge.conf %{buildroot}%_sysconfdir/%name/bridge.conf
install -D -m 0755 %{rpmfilesdir}/qemu-ifup %{buildroot}%_datadir/%name/qemu-ifup
install -D -p -m 0644 %{rpmfilesdir}/80-qemu-ga.rules %{buildroot}/usr/lib/udev/rules.d/80-qemu-ga.rules
install -D -m 0755 scripts/analyze-migration.py  %{buildroot}%_bindir/analyze-migration.py
install -D -m 0755 scripts/vmstate-static-checker.py  %{buildroot}%_bindir/vmstate-static-checker.py
install -D -m 0755 scripts/kvm/vmxcap  %{buildroot}%_bindir/vmxcap
install -D -m 0755 %{rpmfilesdir}/qemu-supportconfig %{buildroot}/usr/lib/supportconfig/plugins/%name
install -D -m 0644 %{rpmfilesdir}/supported.arm.txt %{buildroot}%_docdir/qemu-arm/supported.txt
install -D -m 0644 %{rpmfilesdir}/supported.ppc.txt %{buildroot}%_docdir/qemu-ppc/supported.txt
install -D -m 0644 %{rpmfilesdir}/supported.x86.txt %{buildroot}%_docdir/qemu-x86/supported.txt
install -D -m 0644 %{rpmfilesdir}/supported.s390.txt %{buildroot}%_docdir/qemu-s390x/supported.txt

%if %{legacy_qemu_kvm}
install -D -m 0644 %{rpmfilesdir}/qemu-kvm.1.gz %{buildroot}%_mandir/man1/qemu-kvm.1.gz
install -d %{buildroot}%_docdir/qemu-kvm
# FIXME: Why do we onlly generate the HTML for the legacy package documentation?
%ifarch s390x
ln -s ../qemu-s390x/supported.txt %{buildroot}%_docdir/qemu-kvm/kvm-supported.txt
rst2html --exit-status=2 %{buildroot}%_docdir/qemu-s390x/supported.txt %{buildroot}%_docdir/qemu-kvm/kvm-supported.html
%else
ln -s ../qemu-x86/supported.txt %{buildroot}%_docdir/qemu-kvm/kvm-supported.txt
rst2html --exit-status=2 %{buildroot}%_docdir/qemu-x86/supported.txt %{buildroot}%_docdir/qemu-kvm/kvm-supported.html
# End of "ifarch s390x"
%endif
# End of "if legacy_qemu_kvm"
%endif

%ifarch aarch64 %arm %ix86 ppc ppc64 ppc64le riscv64 s390x x86_64
%ifarch ppc64le
%define qemu_arch ppc64
%endif
ln -s qemu-system-%{qemu_arch} %{buildroot}%_bindir/qemu-kvm
%endif

%if %{kvm_available}
install -D -m 0644 %{rpmfilesdir}/80-kvm.rules %{buildroot}/usr/lib/udev/rules.d/80-kvm.rules
%endif
install -D -p -m 0644 %{rpmfilesdir}/qemu-guest-agent.service %{buildroot}%{_unitdir}/qemu-guest-agent.service
install -D -p -m 0644 %{rpmfilesdir}/ksm.service %{buildroot}%{_unitdir}/ksm.service
%ifarch s390x
install -D -m 0644 %{rpmfilesdir}/kvm.conf %{buildroot}%{_prefix}/lib/modules-load.d/kvm.conf
# End of "if kvm_available"
%endif

# We rely on a separate project / package to provide edk2 firmware
unlink %{buildroot}%_datadir/%name/edk2-licenses.txt
unlink %{buildroot}%_datadir/%name/firmware/50-edk2-i386-secure.json
unlink %{buildroot}%_datadir/%name/firmware/50-edk2-x86_64-secure.json
unlink %{buildroot}%_datadir/%name/firmware/60-edk2-aarch64.json
unlink %{buildroot}%_datadir/%name/firmware/60-edk2-arm.json
unlink %{buildroot}%_datadir/%name/firmware/60-edk2-i386.json
unlink %{buildroot}%_datadir/%name/firmware/60-edk2-x86_64.json
unlink %{buildroot}%_datadir/%name/edk2-aarch64-code.fd
unlink %{buildroot}%_datadir/%name/edk2-arm-code.fd
unlink %{buildroot}%_datadir/%name/edk2-arm-vars.fd
unlink %{buildroot}%_datadir/%name/edk2-i386-code.fd
unlink %{buildroot}%_datadir/%name/edk2-i386-secure-code.fd
unlink %{buildroot}%_datadir/%name/edk2-i386-vars.fd
unlink %{buildroot}%_datadir/%name/edk2-x86_64-code.fd
unlink %{buildroot}%_datadir/%name/edk2-x86_64-secure-code.fd

# this was never meant for customer consumption - delete even though installed
unlink %{buildroot}%_bindir/elf2dmp

# For PPC and x86 firmwares, there are a few extra install steps necessary.
# In general, if we know that we have not built a firmware, remove it from the
# install base, as the one that we have there is the upstream binary, that got
# copied there during `make install`.

%if %{build_ppc_firmware}
# In support of update-alternatives
#
# The reason why we do this, is because we have (only for PPC) an skiboot
# package, shipping an alternative version of skiboot.lid. That is, in fact,
# what's "on the other end" of us supporting update-alternatives for this
# particular firmware.
mv %{buildroot}%_datadir/%name/skiboot.lid %{buildroot}%_datadir/%name/skiboot.lid.qemu
# create a dummy target for /etc/alternatives/skiboot.lid
mkdir -p %{buildroot}%{_sysconfdir}/alternatives
ln -s -f %{_sysconfdir}/alternatives/skiboot.lid %{buildroot}%{_datadir}/%name/skiboot.lid
%else
for f in %{ppc_extra_firmware} ; do
  unlink %{buildroot}%_datadir/%name/$f
done
# End of "if build_ppc_fimrware"
%endif

# For riscv64 firmwares (currently, only opensbi), we leave them there in
# any case, because they're part of the qemu-extra package, and riscv is
# a bit special in many ways already.

%if %{build_x86_firmware}
install -D -m 0644 %{rpmfilesdir}/50-seabios-256k.json %{buildroot}%_datadir/%name/firmware/50-seabios-256k.json
install -D -m 0644 %{rpmfilesdir}/60-seabios-128k.json %{buildroot}%_datadir/%name/firmware/60-seabios-128k.json
install -d -m 0755 %{buildroot}%_docdir/qemu-seabios
%else
for f in %{x86_extra_firmware} ; do
  unlink %{buildroot}%_datadir/%name/$f
done
# End of "if build_x86_firmware"
%endif

%suse_update_desktop_file qemu

# Common install steps for qemu and qemu-linux-user
%fdupes -s %{buildroot}

%check
cd %blddir

# Patch 'increase x86_64 physical bits to 42' requires that the DSDT used for
# acpi [q]tests is modified too. But it's binary, and that means we cannot
# do that in the patch itself. Instead, we keep a copy of the binary in the
# package sources, and put it in place now, before the tests themselves.
# If that patch is removed, the following line needs to go as well.
cp %{rpmfilesdir}/DSDT.pcie %{srcdir}/tests/data/acpi/microvm/

# Patch 'tests/acpi: update tables for new core count test' requires some new
# binaries to be introcuded too. Let's copy them in place as well
cp %{rpmfilesdir}/APIC.core-count2 %{rpmfilesdir}/DSDT.core-count2 %{rpmfilesdir}/FACP.core-count2 %{srcdir}/tests/data/acpi/q35/

%if 0%{?qemu_user_space_build}
# Seccomp is not supported by linux-user emulation
echo 'int main (void) { return 0; }' > %{srcdir}/tests/unit/test-seccomp.c
# keyctl is not yet supported by linux-user emulation
echo 'int main (void) { return 0; }' > %{srcdir}/tests/unit/test-crypto-secret.c
%endif

# Compile the QOM test binary first, so that ...
%make_build tests/qtest/qom-test
# ... make comes in fresh and has lots of address space (needed for 32bit, bsc#957379)
# FIXME: is this still a problem?

# Let's build everything first
%make_build check-build
# Let's now run the 'make check' component individually, so we have
# more control on the options (like -j, etc)
%make_build check-unit
%make_build check-qapi-schema
%make_build check-softfloat
# This would be `make_build check-block`. But iotests are not reliable
# if ran in parallel in OBS, so let's be slow for now.
make -O V=1 VERBOSE=1 -j1 check-block
%if %{with chkqtests} && !0%{?qemu_user_space_build}
# Run qtests sequentially, as it's too unreliable, when run in OBS, if parallelized
make -O V=1 VERBOSE=1 -j1 check-qtest
%endif
# Last step will be to run a full check-report, but we will
# enable this at a later point
#make -O V=1 VERBOSE=1 -j1 check-report.junit.xml

%if 0%{with spice}
%package spice
Summary:        Modules and packages for SPICE
Group:          System/Emulators/PC
Requires:       qemu-audio-spice
Requires:       qemu-chardev-spice
Requires:       qemu-headless
Requires:       qemu-hw-display-qxl
Requires:       qemu-hw-usb-redirect
Requires:       qemu-ui-spice-core

%description spice
%{generic_qemu_description}

This meta-package brings in, as dependencies, the modules and packages
necessary for having SPICE working for your VMs.

%files spice

%package audio-spice
Summary:        Spice based audio support for QEMU
Group:          System/Emulators/PC
Requires:       qemu-ui-spice-core = %{version}-%{release}
%{qemu_module_conflicts}

%description audio-spice
This package contains a module for Spice based audio support for QEMU.

%files audio-spice
%dir %_libdir/%name
%_libdir/%name/audio-spice.so

%package chardev-spice
Summary:        Spice vmc and port chardev support for QEMU
Group:          System/Emulators/PC
Requires:       qemu-ui-spice-core = %{version}-%{release}
%{qemu_module_conflicts}

%description chardev-spice
This package contains a module for Spice chardev support for QEMU.

%files chardev-spice
%dir %_datadir/%name
%dir %_libdir/%name
%_libdir/%name/chardev-spice.so

%package ui-spice-app
Summary:        Spice UI support for QEMU
Group:          System/Emulators/PC
Requires:       qemu-chardev-spice = %{version}-%{release}
Requires:       qemu-ui-spice-core = %{version}-%{release}
%{qemu_module_conflicts}

%description ui-spice-app
This package contains a module for doing Spice based UI for QEMU.

%files ui-spice-app
%dir %_libdir/%name
%_libdir/%name/ui-spice-app.so

%package ui-spice-core
Summary:        Core Spice support for QEMU
Group:          System/Emulators/PC
Requires:       qemu-ui-opengl
# This next Requires is only since virt-manager expects audio support
Requires:       qemu-audio-spice = %{version}-%{release}
%{qemu_module_conflicts}

%description ui-spice-core
This package contains a module with core Spice support for QEMU.

%files ui-spice-core
%dir %_datadir/%name
%dir %_libdir/%name
%_libdir/%name/ui-spice-core.so

%package hw-display-qxl
Summary:        QXL display support for QEMU
Group:          System/Emulators/PC
Requires:       qemu-ui-spice-core = %{version}-%{release}
%{qemu_module_conflicts}

%description hw-display-qxl
This package contains a module for QXL display support for QEMU.

%files hw-display-qxl
%dir %_datadir/%name
%dir %_libdir/%name
%_libdir/%name/hw-display-qxl.so

# End of "with spice"
%endif

%package headless
Summary:        Minimum set of packages for having a functional QEMU
Group:          System/Emulators/PC
Requires:       qemu
Requires:       qemu-block-curl
Requires:       qemu-block-nfs
Requires:       qemu-img
%if %{has_virtiofsd}
Requires:       virtiofsd
%endif
Recommends:     qemu-tools

%description headless
%{generic_qemu_description}

This meta-package brings in, as dependencies, the minimum set of packages
currently necessary for having a functional (headless) QEMU/KVM stack.

%files headless

%package x86
Summary:        Machine emulator and virtualizer for x86 architectures
Group:          System/Emulators/PC
Requires:       %name = %{version}
Requires:       qemu-accel-tcg-x86
Requires:       qemu-ipxe
Requires:       qemu-seabios
Requires:       qemu-vgabios
%ifarch x86_64
Requires:       qemu-ovmf-x86_64
%else
Recommends:     qemu-ovmf-ia32
Recommends:     qemu-ovmf-x86_64
%endif
Recommends:     ovmf
Recommends:     qemu-microvm

%description x86
%{generic_qemu_description}

This package provides i386 and x86_64 emulation.

%files x86
%_bindir/qemu-system-i386
%_bindir/qemu-system-x86_64
%_datadir/%name/kvmvapic.bin
%_datadir/%name/linuxboot.bin
%_datadir/%name/linuxboot_dma.bin
%_datadir/%name/multiboot.bin
%_datadir/%name/multiboot_dma.bin
%_datadir/%name/pvh.bin
%doc %_docdir/qemu-x86

%package ppc
Summary:        Machine emulator and virtualizer for Power architectures
Group:          System/Emulators/PC
Requires:       %name = %{version}
Requires:       qemu-ipxe
Requires:       qemu-SLOF
Requires:       qemu-vgabios

%description ppc
%{generic_qemu_description}

This package provides ppc and ppc64 emulation.

%files ppc
%_bindir/qemu-system-ppc
%_bindir/qemu-system-ppc64
%_datadir/%name/bamboo.dtb
%_datadir/%name/canyonlands.dtb
%_datadir/%name/openbios-ppc
%_datadir/%name/qemu_vga.ndrv
%_datadir/%name/u-boot.e500
%_datadir/%name/u-boot-sam460-20100605.bin
%_datadir/%name/vof*.bin
%doc %_docdir/qemu-ppc

%package s390x
Summary:        Machine emulator and virtualizer for S/390 architectures
Group:          System/Emulators/PC
Requires:       %name = %{version}
Provides:       qemu-s390 = %{version}
Obsoletes:      qemu-s390 < %{version}

%description s390x
%{generic_qemu_description}

This package provides s390x emulation.

%files s390x
%_bindir/qemu-system-s390x
%_datadir/%name/s390-ccw.img
%_datadir/%name/s390-netboot.img
%doc %_docdir/qemu-s390x

%package arm
Summary:        Machine emulator and virtualizer for ARM architectures
Group:          System/Emulators/PC
Requires:       %name = %{version}
Requires:       qemu-ipxe
Recommends:     ovmf
Recommends:     qemu-uefi-aarch64
Recommends:     qemu-vgabios

%description arm
%{generic_qemu_description}

This package provides arm emulation.

%files arm
%_bindir/qemu-system-arm
%_bindir/qemu-system-aarch64
%_datadir/%name/npcm7xx_bootrom.bin
%doc %_docdir/qemu-arm

%package extra
Summary:        Machine emulator and virtualizer for "extra" architectures
Group:          System/Emulators/PC
Requires:       %name = %{version}
Recommends:     qemu-ipxe
Recommends:     qemu-skiboot
Recommends:     qemu-vgabios

%description extra
%{generic_qemu_description}

This package provides some lesser used emulations, including alpha, m68k,
mips, sparc, and xtensa. (The term "extra" is juxtapositioned against more
popular QEMU packages which are dedicated to a single architecture.)

%files extra
%_bindir/qemu-system-alpha
%_bindir/qemu-system-avr
%_bindir/qemu-system-cris
%_bindir/qemu-system-hppa
%_bindir/qemu-system-loongarch64
%_bindir/qemu-system-m68k
%_bindir/qemu-system-microblaze
%_bindir/qemu-system-microblazeel
%_bindir/qemu-system-mips
%_bindir/qemu-system-mipsel
%_bindir/qemu-system-mips64
%_bindir/qemu-system-mips64el
%_bindir/qemu-system-nios2
%_bindir/qemu-system-or1k
%_bindir/qemu-system-riscv32
%_bindir/qemu-system-riscv64
%_bindir/qemu-system-rx
%_bindir/qemu-system-sh4
%_bindir/qemu-system-sh4eb
%_bindir/qemu-system-sparc
%_bindir/qemu-system-sparc64
%_bindir/qemu-system-tricore
%_bindir/qemu-system-xtensa
%_bindir/qemu-system-xtensaeb
%_datadir/%name/hppa-firmware.img
%_datadir/%name/openbios-sparc32
%_datadir/%name/openbios-sparc64
%_datadir/%name/opensbi-riscv32-generic-fw_dynamic.bin
%_datadir/%name/opensbi-riscv64-generic-fw_dynamic.bin
%_datadir/%name/palcode-clipper
%_datadir/%name/petalogix-ml605.dtb
%_datadir/%name/petalogix-s3adsp1800.dtb
%_datadir/%name/QEMU,cgthree.bin
%_datadir/%name/QEMU,tcx.bin

%files lang -f %blddir/%name.lang

%package audio-alsa
Summary:        ALSA based audio support for QEMU
Group:          System/Emulators/PC
%{qemu_module_conflicts}

%description audio-alsa
This package contains a module for ALSA based audio support for QEMU.

%files audio-alsa
%dir %_libdir/%name
%_libdir/%name/audio-alsa.so

%package audio-dbus
Summary:        D-Bus based audio support for QEMU
Group:          System/Emulators/PC
%{qemu_module_conflicts}

%description audio-dbus
This package provides a module for D-Bus based audio support for QEMU.

%files audio-dbus
%dir %_libdir/%name
%_libdir/%name/audio-dbus.so

%package audio-pa
Summary:        Pulse Audio based audio support for QEMU
Group:          System/Emulators/PC
%{qemu_module_conflicts}

%description audio-pa
This package contains a module for Pulse Audio based audio support for QEMU.

%files audio-pa
%dir %_libdir/%name
%_libdir/%name/audio-pa.so

%package audio-jack
Summary:        JACK based audio support for QEMU
Group:          System/Emulators/PC
%{qemu_module_conflicts}

%description audio-jack
This package contains a module for JACK based audio support for QEMU.

%files audio-jack
%dir %_libdir/%name
%_libdir/%name/audio-jack.so

%package audio-oss
Summary:        OSS based audio support for QEMU
Group:          System/Emulators/PC
%{qemu_module_conflicts}

%description audio-oss
This package contains a module for OSS based audio support for QEMU.

%files audio-oss
%dir %_libdir/%name
%_libdir/%name/audio-oss.so

%package audio-pipewire
Summary:        Pipewire based audio support for QEMU
Group:          System/Emulators/PC
%{qemu_module_conflicts}

%description audio-pipewire
This package contains a module for Pipewire based audio support for QEMU.

%files audio-pipewire
%dir %_libdir/%name
%_libdir/%name/audio-pipewire.so

%package block-curl
Summary:        cURL block support for QEMU
Group:          System/Emulators/PC
%{qemu_module_conflicts}

%description block-curl
This package contains a module for accessing network-based image files over
a network connection from qemu-img tool and QEMU system emulation.

%files block-curl
%dir %_libdir/%name
%_libdir/%name/block-curl.so

%package block-dmg
Summary:        DMG block support for QEMU
Group:          System/Emulators/PC
%{qemu_module_conflicts}

%description block-dmg
This package contains a module for accessing Mac OS X image files from
qemu-img tool and QEMU system emulation.

%files block-dmg
%dir %_libdir/%name
%_libdir/%name/block-dmg-bz2.so
%_libdir/%name/block-dmg-lzfse.so

%package block-gluster
Summary:        GlusterFS block support for QEMU
Group:          System/Emulators/PC
%{qemu_module_conflicts}

%description block-gluster
This package contains a module for accessing network-based image files over a
GlusterFS network connection from qemu-img tool and QEMU system emulation.

%files block-gluster
%dir %_libdir/%name
%_libdir/%name/block-gluster.so

%package block-iscsi
Summary:        iSCSI block support for QEMU
Group:          System/Emulators/PC
%{qemu_module_conflicts}

%description block-iscsi
This package contains a module for accessing network-based image files over an
iSCSI network connection from qemu-img tool and QEMU system emulation.

%files block-iscsi
%dir %_libdir/%name
%_libdir/%name/block-iscsi.so

%package block-nfs
Summary:        direct Network File System support for QEMU
Group:          System/Emulators/PC
%{qemu_module_conflicts}

%description block-nfs
This package contains a module for directly accessing nfs based image files
for QEMU.

%files block-nfs
%dir %_libdir/%name
%_libdir/%name/block-nfs.so

%package block-ssh
Summary:        SSH (SFTP) block support for QEMU
Group:          System/Emulators/PC
%{qemu_module_conflicts}

%description block-ssh
This package contains a module for accessing network-based image files over an
SSH network connection from qemu-img tool and QEMU system emulation.

%files block-ssh
%dir %_libdir/%name
%_libdir/%name/block-ssh.so

%package chardev-baum
Summary:        Baum braille chardev support for QEMU
Group:          System/Emulators/PC
%{qemu_module_conflicts}

%description chardev-baum
This package contains a module for baum braille chardev support for QEMU.

%files chardev-baum
%dir %_datadir/%name
%dir %_libdir/%name
%_libdir/%name/chardev-baum.so

%package hw-display-virtio-gpu
Summary:        Virtio GPU display support for QEMU
Group:          System/Emulators/PC
# Make sure that VGA is pretty much always there. Technically, this isn't
# really necessary (and/or, should be dealt with in other places) but it
# makes it easier to deal with strange situation where, e.g., GRUB is
# configured to work only with a graphical terminal (see bsc#1219164),
# and the hw-display-virtio-vga package is small enough, anyway.
Requires:       qemu-hw-display-virtio-vga = %{version}-%{release}
%{qemu_module_conflicts}

%description hw-display-virtio-gpu
This package contains a module for Virtio GPU display support for QEMU.

%files hw-display-virtio-gpu
%dir %_datadir/%name
%_libdir/%name/hw-display-virtio-gpu.so
%_libdir/%name/hw-display-virtio-gpu-gl.so

%package hw-display-virtio-gpu-pci
Summary:        Virtio-gpu pci device for QEMU
Group:          System/Emulators/PC
Requires:       qemu-hw-display-virtio-gpu = %{version}-%{release}
%{qemu_module_conflicts}

%description hw-display-virtio-gpu-pci
This package contains a module providing the virtio gpu pci device for QEMU.

%files hw-display-virtio-gpu-pci
%dir %_datadir/%name
%dir %_libdir/%name
%_libdir/%name/hw-display-virtio-gpu-pci.so
%_libdir/%name/hw-display-virtio-gpu-pci-gl.so

%package hw-display-virtio-vga
Summary:        Virtio vga device for QEMU
Group:          System/Emulators/PC
%{qemu_module_conflicts}

%description hw-display-virtio-vga
This package contains a module providing the virtio vga device for QEMU.

%files hw-display-virtio-vga
%dir %_datadir/%name
%dir %_libdir/%name
%_libdir/%name/hw-display-virtio-vga.so
%_libdir/%name/hw-display-virtio-vga-gl.so

%package hw-s390x-virtio-gpu-ccw
Summary:        S390x virtio-gpu ccw device for QEMU
Group:          System/Emulators/PC
Requires:       qemu-hw-display-virtio-gpu = %{version}-%{release}
%{qemu_module_conflicts}

%description hw-s390x-virtio-gpu-ccw
This package contains a module providing the s390x virtio gpu ccw device for
QEMU.

%files hw-s390x-virtio-gpu-ccw
%dir %_datadir/%name
%dir %_libdir/%name
%_libdir/%name/hw-s390x-virtio-gpu-ccw.so

%package hw-usb-redirect
Summary:        USB redirection support for QEMU
Group:          System/Emulators/PC
%{qemu_module_conflicts}

%description hw-usb-redirect
This package contains a module for USB redirection support for QEMU.

%files hw-usb-redirect
%dir %_datadir/%name
%dir %_libdir/%name
%_libdir/%name/hw-usb-redirect.so

%package hw-usb-smartcard
Summary:        USB smartcard support for QEMU
Group:          System/Emulators/PC
%{qemu_module_conflicts}

%description hw-usb-smartcard
This package contains a modules for USB smartcard support for QEMU.

%files hw-usb-smartcard
%dir %_datadir/%name
%dir %_libdir/%name
%_libdir/%name/hw-usb-smartcard.so

%package hw-usb-host
Summary:        USB passthrough driver support for QEMU
Group:          System/Emulators/PC
%{qemu_module_conflicts}

%description hw-usb-host
This package contains a modules for USB passthrough driver for QEMU.

%files hw-usb-host
%dir %_datadir/%name
%dir %_libdir/%name
%_libdir/%name/hw-usb-host.so

%package ui-dbus
Summary:        D-Bus based UI support for QEMU
Group:          System/Emulators/PC
%{qemu_module_conflicts}

%description ui-dbus
This package contains a module for doing D-Bus based UI for QEMU.

%files ui-dbus
%dir %_libdir/%name
%_libdir/%name/ui-dbus.so

%package ui-curses
Summary:        Curses based UI support for QEMU
Group:          System/Emulators/PC
%{qemu_module_conflicts}

%description ui-curses
This package contains a module for doing curses based UI for QEMU.

%files ui-curses
%dir %_libdir/%name
%_libdir/%name/ui-curses.so

%package ui-gtk
Summary:        GTK based UI support for QEMU
Group:          System/Emulators/PC
Requires:       qemu-ui-opengl
Supplements:    (qemu and libgtk-3-0)
%{qemu_module_conflicts}

%description ui-gtk
This package contains a module for doing GTK based UI for QEMU.

%files ui-gtk
%dir %_libdir/%name
%_libdir/%name/ui-gtk.so

%package ui-opengl
Summary:        OpenGL based UI support for QEMU
Group:          System/Emulators/PC
%{qemu_module_conflicts}

%description ui-opengl
This package contains a module for doing OpenGL based UI for QEMU.

%files ui-opengl
%dir %_datadir/%name
%dir %_libdir/%name
%_libdir/%name/ui-egl-headless.so
%_libdir/%name/ui-opengl.so

%package vhost-user-gpu
Summary:        Vhost user mode virtio-gpu 2D/3D rendering backend for QEMU
Group:          System/Emulators/PC
%{qemu_module_conflicts}

%description vhost-user-gpu
This package contains a vhost user mode virtio-gpu 2D/3D rendering backend for
QEMU.

%files vhost-user-gpu
%dir %_datadir/%name/vhost-user
%_datadir/%name/vhost-user/50-qemu-gpu.json
%_libexecdir/vhost-user-gpu

%package img
Summary:        QEMU disk image utility
Group:          System/Emulators/PC

%description img
This package provides command line tools for manipulating disk images.

%files img
%_bindir/qemu-img
%_bindir/qemu-io
%_bindir/qemu-nbd
%_bindir/qemu-storage-daemon
%_mandir/man1/qemu-img.1.gz
%_mandir/man8/qemu-nbd.8.gz
%_mandir/man1/qemu-storage-daemon.1.gz
%_mandir/man7/qemu-storage-daemon-qmp-ref.7.gz

%package pr-helper
Summary:        QEMU persistent reservation helper
Group:          System/Emulators/PC

%description pr-helper
This package provides a helper utility for SCSI persistent reservations.

%files pr-helper
%_bindir/qemu-pr-helper
%_mandir/man8/qemu-pr-helper.8.gz

%package tools
Summary:        Tools for QEMU
Group:          System/Emulators/PC
Requires(pre):  permissions
Requires:       qemu-img
Requires:       qemu-pr-helper
Requires:       group(kvm)
%if %{has_virtiofsd}
Requires:       virtiofsd
%endif
Recommends:     multipath-tools
Recommends:     qemu-block-curl
%if 0%{?with_rbd}
Recommends:     qemu-block-rbd
%endif

%description tools
This package contains various QEMU related tools, including a bridge helper,
a virtfs helper, ivshmem, disk utilities and scripts for various purposes.

%files tools
%_bindir/analyze-migration.py
%_bindir/qemu-edid
%_bindir/qemu-keymap
%_bindir/vmstate-static-checker.py
%_bindir/vmxcap
%verify(not mode) %attr(4750,root,kvm) %_libexecdir/qemu-bridge-helper
%_libexecdir/virtfs-proxy-helper
%_mandir/man1/virtfs-proxy-helper.1.gz
%dir %_sysconfdir/%name
%config(noreplace) %_sysconfdir/%name/bridge.conf

%post tools
%set_permissions %_libexecdir/qemu-bridge-helper

%verifyscript tools
%verify_permissions %_libexecdir/qemu-bridge-helper

%package ivshmem-tools
Summary:        Inter-VM Shared Memory Tools for QEMU
Group:          System/Emulators/PC

%description ivshmem-tools
This package contains a sample shared memory client and server which utilize
QEMU's Inter-VM shared memory device as specified by the ivshmem client-server
protocol specification documented in docs/specs/ivshmem-spec.txt in QEMU source
code.

%files ivshmem-tools
%dir %_datadir/%name
%_bindir/ivshmem-client
%_bindir/ivshmem-server

%package guest-agent
Summary:        Guest agent for QEMU
Group:          System/Emulators/PC
Requires:       group(kvm)
Requires(post): udev
Supplements:    modalias(acpi*:QEMU0002%3A*)
Supplements:    modalias(pci:v00005853d00000001sv*sd*bc*sc*i*)
Supplements:    modalias(pci:v0000FFFDd00000101sv*sd*bc*sc*i*)
%{?systemd_ordering}

%description guest-agent
This package contains the QEMU guest agent. It is installed in the linux guest
to provide information and control at the guest OS level.

%files guest-agent
%attr(0755,root,kvm) %_bindir/qemu-ga
%_mandir/man8/qemu-ga.8.gz
%{_unitdir}/qemu-guest-agent.service
/usr/lib/udev/rules.d/80-qemu-ga.rules

%pre guest-agent
%service_add_pre qemu-guest-agent.service

%post guest-agent
%service_add_post qemu-guest-agent.service
if [ -e /dev/virtio-ports/org.qemu.guest_agent.0 ]; then
  /usr/bin/systemctl start qemu-guest-agent.service || :
fi

%preun guest-agent
if [ -e /dev/virtio-ports/org.qemu.guest_agent.0 ]; then
  /usr/bin/systemctl stop qemu-guest-agent.service || :
fi

%postun guest-agent
%service_del_postun_without_restart qemu-guest-agent.service
if [ "$1" = "1" ] ; then
  if [ -e /dev/virtio-ports/org.qemu.guest_agent.0 ]; then
    /usr/bin/systemctl restart qemu-guest-agent.service || :
  fi
fi

%package ksm
Summary:        Kernel Samepage Merging services
Group:          System/Emulators/PC
Requires(pre):  coreutils
Requires(post): coreutils

%description ksm
Kernel Samepage Merging (KSM) is a memory-saving de-duplication feature, that
merges anonymous (private) pages (not pagecache ones).

This package provides a service file for starting and stopping KSM.

%files ksm
%{_unitdir}/ksm.service

%pre ksm
%service_add_pre ksm.service

%post ksm
%service_add_post ksm.service

%preun ksm
%service_del_preun ksm.service

%postun ksm
%service_del_postun ksm.service

%package accel-tcg-x86
Summary:        TCG accelerator for QEMU
Group:          System/Emulators/PC
%{qemu_module_conflicts}

%description accel-tcg-x86
TCG is the QEMU binary translator, responsible for converting from target to
host instruction set.

This package provides the TCG accelerator for QEMU.

%files accel-tcg-x86
%dir %_datadir/%name
%dir %_libdir/%name
%_libdir/%name/accel-tcg-i386.so
%_libdir/%name/accel-tcg-x86_64.so

%package accel-qtest
Summary:        QTest accelerator for QEMU
Group:          System/Emulators/PC
%{qemu_module_conflicts}

%description accel-qtest
QTest is a device emulation testing framework. It is useful to test device
models.

This package provides QTest accelerator for testing QEMU.

%files accel-qtest
%dir %_datadir/%name
%dir %_libdir/%name
%_libdir/%name/accel-qtest-aarch64.so
%_libdir/%name/accel-qtest-alpha.so
%_libdir/%name/accel-qtest-arm.so
%_libdir/%name/accel-qtest-avr.so
%_libdir/%name/accel-qtest-cris.so
%_libdir/%name/accel-qtest-hppa.so
%_libdir/%name/accel-qtest-i386.so
%_libdir/%name/accel-qtest-loongarch64.so
%_libdir/%name/accel-qtest-m68k.so
%_libdir/%name/accel-qtest-microblaze.so
%_libdir/%name/accel-qtest-microblazeel.so
%_libdir/%name/accel-qtest-mips.so
%_libdir/%name/accel-qtest-mips64.so
%_libdir/%name/accel-qtest-mips64el.so
%_libdir/%name/accel-qtest-mipsel.so
%_libdir/%name/accel-qtest-nios2.so
%_libdir/%name/accel-qtest-or1k.so
%_libdir/%name/accel-qtest-ppc.so
%_libdir/%name/accel-qtest-ppc64.so
%_libdir/%name/accel-qtest-riscv32.so
%_libdir/%name/accel-qtest-riscv64.so
%_libdir/%name/accel-qtest-rx.so
%_libdir/%name/accel-qtest-s390x.so
%_libdir/%name/accel-qtest-sh4.so
%_libdir/%name/accel-qtest-sh4eb.so
%_libdir/%name/accel-qtest-sparc.so
%_libdir/%name/accel-qtest-sparc64.so
%_libdir/%name/accel-qtest-tricore.so
%_libdir/%name/accel-qtest-x86_64.so
%_libdir/%name/accel-qtest-xtensa.so
%_libdir/%name/accel-qtest-xtensaeb.so

%if 0%{?with_rbd}
%package block-rbd
Summary:        Rados Block Device (Ceph) support for QEMU
Group:          System/Emulators/PC
%{qemu_module_conflicts}

%description block-rbd
This package contains a module for accessing ceph (rbd,rados) image files
for QEMU.

%files block-rbd
%dir %_libdir/%name
%_libdir/%name/block-rbd.so
# End of "if with_rbd"
%endif

%if %{build_ppc_firmware}
%package SLOF
Summary:        Slimline Open Firmware - SLOF
Group:          System/Emulators/PC
BuildArch:      noarch

%description SLOF
Slimline Open Firmware (SLOF) is an implementation of the IEEE 1275 standard.
It can be used as partition firmware for pSeries machines running on QEMU or KVM.

%files SLOF
%dir %_datadir/%name
%_datadir/%name/slof.bin

%package skiboot
Summary:        OPAL firmware (aka skiboot), used in booting OpenPOWER systems
Group:          System/Emulators/PC
BuildArch:      noarch
Requires(post): update-alternatives
Requires(postun): update-alternatives

%description skiboot
Provides OPAL (OpenPower Abstraction Layer) firmware, aka skiboot, as
traditionally packaged with QEMU.

%files skiboot
%dir %_datadir/%name
%_datadir/%name/skiboot.lid
%_datadir/%name/skiboot.lid.qemu
%ghost %_sysconfdir/alternatives/skiboot.lid

%post skiboot
update-alternatives --install \
   %{_datadir}/%name/skiboot.lid skiboot.lid %{_datadir}/%name/skiboot.lid.qemu 15

%preun skiboot
if [ ! -f %{_datadir}/%name/skiboot.lid.qemu ] ; then
   update-alternatives --remove skiboot.lid %{_datadir}/%name/skiboot.lid.qemu
fi
# End of "if build_ppc_firmware"
%endif

%if %{build_x86_firmware}
%package microvm
Summary:        x86 MicroVM firmware for QEMU
Group:          System/Emulators/PC
BuildArch:      noarch

%description microvm
This package provides minimal x86 firmware for booting certain guests under
QEMU. qboot provides the minimum resources needed to boot PVH and bzImages.
bios-microvm, created from a minimal seabios configuration, provides slightly
wider support than qboot, but still focuses on quick boot up.

%files microvm
%dir %_datadir/%name
%_datadir/%name/bios-microvm.bin
%_datadir/%name/qboot.rom

%package seabios
Summary:        x86 Legacy BIOS for QEMU
Group:          System/Emulators/PC
Version:        8.2.10%{sbver}
Release:        150600.3.40.1
BuildArch:      noarch
Conflicts:      %name < 1.6.0

%description seabios
SeaBIOS is an open source implementation of a 16bit x86 BIOS. SeaBIOS
is the default and legacy BIOS for QEMU.

%files seabios
%dir %_datadir/%name
%_datadir/%name/bios.bin
%_datadir/%name/bios-256k.bin
%_datadir/%name/firmware/50-seabios-256k.json
%_datadir/%name/firmware/60-seabios-128k.json
%license roms/seabios/COPYING
%doc %_docdir/qemu-seabios

%package vgabios
Summary:        VGA BIOSes for QEMU
Group:          System/Emulators/PC
Version:        8.2.10%{sbver}
Release:        150600.3.40.1
BuildArch:      noarch
Conflicts:      %name < 1.6.0

%description vgabios
VGABIOS provides the video ROM BIOSes for the following variants of VGA
emulated devices: Std VGA, QXL, Cirrus CLGD 5446 and VMware emulated
video card. For use with QEMU.

%files vgabios
%dir %_datadir/%name
%_datadir/%name/vgabios.bin
%_datadir/%name/vgabios-ati.bin
%_datadir/%name/vgabios-bochs-display.bin
%_datadir/%name/vgabios-cirrus.bin
%_datadir/%name/vgabios-qxl.bin
%_datadir/%name/vgabios-ramfb.bin
%_datadir/%name/vgabios-stdvga.bin
%_datadir/%name/vgabios-virtio.bin
%_datadir/%name/vgabios-vmware.bin
%license roms/seabios/COPYING

%package ipxe
Summary:        PXE ROMs for QEMU NICs
Group:          System/Emulators/PC
Version:        8.2.10
Release:        150600.3.40.1
BuildArch:      noarch
Conflicts:      %name < 1.6.0

%description ipxe
Provides Preboot Execution Environment (PXE) ROM support for various emulated
network adapters available with QEMU.

%files ipxe
%dir %_datadir/%name
%_datadir/%name/efi-e1000.rom
%_datadir/%name/efi-e1000e.rom
%_datadir/%name/efi-eepro100.rom
%_datadir/%name/efi-ne2k_pci.rom
%_datadir/%name/efi-pcnet.rom
%_datadir/%name/efi-rtl8139.rom
%_datadir/%name/efi-virtio.rom
%_datadir/%name/efi-vmxnet3.rom
%_datadir/%name/pxe-e1000.rom
%_datadir/%name/pxe-eepro100.rom
%_datadir/%name/pxe-ne2k_pci.rom
%_datadir/%name/pxe-pcnet.rom
%_datadir/%name/pxe-rtl8139.rom
%_datadir/%name/pxe-virtio.rom
# End of "if build_x86_firmware"
%endif

%package doc
Summary:        Documentation for QEMU
Group:          System/Emulators/PC
BuildArch:      noarch
Suggests:       qemu

%files doc
%doc %_docdir/%name

%description doc
%{generic_qemu_description}

This package contains user and developer documentation for QEMU.

%changelog
* Fri Oct 10 2025 dfaggioli@suse.com
- fix bsc#1230042:
  * .gitmodules: move u-boot mirrors to qemu-project-mirrors
  * [openSUSE] rpm/spec: qemu-vgabios is required on ppc (bsc#1230042)
* Tue Jul 22 2025 dfaggioli@suse.com
- Fix bsc#1246566:
  * [roms] seabios: include "pciinit: don't misalign large BARs" (bsc#1246566)
* Wed Jun 18 2025 dfaggioli@suse.com
- Fix bsc#1240157:
  * [openSUSE][RPM] spec: Require ipxe and virtio-gpu packages for more arch-es (bsc#1240157)
* Fri Jun 13 2025 dfaggioli@suse.com
- Fix bsc#1243013:
  * [openSUSE] rpm: disable LTO for userspace emulation on 15.6
- Update version to 8.2.10
  * Full changelog here:
    https://lore.kernel.org/qemu-devel/7dd1fbc7-a58f-4b2c-82b9-735840246ab2@tls.msk.ru/
  * Some backports:
    hw/misc/aspeed_hace: Fix buffer overflow in has_padding function
    target/ppc: Fix e200 duplicate SPRs
    linux-user/riscv: Fix handling of cpu mask in riscv_hwprobe syscall
    docs/about/emulation: Fix broken link
    vdpa: Allow vDPA to work on big-endian machine
    vdpa: Fix endian bugs in shadow virtqueue
    target/loongarch: Fix vldi inst
    target/arm: Simplify pstate_sm check in sve_access_check
    target/arm: Make DisasContext.{fp, sve}_access_checked tristate
    util/cacheflush: Make first DSB unconditional on aarch64
    ui/cocoa: Temporarily ignore annoying deprecated declaration warnings
    docs: Rename default-configs to configs
    block: Zero block driver state before reopening
    hw/xen/hvm: Fix Aarch64 typo
    hw/net/smc91c111: Don't allow data register access to overrun buffer
    hw/net/smc91c111: Sanitize packet length on tx
    hw/net/smc91c111: Sanitize packet numbers
    hw/net/smc91c111: Ignore attempt to pop from empty RX fifo
    ppc/pnv/occ: Fix common area sensor offsets
    net: move backend cleanup to NIC cleanup
    net: parameterize the removing client from nc list
    util/qemu-timer.c: Don't warp timer from timerlist_rearm()
    target/arm: Correct STRD atomicity
    target/arm: Correct LDRD atomicity and fault behaviour
    ...
* Mon Mar 24 2025 dfaggioli@suse.com
- Fixes for bsc#1237603, bsc#1236709, bsc#1236709:
  * linux-user: Do not define struct sched_attr if libc headers do (bsc#1237603)
  * elfload: Fix alignment when unmapping excess reservation (bsc#1236709)
  * linux-user: Honor elf alignment when placing images (bsc#1236709)
* Thu Feb 13 2025 dfaggioli@suse.com
- Fix bsc#1228343:
  * tests/acpi: q35: Update host address width in DMAR (bsc#1228343)
  * intel_iommu: Set default aw_bits to 48 starting from QEMU 9.2 (bsc#1228343)
  * tests/acpi: q35: allow DMAR acpi table changes (bsc#1228343)
- Update version to 8.2.9
  * Full changelog here:
    https://lore.kernel.org/qemu-devel/568115c5-8908-4931-aae0-a57bf542f68a@tls.msk.ru/
  * Some backports:
    gdbstub/user-target: fix gdbserver int format (%%d -> %%x)
    target/s390x: Fix MVC not always invalidating translation blocks
    physmem: fix qemu_ram_alloc_from_fd size calculation
    hw/usb/canokey: Fix buffer overflow for OUT packet
    target/arm: arm_reset_sve_state() should set FPSR, not FPCR
    hw/cxl: Fix msix_notify: Assertion `vector < dev->msix_entries_nr`
    tests: acpi: update expected blobs
    pci: acpi: Windows 'PCI Label Id' bug workaround
    tests: acpi: whitelist expected blobs
    pci/msix: Fix msix pba read vector poll end calculation
    pci: ensure valid link status bits for downstream ports
    hw/usb/hcd-xhci-pci: Use modulo to select MSI vector as per spec
    backends/cryptodev-vhost-user: Fix local_error leaks
    tests/qtest/boot-serial-test: Correct HPPA machine name
    target/i386/cpu: Fix notes for CPU models
    docs: Correct release of TCG trace-events removal
    target/loongarch: Use actual operand size with vbsrl check
    s390x/s390-virtio-ccw: don't crash on weird RAM sizes
    vhost-user: fix shared object return values
    meson.build: Disallow libnfs v6 to fix the broken macOS build
    hw/intc/arm_gicv3_its: Zero initialize local DTEntry etc structs
    roms: re-add edk2-basetools target
    edk2: commit version info
    edk2: get version + date from git submodule
    x86/loader: only patch linux kernels
    ...
* Fri Dec 20 2024 dfaggioli@suse.com
- Properly fix bsc#1232283:
  * nbd/server: avoid per-NBDRequest nbd_client_get/put() (bsc#1232283)
* Tue Nov 26 2024 dfaggioli@suse.com
- Fix boo#1231166, bsc#1228079, bsc#1221812,
  bsc#1232283, bsc#1230978, bsc#1233530:
  * [openSUSE][RPM] The qemu translation is not being installed (boo#1231166)
  * target/i386: Expose IBPB-BRTYPE and SBPB CPUID bits to the guest (bsc#1228079)
  * [openSUSE] block: Move qcow2 invalidation query-block op_blocker
  * scsi: megasas: Internal cdbs have 16-byte length (bsc#1233530)
* Tue Nov 26 2024 dfaggioli@suse.com
- Update version to 8.2.8
  * Full changelog here:
    https://lore.kernel.org/qemu-devel/1b7d3242-abe9-408a-bd77-85162e0d0de4@tls.msk.ru/
  * Some backports:
    usb-hub: Fix handling port power control messages
    hw/audio/hda: fix memory leak on audio setup (bsc#1232728)
    hw/misc/mos6522: Fix bad class definition of the MOS6522 device
    contrib/plugins: add compat for g_memdup2
    target/i386: fix hang when using slow path for ptw_setl
    tcg: Allow top bit of SIMD_DATA_BITS to be set in simd_desc()
    linux-user/arm: Select vdso for be8 and be32 modes
    linux-user/arm: Reduce vdso alignment to 4k
    linux-user: Tolerate CONFIG_LSM_MMAP_MIN_ADDR
    accel/tcg: Fix user-only probe_access_internal plugin check
    target/arm: Drop user-only special case in sve_stN_r
    linux-user: Fix setreuid and setregid to use direct syscalls
    target/i386: Fix legacy page table walk
    9pfs: fix crash on 'Treaddir' request
    hw/nvme: fix handling of over-committed queues
    migration: Ensure vmstate_save() sets errp
    target/arm: Fix SVE SDOT/UDOT/USDOT (4-way, indexed)
    target/ppc: Fix mtDPDES targeting SMT siblings
    target/ppc: Set ctx->opcode for decode_insn32()
    target/riscv: Fix vcompress with rvv_ta_all_1s
    target/riscv/kvm: clarify how 'riscv-aia' default works
    target/riscv/kvm: set 'aia_mode' to default in error path
    hw/intc/riscv_aplic: Check and update pending when write sourcecfg
    hw/intc/riscv_aplic: Fix in_clrip[x] read emulation
    target/riscv: Set vtype.vill on CPU reset
    ... ... ...
* Thu Oct 17 2024 dfaggioli@suse.com
- Bugfixes and CVEs:
  * hw/usb/hcd-ohci: Fix #1510, #303: pid not IN or OUT (bsc#1230834, CVE-2024-8354)
  * softmmu: Support concurrent bounce buffers (bsc#1230915, CVE-2024-8612)
  * system/physmem: Per-AddressSpace bounce buffering (bsc#1230915, CVE-2024-8612)
  * system/physmem: Propagate AddressSpace to MapClient helpers (bsc#1230915, CVE-2024-8612)
  * system/physmem: Replace qemu_mutex_lock() calls with QEMU_LOCK_GUARD (bsc#1230915, CVE-2024-8612)
* Tue Oct 15 2024 dfaggioli@suse.com
- Update version to 8.2.7
  * Full changelog here:
    https://lore.kernel.org/qemu-devel/d9ff276f-f1ba-4e90-8343-a7a0dc2bf305@tls.msk.ru/
  * Fixes:
    bsc#1229007, CVE-2024-7409
    bsc#1224132, CVE-2024-4693
  * Some backports:
    gitlab: fix logic for changing docker tag on stable branches
    ui/sdl2: set swap interval explicitly when OpenGL is enabled
    hw/intc/arm_gic: fix spurious level triggered interrupts
    hw/audio/virtio-sound: fix heap buffer overflow
    tests/docker: update debian i686 and mipsel images to bookworm
    tests/docker: remove debian-armel-cross
    hw/display/vhost-user-gpu.c: fix vhost_user_gpu_chr_read()
    crypto: check gnutls & gcrypt support the requested pbkdf hash
    crypto: run qcrypto_pbkdf2_count_iters in a new thread
    softmmu/physmem: fix memory leak in dirty_memory_extend()
    target/ppc: Fix migration of CPUs with TLB_EMB TLB type
    gitlab: migrate the s390x custom machine to 22.04
    target/hppa: Fix PSW V-bit packaging in cpu_hppa_get for hppa64
    hw/audio/virtio-snd: fix invalid param check
    virtio-pci: Fix the use of an uninitialized irqfd
* Tue Sep 17 2024 dfaggioli@suse.com
- Fix bsc#1229929, bsc#1230140 (patch already submitted upstream):
  * [openSUSE] target/ppc: Fix lxvx/stxvx facility check (bsc#1229929)
* Tue Aug 27 2024 dfaggioli@suse.com
- Fix bsc#1221812:
  * [openSUSE] block: Reschedule query-block during qcow2 invalidation (bsc#1221812)
- Fixup old patches and infra issues:
  * [openSUSE] fixup patch "pc: q35: Allow 1024 cpus for old machine types"
  * [openSUSE] Point submodule branches to gitlab
* Tue Aug 13 2024 dfaggioli@suse.com
- Fix bsc#1229007, CVE-2024-7409:
  * nbd/server: CVE-2024-7409: Close stray clients at server-stop (bsc#1229007)
  * nbd/server: CVE-2024-7409: Drop non-negotiating clients (bsc#1229007)
  * nbd/server: CVE-2024-7409: Cap default max-connections to 100 (bsc#1229007)
  * nbd/server: Plumb in new args to nbd_client_add() (bsc#1229007, CVE-2024-7409)
  * nbd: Minor style and typo fixes (bsc#1229007, CVE-2024-7409)
* Mon Aug 12 2024 dfaggioli@suse.com
- Update to version 8.2.6:
  Full backport lists (from the various releases) here:
  https://lore.kernel.org/qemu-devel/1721203806.547734.831464.nullmailer@tls.msk.ru/
  Some of the upstream backports are:
  hw/nvme: fix number of PIDs for FDP RUH update
  sphinx/qapidoc: Fix to generate doc for explicit, unboxed arguments
  char-stdio: Restore blocking mode of stdout on exit
  virtio: remove virtio_tswap16s() call in vring_packed_event_read()
  virtio-pci: Fix the failure process in kvm_virtio_pci_vector_use_one()
  block: Parse filenames only when explicitly requested
  iotests/270: Don't store data-file with json: prefix in image
  iotests/244: Don't store data-file with protocol in image
  qcow2: Don't open data_file with BDRV_O_NO_IO (bsc#1227322, CVE-2024-4467)
  target/arm: Fix FJCVTZS vs flush-to-zero
  target/arm: Fix VCMLA Dd, Dn, Dm[idx]
  i386/cpu: fixup number of addressable IDs for processor cores in the physical package
  tests: Update our CI to use CentOS Stream 9 instead of 8
  migration: Fix file migration with fdset
  tcg/loongarch64: Fix tcg_out_movi vs some pcrel pointers
  target/sparc: use signed denominator in sdiv helper
  linux-user: Make TARGET_NR_setgroups affect only the current thread
  accel/tcg: Fix typo causing tb->page_addr[1] to not be recorded
  stdvga: fix screen blanking
  hw/audio/virtio-snd: Always use little endian audio format
  ui/gtk: Draw guest frame at refresh cycle
  virtio-net: drop too short packets early
  target/i386: fix size of EBP writeback in gen_enter()
* Fri Jun 28 2024 dfaggioli@suse.com
- Update to version 8.2.5:
  Full backport lists (from the various releases) here:
  https://lore.kernel.org/qemu-devel/1718081047.648425.1238605.nullmailer@tls.msk.ru/
  Some of the upstream backports are:
  target/loongarch: fix a wrong print in cpu dump
  ui/sdl2: Allow host to power down screen
  target/i386: fix SSE and SSE2 feature check
  target/i386: fix xsave.flat from kvm-unit-tests
  disas/riscv: Decode all of the pmpcfg and pmpaddr CSRs
  target/riscv/kvm.c: Fix the hart bit setting of AIA
  target/riscv: rvzicbo: Fixup CBO extension register calculation
  target/riscv: do not set mtval2 for non guest-page faults
  target/riscv: prioritize pmp errors in raise_mmu_exception()
  target/riscv: rvv: Remove redudant SEW checking for vector fp narrow/widen instructions
  target/riscv: rvv: Check single width operator for vfncvt.rod.f.f.w
  target/riscv: rvv: Check single width operator for vector fp widen instructions
  target/riscv: rvv: Fix Zvfhmin checking for vfwcvt.f.f.v and vfncvt.f.f.w instructions
  target/riscv/cpu.c: fix Zvkb extension config
  target/riscv: Fix the element agnostic function problem
  target/riscv/kvm: tolerate KVM disable ext errors
  hw/intc/riscv_aplic: APLICs should add child earlier than realize
  iotests: test NBD+TLS+iothread
  qio: Inherit follow_coroutine_ctx across TLS
  target/arm: Disable SVE extensions when SVE is disabled
  hw/intc/arm_gic: Fix handling of NS view of GICC_APR<n>
  hvf: arm: Fix encodings for ID_AA64PFR1_EL1 and debug System registers
  gitlab: use 'setarch -R' to workaround tsan bug
  gitlab: use $MAKE instead of 'make'
  dockerfiles: add 'MAKE' env variable to remaining containers
  gitlab: Update msys2-64bit runner tags
  target/i386: no single-step exception after MOV or POP SS
  ...
* Thu May 16 2024 dfaggioli@suse.com
- Update to version 8.2.4. Full changelog/backports here:
  https://lore.kernel.org/qemu-devel/1715632914.382233.1013785.nullmailer@tls.msk.ru/
  Some of the upstream backports are:
  target/sh4: Fix SUBV opcode
  target/sh4: Fix ADDV opcode
  hw/arm/npcm7xx: Store derivative OTP fuse key in little endian
  hw/dmax/xlnx_dpdma: fix handling of address_extension descriptor fields
  hw/ufs: Fix buffer overflow bug
  tests/avocado: update sunxi kernel from armbian to 6.6.16
  target/loongarch/cpu.c: typo fix: expection
  backends/cryptodev-builtin: Fix local_error leaks
  nbd/server: Mark negotiation functions as coroutine_fn
  nbd/server: do not poll within a coroutine context
  linux-user: do_setsockopt: fix SOL_ALG.ALG_SET_KEY
  target/riscv/kvm: change timer regs size to u64
  target/riscv/kvm: change KVM_REG_RISCV_FP_D to u64
  target/riscv/kvm: change KVM_REG_RISCV_FP_F to u32
  ...
* Wed May  8 2024 dfaggioli@suse.com
- Fixes:
  * [openSUSE][RPM] Prioritize PA over PipeWire in SLE (bsc#1222218)
  * [openSUSE][RPM] Hostname normalization (for repr. build ) done properly (boo#1084909)
* Tue Apr 30 2024 dfaggioli@suse.com
- Fix a build issue on riscv:
  * target/riscv/kvm: rename riscv_reg_id() to riscv_reg_id_ulong()
  * target/riscv/kvm: add RISCV_CONFIG_REG()
  * target/riscv/kvm: change timer regs size to u64
  * target/riscv/kvm: change KVM_REG_RISCV_FP_D to u64
  * target/riscv/kvm: change KVM_REG_RISCV_FP_F to u32
- Update to version 8.2.3. Full changelog/backports here:
  https://lore.kernel.org/qemu-devel/1713980341.971368.1218343.nullmailer@tls.msk.ru/
  Some of the upstream backports are:
  * Update version for 8.2.3 release
  * ppc/spapr: Initialize max_cpus limit to SPAPR_IRQ_NR_IPIS.
  * ppc/spapr: Introduce SPAPR_IRQ_NR_IPIS to refer IRQ range for CPU IPIs.
  * hw/pci-host/ppc440_pcix: Do not expose a bridge device on PCI bus
  * hw/isa/vt82c686: Keep track of PIRQ/PINT pins separately
  * virtio-pci: fix use of a released vector
  * linux-user/x86_64: Handle the vsyscall page in open_self_maps_{2,4}
  * hw/audio/virtio-snd: Remove unused assignment
  * hw/net/net_tx_pkt: Fix overrun in update_sctp_checksum()
  * hw/sd/sdhci: Do not update TRNMOD when Command Inhibit (DAT) is set
  * hw/net/lan9118: Fix overflow in MIL TX FIFO
  * hw/net/lan9118: Replace magic '2048' value by MIL_TXFIFO_SIZE definition
  * backends/cryptodev: Do not abort for invalid session ID
  * hw/misc/applesmc: Fix memory leak in reset() handler
  * hw/block/nand: Fix out-of-bound access in NAND block buffer
  * hw/block/nand: Have blk_load() take unsigned offset and return boolean
  * hw/block/nand: Factor nand_load_iolen() method out
  * qemu-options: Fix CXL Fixed Memory Window interleave-granularity typo
  * hw/virtio/virtio-crypto: Protect from DMA re-entrancy bugs
  * hw/char/virtio-serial-bus: Protect from DMA re-entrancy bugs
  * hw/display/virtio-gpu: Protect from DMA re-entrancy bugs
  * mirror: Don't call job_pause_point() under graph lock (bsc#1224179)
  * ...and many more...
* Thu Apr 18 2024 dfaggioli@suse.com
- Backports and bugfixes:
  * hw/net/net_tx_pkt: Fix overrun in update_sctp_checksum() (bsc#1222841, CVE-2024-3567)
  * hw/virtio/virtio-crypto: Protect from DMA re-entrancy bugs (bsc#1222843, CVE-2024-3446)
  * hw/char/virtio-serial-bus: Protect from DMA re-entrancy bugs (bsc#1222843, CVE-2024-3446)
  * hw/display/virtio-gpu: Protect from DMA re-entrancy bugs (bsc#1222843, CVE-2024-3446)
  * hw/virtio: Introduce virtio_bh_new_guarded() helper (bsc#1222843, CVE-2024-3446)
  * hw/sd/sdhci: Do not update TRNMOD when Command Inhibit (DAT) is set (bsc#1222845, CVE-2024-3447)
  * hw/nvme: Use pcie_sriov_num_vfs() (bsc#1220065, CVE-2024-26328)
* Fri Mar 15 2024 dfaggioli@suse.com
- Update to version 8.2.2. Full changelog here:
  https://lore.kernel.org/qemu-devel/1709577077.783602.1474596.nullmailer@tls.msk.ru/
  Some upstream backports:
  * chardev/char-socket: Fix TLS io channels sending too much data to the backend
  * tests/unit/test-util-sockets: Remove temporary file after test
  * hw/usb/bus.c: PCAP adding 0xA in Windows version
  * hw/intc/Kconfig: Fix GIC settings when using "--without-default-devices"
  * gitlab: force allow use of pip in Cirrus jobs
  * tests/vm: avoid re-building the VM images all the time
  * tests/vm: update openbsd image to 7.4
  * target/i386: leave the A20 bit set in the final NPT walk
  * target/i386: remove unnecessary/wrong application of the A20 mask
  * target/i386: Fix physical address truncation
  * target/i386: check validity of VMCB addresses
  * target/i386: mask high bits of CR3 in 32-bit mode
  * pl031: Update last RTCLR value on write in case it's read back
  * hw/nvme: fix invalid endian conversion
  * update edk2 binaries to edk2-stable202402
  * update edk2 submodule to edk2-stable202402
  * target/ppc: Fix crash on machine check caused by ifetch
  * target/ppc: Fix lxv/stxv MSR facility check
  * .gitlab-ci.d/windows.yml: Drop msys2-32bit job
  * system/vl: Update description for input grab key
  * docs/system: Update description for input grab key
  * hw/hppa/Kconfig: Fix building with "configure --without-default-devices"
  * tests/qtest: Depend on dbus_display1_dep
  * meson: Explicitly specify dbus-display1.h dependency
  * audio: Depend on dbus_display1_dep
  * ui/console: Fix console resize with placeholder surface
  * ui/clipboard: add asserts for update and request
  * ui/clipboard: mark type as not available when there is no data
  * ui: reject extended clipboard message if not activated
  * target/i386: Generate an illegal opcode exception on cmp instructions with lock prefix
  * i386/cpuid: Move leaf 7 to correct group
  * i386/cpuid: Decrease cpuid_i when skipping CPUID leaf 1F
  * i386/cpu: Mask with XCR0/XSS mask for FEAT_XSAVE_XCR0_HI and FEAT_XSAVE_XSS_HI leafs
  * i386/cpu: Clear FEAT_XSAVE_XSS_LO/HI leafs when CPUID_EXT_XSAVE is not available
  * .gitlab-ci/windows.yml: Don't install libusb or spice packages on 32-bit
  * iotests: Make 144 deterministic again
  * target/arm: Don't get MDCR_EL2 in pmu_counter_enabled() before checking ARM_FEATURE_PMU
  * target/arm: Fix SVE/SME gross MTE suppression checks
  * target/arm: Handle mte in do_ldrq, do_ldro
- Address bsc#1220310. Backported upstream commits:
  * ppc/spapr: Initialize max_cpus limit to SPAPR_IRQ_NR_IPIS
  * ppc/spapr: Introduce SPAPR_IRQ_NR_IPIS to refer IRQ range for CPU IPIs.
* Mon Mar  4 2024 dfaggioli@suse.com
- Fix bsc#1220799. Amended commit:
  * [openSUSE]: Increase default phys bits to 42, if host supports that
    (bsc#1205978, bsc#1219977, bsc#1220799)
* Thu Feb 29 2024 dfaggioli@suse.com
- Backports and bugfixes:
  * [openSUSE]: Increase default phys bits to 42, if host supports that
    (bsc#1205978, bsc#1219977)
  * vfio/pci: Clear MSI-X IRQ index always (bsc#1220275)
* Wed Feb 21 2024 dfaggioli@suse.com
- Just "prettify" the spec files a little:
  * [openSUSE][RPM] Cosmetic fixes to spec files (copyright, sorting, etc)
* Wed Feb 21 2024 dfaggioli@suse.com
- Patchqueue shrinking and  bugfixing (actually, more of a temporary
  workaround, until a proper solution is found upstream):
  * [openSUSE] roms/seabios: revert some upstream commits that
    break a lot of use-cases
  * [openSUSE] roms/seabios: Drop an old (and no longer necessary)
    downstream patch (bsc#1219977)
* Mon Feb 19 2024 dfaggioli@suse.com
  Update to latest stable version (8.2.1)
- Downstream changes:
  * [openSUSE][RPM]: Install the VGA module "more often" (bsc#1219164)
  * [openSUSE][RPM] Fix handling of qemu-kvm legacy package for RISCV
  * [openSUSE][RPM] factor common definitions between qemu and qemu-linux-user spec files
- Upstream backports:
  * target/arm: Fix incorrect aa64_tidcp1 feature check
  * target/arm: Fix A64 scalar SQSHRN and SQRSHRN
  * target/xtensa: fix OOB TLB entry access
  * qtest: bump aspeed_smc-test timeout to 6 minutes
  * monitor: only run coroutine commands in qemu_aio_context
  * iotests: port 141 to Python for reliable QMP testing
  * iotests: add filter_qmp_generated_node_ids()
  * block/blklogwrites: Fix a bug when logging "write zeroes" operations.
  * virtio-net: correctly copy vnet header when flushing TX (bsc#1218484, CVE-2023-6693)
  * tcg/arm: Fix SIGILL in tcg_out_qemu_st_direct
  * linux-user/riscv: Adjust vdso signal frame cfa offsets
  * linux-user: Fixed cpu restore with pc 0 on SIGBUS
  * block/io: clear BDRV_BLOCK_RECURSE flag after recursing in bdrv_co_block_status
  * coroutine-ucontext: Save fake stack for pooled coroutine
  * tcg/s390x: Fix encoding of VRIc, VRSa, VRSc insns
  * accel/tcg: Revert mapping of PCREL translation block to multiple virtual addresses
  * acpi/tests/avocado/bits: wait for 200 seconds for SHUTDOWN event from bits VM
  * s390x/pci: drive ISM reset from subsystem reset
  * s390x/pci: refresh fh before disabling aif
  * s390x/pci: avoid double enable/disable of aif
  * hw/scsi/esp-pci: set DMA_STAT_BCMBLT when BLAST command issued
  * hw/scsi/esp-pci: synchronise setting of DMA_STAT_DONE with ESP completion interrupt
  * hw/scsi/esp-pci: generate PCI interrupt from separate ESP and PCI sources
  * hw/scsi/esp-pci: use correct address register for PCI DMA transfers
  * migration/rdma: define htonll/ntohll only if not predefined
  * hw/pflash: implement update buffer for block writes
  * hw/pflash: use ldn_{be,le}_p and stn_{be,le}_p
  * hw/pflash: refactor pflash_data_write()
  * backends/cryptodev: Do not ignore throttle/backends Errors
  * target/i386: pcrel: store low bits of physical address in data[0]
  * target/i386: fix incorrect EIP in PC-relative translation blocks
  * target/i386: Do not re-compute new pc with CF_PCREL
  * load_elf: fix iterator's type for elf file processing
  * target/hppa: Update SeaBIOS-hppa to version 15
  * target/hppa: Fix IOR and ISR on error in probe
  * target/hppa: Fix IOR and ISR on unaligned access trap
  * target/hppa: Export function hppa_set_ior_and_isr()
  * target/hppa: Avoid accessing %%gr0 when raising exception
  * hw/hppa: Move software power button address back into PDC
  * target/hppa: Fix PDC address translation on PA2.0 with PSW.W=0
  * hw/pci-host/astro: Add missing astro & elroy registers for NetBSD
  * hw/hppa/machine: Disable default devices with --nodefaults option
  * hw/hppa/machine: Allow up to 3840 MB total memory
  * readthodocs: fully specify a build environment
  * .gitlab-ci.d/buildtest.yml: Work around htags bug when environment is large
  * target/s390x: Fix LAE setting a wrong access register
  * tests/qtest/virtio-ccw: Fix device presence checking
  * tests/acpi: disallow tests/data/acpi/virt/SSDT.memhp changes
  * tests/acpi: update expected data files
  * edk2: update binaries to git snapshot
  * edk2: update build config, set PcdUninstallMemAttrProtocol = TRUE.
  * edk2: update to git snapshot
  * tests/acpi: allow tests/data/acpi/virt/SSDT.memhp changes
  * util: fix build with musl libc on ppc64le
  * tcg/ppc: Use new registers for LQ destination
  * hw/intc/arm_gicv3_cpuif: handle LPIs in in the list registers
  * hw/vfio: fix iteration over global VFIODevice list
  * vfio/container: Replace basename with g_path_get_basename
  * edu: fix DMA range upper bound check
  * hw/net: cadence_gem: Fix MDIO_OP_xxx values
  * audio/audio.c: remove trailing newline in error_setg
  * chardev/char.c: fix "abstract device type" error message
  * target/riscv: Fix mcycle/minstret increment behavior
  * hw/net/can/sja1000: fix bug for single acceptance filter and standard frame
  * target/i386: the sgx_epc_get_section stub is reachable
  * configure: use a native non-cross compiler for linux-user
  * include/ui/rect.h: fix qemu_rect_init() mis-assignment
  * target/riscv/kvm: do not use non-portable strerrorname_np()
  * iotests: Basic tests for internal snapshots
  * vl: Improve error message for conflicting -incoming and -loadvm
  * block: Fix crash when loading snapshot on inactive node
- Fixes:
  * bsc#1218484 (CVE-2023-6693)
* Sat Feb 17 2024 dfaggioli@suse.com
- Try to solve the qemu-kvm dependency issues on all arches
  (see, e.g., bsc#1218684)
  * [openSUSE][RPM] Create the legacy qemu-kvm symlink for all arches
* Fri Feb 16 2024 dfaggioli@suse.com
- Update the service file to use OBS-scm (by fvogt)
- Various fixes:
  * [openSUSE][RPM] Fix enabling features on non-x86_64 (bsc#1220011, bsc#1219818)
  * [openSUSE][RPM] Disable test-crypto-secret in linux-user build
  * [openSUSE] Update ipxe submodule reference (bsc#1219733, bsc#1219722)
  * [openSUSE][RPM] spec: allow building without spice
* Tue Feb  6 2024 dfaggioli@suse.com
- Fix a build issue of OVMF caused by
  https://gitlab.com/qemu-project/qemu/-/issues/2064:
  * target/i386: fix incorrect EIP in PC-relative translation blocks
  * target/i386: Do not re-compute new pc with CF_PCREL
* Fri Jan 12 2024 dfaggioli@suse.com
- Update to latest upstream release, 8.2.0:
  The full list of changes are available at: https://wiki.qemu.org/ChangeLog/8.2
  Highlights include:
  * New virtio-sound device emulation
  * New virtio-gpu rutabaga device emulation used by Android emulator
  * New hv-balloon for dynamic memory protocol device for Hyper-V guests
  * New Universal Flash Storage device emulation
  * Network Block Device (NBD) 64-bit offsets for improved performance
  * dump-guest-memory now supports the standard kdump format
  * ARM: Xilinx Versal board now models the CFU/CFI, and the TRNG device
  * ARM: CPU emulation support for cortex-a710 and neoverse-n2
  * ARM: architectural feature support for PACQARMA3, EPAC, Pauth2, FPAC,
    FPACCOMBINE, TIDCP1, MOPS, HBC, and HPMN0
  * HPPA: CPU emulation support for 64-bit PA-RISC 2.0
  * HPPA: machine emulation support for C3700, including Astro memory
    controller and four Elroy PCI bridges
  * LoongArch: ISA support for LASX extension and PRELDX instruction
  * LoongArch: CPU emulation support for la132
  * RISC-V: ISA/extension support for AIA virtualization support via KVM,
    and vector cryptographic instructions
  * RISC-V: Numerous extension/instruction cleanups, fixes, and reworks
  * s390x: support for vfio-ap passthrough of crypto adapter for
    protected
    virtualization guests
  * Tricore: support for TC37x CPU which implements ISA v1.6.2
  * Tricore: support for CRCN, FTOU, FTOHP, and HPTOF instructions
  * x86: Zen support for PV console and network devices
- Patch added (from upstream stable tree):
  * include/ui/rect.h: fix qemu_rect_init() mis-assignment
* Wed Jan 10 2024 dfaggioli@suse.com
- Some packaging and dependencies fixes:
  * [openSUSE] rpm: restrict canokey to openSUSE only
  * [openSUSE] rpm: fix virtiofsd dependency on 32 bit systems
  * [openSUSE] rpm: add support for canokeys (boo#1217520)
* Fri Dec 15 2023 dfaggioli@suse.com
- Rearrange dependencies and subpackages and filter features for ALP
  * [openSUSE] rpm: disable Xen support in ALP-based distros
  * [openSUSE] rpm: some more refinements of inter-subpackage dependencies
* Fri Dec 15 2023 dfaggioli@suse.com
- Fix boo#1084909 and create a new qemu-spice metapackage:
  * [openSUSE] rpm: normalize hostname, for reproducible builds (#44)
  * [openSUSE] rpm: new subpackage, for SPICE
* Wed Nov 29 2023 dfaggioli@suse.com
- Align to upstream stable release. It includes many of the patches we had
  backported ourself, to fix bugs and issues, plus more. See here for details:
  * https://lore.kernel.org/qemu-devel/1700589639.257680.3420728.nullmailer@tls.msk.ru/
  * https://gitlab.com/qemu-project/qemu/-/commits/stable-8.1?ref_type=heads
  An (incomplete!) list of such backports is:
  * Update version for 8.1.3 release
  * hw/mips: LOONGSON3V depends on UNIMP device
  * target/arm: HVC at EL3 should go to EL3, not EL2
  * s390x/pci: only limit DMA aperture if vfio DMA limit reported
  * target/riscv/kvm: support KVM_GET_REG_LIST
  * target/riscv/kvm: improve 'init_multiext_cfg' error msg
  * tracetool: avoid invalid escape in Python string
  * tests/tcg/s390x: Test LAALG with negative cc_src
  * target/s390x: Fix LAALG not updating cc_src
  * tests/tcg/s390x: Test CLC with inaccessible second operand
  * target/s390x: Fix CLC corrupting cc_src
  * tests/qtest: ahci-test: add test exposing reset issue with pending callback
  * hw/ide: reset: cancel async DMA operation before resetting state
  * target/mips: Fix TX79 LQ/SQ opcodes
  * target/mips: Fix MSA BZ/BNZ opcodes displacement
  * ui/gtk-egl: apply scale factor when calculating window's dimension
  * ui/gtk: force realization of drawing area
  * ati-vga: Implement fallback for pixman routines
  * ...
* Wed Nov 22 2023 dfaggioli@suse.com
- Backports and bugfixes:
  * [openSUSE] Make Sphinx build reproducible (boo#1102408)
  * target/s390x/arch_dump: Add arch cleanup function for PV dumps (bsc#1217227)
  * dump: Add arch cleanup function (bsc#1217227)
  * target/s390x/dump: Remove unneeded dump info function pointer init (bsc#1217227)
* Wed Nov 15 2023 dfaggioli@suse.com
- Fix bsc#1216638:
  * target/s390x: Fix LAALG not updating cc_src
  * target/s390x: Fix CLC corrupting cc_src
* Tue Oct 24 2023 dfaggioli@suse.com
- [openSUSE] supportconfig: Adapt plugin to modern supportconfig
* Wed Oct 18 2023 dfaggioli@suse.com
- [openSUSE] Add -p1 to autosetup in spec files
  * So patches can be applied easily (might be useful for development)
* Wed Oct 18 2023 dfaggioli@suse.com
- Update to version 8.1.2:
  This fixes the following upstream issues:
  * https://gitlab.com/qemu-project/qemu/-/issues/1826
  * https://gitlab.com/qemu-project/qemu/-/issues/1834
  * https://gitlab.com/qemu-project/qemu/-/issues/1846
  It also contains a fix for:
  * CVE-2023-42467 (bsc#1215192)
  As well as several upstream backports (list available at:
  https://lore.kernel.org/qemu-devel/24dfb1c2-dc40-4841-893d-5d530638b684@tls.msk.ru/)
* Wed Oct 11 2023 dfaggioli@suse.com
- Update to version 8.1.1:
  Upstream backports and bugfixes, among which:
  * tpm: fix crash when FD >= 1024 and unnecessary errors due to EINTR (Marc-Andr303251 Lureau)
  * meson: Fix targetos match for illumos and Solaris. (Jonathan Perkin)
  * s390x/ap: fix missing subsystem reset registration (Janosch Frank)
  * ui: fix crash when there are no active_console (Marc-Andr303251 Lureau)
  * virtio-gpu/win32: set the destroy function on load (Marc-Andr303251 Lureau)
  * target/riscv: Allocate itrigger timers only once (Akihiko Odaki)
  * target/riscv/pmp.c: respect mseccfg.RLB for pmpaddrX changes (Leon Schuermann)
  * target/riscv: fix satp_mode_finalize() when satp_mode.supported = 0 (Daniel Henrique Barboza)
  * hw/riscv: virt: Fix riscv,pmu DT node path (Conor Dooley)
  * [...]
  * target/arm: Fix 64-bit SSRA (Richard Henderson)
  * target/arm: Fix SME ST1Q (Richard Henderson)
  * accel/kvm: Specify default IPA size for arm64 (Akihiko Odaki)
  * kvm: Introduce kvm_arch_get_default_type hook (Akihiko Odaki)
  * include/hw/virtio/virtio-gpu: Fix virtio-gpu with blob on big endian hosts (Thomas Huth)
  * target/s390x: Check reserved bits of VFMIN/VFMAX's M5 (Ilya Leoshkevich)
  * target/s390x: Fix VSTL with a large length (Ilya Leoshkevich)
  * target/s390x: Use a 16-bit immediate in VREP (Ilya Leoshkevich)
  * target/s390x: Fix the "ignored match" case in VSTRS (Ilya Leoshkevich)
  Full list available at:
  * https://lore.kernel.org/qemu-devel/d0d2ac8f-313a-c937-dc3a-88e45fce933c@tls.msk.ru/
* Fri Oct  6 2023 dfaggioli@suse.com
- Fix (bsc#1215486):
  * [openSUSE][RPM] spec: enable the Pipewire audio backend (bsc#1215486)
* Thu Oct  5 2023 dfaggioli@suse.com
- (Properly) fix: bsc#1213210
  * linux-user/elfload: Enable vxe2 on s390x
- Fix: bsc#1215677
  * optionrom: Remove build-id section
* Thu Sep 14 2023 dfaggioli@suse.com
- Fix bsc#1215311:
  * roms/ipxe: Backport 0aa2e4ec9635, in preparation of binutils 2.41 (bsc#1215311)
* Thu Sep 14 2023 dfaggioli@suse.com
- Fix the build for SLE/Leap:
  * [openSUSE][RPM] Make the package buildable on SLE/Leap 15.x
* Tue Sep 12 2023 dfaggioli@suse.com
- Fix bsc#1211000:
  * [openSUSE] block: Add a thread-pool version of fstat (bsc#1211000)
  * [openSUSE] block: Convert qmp_query_block() to coroutine_fn (bsc#1211000)
  * [openSUSE] block: Don't query all block devices at hmp_nbd_server_start (bsc#1211000)
  * [openSUSE] block: Convert qmp_query_named_block_nodes to coroutine (bsc#1211000)
  * [openSUSE] block: Convert bdrv_block_device_info into co_wrapper (bsc#1211000)
  * [openSUSE] block: Convert bdrv_query_block_graph_info to coroutine (bsc#1211000)
  * [openSUSE] block: Temporarily mark bdrv_co_get_allocated_file_size as mixed (bsc#1211000)
  * [openSUSE] block: Allow the wrapper script to see functions declared in qapi.h (bsc#1211000)
  * [openSUSE] block: Remove unnecessary variable in bdrv_block_device_info (bsc#1211000)
  * [openSUSE] block: Remove bdrv_query_block_node_info (bsc#1211000)
- Fix bsc#1213210:
  * target/s390x: Fix the "ignored match" case in VSTRS (bsc#1213210)
* Tue Sep 12 2023 dfaggioli@suse.com
- Update to version 8.1.0. Full list of changes are available at:
  https://wiki.qemu.org/ChangeLog/8.1
  Highlights:
  * VFIO: improved live migration support, no longer an experimental feature
  * GTK GUI now supports multi-touch events
  * ARM, PowerPC, and RISC-V can now use AES acceleration on host processor
  * PCIe: new QMP commands to inject CXL General Media events, DRAM
    events and Memory Module events
  * ARM: KVM VMs on a host which supports MTE (the Memory Tagging Extension)
    can now use MTE in the guest
  * ARM: emulation support for bpim2u (Banana Pi BPI-M2 Ultra) board and
    neoverse-v1 (Cortex Neoverse-V1) CPU
  * ARM: new architectural feature support for: FEAT_PAN3 (SCTLR_ELx.EPAN),
    FEAT_LSE2 (Large System Extensions v2), and experimental support for
    FEAT_RME (Realm Management Extensions)
  * Hexagon: new instruction support for v68/v73 scalar, and v68/v69 HVX
  * Hexagon: gdbstub support for HVX
  * MIPS: emulation support for Ingenic XBurstR1/XBurstR2 CPUs, and MXU
    instructions
  * PowerPC: TCG SMT support, allowing pseries and powernv to run with up
    to 8 threads per core
  * PowerPC: emulation support for Power9 DD2.2 CPU model, and perf
    sampling support for POWER CPUs
  * RISC-V: ISA extension support for BF16/Zfa, and disassembly support
    for Zcm*/Z*inx/XVentanaCondOps/Xthead
  * RISC-V: CPU emulation support for Veyron V1
  * RISC-V: numerous KVM/emulation fixes and enhancements
  * s390: instruction emulation fixes for LDER, LCBB, LOCFHR, MXDB, MXDBR,
    EPSW, MDEB, MDEBR, MVCRL, LRA, CKSM, CLM, ICM, MC, STIDP, EXECUTE, and
    CLGEBR(A)
  * SPARC: updated target/sparc to use tcg_gen_lookup_and_goto_ptr() for
    improved performance
  * Tricore: emulation support for TC37x CPU that supports ISA v1.6.2
    instructions
  * Tricore: instruction emulation of POPCNT.W, LHA, CRC32L.W, CRC32.B,
    SHUFFLE, SYSCALL, and DISABLE
  * x86: CPU model support for GraniteRapids
  * and lots more...
- This also (automatically) fixes:
  * bsc#1212850 (CVE-2023-3354)
  * bsc#1213001 (CVE-2023-3255)
  * bsc#1213925 (CVE-2023-3180)
  * bsc#1213414 (CVE-2023-3301)
  * bsc#1207205 (CVE-2023-0330)
  * bsc#1212968 (CVE-2023-2861)
  * bsc#1179993, bsc#1181740, bsc#1211697
* Mon Aug  7 2023 dfaggioli@suse.com
- perl-Text-Markdown is not available in all distros and for all
  arch-es. Use discount instead
- Patches added:
  * [openSUSE][spec] Use discount instead of perl-Text-Markdown
* Mon Aug  7 2023 dfaggioli@suse.com
- Update to version 8.0.4:
  * Official changelog not released on the mailing list yet
  * Security issues fixed:
  - bsc#1212850 (CVE-2023-3354)
  - bsc#1213001 (CVE-2023-3255)
  - bsc#1213925 (CVE-2023-3180)
  - bsc#1207205 (CVE-2023-0330)
* Thu Jul 27 2023 dfaggioli@suse.com
- Fix bsc#1179993, bsc#1181740, bsc#1213001
- Patches added:
  * hw/ide/piix: properly initialize the BMIBA register
  * ui/vnc-clipboard: fix infinite loop in inflate_buffer (CVE-2023-3255)
* Wed Jul 12 2023 dfaggioli@suse.com
- Update to version 8.0.3:
  * See full log: https://lists.nongnu.org/archive/html/qemu-stable/2023-07/msg00086.html
  * Security issues fixed:
  - 9pfs: prevent opening special files (CVE-2023-2861)
  - vhost-vdpa (CVE-2023-3301)
  * Use the official xkb name for Arabic layout, not the
    legacy synonym (bsc#1212966)
  * [openSUSE][RPM] Update to version 8.0.3
* Fri Jun 30 2023 dfaggioli@suse.com
- Patches added (first one is relevant for boo#1197298 and bsc#1212768):
  * [openSUSE][RPM] Use --preserve-argv0 in qemu-linux-user (#32)
  * [openSUSE][RPM] Split qemu-tools package (#31)
* Thu Jun  1 2023 dfaggioli@suse.com
- Update to version 8.0.2:
  * Stability, security and bug fixes
- Patch added:
  * [openSUSE][RPM] Update to version 8.0.2
* Mon May 29 2023 dfaggioli@suse.com
- Patch added:
  [openSUSE][RPM] Fix deps for virtiofsd and improve spec files
* Tue May 23 2023 dfaggioli@suse.com
- Update the _constraints file:
  * the qemu-testsuite package does not exist any longer, but some
    of the tests are done in the qemu package (so "transfer" some of
    the constraints to that one)
  - some of the builds are failing with OOM, happening while the RPM
    is actually put together, at the end of the process. Try to give
    them more RAM
* Mon May 22 2023 dfaggioli@suse.com
- Patch added:
  [openSUSE][RPM] spec: require virtiofsd, now that it is a sep package (#27)
* Mon May 22 2023 dfaggioli@suse.com
- Update to version 8.0.0 (https://wiki.qemu.org/ChangeLog/8.0)
  * Removed features: https://qemu-project.gitlab.io/qemu/about/removed-features.html
  * Deprecated features: https://qemu-project.gitlab.io/qemu/about/deprecated.html
  * Some notable changes:
  - ARM:
  - New emulated CPU types:
  - Cortex-A55 CPU
  - Cortex-R52 CPU
  - x86
  - Add support for Xen guests under KVM with Linux v5.12+
  - New CPU model "SapphireRapids"
  - VFIO
  - Experimental migration support has been updated to the v2 VFIO migration protocol
  - virtio
  - virtio-mem now fully supports combining preallocation with migration
  - vDPA
  - Support live migration of vhost-vdpa net devices without CVQ, with no need of x-svq
  - virtiofs
  - The old C virtiofsd has been removed, use the new Rust implementation instead.
  * Patches added:
  [openSUSE][RPM] Try to avoid recommending too many packages (bsc#1205680)
  [openSUSE][RPM] Move documentation to a subpackage and fix qemu-headless (bsc#1209629)
  roms: add back edk2-basetools target
  async: Suppress GCC13 false positive in aio_bh_poll()
  [openSUSE][OBS] Limit the workflow runs to the factory branch (#25)
  [openSUSE][RPM] Spec file adjustments for 8.0.0
* Wed May 10 2023 dfaggioli@suse.com
- (Radical!) Change of packaging workflow. Now pretty much everything
  happens via git, and interacting with https://github.com/openSUSE/qemu.git.
  See README.PACKAGING for details
  * Patches added:
  linux-user: Add pidfd_open(), pidfd_send_signal() and pidfd_getfd() syscalls
  linux-user: fill out task state in /proc/self/stat
  linux-user: Emulate CLONE_PIDFD flag in clone()
  * Patches transformed in git commits:
  Disable-some-tests-that-have-problems-in.patch
  Make-char-muxer-more-robust-wrt-small-FI.patch
  Make-installed-scripts-explicitly-python.patch
  Makefile-fix-build-with-binutils-2.38.patch
  PPC-KVM-Disable-mmu-notifier-check.patch
  Raise-soft-address-space-limit-to-hard-l.patch
  Revert-linux-user-fix-compat-with-glibc-.patch
  Revert-roms-efirom-tests-uefi-test-tools.patch
  Revert-tests-qtest-enable-more-vhost-use.patch
  Update-linux-headers-to-v6.0-rc4.patch
  accel-abort-if-we-fail-to-load-the-accel.patch
  acpi-cpuhp-fix-guest-visible-maximum-acc.patch
  ath5k-Add-missing-AR5K_EEPROM_READ-in-at.patch
  bios-tables-test-add-test-for-number-of-.patch
  bios-tables-test-teach-test-to-use-smbio.patch
  block-Handle-curl-7.55.0-7.85.0-version-.patch
  block-io_uring-revert-Use-io_uring_regis.patch
  configure-Add-Wno-gnu-variable-sized-typ.patch
  dmg-warn-when-opening-dmg-images-contain.patch
  dump-Add-architecture-section-and-sectio.patch
  dump-Refactor-dump_iterate-and-introduce.patch
  dump-Reintroduce-memory_offset-and-secti.patch
  dump-Rename-write_elf-_phdr_note-to-prep.patch
  dump-Rename-write_elf_loads-to-write_elf.patch
  dump-Reorder-struct-DumpState.patch
  dump-Replace-opaque-DumpState-pointer-wi.patch
  dump-Rework-dump_calculate_size-function.patch
  dump-Rework-filter-area-variables.patch
  dump-Rework-get_start_block.patch
  dump-Split-elf-header-functions-into-pre.patch
  dump-Use-a-buffer-for-ELF-section-data-a.patch
  dump-Write-ELF-section-headers-right-aft.patch
  hw-acpi-erst.c-Fix-memory-handling-issue.patch
  hw-display-qxl-Avoid-buffer-overrun-in-q.patch
  hw-display-qxl-Document-qxl_phys2virt.patch
  hw-display-qxl-Have-qxl_log_command-Retu.patch
  hw-display-qxl-Pass-requested-buffer-siz.patch
  hw-pvrdma-Protect-against-buggy-or-malic.patch
  hw-scsi-megasas-check-for-NULL-frame-in-.patch
  hw-smbios-add-core_count2-to-smbios-tabl.patch
  hw-smbios-handle-both-file-formats-regar.patch
  hw-smbios-support-for-type-8-port-connec.patch
  include-elf.h-add-s390x-note-types.patch
  increase-x86_64-physical-bits-to-42.patch
  linux-user-Fake-proc-cpuinfo.patch
  linux-user-lseek-explicitly-cast-non-set.patch
  linux-user-remove-conditionals-for-many-.patch
  linux-user-use-max-as-default-CPU-model-.patch
  linux-user-use-target_ulong.patch
  meson-install-ivshmem-client-and-ivshmem.patch
  meson-remove-pkgversion-from-CONFIG_STAM.patch
  module-add-Error-arguments-to-module_loa.patch
  module-removed-unused-function-argument-.patch
  module-rename-module_load_one-to-module_.patch
  net-tulip-Restrict-DMA-engine-to-memorie.patch
  openSUSE-Basetools-Ignore-spurious-GCC-1.patch
  openSUSE-Makefile-Fix-csum8-to-be-built-.patch
  openSUSE-Makefile-define-endianess-for-c.patch
  openSUSE-Makefile-fix-issues-of-build-re.patch
  openSUSE-add-cross.ini-file-to-handle-aa.patch
  openSUSE-build-Makefile-fix-issues-of-bu.patch
  openSUSE-build-Silence-GCC-12-spurious-w.patch
  openSUSE-build-be-explicit-about-mx86-us.patch
  openSUSE-build-enable-cross-compilation-.patch
  openSUSE-pc-q35-Bump-max_cpus-to-1024.patch
  openSUSE-pcbios-stub-out-the-SAN-req-s-i.patch
  openSUSE-switch-to-python3-as-needed.patch
  openSUSE-test-help-compiler-out-by-initi.patch
  qemu-binfmt-conf-Modify-default-path.patch
  qemu-bridge-helper-reduce-security-profi.patch
  qemu-osdep-Switch-position-of-extern-and.patch
  roms-Makefile-add-cross-file-to-qboot-me.patch
  roms-Makefile-pass-a-packaging-timestamp.patch
  roms-change-cross-compiler-naming-to-be-.patch
  s390x-Add-KVM-PV-dump-interface.patch
  s390x-Add-protected-dump-cap.patch
  s390x-Introduce-PV-query-interface.patch
  s390x-pci-add-routine-to-get-host-functi.patch
  s390x-pci-don-t-fence-interpreted-device.patch
  s390x-pci-enable-adapter-event-notificat.patch
  s390x-pci-enable-for-load-store-interpre.patch
  s390x-pci-let-intercept-devices-have-sep.patch
  s390x-pci-reflect-proper-maxstbl-for-gro.patch
  s390x-pci-reset-ISM-passthrough-devices-.patch
  s390x-pci-shrink-DMA-aperture-to-be-boun.patch
  s390x-pv-Add-dump-support.patch
  s390x-tod-kvm-don-t-save-restore-the-TOD.patch
  scsi-generic-check-for-additional-SG_IO-.patch
  scsi-generic-replace-logical-block-count.patch
  test-vmstate-fix-bad-GTree-usage-use-aft.patch
  tests-acpi-allow-changes-for-core_count2.patch
  tests-acpi-update-tables-for-new-core-co.patch
  tests-change-error-message-in-test-162.patch
  tests-qemu-iotests-Triple-timeout-of-i-o.patch
  ui-vnc-clipboard-fix-integer-underflow-i.patch
  xen-add-block-resize-support-for-xen-dis.patch
  xen-ignore-live-parameter-from-xen-save-.patch
  xen_disk-Add-suse-specific-flush-disable.patch
* Sat Mar 25 2023 mliska@suse.cz
- Enable again LTO for x86_64 target (boo#1133281).
* Wed Mar 22 2023 dfaggioli@suse.com
- Further fixes for bsc#1209546
  * Patches added:
  test-vmstate-fix-bad-GTree-usage-use-aft.patch
* Tue Mar 21 2023 dfaggioli@suse.com
- Fix bsc#1209546
  * Patches added:
  qemu-osdep-Switch-position-of-extern-and.patch
* Thu Mar 16 2023 dfaggioli@suse.com
- Fix bsc#1209064
- Backport the "acpi: cpuhp: fix guest-visible maximum access size
  to the legacy reg block" patch, as it makes developing and
  testing OVMF/EDK2 easier
  * Patches added:
  acpi-cpuhp-fix-guest-visible-maximum-acc.patch
  s390x-pci-reset-ISM-passthrough-devices-.patch
  s390x-pci-shrink-DMA-aperture-to-be-boun.patch
* Thu Mar  9 2023 mliska@suse.cz
- Disable -Werror as it is very sensitive when one
  updates a new compiler. -Werror is fine for upstream development,
  but not when it comes to stability of a package build.
* Fri Feb 24 2023 dfaggioli@suse.com
- Fix build issue with Linux 6.2's headers (bsc#1208657) by dropping
  linux-user-add-more-compat-ioctl-definit.patch and adding
  Revert-linux-user-fix-compat-with-glibc-.patch
- Patches meson-enforce-a-minimum-Linux-kernel-hea.patch and
  linux-user-drop-conditionals-for-obsolet.patch were added
  as downstream patches as they were part of a series, but
  they never made it upstream, so we don't want them here
  either
  * Patches dropped:
  linux-user-add-more-compat-ioctl-definit.patch
  linux-user-drop-conditionals-for-obsolet.patch
  meson-enforce-a-minimum-Linux-kernel-hea.patch
  * Patches added:
  Revert-linux-user-fix-compat-with-glibc-.patch
* Thu Feb 23 2023 dfaggioli@suse.com
- Fixes bsc#1197653, CVE-2022-1050
  * Patches added:
  block-Handle-curl-7.55.0-7.85.0-version-.patch
  hw-pvrdma-Protect-against-buggy-or-malic.patch
* Mon Feb 13 2023 li.zhang@suse.com
- Fixes: jsc#PED-1716  Add S390 features from IBM requirements
  * Patches added:
  dump-Add-architecture-section-and-sectio.patch
  dump-Refactor-dump_iterate-and-introduce.patch
  dump-Reintroduce-memory_offset-and-secti.patch
  dump-Rename-write_elf_loads-to-write_elf.patch
  dump-Rename-write_elf-_phdr_note-to-prep.patch
  dump-Reorder-struct-DumpState.patch
  dump-Replace-opaque-DumpState-pointer-wi.patch
  dump-Rework-dump_calculate_size-function.patch
  dump-Rework-filter-area-variables.patch
  dump-Rework-get_start_block.patch
  dump-Split-elf-header-functions-into-pre.patch
  dump-Use-a-buffer-for-ELF-section-data-a.patch
  dump-Write-ELF-section-headers-right-aft.patch
  include-elf.h-add-s390x-note-types.patch
  s390x-Add-KVM-PV-dump-interface.patch
  s390x-Add-protected-dump-cap.patch
  s390x-Introduce-PV-query-interface.patch
  s390x-pv-Add-dump-support.patch
* Fri Feb 10 2023 dfaggioli@suse.com
- Fixed: bsc#1205847 (CVE-2022-4172), bsc#1203788 (CVE-2022-3165),
  bsc#1205808 (CVE-2022-4144), bsc#1206527, bsc#1208139
- Improved handling of: bsc#1202282 (jsc#PED-2592)
  * Patches dropped:
  pc-q35-Bump-max_cpus-to-1024.patch
  * Patches added:
  accel-abort-if-we-fail-to-load-the-accel.patch
  bios-tables-test-add-test-for-number-of-.patch
  bios-tables-test-teach-test-to-use-smbio.patch
  dmg-warn-when-opening-dmg-images-contain.patch
  hw-acpi-erst.c-Fix-memory-handling-issue.patch
  hw-display-qxl-Avoid-buffer-overrun-in-q.patch
  hw-display-qxl-Document-qxl_phys2virt.patch
  hw-display-qxl-Have-qxl_log_command-Retu.patch
  hw-display-qxl-Pass-requested-buffer-siz.patch
  hw-smbios-add-core_count2-to-smbios-tabl.patch
  hw-smbios-support-for-type-8-port-connec.patch
  module-add-Error-arguments-to-module_loa.patch
  module-removed-unused-function-argument-.patch
  module-rename-module_load_one-to-module_.patch
  openSUSE-pc-q35-Bump-max_cpus-to-1024.patch
  s390x-tod-kvm-don-t-save-restore-the-TOD.patch
  tests-acpi-allow-changes-for-core_count2.patch
  tests-acpi-update-tables-for-new-core-co.patch
  ui-vnc-clipboard-fix-integer-underflow-i.patch
* Fri Jan 27 2023 dfaggioli@suse.com
- Refactor building and installing SeaBIOS docs
* Fri Jan 27 2023 dfaggioli@suse.com
- Rename submodule patches so that it's clear which ones
  are backports and which ones are downstream only fixes;
- No functional change intended.
  * Patches dropped:
  Ignore-spurious-GCC-12-warning.patch
  roms-sgabios-Fix-csum8-to-be-built-by-ho.patch
  Makefile-define-endianess-for-cross-buil.patch
  ipxe-Makefile-fix-issues-of-build-reprod.patch
  qboot-add-cross.ini-file-to-handle-aarch.patch
  sgabios-Makefile-fix-issues-of-build-rep.patch
  Silence-GCC-12-spurious-warnings.patch
  build-be-explicit-about-mx86-used-note-n.patch
  enable-cross-compilation-on-ARM.patch
  stub-out-the-SAN-req-s-in-int13.patch
  help-compiler-out-by-initializing-array.patch
  seabios-switch-to-python3-as-needed.patch
  * Patches added:
  openSUSE-Basetools-Ignore-spurious-GCC-1.patch
  openSUSE-Makefile-Fix-csum8-to-be-built-.patch
  openSUSE-Makefile-define-endianess-for-c.patch
  openSUSE-Makefile-fix-issues-of-build-re.patch
  openSUSE-add-cross.ini-file-to-handle-aa.patch
  openSUSE-build-Makefile-fix-issues-of-bu.patch
  openSUSE-build-Silence-GCC-12-spurious-w.patch
  openSUSE-build-be-explicit-about-mx86-us.patch
  openSUSE-build-enable-cross-compilation-.patch
  openSUSE-pcbios-stub-out-the-SAN-req-s-i.patch
  openSUSE-switch-to-python3-as-needed.patch
  openSUSE-test-help-compiler-out-by-initi.patch
* Wed Jan 11 2023 li.zhang@suse.com
- Fixes jsc#PED-1716
  * Patches added:
  configure-Add-Wno-gnu-variable-sized-typ.patch
  s390x-pci-add-routine-to-get-host-functi.patch
  s390x-pci-don-t-fence-interpreted-device.patch
  s390x-pci-enable-adapter-event-notificat.patch
  s390x-pci-enable-for-load-store-interpre.patch
  s390x-pci-let-intercept-devices-have-sep.patch
  s390x-pci-reflect-proper-maxstbl-for-gro.patch
  Update-linux-headers-to-v6.0-rc4.patch
* Thu Jan  5 2023 dmueller@suse.com
- install binfmt-misc handlers for systemd (bsc#1206838)
* Wed Nov 16 2022 dfaggioli@suse.com
- Raise the maximum number of vCPUs a VM can have to 1024 (jsc#PED-2592)
  * Patches added:
  pc-q35-Bump-max_cpus-to-1024.patch
* Wed Nov  9 2022 giecrilj@stegny.2a.pl
- install SeaBIOS documentation
* Thu Nov  3 2022 schwab@suse.de
- Enable KVM support on riscv64
* Wed Oct 26 2022 dfaggioli@suse.com
- qtests test are not realiable when run inside OBS builders, so
  let's disable that part of the testsuite for now. There is work
  ongoing to run it somewhere else (on dedicated hosts) to avoid
  loosing coverage. (bsc#1204566)
* Mon Oct 24 2022 dfaggioli@suse.com
- Improve dependency handling (e.g., what's recommended vs. what's
  required.
- Add a subpackage (qemu-headless) that brings in all the packages
  that are needed for creating VMs with tools like virt-install
  or VirtManager, run either locally or from a remote host.
  (bsc#1202166)
* Mon Oct 10 2022 hpj@urpla.net
- Build fails due to exceeding 10 GB disk limit (10430 MB):
  raise disk space contraint to 12 GB
* Fri Oct  7 2022 dfaggioli@suse.com
- Fixes bsc#1204082
  * Patches added:
  block-io_uring-revert-Use-io_uring_regis.patch
* Wed Oct  5 2022 dfaggioli@suse.com
- Due to change in where some documentation files are, if
  qemu-guest-agent is installed, we need to make sure we update it
  to our version (bsc#1203995)
* Wed Oct  5 2022 dfaggioli@suse.com
- The links in the forsplit dirs, in each subpackage, born to deal with
  package & subpackage splitting, are not really used. In fact, they're
  "Provides:"-ed by a bunch of subpackages, but there's no "Requires:"
  for any of them. Let's just get rid of them.
* Tue Oct  4 2022 dfaggioli@suse.com
- The old qemu-binfmt weappers around the various qemu-$ARCH Linux
  user emulation binaries (see, e.g., bsc#1186256) are not necessary
  any longer, and bsc#1143725 can now be considered fixed.
  * Patches dropped:
  linux-user-add-binfmt-wrapper-for-argv-0.patch
  linux-user-binfmt-support-host-binaries.patch
* Tue Oct  4 2022 dfaggioli@suse.com
- Fix bsc#1204001. Patches are not upstream, and have been picked up
  and backported from the ML. This is something we usually prefer to
  avoid, but this is urgent, and the patches looks fine, with high
  chances for them to be included as they are (and if they're not, we
  will revisit this, i.e., drop them and re-include the ones that are
  actually committed)
  * Patches added:
  linux-user-add-more-compat-ioctl-definit.patch
  linux-user-drop-conditionals-for-obsolet.patch
  linux-user-remove-conditionals-for-many-.patch
  meson-enforce-a-minimum-Linux-kernel-hea.patch
* Tue Oct  4 2022 dfaggioli@suse.com
- Improve the output of update_git.sh, by including the list of
  repos to which we have downstream patches.
* Fri Sep 30 2022 dfaggioli@suse.com
- Fix: bsc#1202665, CVE-2022-2962
  * Patches added:
  net-tulip-Restrict-DMA-engine-to-memorie.patch
* Fri Sep 30 2022 dmueller@suse.com
- skip tests that don't work under qemu-linux-user emulation
* Wed Sep 28 2022 dfaggioli@suse.com
- Runs of the test-suite seem much more stable now, in this version
  of QEMU. (bsc#1203610) We are also fine re-enabling running them
  in parallel.
* Wed Sep 28 2022 dfaggioli@suse.com
- Switch QEMU Linux user to emulate the same CPU as the one of the
  host by default. This is a bit conrtoversial and tricky, when
  thinking about system emulation/virtualization. But for linux-user,
  it should be just fine. (bsc#1203684)
  * Patches added:
  linux-user-use-max-as-default-CPU-model-.patch
* Tue Sep 27 2022 dfaggioli@suse.com
- Be less verbose when packaging documentation. In fact, with just
  a couple of (minor) re-arrangements, we can get rid of having to
  list all the files all the time
- Package /etc/qemu/bridge.conf as '%%config(noreplace). Next step
  will probably be to move it to /usr/etc/qemu (bsc#1201944)
* Wed Sep 21 2022 dfaggioli@suse.com
- Switch to %%autosetup for all products (this required some changes
  in update_git.sh)
- Run check-qtest sequentially, as it's more reliable, when in OBS
- Build with libbpf, fdt and capstone support
- Drop the patch adding our support document, and deal with that
  in the spec file directly
  * Patches dropped:
  doc-add-our-support-doc-to-the-main-proj.patch
* Tue Sep 20 2022 dfaggioli@suse.com
- Updated to latest upstream version 7.1
  * https://wiki.qemu.org/ChangeLog/7.1
  Be sure to also check the following pages:
  * https://qemu-project.gitlab.io/qemu/about/removed-features.html
  * https://qemu-project.gitlab.io/qemu/about/deprecated.html
  Some notable changes:
  * [x86] Support for architectural LBRs on KVM virtual machines
  * [x86] The libopcode-based disassembler has been removed. Use
    Capstone instead
  * [LoongArch] Add initial support for the LoongArch64 architecture.
  * [ARM] The emulated SMMUv3 now advertises support for SMMUv3.2-BBML2
  * [ARM] The xlnx-zynqmp SoC model now implements the 4 TTC timers
  * [ARM] The versal machine now models the Cortex-R5s in the Real-Time
    Processing Unit (RPU) subsystem
  * [ARM] The virt board now supports emulation of the GICv4.0
  * [ARM] New emulated CPU types: Cortex-A76, Neoverse-N1
  * [HPPA] Fix serial port pass-through from host to guest
  * [HPPA] Lots of general code improvements and tidy-ups
  * [RISC-V] RISC-V
  * [RISC-V] Add support for privileged spec version 1.12.0
  * [RISC-V] Use privileged spec version 1.12.0 for virt machine by default
  * [RISC-V] Allow software access to MIP SEIP
  * [RISC-V] Add initial support for the Sdtrig extension
  * [RISC-V] Optimisations and improvements for the vector extension
  * [VFIO] Experimental support for exposing emulated PCI devices over the
    new vfio-user protocol (a vfio-user client is not yet available
    in QEMU, though)
  * [QMP] The on-cbw-error option for copy-before-write filter, to specify
    behavior on CBW (copy before write) operation failure.
  * [QMP] The cbw-timeout option for copy-before-write filter, to specify
    timeout for CBW operation.
  * [QMP] New commands query-stats and query-stats-schema to retrieve
    statistics from various QEMU subsystems (right now only from
    KVM).
  * [QMP] The PanicAction can now be configured to report an exit-failure
    (useful for automated testing)
  * [Networking] QEMU can be compiled with the system slirp library even
    when using CFI. This requires libslirp 4.7.
  * [Migration] Support for zero-copy-send on Linux, which reduces CPU
    usage on the source host. Note that locked memory is needed
    to support this
  * Patches added:
  Revert-tests-qtest-enable-more-vhost-use.patch
  meson-remove-pkgversion-from-CONFIG_STAM.patch
  * Patches dropped:
  AIO-Reduce-number-of-threads-for-32bit-h.patch
  Makefile-Don-t-check-pc-bios-as-pre-requ.patch
  Revert-8dcb404bff6d9147765d7dd3e9c849337.patch
  Revert-qht-constify-qht_statistics_init.patch
  XXX-dont-dump-core-on-sigabort.patch
  acpi_piix4-Fix-migration-from-SLE11-SP2.patch
  configure-only-populate-roms-if-softmmu.patch
  configure-remove-pkgversion-from-CONFIG_.patch
  coroutine-ucontext-use-QEMU_DEFINE_STATI.patch
  coroutine-use-QEMU_DEFINE_STATIC_CO_TLS.patch
  coroutine-win32-use-QEMU_DEFINE_STATIC_C.patch
  hostmem-default-the-amount-of-prealloc-t.patch
  hw-usb-hcd-ehci-fix-writeback-order.patch
  i8254-Fix-migration-from-SLE11-SP2.patch
  intc-exynos4210_gic-replace-snprintf-wit.patch
  modules-generates-per-target-modinfo.patch
  modules-introduces-module_kconfig-direct.patch
  pc-bios-s390-ccw-net-avoid-warning-about.patch
  pci-fix-overflow-in-snprintf-string-form.patch
  qemu-cvs-gettimeofday.patch
  qemu-cvs-ioctl_debug.patch
  qemu-cvs-ioctl_nodirection.patch
  qht-Revert-some-constification-in-qht.c.patch
  qom-handle-case-of-chardev-spice-module-.patch
  scsi-lsi53c895a-fix-use-after-free-in-ls.patch
  scsi-lsi53c895a-really-fix-use-after-fre.patch
  softmmu-Always-initialize-xlat-in-addres.patch
  sphinx-change-default-language-to-en.patch
  test-add-mapping-from-arch-of-i686-to-qe.patch
  tests-Fix-block-tests-to-be-compatible-w.patch
  tests-qtest-Move-the-fuzz-tests-to-x86-o.patch
  usb-Help-compiler-out-to-avoid-a-warning.patch
* Thu Sep 15 2022 dimstar@opensuse.org
- pcre-devel-static is only needed when building against
  glib2 < 2.73. After that, glib2 was migrated to pcre2.
* Thu Jul 21 2022 dfaggioli@suse.com
- Substantial rework of the spec file:
  * the 'make check' testsuite now runs in the %%check section of
    the main package, not in a subpackage
  * switched from %%setup to %%autosetup
  * rearranged the content in order to minimize the use of %%if,
    %%ifarch, etc
* Thu Jul 21 2022 dfaggioli@suse.com
- Properly fix bsc#1198038, CVE-2022-0216
  * Patches added:
  scsi-lsi53c895a-really-fix-use-after-fre.patch
  tests-qtest-Move-the-fuzz-tests-to-x86-o.patch
* Thu Jul 21 2022 dfaggioli@suse.com
- Make temp dir (for update_git.sh) configurable
- Added new subpackages (audio-dbus, ui-dbus)
- bsc#1199018 was never fixed in Factory's QEMU 6.2. It is
  now (since the patches are already in SeaBIOS 1.16.0)
- Some tests are having issues when run in OBS. They seem to be
  due to race conditions, triggered by resource constraints of
  OBS workers. Let's disable them for now, while looking for a fix
- Update to v7.0.0 (bsc#1201307). For full release notes, see:
  * https://wiki.qemu.org/ChangeLog/7.0
  Be sure to also check the following pages:
  * https://qemu-project.gitlab.io/qemu/about/removed-features.html
  * https://qemu-project.gitlab.io/qemu/about/deprecated.html
  Some notable changes:
  * [ARM] The virt board has gained a new control knob to disable passing a RNG seed in the DTB (dtb-kaslr-seed)
  * [ARM] The AST2600 SoC now supports a dummy version of the i3c device
  * [ARM] The virt board can now run guests with KVM on hosts with restricted IPA ranges
  * [ARM] The virt board now supports virtio-mem-pci
  * [ARM] The virt board now supports specifying the guest CPU topology
  * [ARM] On the virt board, we now enable PAuth when using KVM or hvf and the host CPU supports it
  * [RISC-V] Add support for ratified 1.0 Vector extension
  * [RISC-V] Support for the Zve64f and Zve32f extensions
  * [RISC-V] Drop support for draft 0.7.1 Vector extension
  * [RISC-V] Support Zfhmin and Zfh extensions
  * [RISC-V] RISC-V KVM support
  * [RISC-V] Mark Hypervisor extension as non experimental
  * [RISC-V] Enable Hypervisor extension by default
  * [x86] Support for Intel AMX.
  * [PCI/PCIe] Q35: fix PCIe device becoming disabled after migration when ACPI based PCI hotplug is used (6b0969f1ec)
  * [PCI/PCIe] initial bits of SR/IOV support (250346169)
  * [PCI/PCIe] arm/virt: fixed PXB interrupt routing (e609301b45)
  * [PCI/PCIe] arm/virt: support for virtio-mem-pci (b1b87327a9)
  * [virtiofs] Fix for CVE-2022-0358 - behaviour with supplementary groups and SGID directories
  * [virtiofs] Improved security label support
  * [virtiofs] The virtiofsd in qemu is now starting to be deprecated; please start using and contributing to Rust virtiofsd
  * Patches dropped:
  acpi-validate-hotplug-selector-on-access.patch
  block-backend-Retain-permissions-after-m.patch
  block-qdict-Fix-Werror-maybe-uninitializ.patch
  brotli-fix-actual-variable-array-paramet.patch
  display-qxl-render-fix-race-condition-in.patch
  doc-Add-the-SGX-numa-description.patch
  hw-i386-amd_iommu-Fix-maybe-uninitialize.patch
  hw-intc-exynos4210_gic-provide-more-room.patch
  hw-nvme-fix-CVE-2021-3929.patch
  hw-nvram-at24-return-0xff-if-1-byte-addr.patch
  iotest-065-explicit-compression-type.patch
  iotest-214-explicit-compression-type.patch
  iotest-302-use-img_info_log-helper.patch
  iotest-303-explicit-compression-type.patch
  iotest-39-use-_qcow2_dump_header.patch
  iotests-60-more-accurate-set-dirty-bit-i.patch
  iotests-bash-tests-filter-compression-ty.patch
  iotests-common.rc-introduce-_qcow2_dump_.patch
  iotests-declare-lack-of-support-for-comp.patch
  iotests-drop-qemu_img_verbose-helper.patch
  iotests-massive-use-_qcow2_dump_header.patch
  iotests-MRCE-Write-data-to-source.patch
  iotests.py-filter-out-successful-output-.patch
  iotests.py-img_info_log-rename-imgopts-a.patch
  iotests.py-implement-unsupported_imgopts.patch
  iotests.py-qemu_img-create-support-IMGOP.patch
  iotests.py-rewrite-default-luks-support-.patch
  iotests-specify-some-unsupported_imgopts.patch
  meson-build-all-modules-by-default.patch
  numa-Enable-numa-for-SGX-EPC-sections.patch
  numa-Support-SGX-numa-in-the-monitor-and.patch
  python-aqmp-add-__del__-method-to-legacy.patch
  python-aqmp-add-_session_guard.patch
  python-aqmp-add-SocketAddrT-to-package-r.patch
  python-aqmp-add-socket-bind-step-to-lega.patch
  python-aqmp-add-start_server-and-accept-.patch
  python-aqmp-copy-type-definitions-from-q.patch
  python-aqmp-drop-_bind_hack.patch
  python-aqmp-fix-docstring-typo.patch
  python-aqmp-Fix-negotiation-with-pre-oob.patch
  python-aqmp-fix-race-condition-in-legacy.patch
  Python-aqmp-fix-type-definitions-for-myp.patch
  python-aqmp-handle-asyncio.TimeoutError-.patch
  python-aqmp-refactor-_do_accept-into-two.patch
  python-aqmp-remove-_new_session-and-_est.patch
  python-aqmp-rename-accept-to-start_serve.patch
  python-aqmp-rename-AQMPError-to-QMPError.patch
  python-aqmp-split-_client_connected_cb-o.patch
  python-aqmp-squelch-pylint-warning-for-t.patch
  python-aqmp-stop-the-server-during-disco.patch
  python-introduce-qmp-shell-wrap-convenie.patch
  python-machine-raise-VMLaunchFailure-exc.patch
  python-move-qmp-shell-under-the-AQMP-pac.patch
  python-move-qmp-utilities-to-python-qemu.patch
  python-qmp-switch-qmp-shell-to-AQMP.patch
  python-support-recording-QMP-session-to-.patch
  python-upgrade-mypy-to-0.780.patch
  qcow2-simple-case-support-for-downgradin.patch
  qemu-binfmt-conf.sh-should-use-F-as-shor.patch
  tests-qemu-iotests-040-Skip-TestCommitWi.patch
  tests-qemu-iotests-Fix-051-for-binaries-.patch
  tests-qemu-iotests-testrunner-Quote-case.patch
  tools-virtiofsd-Add-rseq-syscall-to-the-.patch
  ui-cursor-fix-integer-overflow-in-cursor.patch
  vhost-vsock-detach-the-virqueue-element-.patch
  virtiofsd-Drop-membership-of-all-supplem.patch
  virtio-net-fix-map-leaking-on-error-duri.patch
  Disable-some-tests-that-have-problems-in.patch
  * Patches added:
  intc-exynos4210_gic-replace-snprintf-wit.patch
  Revert-8dcb404bff6d9147765d7dd3e9c849337.patch
* Wed Jul 20 2022 dfaggioli@suse.com
- Fix bsc#1197084
  * Patches added:
  hostmem-default-the-amount-of-prealloc-t.patch
* Sun Jul 17 2022 dfaggioli@suse.com
- Get rid of downstream patches breaking s390 modules. Replace
  them with the upstream proposed and Acked (but never committed)
  solution (bsc#1199015)
  * Patches added:
  modules-generates-per-target-modinfo.patch
  modules-introduces-module_kconfig-direct.patch
  * Patches dropped:
  Fix-the-module-building-problem-for-s390.patch
  modules-quick-fix-a-fundamental-error-in.patch
* Sat Jul 16 2022 dfaggioli@suse.com
- backport patches for having coroutine work well when LTO is used
  * Patches added:
  coroutine-ucontext-use-QEMU_DEFINE_STATI.patch
  coroutine-use-QEMU_DEFINE_STATIC_CO_TLS.patch
  coroutine-win32-use-QEMU_DEFINE_STATIC_C.patch
* Sat Jul 16 2022 dfaggioli@suse.com
- seabios: drop patch that changes python in python2.
  Just go to python3 directly.
  * Patches dropped:
  seabios-use-python2-explicitly-as-needed.patch
* Fri Jul 15 2022 dfaggioli@suse.com
- Fix the following bugs:
  - bsc#1198037, CVE-2021-4207
  - bsc#1198038, CVE-2022-0216
  - bsc#1201367, CVE-2022-35414
  - bsc#1198035, CVE-2021-4206
  - bsc#1198712, CVE-2022-26354
  - bsc#1198711, CVE-2022-26353
  * Patches added:
  display-qxl-render-fix-race-condition-in.patch
  scsi-lsi53c895a-fix-use-after-free-in-ls.patch
  softmmu-Always-initialize-xlat-in-addres.patch
  ui-cursor-fix-integer-overflow-in-cursor.patch
  vhost-vsock-detach-the-virqueue-element-.patch
  virtio-net-fix-map-leaking-on-error-duri.patch
* Fri Jul  1 2022 lma@suse.com
- Fix usb ehci boot failure (bsc#1192115)
  * Patches added:
  hw-usb-hcd-ehci-fix-writeback-order.patch
* Tue Jun 21 2022 dfaggioli@suse.com
- Fix bugs boo#1200557 and boo#1199924
- Now that boo#1199924 is fixed, re-enable FORTIFY_SOURCE=3
  * Patches added:
  pci-fix-overflow-in-snprintf-string-form.patch
  sphinx-change-default-language-to-en.patch
* Fri May 27 2022 dfaggioli@suse.com
- It has been observed that building QEMU with _FORTIFY_SOURCE=3
  causes problem (see bsc#1199924). Force it to =2 for now, while
  we investigate the issue.
* Mon May 23 2022 dfaggioli@suse.com
- Backport a GCC 12 aarch64 build fix (bsc#1199625)
  * Patches added:
  block-qdict-Fix-Werror-maybe-uninitializ.patch
* Wed May 11 2022 mliska@suse.cz
- Filter out rpmlint error that is valid for qemu, but will
  have its badness increased in the future.
* Tue Apr 19 2022 dmueller@suse.com
- enable aio=io_uring on all kvm architectures (bsc#1197699)
* Tue Apr  5 2022 li.zhang@suse.com
- Backport aqmp patches from upstream which can fix iotest issues
  * Patches added:
  python-aqmp-add-__del__-method-to-legacy.patch
  python-aqmp-add-_session_guard.patch
  python-aqmp-add-SocketAddrT-to-package-r.patch
  python-aqmp-add-socket-bind-step-to-lega.patch
  python-aqmp-add-start_server-and-accept-.patch
  python-aqmp-copy-type-definitions-from-q.patch
  python-aqmp-drop-_bind_hack.patch
  python-aqmp-fix-docstring-typo.patch
  python-aqmp-Fix-negotiation-with-pre-oob.patch
  python-aqmp-fix-race-condition-in-legacy.patch
  Python-aqmp-fix-type-definitions-for-myp.patch
  python-aqmp-handle-asyncio.TimeoutError-.patch
  python-aqmp-refactor-_do_accept-into-two.patch
  python-aqmp-remove-_new_session-and-_est.patch
  python-aqmp-rename-accept-to-start_serve.patch
  python-aqmp-rename-AQMPError-to-QMPError.patch
  python-aqmp-split-_client_connected_cb-o.patch
  python-aqmp-squelch-pylint-warning-for-t.patch
  python-aqmp-stop-the-server-during-disco.patch
  python-introduce-qmp-shell-wrap-convenie.patch
  python-machine-raise-VMLaunchFailure-exc.patch
  python-move-qmp-shell-under-the-AQMP-pac.patch
  python-move-qmp-utilities-to-python-qemu.patch
  python-qmp-switch-qmp-shell-to-AQMP.patch
  python-support-recording-QMP-session-to-.patch
  python-upgrade-mypy-to-0.780.patch
* Tue Apr  5 2022 li.zhang@suse.com
- Drop the patches which are workaround to fix iotest issues
  * Patches dropped:
  Revert-python-iotests-replace-qmp-with-a.patch
  Revert-python-machine-add-instance-disam.patch
  Revert-python-machine-add-sock_dir-prope.patch
  Revert-python-machine-handle-fast-QEMU-t.patch
  Revert-python-machine-move-more-variable.patch
  Revert-python-machine-remove-_remove_mon.patch
* Thu Mar 31 2022 li.zhang@suse.com
- Support the SGX feature (bsc#1197807)
  * Patches added:
  doc-Add-the-SGX-numa-description.patch
  numa-Enable-numa-for-SGX-EPC-sections.patch
  numa-Support-SGX-numa-in-the-monitor-and.patch
* Tue Mar 29 2022 li.zhang@suse.com
- Backport CVE-2021-3929 (bsc#1193880)
  * Patches added:
  hw-nvme-fix-CVE-2021-3929.patch
* Tue Mar 29 2022 li.zhang@suse.com
- The patches from upstream cause testsuit failures (bsc#1197150 bsc#1197528)
  * Patches added:
  Revert-python-iotests-replace-qmp-with-a.patch
  Revert-python-machine-add-instance-disam.patch
  Revert-python-machine-add-sock_dir-prope.patch
  Revert-python-machine-handle-fast-QEMU-t.patch
  Revert-python-machine-move-more-variable.patch
  Revert-python-machine-remove-_remove_mon.patch
* Tue Mar 29 2022 dfaggioli@suse.com
- Add missing patch from a PTFs (bsc#1194938)
  * Patches added:
  scsi-generic-check-for-additional-SG_IO-.patch
* Fri Mar 25 2022 dfaggioli@suse.com
- Kill downstream patches around bifmt handling that makes
  cumbersome to run multi-arch containers, and switch to the
  upstream behavior, which is well documented and valid on
  all other distros. This is possible thanks to Linux kernel
  commit 2347961b11d4 and QEMU commit 6e1c0d7b951e19c53 (so
  it can only work on Leap/SLE 15.4 and higher). (bsc#1197298)
  * Patches dropped:
  qemu-binfmt-conf.sh-allow-overriding-SUS.patch
  qemu-binfmt-conf-use-qemu-ARCH-binfmt.patch
* Fri Mar 25 2022 dfaggioli@suse.com
- Fix update_git.sh wiping all the package file of the local
  checkout while cloning the git repository on demand (in case they
  don't exist and the user as to do so).
* Thu Mar 24 2022 li.zhang@suse.com
- Improve test reliability
  * Patches added:
  Fix-the-module-building-problem-for-s390.patch
  tests-qemu-iotests-040-Skip-TestCommitWi.patch
  tests-qemu-iotests-testrunner-Quote-case.patch
* Wed Mar 23 2022 dfaggioli@suse.com
- Fix virtiofs crashing with glibc >= 2.35, due to rseq syscall
  (bsc#1196924)
  * Patches added:
  tools-virtiofsd-Add-rseq-syscall-to-the-.patch
* Tue Mar 22 2022 dfaggioli@suse.com
- Avoid warnings caused by a GCC 12 bug, see https://gcc.gnu.org/bugzilla/show_bug.cgi?id=98503
  (bsc#1197018)
  * Patches added:
  hw-i386-amd_iommu-Fix-maybe-uninitialize.patch
  Silence-GCC-12-spurious-warnings.patch
  Ignore-spurious-GCC-12-warning.patch
* Tue Mar 15 2022 li.zhang@suse.com
- Proactive fix
  * Patches added:
  hw-nvram-at24-return-0xff-if-1-byte-addr.patch
* Mon Mar  7 2022 dfaggioli@suse.com
- Build PPC firmwares from sources on non-PPC builds as well
  (bsc#1193545)
- Build RiscV firmwares on non-RiscV builds as well
- While there, refactor (and simplify!) the firmware building
  logic and code
  * Patches added:
  Makefile-define-endianess-for-cross-buil.patch
  Makefile-fix-build-with-binutils-2.38.patch
* Mon Mar  7 2022 dfaggioli@suse.com
- qemu,kvm,xen: NULL pointer dereference issue in megasas-gen2 host
  bus adapter (bsc#1180432, CVE-2020-35503)
  * Patches added:
  hw-scsi-megasas-check-for-NULL-frame-in-.patch
* Fri Feb 18 2022 dfaggioli@suse.com
- Include vmxcap in the qemu-tools package (is being very useful
  for debugging bsc#1193364)
* Fri Feb 18 2022 dfaggioli@suse.com
- The qemu package should require qemu-x86, qemu-arm, etc, as there's
  no point installing it without _any_ of them. Additionally, right
  now, the user does not get a working qemu, if recommended packages
  are disabled (e.g., on MicroOS or SLE Micro). bsc#1196087
* Fri Feb 18 2022 dfaggioli@suse.com
- Give clearer instructions on how to modify the package patches
  from the output of update_git.sh (docs change only, no functional
  change)
* Fri Feb 18 2022 dfaggioli@suse.com
- qemu,kvm: potential privilege escalation via virtiofsd
  (bsc#1195161, CVE-2022-0358)
  * Patches added:
  virtiofsd-Drop-membership-of-all-supplem.patch
* Fri Feb 18 2022 li.zhang@suse.com
  * Patches added:
  block-backend-Retain-permissions-after-m.patch
  iotest-065-explicit-compression-type.patch
  iotest-214-explicit-compression-type.patch
  iotest-302-use-img_info_log-helper.patch
  iotest-303-explicit-compression-type.patch
  iotest-39-use-_qcow2_dump_header.patch
  iotests-60-more-accurate-set-dirty-bit-i.patch
  iotests-bash-tests-filter-compression-ty.patch
  iotests-common.rc-introduce-_qcow2_dump_.patch
  iotests-declare-lack-of-support-for-comp.patch
  iotests-drop-qemu_img_verbose-helper.patch
  iotests-massive-use-_qcow2_dump_header.patch
  iotests-MRCE-Write-data-to-source.patch
  iotests.py-filter-out-successful-output-.patch
  iotests.py-img_info_log-rename-imgopts-a.patch
  iotests.py-implement-unsupported_imgopts.patch
  iotests.py-qemu_img-create-support-IMGOP.patch
  iotests.py-rewrite-default-luks-support-.patch
  iotests-specify-some-unsupported_imgopts.patch
  qcow2-simple-case-support-for-downgradin.patch
  tests-qemu-iotests-Fix-051-for-binaries-.patch
* Wed Feb 16 2022 li.zhang@suse.com
-Backport patch from upstream, bsc#1194063 CVE-2021-4158
  * Patches added:
  acpi-validate-hotplug-selector-on-access.patch
* Sun Jan 23 2022 li.zhang@suse.com
- Enable modules for testsuite
* Mon Jan 17 2022 li.zhang@suse.com
  * Patches added:
  meson-build-all-modules-by-default.patch
* Wed Jan 12 2022 dfaggioli@suse.com
- It's time to really start requiring -F when using -b in
  qemu-img for us as well. Users/customers have been warned
  in the relevant release notes (bsc#1190135)
  * Patches dropped:
  Revert-qemu-img-Improve-error-for-rebase.patch
  Revert-qemu-img-Require-F-with-b-backing.patch
* Wed Dec 22 2021 dfaggioli@suse.com
- Fix testsuite failures by not using modules when building tests
  (and some other, also testsuite related, spec file problems)
* Tue Dec 21 2021 lma@suse.com
- [JIRA] (SLE-20965) Make QEMU guests more failsafe when resizing
  SCSI passthrough disks
  * Patches added:
    scsi-generic-replace-logical-block-count.patch
* Thu Dec 16 2021 li.zhang@suse.com
- Add an audio-oss sub-package
* Thu Dec 16 2021 dfaggioli@suse.com
- Add some new (mostly documentation) files in the package
* Thu Dec 16 2021 li.zhang@suse.com
- Remove option --audio-drv-list because audio is detected by
  meson automatically in latest version.
* Thu Dec 16 2021 li.zhang@suse.com
- Remove options --disable-jemalloc and --disable-tcmalloc
  which are changed in v6.2.0.
* Thu Dec 16 2021 dfaggioli@suse.com
- Update to v 6.2.0. For full release notese, see:
  * https://wiki.qemu.org/ChangeLog/6.2.
  Be sure to also check the following pages:
  * https://qemu-project.gitlab.io/qemu/about/removed-features.html
  * https://qemu-project.gitlab.io/qemu/about/deprecated.html
  Some notable changes:
  * virtio-mem: guest memory dumps are now fully supported, along
    with pre-copy/post-copy migration and background guest snapshots
  * QMP: support for nw DEVICE_UNPLUG_GUEST_ERROR to detect
    guest-reported hotplug failures
  * TCG: improvements to TCG plugin argument syntax, and multi-core
    support for cache plugin
  * 68k: improved support for Apple’s NuBus, including ability to
    load declaration ROMs, and slot IRQ support
  * ARM: macOS hosts with Apple Silicon CPUs now support ‘hvf’
    accelerator for AArch64 guests
  * ARM: emulation support for Fujitsu A64FX processor model
  * ARM: emulation support for kudo-mbc machine type
  * ARM: M-profile MVE extension is now supported for Cortex-M55
  * ARM: ‘virt’ machine now supports an emulated ITS (Interrupt
    Translation Service) and supports more than 123 CPUs in
    emulation mode
  * ARM: xlnx-zcu102 and xlnx-versal-virt machines now support
    BBRAM and eFUSE devices
  * PowerPC: improved POWER10 support for the ‘powernv’ machine type
  * PowerPC: initial support for POWER10 DD2.0 CPU model
  * PowerPC: support for FORM2 PAPR NUMA descriptions for ‘pseries’ machine type
  * RISC-V: support for Zb[abcs] instruction set extensions
  * RISC-V: support for vhost-user and numa mem options across all boards
  * RISC-V: SiFive PWM support
  * x86: support for new Snowridge-v4 CPU model
  * x86: guest support for Intel SGX
  * x86: AMD SEV guests now support measurement of kernel binary when doing
    direct kernel boot (not using a bootloader)
  * Patches dropped:
  9pfs-fix-crash-in-v9fs_walk.patch
  block-introduce-max_hw_iov-for-use-in-sc.patch
  hmp-Unbreak-change-vnc.patch
  hw-acpi-ich9-Add-compat-prop-to-keep-HPC.patch
  hw-i386-acpi-build-Deny-control-on-PCIe-.patch
  i386-cpu-Remove-AVX_VNNI-feature-from-Co.patch
  net-vmxnet3-validate-configuration-value.patch
  pcie-rename-native-hotplug-to-x-native-h.patch
  plugins-do-not-limit-exported-symbols-if.patch
  plugins-execlog-removed-unintended-s-at-.patch
  qemu-nbd-Change-default-cache-mode-to-wr.patch
  qemu-sockets-fix-unix-socket-path-copy-a.patch
  target-arm-Don-t-skip-M-profile-reset-en.patch
  target-i386-add-missing-bits-to-CR4_RESE.patch
  tcg-arm-Fix-tcg_out_vec_op-function-sign.patch
  uas-add-stream-number-sanity-checks.patch
  vhost-vsock-fix-migration-issue-when-seq.patch
  virtio-balloon-don-t-start-free-page-hin.patch
  virtio-mem-pci-Fix-memory-leak-when-crea.patch
  virtio-net-fix-use-after-unmap-free-for-.patch
* Tue Dec 14 2021 dfaggioli@suse.com
- Reinstate Lin Ma's fixes for bsc#1192147 as they were
  submitted only to IBS.
  * Patches added:
  hw-acpi-ich9-Add-compat-prop-to-keep-HPC.patch
  hw-i386-acpi-build-Deny-control-on-PCIe-.patch
  pcie-rename-native-hotplug-to-x-native-h.patch
* Tue Dec 14 2021 dfaggioli@suse.com
- Rename the Guest Agent service qemu-guest-agent, like in other
  distros (and upstream). bsc#1185543
* Mon Dec 13 2021 dmueller@suse.com
- disable QOM cast debug outside the testsuite as the corresponding
  asserts show up occassionally as top #1 in perf(1) traces under
  heavy virtio load
- enable LTO when we'd like to use LTO
* Tue Dec  7 2021 dfaggioli@suse.com
  * Patches added (bsc#1186256):
  qemu-binfmt-conf.sh-allow-overriding-SUS.patch
* Mon Dec  6 2021 guillaume.gardet@opensuse.org
- cross-i386-binutils and cross-i386-gcc are not needed and were
  dropped from Factory - boo#1193424
* Tue Oct 26 2021 jose.ziviani@suse.com
- qemu: virtio-net: heap use-after-free in virtio_net_receive_rcu
  (bsc#1189938 CVE-2021-3748)
  solved by virtio-net-fix-use-after-unmap-free-for-.patch
- kvm,qemu: out-of-bounds write in UAS (USB Attached SCSI) device emulation
  (bsc#1189702 CVE-2021-3713)
  * Patches added:
  uas-add-stream-number-sanity-checks.patch
* Fri Oct  8 2021 li.zhang@suse.com
- Stable fixes from upstream
  * Patches added:
  block-introduce-max_hw_iov-for-use-in-sc.patch
  hmp-Unbreak-change-vnc.patch
  qemu-nbd-Change-default-cache-mode-to-wr.patch
  target-arm-Don-t-skip-M-profile-reset-en.patch
  vhost-vsock-fix-migration-issue-when-seq.patch
  virtio-mem-pci-Fix-memory-leak-when-crea.patch
  virtio-net-fix-use-after-unmap-free-for-.patch
* Thu Sep 16 2021 jose.ziviani@suse.com
- Fix testsuite dependencies (bsc#1190573)
  * Patches added:
  modules-quick-fix-a-fundamental-error-in.patch
* Tue Sep 14 2021 jose.ziviani@suse.com
- Replace patch to fix hardcoded binfmt handler
  (bsc#1186256)
  * Patches dropped:
  qemu-binfmt-conf.sh-allow-overriding-SUS.patch
  * Patches added:
  qemu-binfmt-conf.sh-should-use-F-as-shor.patch
- Stable fixes from upstream
  * Patches added:
  9pfs-fix-crash-in-v9fs_walk.patch
  i386-cpu-Remove-AVX_VNNI-feature-from-Co.patch
  plugins-do-not-limit-exported-symbols-if.patch
  plugins-execlog-removed-unintended-s-at-.patch
  qemu-sockets-fix-unix-socket-path-copy-a.patch
  target-i386-add-missing-bits-to-CR4_RESE.patch
  virtio-balloon-don-t-start-free-page-hin.patch
* Wed Sep  8 2021 jose.ziviani@suse.com
- Fix qemu build on ARMv7 (bsc#1190211)
  * Patches added:
  tcg-arm-Fix-tcg_out_vec_op-function-sign.patch
* Wed Sep  8 2021 jose.ziviani@suse.com
- Update supported file for ARM machines.
* Thu Sep  2 2021 jose.ziviani@suse.com
- Keep qemu-img without backing format still deprecated
  (bsc#1190135)
  * Patches added:
  Revert-qemu-img-Improve-error-for-rebase.patch
  Revert-qemu-img-Require-F-with-b-backing.patch
- Update the support files to reflect the deprecation.
* Tue Aug 31 2021 jose.ziviani@suse.com
- Update build dependencies versions: libgcrypt >= 1.8.0,
  gnutls >= 3.5.18, glib >= 2.56, libssh >= 0.8.7
* Fri Aug 27 2021 jose.ziviani@suse.com
- Fix hardcoded binfmt handler doesn't play well with containers
  (bsc#1186256)
  * Patches added:
  qemu-binfmt-conf.sh-allow-overriding-SUS.patch
* Thu Aug 26 2021 jose.ziviani@suse.com
- Update to v6.1: see https://wiki.qemu.org/ChangeLog/6.1
  For a full list of formely deprecated features that are removed,
  consult: https://qemu-project.gitlab.io/qemu/about/removed-features.html
  For a list of new deprecated features, consult:
  https://qemu-project.gitlab.io/qemu/about/deprecated.html
  Some noteworthy changes:
  * Removed moxie CPU.
  * Removed lm32 CPU.
  * Removed unicore32 CPU.
  * Removed 'info cpustats'.
  * Added Aspeed machines: rainier-bmc, quanta-q7l1-bmc.
  * Added npcm7xx machine: quanta-gbs-bmc.
  * Model for Aspeed's Hash and Crypto Engine.
  * SVE2 is now emulated, including bfloat16 support
  * FEAT_I8MM, FEAT_TLBIOS, FEAT_TLBRANGE, FEAT_BF16, FEAT_AA32BF16, and
  FEAT_MTE3 are now emulated.
  * Improved hot-unplug failures on PowerPC pseries machine.
  * Implemented some POWER10 instructions in TCG.
  * Added shakti_c RISC-V machine.
  * Improved documentation for RISC-V machines.
  * CPU models for gen16 have been added for s390x.
  * New CPU model versions added with XSAVES enabled:
  Skylake-Client-v4, Skylake-Server-v5, Cascadelake-Server-v5,
  Cooperlake-v2, Icelake-Client-v3, Icelake-Server-v5, Denverton-v3,
  Snowridge-v3, Dhyana-v2
  * Added ACPI based PCI hotplug support to Q35 machine. Enabled and
  used by default since pc-q35-6.1 machine type.
  * Added support for the pca9546 and pca9548 I2C muxes.
  * Added support for PMBus and several PMBus devices.
  * Crypto subsystem:
  The preferred crypto backend driver now gnutls, with libgcrypt as the
  second choice, and nettle as third choice, with ordering driven mostly
  by performance of the ciphers.
  * Misc doc improvements.
  * Patches removed:
  block-nvme-Fix-VFIO_MAP_DMA-failed-No-sp.patch
  hmp-Fix-loadvm-to-resume-the-VM-on-succe.patch
  hw-block-nvme-align-with-existing-style.patch
  hw-block-nvme-consider-metadata-read-aio.patch
  hw-net-can-sja1000-fix-buff2frame_bas-an.patch
  hw-nvme-fix-missing-check-for-PMR-capabi.patch
  hw-nvme-fix-pin-based-interrupt-behavior.patch
  hw-pci-host-q35-Ignore-write-of-reserved.patch
  hw-rdma-Fix-possible-mremap-overflow-in-.patch
  hw-rx-rx-gdbsim-Do-not-accept-invalid-me.patch
  hw-usb-Do-not-build-USB-subsystem-if-not.patch
  hw-usb-host-stub-Remove-unused-header.patch
  linux-user-aarch64-Enable-hwcap-for-RND-.patch
  module-for-virtio-gpu-pre-load-module-to.patch
  monitor-qmp-fix-race-on-CHR_EVENT_CLOSED.patch
  pvrdma-Ensure-correct-input-on-ring-init.patch
  pvrdma-Fix-the-ring-init-error-flow-CVE-.patch
  qemu-config-load-modules-when-instantiat.patch
  qemu-config-parse-configuration-files-to.patch
  qemu-config-use-qemu_opts_from_qdict.patch
  runstate-Initialize-Error-to-NULL.patch
  sockets-update-SOCKET_ADDRESS_TYPE_FD-li.patch
  target-i386-Exit-tb-after-wrmsr.patch
  target-sh4-Return-error-if-CPUClass-get_.patch
  tcg-Allocate-sufficient-storage-in-temp_.patch
  tcg-arm-Fix-tcg_out_op-function-signatur.patch
  tcg-sparc-Fix-temp_allocate_frame-vs-spa.patch
  ui-Fix-memory-leak-in-qemu_xkeymap_mappi.patch
  usb-hid-avoid-dynamic-stack-allocation.patch
  usb-limit-combined-packets-to-1-MiB-CVE-.patch
  usb-mtp-avoid-dynamic-stack-allocation.patch
  usb-redir-avoid-dynamic-stack-allocation.patch
  usbredir-fix-free-call.patch
  vfio-ccw-Permit-missing-IRQs.patch
  vhost-user-blk-Check-that-num-queues-is-.patch
  vhost-user-blk-Don-t-reconnect-during-in.patch
  vhost-user-blk-Fail-gracefully-on-too-la.patch
  vhost-user-blk-Get-more-feature-flags-fr.patch
  vhost-user-blk-Make-sure-to-set-Error-on.patch
  vhost-user-gpu-abstract-vg_cleanup_mappi.patch
  vhost-user-gpu-fix-leak-in-virgl_cmd_res.patch
  vhost-user-gpu-fix-leak-in-virgl_resourc.patch
  vhost-user-gpu-fix-memory-disclosure-in-.patch
  vhost-user-gpu-fix-memory-leak-in-vg_res.patch
  vhost-user-gpu-fix-memory-leak-while-cal.patch
  vhost-user-gpu-fix-OOB-write-in-virgl_cm.patch
  vhost-user-gpu-fix-resource-leak-in-vg_r.patch
  vhost-vdpa-don-t-initialize-backend_feat.patch
  virtio-blk-Fix-rollback-path-in-virtio_b.patch
  virtio-Fail-if-iommu_platform-is-request.patch
  virtiofsd-Fix-side-effect-in-assert.patch
  vl-allow-not-specifying-size-in-m-when-u.patch
  vl-Fix-an-assert-failure-in-error-path.patch
  vl-plug-object-back-into-readconfig.patch
  vl-plumb-keyval-based-options-into-readc.patch
  x86-acpi-use-offset-instead-of-pointer-w.patch
* Tue Aug 10 2021 jose.ziviani@suse.com
- usb: unbounded stack allocation in usbredir
  (bsc#1186012, CVE-2021-3527)
  hw-usb-Do-not-build-USB-subsystem-if-not.patch
  hw-usb-host-stub-Remove-unused-header.patch
  usb-hid-avoid-dynamic-stack-allocation.patch
  usb-limit-combined-packets-to-1-MiB-CVE-.patch
  usb-mtp-avoid-dynamic-stack-allocation.patch
* Fri Aug  6 2021 jose.ziviani@suse.com
- usbredir: free call on invalid pointer in bufp_alloc
  (bsc#1189145, CVE-2021-3682)
  usbredir-fix-free-call.patch
* Tue Aug  3 2021 jose.ziviani@suse.com
- Add stable patches from upstream:
  block-nvme-Fix-VFIO_MAP_DMA-failed-No-sp.patch
  hw-net-can-sja1000-fix-buff2frame_bas-an.patch
  hw-pci-host-q35-Ignore-write-of-reserved.patch
* Fri Jul 23 2021 jose.ziviani@suse.com
- Disabled skiboot building for PowerPC due to the following issue:
  https://github.com/open-power/skiboot/issues/265
* Fri Jul 23 2021 jose.ziviani@suse.com
- Fix possible mremap overflow in the pvrdma
  (CVE-2021-3582, bsc#1187499)
  hw-rdma-Fix-possible-mremap-overflow-in-.patch
- Ensure correct input on ring init
  (CVE-2021-3607, bsc#1187539)
  pvrdma-Ensure-correct-input-on-ring-init.patch
- Fix the ring init error flow
  (CVE-2021-3608, bsc#1187538)
  pvrdma-Fix-the-ring-init-error-flow-CVE-.patch
* Sat Jul  3 2021 jose.ziviani@suse.com
- Fix qemu-supportconfig network-manager verification
* Thu Jul  1 2021 jose.ziviani@suse.com
- Fix stable issues found in upstream:
  hmp-Fix-loadvm-to-resume-the-VM-on-succe.patch
  hw-block-nvme-align-with-existing-style.patch
  hw-nvme-fix-missing-check-for-PMR-capabi.patch
  hw-nvme-fix-pin-based-interrupt-behavior.patch
  linux-user-aarch64-Enable-hwcap-for-RND-.patch
  qemu-config-load-modules-when-instantiat.patch
  qemu-config-parse-configuration-files-to.patch
  qemu-config-use-qemu_opts_from_qdict.patch
  runstate-Initialize-Error-to-NULL.patch
  target-i386-Exit-tb-after-wrmsr.patch
  tcg-Allocate-sufficient-storage-in-temp_.patch
  tcg-sparc-Fix-temp_allocate_frame-vs-spa.patch
  vhost-vdpa-don-t-initialize-backend_feat.patch
  vl-allow-not-specifying-size-in-m-when-u.patch
  vl-Fix-an-assert-failure-in-error-path.patch
  vl-plug-object-back-into-readconfig.patch
  vl-plumb-keyval-based-options-into-readc.patch
  x86-acpi-use-offset-instead-of-pointer-w.patch
* Thu Jul  1 2021 jose.ziviani@suse.com
- Update qemu-supportconfig plugin
* Fri Jun 25 2021 jose.ziviani@suse.com
- Fix an update-alternative warning when removing qemu-skiboot package
  bsc#1178678
* Wed Jun 23 2021 jose.ziviani@suse.com
- Use doc directive to build QEMU documentation
* Fri Jun 11 2021 jose.ziviani@suse.com
- Improve compatibility with gcc 11:
  target-sh4-Return-error-if-CPUClass-get_.patch
  tcg-arm-Fix-tcg_out_op-function-signatur.patch
* Wed Jun  9 2021 jose.ziviani@suse.com
- Enable zstd compression option to qcow2
* Mon Jun  7 2021 jose.ziviani@suse.com
- Fix out-of-bounds write in virgl_cmd_get_capset
  CVE-2021-3546 bsc#1185981
  vhost-user-gpu-abstract-vg_cleanup_mappi.patch
- Fix memory leaks found in the virtio vhost-user GPU device
  CVE-2021-3544 bsc#1186010
  vhost-user-gpu-fix-leak-in-virgl_cmd_res.patch
  vhost-user-gpu-fix-leak-in-virgl_resourc.patch
  vhost-user-gpu-fix-memory-disclosure-in-.patch
  vhost-user-gpu-fix-memory-leak-in-vg_res.patch
  vhost-user-gpu-fix-memory-leak-while-cal.patch
  vhost-user-gpu-fix-OOB-write-in-virgl_cm.patch
- Fix information disclosure due to uninitialized memory read
  CVE-2021-3545 bsc#1185990
  vhost-user-gpu-fix-resource-leak-in-vg_r.patch
* Thu Jun  3 2021 dmueller@suse.com
- disable sheepdog, it was dropped upstream (
  https://gitlab.com/qemu-project/qemu/-/commit/09ec85176e4095be15f233ebc870d5680123f024)
  and fails to build with gcc 11 on non-x86
* Mon May 24 2021 jose.ziviani@suse.com
- Fix CVE-2021-3527 in usb/redir:
  usb-redir-avoid-dynamic-stack-allocation.patch
- Fix issues found upstream:
  hw-block-nvme-consider-metadata-read-aio.patch
  sockets-update-SOCKET_ADDRESS_TYPE_FD-li.patch
  vfio-ccw-Permit-missing-IRQs.patch
  vhost-user-blk-Check-that-num-queues-is-.patch
  vhost-user-blk-Don-t-reconnect-during-in.patch
  vhost-user-blk-Fail-gracefully-on-too-la.patch
  vhost-user-blk-Get-more-feature-flags-fr.patch
  vhost-user-blk-Make-sure-to-set-Error-on.patch
  virtio-blk-Fix-rollback-path-in-virtio_b.patch
  virtio-Fail-if-iommu_platform-is-request.patch
  virtiofsd-Fix-side-effect-in-assert.patch
  monitor-qmp-fix-race-on-CHR_EVENT_CLOSED.patch
* Mon May 17 2021 jose.ziviani@suse.com
- Brotli VLA error was already fixed in v5.2 but the patches wasn't
  included in v6.0. This change fixed that
- Patches added:
  brotli-fix-actual-variable-array-paramet.patch
  hw-rx-rx-gdbsim-Do-not-accept-invalid-me.patch
  ui-Fix-memory-leak-in-qemu_xkeymap_mappi.patch
* Thu May  6 2021 brogers@suse.com
- For the record, these issues are fixed in this package already.
  Most are alternate references to previously mentioned issues:
  (CVE-2019-15890, bsc#1149813, CVE-2020-8608, bsc#1163019,
  CVE-2020-14364, bsc#1175534, CVE-2020-25707, bsc#1178683,
  CVE-2020-25723, bsc#1178935, CVE-2020-29130, bsc#1179477,
  CVE-2020-29129, bsc#1179484, CVE-2021-3419, bsc#1182975)
* Fri Apr 30 2021 jose.ziviani@suse.com
- Update to v6.0: see https://wiki.qemu.org/ChangeLog/6.0
  For a full list of formely deprecated features that are removed now,
  consult: https://qemu-project.gitlab.io/qemu/system/removed-features.html.
  For a list of new deprecated features, consult:
  https://qemu-project.gitlab.io/qemu/system/deprecated.html
  Some noteworthy changes:
  * Removed tileGX CPU (linux-user mode).
  * Removed ide-drive device (use ide-hd or ide-cd instead).
  * Removed scsi-disk device (use scsi-hd or scsi-cd instead).
  * Removed pc-1.0, pc-1.1, pc-1.2, and pc-1.3 machine types.
  * Added emulation of Arm-v8.1M arch and Cortex-M55 CPU.
  * Added boards mps3-an524 (Cortex-M33) and mps3-an547 (Cortex-M55).
  * x86: Support for running SEV-ES encrypted guests; TCG can emulate
  the PKS feature; WHPX accelerator supports accelerated APIC.
  * ARM: ARMv8.4-TTST, ARMv8.4-SEL2, FEAT_SSBS, and ARMv8.4-DIT emulation
  are now supported; Added ARMv8.5-MemTag extension is now supported formely
  linux-user. Additional device emulation support for xlnx-zynqmp, xlnx-versal,
  sbsa-ref, npcm7xx, and sabrelite board models.
  * PowerPC: powernv now allows external BMC; pseries can send QAPI message
  if it detects a memory hotplug failure; CPU unplug request can be retried.
  * s390: TCG works with Linux kernels built with clang-11 and clang12.
  * RISC-V: OpenSBI upgraded to v0.9; Support the QMP dump-guest-memory
  command; Add support for the SiFive SPI controller (sifive_u); Add QSPI
  NOR flash to Microchip PFSoC.
  * Misc doc improvements.
  * Multiprocess: Add experimental options to support out-of-process device
  emulation.
  * ACPI: support for assigning NICs to known names in guest OS independently of
  PCI slot placement.
  * NVMe: new emulation support for v1.4 spec with many new features, experimental
  support for Zoned Namespaces, multipath I/O, and End-to-End Data Protection.
  * Xen: New guest loader for testing of Xen-like hypervisors booting kernels.
  * virtiofs: misc. security fixes and performance improvements.
  * Tools: FUSE block exports to allow mounting any QEMU block device node
  as a host file.
  * Migration: query/info-migrate now display the migration blocker status and
  the reasons for blocking.
  * User-mode: Added support for the Qualcomm Hexagon processor.
  * TCG: Added support for Apple Silicon hosts (macOS).
  * QMP: backup jobs now support multiple asynchronous requests in parallel
  * VNC: virtio-vga support for scaling resolution based on client window size
  * Patches added:
  doc-add-our-support-doc-to-the-main-proj.patch
  * Patches removed:
  9pfs-Fully-restart-unreclaim-loop-CVE-20.patch
  audio-add-sanity-check.patch
  block-Fix-deadlock-in-bdrv_co_yield_to_d.patch
  block-Fix-locking-in-qmp_block_resize.patch
  blockjob-Fix-crash-with-IOthread-when-bl.patch
  block-nfs-fix-int-overflow-in-nfs_client.patch
  block-rbd-fix-memory-leak-in-qemu_rbd_co.patch
  block-rbd-Fix-memory-leak-in-qemu_rbd_co.patch
  block-Separate-blk_is_writable-and-blk_s.patch
  block-Simplify-qmp_block_resize-error-pa.patch
  brotli-fix-actual-variable-array-paramet.patch
  build-no-pie-is-no-functional-linker-fla.patch
  cadence_gem-switch-to-use-qemu_receive_p.patch
  cpu-core-Fix-help-of-CPU-core-device-typ.patch
  docs-add-SUSE-support-statements-to-html.patch
  dp8393x-switch-to-use-qemu_receive_packe.patch
  e1000-fail-early-for-evil-descriptor.patch
  e1000-switch-to-use-qemu_receive_packet-.patch
  hw-arm-virt-acpi-build-Fix-GSIV-values-o.patch
  hw-arm-virt-Disable-pl011-clock-migratio.patch
  hw-block-fdc-Fix-fallback-property-on-sy.patch
  hw-intc-arm_gic-Fix-interrupt-ID-in-GICD.patch
  hw-isa-Kconfig-Add-missing-dependency-VI.patch
  hw-isa-piix4-Migrate-Reset-Control-Regis.patch
  hw-net-lan9118-Fix-RX-Status-FIFO-PEEK-v.patch
  hw-s390x-fix-build-for-virtio-9p-ccw.patch
  hw-sd-sd-Actually-perform-the-erase-oper.patch
  hw-sd-sd-Fix-build-error-when-DEBUG_SD-i.patch
  hw-sd-sdhci-Correctly-set-the-controller.patch
  hw-sd-sdhci-Don-t-transfer-any-data-when.patch
  hw-sd-sdhci-Don-t-write-to-SDHC_SYSAD-re.patch
  hw-sd-sdhci-Limit-block-size-only-when-S.patch
  hw-sd-sdhci-Reset-the-data-pointer-of-s-.patch
  hw-sd-sd-Move-the-sd_block_-read-write-a.patch
  hw-sd-sd-Skip-write-protect-groups-check.patch
  hw-timer-slavio_timer-Allow-64-bit-acces.patch
  hw-virtio-pci-Added-AER-capability.patch
  hw-virtio-pci-Added-counter-for-pcie-cap.patch
  i386-acpi-restore-device-paths-for-pre-5.patch
  iotests-Fix-_send_qemu_cmd-with-bash-5.1.patch
  lan9118-switch-to-use-qemu_receive_packe.patch
  lsilogic-Use-PCIDevice-exit-instead-of-D.patch
  Make-keycode-gen-output-reproducible-use.patch
  memory-clamp-cached-translation-in-case-.patch
  monitor-Fix-assertion-failure-on-shutdow.patch
  mptsas-Remove-unused-MPTSASState-pending.patch
  msf2-mac-switch-to-use-qemu_receive_pack.patch
  net-Fix-handling-of-id-in-netdev_add-and.patch
  net-introduce-qemu_receive_packet.patch
  pcnet-switch-to-use-qemu_receive_packet-.patch
  qemu-nbd-Use-SOMAXCONN-for-socket-listen.patch
  qemu-storage-daemon-Enable-object-add.patch
  rtl8139-switch-to-use-qemu_receive_packe.patch
  s390x-add-have_virtio_ccw.patch
  s390x-css-report-errors-from-ccw_dstream.patch
  s390x-Fix-stringop-truncation-issue-repo.patch
  s390x-modularize-virtio-gpu-ccw.patch
  s390x-move-S390_ADAPTER_SUPPRESSIBLE.patch
  s390x-pci-restore-missing-Query-PCI-Func.patch
  spice-app-avoid-crash-when-core-spice-mo.patch
  sungem-switch-to-use-qemu_receive_packet.patch
  target-arm-Don-t-decode-insns-in-the-XSc.patch
  target-arm-Fix-MTE0_ACTIVE.patch
  target-arm-Introduce-PREDDESC-field-defi.patch
  target-arm-Update-PFIRST-PNEXT-for-pred_.patch
  target-arm-Update-REV-PUNPK-for-pred_des.patch
  target-arm-Update-ZIP-UZP-TRN-for-pred_d.patch
  target-xtensa-fix-meson.build-rule-for-x.patch
  tcg-Use-memset-for-large-vector-byte-rep.patch
  tools-virtiofsd-Replace-the-word-whiteli.patch
  tx_pkt-switch-to-use-qemu_receive_packet.patch
  ui-vnc-Add-missing-lock-for-send_color_m.patch
  update-linux-headers-Include-const.h.patch
  Update-linux-headers-to-5.11-rc2.patch
  util-fix-use-after-free-in-module_load_o.patch
  vfio-ccw-Connect-the-device-request-noti.patch
  vhost-user-blk-fix-blkcfg-num_queues-end.patch
  viriofsd-Add-support-for-FUSE_HANDLE_KIL.patch
  virtiofsd-extract-lo_do_open-from-lo_ope.patch
  virtiofsd-optionally-return-inode-pointe.patch
  virtiofsd-prevent-opening-of-special-fil.patch
  virtiofs-drop-remapped-security.capabili.patch
  virtiofsd-Save-error-code-early-at-the-f.patch
  virtio-move-use-disabled-flag-property-t.patch
  virtio-pci-compat-page-aligned-ATS.patch
  xen-block-Fix-removal-of-backend-instanc.patch
* Mon Apr 19 2021 brogers@suse.com
- Include upstream patch designated as stable material and reviewed
  for applicability to include here
  mptsas-Remove-unused-MPTSASState-pending.patch
- Clarify in support documents that cpu-add was removed in this
  release from both the human monitor protocol (HMP) and QMP
  interfaces
* Thu Apr 15 2021 brogers@suse.com
- 6.0.0 qemu is about to be released. Add comments to the in-
  package support documents (supported.<arch>.txt) about the new
  deprecations as of that release as an early head's up for qemu
  users. These deprecations include these command-line options:
  - M option: kernel-irqchip=off
  - chardev tty
  - chardev paraport
  - enable-fips
  - writeconfig
  - spice password=string
* Tue Apr 13 2021 brogers@suse.com
- Include upstream patches designated as stable material and
  reviewed for applicability to include here. NOTE that the
  PIIX4 patch has migration implications: the change will also be
  applied to the SLE-15-SP2 qemu, and a live migration from that
  version to this SLE-15-SP3 qemu would require this patch to be
  applied for a successful migration if PIIX4 southbridge is used
  in the machine emulation (x86 i440fx)
  block-rbd-fix-memory-leak-in-qemu_rbd_co.patch
  block-rbd-Fix-memory-leak-in-qemu_rbd_co.patch
  cpu-core-Fix-help-of-CPU-core-device-typ.patch
  hw-arm-virt-acpi-build-Fix-GSIV-values-o.patch
  hw-block-fdc-Fix-fallback-property-on-sy.patch
  hw-isa-Kconfig-Add-missing-dependency-VI.patch
  hw-isa-piix4-Migrate-Reset-Control-Regis.patch
  hw-virtio-pci-Added-AER-capability.patch
  hw-virtio-pci-Added-counter-for-pcie-cap.patch
  s390x-css-report-errors-from-ccw_dstream.patch
  target-xtensa-fix-meson.build-rule-for-x.patch
  util-fix-use-after-free-in-module_load_o.patch
  virtio-pci-compat-page-aligned-ATS.patch
* Tue Mar 30 2021 brogers@suse.com
- Switch method of splitting off hw-s390x-virtio-gpu-ccw.so as a
  module to what was accepted upstream (bsc#1181103)
  * Patches dropped:
  hw-s390x-modularize-virtio-gpu-ccw.patch
  * Patches added:
  s390x-add-have_virtio_ccw.patch
  s390x-modularize-virtio-gpu-ccw.patch
  s390x-move-S390_ADAPTER_SUPPRESSIBLE.patch
* Tue Mar 23 2021 brogers@suse.com
- Fix OOB access in sdhci interface (CVE-2020-17380, bsc#1175144,
  CVE-2020-25085, bsc#1176681, CVE-2021-3409, bsc#1182282)
  hw-sd-sd-Actually-perform-the-erase-oper.patch
  hw-sd-sd-Fix-build-error-when-DEBUG_SD-i.patch
  hw-sd-sdhci-Correctly-set-the-controller.patch
  hw-sd-sdhci-Don-t-transfer-any-data-when.patch
  hw-sd-sdhci-Don-t-write-to-SDHC_SYSAD-re.patch
  hw-sd-sdhci-Limit-block-size-only-when-S.patch
  hw-sd-sdhci-Reset-the-data-pointer-of-s-.patch
  hw-sd-sd-Move-the-sd_block_-read-write-a.patch
  hw-sd-sd-Skip-write-protect-groups-check.patch
- Fix potential privilege escalation in virtiofsd tool
  (CVE-2021-20263, bsc#1183373)
  tools-virtiofsd-Replace-the-word-whiteli.patch
  viriofsd-Add-support-for-FUSE_HANDLE_KIL.patch
  virtiofsd-extract-lo_do_open-from-lo_ope.patch
  virtiofsd-optionally-return-inode-pointe.patch
  virtiofsd-prevent-opening-of-special-fil.patch
  virtiofs-drop-remapped-security.capabili.patch
  virtiofsd-Save-error-code-early-at-the-f.patch
- Fix OOB access (stack overflow) in rtl8139 NIC emulation
  (CVE-2021-3416, bsc#1182968)
  net-introduce-qemu_receive_packet.patch
  rtl8139-switch-to-use-qemu_receive_packe.patch
- Fix OOB access (stack overflow) in other NIC emulations
  (CVE-2021-3416)
  cadence_gem-switch-to-use-qemu_receive_p.patch
  dp8393x-switch-to-use-qemu_receive_packe.patch
  e1000-switch-to-use-qemu_receive_packet-.patch
  lan9118-switch-to-use-qemu_receive_packe.patch
  msf2-mac-switch-to-use-qemu_receive_pack.patch
  pcnet-switch-to-use-qemu_receive_packet-.patch
  sungem-switch-to-use-qemu_receive_packet.patch
  tx_pkt-switch-to-use-qemu_receive_packet.patch
- Fix heap overflow in MSIx emulation (CVE-2020-27821, bsc#1179686)
  memory-clamp-cached-translation-in-case-.patch
- Include upstream patches designated as stable material and
  reviewed for applicability to include here
  hw-arm-virt-Disable-pl011-clock-migratio.patch
  xen-block-Fix-removal-of-backend-instanc.patch
- Fix package scripts to not use hard coded paths for temporary
  working directories and log files (bsc#1182425)
* Wed Mar 17 2021 brogers@suse.com
- Fix s390x "mediated device is in use" error condition
  (bsc#1183634)
  update-linux-headers-Include-const.h.patch
  Update-linux-headers-to-5.11-rc2.patch
  vfio-ccw-Connect-the-device-request-noti.patch
* Tue Mar 16 2021 brogers@suse.com
- Fix DoS in e1000 emulated device (CVE-2021-20257 bsc#1182577)
  e1000-fail-early-for-evil-descriptor.patch
* Mon Mar 15 2021 brogers@suse.com
- Fix incorrect guest data in s390x PCI passthrough (bsc#1183372)
  s390x-pci-restore-missing-Query-PCI-Func.patch
* Fri Mar 12 2021 brogers@suse.com
- Include upstream patches designated as stable material and
  reviewed for applicability to include here
  lsilogic-Use-PCIDevice-exit-instead-of-D.patch
  vhost-user-blk-fix-blkcfg-num_queues-end.patch
- Fix potential privilege escalation in virtfs (CVE-2021-20181
  bsc#1182137)
  9pfs-Fully-restart-unreclaim-loop-CVE-20.patch
- Fix OOB access in vmxnet3 emulation (CVE-2021-20203 bsc#1181639)
  net-vmxnet3-validate-configuration-value.patch
* Fri Mar 12 2021 dimstar@opensuse.org
- Add #!ForceMultiversion to qemu.spec:
  + As the spec file defines different Version: fiels for various
    subpackages, we must instruct OBS to not ever reset the
    checkin-counter, as it would by defalut on a version increase.
    Resetting the version counter results in sub-packages reusing
    their VERSION-RELEASE from the past (e.g. qemu-ipxe is version
    1.0.0+, and upon checkin of a new qemu version, RELEASE is
    reset to 1.1, thus again producing
    qemu-ipxe-1.0.0+-1.1.noarch.rpm.
* Fri Mar  5 2021 brogers@suse.com
- Fix GCC11 compiler issue in brotli (edk2) code (boo#1181922)
  brotli-fix-actual-variable-array-paramet.patch
- Tweak a few submodule descriptions and summaries
- Fix a backward compatibility issue in ACPI data
  i386-acpi-restore-device-paths-for-pre-5.patch
* Wed Mar  3 2021 brogers@suse.com
- Add patch from IBM to improve modularization situation on s390
  where a new qemu module, hw-s390x-virtio-gpu-ccw.so, and a
  corresponding new qemu-hw-s390x-virtio-gpu-ccw subpackage, is
  split out (this parallels the hw-display-virtio-gpu-pci.so module).
  Split-provides file is also used to track this functionality
  splitout. Both the packages supplying the above mentioned modules
  now have a Requires on the qemu-hw-display-virtio-gpu package. It
  is anticipated that this change is going in upstream as well, and
  if done differently the plan is to update to the upstream
  implementation if possible (bsc#1181103)
  hw-s390x-modularize-virtio-gpu-ccw.patch
* Fri Feb 26 2021 brogers@suse.com
- Added a few more usability improvements for our git packaging
  workflow
* Fri Feb 19 2021 brogers@suse.com
- Fix issue of virtio-9p-ccw having been mistakenly dropped from
  qemu (bsc#1182496)
  hw-s390x-fix-build-for-virtio-9p-ccw.patch
* Thu Feb 18 2021 brogers@suse.com
- Tweaked some spec file details to be again compatible with quilt
  setup using the spec file as input
- Remove BuildRequires that were added in anticipation of building
  ovmf within this package. We have not taken that route
* Wed Feb 17 2021 brogers@suse.com
- Fix uninitialized variable in ipxe driver code (boo#1181922)
  ath5k-Add-missing-AR5K_EEPROM_READ-in-at.patch
- Add a few improvements to the git-based package workflow scripts
* Tue Feb 16 2021 brogers@suse.com
- Include additional upstream patches designated as stable material
  and reviewed for applicability to include here
  blockjob-Fix-crash-with-IOthread-when-bl.patch
  monitor-Fix-assertion-failure-on-shutdow.patch
  qemu-nbd-Use-SOMAXCONN-for-socket-listen.patch
  qemu-storage-daemon-Enable-object-add.patch
* Mon Feb  8 2021 brogers@suse.com
- Switch the modules qemu-ui-display-gpu and qemu-ui-display-gpu-pci
  from being an x86 only Recommends, to a Recommends for all arch's
  except s390x (boo#1181350)
- Fix qemu-hw-usb-smartcard to not be a Recommends for s390x
- Minor spec file tweaks for compatibility with upcoming spec file
  formatter
* Mon Feb  8 2021 brogers@suse.com
- Make note that this patch takes care of an OOB access in ARM
  interrupt handling (CVE-2021-20221 bsc#1181933)
  hw-intc-arm_gic-Fix-interrupt-ID-in-GICD.patch
* Fri Feb  5 2021 brogers@suse.com
- Include upstream patches designated as stable material and
  reviewed for applicability to include here
  block-Separate-blk_is_writable-and-blk_s.patch
  hw-intc-arm_gic-Fix-interrupt-ID-in-GICD.patch
  hw-net-lan9118-Fix-RX-Status-FIFO-PEEK-v.patch
  hw-timer-slavio_timer-Allow-64-bit-acces.patch
  net-Fix-handling-of-id-in-netdev_add-and.patch
  target-arm-Don-t-decode-insns-in-the-XSc.patch
  target-arm-Fix-MTE0_ACTIVE.patch
  target-arm-Introduce-PREDDESC-field-defi.patch
  target-arm-Update-PFIRST-PNEXT-for-pred_.patch
  target-arm-Update-REV-PUNPK-for-pred_des.patch
  target-arm-Update-ZIP-UZP-TRN-for-pred_d.patch
  tcg-Use-memset-for-large-vector-byte-rep.patch
  ui-vnc-Add-missing-lock-for-send_color_m.patch
  virtio-move-use-disabled-flag-property-t.patch
* Thu Feb  4 2021 brogers@suse.com
- binutils v2.36 has changed the handling of the assembler's
  - mx86-used-note, resulting in a build failure. To compensate, we
  now explicitly specify -mx86-used-note=no in the seabios Makefile
  (boo#1181775)
  build-be-explicit-about-mx86-used-note-n.patch
* Fri Jan 29 2021 brogers@suse.com
- Additional tweaks to ensure libvirt runs ok when
  qemu-hw-display-virtio-gpu package is not installed
* Fri Jan 29 2021 lma@suse.com
- Use '%%service_del_postun_without_restart' instead of
  '%%service_del_postun' to avoid "Failed to try-restart
  qemu-ga@.service" error while updating the qemu-guest-agent.
  (bsc#1178565)
* Tue Jan 26 2021 brogers@suse.com
- Fix two additional cases of qemu crashing due to qemu module
  packages not being loaded.
  qom-handle-case-of-chardev-spice-module-.patch
  spice-app-avoid-crash-when-core-spice-mo.patch
* Fri Jan 22 2021 brogers@suse.com
- Fix issue of qemu crashing (abort called) when virtio-gpu device
  is asked for and the qemu-hw-display-virtio-gpu package isn't
  installed. (bsc#1181103)
  module-for-virtio-gpu-pre-load-module-to.patch
- Add additional inter-module package dependencies, to reflect the
  current module dependencies (see qemu source file: util/module.c)
- As of v3.1.0 virt-manager, new VM's are created by default with
  audio/sound enabled, so it's time to reflect the need, at least
  in the spice case, by having spice-audio available when spice in
  general is used (boo#1180210 boo#1181132)
- Further refine package Recommends/Suggests based on architecture
- Remove no longer needed dependency on pwdutils (boo#1181235)
* Mon Jan 18 2021 brogers@suse.com
- Fix qemu-testsuite issue where white space processing gets
  handled differently under bash 5.1 (boo#1181054)
  iotests-Fix-_send_qemu_cmd-with-bash-5.1.patch
* Wed Jan 13 2021 brogers@suse.com
- Convert qemu-kvm from a script to a symlink. Using qemu-kvm to
  invoke the QEMU emulator has been deprecated for some time,
  but is still provided. It has as it's ancient origins a version
  of QEMU which had KVM acceleration enabled by default, and then
  recently, until now, it is a shell script which execs the QEMU
  emulator, adding '-machine accel=kvm' to the beginning of the
  list of command line options passed to the emulator.
  This method collides with the now preferred method of specifying
  acceleration options by using -accel. qemu-kvm is now changed to
  simply be a symlink to the same QEMU binary which the prior
  script exec'd. This new approach takes advantage of a built-in
  QEMU feature where if QEMU is invoked using a program name ending
  in 'kvm', KVM emulation is enabled. This approach is better in
  that it is more compatible with any other command line option
  that may be added for describing acceleration.
  For those who have modified qemu-kvm to add additional command
  line options, or take other actions in the context of the script
  you will now need to create an alternate script "emulator" to
  achieve the same result. Note that it's possible there may be
  some very subtle behavioral difference in the switch from a
  script to a symlink, but given that qemu-kvm is a deprecated
  package, we're not going to worry about that.
* Tue Jan  5 2021 brogers@suse.com
- Fix crash when spice used and the qemu-audio-spice package isn't
  installed (boo#1180210)
  audio-add-sanity-check.patch
- Add some stable patches from upstream
  block-Fix-deadlock-in-bdrv_co_yield_to_d.patch
  block-Fix-locking-in-qmp_block_resize.patch
  block-nfs-fix-int-overflow-in-nfs_client.patch
  block-Simplify-qmp_block_resize-error-pa.patch
  build-no-pie-is-no-functional-linker-fla.patch
* Tue Dec  8 2020 brogers@suse.com
- Update to v5.2.0: See http://wiki.qemu.org/ChangeLog/5.2
  Take note that ongoing feature deprecation is tracked at both
  http://wiki.qemu-project.org/Features/LegacyRemoval and in
  the deprecated.html file installed with the qemu package
  Some noteworthy changes:
  * Dropped system emulators: qemu-system-lm32, qemu-system-unicore32
  * Dropped linux user emulator: qemu-ppc64abi32
  * Added linux user emulator: qemu-extensaeb
  * Unicore32 and lm32 guest support dropped
  * New sub-packages (most due to ongoing modularization of QEMU):
  qemu-audio-spice, qemu-hw-chardev-spice, qemu-hw-display-virtio-vga,
  qemu-hw-display-virtio-gpu, qemu-hw-display-virtio-gpu-pci,
  qemu-ui-spice-core, qemu-ui-opengl, qemu-ivshmem-tools
  * x86: A new KVM feature which improves the handling of asynchronous page
  faults is available with -cpu ...,kvm-async-pf-int (requires Linux 5.8)
  * s390: More instructions emulated under TCG
  * PowerPC: nvdimm= machine option now functions correctly; misc improvements
  * ARM: new boards: mps2-an386 (Cortex-M4 based) and mps2-an500
  (Cortex-M7 based),  raspi3ap (the Pi 3 model A+), raspi0 (the Pi Zero)
  and raspi1ap (the Pi A+)
  * RISC-V: OpenSBI v0.8 included by default; Generic OpenSBI platform used
  when no -bios argument is supplied; Support for NUMA sockets on Virt
  and Spike Machines; Support for migrating machines; misc improvements
  * Misc NVMe improvements
  * The 'vhost-user-blk' export type has been added, allowing
  qemu-storage-daemon to act as a vhost-user-blk device backend
  * The SMBIOS OEM strings can now come from a file
  * 9pfs - misc performance related improvements
  * virtiofs - misc improvements
  * migration: The default migration bandwidth has been increased to 1Gbps
  (users are still encouraged to tune it to their own hardware); The new
  'calc-dirty-rate' and 'query-dirty-rate' QMP commands can help determine
  the likelihood of precopy migration success; TLS+multifd now supported
  for higher bandwidth encrypted migration; misc minor features added
  * Misc minor block features added
  * Misc doc improvements
  * qemu-microvm subpackage change: the bios-microvm.bin is now SeaBIOS based,
  and the qboot based on is now qboot.rom
  * elf2dmp is no longer part of qemu-tools (it was never intended to be
  a packaged binary)
  * Some subpackages which were 'Requires' are now 'Recommends', allowing for
  a smaller qemu packaging footprint if needed
  * Patches dropped (included in release tarball, unless otherwise noted):
  docs-fix-trace-docs-build-with-sphinx-3..patch (fixed differently)
  hw-hyperv-vmbus-Fix-32bit-compilation.patch
  linux-user-properly-test-for-infinite-ti.patch
  Switch-order-of-libraries-for-mpath-supp.patch (fixed differently)
  Conditionalize-ui-bitmap-installation-be.patch (fixed differently)
  hw-usb-hcd-xhci-Fix-GCC-9-build-warning.patch (no longer using gcc9)
  hw-usb-dev-mtp-Fix-GCC-9-build-warning.patch (no longer using gcc9)
  roms-Makefile-enable-cross-compile-for-b.patch (fixed with different patch)
  libvhost-user-handle-endianness-as-manda.patch
  virtio-add-vhost-user-fs-ccw-device.patch
  Fix-s-directive-argument-is-null-error.patch
  build-Workaround-compilation-error-with-.patch
  build-Be-explicit-about-fcommon-compiler.patch
  intel-Avoid-spurious-compiler-warning-on.patch
  golan-Add-explicit-type-casts-for-nodnic.patch
  Do-not-apply-WORKAROUND_CFLAGS-for-host-.patch
  ensure-headers-included-are-compatible-w.patch
  Enable-cross-compile-prefix-for-C-compil.patch (fixed differently)
  hw-net-net_tx_pkt-fix-assertion-failure-.patch
  hw-net-xgmac-Fix-buffer-overflow-in-xgma.patch
  s390x-protvirt-allow-to-IPL-secure-guest.patch
  usb-fix-setup_len-init-CVE-2020-14364.patch
  * Patches added:
  meson-install-ivshmem-client-and-ivshmem.patch
  Revert-roms-efirom-tests-uefi-test-tools.patch
  Makefile-Don-t-check-pc-bios-as-pre-requ.patch
  roms-Makefile-add-cross-file-to-qboot-me.patch
  qboot-add-cross.ini-file-to-handle-aarch.patch
  usb-Help-compiler-out-to-avoid-a-warning.patch
- In spec file, where reasonable, switch BuildRequires: XXX-devel
  to be pkgconfig(XXX') instead
- No longer disable link time optimization for qemu for x86. It looks like
  either the build service, qemu code changes and/or the switch to meson
  have resolved issues previously seen there. We still see problems for
  other architectures however.
- For the record, the following issues reported for SUSE SLE15-SP2
  are either fixed in this current package, or are otherwise no longer
  an issue: bsc#1172384 bsc#1174386 bsc#1174641 bsc#1174863 bsc#1175370
  bsc#1175441 bsc#1176494 CVE-2020-13361 CVE-2020-14364 CVE-2020-15863
  CVE-2020-16092 CVE-2020-24352
  and the following feature requests are satisfied by this package:
  jsc#SLE-13689 jsc#SEL-13780 jsc#SLE-13840
- To be more accurate, and to align with other qemu packaging
  practices, rename the qemu-s390 package to qemu-s390x. The old
  name (in the rpm namespace) is provided with a "Provides"
  directive, and an "Obsoletes" done against that name for prior
  qemu versions, as is standard practice (boo#1177764 jsc#SLE-17060)
- Take this opportunity to remove some ancient Split-Provides
  mechanisms which can't conceivably be needed any more:
  qemu-block-curl provided: qemu:%%_libdir/%%name/block-curl.so
  qemu-guest-agent provided: qemu:%%_bindir/qemu-ga
  qemu-tools provided: qemu:%%_libexecdir/qemu-bridge-helper
* Mon Nov 30 2020 brogers@suse.com
- Disable linux-user 'ls' test on 32 bit arm. It's failing with
  "Allocating guest commpage: Cannot allocate memory" error, which
  we should hunt down, but for now we don't want it to prevent the
  package from being built
* Tue Nov 24 2020 brogers@suse.com
- Be more careful about what directives are used for qemu-testsuite
* Tue Nov 24 2020 brogers@suse.com
- Fix some spec file 'Requires' statements to be accurate to the
  new model of relying on system-user-qemu and system-group-kvm to
  provide the needed users and groups
* Thu Nov 19 2020 milsav92@outlook.com
- Added io_uring support.
* Tue Nov 17 2020 brogers@suse.com
- A patch has been applied to virt-manager to handle qemu spice
  related modules not being present, so undo the change from Sep
  30, 2020. Once again qemu-hw-display-qxl and qemu-hw-usb-redirect
  are Recommends and not Required by the qemu package
  (boo#1157320 boo#1176517, boo#1178141)
- For jsc#SLE-11629, change qemu, qemu-tools, and qemu-guest-agent
  to rely on system-user-qemu and system-group-kvm to provide now
  static system UIDs and GID's for qemu user and group, and kvm
  group. This will make guest migration more seamless for new
  installations since there is no chance of having required ID's
  differ in value.
* Wed Nov  4 2020 lyan@suse.com
- Add virtio-fs support for s390x (jsc#SLE-13822)
  libvhost-user-handle-endianness-as-manda.patch
  virtio-add-vhost-user-fs-ccw-device.patch
* Wed Oct 14 2020 brogers@suse.com
- Note: As part of the "Close the Leap Gap" effort, it's been
  decided that our SDL2 support in qemu is not worth trying
  to maintain. Long ago SLE qemu stopped including SDL2 support and
  now we will do the same for the openSUSE releases going forward.
  Accordingly SDL2 options are now configured out, and the two sub-
  packages which are SDL2 specific, namely qemu-audio-sdl and
  qemu-ui-sdl, are no longer generated, and due to the rpm package
  conflicts used for those packages, they will be uninstalled from
  systems as qemu updates move forward
- Drop e2fsprogs-devel and libpcap-devel as BuildRequires packages.
  They have not actually been needed to build qemu for a very long
  time
- Add more forsplits files
* Wed Oct  7 2020 brogers@suse.com
- Create qemu-skiboot sub-package. Use update-alternatives mechanism
  to coordinate with opal-firmware (provided with skiboot package set)
  on the provider of the /usr/share/qemu/skiboot.lid firmware file.
  qemu-skiboot uses a priority of 15, while opal-firmware uses a
  priority of 10 (jsc#SLE-13240)
* Wed Sep 30 2020 brogers@suse.com
- Undo part of the split-provides recently done. We have to wait on
  virt-manager to handle qemu modularization better before we make
  qemu-hw-display-qxl and qemu-hw-usb-redirect non-required
  (boo#1157320 boo#1176517)
* Mon Sep 21 2020 brogers@suse.com
- Fix spec file, where a conditional macro didn't have the correct
  syntax (bsc#1176766)
* Tue Sep 15 2020 brogers@suse.com
- Change qemu-x86 packaging relationship with qemu-microvm from
  Requires to Recommends
* Thu Sep 10 2020 brogers@suse.com
- In an effort to "Close the Leap Gap", remove use of is_opensuse
  from the spec file, so that the same packages built for SLE can
  be reused for Leap. Some sub-packages will not be included for
  SLE which are included for Leap. They wil be provided in Package
  Hub for SLE users as unsupported packages. (jsc#SLE-11660,
  jsc#SLE-11661, jsc#SLE-11662, jsc#SLE-11691, jse#SLE-11692,
  jsc#SLE-11894)
* Thu Sep 10 2020 brogers@suse.com
- Add infrastructure to do package splits when split-off package
  isn't required and doesn't (otherwise) include any previously
  installed files. This version of qemu has split out non-essential
  functionality into loadable modules, as noted in Aug 20, 2020 log
  entry, which describes the emergency Split-Provides. That approach
  will be superseded by this planned approach, and those dummy doc
  files will be removed in time
  Here is the new mapping:
  subpackage            continuity file provided (files are dummies)
  ==========            ============================================
  qemu-chardev-baum     /usr/share/qemu/forsplits/00
  qemu-hw-display-qxl   /usr/share/qemu/forsplits/01
  qemu-hw-usb-redirect  /usr/share/qemu/forsplits/02
  qemu-hw-usb-smartcard /usr/share/qemu/forsplits/03
* Wed Sep  9 2020 brogers@suse.com
- Fix path of qemu-pr-helper. It was a mistake to move it from
  %%_bindir to _libexecdir. In more recent qemu code it's been moved
  back, so undo this mistake by providing it at the same location
  as it has been all along
* Tue Sep  1 2020 brogers@suse.com
- For SLE15-SP3, note that this update to v5.1.0 is a step towards
  fulfilling jsc#SLE-13689, which asks for qemu v5.2.0 or higher
* Wed Aug 26 2020 brogers@suse.com
- Fix some shell syntax in update_git.sh, esp. an issue exposed by
  the most recent patch added
* Tue Aug 25 2020 brogers@suse.com
- Fix OOB access while processing USB packets (CVE-2020-14364
  bsc#1175441)
  usb-fix-setup_len-init-CVE-2020-14364.patch
- Re-sync openSUSE and SUSE SLE qemu packages. This changes file
  is the openSUSE one with this entry providing the intervening
  SLE CVE, JIRA, and bugzilla references, which are still addressed
  in this package, and not yet called out in this changes file.
  * CVE-2020-1983  CVE-2020-10761 CVE-2020-13361 CVE-2020-13362
  CVE-2020-13659 CVE-2020-13800
  * bsc#1167816 bsc#1170940 boo#1171712 bsc#1172383 bsc#1172384
  bsc#1172386 bsc#1172495 bsc#1172710
  * Patches dropped (SLE) (included in current release tarball):
  exec-set-map-length-to-zero-when-returni.patch
  i386-acpi-Remove-_HID-from-the-SMBus-ACP.patch
  megasas-use-unsigned-type-for-reply_queu.patch
* Thu Aug 20 2020 brogers@suse.com
- Fix compilation errors seen with pre-release gcc 11
  qht-Revert-some-constification-in-qht.c.patch
  Revert-qht-constify-qht_statistics_init.patch
  help-compiler-out-by-initializing-array.patch
  s390x-Fix-stringop-truncation-issue-repo.patch
- Add Split-Provides mechanism, using doc files which were moved
  in v5.1.0. This allows for the new subpackages to be selected for
  install when the v5.0.0 qemu is updated. These new subpackages are
  not marked as "Required" by any packages, in an effort to reduce
  the dependencies of the core qemu components (boo#1175320)
  v5.0.0 qemu file mapping is provided as follows:
  subpackage            continuity file provided (files are dummies)
  ==========            ============================================
  qemu-chardev-baum     /usr/share/doc/packages/qemu/qemu-ga-ref.html
  qemu-hw-display-qxl   /usr/share/doc/packages/qemu/qemu-ga-ref.txt
  qemu-hw-usb-redirect  /usr/share/doc/packages/qemu/qemu-qmp-ref.html
  qemu-hw-usb-smartcard /usr/share/doc/packages/qemu/qemu-qmp-ref.txt
* Tue Aug 18 2020 dimstar@opensuse.org
- Fix wrong usage of %%{_libexecdir} for systemd owned paths below
  %%{_prefix}/lib.
* Tue Aug 11 2020 brogers@suse.com
- Update to v5.1.0: See http://wiki.qemu.org/ChangeLog/5.1
  Take note that ongoing feature deprecation is tracked at both
  http://wiki.qemu-project.org/Features/LegacyRemoval and in
  the deprecated.html file installed with the qemu package
  Some noteworthy changes:
  * s390: Protected virtualization (secure execute) is fully merged
  upstream
  * s390: vfio-ccw devices no longer require setting the allow
  prefetch bit in the ORB, but is still dependent on host kernel
  support
  * s390: vfio-ccw now has basic support for relaying path state
  changes to the guest
  * PowerPC: pseries: NVDIMMs require label-size property
  * PowerPC: pseries: POWER10 support
  * PowerPC: added interface to inject POWER style NMIs
  * ARM: new board: sonorapass-bmc
  * ARM: new emulated features: ARMv8.2-TTSUXN, ARMv8.5-MemTag
  * ARM: Raspberry Pi boards now support a USB controller
  * ARM: virt board now supports hot-remove memory
  * RISC-V lots of improvements
  * qemu-img resize now requires -shrink to shrinking raw images
  * The mem parameter of the -numa option is no longer recognized
  starting with 5.1 machine types - instead use the memdev parameter
  * The ACPI WAET table is now exposed to guests
  * The max blocksize for virtual storage device is now 2 MiB
  * NVMe improvements
  * Crypto subsystem improvements
  * Block backends and tools: Numerous improvements and fixes
  * Firmware updates: SeaBIOS (essentially v1.14.0), OpenBIOS, SLOF
  (20200717), OpenSBI (v0.7)
  * Patches dropped (upstream unless otherwise noted):
  ati-vga-check-mm_index-before-recursive-.patch
  audio-fix-wavcapture-segfault.patch
  es1370-check-total-frame-count-against-c.patch
  exec-set-map-length-to-zero-when-returni.patch
  gcc10-maybe-uninitialized.patch
  hw-vfio-pci-quirks-Fix-broken-legacy-IGD.patch
  megasas-use-unsigned-type-for-reply_queu.patch
  nbd-server-Avoid-long-error-message-asse.patch
  ppc-spapr_caps-Don-t-disable-cap_cfpc-on.patch
  s390x-Add-SIDA-memory-ops.patch
  s390x-Add-unpack-facility-feature-to-GA1.patch
  s390x-Move-diagnose-308-subcodes-and-rcs.patch
  s390x-protvirt-Add-migration-blocker.patch
  s390x-protvirt-Disable-address-checks-fo.patch
  s390x-protvirt-Handle-SIGP-store-status-.patch
  s390x-protvirt-Inhibit-balloon-when-swit.patch
  s390x-protvirt-KVM-intercept-changes.patch
  s390x-protvirt-Move-diag-308-data-over-S.patch
  s390x-protvirt-Move-IO-control-structure.patch
  s390x-protvirt-Move-STSI-data-over-SIDAD.patch
  s390x-protvirt-SCLP-interpretation.patch
  s390x-protvirt-Set-guest-IPL-PSW.patch
  s390x-protvirt-Support-unpack-facility.patch
  s390x-s390-virtio-ccw-Fix-build-on-syste.patch
  Sync-pv.patch
  tests-Disable-some-block-tests-for-now.patch (no longer needed)
  vga-fix-cirrus-bios.patch
  virtiofsd-add-rlimit-nofile-NUM-option.patch
  virtiofsd-stay-below-fs.file-max-sysctl-.patch
  * Patches renamed:
  build-Do-not-apply-WORKAROUND_CFLAGS-for.patch
  - > Do-not-apply-WORKAROUND_CFLAGS-for-host-.patch
  build-Fix-s-directive-argument-is-null-e.patch
  - > Fix-s-directive-argument-is-null-error.patch
  * Patches added:
  hw-hyperv-vmbus-Fix-32bit-compilation.patch
- New subpackages, due to modularization: qemu-chardev-baum,
  qemu-hw-display-qxl, qemu-hw-usb-redirect, qemu-hw-usb-smartcard
- Configure to use "system" libslirp and libdaxctl (libnvdimm)
  when available
* Thu Jul 23 2020 lyan@suse.com
- Don't disable cap_cfpc on POWER8 by default (bsc#1174374)
  ppc-spapr_caps-Don-t-disable-cap_cfpc-on.patch
* Tue Jul 21 2020 brogers@suse.com
- Updating to Sphinx v3.1.2 in Factory is exposing an issue in
  qemu doc sources. Fix it
  docs-fix-trace-docs-build-with-sphinx-3..patch
* Wed Jul  8 2020 brogers@suse.com
- Fix DoS possibility in ati-vga emulation (CVE-2020-13800
  bsc#1172495)
  ati-vga-check-mm_index-before-recursive-.patch
- Fix DoS possibility in Network Block Device (nbd) support
  infrastructure (CVE-2020-10761 bsc#1172710)
  nbd-server-Avoid-long-error-message-asse.patch
- Fix null pointer dereference possibility (DoS) in MegaRAID SAS
  8708EM2 emulation (CVE-2020-13659 bsc#1172386)
  exec-set-map-length-to-zero-when-returni.patch
- Fix OOB access possibility in MegaRAID SAS 8708EM2 emulation
  (CVE-2020-13362 bsc#1172383)
  megasas-use-unsigned-type-for-reply_queu.patch
- Fix legacy IGD passthrough
  hw-vfio-pci-quirks-Fix-broken-legacy-IGD.patch
* Wed Jul  1 2020 brogers@suse.com
- The latest gcc10 available in Factory has the fix for the
  issue this patch was created to avoid, so drop it
  build-Work-around-gcc10-bug-by-not-using.patch
* Sun Jun 28 2020 brogers@suse.com
- Switch to upstream versions of some patches we carry
  add-enum-cast-to-avoid-gcc10-warning.patch
  - > golan-Add-explicit-type-casts-for-nodnic.patch
  Be-explicit-about-fcommon-compiler-direc.patch
  - > build-Be-explicit-about-fcommon-compiler.patch
  Do-not-apply-WORKAROUND_CFLAGS-for-host-.patch
  - > build-Do-not-apply-WORKAROUND_CFLAGS-for.patch
  Fix-s-directive-argument-is-null-error.patch
  - > build-Fix-s-directive-argument-is-null-e.patch
  Workaround-compilation-error-with-gcc-9..patch
  - > build-Workaround-compilation-error-with-.patch
  work-around-gcc10-problem-with-zero-leng.patch
  - > intel-Avoid-spurious-compiler-warning-on.patch
- Fix vgabios issue for cirrus graphics emulation, which
  effectively downgraded it to standard VGA behavior
  vga-fix-cirrus-bios.patch
* Fri Jun 12 2020 brogers@suse.com
- Fix OOB access possibility in ES1370 audio device emulation
  (CVE-2020-13361 bsc#1172384)
  es1370-check-total-frame-count-against-c.patch
* Tue Jun  2 2020 brogers@suse.com
- Work around gcc 10 bug (boo#1172411)
  build-Work-around-gcc10-bug-by-not-using.patch
* Mon Jun  1 2020 brogers@suse.com
- Now that gcc10 compatibility is figured out, remove NO_WERROR=1
  again from ipxe make.
* Tue May 26 2020 brogers@suse.com
- Fix segfault when doing HMP wavcapture (boo#1171712)
  audio-fix-wavcapture-segfault.patch
* Tue May  5 2020 brogers@suse.com
- Fix DoS in virtiofsd, where a FUSE client could exhaust the
  number of available open files on the host (CVE-2020-10717
  bsc#1171110)
  virtiofsd-add-rlimit-nofile-NUM-option.patch
  virtiofsd-stay-below-fs.file-max-sysctl-.patch
* Tue May  5 2020 brogers@suse.com
- Add more fixes for gcc10 compatibility: Use NO_WERROR=1 when
  building ipxe sources, at least until we get gcc10 compatibility
  figured out. Also add patch for explicitly using -fcommon
  (boo#1171140)
  Be-explicit-about-fcommon-compiler-direc.patch
  and fix for tighter enum compatibility checking (boo#1171139)
  add-enum-cast-to-avoid-gcc10-warning.patch
  and a work around for what seems to be a compiler regression
  (boo#1171123)
  work-around-gcc10-problem-with-zero-leng.patch
* Tue Apr 28 2020 brogers@suse.com
- Update to v5.0.0: See http://wiki.qemu.org/ChangeLog/5.0
  Take note that ongoing feature deprecation is tracked at both
  http://wiki.qemu-project.org/Features/LegacyRemoval and in
  the deprecated.html file installed with the qemu package
  Some noteworthy changes:
  * x86: EPYC-Rome vcpu model
  * x86: vcpu model fixes for EPYC, Denverton, and Icelake-Server
  * s390: (as previously mentioned) Protected Virtualization support:
  start and control guest in secure mode (bsc#1167075 jsc#SLE-7407)
  * s390: support for Adapter Interrupt Suppression while running in
  KVM mode
  * PowerPC: pseries: NVDIMMs with file backend supported
  * PowerPC: powernv: KVM guests now runnable under TCG emulation
  * PowerPC: powernv: Basic POWER10 support
  * ARM: new boards: tacoma-bmc, Netduindo Plus 2, Orangepi PC
  * ARM: 'virt' machine now supports vTPM and virtio-iommu devices
  * ARM:Cortex-M7 CPU support
  * ARM: Lots of architecture features now emulated
  * ARM: TPM supported
  * ARM: Timekeeping improvements
  * ARM: LOTS more - refer to upstream changelog
  * virtio-iommu
  * VNC compatibility with noVNC improved
  * Support for using memory backends for main/"built-in" guest RAM
  * hostmem backends can now specify prealloc thread count
  * Better Azure compatibility of VHD images
  * Ceph namespaces supported
  * Compress block filter driver can create compressed backup images
  * virtiofsd availble for host filesystem passthrough
  * Improved html based documentation is provided with this release
  * Live migration support for external processes running on QEMU D-Bus
  * Patches dropped (upstream unless otherwise noted):
  i386-Add-MSR-feature-bit-for-MDS-NO.patch
  i386-Add-macro-for-stibp.patch
  i386-Add-new-CPU-model-Cooperlake.patch
  arm-arm-powerctl-set-NSACR.-CP11-CP10-bi.patch
  iotests-Skip-test-060-if-it-is-not-possi.patch
  iotests-Skip-test-079-if-it-is-not-possi.patch
  Revert-qemu-options.hx-Update-for-reboot.patch
  iotests-Provide-a-function-for-checking-.patch
  Fix-double-free-issue-in-qemu_set_log_fi.patch
  iotests-Fix-IMGOPTSSYNTAX-for-nbd.patch
  virtio-blk-fix-out-of-bounds-access-to-b.patch
  block-Activate-recursively-even-for-alre.patch
  i386-Resolve-CPU-models-to-v1-by-default.patch
  numa-properly-check-if-numa-is-supported.patch
  vhost-user-gpu-Drop-trailing-json-comma.patch
  display-bochs-display-fix-memory-leak.patch
  hw-arm-smmuv3-Apply-address-mask-to-line.patch
  hw-arm-smmuv3-Correct-SMMU_BASE_ADDR_MAS.patch
  hw-arm-smmuv3-Check-stream-IDs-against-a.patch
  hw-arm-smmuv3-Align-stream-table-base-ad.patch
  hw-arm-smmuv3-Use-correct-bit-positions-.patch
  hw-arm-smmuv3-Report-F_STE_FETCH-fault-a.patch
  block-Add-bdrv_qapi_perm_to_blk_perm.patch
  blkdebug-Allow-taking-unsharing-permissi.patch
  virtio-add-ability-to-delete-vq-through-.patch
  virtio-update-queue-size-on-guest-write.patch
  virtio-don-t-enable-notifications-during.patch
  numa-Extend-CLI-to-provide-initiator-inf.patch
  numa-Extend-CLI-to-provide-memory-latenc.patch
  numa-Extend-CLI-to-provide-memory-side-c.patch
  hmat-acpi-Build-Memory-Proximity-Domain-.patch
  hmat-acpi-Build-System-Locality-Latency-.patch
  hmat-acpi-Build-Memory-Side-Cache-Inform.patch
  tests-numa-Add-case-for-QMP-build-HMAT.patch
  qcow2-bitmaps-fix-qcow2_can_store_new_di.patch
  backup-top-Begin-drain-earlier.patch
  virtio-mmio-update-queue-size-on-guest-w.patch
  virtio-net-delete-also-control-queue-whe.patch
  intel_iommu-a-fix-to-vtd_find_as_from_bu.patch
  target-i386-Add-new-bit-definitions-of-M.patch
  target-i386-Add-missed-features-to-Coope.patch
  hw-i386-pc-fix-regression-in-parsing-vga.patch
  migration-test-ppc64-fix-FORTH-test-prog.patch
  target-arm-Return-correct-IL-bit-in-merg.patch
  target-arm-Set-ISSIs16Bit-in-make_issinf.patch
  runstate-ignore-finishmigrate-prelaunch-.patch
  migration-Rate-limit-inside-host-pages.patch
  m68k-Fix-regression-causing-Single-Step-.patch
  Revert-vnc-allow-fall-back-to-RAW-encodi.patch
  vnc-prioritize-ZRLE-compression-over-ZLI.patch
  target-i386-kvm-initialize-feature-MSRs-.patch
  s390x-adapter-routes-error-handling.patch
  iscsi-Cap-block-count-from-GET-LBA-STATU.patch
  block-backup-fix-memory-leak-in-bdrv_bac.patch
  tpm-ppi-page-align-PPI-RAM.patch
  hw-intc-arm_gicv3_kvm-Stop-wrongly-progr.patch
  target-arm-fix-TCG-leak-for-fcvt-half-do.patch
  block-fix-memleaks-in-bdrv_refresh_filen.patch
  block-backup-top-fix-failure-path.patch
  iotests-add-test-for-backup-top-failure-.patch
  audio-oss-fix-buffer-pos-calculation.patch
  target-arm-monitor-query-cpu-model-expan.patch
  block-fix-crash-on-zero-length-unaligned.patch
  block-Fix-VM-size-field-width-in-snapsho.patch
  target-arm-Correct-definition-of-PMCRDP.patch
  block-nbd-extract-the-common-cleanup-cod.patch
  block-nbd-fix-memory-leak-in-nbd_open.patch
  virtio-crypto-do-delete-ctrl_vq-in-virti.patch
  virtio-pmem-do-delete-rq_vq-in-virtio_pm.patch
  vhost-user-blk-delete-virtioqueues-in-un.patch
  hw-arm-cubieboard-use-ARM-Cortex-A8-as-t.patch
  pc-bios-s390x-Save-iplb-location-in-lowc.patch
  iotests-Fix-nonportable-use-of-od-endian.patch
  block-qcow2-threads-fix-qcow2_decompress.patch
  job-refactor-progress-to-separate-object.patch
  block-block-copy-fix-progress-calculatio.patch
  block-io-fix-bdrv_co_do_copy_on_readv.patch
  scsi-qemu-pr-helper-Fix-out-of-bounds-ac.patch
  target-ppc-Fix-rlwinm-on-ppc64.patch
  compat-disable-edid-on-correct-virtio-gp.patch
  ppc-ppc405_boards-Remove-unnecessary-NUL.patch
  block-Avoid-memleak-on-qcow2-image-info-.patch
  block-bdrv_set_backing_bs-fix-use-after-.patch
  hmp-vnc-Fix-info-vnc-list-leak.patch
  migration-colo-fix-use-after-free-of-loc.patch
  migration-ram-fix-use-after-free-of-loca.patch
  qcow2-List-autoclear-bit-names-in-header.patch
  sheepdog-Consistently-set-bdrv_has_zero_.patch
  target-arm-Fix-PAuth-sbox-functions.patch
  tcg-i386-Fix-INDEX_op_dup2_vec.patch
  net-tulip-check-frame-size-and-r-w-data-.patch
  target-i386-do-not-set-unsupported-VMX-s.patch
  spapr-Fix-failure-path-for-attempting-to.patch
  ati-vga-Fix-checks-in-ati_2d_blt-to-avoi.patch
  xen-block-Fix-double-qlist-remove-and-re.patch
  vpc-Don-t-round-up-already-aligned-BAT-s.patch
  target-xtensa-fix-pasto-in-pfwait.r-opco.patch
  aio-wait-delegate-polling-of-main-AioCon.patch
  async-use-explicit-memory-barriers.patch
  tcg-mips-mips-sync-encode-error.patch
  vhost-user-gpu-Release-memory-returned-b.patch
  vga-Raise-VRAM-to-16-MiB-for-pc-0.15-and.patch (no pc-0.15)
  hw-i386-disable-smbus-migration-for-xenf.patch
  s390x-Don-t-do-a-normal-reset-on-the-ini.patch
  s390x-Move-reset-normal-to-shared-reset-.patch
  s390x-Move-initial-reset.patch
  s390x-Move-clear-reset.patch
  s390x-kvm-Make-kvm_sclp_service_call-voi.patch
  s390x-ipl-Consolidate-iplb-validity-chec.patch
  s390x-Beautify-diag308-handling.patch
  s390x-Add-missing-vcpu-reset-functions.patch
  s390-sclp-improve-special-wait-psw-logic.patch
  vhost-correctly-turn-on-VIRTIO_F_IOMMU_P.patch
  util-add-slirp_fmt-helpers.patch
  slirp-use-correct-size-while-emulating-I.patch
  tcp_emu-Fix-oob-access.patch
  slirp-use-correct-size-while-emulating-c.patch
  tcp_emu-fix-unsafe-snprintf-usages.patch
- For SLE builds, leverage the html documentation by adding a link
  to the SUSE specific support documentation (the *.txt support doc
  was slightly tweaked to be acceptable as reStructuredText for
  conversion to html)
  docs-add-SUSE-support-statements-to-html.patch
* Mon Apr 27 2020 brogers@suse.com
-Fix potential DoS in ATI VGA emulation (CVE-2020-11869
  bsc#1170537)
  ati-vga-Fix-checks-in-ati_2d_blt-to-avoi.patch
* Wed Apr 22 2020 brogers@suse.com
- Minor tweaks to patches and support doc
* Tue Apr 21 2020 mliska@suse.cz
- Add gcc10-maybe-uninitialized.patch in order to fix
  boo#1169728.
* Fri Apr 17 2020 brogers@suse.com
- Include upstream patches targeted for the next stable release
  (bug fixes only)
  spapr-Fix-failure-path-for-attempting-to.patch
  target-i386-do-not-set-unsupported-VMX-s.patch
  target-xtensa-fix-pasto-in-pfwait.r-opco.patch
  tcg-i386-Fix-INDEX_op_dup2_vec.patch
  tcg-mips-mips-sync-encode-error.patch
  vhost-user-gpu-Release-memory-returned-b.patch
  vpc-Don-t-round-up-already-aligned-BAT-s.patch
  xen-block-Fix-double-qlist-remove-and-re.patch
- Fix bug causing weak encryption in PAuth for ARM
  (CVE-2020-10702 bsc#1168681)
  target-arm-Fix-PAuth-sbox-functions.patch
- Fix OOB in tulip NIC emulation (CVE-2020-11102 bsc#1168713
  net-tulip-check-frame-size-and-r-w-data-.patch
- Note that previously included patch addresses CVE-2020-1711
  and bsc#1166240
  iscsi-Cap-block-count-from-GET-LBA-STATU.patch
- Include performance improvement (and related?) patch
  aio-wait-delegate-polling-of-main-AioCon.patch
  async-use-explicit-memory-barriers.patch
- Rework previous patch at Olaf H.'s direction
  hw-i386-disable-smbus-migration-for-xenf.patch
- Eliminate is_opensuse usage in producing seabios version string
  what we are doing here is just replacing the upstream string
  with one indicating that the openSUSE build service built it,
  and so just leave it as "-rebuilt.opensuse.org"
- Alter algorithm used to produce "unique" symbol for coordinating
  qemu with the optional modules it may load. This is a reasonable
  relaxation for broader compatibility
  configure-remove-pkgversion-from-CONFIG_.patch
- Tweak supported.*.txt for latest deprecations, and other fixes
- Tweak update_git.sh, config.sh
* Mon Apr  6 2020 brogers@suse.com
- One more fix is needed for: s390x Protected Virtualization support
  - start and control guest in secure mode (bsc#1167075 jsc#SLE-7407)
  s390x-s390-virtio-ccw-Fix-build-on-syste.patch
* Thu Mar 26 2020 brogers@suse.com
- Include upstream patches targeted for the next stable release
  (bug fixes only)
  block-Avoid-memleak-on-qcow2-image-info-.patch
  block-bdrv_set_backing_bs-fix-use-after-.patch
  hmp-vnc-Fix-info-vnc-list-leak.patch
  migration-colo-fix-use-after-free-of-loc.patch
  migration-ram-fix-use-after-free-of-loca.patch
  ppc-ppc405_boards-Remove-unnecessary-NUL.patch
  qcow2-List-autoclear-bit-names-in-header.patch
  scsi-qemu-pr-helper-Fix-out-of-bounds-ac.patch
  sheepdog-Consistently-set-bdrv_has_zero_.patch
* Tue Mar 24 2020 brogers@suse.com
- Note The previous set of s390x patches also includes the fix for:
  bsc#1167445
* Fri Mar 20 2020 brogers@suse.com
- Include upstream patches targeted for the next stable release
  (bug fixes only)
  block-io-fix-bdrv_co_do_copy_on_readv.patch
  compat-disable-edid-on-correct-virtio-gp.patch
  target-ppc-Fix-rlwinm-on-ppc64.patch
  vhost-correctly-turn-on-VIRTIO_F_IOMMU_P.patch
- s390x Protected Virtualization support - start and control guest
  in secure mode. (note: binary patch from patch series dropped since
  for s390x we rebuild the patched binary anyways) (bsc#1167075
  jsc#SLE-7407)
  s390-sclp-improve-special-wait-psw-logic.patch
  s390x-Add-missing-vcpu-reset-functions.patch
  s390x-Add-SIDA-memory-ops.patch
  s390x-Add-unpack-facility-feature-to-GA1.patch
  s390x-Beautify-diag308-handling.patch
  s390x-Don-t-do-a-normal-reset-on-the-ini.patch
  s390x-ipl-Consolidate-iplb-validity-chec.patch
  s390x-kvm-Make-kvm_sclp_service_call-voi.patch
  s390x-Move-clear-reset.patch
  s390x-Move-diagnose-308-subcodes-and-rcs.patch
  s390x-Move-initial-reset.patch
  s390x-Move-reset-normal-to-shared-reset-.patch
  s390x-protvirt-Add-migration-blocker.patch
  s390x-protvirt-Disable-address-checks-fo.patch
  s390x-protvirt-Handle-SIGP-store-status-.patch
  s390x-protvirt-Inhibit-balloon-when-swit.patch
  s390x-protvirt-KVM-intercept-changes.patch
  s390x-protvirt-Move-diag-308-data-over-S.patch
  s390x-protvirt-Move-IO-control-structure.patch
  s390x-protvirt-Move-STSI-data-over-SIDAD.patch
  s390x-protvirt-SCLP-interpretation.patch
  s390x-protvirt-Set-guest-IPL-PSW.patch
  s390x-protvirt-Support-unpack-facility.patch
  Sync-pv.patch
* Fri Mar 20 2020 lyan@suse.com
- Fix the issue that s390x could not read IPL channel program when using
  dasd as boot device (bsc#1163140)
  pc-bios-s390x-Save-iplb-location-in-lowc.patch
* Mon Mar 16 2020 brogers@suse.com
- Fix potential OOB accesses in slirp (CVE-2020-8608 bsc#1163018
  bsc#1161066 CVE-2020-7039)
  slirp-use-correct-size-while-emulating-c.patch
  slirp-use-correct-size-while-emulating-I.patch
  tcp_emu-Fix-oob-access.patch
  tcp_emu-fix-unsafe-snprintf-usages.patch
  util-add-slirp_fmt-helpers.patch
- Replace this patch with upstream version
  target-arm-monitor-query-cpu-model-expan.patch
* Thu Mar 12 2020 brogers@suse.com
- Include upstream patches targeted for the next stable release
  (bug fixes only)
  audio-oss-fix-buffer-pos-calculation.patch
  blkdebug-Allow-taking-unsharing-permissi.patch
  block-Add-bdrv_qapi_perm_to_blk_perm.patch
  block-backup-top-fix-failure-path.patch
  block-block-copy-fix-progress-calculatio.patch
  block-fix-crash-on-zero-length-unaligned.patch
  block-fix-memleaks-in-bdrv_refresh_filen.patch
  block-Fix-VM-size-field-width-in-snapsho.patch
  block-nbd-extract-the-common-cleanup-cod.patch
  block-nbd-fix-memory-leak-in-nbd_open.patch
  block-qcow2-threads-fix-qcow2_decompress.patch
  hw-arm-cubieboard-use-ARM-Cortex-A8-as-t.patch
  hw-intc-arm_gicv3_kvm-Stop-wrongly-progr.patch
  iotests-add-test-for-backup-top-failure-.patch
  iotests-Fix-nonportable-use-of-od-endian.patch
  job-refactor-progress-to-separate-object.patch
  target-arm-Correct-definition-of-PMCRDP.patch
  target-arm-fix-TCG-leak-for-fcvt-half-do.patch
  tpm-ppi-page-align-PPI-RAM.patch
  vhost-user-blk-delete-virtioqueues-in-un.patch
  virtio-add-ability-to-delete-vq-through-.patch
  virtio-crypto-do-delete-ctrl_vq-in-virti.patch
  virtio-pmem-do-delete-rq_vq-in-virtio_pm.patch
* Thu Feb 20 2020 brogers@suse.com
- Add Obsoletes directive for qemu-audio-sdl and qemu-ui-sdl since
  for a qemu package upgrade from SLE12-SP5, support for SDL is
  dropped
* Wed Feb 19 2020 brogers@suse.com
- Fix xenfv migration from xen host with pre-v4.0 qemu. We had
  previously dropped a similar patch, but have decided that for now
  we need to go with this type of solution (bsc#1159755)
  hw-i386-disable-smbus-migration-for-xenf.patch
* Tue Feb  4 2020 lyan@suse.com
- Avoid query-cpu-model-expansion crashed qemu when using
  machine type none, patch is queued in upstream now, will
  update commit id later (bsc#1159443)
  target-arm-monitor-query-cpu-model-expan.patch
* Mon Feb  3 2020 dimstar@opensuse.org
- BuildRequire pkgconfig(libudev) instead of libudev-devel: Allow
  OBS to shortcut through -mini flavors.
* Tue Jan 28 2020 brogers@suse.com
- Stop using system membarriers (ie switch from --enable-membarrier
  to --disable-membarrier). This is a blocker for using qemu in the
  context of containers (boo#1130134 jsc#SLE-11089)
- Drop this recently added patch - in consultation with upstream it
  was decided it needed to be solved a different way (bsc#1159755)
  hw-i386-disable-smbus-migration-for-xenf.patch
- Include upstream patches targeted for the next stable release
  (bug fixes only)
  block-backup-fix-memory-leak-in-bdrv_bac.patch
  iscsi-Cap-block-count-from-GET-LBA-STATU.patch
  s390x-adapter-routes-error-handling.patch
  target-i386-kvm-initialize-feature-MSRs-.patch
* Fri Jan 24 2020 brogers@suse.com
- Include upstream patches targeted for the next stable release
  (bug fixes only)
  hw-i386-pc-fix-regression-in-parsing-vga.patch
  m68k-Fix-regression-causing-Single-Step-.patch
  migration-Rate-limit-inside-host-pages.patch
  migration-test-ppc64-fix-FORTH-test-prog.patch
  Revert-vnc-allow-fall-back-to-RAW-encodi.patch
  runstate-ignore-finishmigrate-prelaunch-.patch
  target-arm-Return-correct-IL-bit-in-merg.patch
  target-arm-Set-ISSIs16Bit-in-make_issinf.patch
  vnc-prioritize-ZRLE-compression-over-ZLI.patch
* Fri Jan 24 2020 dimstar@opensuse.org
- BuildRequire pkconfig(systemd) instead of systemd: allow OBS to
  shortcut through the -mini flavors.
- Use systemd_ordering in place of systemd_requires: systemd is
  never a strict requirement for qemu; but when installing qemu on
  a systemd-managed system, we want system to be present first.
* Tue Jan 14 2020 brogers@suse.com
- Fix xenfv migration from xen host with pre-v4.0 qemu (bsc#1159755)
  hw-i386-disable-smbus-migration-for-xenf.patch
* Tue Jan 14 2020 ohering@suse.de
- Create files within bundles.tar.xz with fixed timestamp and uid
* Sat Jan 11 2020 brogers@suse.com
- Add a %%bcond_without system_membarrier along with related
  processing to the spec file, to better investigate running QEMU
  with the --disable-membarrier configure option
* Fri Jan 10 2020 brogers@suse.com
- Include upstream patches targeted for the next stable release
  (bug fixes only)
  arm-arm-powerctl-set-NSACR.-CP11-CP10-bi.patch
  backup-top-Begin-drain-earlier.patch
  block-Activate-recursively-even-for-alre.patch
  display-bochs-display-fix-memory-leak.patch
  Fix-double-free-issue-in-qemu_set_log_fi.patch
  hw-arm-smmuv3-Align-stream-table-base-ad.patch
  hw-arm-smmuv3-Apply-address-mask-to-line.patch
  hw-arm-smmuv3-Check-stream-IDs-against-a.patch
  hw-arm-smmuv3-Correct-SMMU_BASE_ADDR_MAS.patch
  hw-arm-smmuv3-Report-F_STE_FETCH-fault-a.patch
  hw-arm-smmuv3-Use-correct-bit-positions-.patch
  i386-Resolve-CPU-models-to-v1-by-default.patch
  intel_iommu-a-fix-to-vtd_find_as_from_bu.patch
  iotests-Fix-IMGOPTSSYNTAX-for-nbd.patch
  iotests-Provide-a-function-for-checking-.patch
  iotests-Skip-test-060-if-it-is-not-possi.patch
  iotests-Skip-test-079-if-it-is-not-possi.patch
  numa-properly-check-if-numa-is-supported.patch
  qcow2-bitmaps-fix-qcow2_can_store_new_di.patch
  Revert-qemu-options.hx-Update-for-reboot.patch
  vhost-user-gpu-Drop-trailing-json-comma.patch
  virtio-blk-fix-out-of-bounds-access-to-b.patch
  virtio-mmio-update-queue-size-on-guest-w.patch
  virtio-net-delete-also-control-queue-whe.patch
  virtio-update-queue-size-on-guest-write.patch
- Include performance improvement
  virtio-don-t-enable-notifications-during.patch
- Repair incorrect packaging references to Jira tracked features
* Thu Jan  9 2020 brogers@suse.com
- Add Cooperlake vcpu model (jsc#SLE-7923)
  i386-Add-MSR-feature-bit-for-MDS-NO.patch
  i386-Add-macro-for-stibp.patch
  i386-Add-new-CPU-model-Cooperlake.patch
  target-i386-Add-new-bit-definitions-of-M.patch
  target-i386-Add-missed-features-to-Coope.patch
- Add HMAT support (jsc#SLE-8897) (the test case for this series
  isn't included because we aren't set up to handle binary patches)
  numa-Extend-CLI-to-provide-initiator-inf.patch
  numa-Extend-CLI-to-provide-memory-latenc.patch
  numa-Extend-CLI-to-provide-memory-side-c.patch
  hmat-acpi-Build-Memory-Proximity-Domain-.patch
  hmat-acpi-Build-System-Locality-Latency-.patch
  hmat-acpi-Build-Memory-Side-Cache-Inform.patch
  tests-numa-Add-case-for-QMP-build-HMAT.patch
* Thu Dec 12 2019 brogers@suse.com
- Update to v4.2.0: See http://wiki.qemu.org/ChangeLog/4.2
  Take note that ongoing feature deprecation is tracked at both
  http://wiki.qemu-project.org/Features/LegacyRemoval and in
  Appendix B of the qemu-doc.* files installed with the qemu package
  Some noteworthy changes:
  * x86: Denverton, Snowridge, and Dhyana CPU models added
  * x86: Latest version of all CPU models how have TSX (HLE and RTM)
  disabled by default
  * x86: Support for AVX512 BFloat16 extensions
  * x86: VMX features exposed more accurately and controllably
  * s390: TCG now implements IEP (Instruction Execution Protection)
  * PowerPC: POWER8 and POWER9 non-virtualized machines separated out
  * PowerPC: RTAS now comes from SLOF instead of QEMU itself
  * PowerPC: Unplug of multifunction PCI devices now unplugs the
  whole slot, as in x86
  * ARM: Support for >256 CPUs with KVM is fixed
  * ARM: Memory hotplug now supported , when using UEFI, ACPI, for
  virt machine type
  * ARM: SVE support possuble now for KVM guests
  * ARM: ACPI generic event device can now deliver powerdown event
  * The backend device can be specified for a guest audio device
  * virtio v1.1 packed virtqueues supported
  * Socket based character device backends now support TCP keep-alive
  * Use encryption library cipher mode facilities, allowing improved
  performance for eg. AES-XTS encrption
  * Misc block device improvements, esp. with nbd
- See the following few release-candidate changelog entries for
  additional changes related to this release
- Switched package build to be out-of-tree
* Tue Dec 10 2019 brogers@suse.com
- Update to v4.2.0-rc5: See http://wiki.qemu.org/ChangeLog/4.2
* Wed Dec  4 2019 brogers@suse.com
- Update to v4.2.0-rc4: See http://wiki.qemu.org/ChangeLog/4.2
  * Update the support documents used for SUSE SLE releases to cover
  this qemu release
* Wed Nov 27 2019 brogers@suse.com
- Update to v4.2.0-rc3: See http://wiki.qemu.org/ChangeLog/4.2
  * Patches dropped (upstream unless otherwise noted):
  ati-add-edid-support.patch
  ati-vga-add-rage128-edid-support.patch
  ati-vga-fix-ati_read.patch
  ati-vga-make-i2c-register-and-bits-confi.patch
  ati-vga-make-less-verbose.patch
  ati-vga-try-vga-ddc-first.patch
  Disable-Waddress-of-packed-member-for-GC.patch
  hdata-vpd-fix-printing-char-0x00.patch
  target-i386-add-PSCHANGE_NO-bit-for-the-.patch
  target-i386-Export-TAA_NO-bit-to-guests.patch
  vbe-add-edid-support.patch
  vga-add-ati-bios-tables.patch
  vga-add-atiext-driver.patch
  vga-make-memcpy_high-public.patch
  vga-move-modelist-from-bochsvga.c-to-new.patch
  * Patches added:
  Enable-cross-compile-prefix-for-C-compil.patch
  ensure-headers-included-are-compatible-w.patch
  roms-Makefile-enable-cross-compile-for-b.patch
  * Add qemu-ui-spice-app package containing ui-spice-app.so
  * Add qemu-microvm package containing bios-microvm.bin
- Add descriptors for the 128k and 256k SeaBios firmware images
- For the record, the following issues reported for SUSE SLE15-SP1
  are either fixed in this current package, or are otherwise not an
  issue: bsc#1079730 bsc#1098403 bsc#1111025 bsc#1128106 bsc#1133031
  bsc#1134883 bsc#1135210 bsc#1135902 bsc#1136540 bsc#1136778
  bsc#1138534 bsc#1140402 bsc#1143794 bsc#1145379 bsc#1144087
  bsc#1145427 bsc#1145436 bsc#1145774 bsc#1146873 bsc#1149811
  bsc#1152506 bsc#1155812 bsc#1156642 CVE-2018-12207 CVE-2019-5008
  CVE-2019-11135 CVE-2019-12068 CVE-2019-12155 CVE-2019-13164
  CVE-2019-14378 CVE-2019-15890, and the following feature requests
  are satisfied by this package: fate#327410 fate#327764 fate#327796
  jsc#SLE-4883 jsc#SLE-6132 jsc#SLE-6237 jsc#SLE-6754
* Tue Nov 19 2019 brogers@suse.com
- Expose pschange-mc-no "feature", indicating CPU does not have
  the page size change machine check vulnerability (CVE-2018-12207
  bsc#1155812)
  target-i386-add-PSCHANGE_NO-bit-for-the-.patch
- Expose taa-no "feature", indicating CPU does not have the
  TSX Async Abort vulnerability. (CVE-2019-11135 bsc#1152506)
  target-i386-Export-TAA_NO-bit-to-guests.patch
  Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-4.1
* Fri Nov 15 2019 brogers@suse.com
- Update to v4.1.1, a stable, bug-fix-only release
  * Besides incorporating the following fixes we already carried, it
  includes about the same number of other, similar type fixes
  which we hadn't yet incorporated.
  * Patches dropped (subsumed by stable update):
  block-Add-bdrv_co_get_self_request.patch
  block-create-Do-not-abort-if-a-block-dri.patch
  block-file-posix-Let-post-EOF-fallocate-.patch
  block-file-posix-Reduce-xfsctl-use.patch
  block-io-refactor-padding.patch
  blockjob-update-nodes-head-while-removin.patch
  block-Make-wait-mark-serialising-request.patch
  block-nfs-tear-down-aio-before-nfs_close.patch
  coroutine-Add-qemu_co_mutex_assert_locke.patch
  curl-Check-completion-in-curl_multi_do.patch
  curl-Handle-success-in-multi_check_compl.patch
  curl-Keep-pointer-to-the-CURLState-in-CU.patch
  curl-Keep-socket-until-the-end-of-curl_s.patch
  curl-Pass-CURLSocket-to-curl_multi_do.patch
  curl-Report-only-ready-sockets.patch
  hw-arm-boot.c-Set-NSACR.-CP11-CP10-for-N.patch
  hw-core-loader-Fix-possible-crash-in-rom.patch
  make-release-pull-in-edk2-submodules-so-.patch
  memory-Provide-an-equality-function-for-.patch
  mirror-Keep-mirror_top_bs-drained-after-.patch
  pr-manager-Fix-invalid-g_free-crash-bug.patch
  qcow2-bitmap-Fix-uint64_t-left-shift-ove.patch
  qcow2-Fix-corruption-bug-in-qcow2_detect.patch
  qcow2-Fix-QCOW2_COMPRESSED_SECTOR_MASK.patch
  qcow2-Fix-the-calculation-of-the-maximum.patch
  roms-Makefile.edk2-don-t-pull-in-submodu.patch
  s390-PCI-fix-IOMMU-region-init.patch
  s390x-tcg-Fix-VERIM-with-32-64-bit-eleme.patch
  target-alpha-fix-tlb_fill-trap_arg2-valu.patch
  target-arm-Don-t-abort-on-M-profile-exce.patch
  target-arm-Free-TCG-temps-in-trans_VMOV_.patch
  util-iov-introduce-qemu_iovec_init_exten.patch
  vhost-Fix-memory-region-section-comparis.patch
  vpc-Return-0-from-vpc_co_create-on-succe.patch
  Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-4.1
* Fri Nov  8 2019 guillaume.gardet@opensuse.org
- Fix %%arm builds
* Thu Nov  7 2019 brogers@suse.com
- Fix two issues with qcow2 image processing which could affect
  disk integrity
  qcow2-Fix-QCOW2_COMPRESSED_SECTOR_MASK.patch
  qcow2-bitmap-Fix-uint64_t-left-shift-ove.patch
* Wed Nov  6 2019 brogers@suse.com
- Work around a host kernel xfs bug which can result in qcow2 image
  corruption
  block-io-refactor-padding.patch
  util-iov-introduce-qemu_iovec_init_exten.patch
  block-Make-wait-mark-serialising-request.patch
  block-Add-bdrv_co_get_self_request.patch
  block-file-posix-Let-post-EOF-fallocate-.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-4.1
* Mon Nov  4 2019 stefan.bruens@rwth-aachen.de
- Correct package names in _constraints after switch to multibuild.
* Sat Oct 26 2019 brogers@suse.com
- Address potential corruption when using qcow2 images
  coroutine-Add-qemu_co_mutex_assert_locke.patch
  qcow2-Fix-corruption-bug-in-qcow2_detect.patch
- Include more tweaks to our packaging workflow scripts - this will
  continue as we refine the scripts
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-4.1
* Thu Oct 17 2019 lnussel@suse.de
- use %%gcc_version for cross compilers (boo#1153703)
* Tue Oct 15 2019 brogers@suse.com
- Add upstream edk2 submodule fix for creating tarball
- Switch to upstream patch for avoiding git ref in edk2 makefile
- Fix failing block tests which aren't compatible with the configure
  option --enable-membarrier
  * Patches dropped:
  roms-Makefile.edk2-don-t-invoke-git-sinc.patch
  tests-block-io-test-130-needs-some-delay.patch
  * Patches added:
  make-release-pull-in-edk2-submodules-so-.patch
  roms-Makefile.edk2-don-t-pull-in-submodu.patch
  tests-Fix-block-tests-to-be-compatible-w.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-4.1
* Thu Oct 10 2019 brogers@suse.com
- Reduce the cross compiler versions we rely on
- Fix some qemu-testsuite issues, reducing known error cases
  test-add-mapping-from-arch-of-i686-to-qe.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-4.1
* Wed Oct  2 2019 brogers@suse.com
- Since our spec file has bashisms, include the following in the
  spec file: %%define _buildshell /bin/bash
* Wed Oct  2 2019 brogers@suse.com
- Disable some block tests which randomly fail. This is in context
  of the build service build of qemu-testsuite
  tests-Disable-some-block-tests-for-now.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-4.1
* Tue Oct  1 2019 brogers@suse.com
- Add some post v4.1.0 upstream stable patches
  * Patches added:
  mirror-Keep-mirror_top_bs-drained-after-.patch
  s390x-tcg-Fix-VERIM-with-32-64-bit-eleme.patch
  target-alpha-fix-tlb_fill-trap_arg2-valu.patch
  target-arm-Free-TCG-temps-in-trans_VMOV_.patch
  target-arm-Don-t-abort-on-M-profile-exce.patch
  qcow2-Fix-the-calculation-of-the-maximum.patch
  block-file-posix-Reduce-xfsctl-use.patch
  pr-manager-Fix-invalid-g_free-crash-bug.patch
  vpc-Return-0-from-vpc_co_create-on-succe.patch
  block-nfs-tear-down-aio-before-nfs_close.patch
  block-create-Do-not-abort-if-a-block-dri.patch
  curl-Keep-pointer-to-the-CURLState-in-CU.patch
  curl-Keep-socket-until-the-end-of-curl_s.patch
  curl-Check-completion-in-curl_multi_do.patch
  curl-Pass-CURLSocket-to-curl_multi_do.patch
  curl-Report-only-ready-sockets.patch
  curl-Handle-success-in-multi_check_compl.patch
  blockjob-update-nodes-head-while-removin.patch
  memory-Provide-an-equality-function-for-.patch
  vhost-Fix-memory-region-section-comparis.patch
  hw-arm-boot.c-Set-NSACR.-CP11-CP10-for-N.patch
  s390-PCI-fix-IOMMU-region-init.patch
  hw-core-loader-Fix-possible-crash-in-rom.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-4.1
* Wed Sep 11 2019 brogers@suse.com
- Include more tweaks to our packaging workflow scripts
- Produce qemu-linux-user and qemu-testsuite via the build service
  multibuild capability, instead of duplicating the spec file and
  using package link in build service
  * combine qemu-linux-user spec file into main qemu spec file. Since
  this model uses a single changelog, here are some historicial
  mentions from the now unused qemu-linux-user.changes (delta from
  qemu's was quite minimal):
  - Adjust to a v5.2 linux kernel change regarding SIOCGSTAMP
  - Fix pwrite64/pread64 to return 0 over -1 for a
    zero length NULL buffer in qemu (bsc#1121600)
  * bsc#1112499
  * Since qemu-testsuite.spec and qemu-testsuite.changes were just
  copies of the main qemu version nothing needs to be done there
* Fri Sep  6 2019 brogers@suse.com
- Build opensbi from source on riscv64
* Fri Sep  6 2019 brogers@suse.com
- Update to v4.1.0: See http://wiki.qemu.org/ChangeLog/4.1
  Take note that ongoing feature deprecation is tracked at both
  http://wiki.qemu-project.org/Features/LegacyRemoval and in
  Appendix B of the qemu-doc.* files installed with the qemu package
  Some noteworthy changes:
  * x86: CPU models are now versioned
  * x86: CPU die topology can now be configured
  * x86: New Hygon Dhyana and Intel Snowridge CPU models
  * s390: The bios now supports IPL (boot) from ECKD DASD assigned
  to the guest via vfio-ccw
  * s390: The bios now tolerates the presence of bootmap signature
  entries written by zipl
  * PowerPC: pseries machine now supports KVM acceleration
  (kernel_irqchip=on) of the XIVE interrupt controller
  * PowerPC: pseries now supports hot-plug of PCI bridges and hot-plug
  and unplug of devices under PCI bridges
  * ARM: QEMU now supports emulating an FPU for Cortex-M CPUs, and the
  Cortex-M4 and Cortex-M33 now provide the FP
  * Python 2 support is deprecated
  * UEFI platform firmware binaries, and matching variable store
  templates are now installed
  * Now it's possible to specify memory-less NUMA node when using
  "-numa node,memdev" options
  * Possible to trigger self announcement on specific network interfaces
  * Default memory distribution between NUMA nodes is now deprecated
  * Fallback to normal RAM allocation if QEMU is not able to allocate
  from the "-mem-path" provided file/filesystem is now deprecated
  * virtio-gpu 2d/3d rendering may now be offloaded to an external
  vhost-user process, such as QEMU vhost-user-gpu
  * QEMU will automatically try to use the MAP_SYNC mmap flag for memory
  backends configured with pmem=on,share=on
  * Additional SeaVGABIOS patches added for vga-ati compatibility
- Drop attempt at build compatibility with SLE12
- New sub-packages: qemu-edk2, qemu-vhost-user-gpu
- Conditionalize building of qemu-edk2 (and leave unbuilt for now)
- Implement new packaging workflow, includes no longer numbering
  patches, and having the "current git repo" stored with the package
  in the form of git bundles
  * Patches dropped (upstream unless otherwise noted):
  0027-tests-test-thread-pool-is-racy-add-.patch
  0032-tests-Fix-Makefile-handling-of-chec.patch
  0034-Revert-target-i386-kvm-add-VMX-migr.patch
  0036-sockets-avoid-string-truncation-war.patch
  0039-linux-user-avoid-string-truncation-.patch
  0040-linux-user-elfload-Fix-GCC-9-build-.patch
  0041-qxl-avoid-unaligned-pointer-reads-w.patch
  0042-libvhost-user-fix-Waddress-of-packe.patch
  0043-target-i386-define-md-clear-bit.patch
  0045-kbd-state-fix-autorepeat-handling.patch
  0046-target-ppc-ensure-we-get-null-termi.patch
  0049-qxl-check-release-info-object.patch
  0050-qemu-bridge-helper-restrict-interfa.patch
  0051-linux-user-fix-to-handle-variably-s.patch
  ipxe-use-gcc6-for-more-compact-code.patch (no longer needed)
  (the next three are replaced by the upstream equivalent)
  ipxe-efi-Simplify-diagnostic-for-NULL-handle.patch
  ipxe-build-Disable-gcc-address-of-packed-member-warning.patch
  ipxe-efi-Avoid-string-op-warning-with-cross-gcc-7-compile.patch
  slirp-fix-heap-overflow-in-ip_reass-on-big-packet-input.patch
  * Patches renamed:
  0001-XXX-dont-dump-core-on-sigabort.patch
  - > XXX-dont-dump-core-on-sigabort.patch
  0002-qemu-binfmt-conf-Modify-default-pat.patch
  - > qemu-binfmt-conf-Modify-default-path.patch
  0003-qemu-cvs-gettimeofday.patch
  - > qemu-cvs-gettimeofday.patch
  0004-qemu-cvs-ioctl_debug.patch
  - > qemu-cvs-ioctl_debug.patch
  0005-qemu-cvs-ioctl_nodirection.patch
  - > qemu-cvs-ioctl_nodirection.patch
  0006-linux-user-add-binfmt-wrapper-for-a.patch
  - > linux-user-add-binfmt-wrapper-for-argv-0.patch
  0007-PPC-KVM-Disable-mmu-notifier-check.patch
  - > PPC-KVM-Disable-mmu-notifier-check.patch
  0008-linux-user-binfmt-support-host-bina.patch
  - > linux-user-binfmt-support-host-binaries.patch
  0009-linux-user-Fake-proc-cpuinfo.patch
  - > linux-user-Fake-proc-cpuinfo.patch
  0010-linux-user-use-target_ulong.patch
  - > linux-user-use-target_ulong.patch
  0011-Make-char-muxer-more-robust-wrt-sma.patch
  - > Make-char-muxer-more-robust-wrt-small-FI.patch
  0012-linux-user-lseek-explicitly-cast-no.patch
  - > linux-user-lseek-explicitly-cast-non-set.patch
  0013-AIO-Reduce-number-of-threads-for-32.patch
  - > AIO-Reduce-number-of-threads-for-32bit-h.patch
  0014-xen_disk-Add-suse-specific-flush-di.patch
  - > xen_disk-Add-suse-specific-flush-disable.patch
  0015-qemu-bridge-helper-reduce-security-.patch
  - > qemu-bridge-helper-reduce-security-profi.patch
  0016-qemu-binfmt-conf-use-qemu-ARCH-binf.patch
  - > qemu-binfmt-conf-use-qemu-ARCH-binfmt.patch
  0017-linux-user-properly-test-for-infini.patch
  - > linux-user-properly-test-for-infinite-ti.patch
  0018-roms-Makefile-pass-a-packaging-time.patch
  - > roms-Makefile-pass-a-packaging-timestamp.patch
  0019-Raise-soft-address-space-limit-to-h.patch
  - > Raise-soft-address-space-limit-to-hard-l.patch
  0020-increase-x86_64-physical-bits-to-42.patch
  - > increase-x86_64-physical-bits-to-42.patch
  0021-vga-Raise-VRAM-to-16-MiB-for-pc-0.1.patch
  - > vga-Raise-VRAM-to-16-MiB-for-pc-0.15-and.patch
  0022-i8254-Fix-migration-from-SLE11-SP2.patch
  - > i8254-Fix-migration-from-SLE11-SP2.patch
  0023-acpi_piix4-Fix-migration-from-SLE11.patch
  - > acpi_piix4-Fix-migration-from-SLE11-SP2.patch
  0024-Switch-order-of-libraries-for-mpath.patch
  - > Switch-order-of-libraries-for-mpath-supp.patch
  0025-Make-installed-scripts-explicitly-p.patch
  - > Make-installed-scripts-explicitly-python.patch
  0026-hw-smbios-handle-both-file-formats-.patch
  - > hw-smbios-handle-both-file-formats-regar.patch
  0028-xen-add-block-resize-support-for-xe.patch
  - > xen-add-block-resize-support-for-xen-dis.patch
  0029-tests-qemu-iotests-Triple-timeout-o.patch
  - > tests-qemu-iotests-Triple-timeout-of-i-o.patch
  0030-tests-block-io-test-130-needs-some-.patch
  - > tests-block-io-test-130-needs-some-delay.patch
  0031-xen-ignore-live-parameter-from-xen-.patch
  - > xen-ignore-live-parameter-from-xen-save-.patch
  0033-Conditionalize-ui-bitmap-installati.patch
  - > Conditionalize-ui-bitmap-installation-be.patch
  0035-tests-change-error-message-in-test-.patch
  - > tests-change-error-message-in-test-162.patch
  0037-hw-usb-hcd-xhci-Fix-GCC-9-build-war.patch
  - > hw-usb-hcd-xhci-Fix-GCC-9-build-warning.patch
  0038-hw-usb-dev-mtp-Fix-GCC-9-build-warn.patch
  - > hw-usb-dev-mtp-Fix-GCC-9-build-warning.patch
  0044-hw-intc-exynos4210_gic-provide-more.patch
  - > hw-intc-exynos4210_gic-provide-more-room.patch
  0047-configure-only-populate-roms-if-sof.patch
  - > configure-only-populate-roms-if-softmmu.patch
  0048-pc-bios-s390-ccw-net-avoid-warning-.patch
  - > pc-bios-s390-ccw-net-avoid-warning-about.patch
  keycodemapdb-make-keycode-gen-output-reproducible.patch
  - > Make-keycode-gen-output-reproducible-use.patch
  ipxe-stub-out-the-SAN-req-s-in-int13.patch
  - > stub-out-the-SAN-req-s-in-int13.patch
  sgabios-fix-cross-build.patch deleted
  - > roms-sgabios-Fix-csum8-to-be-built-by-ho.patch
  sgabios-stable-buildid.patch
  - > sgabios-Makefile-fix-issues-of-build-rep.patch
  skiboot-gcc9-compat.patch
  - > Disable-Waddress-of-packed-member-for-GC.patch
  ipxe-stable-buildid.patch
  - > ipxe-Makefile-fix-issues-of-build-reprod.patch
  seabios-fix_cross_compilation.patch
  - > enable-cross-compilation-on-ARM.patch
  * Patches added:
  roms-change-cross-compiler-naming-to-be-.patch
  roms-Makefile.edk2-don-t-invoke-git-sinc.patch
  vga-move-modelist-from-bochsvga.c-to-new.patch
  vga-make-memcpy_high-public.patch
  vga-add-atiext-driver.patch
  vga-add-ati-bios-tables.patch
  vbe-add-edid-support.patch
  ati-add-edid-support.patch
  ati-vga-make-less-verbose.patch
  ati-vga-fix-ati_read.patch
  ati-vga-make-i2c-register-and-bits-confi.patch
  ati-vga-try-vga-ddc-first.patch
  ati-vga-add-rage128-edid-support.patch
  Fix-s-directive-argument-is-null-error.patch
  Workaround-compilation-error-with-gcc-9..patch
  Do-not-apply-WORKAROUND_CFLAGS-for-host-.patch
  hdata-vpd-fix-printing-char-0x00.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-4.1
* Fri Aug  9 2019 brogers@suse.com
- Since we build seabios, take advantage of ability to add our own
  identifying version info by changing SEABIOS_EXTRAVERSION from
  "-prebuilt.qemu.org" to "-rebuilt.suse.com" (or
  "-rebuilt.opensuse.org for openSUSE releases)
* Wed Aug  7 2019 lyan@suse.com
- Security fix for heap overflow in ip_reass on big packet input
  (CVE-2019-14378, bsc#1143794)
  slirp-fix-heap-overflow-in-ip_reass-on-big-packet-input.patch
* Tue Aug  6 2019 brogers@suse.com
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-4.0
  * Patches added:
  0051-linux-user-fix-to-handle-variably-s.patch
* Mon Aug  5 2019 stefan.bruens@rwth-aachen.de
- Make keycode-gen output reproducible (use SOURCE_DATE_EPOCH timestamp)
  keycodemapdb-make-keycode-gen-output-reproducible.patch
* Thu Jul 25 2019 lyan@suse.com
- Security fix for null pointer dereference while releasing spice resources
  (CVE-2019-12155, bsc#1135902)
  0049-qxl-check-release-info-object.patch
- Security fix for qemu-bridge-helper ACL can be bypassed when names are too long
  (CVE-2019-13164, bsc#1140402)
  0050-qemu-bridge-helper-restrict-interfa.patch
- Replace patch 0043 with an upstream version
  0043-target-i386-define-md-clear-bit.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-4.0
* Tue Jun 11 2019 cgoll@suse.com
- fixed regression for ksm.service was (bsc#1112646)
* Mon Jun 10 2019 brogers@suse.com
- Content of packaged %%_docdir/%%name/interop/_static/ dir depends
  on python-Sphinx version, so lets just wildcard specifying
  those files, rather than trying to manage a specific file list
* Fri Jun  7 2019 brogers@suse.com
- Last change exposed that we still do rely on python2. Make
  spec file adjustment
* Fri Jun  7 2019 tchvatal@suse.com
- Switch from python-Sphinx to Sphinx from python variant we are
  building with (new Sphinx is for python3 only)
* Wed May 29 2019 brogers@suse.com
- Fix a number of compatibility issues with the stricter gcc9 checks
  * Disable warning for taking address of packed structure members
  0048-pc-bios-s390-ccw-net-avoid-warning-.patch
  * Fix case of strncpy where null terminated string not guaranteed
  0046-target-ppc-ensure-we-get-null-termi.patch
  * Disable warning for taking address of packed structure members
  and fix case of passing null pointer as "%%s" format parameter
  skiboot-gcc9-compat.patch
- Fix configure script which caused firmware to be built in
  linux-user only build.
  0047-configure-only-populate-roms-if-sof.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-4.0
* Fri May 17 2019 brogers@suse.com
- Fix regression in autorepeat key handling
  0045-kbd-state-fix-autorepeat-handling.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-4.0
* Fri May 17 2019 schwab@suse.de
- Fix file list
* Wed May 15 2019 brogers@suse.com
- Yet another gcc9 related code fix (bsc#1121464)
  0044-hw-intc-exynos4210_gic-provide-more.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-4.0
* Tue May 14 2019 brogers@suse.com
- Switch to now upstreamed version of patch and add one more
  gcc9 related patch
  * Patches renamed:
  0041-qxl-fix-Waddress-of-packed-member.patch
  - > 0041-qxl-avoid-unaligned-pointer-reads-w.patch
  0042-libvhost-user-fix-Waddress-of-packe.patch
- Add x86 cpu feature "md-clear" (CVE-2018-12126 CVE-2018-12127
  CVE-2018-12130 CVE-2019-11091 bsc#1111331)
  0043-target-i386-define-md-clear-bit.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-4.0
* Sat May 11 2019 brogers@suse.com
- Correct logic of which ipxe patches get included based on
  suse_version. We were wrongly excluding a gcc9 related patch for
  example
* Fri May 10 2019 brogers@suse.com
- Switch to now upstreamed version of some patches
  * Patches renamed:
  0036-util-qemu-sockets-Fix-GCC-9-build-w.patch
  - > 0036-sockets-avoid-string-truncation-war.patch
  0039-linux-user-uname-Fix-GCC-9-build-wa.patch
  - > 0039-linux-user-avoid-string-truncation-.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-4.0
* Thu May  9 2019 brogers@suse.com
- Create /usr/share/qemu/firmware and /etc/qemu/firmware directories
  in support of the firmware descriptor feature now in use as of
  libvirt v5.2
* Mon May  6 2019 brogers@suse.com
- Disable LTO as suggested by Martin Liska (boo#1133281)
- Remove and obsolete qemu-oss-audio subpackage. OSS audio is very
  old, and we didn't really even configure the package properly for
  it for a very long time, so presumably there can't be any users
  of it as far as qemu is concerned
- Avoid warnings which gcc9 complains about
  0036-util-qemu-sockets-Fix-GCC-9-build-w.patch
  0037-hw-usb-hcd-xhci-Fix-GCC-9-build-war.patch
  0038-hw-usb-dev-mtp-Fix-GCC-9-build-warn.patch
  0039-linux-user-uname-Fix-GCC-9-build-wa.patch
  0040-linux-user-elfload-Fix-GCC-9-build-.patch
  0041-qxl-fix-Waddress-of-packed-member.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-4.0
* Wed May  1 2019 brogers@suse.com
- Update to v4.0.0: See http://wiki.qemu.org/ChangeLog/4.0
  Take note that ongoing feature deprecation is tracked at both
  http://wiki.qemu-project.org/Features/LegacyRemoval and in
  Appendix B of the qemu-doc.* files installed with the qemu package
  Some noteworthy changes:
  * ARM: ARMv8+ extensions for SB, PredInv, HPD, LOR, FHM, AA32HPD,
  PAuth, JSConv, CondM, FRINT, and BTI
  * ARM: new emulation support for "Musca" and "MPS2" development boards
  * ARM: virt: support for >255GB of RAM and u-boot "noload" image types
  * ARM: improved emulation of ARM PMU
  * HPPA: support for TLB protection IDs and TLB trace events
  * MIPS: support for multi-threaded TCG emulation
  * MIPS: emulation support for I7200 I6500 CPUs, QMP-base querying of
  CPU types, and improved support for SAARI and SAAR configuration registers
  * MIPS: improvements to Interthread Communication Unit, Fulong 2E
  machine types, and end-user documentation.
  * PowerPC: pseries/powernv: support for POWER9 large decrementer
  * PowerPC: pseries: emulation support for XIVE interrupt controller
  * PowerPC: pseries: support for hotplugging PCI host bridges (PHBs)
  * PowerPC: pseries: Spectre/Meltdown mitigations enabled by default,
  additional support for count-cache-flush mitigation
  * RISC-V: virt: support for PCI and USB
  * RISC-V: support for TSR, TW, and TVM fields of mstatus, FS field now
  supports three stats (dirty, clean, and off)
  * RISC-V: built-in gdbserver supports register lists via XML files
  * s390: support for z14 GA 2 CPU model, Multiple-epoch and PTFF
  features now enabled in z14 CPU model by default
  * s390: vfio-ap: now supports hot plug/unplug, and no longer inhibits memory
  ballooning
  * s390: emulation support for floating-point extension facility and
  vector support instructions
  * x86: HAX accelerator now supported POSIX hosts other than Darwin,
  including Linux and NetBSD
  * x86: Q35: advertised PCIe root port speeds will now optimally default
  to maximum link speed (16GT/s) and width (x32) provided by PCIe 4.0 for
  QEMU 4.0+ machine types; older machine types will retain 2.5GT/x1
  defaults for compatibility.
  * x86: Xen PVH images can now be booted with "-kernel" option
  * Xtensa: xtfpga: improved SMP support for linux (interrupt
  distributor, IPI, and runstall) and new SMP-capable test_mmuhifi_c3
  core configuration
  * Xtensa: support for Flexible length instructions extension (FLIX)
  * GUI: new '-display spice-app' to configure/launch a Spice client GUI with
  a similar UI to QEMU GTK. VNC server now supports access controls via
  tls-authz/sasl-authz options
  * QMP: support for "out-of-band" command execution, can be useful for
  postcopy migration recovery. Additional QMP commands for working with
  block devices and dirty bitmaps
  * VFIO: EDID interface for supported mdev (Intel vGPU for kernel 5.0+),
  allows resolution setting via xres/yres options.
  * Xen: new 'xen-disk' device which can create a Xen PV disk backend,
  and performance improvements for Xen PV disk backend.
  * Network Block Device: improved tracing and error diagnostics, improved
  client compatibility with buggy NBD server implementations, new
  - -bitmap, --list, --tls-authz options for qemu-nbd
  * virtio-blk now supports DISCARD and WRITE_ZEROES
  * qemu-test-suite output is now in TAP format
  * Sphinx now used for part of qemu documentation
  * A few more configure features are enabled: iconv, lzfse (for openSUSE)
  * Provide better logo icons
- Made these package building changes:
  * Removed this token from spec file: #!BuildIgnore:  gcc-PIE
  * Created ability to build qemu source out-of-tree
  * Added BSD-2-Clause license clause due to EDK II code inclusion
  * Patches dropped (upstream unless otherwise noted):
  0010-Remove-problematic-evdev-86-key-fro.patch
  0025-Fix-tigervnc-long-press-issue.patch
  0026-string-input-visitor-Fix-uint64-par.patch
  0027-test-string-input-visitor-Add-int-t.patch
  0028-test-string-input-visitor-Add-uint6.patch
  0029-tests-Add-QOM-property-unit-tests.patch
  0030-tests-Add-scsi-disk-test.patch
  0033-smbios-Add-1-terminator-if-any-stri.patch (different approach used)
  0034-qemu-io-tests-comment-out-problemat.patch (not as needed)
  0039-xen_disk-Avoid-repeated-memory-allo.patch
  0041-vfio-ap-flag-as-compatible-with-bal.patch
  0042-hw-s390x-Fix-bad-mask-in-time2tod.patch
  0043-pcie-set-link-state-inactive-active.patch
  0044-pc-piix4-Update-smbus-I-O-space-aft.patch
  0045-hw-usb-fix-mistaken-de-initializati.patch
  0046-usb-mtp-use-O_NOFOLLOW-and-O_CLOEXE.patch
  0047-pvrdma-release-device-resources-in-.patch
  0048-rdma-check-num_sge-does-not-exceed-.patch
  0049-pvrdma-add-uar_read-routine.patch
  0050-pvrdma-check-number-of-pages-when-c.patch
  0051-pvrdma-check-return-value-from-pvrd.patch
  0052-pvrdma-release-ring-object-in-case-.patch
  0053-block-Fix-hangs-in-synchronous-APIs.patch
  0054-linux-user-make-pwrite64-pread64-fd.patch
  0055-xen-Add-xen-v4.12-based-xc_domain_c.patch
  0056-slirp-check-data-length-while-emula.patch
  0057-s390x-Return-specification-exceptio.patch
  0059-memory-Fix-the-memory-region-type-a.patch
  0060-target-i386-sev-Do-not-pin-the-ram-.patch
  0061-slirp-check-sscanf-result-when-emul.patch
  0062-ppc-add-host-serial-and-host-model-.patch
  0063-i2c-ddc-fix-oob-read.patch
  0064-device_tree.c-Don-t-use-load_image.patch
  0065-spapr-Simplify-handling-of-host-ser.patch
  ipxe-efi-guard-strncpy-with-gcc-warning-ignore-pragma.patch
  ipxe-fix-build.patch
  skiboot-hdata-i2c.c-fix-building-with-gcc8.patch
  * Patches renamed:
  0011-linux-user-use-target_ulong.patch
  - > 0010-linux-user-use-target_ulong.patch
  0012-Make-char-muxer-more-robust-wrt-sma.patch
  - > 0011-Make-char-muxer-more-robust-wrt-sma.patch
  0013-linux-user-lseek-explicitly-cast-no.patch
  - > 0012-linux-user-lseek-explicitly-cast-no.patch
  0014-AIO-Reduce-number-of-threads-for-32.patch
  - > 0013-AIO-Reduce-number-of-threads-for-32.patch
  0015-xen_disk-Add-suse-specific-flush-di.patch
  - > 0014-xen_disk-Add-suse-specific-flush-di.patch
  0016-qemu-bridge-helper-reduce-security-.patch
  - > 0015-qemu-bridge-helper-reduce-security-.patch
  0017-qemu-binfmt-conf-use-qemu-ARCH-binf.patch
  - > 0016-qemu-binfmt-conf-use-qemu-ARCH-binf.patch
  0018-linux-user-properly-test-for-infini.patch
  - > 0017-linux-user-properly-test-for-infini.patch
  0019-roms-Makefile-pass-a-packaging-time.patch
  - > 0018-roms-Makefile-pass-a-packaging-time.patch
  0020-Raise-soft-address-space-limit-to-h.patch
  - > 0019-Raise-soft-address-space-limit-to-h.patch
  0021-increase-x86_64-physical-bits-to-42.patch
  - > 0020-increase-x86_64-physical-bits-to-42.patch
  0022-vga-Raise-VRAM-to-16-MiB-for-pc-0.1.patch
  - > 0021-vga-Raise-VRAM-to-16-MiB-for-pc-0.1.patch
  0023-i8254-Fix-migration-from-SLE11-SP2.patch
  - > 0022-i8254-Fix-migration-from-SLE11-SP2.patch
  0024-acpi_piix4-Fix-migration-from-SLE11.patch
  - > 0023-acpi_piix4-Fix-migration-from-SLE11.patch
  0031-Switch-order-of-libraries-for-mpath.patch
  - > 0024-Switch-order-of-libraries-for-mpath.patch
  0032-Make-installed-scripts-explicitly-p.patch
  - > 0025-Make-installed-scripts-explicitly-p.patch
  0035-tests-test-thread-pool-is-racy-add-.patch
  - > 0027-tests-test-thread-pool-is-racy-add-.patch
  0036-xen-add-block-resize-support-for-xe.patch
  - > 0028-xen-add-block-resize-support-for-xe.patch
  0037-tests-qemu-iotests-Triple-timeout-o.patch
  - > 0029-tests-qemu-iotests-Triple-timeout-o.patch
  0038-tests-block-io-test-130-needs-some-.patch
  - > 0030-tests-block-io-test-130-needs-some-.patch
  0040-xen-ignore-live-parameter-from-xen-.patch
  - > 0031-xen-ignore-live-parameter-from-xen-.patch
  0058-Revert-target-i386-kvm-add-VMX-migr.patch
  - > 0034-Revert-target-i386-kvm-add-VMX-migr.patch
  * Patches added:
  0026-hw-smbios-handle-both-file-formats-.patch
  0032-tests-Fix-Makefile-handling-of-chec.patch
  0033-Conditionalize-ui-bitmap-installati.patch
  0035-tests-change-error-message-in-test-.patch
  ipxe-efi-Avoid-string-op-warning-with-cross-gcc-7-compile.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-4.0
* Fri Mar 29 2019 brogers@suse.com
- Adjust fix for CVE-2019-8934 (bsc#1126455) to match the latest
  upstream adjustments for the same. Basically now the security fix
  is to provide a dummy host-model and host-serial value, which
  overrides getting that value from the host
  0065-spapr-Simplify-handling-of-host-ser.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-3.1
* Wed Mar 27 2019 brogers@suse.com
- Tweak last spec file change to guard new Requires with conditional
- Fix DOS possibility in device tree processing (CVE-2018-20815
  bsc#1130675)
  0064-device_tree.c-Don-t-use-load_image.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-3.1
* Mon Mar 25 2019 brogers@suse.com
- Remove an unneeded BuildRequires which impacts bsc#1119414 fix
  Also add a corresponding Recommends for qemu-tools as part of
  this packaging adjustment (bsc#1130484)
- Fix information leak in slirp (CVE-2019-9824 bsc#1129622)
  0061-slirp-check-sscanf-result-when-emul.patch
- Add method to specify whether or not to expose certain ppc64 host
  information, which can be considered a security issue
  (CVE-2019-8934 bsc#1126455)
  0062-ppc-add-host-serial-and-host-model-.patch
- Fix OOB memory access and information leak in virtual monitor
  interface (CVE-2019-03812 bsc#1125721)
  0063-i2c-ddc-fix-oob-read.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-3.1
* Fri Mar  8 2019 brogers@suse.com
- Again address ipxe GCC 9 incompatibilities. Previously included
  patch to disable unneeded warning got muffed somehow (bsc#1121464)
* Fri Feb 22 2019 guillaume.gardet@opensuse.org
- Package and cross-build rom files for aarch64 from
  SLE15/Leap15.0 to fix boo#1125964
- Add patch to fix seabios cross-compilation:
  * seabios-fix_cross_compilation.patch
- Add patch to fix sgabios cross-compilation:
  * sgabios-fix-cross-build.patch
* Thu Feb 21 2019 guillaume.gardet@opensuse.org
- Fix _constraints to include all architectures for disk size
  (fix aarch64)
* Fri Feb 15 2019 brogers@suse.com
- Revert upstream patch which declares x86 vmx feature a migration
  blocker. Given the proliferation of using vm's with host features
  passed through and the general knowledge that nested
  virtualization has many usage caveats, but still gets put in use
  in restricted scenarios, this patch did more harm than good, I
  feel. So despite this relaxation, please consider yourself warned
  that nested virtualization is not yet a supportable feature.
  (bsc#1121604)
  0058-Revert-target-i386-kvm-add-VMX-migr.patch
- Fix SEV VM device assignment (bsc#1123205)
  0059-memory-Fix-the-memory-region-type-a.patch
  0060-target-i386-sev-Do-not-pin-the-ram-.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-3.1
* Mon Feb 11 2019 brogers@suse.com
- Remove 71-sev.rules, which modifies the default permissions of
  /dev/sev by adding the kvm group as reader/writer. Upstream
  decided to take a different approach for libvirt to manage SEV
  due to security concerns which I agree overrides the convenience
  of providing /dev/sev access to all the kvm group (bsc#1124842
  bsc#1102604)
* Fri Feb  1 2019 brogers@suse.com
- Increase memory needed to build qemu-testsuite for ppc* arch's
  in _constraints file
* Wed Jan 30 2019 lyan@suse.com
- Return specification exception for unimplemented diag 308 subcodes
  rather than a hardware error (bsc#1123179)
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-3.1
  * Patches added:
  0057-s390x-Return-specification-exceptio.patch
* Fri Jan 25 2019 brogers@suse.com
- Fix OOB issue in slirp (CVE-2019-6778 bsc#1123156)
  0056-slirp-check-data-length-while-emula.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-3.1
- Fix ipxe GCC 9 incompatibilities (bsc#1121464)
  ipxe-efi-Simplify-diagnostic-for-NULL-handle.patch
  ipxe-build-Disable-gcc-address-of-packed-member-warning.patch
* Thu Jan 17 2019 brogers@suse.com
- Tweak Xen interface to be compatible with upcoming v4.12 Xen
  0055-xen-Add-xen-v4.12-based-xc_domain_c.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-3.1
* Tue Jan 15 2019 lyan@suse.com
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-3.1
  * Patches added:
  0054-linux-user-make-pwrite64-pread64-fd.patch
  (bsc#1121600)
* Mon Jan 14 2019 brogers@suse.com
- Clarify that move to include v3.1.0 in qemu package corresponds
  with fate#327089, which of course builds on v3.0.0 mentioned
  previously, and that among other patches which this change
  obsoletes (because functionality is included in base version) I
  will mention one pointed out by reviewers:
  0094-s390x-cpumodels-add-z14-Model-ZR1.patch
* Thu Jan 10 2019 brogers@suse.com
- include post v3.1.0 patches marked for next stable release:
  0041-vfio-ap-flag-as-compatible-with-bal.patch
  0042-hw-s390x-Fix-bad-mask-in-time2tod.patch
  0043-pcie-set-link-state-inactive-active.patch
  0044-pc-piix4-Update-smbus-I-O-space-aft.patch
  0045-hw-usb-fix-mistaken-de-initializati.patch
- Address various security/stability issues
  * Fix host access vulnerability in usb-mtp infrastructure
  (CVE-2018-16872 bsc#1119493)
  0046-usb-mtp-use-O_NOFOLLOW-and-O_CLOEXE.patch
  * Fix DoS in pvrdma interface (CVE-2018-20123 bsc#1119437)
  0047-pvrdma-release-device-resources-in-.patch
  * Fix OOB access issue in rdma backend (CVE-2018-20124 bsc#1119840)
  0048-rdma-check-num_sge-does-not-exceed-.patch
  * Fix NULL pointer reference in pvrdma emulation (CVE-2018-20191
  bsc#1119979)
  0049-pvrdma-add-uar_read-routine.patch
  * Fix DoS in pvrdma interface (CVE-2018-20125 bsc#1119989)
  0050-pvrdma-check-number-of-pages-when-c.patch
  * Fix DoS in pvrdma interface (CVE-2018-20216 bsc#1119984)
  0051-pvrdma-check-return-value-from-pvrd.patch
  * Fix DoS in pvrdma interface (CVE-2018-20126 bsc#1119991)
  0052-pvrdma-release-ring-object-in-case-.patch
- one more post v3.1.0 patches marked for next stable release:
  0053-block-Fix-hangs-in-synchronous-APIs.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-3.1
* Tue Jan  8 2019 olaf@aepfle.de
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-3.1
  * Patches added:
  0040-xen-ignore-live-parameter-from-xen-.patch
  (bsc#1079730, bsc#1101982, bsc#1063993)
* Fri Jan  4 2019 brogers@suse.com
- Follow up on ideas prompted by last change: clean up the patches
  generated by git workflow. There is no value to the first line
  (mbox From line), or [PATCH] on subject line. Get rid of those
- Other minor fixes and improvements to update_git.sh
* Thu Jan  3 2019 olaf@aepfle.de
- Modify update_git.sh script:
  pass --zero-commit to format-patch
  This removes needless noise in the buildservice when the same set
  of patches is imported/exported at different times by different users.
  pass --no-signature to format-patch
  Remove sed call which used to remove the signature, use mv instead
* Fri Dec 14 2018 lma@suse.com
- Use /bin/bash to echo value into sys fs for ksm control (bsc#1112646)
* Wed Dec 12 2018 brogers@suse.com
- fix memory leak in xen_disk (bsc#1100408)
  0039-xen_disk-Avoid-repeated-memory-allo.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-3.1
* Wed Dec 12 2018 olaf@aepfle.de
- building against xen-devel requires the XC_* compat macros to be
  set because this version of QEMU will be built against many
  versions of Xen. configure will decide on the appropriate function
  names it knows about today. To actually call these functions,
  future versions of Xen may require XC_* to be set.
  Furthermore, fix a bug in QEMU: xen_common.h undefines the XC_*
  macros unconditionally.
* Tue Dec 11 2018 brogers@suse.com
- Update to v3.1.0: See http://wiki.qemu.org/ChangeLog/3.1
  Take note that ongoing feature deprecation is tracked at both
  http://wiki.qemu-project.org/Features/LegacyRemoval and in
  Appendix B of the qemu-doc.* files installed with the qemu package
  Some noteworthy changes:
  * x86 IceLake-Server and IceLake-Client cpu models added
  * Document recommendations for choosing cpu modesl for x86 guests
  * Support for Hyper-V enlightened VMCS
  * stdvga and bochs-display devices can expose EDID information to the
  guest. stdvga xres and yres properties are exposed in the EDID information
  * s390 improvements: vfio-ap crypto device support, max-cpu model added,
  etoken support, huge page backing support
  * ARM: ARMv6M architecture and Cortex-M0 cpu host support added,
  Cortex-A72 cpu model added, GICv2 virtualization extensions, emulation
  of AArch32 virtualization, Scalable Vector Extension implemented
  * Support for AMD IOMMU interrupt remapping and guest virtual APIC mode
  * Multithreaded TCG on x86 is considered supportable
  * Add a patch to triple timeout of block io tests, since the obs
  environment is fickle
  * x86 save/restore and live migration is prohibited if Intel KVM nested
  virtualization is enabled
  * Patches dropped (upstream unless otherwise noted):
  0033-migration-warn-about-inconsistent-s.patch (shouldn't be needed anymore)
  0035-configure-Modify-python-used-for-io.patch (upstream now python3 friendly)
  0039-tests-boot-serial-test-Bump-timeout.patch
  0040-linux-headers-update.patch
  0041-s390x-kvm-add-etoken-facility.patch
  0042-seccomp-prefer-SCMP_ACT_KILL_PROCES.patch
  0043-configure-require-libseccomp-2.2.0.patch
  0044-seccomp-set-the-seccomp-filter-to-a.patch
  0045-sandbox-disable-sandbox-if-CONFIG_S.patch
  0046-seccomp-check-TSYNC-host-capability.patch
  0047-linux-user-init_guest_space-Try-to-.patch
  0048-ne2000-fix-possible-out-of-bound-ac.patch
  0049-rtl8139-fix-possible-out-of-bound-a.patch
  0050-pcnet-fix-possible-buffer-overflow.patch
  0051-net-ignore-packet-size-greater-than.patch
  0052-lsi53c895a-check-message-length-val.patch
  0053-nvme-fix-oob-access-issue-CVE-2018-.patch (fixed differently upstream)
  * Patches renamed:
  0034-smbios-Add-1-terminator-if-any-stri.patch
  - > 0033-smbios-Add-1-terminator-if-any-stri.patch
  0036-qemu-io-tests-comment-out-problemat.patch
  - > 0034-qemu-io-tests-comment-out-problemat.patch
  0037-tests-test-thread-pool-is-racy-add-.patch
  - > 0035-tests-test-thread-pool-is-racy-add-.patch
  0038-xen-add-block-resize-support-for-xe.patch
  - > 0036-xen-add-block-resize-support-for-xe.patch
  * Patches added:
  0037-tests-qemu-iotests-Triple-timeout-o.patch
  0038-tests-block-io-test-130-needs-some-.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-3.1
- Update includes the following bug fixes: bsc#1108474, bsc#1117615
- Update includes the following SLE requested functionality: FATE#324810,
  FATE#325875, FATE#326369, FATE#326378, FATE#326379, FATE#326401,
  FATE#326672, FATE#326829
- Make the following packaging changes related to the new release
  * Enable libpmem, pvrdma, vhost-crypto features and qemu-block-nfs
  subpackage
  * New roms available: vgabios-bochs-display.bin, vgabios-ramfb.bin
  * New binary tool included (qemu-edid) for testing the new qemu edid
  generator
- Tweaked patches we carry to pass qemu's checkpatch checker
- Modify update_git.sh script to enable packaging qemu from development
  time sources, not just at release time
- Removed erroneous (and now useless) tests for tar and gzip formats
- Don't exclude s390x anymore from building the qemu-testsuite
- Based on current OBS building observations make changes to storage
  and memory requires specified in the _constraints file
* Thu Nov  8 2018 brogers@suse.com
- Re-sync openSUSE and SUSE SLE qemu packages. This changes file
  is the openSUSE one with this entry providing the intervening
  SLE CVE, FATE, and bugzilla references, which are still addressed
  in this package, and not yet called out in this changes file.
  * CVE-2018-10839 CVE-2018-16847 CVE-2018-17958 CVE-2018-17962
  CVE-2018-17963 CVE-2018-18849
  * bsc#1110910 bsc#1111006 bsc#1111010 bsc#1111013 bsc#1114422
  bsc#1114529
  * Patches added:
  0047-linux-user-init_guest_space-Try-to-.patch
  0048-ne2000-fix-possible-out-of-bound-ac.patch
  0049-rtl8139-fix-possible-out-of-bound-a.patch
  0050-pcnet-fix-possible-buffer-overflow.patch
  0051-net-ignore-packet-size-greater-than.patch
  0052-lsi53c895a-check-message-length-val.patch
  0053-nvme-fix-oob-access-issue-CVE-2018-.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-3.0
* Fri Oct  5 2018 ldewey@suse.com
  * Adding changes to mitigate seccomp vulnerability
  (CVE-2018-15746 bsc#1106222)
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-3.0
  * Patches added:
  0042-seccomp-prefer-SCMP_ACT_KILL_PROCES.patch
  0043-configure-require-libseccomp-2.2.0.patch
  0044-seccomp-set-the-seccomp-filter-to-a.patch
  0045-sandbox-disable-sandbox-if-CONFIG_S.patch
  0046-seccomp-check-TSYNC-host-capability.patch
* Mon Sep 24 2018 brogers@suse.com
- Do more misc spec file fixes:
  * Be explicit in spec file about Version used for all subpackages
  (again, to avoid subpackage ordering issues). Default Release
  tag is also brought in by obs format_spec_file service
  * Delete binary blob s390-netboot.img, which we rebuild
  * Don't provide separate Url for qemu-kvm package - the main qemu
  website provides easily findable link for kvm specifics
  * Associate petalogix-ml605.dtb with qemu-extra instead of qemu-ppc
  * More entry sorting
* Fri Sep 21 2018 brogers@suse.com
- Correct some versioning as follows:
  * Accurately reflect the qemu-ipxe package version value by adding
  "+" at the end
  * Don't overwrite seabios .version file, since now (for quite some
  time actually) upstream tarball creation creates this file and
  the value we are writing to it is actually wrong
- Make spec file improvements, including the following:
  * Add qemu.keyring to enable package source verification
  * Create srcname macro to identify source file name separately from
  package name
  * Create alternate to %%version to avoid subpackage ordering
  causing inadvertantly wrong %%version value at point of use
  * Sort some entries
  * Be more consistent with macro syntax usage
  * Minor file tweaks as done by osc format_spec_file service
* Fri Sep 14 2018 brogers@suse.com
- Re-sync openSUSE and SUSE SLE qemu packages. This changes file
  is the openSUSE one with this entry providing the intervening
  SLE CVE, FATE, and bugzilla references, which are still addressed
  in this package, and not yet called out in this changes file.
  * CVE-2018-11806 CVE-2018-12617 CVE-2018-7550 CVE-2018-15746
  * fate#325467
  * bsc#1091695 bsc#1094725 bsc#1094913 bsc#1096223 bsc#1098735
  bsc#1103628 bsc#1105279 bsc#1106222 bsc#1106222 bsc#1107489
  * Patches added:
  * only enable glusterfs for openSUSE
  0040-linux-headers-update.patch
  0041-s390x-kvm-add-etoken-facility.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-3.0
* Mon Aug 20 2018 brogers@suse.com
- Increase timeout for boot-serial-test, since we've hit the timeout
  for armv7l arch in qemu-testsuite.
  0039-tests-boot-serial-test-Bump-timeout.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-3.0
* Wed Aug 15 2018 brogers@suse.com
- Drop legacy kvm_stat script and man page. We'll rely on the kvm_stat
  package only going forward
  kvm_stat
  kvm_stat.1.gz
- Update SLE support documentation to match v3.0.0 release
* Tue Aug 14 2018 brogers@suse.com
- Update to v3.0.0: See http://wiki.qemu.org/ChangeLog/3.0
  Don't read anything into the major version number update. It's
  been decided to increase the major version number each year.
  Take note that ongoing feature deprecation is tracked at both
  http://wiki.qemu-project.org/Features/LegacyRemoval and in
  Appendix B of the qemu-doc.* files installed with the qemu package.
  Some noteworthy changes:
  * Support for additional x86/AMD mitigations against Speculative
  Store Bypass (Spectre Variant 4, CVE-2018-3639)
  * Improved support for nested KVM guests running on Hyper-V
  * Block device support for active disk-mirroring, which avoids
  convergence issues which may arise when doing passive/background
  mirroring of busy devices
  * Improved support for AHCI emulation, SCSI emulation, and persistent
  reservations / cluster management
  * OpenGL ES support for SDL front-end, additional framebuffer
  device options for early boot display without using legacy VGA
  emulation
  * Live migration support for TPM TIS devices, capping bandwidth
  usage during post-copy migration, and recovering from a failed
  post-copy migration
  * Improved latency when using user-mode networking / SLIRP
  * ARM: support for SMMUv3 IOMMU when using 'virt' machine type
  * ARM: v8M extensions for VLLDM and VLSTM floating-point instructions,
  and improved support for AArch64 v8.2 FP16 extensions
  * ARM: support for Scalable Vector Extensions in linux-user mode
  * Microblaze: support for 64-bit address sizes and translation bug
  fixes
  * PowerPC: PMU support for mac99 machine type and improvements for
  Uninorth PCI host bridge emulation for Mac machine types
  * PowerPC: preliminary support for emulating POWER9 hash MMU mode when
  using powernv machine type
  * RISC-V: improvement for privileged ISA emulation
  * s390: support for z14 ZR1 CPU model
  * s390: bpb/ppa15 Spectre mitigations enabled by default for z196 and
  later CPU models
  * s390: support for configuring consoles via -serial options
  * Patches dropped (upstream unless otherwise noted):
  0008-linux-user-fix-segfault-deadlock.patch (no longer needed)
  0039-blockjob-Fix-assertion-in-block_job.patch
  0041-seccomp-allow-sched_setscheduler-wi.patch
  Make-installed-scripts-explicitly-python3.patch (we now make
  python3 explicit in other patch)
  * Patches renamed:
  0009-linux-user-binfmt-support-host-bina.patch
  - > 0008-linux-user-binfmt-support-host-bina.patch
  0010-linux-user-Fake-proc-cpuinfo.patch
  - > 0009-linux-user-Fake-proc-cpuinfo.patch
  0011-Remove-problematic-evdev-86-key-fro.patch
  - > 0010-Remove-problematic-evdev-86-key-fro.patch
  0012-linux-user-use-target_ulong.patch
  - > 0011-linux-user-use-target_ulong.patch
  0013-Make-char-muxer-more-robust-wrt-sma.patch
  - > 0012-Make-char-muxer-more-robust-wrt-sma.patch
  0014-linux-user-lseek-explicitly-cast-no.patch
  - > 0013-linux-user-lseek-explicitly-cast-no.patch
  0015-AIO-Reduce-number-of-threads-for-32.patch
  - > 0014-AIO-Reduce-number-of-threads-for-32.patch
  0016-xen_disk-Add-suse-specific-flush-di.patch
  - > 0015-xen_disk-Add-suse-specific-flush-di.patch
  0017-qemu-bridge-helper-reduce-security-.patch
  - > 0016-qemu-bridge-helper-reduce-security-.patch
  0018-qemu-binfmt-conf-use-qemu-ARCH-binf.patch
  - > 0017-qemu-binfmt-conf-use-qemu-ARCH-binf.patch
  0019-linux-user-properly-test-for-infini.patch
  - > 0018-linux-user-properly-test-for-infini.patch
  0020-roms-Makefile-pass-a-packaging-time.patch
  - > 0019-roms-Makefile-pass-a-packaging-time.patch
  0021-Raise-soft-address-space-limit-to-h.patch
  - > 0020-Raise-soft-address-space-limit-to-h.patch
  0022-increase-x86_64-physical-bits-to-42.patch
  - > 0021-increase-x86_64-physical-bits-to-42.patch
  0023-vga-Raise-VRAM-to-16-MiB-for-pc-0.1.patch
  - > 0022-vga-Raise-VRAM-to-16-MiB-for-pc-0.1.patch
  0024-i8254-Fix-migration-from-SLE11-SP2.patch
  - > 0023-i8254-Fix-migration-from-SLE11-SP2.patch
  0025-acpi_piix4-Fix-migration-from-SLE11.patch
  - > 0024-acpi_piix4-Fix-migration-from-SLE11.patch
  0026-Fix-tigervnc-long-press-issue.patch
  - > 0025-Fix-tigervnc-long-press-issue.patch
  0027-string-input-visitor-Fix-uint64-par.patch
  - > 0026-string-input-visitor-Fix-uint64-par.patch
  0028-test-string-input-visitor-Add-int-t.patch
  - > 0027-test-string-input-visitor-Add-int-t.patch
  0029-test-string-input-visitor-Add-uint6.patch
  - > 0028-test-string-input-visitor-Add-uint6.patch
  0030-tests-Add-QOM-property-unit-tests.patch
  - > 0029-tests-Add-QOM-property-unit-tests.patch
  0031-tests-Add-scsi-disk-test.patch
  - > 0030-tests-Add-scsi-disk-test.patch
  0032-Switch-order-of-libraries-for-mpath.patch
  - > 0031-Switch-order-of-libraries-for-mpath.patch
  0033-Make-installed-scripts-explicitly-p.patch
  - > 0032-Make-installed-scripts-explicitly-p.patch (python2->python3)
  0034-migration-warn-about-inconsistent-s.patch
  - > 0033-migration-warn-about-inconsistent-s.patch
  0035-smbios-Add-1-terminator-if-any-stri.patch
  - > 0034-smbios-Add-1-terminator-if-any-stri.patch
  0036-configure-Modify-python-used-for-io.patch
  - > 0035-configure-Modify-python-used-for-io.patch
  0037-qemu-io-tests-comment-out-problemat.patch
  - > 0036-qemu-io-tests-comment-out-problemat.patch
  0038-tests-test-thread-pool-is-racy-add-.patch
  - > 0037-tests-test-thread-pool-is-racy-add-.patch
  0040-xen-add-block-resize-support-for-xe.patch
  - > 0038-xen-add-block-resize-support-for-xe.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-3.0
* Thu Aug  9 2018 ldewey@suse.com
- Update QEMU to allow kvm group access to /dev/sev (bsc#1102604).
  71-sev.rules
* Fri Aug  3 2018 brogers@suse.com
- Update to v2.12.1, a stable, (mostly) bug-fix-only release
  * This update contains new mitigation functionality for
  CVE-2018-3639 (Speculative Store Bypass) in x86. There are also
  bug fixes for migration, Intel IOMMU emulation, block layer/image
  handling, ARM emulation, and various other areas.
  (Note that a number of 2.12.1 patches were already included by us
  previously) (CVE-2018-3639 bsc#1092885)
  * Patches dropped (subsumed by stable update):
  0039-device_tree-Increase-FDT_MAX_SIZE-t.patch
  0040-vnc-fix-use-after-free.patch
  0041-ccid-Fix-dwProtocols-advertisement-.patch
  0042-tcg-arm-Fix-memory-barrier-encoding.patch
  0043-s390-ccw-force-diag-308-subcode-to-.patch
  0044-nbd-client-fix-nbd_negotiate_simple.patch
  0045-migration-block-dirty-bitmap-fix-me.patch
  0046-nbd-client-Fix-error-messages-durin.patch
  0047-nbd-client-Relax-handling-of-large-.patch
  0048-qxl-fix-local-renderer-crash.patch
  0049-tcg-Limit-the-number-of-ops-in-a-TB.patch
  0050-target-arm-Clear-SVE-high-bits-for-.patch
  0051-cpus-tcg-fix-never-exiting-loop-on-.patch
  0052-s390x-css-disabled-subchannels-cann.patch
  0053-pc-bios-s390-ccw-struct-tpi_info-mu.patch
  0054-virtio-ccw-common-reset-handler.patch
  0055-s390x-ccw-make-sure-all-ccw-devices.patch
  0056-blockjob-expose-error-string-via-qu.patch
  0058-qemu-io-Use-purely-string-blockdev-.patch
  0059-qemu-img-Use-only-string-options-in.patch
  0060-nfs-Remove-processed-options-from-Q.patch
  0061-i386-define-the-ssbd-CPUID-feature-.patch
  0062-i386-Define-the-Virt-SSBD-MSR-and-h.patch
  0063-i386-define-the-AMD-virt-ssbd-CPUID.patch
  0064-ahci-fix-PxCI-register-race.patch
  0065-ccid-card-passthru-fix-regression-i.patch
  * Patches renamed:
  0057-blockjob-Fix-assertion-in-block_job.patch
  - > 0039-blockjob-Fix-assertion-in-block_job.patch
  0066-xen-add-block-resize-support-for-xe.patch
  - > 0040-xen-add-block-resize-support-for-xe.patch
  0067-seccomp-allow-sched_setscheduler-wi.patch
  - > 0041-seccomp-allow-sched_setscheduler-wi.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.12
* Wed Jul 25 2018 ldewey@suse.com
- Fixing seccomp resourcecontrol defunct issue (bsc#1102627)
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.12
  * Patches added:
  0067-seccomp-allow-sched_setscheduler-wi.patch
* Wed Jul 25 2018 matz@suse.com
- Add ipxe-fix-build.patch to not error out with
  binutils >= 2.31 .
* Thu Jul  5 2018 brogers@suse.com
- Remove linux-user patch which is no longer needed (bsc#1098056)
  * Patches dropped:
  0011-linux-user-XXX-disable-fiemap.patch
  * Patches renamed:
  0036-Remove-problematic-evdev-86-key-fro.patch
  - > 0011-Remove-problematic-evdev-86-key-fro.patch
  0037-configure-Modify-python-used-for-io.patch
  - > 0036-configure-Modify-python-used-for-io.patch
  0038-qemu-io-tests-comment-out-problemat.patch
  - > 0037-qemu-io-tests-comment-out-problemat.patch
  0039-tests-test-thread-pool-is-racy-add-.patch
  - > 0038-tests-test-thread-pool-is-racy-add-.patch
  0040-device_tree-Increase-FDT_MAX_SIZE-t.patch
  - > 0039-device_tree-Increase-FDT_MAX_SIZE-t.patch
  0041-vnc-fix-use-after-free.patch
  - > 0040-vnc-fix-use-after-free.patch
  0042-ccid-Fix-dwProtocols-advertisement-.patch
  - > 0041-ccid-Fix-dwProtocols-advertisement-.patch
  0043-tcg-arm-Fix-memory-barrier-encoding.patch
  - > 0042-tcg-arm-Fix-memory-barrier-encoding.patch
  0044-s390-ccw-force-diag-308-subcode-to-.patch
  - > 0043-s390-ccw-force-diag-308-subcode-to-.patch
  0045-nbd-client-fix-nbd_negotiate_simple.patch
  - > 0044-nbd-client-fix-nbd_negotiate_simple.patch
  0046-migration-block-dirty-bitmap-fix-me.patch
  - > 0045-migration-block-dirty-bitmap-fix-me.patch
  0047-nbd-client-Fix-error-messages-durin.patch
  - > 0046-nbd-client-Fix-error-messages-durin.patch
  0048-nbd-client-Relax-handling-of-large-.patch
  - > 0047-nbd-client-Relax-handling-of-large-.patch
  0049-qxl-fix-local-renderer-crash.patch
  - > 0048-qxl-fix-local-renderer-crash.patch
  0050-tcg-Limit-the-number-of-ops-in-a-TB.patch
  - > 0049-tcg-Limit-the-number-of-ops-in-a-TB.patch
  0051-target-arm-Clear-SVE-high-bits-for-.patch
  - > 0050-target-arm-Clear-SVE-high-bits-for-.patch
  0052-cpus-tcg-fix-never-exiting-loop-on-.patch
  - > 0051-cpus-tcg-fix-never-exiting-loop-on-.patch
  0053-s390x-css-disabled-subchannels-cann.patch
  - > 0052-s390x-css-disabled-subchannels-cann.patch
  0054-pc-bios-s390-ccw-struct-tpi_info-mu.patch
  - > 0053-pc-bios-s390-ccw-struct-tpi_info-mu.patch
  0055-virtio-ccw-common-reset-handler.patch
  - > 0054-virtio-ccw-common-reset-handler.patch
  0056-s390x-ccw-make-sure-all-ccw-devices.patch
  - > 0055-s390x-ccw-make-sure-all-ccw-devices.patch
  0057-blockjob-expose-error-string-via-qu.patch
  - > 0056-blockjob-expose-error-string-via-qu.patch
  0058-blockjob-Fix-assertion-in-block_job.patch
  - > 0057-blockjob-Fix-assertion-in-block_job.patch
  0059-qemu-io-Use-purely-string-blockdev-.patch
  - > 0058-qemu-io-Use-purely-string-blockdev-.patch
  0060-qemu-img-Use-only-string-options-in.patch
  - > 0059-qemu-img-Use-only-string-options-in.patch
  0061-nfs-Remove-processed-options-from-Q.patch
  - > 0060-nfs-Remove-processed-options-from-Q.patch
  0062-i386-define-the-ssbd-CPUID-feature-.patch
  - > 0061-i386-define-the-ssbd-CPUID-feature-.patch
  0063-i386-Define-the-Virt-SSBD-MSR-and-h.patch
  - > 0062-i386-Define-the-Virt-SSBD-MSR-and-h.patch
  0064-i386-define-the-AMD-virt-ssbd-CPUID.patch
  - > 0063-i386-define-the-AMD-virt-ssbd-CPUID.patch
  0065-ahci-fix-PxCI-register-race.patch
  - > 0064-ahci-fix-PxCI-register-race.patch
  0066-ccid-card-passthru-fix-regression-i.patch
  - > 0065-ccid-card-passthru-fix-regression-i.patch
  0067-xen-add-block-resize-support-for-xe.patch
  - > 0066-xen-add-block-resize-support-for-xe.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.12
* Thu Jul  5 2018 brogers@suse.com
- Fix build failure of skiboot with gcc8 compiler
  skiboot-hdata-i2c.c-fix-building-with-gcc8.patch
* Thu Jul  5 2018 brogers@suse.com
- Tweak build service constraints information to avoid failures
* Fri Jun  8 2018 brogers@suse.com
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.12
  * Patches added:
  0067-xen-add-block-resize-support-for-xe.patch
* Thu Jun  7 2018 brogers@suse.com
- Tweak patch file generation to be more git version agnostic.
  Also change update_git.sh to not reformat spec file by default.
* Thu May 31 2018 brogers@suse.com
- Looks like the right fix for the AHCI issue has been identified
  upstream. Turns out to also affect Linux guests as well.
  (bsc#1094406)
  * Patches dropped:
  0065-Revert-replay-don-t-process-async-e.patch
  0066-Revert-replay-avoid-recursive-call-.patch
  0067-Revert-replay-check-return-values-o.patch
  0068-Revert-replay-push-replay_mutex_loc.patch
  * Patches added:
  0065-ahci-fix-PxCI-register-race.patch
- Fix a regresssion introduced in v2.12.0 for ccid-card-passthrough
  (bsc#1095419)
  0066-ccid-card-passthru-fix-regression-i.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.12
* Tue May 29 2018 lma@suse.com
- Fix qemu-guest-agent service issue (bsc#1094898)
* Fri May 25 2018 brogers@suse.com
- Spectre v4 vulnerability mitigation support for KVM guests.
  High level description of vulnerability: Systems with microprocessors
  utilizing speculative execution and speculative execution of memory
  reads before the addresses of all prior memory writes are known may
  allow unauthorized disclosure of information to an attacker with
  local user access via a side-channel analysis.
  This change permits the new x86 cpu feature flag named "ssbd" to be
  presented to the guest, given that the host has this feature, and
  KVM exposes it to the guest as well.
  For this feature to be enabled, via adding it to the qemu commandline
  (eg: -cpu <model>,+spec-ctrl,+ssbd), so the guest OS can take advantage
  of the feature, spec-ctrl and ssbd support is also required in the host.
  Another new x86 cpu feature flag named "virt-ssbd" is also added to
  handle this vulnerability for AMD processors.
  (CVE-2018-3639 bsc#1092885)
  0062-i386-define-the-ssbd-CPUID-feature-.patch
  0063-i386-Define-the-Virt-SSBD-MSR-and-h.patch
  0064-i386-define-the-AMD-virt-ssbd-CPUID.patch
- Replay code introduced an issue for AHCI emulation, where on Windows 10
  I/O would stop randomly, and Windows would then reset the AHCI device.
  The issue is not yet fully identified, but reverting some of those
  changes is at least for now a workaround. (bsc#1094406)
  0065-Revert-replay-don-t-process-async-e.patch
  0066-Revert-replay-avoid-recursive-call-.patch
  0067-Revert-replay-check-return-values-o.patch
  0068-Revert-replay-push-replay_mutex_loc.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.12
* Thu May 17 2018 brogers@suse.com
- Add some upstream fixes targeted for the next stable release
  0040-device_tree-Increase-FDT_MAX_SIZE-t.patch
  0041-vnc-fix-use-after-free.patch
  0042-ccid-Fix-dwProtocols-advertisement-.patch
  0043-tcg-arm-Fix-memory-barrier-encoding.patch
  0044-s390-ccw-force-diag-308-subcode-to-.patch
  0045-nbd-client-fix-nbd_negotiate_simple.patch
  0046-migration-block-dirty-bitmap-fix-me.patch
  0047-nbd-client-Fix-error-messages-durin.patch
  0048-nbd-client-Relax-handling-of-large-.patch
  0049-qxl-fix-local-renderer-crash.patch
  0050-tcg-Limit-the-number-of-ops-in-a-TB.patch
  0051-target-arm-Clear-SVE-high-bits-for-.patch
  0052-cpus-tcg-fix-never-exiting-loop-on-.patch
  0053-s390x-css-disabled-subchannels-cann.patch
  0054-pc-bios-s390-ccw-struct-tpi_info-mu.patch
  0055-virtio-ccw-common-reset-handler.patch
  0056-s390x-ccw-make-sure-all-ccw-devices.patch
  0057-blockjob-expose-error-string-via-qu.patch
  0058-blockjob-Fix-assertion-in-block_job.patch
  0059-qemu-io-Use-purely-string-blockdev-.patch
  0060-qemu-img-Use-only-string-options-in.patch
  0061-nfs-Remove-processed-options-from-Q.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.12
* Wed May 16 2018 brogers@suse.com
- Fix qemu-guest-agent uninstall (bsc#1093169)
- Minor tweak to qemu spec file
* Fri Apr 27 2018 brogers@suse.com
- Update to v2.12.0: See http://wiki.qemu.org/ChangeLog/2.12
  Some noteworthy changes:
  CLI options removed: -tdf, -no-kvm-pit, -drive boot, -net channel,
  - net dump, -hdachs, -drive,if=scsi
  HMP commands removed: usb_add, usb_del, host_net_add, host_net_remove
  Q35 default nic now e1000e
  AMD SEV support
  - smbios supports setting data for type 11 tables
  audio and display support split out as modules
  - nic for simple creation of guest NIC and host back-end
  QMP monitor "out-of-band" capability
  lots of ARM and s390 improvements
- Include more of upstream's in-tree tests in the qemu-testsuite
  package
  * Patches dropped:
  0033-memfd-fix-configure-test.patch
  0034-qapi-use-items-values-intead-of-ite.patch
  0035-qapi-Use-OrderedDict-from-standard-.patch
  0036-qapi-adapt-to-moved-location-of-Str.patch
  0037-qapi-Adapt-to-moved-location-of-mak.patch
  0038-qapi-remove-q-arg-to-diff-when-comp.patch
  0039-qapi-ensure-stable-sort-ordering-wh.patch
  0040-qapi-force-a-UTF-8-locale-for-runni.patch
  0041-scripts-ensure-signrom-treats-data-.patch
  0042-configure-allow-use-of-python-3.patch
  0043-input-add-missing-JIS-keys-to-virti.patch
  0045-pc-fail-memory-hot-plug-unplug-with.patch
  0046-memattrs-add-debug-attribute.patch
  0047-exec-add-ram_debug_ops-support.patch
  0048-exec-add-debug-version-of-physical-.patch
  0049-monitor-i386-use-debug-APIs-when-ac.patch
  0050-machine-add-memory-encryption-prope.patch
  0051-kvm-update-kvm.h-to-include-memory-.patch
  0052-docs-add-AMD-Secure-Encrypted-Virtu.patch
  0053-target-i386-add-Secure-Encrypted-Vi.patch
  0054-qmp-add-query-sev-command.patch
  0055-sev-i386-add-command-to-initialize-.patch
  0056-qmp-populate-SevInfo-fields-with-SE.patch
  0057-sev-i386-register-the-guest-memory-.patch
  0058-kvm-introduce-memory-encryption-API.patch
  0059-hmp-add-info-sev-command.patch
  0060-sev-i386-add-command-to-create-laun.patch
  0061-sev-i386-add-command-to-encrypt-gue.patch
  0062-target-i386-encrypt-bios-rom.patch
  0063-sev-i386-add-support-to-LAUNCH_MEAS.patch
  0064-sev-i386-finalize-the-SEV-guest-lau.patch
  0065-hw-i386-set-ram_debug_ops-when-memo.patch
  0066-sev-i386-add-debug-encrypt-and-decr.patch
  0067-target-i386-clear-C-bit-when-walkin.patch
  0068-include-add-psp-sev.h-header-file.patch
  0069-sev-i386-add-support-to-query-PLATF.patch
  0070-sev-i386-add-support-to-KVM_SEV_GUE.patch
  0071-qmp-add-query-sev-launch-measure-co.patch
  0072-tests-qmp-test-blacklist-query-sev-.patch
  0073-sev-i386-add-migration-blocker.patch
  0074-cpu-i386-populate-CPUID-0x8000_001F.patch
  0075-migration-warn-about-inconsistent-s.patch
  0076-smbios-support-setting-OEM-strings-.patch
  0077-smbios-Add-1-terminator-if-any-stri.patch
  0078-Remove-problematic-evdev-86-key-fro.patch
  0079-tpm-lookup-cancel-path-under-tpm-de.patch
  0080-vga-fix-region-calculation.patch
  skiboot-GCC7-fixes-for-Wimplicit-fallthr.patch
  skiboot-libc-stdio-vsnprintf.c-add-expli.patch
  skiboot-build-LDFLAGS-pass-pie-flag-explicitly-to-ld.patch
  ui-keycodemapdb-Add-missing-QKeyCode-val.patch
  ui-keycodemapdb-Fix-compat-with-py3-dict.patch
  * Patches renamed:
  0044-Make-installed-scripts-explicitly-p.patch
  - > 0033-Make-installed-scripts-explicitly-p.patch
  0075-migration-warn-about-inconsistent-s.patch
  - > 0034-migration-warn-about-inconsistent-s.patch
  0077-smbios-Add-1-terminator-if-any-stri.patch
  - > 0035-smbios-Add-1-terminator-if-any-stri.patch
  0078-Remove-problematic-evdev-86-key-fro.patch
  - > 0036-Remove-problematic-evdev-86-key-fro.patch
  * Patches added:
  0037-configure-Modify-python-used-for-io.patch
  0038-qemu-io-tests-comment-out-problemat.patch
  0039-tests-test-thread-pool-is-racy-add-.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.12
* Fri Apr 27 2018 brogers@suse.com
- Fix autoinstall of qemu-guest-agent by getting the modalias string
  right (bsc#1091143)
* Thu Apr 26 2018 brogers@suse.com
- Guard strncpy call with GCC pragma to disable warning about possible
  incorrect usage, when in fact it is correct. This is for gcc 8
  compatibility (bsc#1090355)
  ipxe-efi-guard-strncpy-with-gcc-warning-ignore-pragma.patch
* Mon Apr 23 2018 lma@suse.com
- Add WantedBy for enable qemu-ga@.service auto start (bsc#1090369)
* Tue Apr 10 2018 kwalter@suse.com
- fix qemu-ga service file name (bsc#1089067)
* Thu Apr  5 2018 brogers@suse.com
- Fix OOB access in VGA emulation (CVE-2018-7858 bsc#1084604)
  0080-vga-fix-region-calculation.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.11
* Thu Apr  5 2018 lyan@suse.com
- Add new look up path "sys/class/tpm" for tpm cancel path based
  on Linux 4.0 change (commit 313d21eeab9282e)(bsc#1070615)
  0079-tpm-lookup-cancel-path-under-tpm-de.patch
* Wed Mar 14 2018 brogers@suse.com
- Fix issue with key codes in qemu v2.11
  0078-Remove-problematic-evdev-86-key-fro.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.11
* Wed Mar 14 2018 lma@suse.com
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.11
  * Patches added:
  0077-smbios-Add-1-terminator-if-any-stri.patch
  bsc#994082 and bsc#1084316
* Fri Mar  9 2018 brogers@suse.com
- Add support for setting OEM strings table (fate#323624)
  0076-smbios-support-setting-OEM-strings-.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.11
* Sat Mar  3 2018 brogers@suse.com
- SLE15 KVM (as targeted for RC1) now has the feature exposed.
  Drop the patch. (bsc#1082276)
  0076-i386-Compensate-for-KVM-SPEC_CTRL-f.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.11
* Fri Mar  2 2018 brogers@suse.com
- Change example qemu-ifup script to not depend on bridge-utils.
  Also update the paths used for ip binary.
* Wed Feb 28 2018 brogers@suse.com
- Eliminate bogus use of CPUID_7_0_EDX_PRED_CMD which we've
  carried since the initial Spectre v2 patch was added. EDX bit
  27 of CPUID Leaf 07H, Sub-leaf 0 provides status on STIBP, and
  not the PRED_CMD MSR. Exposing the STIBP CPUID feature bit to the
  guest is wrong in general, since the VM doesn't directly control
  the scheduling of physical hyperthreads. This is left strictly to
  the L0 hypervisor.
* Thu Feb 22 2018 brogers@suse.com
- Update to v2.11.1, a stable, (mostly) bug-fix-only release
  In addition to bug fixes, of necessity fixes are needed to
  address the Spectre v2 vulnerability by passing along to the
  guest new hardware features introduced by host microcode updates.
  A January 2018 release of qemu initially addressed this issue
  by exposing the feature for all x86 vcpu types, which was the
  quick and dirty approach, but not the proper solution. We remove
  that initial patch and now rely on the upstream solution. This
  update instead defines spec_ctrl and ibpb cpu feature flags as
  well as new cpu models which are clones of existing models with
  either -IBRS or -IBPB added to the end of the model name. These
  new vcpu models explicitly include the new feature(s), whereas
  the feature flags can be added to the cpu parameter as with other
  features.  In short, for continued Spectre v2 protection, ensure
  that either the appropriate cpu feature flag is added to the QEMU
  command-line, or one of the new cpu models is used. Although
  migration from older versions is supported, the new cpu features
  won't be properly exposed to the guest until it is restarted with
  the cpu features explicitly added. A reboot is insufficient.
  A warning patch is added which attempts to detect a migration
  from a qemu version which had the quick and dirty fix (it only
  detects certain cases, but hopefully is helpful.)
  s390x guest vulnerability to Spectre v2 is also addressed in this
  update by including support for bpb and ppa/stfle.81 features.
  (CVE-2017-5715 bsc#1068032)
  For additional information on Spectre v2 as it relates to QEMU,
  see: https://www.qemu.org/2018/02/14/qemu-2-11-1-and-spectre-update/
- Unfortunately, it was found that our current KVM isn't correctly
  indicating support for the spec-ctrl feature, so I've added a patch
  to still detect that support within QEMU. This is of course a
  temporary kludge until KVM gets fixed. (bsc#1082276)
- The SEV support patches are updated to the v9 series.
- Fix incompatibility with recent glibc (boo#1081154)
- Add Supplements tags for the guest agent package in an attempt to
  auto-install for QEMU and Xen SUSE Linux guests (fate#323570)
  * Patches dropped (subsumed by stable update, or reworked in v9):
  0033-i386-kvm-MSR_IA32_SPEC_CTRL-and-MSR.patch
  0050-target-i386-add-memory-encryption-f.patch
  0054-accel-add-Secure-Encrypted-Virtuliz.patch
  0072-sev-Fix-build-for-non-x86-hosts.patch
  * Patches added:
  0033-memfd-fix-configure-test.patch
  0053-target-i386-add-Secure-Encrypted-Vi.patch
  0056-qmp-populate-SevInfo-fields-with-SE.patch
  0072-tests-qmp-test-blacklist-query-sev-.patch
  0073-sev-i386-add-migration-blocker.patch
  0074-cpu-i386-populate-CPUID-0x8000_001F.patch
  0075-migration-warn-about-inconsistent-s.patch
  0076-i386-Compensate-for-KVM-SPEC_CTRL-f.patch
  * Patches renamed (plus some minor code changes):
  0051-machine-add-memory-encryption-prope.patch
  - > 0050-machine-add-memory-encryption-prope.patch
  0052-kvm-update-kvm.h-to-include-memory-.patch
  - > 0051-kvm-update-kvm.h-to-include-memory-.patch
  0053-docs-add-AMD-Secure-Encrypted-Virtu.patch
  - > 0052-docs-add-AMD-Secure-Encrypted-Virtu.patch
  0055-sev-add-command-to-initialize-the-m.patch
  - > 0055-sev-i386-add-command-to-initialize-.patch
  0056-sev-register-the-guest-memory-range.patch
  - > 0057-sev-i386-register-the-guest-memory-.patch
  0057-kvm-introduce-memory-encryption-API.patch
  - > 0058-kvm-introduce-memory-encryption-API.patch
  0058-qmp-add-query-sev-command.patch
  - > 0054-qmp-add-query-sev-command.patch
  0060-sev-add-command-to-create-launch-me.patch
  - > 0060-sev-i386-add-command-to-create-laun.patch
  0061-sev-add-command-to-encrypt-guest-me.patch
  - > 0061-sev-i386-add-command-to-encrypt-gue.patch
  0063-sev-add-support-to-LAUNCH_MEASURE-c.patch
  - > 0063-sev-i386-add-support-to-LAUNCH_MEAS.patch
  0064-sev-Finalize-the-SEV-guest-launch-f.patch
  - > 0064-sev-i386-finalize-the-SEV-guest-lau.patch
  0066-sev-add-debug-encrypt-and-decrypt-c.patch
  - > 0066-sev-i386-add-debug-encrypt-and-decr.patch
  0069-sev-add-support-to-query-PLATFORM_S.patch
  - > 0069-sev-i386-add-support-to-query-PLATF.patch
  0070-sev-add-support-to-KVM_SEV_GUEST_ST.patch
  - > 0070-sev-i386-add-support-to-KVM_SEV_GUE.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.11
* Thu Feb  8 2018 brogers@suse.com
- Add AMD SEV (Secure Encrypted Virtualization) support by taking
  the v7 series of the patches posted to qemu ml. (fate#322124)
  0046-memattrs-add-debug-attribute.patch
  0047-exec-add-ram_debug_ops-support.patch
  0048-exec-add-debug-version-of-physical-.patch
  0049-monitor-i386-use-debug-APIs-when-ac.patch
  0050-target-i386-add-memory-encryption-f.patch
  0051-machine-add-memory-encryption-prope.patch
  0052-kvm-update-kvm.h-to-include-memory-.patch
  0053-docs-add-AMD-Secure-Encrypted-Virtu.patch
  0054-accel-add-Secure-Encrypted-Virtuliz.patch
  0055-sev-add-command-to-initialize-the-m.patch
  0056-sev-register-the-guest-memory-range.patch
  0057-kvm-introduce-memory-encryption-API.patch
  0058-qmp-add-query-sev-command.patch
  0059-hmp-add-info-sev-command.patch
  0060-sev-add-command-to-create-launch-me.patch
  0061-sev-add-command-to-encrypt-guest-me.patch
  0062-target-i386-encrypt-bios-rom.patch
  0063-sev-add-support-to-LAUNCH_MEASURE-c.patch
  0064-sev-Finalize-the-SEV-guest-launch-f.patch
  0065-hw-i386-set-ram_debug_ops-when-memo.patch
  0066-sev-add-debug-encrypt-and-decrypt-c.patch
  0067-target-i386-clear-C-bit-when-walkin.patch
  0068-include-add-psp-sev.h-header-file.patch
  0069-sev-add-support-to-query-PLATFORM_S.patch
  0070-sev-add-support-to-KVM_SEV_GUEST_ST.patch
  0071-qmp-add-query-sev-launch-measure-co.patch
  0072-sev-Fix-build-for-non-x86-hosts.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.11
* Thu Feb  8 2018 brogers@suse.com
- Update python3 related patches now that they are upstream
* Thu Feb  8 2018 jfehlig@suse.com
- guest agent: change service file to a template so it can be
  used by Xen as well. Adjust udev rule accordingly.
  FATE#324963
* Mon Jan 29 2018 brogers@suse.com
- Fix machine inconsistency with -no-acpi and nvdimm (bsc#1077823)
  0045-pc-fail-memory-hot-plug-unplug-with.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.11
* Fri Jan 26 2018 brogers@suse.com
- Modify BuildRequires python references
- seabios also needed tweaks for python2 vs python3
  * Patches added:
  seabios-use-python2-explicitly-as-needed.patch
  seabios-switch-to-python3-as-needed.patch
* Thu Jan 25 2018 brogers@suse.com
- Try to get our story right wrt python2 vs python3 (bsc#1077564)
  * Get rid of use of #!/usr/bin/env python in scripts we install
  * include proposed upstream build system changes needed for building
  with python2 or python3
  * Patches dropped:
  0032-scripts-avoid-usr-bin-python-refere.patch
  * Patches renamed:
  0033-Switch-order-of-libraries-for-mpath.patch
  - > 0032-Switch-order-of-libraries-for-mpath.patch
  0034-i386-kvm-MSR_IA32_SPEC_CTRL-and-MSR.patch
  - > 0033-i386-kvm-MSR_IA32_SPEC_CTRL-and-MSR.patch
  * Patches added:
  0034-qapi-use-items-values-intead-of-ite.patch
  0035-qapi-Use-OrderedDict-from-standard-.patch
  0036-qapi-adapt-to-moved-location-of-Str.patch
  0037-qapi-Adapt-to-moved-location-of-mak.patch
  0038-qapi-remove-q-arg-to-diff-when-comp.patch
  0039-qapi-ensure-stable-sort-ordering-wh.patch
  0040-qapi-force-a-UTF-8-locale-for-runni.patch
  0041-scripts-ensure-signrom-treats-data-.patch
  0042-configure-allow-use-of-python-3.patch
  0043-input-add-missing-JIS-keys-to-virti.patch
  0044-Make-installed-scripts-explicitly-p.patch
  Make-installed-scripts-explicitly-python3.patch
  ui-keycodemapdb-Add-missing-QKeyCode-val.patch
  ui-keycodemapdb-Fix-compat-with-py3-dict.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.11
* Fri Jan 12 2018 brogers@suse.com
- Fix packaging dependencies (coreutils) for qemu-ksm package
  (bsc#1040202)
* Thu Jan  4 2018 brogers@suse.com
- Pass through to guest info related to x86 security vulnerability
  (CVE-2017-5715 bsc#1068032)
  0034-i386-kvm-MSR_IA32_SPEC_CTRL-and-MSR.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.11
* Wed Dec 13 2017 brogers@suse.com
- Update to v2.11.0: See http://wiki.qemu.org/ChangeLog/2.11
  Some noteworthy changes:
  - nodefconfig is now deprecated
  legacy pci-assignment code removed
  qemu-pr-helper added for handling guest persistant reservations (bsc#891066,
  bsc#910704, bsc#943807)
  qemu-keymap tool added for generating keymap files
  throttle block filter driver added
  support for a TPM emulator
  qcow2 image shrink support
  better support for >=64 vcpus for Windows guests
  nested KVM related improvements
  s390 pgste handling now done better
  EPYC cpu model added (bsc#1052825)
  improvements in qcow2 buffer handling
  vhost-user resume issue fixed
  migration hardening
  ARMv8-M security extension support
  more seccomp/sandboxing options available
  s390 cpu hot-plug improvements
  misc. virtfs improvements
  nbd improvements
  MTTCG improvements
  misc. TCG improvements
  scsi correctness improvements
  SEABIOS now has serial output option
  * Includes fixes for CVE-2017-15118 bsc#1070147, CVE-2017-15119 bsc#1070144
  * Adds KASLR support (fate#323473, bsc#1070281)
  * Update SLE support docs to match this release
  * simplify spec file to expect at least sle_version >= 1315
  * Patches dropped (upstream):
  0013-console-add-question-mark-escape-op.patch
  0020-configure-Fix-detection-of-seccomp-.patch
  0034-target-i386-cpu-Add-new-EPYC-CPU-mo.patch
  0035-chardev-baum-fix-baum-that-releases.patch
  0036-io-fix-temp-directory-used-by-test-.patch
  0037-io-fix-check-for-handshake-completi.patch
  0038-crypto-fix-test-cert-generation-to-.patch
  0039-vhost-user-disable-the-broken-subpr.patch
  0040-io-monitor-encoutput-buffer-size-fr.patch
  0041-cirrus-fix-oob-access-in-mode4and5-.patch
  0042-9pfs-use-g_malloc0-to-allocate-spac.patch
  * Patches renamed:
  0014-Make-char-muxer-more-robust-wrt-sma.patch
  - > 0013-Make-char-muxer-more-robust-wrt-sma.patch
  0015-linux-user-lseek-explicitly-cast-no.patch
  - > 0014-linux-user-lseek-explicitly-cast-no.patch
  0016-AIO-Reduce-number-of-threads-for-32.patch
  - > 0015-AIO-Reduce-number-of-threads-for-32.patch
  0017-xen_disk-Add-suse-specific-flush-di.patch
  - > 0016-xen_disk-Add-suse-specific-flush-di.patch
  0018-qemu-bridge-helper-reduce-security-.patch
  - > 0017-qemu-bridge-helper-reduce-security-.patch
  0019-qemu-binfmt-conf-use-qemu-ARCH-binf.patch
  - > 0018-qemu-binfmt-conf-use-qemu-ARCH-binf.patch
  0021-linux-user-properly-test-for-infini.patch
  - > 0019-linux-user-properly-test-for-infini.patch
  0022-roms-Makefile-pass-a-packaging-time.patch
  - > 0020-roms-Makefile-pass-a-packaging-time.patch
  0023-Raise-soft-address-space-limit-to-h.patch
  - > 0021-Raise-soft-address-space-limit-to-h.patch
  0024-increase-x86_64-physical-bits-to-42.patch
  - > 0022-increase-x86_64-physical-bits-to-42.patch
  0025-vga-Raise-VRAM-to-16-MiB-for-pc-0.1.patch
  - > 0023-vga-Raise-VRAM-to-16-MiB-for-pc-0.1.patch
  0026-i8254-Fix-migration-from-SLE11-SP2.patch
  - > 0024-i8254-Fix-migration-from-SLE11-SP2.patch
  0027-acpi_piix4-Fix-migration-from-SLE11.patch
  - > 0025-acpi_piix4-Fix-migration-from-SLE11.patch
  0028-Fix-tigervnc-long-press-issue.patch
  - > 0026-Fix-tigervnc-long-press-issue.patch
  0029-string-input-visitor-Fix-uint64-par.patch
  - > 0027-string-input-visitor-Fix-uint64-par.patch
  0030-test-string-input-visitor-Add-int-t.patch
  - > 0028-test-string-input-visitor-Add-int-t.patch
  0031-test-string-input-visitor-Add-uint6.patch
  - > 0029-test-string-input-visitor-Add-uint6.patch
  0032-tests-Add-QOM-property-unit-tests.patch
  - > 0030-tests-Add-QOM-property-unit-tests.patch
  0033-tests-Add-scsi-disk-test.patch
  - > 0031-tests-Add-scsi-disk-test.patch
  0043-scripts-avoid-usr-bin-python-refere.patch
  - > 0032-scripts-avoid-usr-bin-python-refere.patch
  * We need the multipath libraries link order switched
  0033-Switch-order-of-libraries-for-mpath.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.11
* Wed Nov 29 2017 brogers@suse.com
- Avoid ref to /usr/bin/python in vmstate-static-checker.py script
  0043-scripts-avoid-usr-bin-python-refere.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.10
* Tue Nov 28 2017 brogers@suse.com
- For SLE15, it's been decided to stop providing SDL based graphics
  due to packaging constraints. Long ago GTK became the default,
  and there is little benefit to providing both. For now, keep it
  enabled for openSUSE (Tumblweed and Leap), but consider it marked
  deprecated there and if no one complains it will be removed for
  openSUSE as well in the near future. (fate#324465)
- Fix problem building skiboot.lid
  skiboot-build-LDFLAGS-pass-pie-flag-explicitly-to-ld.patch
* Thu Oct 26 2017 lyan@suse.com
- Wrap analyze-migration and vmstate-static-checker into tools from
  qemu scripts folder, also changed introduction of qemu-tools in
  spec file
- Move supportplugin position in spec file
* Thu Oct 19 2017 brogers@suse.com
- Add announcement in support docs about qed storage format no
  longer being supported in next major SLE release (SLE15)
  (fate#324200)
- Address various security/stability issues
  * Fix DoS in I/O channel websockets (CVE-2017-15268 bsc#1062942)
  0040-io-monitor-encoutput-buffer-size-fr.patch
  * Fix OOB access in cirrus vga device emulation (CVE-2017-15289
  bsc#1063122)
  0041-cirrus-fix-oob-access-in-mode4and5-.patch
  * Fix information leak in 9pfs interface (CVE-2017-15038 bsc#1062069)
  0042-9pfs-use-g_malloc0-to-allocate-spac.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.10
* Mon Oct 16 2017 brogers@suse.com
- Don't tie glusterfs support to specific arch
- Build skiboot firmware (OPAL), particularly since it's fairly
  easy to do so
  skiboot-GCC7-fixes-for-Wimplicit-fallthr.patch
  skiboot-libc-stdio-vsnprintf.c-add-expli.patch
* Fri Oct 13 2017 henrik.kuhn@origenis.de
- Added the global macro 'with_glusterfs' in order to re-enable
  glusterfs support. The macro enable easier future adjustments
  for various ARCH/targets/requiremnets.
  At first glusterfs support is enabled for openSUSE Leap 42.x and
  Factory for ARCH x86_64.
* Wed Oct  4 2017 brogers@suse.com
- Add dependencies on ovmf (uefi) for the qemu-x86 and qemu-arm
  packages
- Fix s390-netboot.img to be included with qemu-s390 package, not
  qemu-ppc
* Tue Oct  3 2017 brogers@suse.com
- Update to v2.10.1, a stable, bug-fix-only release
  * fixes bsc#1056386 CVE-2017-13673, bsc#1056334 CVE-2017-13672,
  bsc#1057585 CVE-2017-14167
  * Patches dropped (upstream):
  0034-slirp-fix-clearing-ifq_so-from-pend.patch
  0035-s390-ccw-Fix-alignment-for-CCW1.patch
  0038-s390x-ais-for-2.10-stable-disable-a.patch
  0039-s390x-cpumodel-remove-ais-from-z14-.patch
  * Patches renamed:
  0036-target-i386-cpu-Add-new-EPYC-CPU-mo.patch
  - > 0034-target-i386-cpu-Add-new-EPYC-CPU-mo.patch
  0037-chardev-baum-fix-baum-that-releases.patch
  - > 0035-chardev-baum-fix-baum-that-releases.patch
  0040-io-fix-temp-directory-used-by-test-.patch
  - > 0036-io-fix-temp-directory-used-by-test-.patch
  0041-io-fix-check-for-handshake-completi.patch
  - > 0037-io-fix-check-for-handshake-completi.patch
  0042-crypto-fix-test-cert-generation-to-.patch
  - > 0038-crypto-fix-test-cert-generation-to-.patch
  0043-vhost-user-disable-the-broken-subpr.patch
  - > 0039-vhost-user-disable-the-broken-subpr.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.10
* Mon Oct  2 2017 brogers@suse.com
- Fix failures and potential failures in qemu-testsuite
  0040-io-fix-temp-directory-used-by-test-.patch
  0041-io-fix-check-for-handshake-completi.patch
  0042-crypto-fix-test-cert-generation-to-.patch
  0043-vhost-user-disable-the-broken-subpr.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.10
* Thu Sep 28 2017 brogers@suse.com
- Fix migration issue on s390
  0038-s390x-ais-for-2.10-stable-disable-a.patch
  0039-s390x-cpumodel-remove-ais-from-z14-.patch
- Fix case of not being able to build from rpm sources due to
  undefined macro (boo#1057966)
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.10
* Thu Sep 28 2017 lyan@suse.com
- Fix baum that release brlapi twice (bsc#1060045)
  0037-chardev-baum-fix-baum-that-releases.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.10
* Thu Sep 21 2017 brogers@suse.com
- For SLE15 pre-release testing, add support for the EPYC processor.
  This will be officially supported once it is included in the v2.11
  release. (bsc#1052825)
  0036-target-i386-cpu-Add-new-EPYC-CPU-mo.patch
- Fix some support statements in our SLE support documents.
* Thu Aug 31 2017 brogers@suse.com
- Update BuildRequires packages libibverbs-devel and librdmacm-devel
  to the more correct rdma-core-devel
- Enable seccomp for s390x, aarch64, and ppc64le
- Fix OOB issue (use after free) in slirp network stack (CVE-2017-13711
  bsc#1056291)
  0034-slirp-fix-clearing-ifq_so-from-pend.patch
- Fix a misalignment in the s390 ccw firmware (bsc#1056680)
  0035-s390-ccw-Fix-alignment-for-CCW1.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.10
* Thu Aug 31 2017 jfehlig@suse.com
- Add a supportconfig plugin
  qemu-supportconfig
  FATE#323661
* Wed Aug 30 2017 brogers@suse.com
- Update to v2.10.0: See http://wiki.qemu.org/ChangeLog/2.10
- Dropped internal only patches used to support SUSE Studio
  Testdrive as well as other miscellaneous patches deemed unused
  and not worth carrying (bsc#1046783, bsc#1055125, bsc#1055127)
- Update SLE support statements in anticipation of SLE15
- disable SAN boot capability from virtio pxe rom used in v1.4 and
  older pc machine types due to rom size requirements. Hopefully
  a better solution can be found which doesn't impact functionality
  * Patches added:
  ipxe-stub-out-the-SAN-req-s-in-int13.patch
  * Patches renamed:
  0006-qemu-cvs-gettimeofday.patch -> 0003-qemu-cvs-gettimeofday.patch
  0007-qemu-cvs-ioctl_debug.patch -> 0004-qemu-cvs-ioctl_debug.patch
  0008-qemu-cvs-ioctl_nodirection.patch -> 0005-qemu-cvs-ioctl_nodirection.patch
  0009-linux-user-add-binfmt-wrapper-for-a.patch -> 0006-linux-user-add-binfmt-wrapper-for-a.patch
  0010-PPC-KVM-Disable-mmu-notifier-check.patch -> 0007-PPC-KVM-Disable-mmu-notifier-check.patch
  0011-linux-user-fix-segfault-deadlock.patch -> 0008-linux-user-fix-segfault-deadlock.patch
  0012-linux-user-binfmt-support-host-bina.patch -> 0009-linux-user-binfmt-support-host-bina.patch
  0013-linux-user-Fake-proc-cpuinfo.patch -> 0010-linux-user-Fake-proc-cpuinfo.patch
  0014-linux-user-XXX-disable-fiemap.patch -> 0011-linux-user-XXX-disable-fiemap.patch
  0017-linux-user-use-target_ulong.patch -> 0012-linux-user-use-target_ulong.patch
  0021-console-add-question-mark-escape-op.patch -> 0013-console-add-question-mark-escape-op.patch
  0022-Make-char-muxer-more-robust-wrt-sma.patch -> 0014-Make-char-muxer-more-robust-wrt-sma.patch
  0023-linux-user-lseek-explicitly-cast-no.patch -> 0015-linux-user-lseek-explicitly-cast-no.patch
  0025-AIO-Reduce-number-of-threads-for-32.patch -> 0016-AIO-Reduce-number-of-threads-for-32.patch
  0027-xen_disk-Add-suse-specific-flush-di.patch -> 0017-xen_disk-Add-suse-specific-flush-di.patch
  0028-qemu-bridge-helper-reduce-security-.patch -> 0018-qemu-bridge-helper-reduce-security-.patch
  0029-qemu-binfmt-conf-use-qemu-ARCH-binf.patch -> 0019-qemu-binfmt-conf-use-qemu-ARCH-binf.patch
  0030-configure-Fix-detection-of-seccomp-.patch -> 0020-configure-Fix-detection-of-seccomp-.patch
  0031-linux-user-properly-test-for-infini.patch -> 0020-linux-user-properly-test-for-infini.patch
  0033-roms-Makefile-pass-a-packaging-time.patch -> 0022-roms-Makefile-pass-a-packaging-time.patch
  0034-Raise-soft-address-space-limit-to-h.patch -> 0023-Raise-soft-address-space-limit-to-h.patch
  0035-increase-x86_64-physical-bits-to-42.patch -> 0024-increase-x86_64-physical-bits-to-42.patch
  0036-vga-Raise-VRAM-to-16-MiB-for-pc-0.1.patch -> 0025-vga-Raise-VRAM-to-16-MiB-for-pc-0.1.patch
  0037-i8254-Fix-migration-from-SLE11-SP2.patch -> 0026-i8254-Fix-migration-from-SLE11-SP2.patch
  0038-acpi_piix4-Fix-migration-from-SLE11.patch -> 0027-acpi_piix4-Fix-migration-from-SLE11.patch
  0039-Fix-tigervnc-long-press-issue.patch -> 0028-Fix-tigervnc-long-press-issue.patch
  0041-string-input-visitor-Fix-uint64-par.patch -> 0029-string-input-visitor-Fix-uint64-par.patch
  0042-test-string-input-visitor-Add-int-t.patch -> 0030-test-string-input-visitor-Add-int-t.patch
  0043-test-string-input-visitor-Add-uint6.patch -> 0031-test-string-input-visitor-Add-uint6.patch
  0044-tests-Add-QOM-property-unit-tests.patch -> 0032-tests-Add-QOM-property-unit-tests.patch
  0045-tests-Add-scsi-disk-test.patch -> 0033-tests-Add-scsi-disk-test.patch
  * Patches dropped (upstream unless otherwise noted):
  0003-qemu-cvs-alsa_bitfield.patch (deemed not needed)
  0004-qemu-cvs-alsa_ioctl.patch (deemed not needed)
  0005-qemu-cvs-alsa_mmap.patch (deemed not needed)
  0015-slirp-nooutgoing.patch (bsc#1055125)
  0016-vnc-password-file-and-incoming-conn.patch (bsc#1055127)
  0018-block-Add-support-for-DictZip-enabl.patch (bsc#1046783)
  0019-block-Add-tar-container-format.patch (bsc#1046783)
  0020-Legacy-Patch-kvm-qemu-preXX-dictzip.patch (bsc#1046783)
  0024-configure-Enable-PIE-for-ppc-and-pp.patch (obsolete)
  0026-dictzip-Fix-on-big-endian-systems.patch (bsc#1046783)
  0032-linux-user-remove-all-traces-of-qem.patch
  0040-fix-xen-hvm-direct-kernel-boot.patch (bsc#970791)
  0046-RFC-update-Linux-headers-from-irqs-.patch
  0047-ARM-KVM-Enable-in-kernel-timers-wit.patch
  0048-input-Add-trace-event-for-empty-key.patch
  0049-ACPI-don-t-call-acpi_pcihp_device_p.patch
  0050-i386-Allow-cpuid-bit-override.patch (was for testing only)
  0051-input-limit-kbd-queue-depth.patch
  0052-audio-release-capture-buffers.patch
  0053-scsi-avoid-an-off-by-one-error-in-m.patch
  0054-vmw_pvscsi-check-message-ring-page-.patch
  0055-9pfs-local-forbid-client-access-to-.patch
  0056-jazz_led-fix-bad-snprintf.patch
  0057-slirp-smb-Replace-constant-strings-.patch
  0058-altera_timer-fix-incorrect-memset.patch
  0059-Hacks-for-building-on-gcc-7-Fedora-.patch
  0060-9pfs-local-fix-unlink-of-alien-file.patch
  0061-megasas-do-not-read-DCMD-opcode-mor.patch
  0062-megasas-always-store-SCSIRequest-in.patch
  0063-nbd-Fully-initialize-client-in-case.patch
  0064-9pfs-local-remove-use-correct-path-.patch
  0065-hid-Reset-kbd-modifiers-on-reset.patch
  0066-input-Decrement-queue-count-on-kbd-.patch
  0067-xhci-only-update-dequeue-ptr-on-com.patch
  0068-vnc-Set-default-kbd-delay-to-10ms.patch
  0069-qemu-nbd-Ignore-SIGPIPE.patch
  0070-usb-redir-fix-stack-overflow-in-usb.patch
  0072-slirp-check-len-against-dhcp-option.patch
  0071-exec-use-qemu_ram_ptr_length-to-acc.patch
  0073-xen-mapcache-store-dma-information-.patch
  0074-exec-Add-lock-parameter-to-qemu_ram.patch
  0075-Replace-struct-ucontext-with-uconte.patch
  ipxe-build-Avoid-implicit-fallthrough-warnings-on-GCC-7.patch
  ipxe-iscsi-Always-send-FirstBurstLength-parameter.patch
  ipxe-ath-Add-missing-break-statements.patch
  ipxe-mucurses-Fix-erroneous-__nonnull-attribute.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.10
* Thu Aug 24 2017 brogers@suse.com
- Fix package build failure as of glibc v2.26 update in Factory
  (boo#1055587)
  0075-Replace-struct-ucontext-with-uconte.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.9
* Mon Aug 21 2017 schwab@suse.de
- Remove redundant prerequire for pwdutils
* Fri Aug 18 2017 brogers@suse.com
- Postrequire acl for setfacl
* Wed Aug 16 2017 schwab@suse.de
- Prerequire shadow for groupadd
* Tue Aug 15 2017 brogers@suse.com
- The recent security fix for CVE-2017-11334 adversely affects Xen.
  Include two additional patches to make sure Xen is going to be OK.
  0073-xen-mapcache-store-dma-information-.patch
  0074-exec-Add-lock-parameter-to-qemu_ram.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.9
* Wed Aug  9 2017 lyan@suse.com
- Pre-add group kvm for qemu-tools (bsc#1011144)
* Mon Aug  7 2017 brogers@suse.com
- Fixed a few more inaccuracies in the support docs.
* Tue Jul 25 2017 brogers@suse.com
- Address various security/stability issues
  * Fix DOS vulnerability in qemu-nbd (bsc#1046636 CVE-2017-10664)
  0069-qemu-nbd-Ignore-SIGPIPE.patch
  * Fix DOS from stack overflow in debug messages of usb redirection
  support (bsc#1047674 CVE-2017-10806)
  0070-usb-redir-fix-stack-overflow-in-usb.patch
  * Fix OOB access during DMA operation (CVE-2017-11334 bsc#1048902)
  0071-exec-use-qemu_ram_ptr_length-to-acc.patch
  * Fix OOB access parsing dhcp slirp options (CVE-2017-11434 bsc#1049381)
  0072-slirp-check-len-against-dhcp-option.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.9
* Tue Jul 25 2017 brogers@suse.com
- Fix support docs to indicate ARM64 is now fully L3 supported in
  SLES 12 SP3. Apply a few additional clarifications in the support
  docs. (bsc#1050268)
- Adjust to libvdeplug-devel package naming changes.
* Tue Jul 18 2017 brogers@suse.com
- Fix migration with xhci (bsc#1048296)
  0067-xhci-only-update-dequeue-ptr-on-com.patch
- Increase VNC delay to fix missing keyboard input events (bsc#1031692)
  0068-vnc-Set-default-kbd-delay-to-10ms.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.9
* Thu Jun 29 2017 lyan@suse.com
- Remove build dependency package iasl used for seabios
* Mon Jun 26 2017 brogers@suse.com
- Fixed stuck state during usb keyboard reset (bsc#1044936)
  0065-hid-Reset-kbd-modifiers-on-reset.patch
- Fixed keyboard events getting lost (bsc#1044936)
  0066-input-Decrement-queue-count-on-kbd-.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.9
* Tue Jun 20 2017 brogers@suse.com
- Use most recent compiler to build size-critical firmware, instead
  of hard-coding gcc6 for all target versions (bsc#1043390)
  * A few upstream ipxe patches were needed for gcc7 compatibility:
  ipxe-ath-Add-missing-break-statements.patch
  ipxe-mucurses-Fix-erroneous-__nonnull-attribute.patch
- Add --no-renames to the git format-patch command in the git
  workflow script for better patch compatibility
- Address various security/stability issues
  * Fix potential privilege escalation in virtfs (CVE-2016-9602
  bsc#1020427)
  0060-9pfs-local-fix-unlink-of-alien-file.patch
  * Fix DOS in megasas device emulation (CVE-2017-9503 bsc#1043296)
  0061-megasas-do-not-read-DCMD-opcode-mor.patch
  0062-megasas-always-store-SCSIRequest-in.patch
  * Fix DOS in qemu-nbd server (CVE-2017-9524 bsc#1043808)
  0063-nbd-Fully-initialize-client-in-case.patch
  * Fix regression introduced by recent virtfs security fixes (bsc#1045035)
  0064-9pfs-local-remove-use-correct-path-.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.9
* Tue Jun  6 2017 lyan@suse.com
- Backport ipxe to support FirstBurstLength (bsc#1040476)
  ipxe-iscsi-Always-send-FirstBurstLength-parameter.patch
* Mon May 22 2017 brogers@suse.com
- Fixes for gcc7 compatability (bsc#1040228) (in behalf of Liang Yan)
  0056-jazz_led-fix-bad-snprintf.patch
  0057-slirp-smb-Replace-constant-strings-.patch
  0058-altera_timer-fix-incorrect-memset.patch
  0059-Hacks-for-building-on-gcc-7-Fedora-.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.9
* Mon May 22 2017 brogers@suse.com
- Protect access to metadata in virtio-9pfs (CVE-2017-7493 bsc#1039495)
  0055-9pfs-local-forbid-client-access-to-.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.9
* Thu May 11 2017 brogers@suse.com
- Address various security/stability issues
  * Fix DOS potential in vnc interface (CVE-2017-8379 bsc#1037334)
  0051-input-limit-kbd-queue-depth.patch
  * Fix DOS potential in vnc interface (CVE-2017-8309 bsc#1037242)
  0052-audio-release-capture-buffers.patch
  * Fix OOB access in megasas device emulation (CVE-2017-8380
  bsc#1037336)
  0053-scsi-avoid-an-off-by-one-error-in-m.patch
  * Fix DOS in Vmware pv scsi emulation (CVE-2017-8112 bsc#1036211)
  0054-vmw_pvscsi-check-message-ring-page-.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.9
* Thu May  4 2017 brogers@suse.com
- Fix building packages for some older distros.
- Further refine our handling of building firmware (or not) for
  the various arch's and distro versions we build for. Note that
  if we don't build x86 firmware, (eg: x86 Leap 42.1) the upstream
  binary blobs are used, which may have migration incompatibilities
  with previous versions of qemu provided.
* Thu Apr 27 2017 brogers@suse.com
- Fix issue in shipping qemu v2.9.0, where pci-passthrough for Xen
  HVM guests got broken (bsc#1034131)
  0049-ACPI-don-t-call-acpi_pcihp_device_p.patch
- Include experimental, unsupported feature to assist in some
  performance analysis work.
  0050-i386-Allow-cpuid-bit-override.patch
* Thu Apr 20 2017 brogers@suse.com
- Updated to v2.9.0: See http://wiki.qemu-project.org/ChangeLog/2.9
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.9
* Tue Apr 18 2017 brogers@suse.com
- Updated to v2.9.0-rc5: See http://wiki.qemu-project.org/ChangeLog/2.9
  * Includes fix for CVE-2017-7471, a virtfs security issue.
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.9
* Wed Apr 12 2017 brogers@suse.com
- Add empty keyboard queue tracepoint to help openQA testing work
  better (bsc#1031692)
  0048-input-Add-trace-event-for-empty-key.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.9
* Tue Apr 11 2017 brogers@suse.com
- Updated to v2.9.0-rc4: See http://wiki.qemu-project.org/ChangeLog/2.9
- Enable ceph/rbd support for s390x (bsc#1030068)
- Enable ceph/rbd support for ppc* as available
- Update ARM in-kernel-timers patch (bsc#1033416)
  * Patches renamed:
  0041-ARM-KVM-Enable-in-kernel-timers-wit.patch -> 0047-ARM-KVM-Enable-in-kernel-timers-wit.patch
  0042-string-input-visitor-Fix-uint64-par.patch -> 0041-string-input-visitor-Fix-uint64-par.patch
  0043-test-string-input-visitor-Add-int-t.patch -> 0042-test-string-input-visitor-Add-int-t.patch
  0044-test-string-input-visitor-Add-uint6.patch -> 0043-test-string-input-visitor-Add-uint6.patch
  0045-tests-Add-QOM-property-unit-tests.patch -> 0044-tests-Add-QOM-property-unit-tests.patch
  0046-tests-Add-scsi-disk-test.patch -> 0045-tests-Add-scsi-disk-test.patch
  * Patches added (support patch):
  0046-RFC-update-Linux-headers-from-irqs-.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.9
* Wed Apr  5 2017 brogers@suse.com
- Updated to v2.9.0-rc3: See http://wiki.qemu-project.org/ChangeLog/2.9
  * Patches dropped (included in upstream source archive):
  0047-hw-intc-arm_gicv3_kvm-Check-KVM_DEV.patch
  0048-i386-Replace-uint32_t-with-FeatureW.patch
  0049-i386-Don-t-override-cpu-options-on-.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.9
- Added additional documentation provided with v2.9.0
- Fix build failure with gcc7 (bsc#1031340)
  ipxe-build-Avoid-implicit-fallthrough-warnings-on-GCC-7.patch
- Made miscellaneous spec file refinements
* Thu Mar 30 2017 brogers@suse.com
- The support documents included are now fairly accurate for the
  arm and s390 world, and the x86 version also received a few
  tweaks. Also included in those docs is a url reference to upstream
  qemu deprecation plans and discussions.
  (fate#321146)
- Add post v2.9.0-rc2 upstream patches which fix -cpu host and -cpu
  max feature overrides for libvirt compatability.
  0048-i386-Replace-uint32_t-with-FeatureW.patch
  0049-i386-Don-t-override-cpu-options-on-.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.9
* Tue Mar 28 2017 brogers@suse.com
- Updated to v2.9.0-rc2: See http://wiki.qemu-project.org/ChangeLog/2.9
  * Includes fix for in guest privilege escalation when using TCG
  (bsc#1030624)
  * Patches dropped (equivalent included in upstream source archive):
  0047-linux-user-exclude-cpu-model-code-w.patch
- Fix failure booting SLE12-SP2 Aarch64 guest (bsc#1031384)
  0047-hw-intc-arm_gicv3_kvm-Check-KVM_DEV.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.9
* Tue Mar 21 2017 brogers@suse.com
- Updated to v2.9.0-rc1: See http://wiki.qemu-project.org/ChangeLog/2.9
  * Patches dropped (no longer needed based on what we now build for):
  0024-virtfs-proxy-helper-Provide-__u64-f.patch
  * Patches dropped (included in upstream source archive):
  0034-dma-rc4030-limit-interval-timer-rel.patch
  * Patches renamed:
  0025-configure-Enable-PIE-for-ppc-and-pp.patch -> 0024-configure-Enable-PIE-for-ppc-and-pp.patch
  0026-AIO-Reduce-number-of-threads-for-32.patch -> 0025-AIO-Reduce-number-of-threads-for-32.patch
  0027-dictzip-Fix-on-big-endian-systems.patch -> 0026-dictzip-Fix-on-big-endian-systems.patch
  0028-xen_disk-Add-suse-specific-flush-di.patch -> 0027-xen_disk-Add-suse-specific-flush-di.patch
  0029-qemu-bridge-helper-reduce-security-.patch -> 0028-qemu-bridge-helper-reduce-security-.patch
  0030-qemu-binfmt-conf-use-qemu-ARCH-binf.patch -> 0029-qemu-binfmt-conf-use-qemu-ARCH-binf.patch
  0031-configure-Fix-detection-of-seccomp-.patch -> 0030-configure-Fix-detection-of-seccomp-.patch
  0032-linux-user-properly-test-for-infini.patch -> 0031-linux-user-properly-test-for-infini.patch
  0033-linux-user-remove-all-traces-of-qem.patch -> 0032-linux-user-remove-all-traces-of-qem.patch
  0035-roms-Makefile-pass-a-packaging-time.patch -> 0033-roms-Makefile-pass-a-packaging-time.patch
  0036-Raise-soft-address-space-limit-to-h.patch -> 0034-Raise-soft-address-space-limit-to-h.patch
  0037-increase-x86_64-physical-bits-to-42.patch -> 0035-increase-x86_64-physical-bits-to-42.patch
  0038-vga-Raise-VRAM-to-16-MiB-for-pc-0.1.patch -> 0036-vga-Raise-VRAM-to-16-MiB-for-pc-0.1.patch
  0039-i8254-Fix-migration-from-SLE11-SP2.patch -> 0037-i8254-Fix-migration-from-SLE11-SP2.patch
  0040-acpi_piix4-Fix-migration-from-SLE11.patch -> 0038-acpi_piix4-Fix-migration-from-SLE11.patch
  0041-Fix-tigervnc-long-press-issue.patch -> 0039-Fix-tigervnc-long-press-issue.patch
  0042-fix-xen-hvm-direct-kernel-boot.patch -> 0040-fix-xen-hvm-direct-kernel-boot.patch
  0043-ARM-KVM-Enable-in-kernel-timers-wit.patch -> 0041-ARM-KVM-Enable-in-kernel-timers-wit.patch
  0044-string-input-visitor-Fix-uint64-par.patch -> 0042-string-input-visitor-Fix-uint64-par.patch
  0045-test-string-input-visitor-Add-int-t.patch -> 0043-test-string-input-visitor-Add-int-t.patch
  0046-test-string-input-visitor-Add-uint6.patch -> 0044-test-string-input-visitor-Add-uint6.patch
  0047-tests-Add-QOM-property-unit-tests.patch -> 0045-tests-Add-QOM-property-unit-tests.patch
  0048-tests-Add-scsi-disk-test.patch -> 0046-tests-Add-scsi-disk-test.patch
  0049-linux-user-exclude-cpu-model-code-w.patch -> 0047-linux-user-exclude-cpu-model-code-w.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.9
* Mon Mar 20 2017 brogers@suse.com
- Updated to v2.9.0-rc0: See http://wiki.qemu-project.org/ChangeLog/2.9
  * Updated version carries fixes for the following reported issues:
  CVE-2016-9602 bsc#1020427, CVE-2016-9923 bsc#1014703,
  CVE-2017-2630 bsc#1025396, CVE-2017-2633 bsc#1026612,
  CVE-2017-5579 bsc#1021741, CVE-2017-5931 bsc#1024114,
  CVE-2017-5973 bsc#1025109, CVE-2017-5987 bsc#1025311,
  CVE-2017-6058 bsc#1025837, CVE-2017-6505 bsc#1028184
  * Patches dropped:
  seabios_128kb.patch (no longer required)
  * Patches dropped (included in upstream source archive):
  0035-net-imx-limit-buffer-descriptor-cou.patch
  0045-virtio-gpu-call-cleanup-mapping-fun.patch
  0051-virtio-gpu-fix-information-leak-in-.patch
  0052-display-cirrus-ignore-source-pitch-.patch
  0053-s390x-kvm-fix-small-race-reboot-vs..patch
  0054-target-s390x-use-qemu-cpu-model-in-.patch
  0056-tests-check-path-to-avoid-a-failing.patch
  0057-display-virtio-gpu-3d-check-virgl-c.patch
  0058-watchdog-6300esb-add-exit-function.patch
  0059-virtio-gpu-3d-fix-memory-leak-in-re.patch
  0060-virtio-gpu-fix-memory-leak-in-resou.patch
  0061-virtio-fix-vq-inuse-recalc-after-mi.patch
  0062-audio-es1370-add-exit-function.patch
  0063-audio-ac97-add-exit-function.patch
  0064-megasas-fix-guest-triggered-memory-.patch
  0065-cirrus-handle-negative-pitch-in-cir.patch
  0066-cirrus-fix-blit-address-mask-handli.patch
  0067-cirrus-fix-oob-access-issue-CVE-201.patch
  0068-usb-ccid-check-ccid-apdu-length.patch
  0069-sd-sdhci-check-data-length-during-d.patch
  0070-virtio-gpu-fix-resource-leak-in-vir.patch
  0071-cirrus-fix-patterncopy-checks.patch
  0072-cirrus-add-blit_is_unsafe-call-to-c.patch
  * Patches renamed:
  0036-roms-Makefile-pass-a-packaging-time.patch -> 0035-roms-Makefile-pass-a-packaging-time.patch
  0037-Raise-soft-address-space-limit-to-h.patch -> 0036-Raise-soft-address-space-limit-to-h.patch
  0038-increase-x86_64-physical-bits-to-42.patch -> 0037-increase-x86_64-physical-bits-to-42.patch
  0039-vga-Raise-VRAM-to-16-MiB-for-pc-0.1.patch -> 0038-vga-Raise-VRAM-to-16-MiB-for-pc-0.1.patch
  0040-i8254-Fix-migration-from-SLE11-SP2.patch -> 0039-i8254-Fix-migration-from-SLE11-SP2.patch
  0041-acpi_piix4-Fix-migration-from-SLE11.patch -> 0040-acpi_piix4-Fix-migration-from-SLE11.patch
  0042-Fix-tigervnc-long-press-issue.patch -> 0041-Fix-tigervnc-long-press-issue.patch
  0043-fix-xen-hvm-direct-kernel-boot.patch -> 0042-fix-xen-hvm-direct-kernel-boot.patch
  0044-ARM-KVM-Enable-in-kernel-timers-wit.patch -> 0043-ARM-KVM-Enable-in-kernel-timers-wit.patch
  0046-string-input-visitor-Fix-uint64-par.patch -> 0044-string-input-visitor-Fix-uint64-par.patch
  0047-test-string-input-visitor-Add-int-t.patch -> 0045-test-string-input-visitor-Add-int-t.patch
  0048-test-string-input-visitor-Add-uint6.patch -> 0046-test-string-input-visitor-Add-uint6.patch
  0049-tests-Add-QOM-property-unit-tests.patch -> 0047-tests-Add-QOM-property-unit-tests.patch
  0050-tests-Add-scsi-disk-test.patch -> 0048-tests-Add-scsi-disk-test.patch
  0055-linux-user-exclude-cpu-model-code-w.patch -> 0049-linux-user-exclude-cpu-model-code-w.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.9
* Wed Mar  1 2017 brogers@suse.com
-  Buildignore for the global gcc-PIE, as this package enables PIE
  on its own and has troubles if all use it. (meissner@suse.com)
* Wed Mar  1 2017 brogers@suse.com
- Address various security/stability issues
  * Fix OOB access in virito-gpu-3d (CVE-2016-10028 bsc#1017084
  bsc#1016503)
  0057-display-virtio-gpu-3d-check-virgl-c.patch
  * Fix DOS in Intel 6300ESB device emulation (CVE-2016-10155 bsc#1021129)
  0058-watchdog-6300esb-add-exit-function.patch
  * Fix DOS in virtio-gpu-3d (CVE-2017-5552 bsc#1021195)
  0059-virtio-gpu-3d-fix-memory-leak-in-re.patch
  * Fix DOS in virtio-gpu (CVE-2017-5578 bsc#1021481)
  0060-virtio-gpu-fix-memory-leak-in-resou.patch
  * Fix cause of infrequent migration failures from bad virtio device
  state. (bsc#1020928)
  0061-virtio-fix-vq-inuse-recalc-after-mi.patch
  * Fix DOS in es1370 emulated audio device (CVE-2017-5526 bsc#1020589)
  0062-audio-es1370-add-exit-function.patch
  * Fix DOS in ac97 emulated audio device (CVE-2017-5525 bsc#1020491)
  0063-audio-ac97-add-exit-function.patch
  * Fix DOS in megasas device emulation (CVE-2017-5856 bsc#1023053)
  0064-megasas-fix-guest-triggered-memory-.patch
  * Fix various inaccuracies in cirrus vga device emulation
  0065-cirrus-handle-negative-pitch-in-cir.patch
  0066-cirrus-fix-blit-address-mask-handli.patch
  * Fix OOB access in cirrus vga emulation (CVE-2017-2615 bsc#1023004)
  0067-cirrus-fix-oob-access-issue-CVE-201.patch
  * Fix DOS in usb CCID card device emulator (CVE-2017-5898 bsc#1023907)
  0068-usb-ccid-check-ccid-apdu-length.patch
  * Fix OOB access in SDHCI device emulation (CVE-2017-5667 bsc#1022541)
  0069-sd-sdhci-check-data-length-during-d.patch
  * Fix DOS in virtio-gpu-3d (CVE-2017-5857 bsc#1023073)
  0070-virtio-gpu-fix-resource-leak-in-vir.patch
  * Fix cirrus patterncopy checks
  0071-cirrus-fix-patterncopy-checks.patch
  * Fix OOB access in cirrus vga emulation (CVE-2017-2620 bsc#1024972)
  0072-cirrus-add-blit_is_unsafe-call-to-c.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.8
* Thu Feb 23 2017 brogers@suse.com
- Fix name of s390x specific sysctl configuration file to end with
  .conf (bsc#1026583)
* Fri Feb 17 2017 brogers@suse.com
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.8
  * Check that sysfs path exists before running test which requires
  it. This allows qemu-testsuite to succeed in local build service
  chroot based package build.
  0056-tests-check-path-to-avoid-a-failing.patch
* Wed Feb 15 2017 brogers@suse.com
- Factory and SLE12-SP3 got a name change in the dtc devel package:
  libfdt1-devel -> libfdt-devel. Adjust our spec file accordingly.
* Tue Feb 14 2017 brogers@suse.com
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.8
  * Patches added:
  0055-linux-user-exclude-cpu-model-code-w.patch
* Thu Feb  2 2017 brogers@suse.com
- Make sure qemu guest agent is usable as soon as qemu-guest-agent
  package is installed. The previous post script was still not
  doing the job.
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.8
  * Fix potential hang/crash rebooting s390x guest
  0053-s390x-kvm-fix-small-race-reboot-vs..patch
  * Fix s390x linux-user failure since v2.8.0 update
  0054-target-s390x-use-qemu-cpu-model-in-.patch
* Wed Jan 25 2017 brogers@suse.com
- Merge qemu packages from openSUSE and SUSE SLE releases together
  for the v2.8 qemu update. The qemu.changes file is the openSUSE
  version with this entry providing CVE, FATE, and bugzilla
  references from the SUSE SLE qemu package to date (see below)
- Updated to v2.8.0: See http://wiki.qemu-project.org/ChangeLog/2.8
  * For SUSE SLE-12-SP3, update relates to fate#319684, fate#321331,
  fate#321335, fate#321339, fate#321349, fate#321857
  * For best compatibility, qemu-ifup and kvm_stat scripts now owned
  by qemu package
  * Build ipxe roms with gcc6 to maintain SLE legacy migration
  compatibility requirements
  * qmp-commands.txt file removed, to resurface in future doc reorganization
  * qemu-tech.html file merged into other existing doc
  * trace-events renamed to trace-events-all
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.8
  * Patches dropped (upstream):
  0013-linux-user-lock-tcg.patch
  0014-linux-user-Run-multi-threaded-code-.patch
  0015-linux-user-lock-tb-flushing-too.patch
  0017-linux-user-implement-FS_IOC_GETFLAG.patch
  0018-linux-user-implement-FS_IOC_SETFLAG.patch
  0034-xen-SUSE-xenlinux-unplug-for-emulat.patch
  0039-Fix-tlb_vaddr_to_host-with-CONFIG_U.patch
  0041-vmsvga-correct-bitmap-and-pixmap-si.patch
  0042-scsi-mptconfig-fix-an-assert-expres.patch
  0043-scsi-mptconfig-fix-misuse-of-MPTSAS.patch
  0044-scsi-pvscsi-limit-loop-to-fetch-SG-.patch
  0045-usb-xhci-fix-memory-leak-in-usb_xhc.patch
  0046-scsi-mptsas-use-g_new0-to-allocate-.patch
  0047-scsi-pvscsi-limit-process-IO-loop-t.patch
  0048-virtio-add-check-for-descriptor-s-m.patch
  0049-net-mcf-limit-buffer-descriptor-cou.patch
  0050-usb-ehci-fix-memory-leak-in-ehci_pr.patch
  0051-xhci-limit-the-number-of-link-trbs-.patch
  0052-9pfs-allocate-space-for-guest-origi.patch
  0053-9pfs-fix-memory-leak-in-v9fs_link.patch
  0054-9pfs-fix-potential-host-memory-leak.patch
  0055-9pfs-fix-information-leak-in-xattr-.patch
  0056-9pfs-fix-memory-leak-in-v9fs_xattrc.patch
  0057-9pfs-fix-memory-leak-in-v9fs_write.patch
  0058-char-serial-check-divider-value-aga.patch
  0059-net-pcnet-check-rx-tx-descriptor-ri.patch
  0060-net-eepro100-fix-memory-leak-in-dev.patch
  0061-net-rocker-set-limit-to-DMA-buffer-.patch
  0062-net-vmxnet-initialise-local-tx-desc.patch
  0063-net-rtl8139-limit-processing-of-rin.patch
  0064-audio-intel-hda-check-stream-entry-.patch
  0065-virtio-gpu-fix-memory-leak-in-virti.patch
  0066-9pfs-fix-integer-overflow-issue-in-.patch
  slof_xhci.patch
  * Patches renamed:
  0016-linux-user-Fake-proc-cpuinfo.patch -> 0013-linux-user-Fake-proc-cpuinfo.patch
  0019-linux-user-XXX-disable-fiemap.patch -> 0014-linux-user-XXX-disable-fiemap.patch
  0020-slirp-nooutgoing.patch -> 0015-slirp-nooutgoing.patch
  0021-vnc-password-file-and-incoming-conn.patch -> 0016-vnc-password-file-and-incoming-conn.patch
  0022-linux-user-use-target_ulong.patch -> 0017-linux-user-use-target_ulong.patch
  0023-block-Add-support-for-DictZip-enabl.patch -> 0018-block-Add-support-for-DictZip-enabl.patch
  0024-block-Add-tar-container-format.patch -> 0019-block-Add-tar-container-format.patch
  0025-Legacy-Patch-kvm-qemu-preXX-dictzip.patch -> 0020-Legacy-Patch-kvm-qemu-preXX-dictzip.patch
  0026-console-add-question-mark-escape-op.patch -> 0021-console-add-question-mark-escape-op.patch
  0027-Make-char-muxer-more-robust-wrt-sma.patch -> 0022-Make-char-muxer-more-robust-wrt-sma.patch
  0028-linux-user-lseek-explicitly-cast-no.patch -> 0023-linux-user-lseek-explicitly-cast-no.patch
  0029-virtfs-proxy-helper-Provide-__u64-f.patch -> 0024-virtfs-proxy-helper-Provide-__u64-f.patch
  0030-configure-Enable-PIE-for-ppc-and-pp.patch -> 0025-configure-Enable-PIE-for-ppc-and-pp.patch
  0031-AIO-Reduce-number-of-threads-for-32.patch -> 0026-AIO-Reduce-number-of-threads-for-32.patch
  0032-dictzip-Fix-on-big-endian-systems.patch -> 0027-dictzip-Fix-on-big-endian-systems.patch
  0033-xen_disk-Add-suse-specific-flush-di.patch -> 0028-xen_disk-Add-suse-specific-flush-di.patch
  0035-qemu-bridge-helper-reduce-security-.patch -> 0029-qemu-bridge-helper-reduce-security-.patch
  0036-qemu-binfmt-conf-use-qemu-ARCH-binf.patch -> 0030-qemu-binfmt-conf-use-qemu-ARCH-binf.patch
  0037-configure-Fix-detection-of-seccomp-.patch -> 0031-configure-Fix-detection-of-seccomp-.patch
  0038-linux-user-properly-test-for-infini.patch -> 0032-linux-user-properly-test-for-infini.patch
  0040-linux-user-remove-all-traces-of-qem.patch -> 0033-linux-user-remove-all-traces-of-qem.patch
  0067-dma-rc4030-limit-interval-timer-rel.patch -> 0034-dma-rc4030-limit-interval-timer-rel.patch
  0068-net-imx-limit-buffer-descriptor-cou.patch -> 0035-net-imx-limit-buffer-descriptor-cou.patch
  0069-roms-Makefile-pass-a-packaging-time.patch -> 0036-roms-Makefile-pass-a-packaging-time.patch
  * Patches added:
  0037-Raise-soft-address-space-limit-to-h.patch
  0038-increase-x86_64-physical-bits-to-42.patch
  0039-vga-Raise-VRAM-to-16-MiB-for-pc-0.1.patch
  0040-i8254-Fix-migration-from-SLE11-SP2.patch
  0041-acpi_piix4-Fix-migration-from-SLE11.patch
  0042-Fix-tigervnc-long-press-issue.patch
  0043-fix-xen-hvm-direct-kernel-boot.patch
  0044-ARM-KVM-Enable-in-kernel-timers-wit.patch
  0045-virtio-gpu-call-cleanup-mapping-fun.patch
  0046-string-input-visitor-Fix-uint64-par.patch
  0047-test-string-input-visitor-Add-int-t.patch
  0048-test-string-input-visitor-Add-uint6.patch
  0049-tests-Add-QOM-property-unit-tests.patch
  0050-tests-Add-scsi-disk-test.patch
  0051-virtio-gpu-fix-information-leak-in-.patch
  0052-display-cirrus-ignore-source-pitch-.patch
  ipxe-use-gcc6-for-more-compact-code.patch
  * SLE patches dropped (accounted for in above listed changes):
  0002-qemu-0.9.0.cvs-binfmt.patch
  0009-block-vmdk-Support-creation-of-SCSI.patch
  0010-linux-user-add-binfmt-wrapper-for-a.patch
  0011-PPC-KVM-Disable-mmu-notifier-check.patch
  0012-linux-user-fix-segfault-deadlock.patch
  0013-linux-user-binfmt-support-host-bina.patch
  0014-linux-user-Ignore-broken-loop-ioctl.patch
  0015-linux-user-lock-tcg.patch
  0016-linux-user-Run-multi-threaded-code-.patch
  0017-linux-user-lock-tb-flushing-too.patch
  0018-linux-user-Fake-proc-cpuinfo.patch
  0019-linux-user-implement-FS_IOC_GETFLAG.patch
  0020-linux-user-implement-FS_IOC_SETFLAG.patch
  0021-linux-user-XXX-disable-fiemap.patch
  0022-slirp-nooutgoing.patch
  0023-vnc-password-file-and-incoming-conn.patch
  0024-linux-user-add-more-blk-ioctls.patch
  0025-linux-user-use-target_ulong.patch
  0026-block-Add-support-for-DictZip-enabl.patch
  0027-block-Add-tar-container-format.patch
  0028-Legacy-Patch-kvm-qemu-preXX-dictzip.patch
  0029-console-add-question-mark-escape-op.patch
  0030-Make-char-muxer-more-robust-wrt-sma.patch
  0031-linux-user-lseek-explicitly-cast-no.patch
  0032-virtfs-proxy-helper-Provide-_u64-f.patch
  0033-configure-Enable-PIE-for-ppc-and-pp.patch
  0034-Raise-soft-address-space-limit-to-h.patch
  0035-increase-x86_64-physical-bits-to-42.patch
  0036-vnc-provide-fake-color-map.patch
  0037-vga-Raise-VRAM-to-16-MiB-for-pc-0.1.patch
  0038-i8254-Fix-migration-from-SLE11-SP2.patch
  0039-acpi_piix4-Fix-migration-from-SLE11.patch
  0040-qtest-Increase-socket-timeout-to-ac.patch
  0041-dictzip-Fix-on-big-endian-systems.patch
  0043-xen_disk-Add-suse-specific-flush-di.patch
  0044-Split-large-discard-requests-from-b.patch
  0045-fix-xen-hvm-direct-kernel-boot.patch
  0046-xen-introduce-dummy-system-device.patch
  0047-xen-write-information-about-support.patch
  0048-xen-add-pvUSB-backend.patch
  0049-xen-move-xen_sysdev-to-xen_backend..patch
  0050-vnc-add-configurable-keyboard-delay.patch
  0051-xen-SUSE-xenlinux-unplug-for-emulat.patch
  0052-configure-add-echo_version-helper.patch
  0053-configure-support-vte-2.91.patch
  0054-scsi-esp-fix-migration.patch
  0055-hw-arm-virt-mark-the-PCIe-host-cont.patch
  0056-xen-when-removing-a-backend-don-t-r.patch
  0057-xen-drain-submit-queue-in-xen-usb-b.patch
  0058-qcow2-avoid-extra-flushes-in-qcow2.patch
  0059-qemu-bridge-helper-reduce-security-.patch
  0060-xen-use-a-common-function-for-pv-an.patch
  0061-xen_platform-unplug-also-SCSI-disks.patch
  0062-virtio-check-vring-descriptor-buffe.patch
  0063-net-vmxnet3-check-for-device_active.patch
  0064-net-vmxnet-initialise-local-tx-desc.patch
  0065-scsi-pvscsi-avoid-infinite-loop-whi.patch
  0066-ARM-KVM-Enable-in-kernel-timers-wit.patch
  0067-hw-net-Fix-a-heap-overflow-in-xlnx..patch
  0068-vmsvga-correct-bitmap-and-pixmap-si.patch
  0069-usb-xhci-fix-memory-leak-in-usb_xhc.patch
  0070-virtio-add-check-for-descriptor-s-m.patch
  0071-net-mcf-limit-buffer-descriptor-cou.patch
  0072-usb-ehci-fix-memory-leak-in-ehci_pr.patch
  0073-xhci-limit-the-number-of-link-trbs-.patch
  0074-9pfs-allocate-space-for-guest-origi.patch
  0075-9pfs-fix-memory-leak-in-v9fs_link.patch
  0076-9pfs-fix-potential-host-memory-leak.patch
  0077-9pfs-fix-memory-leak-in-v9fs_write.patch
  0078-char-serial-check-divider-value-aga.patch
  0079-net-pcnet-check-rx-tx-descriptor-ri.patch
  0080-net-eepro100-fix-memory-leak-in-dev.patch
  0081-net-rocker-set-limit-to-DMA-buffer-.patch
  0082-net-rtl8139-limit-processing-of-rin.patch
  0083-audio-intel-hda-check-stream-entry-.patch
  0084-virtio-gpu-fix-memory-leak-in-virti.patch
  0085-9pfs-fix-integer-overflow-issue-in-.patch
  0086-dma-rc4030-limit-interval-timer-rel.patch
  0087-net-imx-limit-buffer-descriptor-cou.patch
  0088-target-i386-Implement-CPUID-0xB-Ext.patch
  0089-target-i386-present-virtual-L3-cach.patch
  0090-migration-fix-inability-to-save-VM-.patch
  0091-ui-gtk-Fix-a-runtime-warning-on-vte.patch
  0092-gtk-don-t-leak-the-GtkBorder-with-V.patch
  0093-xen-fix-ioreq-handling.patch
  0094-macio-Use-blk_drain-instead-of-blk_.patch
  0095-rbd-Switch-rbd_start_aio-to-byte-ba.patch
  0096-virtio-blk-Release-s-rq-queue-at-sy.patch
  0097-virtio-blk-Remove-stale-comment-abo.patch
  0098-block-reintroduce-bdrv_flush_all.patch
  0099-qemu-use-bdrv_flush_all-for-vm_stop.patch
  0100-block-backend-remove-blkflush_all.patch
  0101-char-fix-missing-return-in-error-pa.patch
  0102-rbd-shift-byte-count-as-a-64-bit-va.patch
  0103-mirror-use-bdrv_drained_begin-bdrv_.patch
  0104-block-curl-Use-BDRV_SECTOR_SIZE.patch
  0105-block-curl-Fix-return-value-from-cu.patch
  0106-block-curl-Remember-all-sockets.patch
  0107-block-curl-Do-not-wait-for-data-bey.patch
  0108-virtio-allow-per-device-class-legac.patch
  0109-virtio-net-mark-VIRTIO_NET_F_GSO-as.patch
  0110-vhost-adapt-vhost_verify_ring_mappi.patch
  0111-ivshmem-Fix-64-bit-memory-bar-confi.patch
  0112-intel_iommu-fix-incorrect-device-in.patch
  0113-9pfs-fix-information-leak-in-xattr-.patch
  0114-9pfs-fix-memory-leak-in-v9fs_xattrc.patch
  0115-net-mcf-check-receive-buffer-size-r.patch
  0116-virtio-gpu-fix-memory-leak-in-updat.patch
  0117-virtio-gpu-fix-information-leak-in-.patch
  0118-9pfs-adjust-the-order-of-resource-c.patch
  0119-9pfs-add-cleanup-operation-in-FileO.patch
  0120-9pfs-add-cleanup-operation-for-hand.patch
  0121-9pfs-add-cleanup-operation-for-prox.patch
  0122-virtio-gpu-call-cleanup-mapping-fun.patch
  0123-string-input-visitor-Fix-uint64-par.patch
  0124-test-string-input-visitor-Add-int-t.patch
  0125-test-string-input-visitor-Add-uint6.patch
  0126-tests-Add-QOM-property-unit-tests.patch
  0127-tests-Add-scsi-disk-test.patch
  0128-usb-ehci-fix-memory-leak-in-ehci_in.patch
  0129-usbredir-free-vm_change_state_handl.patch
  0130-virtio-gpu-fix-information-leak-in-.patch
  ipxe-ath9k-Fix-buffer-overrun-for-ar9287.patch
  ipxe-ath-Fix-building-with-GCC-6.patch
  ipxe-efi-fix-garbage-bytes-in-device-path.patch
  ipxe-efi-fix-uninitialised-data-in-HII.patch
  ipxe-legacy-Fix-building-with-GCC-6.patch
  ipxe-mucurses-Fix-GCC-6-nonnull-compare-errors.patch
  ipxe-sis190-Fix-building-with-GCC-6.patch
  ipxe-skge-Fix-building-with-GCC-6.patch
  ipxe-util-v5.24-perl-errors-on-redeclare.patch
- SLE CVE, FATE, and bugzilla references not otherwise listed in
  this changelog file. The intent of this list is to indicate that
  the fix or feature continues the line of inheritance in the
  development stream of this package. The list is intended to
  satisfy searches only - refer to the SLE-12-SP2 changelog file
  for additional details.
  * fate#314468 fate#314497 fate#315125 fate#315467 fate#317015
  fate#317741 fate#317763 fate#318349 fate#319660 fate#319979
  fate#321010
  * bnc#812983 bnc#869026 bnc#869746 bnc#874413 bnc#875582 bnc#875870
  bnc#877642 bnc#877645 bnc#878541 bsc#882405 bsc#886378 bnc#893339
  bnc#893892 bnc#895369 bnc#896726 bnc#897654 bnc#905097 bnc#907805
  bnc#908380 bnc#914521 bsc#924018 bsc#929339 bsc#932267 bsc#932770
  bsc#933981 bsc#936537 bsc#937125 bsc#938344 bsc#940929 bsc#942845
  bsc#943446 bsc#944697 bsc#945404 bsc#945987 bsc#945989 bsc#946020
  bsc#947159 bnc#953518 bsc#954864 bsc#956829 bsc#957162 bsc#958491
  bsc#958917 bsc#959005 bsc#959386 bsc#960334 bsc#960708 bsc#960725
  bsc#960835 bsc#961333 bsc#961556 bsc#961691 bsc#962320 bsc#963782
  bsc#964413 bsc#970791 bsc#974141 bsc#978158 bsc#979473 bsc#982365
  bsc#989655 bsc#991466 bsc#994771 bsc#994774 bsc#996441 bsc#997858
  bsc#999212
  bsc#1001151 bsc#1002116 bsc#1005353 boo#1007263 bsc#1007769
  bsc#1008519 bsc#1009109 bsc#1013285 bsc#1013341 bsc#1013764
  bsc#1013767 bsc#1014109 bsc#1014110 bsc#1014111 bsc#1014112
  bsc#1014256 bsc#1014514 bsc#1014702 bsc#1015169 bsc#1016779
  * CVE-2014-0222 CVE-2014-0223 CVE-2014-3461 CVE-2014-3640 CVE-2014-7840
  CVE-2014-8106 CVE-2015-1779 CVE-2015-3209 CVE-2015-4037 CVE-2015-5154
  CVE-2015-5225 CVE-2015-5278 CVE-2015-5279 CVE-2015-5745 CVE-2015-6815
  CVE-2015-6855 CVE-2015-7295 CVE-2015-7512 CVE-2015-7549 CVE-2015-8345
  CVE-2015-8504 CVE-2015-8558 CVE-2015-8567 CVE-2015-8568 CVE-2015-8613
  CVE-2015-8619 CVE-2015-8743 CVE-2015-8744 CVE-2015-8745 CVE-2016-1568
  CVE-2016-1714 CVE-2016-1922 CVE-2016-1981 CVE-2016-2198 CVE-2016-3710
  CVE-2016-6490 CVE-2016-6833 CVE-2016-6888 CVE-2016-7116 CVE-2016-7155
  CVE-2016-7161 CVE-2016-9381 CVE-2016-9776 CVE-2016-9845 CVE-2016-9846
  CVE-2016-9907 CVE-2016-9908 CVE-2016-9911 CVE-2016-9912 CVE-2016-9913
  CVE-2016-9921 CVE-2016-9922
* Fri Jan 13 2017 brogers@suse.com
- Despite the previous entry about re-enabling ceph on Nov 19, 2016
  the change wasn't actually done. Do it now.
* Wed Jan 11 2017 afaerber@suse.de
- sgabios-stable-buildid.patch: Use geeko@buildhost
* Mon Nov 28 2016 afaerber@suse.de
- slof_xhci.patch: XHCI fixes (boo#977027)
* Mon Nov 28 2016 afaerber@suse.de
- Recommend x86 ROMs for emulated PCI cards on ppc, arm, others
  (bsc#1005869, michals)
* Tue Nov 22 2016 afaerber@suse.de
- Tidy SLOF patch boilerplate (michals)
* Mon Nov 21 2016 afaerber@suse.de
- Build with spice on all archs. (boo#1009438, michals)
* Sat Nov 19 2016 brogers@suse.com
- Refine the approach to producing stable builds in our ROM based
  packages. All built roms which have hostname or date calls now
  produce consistent results build to build via patch changes, so
  remove the hostname and date call workarounds. (bsc#1011213)
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.7
  * Patches added:
  0069-roms-Makefile-pass-a-packaging-time.patch
  sgabios-stable-buildid.patch
* Sat Nov 19 2016 brogers@suse.com
- Re-enable ceph (rbd) functionality in OBS builds as we've been told
  the issues which prompted us to disable it are resolved
- Address various security/stability issues
  * Fix OOB access in VMware SVGA emulation (CVE-2016-7170 bsc#998516)
  0041-vmsvga-correct-bitmap-and-pixmap-si.patch
  * Fix DOS in LSI SAS1068 emulation (CVE-2016-7157 bsc#997860)
  0042-scsi-mptconfig-fix-an-assert-expres.patch
  0043-scsi-mptconfig-fix-misuse-of-MPTSAS.patch
  * Fix DOS in Vmware pv scsi interface (CVE-2016-7156 bsc#997859)
  0044-scsi-pvscsi-limit-loop-to-fetch-SG-.patch
  * Fix DOS in USB xHCI emulation (CVE-2016-7466 bsc#1000345)
  0045-usb-xhci-fix-memory-leak-in-usb_xhc.patch
  * Fix OOB access in LSI SAS1068 emulation (CVE-2016-7423 bsc#1000397)
  0046-scsi-mptsas-use-g_new0-to-allocate-.patch
  * Fix DOS in Vmware pv scsi interface (CVE-2016-7421 bsc#999661)
  0047-scsi-pvscsi-limit-process-IO-loop-t.patch
  * Fix NULL pointer dereference in virtio processing
  (CVE-2016-7422 bsc#1000346)
  0048-virtio-add-check-for-descriptor-s-m.patch
  * Fix DOS in ColdFire Fast Ethernet Controller emulation
  (CVE-2016-7908 bsc#1002550)
  0049-net-mcf-limit-buffer-descriptor-cou.patch
  * Fix DOS in USB EHCI emulation (CVE-2016-7995 bsc#1003612)
  0050-usb-ehci-fix-memory-leak-in-ehci_pr.patch
  * Fix DOS in USB xHCI emulation (CVE-2016-8576 bsc#1003878)
  0051-xhci-limit-the-number-of-link-trbs-.patch
  * Fix DOS in virtio-9pfs (CVE-2016-8578 bsc#1003894)
  0052-9pfs-allocate-space-for-guest-origi.patch
  * Fix DOS in virtio-9pfs (CVE-2016-9105 bsc#1007494)
  0053-9pfs-fix-memory-leak-in-v9fs_link.patch
  * Fix DOS in virtio-9pfs (CVE-2016-8577 bsc#1003893)
  0054-9pfs-fix-potential-host-memory-leak.patch
  * Plug data leak in virtio-9pfs interface (CVE-2016-9103 bsc#1007454)
  0055-9pfs-fix-information-leak-in-xattr-.patch
  * Fix DOS in virtio-9pfs interface (CVE-2016-9102 bsc#1007450)
  0056-9pfs-fix-memory-leak-in-v9fs_xattrc.patch
  * Fix DOS in virtio-9pfs (CVE-2016-9106 bsc#1007495)
  0057-9pfs-fix-memory-leak-in-v9fs_write.patch
  * Fix DOS in 16550A UART emulation (CVE-2016-8669 bsc#1004707)
  0058-char-serial-check-divider-value-aga.patch
  * Fix DOS in PC-Net II emulation (CVE-2016-7909 bsc#1002557)
  0059-net-pcnet-check-rx-tx-descriptor-ri.patch
  * Fix DOS in PRO100 emulation (CVE-2016-9101 bsc#1007391)
  0060-net-eepro100-fix-memory-leak-in-dev.patch
  * Fix OOB access in Rocker switch emulation (CVE-2016-8668 bsc#1004706)
  0061-net-rocker-set-limit-to-DMA-buffer-.patch
  * Plug data leak in vmxnet3 emulation (CVE-2016-6836 bsc#994760)
  0062-net-vmxnet-initialise-local-tx-desc.patch
  * Fix DOS in RTL8139 emulation (CVE-2016-8910 bsc#1006538)
  0063-net-rtl8139-limit-processing-of-rin.patch
  * Fix DOS in Intel HDA controller emulation (CVE-2016-8909 bsc#1006536)
  0064-audio-intel-hda-check-stream-entry-.patch
  * Fix DOS in virtio-gpu (CVE-2016-7994 bsc#1003613)
  0065-virtio-gpu-fix-memory-leak-in-virti.patch
  * Fix DOS in virtio-9pfs (CVE-2016-9104 bsc#1007493)
  0066-9pfs-fix-integer-overflow-issue-in-.patch
  * Fix DOS in JAZZ RC4030 emulation (CVE-2016-8667 bsc#1004702)
  0067-dma-rc4030-limit-interval-timer-rel.patch
  * Fix DOS in i.MX NIC emulation (CVE-2016-7907 bsc#1002549)
  0068-net-imx-limit-buffer-descriptor-cou.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.7
* Fri Nov 11 2016 ohering@suse.de
- Use fixed timestamps and stable build_id in ipxe and other ROMs
  * Patches added:
  ipxe-stable-buildid.patch
* Mon Nov  7 2016 afaerber@suse.de
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.7
  * Patch updated:
  0040-linux-user-skip-0-flag-from-proc-se.patch -> 0040-linux-user-remove-all-traces-of-qem.patch
* Tue Sep 27 2016 afaerber@suse.de
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.7
  * Patches added:
  0040-linux-user-skip-0-flag-from-proc-se.patch
* Thu Sep 22 2016 afaerber@suse.de
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.7
  * Patches added:
  0039-Fix-tlb_vaddr_to_host-with-CONFIG_U.patch
* Wed Sep 21 2016 afaerber@suse.de
- Document two new options, but leave jemalloc disabled for now
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.7
  * Patches dropped:
  0034-build-link-with-libatomic-on-powerp.patch
  * Patches renamed:
  0035-xen-SUSE-xenlinux-unplug-for-emulat.patch -> 0034-xen-SUSE-xenlinux-unplug-for-emulat.patch
  0036-qemu-bridge-helper-reduce-security-.patch -> 0035-qemu-bridge-helper-reduce-security-.patch
  0037-qemu-binfmt-conf-use-qemu-ARCH-binf.patch -> 0036-qemu-binfmt-conf-use-qemu-ARCH-binf.patch
  0038-configure-Fix-detection-of-seccomp-.patch -> 0037-configure-Fix-detection-of-seccomp-.patch
  0039-linux-user-properly-test-for-infini.patch -> 0038-linux-user-properly-test-for-infini.patch
* Sat Sep 10 2016 afaerber@suse.de
- Updated to v2.7.0: See http://wiki.qemu-project.org/ChangeLog/2.7
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.7
  * Patches added:
  0039-linux-user-properly-test-for-infini.patch
* Tue Sep  6 2016 brogers@suse.com
- Use new kvm_stat package where available, else provide updated
  kvm_stat script.
* Wed Aug 31 2016 afaerber@suse.de
- Update to v2.7.0-rc5: See http://wiki.qemu-project.org/ChangeLog/2.7
* Tue Aug 23 2016 afaerber@suse.de
- Updated to v2.7.0-rc2: See http://wiki.qemu-project.org/ChangeLog/2.7
  * Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.7
  * Patches dropped:
  0002-qemu-0.9.0.cvs-binfmt.patch (script rewritten upstream)
  0009-block-vmdk-Support-creation-of-SCSI.patch (deprecated)
  0014-linux-user-Ignore-broken-loop-ioctl.patch (implemented upstream)
  0024-linux-user-add-more-blk-ioctls.patch (more implemented upstream)
  0034-qtest-Increase-socket-timeout.patch (increased further upstream)
  0036-configure-Enable-libseccomp-for-ppc.patch (enabled upstream)
  0038-block-split-large-discard-requests-.patch
  0041-xen-introduce-dummy-system-device.patch
  0042-xen-write-information-about-support.patch
  0043-xen-add-pvUSB-backend.patch
  0044-xen-move-xen_sysdev-to-xen_backend..patch
  0045-vnc-add-configurable-keyboard-delay.patch
  0046-configure-add-echo_version-helper.patch
  0047-configure-support-vte-2.91.patch
  0048-hw-arm-virt-mark-the-PCIe-host-cont.patch
  0050-scsi-esp-fix-migration.patch
  0051-xen-when-removing-a-backend-don-t-r.patch
  0052-xen-drain-submit-queue-in-xen-usb-b.patch
  0053-qcow2-avoid-extra-flushes-in-qcow2.patch
  0055-xen-use-a-common-function-for-pv-an.patch
  ipxe-ath9k-Fix-buffer-overrun-for-ar9287.patch
  ipxe-mucurses-Fix-GCC-6-nonnull-compare-errors.patch
  ipxe-sis190-Fix-building-with-GCC-6.patch
  ipxe-skge-Fix-building-with-GCC-6.patch
  ipxe-ath-Fix-building-with-GCC-6.patch
  ipxe-legacy-Fix-building-with-GCC-6.patch
  ipxe-util-v5.24-perl-errors-on-redeclare.patch
  ipxe-efi-fix-garbage-bytes-in-device-path.patch
  ipxe-efi-fix-uninitialised-data-in-HII.patch
  * Patches renamed:
  0010-linux-user-add-binfmt-wrapper-for-a.patch -> 0009-linux-user-add-binfmt-wrapper-for-a.patch
  0011-PPC-KVM-Disable-mmu-notifier-check.patch  -> 0010-PPC-KVM-Disable-mmu-notifier-check.patch
  0012-linux-user-fix-segfault-deadlock.patch    -> 0011-linux-user-fix-segfault-deadlock.patch
  0013-linux-user-binfmt-support-host-bina.patch -> 0012-linux-user-binfmt-support-host-bina.patch
  0015-linux-user-lock-tcg.patch                 -> 0013-linux-user-lock-tcg.patch
  0016-linux-user-Run-multi-threaded-code-.patch -> 0014-linux-user-Run-multi-threaded-code-.patch
  0017-linux-user-lock-tb-flushing-too.patch     -> 0015-linux-user-lock-tb-flushing-too.patch
  0018-linux-user-Fake-proc-cpuinfo.patch        -> 0016-linux-user-Fake-proc-cpuinfo.patch
  0019-linux-user-implement-FS_IOC_GETFLAG.patch -> 0017-linux-user-implement-FS_IOC_GETFLAG.patch
  0020-linux-user-implement-FS_IOC_SETFLAG.patch -> 0018-linux-user-implement-FS_IOC_SETFLAG.patch
  0021-linux-user-XXX-disable-fiemap.patch       -> 0019-linux-user-XXX-disable-fiemap.patch
  0022-slirp-nooutgoing.patch                    -> 0020-slirp-nooutgoing.patch
  0023-vnc-password-file-and-incoming-conn.patch -> 0021-vnc-password-file-and-incoming-conn.patch
  0025-linux-user-use-target_ulong.patch         -> 0022-linux-user-use-target_ulong.patch
  0026-block-Add-support-for-DictZip-enabl.patch -> 0023-block-Add-support-for-DictZip-enabl.patch
  0027-block-Add-tar-container-format.patch      -> 0024-block-Add-tar-container-format.patch
  0028-Legacy-Patch-kvm-qemu-preXX-dictzip.patch -> 0025-Legacy-Patch-kvm-qemu-preXX-dictzip.patch
  0029-console-add-question-mark-escape-op.patch -> 0026-console-add-question-mark-escape-op.patch
  0030-Make-char-muxer-more-robust-wrt-sma.patch -> 0027-Make-char-muxer-more-robust-wrt-sma.patch
  0031-linux-user-lseek-explicitly-cast-no.patch -> 0028-linux-user-lseek-explicitly-cast-no.patch
  0032-virtfs-proxy-helper-Provide-__u64-f.patch -> 0029-virtfs-proxy-helper-Provide-__u64-f.patch
  0033-configure-Enable-PIE-for-ppc-and-pp.patch -> 0030-configure-Enable-PIE-for-ppc-and-pp.patch
  0035-AIO-Reduce-number-of-threads-for-32.patch -> 0031-AIO-Reduce-number-of-threads-for-32.patch
  0037-dictzip-Fix-on-big-endian-systems.patch   -> 0032-dictzip-Fix-on-big-endian-systems.patch
  0039-xen_disk-Add-suse-specific-flush-di.patch -> 0033-xen_disk-Add-suse-specific-flush-di.patch
  0040-build-link-with-libatomic-on-powerp.patch -> 0034-build-link-with-libatomic-on-powerp.patch
  0049-xen-SUSE-xenlinux-unplug-for-emulat.patch -> 0035-xen-SUSE-xenlinux-unplug-for-emulat.patch
  0054-qemu-bridge-helper-reduce-security-.patch -> 0036-qemu-bridge-helper-reduce-security-.patch
  * Patches added:
  0002-qemu-binfmt-conf-Modify-default-pat.patch
  0037-qemu-binfmt-conf-use-qemu-ARCH-binf.patch
  * Package renamed trace-events-all file and linuxboot_dma.bin
  * Handle building and packaging roms for e1000e and vmxnet3 (Bruce)
  * Remove ipxe patches which are now enabled upstream (Bruce)
  * Enable seccomp for s390x (Mark Post):
  0038-configure-Fix-detection-of-seccomp-.patch
* Wed Aug 17 2016 brogers@suse.com
- Update to v2.6.1 a stable, bug-fix-only release (fate#316228)
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.6
  * Patches dropped (upstreamed):
  0041-net-mipsnet-check-packet-length-aga.patch
  0042-i386-kvmvapic-initialise-imm32-vari.patch
  0043-esp-check-command-buffer-length-bef.patch
  0044-esp-check-dma-length-before-reading.patch
  0045-scsi-pvscsi-check-command-descripto.patch
  0046-scsi-mptsas-infinite-loop-while-fet.patch
  0047-vga-add-sr_vbe-register-set.patch
  0048-scsi-megasas-use-appropriate-proper.patch
  0049-scsi-megasas-check-read_queue_head-.patch
  0050-scsi-megasas-null-terminate-bios-ve.patch
  0051-vmsvga-move-fifo-sanity-checks-to-v.patch
  0052-vmsvga-don-t-process-more-than-1024.patch
  0053-block-iscsi-avoid-potential-overflo.patch
  0054-scsi-esp-check-TI-buffer-index-befo.patch
  0060-scsi-megasas-initialise-local-confi.patch
  0065-scsi-esp-check-buffer-length-before.patch
  0066-scsi-esp-respect-FIFO-invariant-aft.patch
  0067-pci-assign-Move-Invalid-ROM-error-m.patch
  0068-Xen-PCI-passthrough-fix-passthrough.patch
  0069-scsi-esp-make-cmdbuf-big-enough-for.patch
  0071-virtio-error-out-if-guest-exceeds-v.patch
  * Patches renamed:
  0055-xen-introduce-dummy-system-device.patch
  - > 0041-xen-introduce-dummy-system-device.patch
  0056-xen-write-information-about-support.patch
  - > 0042-xen-write-information-about-support.patch
  0057-xen-add-pvUSB-backend.patch
  - > 0043-xen-add-pvUSB-backend.patch
  0058-xen-move-xen_sysdev-to-xen_backend..patch
  - > 0044-xen-move-xen_sysdev-to-xen_backend..patch
  0059-vnc-add-configurable-keyboard-delay.patch
  - > 0045-vnc-add-configurable-keyboard-delay.patch
  0061-configure-add-echo_version-helper.patch
  - > 0046-configure-add-echo_version-helper.patch
  0062-configure-support-vte-2.91.patch
  - > 0047-configure-support-vte-2.91.patch
  0063-hw-arm-virt-mark-the-PCIe-host-cont.patch
  - > 0048-hw-arm-virt-mark-the-PCIe-host-cont.patch
  0064-xen-SUSE-xenlinux-unplug-for-emulat.patch
  - > 0049-xen-SUSE-xenlinux-unplug-for-emulat.patch
  0070-scsi-esp-fix-migration.patch
  - > 0050-scsi-esp-fix-migration.patch
  0072-xen-when-removing-a-backend-don-t-r.patch
  - > 0051-xen-when-removing-a-backend-don-t-r.patch
  0073-xen-drain-submit-queue-in-xen-usb-b.patch
  - > 0052-xen-drain-submit-queue-in-xen-usb-b.patch
  0074-qcow2-avoid-extra-flushes-in-qcow2.patch
  - > 0053-qcow2-avoid-extra-flushes-in-qcow2.patch
  0075-qemu-bridge-helper-reduce-security-.patch
  - > 0054-qemu-bridge-helper-reduce-security-.patch
  0076-xen-use-a-common-function-for-pv-an.patch
  - > 0055-xen-use-a-common-function-for-pv-an.patch
* Wed Aug  3 2016 brogers@suse.com
- Temporarily disable ceph (rbd) functionality in OBS due to staging
  issues.
* Wed Aug  3 2016 brogers@suse.com
- use upstream solution for building xen-usb.c correctly
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.6
  * Patches dropped:
  0058-usb-Fix-conditions-that-xen-usb.c-i.patch
  * Patches added:
  0058-xen-move-xen_sysdev-to-xen_backend..patch
* Wed Aug  3 2016 brogers@suse.com
- Incorporate patch carried in Xen's qemu to get same support
  as Xen switches to use the qemu package (bsc#953339, bsc#953362,
  bsc#953518, bsc#984981)
  0064-xen-SUSE-xenlinux-unplug-for-emulat.patch
- Fix more potential OOB accesses in 53C9X emulation
  (CVE-2016-5238 bsc#982959)
  0065-scsi-esp-check-buffer-length-before.patch
  0066-scsi-esp-respect-FIFO-invariant-aft.patch
- Avoid "Invalid ROM" error message when it is not appropriate
  (bsc#982927)
  0067-pci-assign-Move-Invalid-ROM-error-m.patch
- Fix failure in Xen HVM PCI passthrough (bsc#981925, bsc#989250)
  0068-Xen-PCI-passthrough-fix-passthrough.patch
- Fix OOB access in 53C9X emulation (CVE-2016-6351 bsc#990835)
  0069-scsi-esp-make-cmdbuf-big-enough-for.patch
  0070-scsi-esp-fix-migration.patch
- Avoid potential for guest initiated OOM condition in qemu through
  virtio interface (CVE-2016-5403 bsc#991080)
  0071-virtio-error-out-if-guest-exceeds-v.patch
- Fix potential crashes in qemu from pvusb bugs (bsc#986156)
  0072-xen-when-removing-a-backend-don-t-r.patch
  0073-xen-drain-submit-queue-in-xen-usb-b.patch
- Avoid unneeded flushes in qcow2 which impact performance (bsc#991296)
  0074-qcow2-avoid-extra-flushes-in-qcow2.patch
- Finally get qemu-bridge-helper the permissions it needs for non-
  root usage. The kvm group is leveraged to control access. (boo#988279)
  0075-qemu-bridge-helper-reduce-security-.patch
- Fix pvusb not working for HVM guests (bsc#991785)
  0076-xen-use-a-common-function-for-pv-an.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.6
- Minor spec file formatting fixes
* Thu Jul 28 2016 agraf@suse.com
- Fix ARM PCIe DMA coherency bug (bsc#991034)
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.6
  * Patches added:
  0063-hw-arm-virt-mark-the-PCIe-host-cont.patch
* Tue Jul 19 2016 afaerber@suse.de
- Clean up the udev ifdeffery to cover systemd as well (boo#860275)
- Trigger udev rules also under systemd (boo#989655)
- Suppress s390x sysctl in chroot
- Ignore s390x sysctl failures (agraf)
* Thu Jul 14 2016 brogers@suse.com
- Build SLOF for SLE12 now that we have gcc fix (bsc#949000)
- Add script for loading kvm module on s390x
- Enable seccomp and iscsi support in more configurations
- Enable more support for virtio-gpu
- Fix /dev/kvm permissions problem with package install and no
  reboot (bnc#867867)
- Remove libtool dependency
- Disable more aggressive stack protector for performance reasons
- Enable vte to be used again in more configurations (bsc#988855)
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.6
  * Patches added:
  0061-configure-add-echo_version-helper.patch
  0062-configure-support-vte-2.91.patch
* Thu Jul 14 2016 agraf@suse.com
- Remove deprecated patch "work-around-SA_RESTART-race" (boo#982208)
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.6
  * Patches dropped:
  0002-XXX-work-around-SA_RESTART-race-wit.patch
  0003-qemu-0.9.0.cvs-binfmt.patch
  0004-qemu-cvs-alsa_bitfield.patch
  0005-qemu-cvs-alsa_ioctl.patch
  0006-qemu-cvs-alsa_mmap.patch
  0007-qemu-cvs-gettimeofday.patch
  0008-qemu-cvs-ioctl_debug.patch
  0009-qemu-cvs-ioctl_nodirection.patch
  0010-block-vmdk-Support-creation-of-SCSI.patch
  0011-linux-user-add-binfmt-wrapper-for-a.patch
  0012-PPC-KVM-Disable-mmu-notifier-check.patch
  0013-linux-user-fix-segfault-deadlock.patch
  0014-linux-user-binfmt-support-host-bina.patch
  0015-linux-user-Ignore-broken-loop-ioctl.patch
  0016-linux-user-lock-tcg.patch
  0017-linux-user-Run-multi-threaded-code-.patch
  0018-linux-user-lock-tb-flushing-too.patch
  0019-linux-user-Fake-proc-cpuinfo.patch
  0020-linux-user-implement-FS_IOC_GETFLAG.patch
  0021-linux-user-implement-FS_IOC_SETFLAG.patch
  0022-linux-user-XXX-disable-fiemap.patch
  0023-slirp-nooutgoing.patch
  0024-vnc-password-file-and-incoming-conn.patch
  0025-linux-user-add-more-blk-ioctls.patch
  0026-linux-user-use-target_ulong.patch
  0027-block-Add-support-for-DictZip-enabl.patch
  0028-block-Add-tar-container-format.patch
  0029-Legacy-Patch-kvm-qemu-preXX-dictzip.patch
  0030-console-add-question-mark-escape-op.patch
  0031-Make-char-muxer-more-robust-wrt-sma.patch
  0032-linux-user-lseek-explicitly-cast-no.patch
  0033-virtfs-proxy-helper-Provide-__u64-f.patch
  0034-configure-Enable-PIE-for-ppc-and-pp.patch
  0035-qtest-Increase-socket-timeout.patch
  0036-AIO-Reduce-number-of-threads-for-32.patch
  0037-configure-Enable-libseccomp-for-ppc.patch
  0038-dictzip-Fix-on-big-endian-systems.patch
  0039-block-split-large-discard-requests-.patch
  0040-xen_disk-Add-suse-specific-flush-di.patch
  0041-build-link-with-libatomic-on-powerp.patch
  0042-net-mipsnet-check-packet-length-aga.patch
  0043-i386-kvmvapic-initialise-imm32-vari.patch
  0044-esp-check-command-buffer-length-bef.patch
  0045-esp-check-dma-length-before-reading.patch
  0046-scsi-pvscsi-check-command-descripto.patch
  0047-scsi-mptsas-infinite-loop-while-fet.patch
  0048-vga-add-sr_vbe-register-set.patch
  0049-scsi-megasas-use-appropriate-proper.patch
  0050-scsi-megasas-check-read_queue_head-.patch
  0051-scsi-megasas-null-terminate-bios-ve.patch
  0052-vmsvga-move-fifo-sanity-checks-to-v.patch
  0053-vmsvga-don-t-process-more-than-1024.patch
  0054-block-iscsi-avoid-potential-overflo.patch
  0055-scsi-esp-check-TI-buffer-index-befo.patch
  0056-xen-introduce-dummy-system-device.patch
  0057-xen-write-information-about-support.patch
  0058-xen-add-pvUSB-backend.patch
  0059-usb-Fix-conditions-that-xen-usb.c-i.patch
  0060-vnc-add-configurable-keyboard-delay.patch
  0061-scsi-megasas-initialise-local-confi.patch
  * Patches added:
  0002-qemu-0.9.0.cvs-binfmt.patch
  0003-qemu-cvs-alsa_bitfield.patch
  0004-qemu-cvs-alsa_ioctl.patch
  0005-qemu-cvs-alsa_mmap.patch
  0006-qemu-cvs-gettimeofday.patch
  0007-qemu-cvs-ioctl_debug.patch
  0008-qemu-cvs-ioctl_nodirection.patch
  0009-block-vmdk-Support-creation-of-SCSI.patch
  0010-linux-user-add-binfmt-wrapper-for-a.patch
  0011-PPC-KVM-Disable-mmu-notifier-check.patch
  0012-linux-user-fix-segfault-deadlock.patch
  0013-linux-user-binfmt-support-host-bina.patch
  0014-linux-user-Ignore-broken-loop-ioctl.patch
  0015-linux-user-lock-tcg.patch
  0016-linux-user-Run-multi-threaded-code-.patch
  0017-linux-user-lock-tb-flushing-too.patch
  0018-linux-user-Fake-proc-cpuinfo.patch
  0019-linux-user-implement-FS_IOC_GETFLAG.patch
  0020-linux-user-implement-FS_IOC_SETFLAG.patch
  0021-linux-user-XXX-disable-fiemap.patch
  0022-slirp-nooutgoing.patch
  0023-vnc-password-file-and-incoming-conn.patch
  0024-linux-user-add-more-blk-ioctls.patch
  0025-linux-user-use-target_ulong.patch
  0026-block-Add-support-for-DictZip-enabl.patch
  0027-block-Add-tar-container-format.patch
  0028-Legacy-Patch-kvm-qemu-preXX-dictzip.patch
  0029-console-add-question-mark-escape-op.patch
  0030-Make-char-muxer-more-robust-wrt-sma.patch
  0031-linux-user-lseek-explicitly-cast-no.patch
  0032-virtfs-proxy-helper-Provide-__u64-f.patch
  0033-configure-Enable-PIE-for-ppc-and-pp.patch
  0034-qtest-Increase-socket-timeout.patch
  0035-AIO-Reduce-number-of-threads-for-32.patch
  0036-configure-Enable-libseccomp-for-ppc.patch
  0037-dictzip-Fix-on-big-endian-systems.patch
  0038-block-split-large-discard-requests-.patch
  0039-xen_disk-Add-suse-specific-flush-di.patch
  0040-build-link-with-libatomic-on-powerp.patch
  0041-net-mipsnet-check-packet-length-aga.patch
  0042-i386-kvmvapic-initialise-imm32-vari.patch
  0043-esp-check-command-buffer-length-bef.patch
  0044-esp-check-dma-length-before-reading.patch
  0045-scsi-pvscsi-check-command-descripto.patch
  0046-scsi-mptsas-infinite-loop-while-fet.patch
  0047-vga-add-sr_vbe-register-set.patch
  0048-scsi-megasas-use-appropriate-proper.patch
  0049-scsi-megasas-check-read_queue_head-.patch
  0050-scsi-megasas-null-terminate-bios-ve.patch
  0051-vmsvga-move-fifo-sanity-checks-to-v.patch
  0052-vmsvga-don-t-process-more-than-1024.patch
  0053-block-iscsi-avoid-potential-overflo.patch
  0054-scsi-esp-check-TI-buffer-index-befo.patch
  0055-xen-introduce-dummy-system-device.patch
  0056-xen-write-information-about-support.patch
  0057-xen-add-pvUSB-backend.patch
  0058-usb-Fix-conditions-that-xen-usb.c-i.patch
  0059-vnc-add-configurable-keyboard-delay.patch
  0060-scsi-megasas-initialise-local-confi.patch
- Enable ceph (rbd) support for aarch64
* Tue Jul 12 2016 brogers@suse.com
- Enable ceph (rbd) support
* Mon Jul  4 2016 glin@suse.com
- Fix OVMF iPXE network menu (bsc#986033, boo#987488)
  ipxe-efi-fix-garbage-bytes-in-device-path.patch
  ipxe-efi-fix-uninitialised-data-in-HII.patch
* Fri Jun 10 2016 brogers@suse.com
- Fix host information leak to guest in MegaRAID SAS 8708EM2 Host
  Bus AdapterMegaRAID SAS 8708EM2 Host Bus Adapter emulation support
  (CVE-2016-5105 bsc#982017)
  * Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.6
  0061-scsi-megasas-initialise-local-confi.patch
* Fri Jun 10 2016 brogers@suse.com
- Address various security/stability issues
  * Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.6
  * Fix OOB access in megasas emulated device (CVE-2016-5106 bsc#982018)
  0049-scsi-megasas-use-appropriate-proper.patch
  * Fix OOB access in megasas emulated device (CVE-2016-5107 bsc#982019)
  0050-scsi-megasas-check-read_queue_head-.patch
  * Fix OOB access in megasas emulated device (CVE-2016-5337 bsc#983961)
  0051-scsi-megasas-null-terminate-bios-ve.patch
  * Correct the vmvga fifo access checks (CVE-2016-4454 bsc#982222)
  0052-vmsvga-move-fifo-sanity-checks-to-v.patch
  * Fix potential DoS issue in vmvga processing (CVE-2016-4453 bsc#982223)
  0053-vmsvga-don-t-process-more-than-1024.patch
  * Fix heap buffer overflow flaw when iscsi protocol is used
  (CVE-2016-5126 bsc#982285)
  0054-block-iscsi-avoid-potential-overflo.patch
  * Fix OOB access in 53C9X emulation (CVE-2016-5338 bsc#983982)
  0055-scsi-esp-check-TI-buffer-index-befo.patch
- Add support to qemu for pv-usb under Xen (fate#316612)
  0056-xen-introduce-dummy-system-device.patch
  0057-xen-write-information-about-support.patch
  0058-xen-add-pvUSB-backend.patch
  0059-usb-Fix-conditions-that-xen-usb.c-i.patch
- Provide ability to rate limit keyboard events from the vnc server.
  This is part of the solution to an issue affecting openQA testing,
  where characters are lost, resulting in unexpected failures
  (bsc#974914)
  0060-vnc-add-configurable-keyboard-delay.patch
* Fri Jun  3 2016 brogers@suse.com
- Adjust to parallel changes in virglrenderer packages - no longer
  "BuildRequires" virglrenderer directly, just the devel package.
* Wed Jun  1 2016 brogers@suse.com
- Fix build compatibility with gcc6 wrt ipxe rom where compiler
  warnings are treated as errors.
  ipxe-ath9k-Fix-buffer-overrun-for-ar9287.patch
  ipxe-mucurses-Fix-GCC-6-nonnull-compare-errors.patch
  ipxe-sis190-Fix-building-with-GCC-6.patch
  ipxe-skge-Fix-building-with-GCC-6.patch
  ipxe-ath-Fix-building-with-GCC-6.patch
  ipxe-legacy-Fix-building-with-GCC-6.patch
- Fix ipxe build script which fails under perl v5.24
  ipxe-util-v5.24-perl-errors-on-redeclare.patch
- Specify build time disk space requirements for ppc64 and ppc64le
* Wed Jun  1 2016 agraf@suse.com
- Add sysctl script and %%post on s390x to allow kvm usage (bsc#975331)
* Thu May 26 2016 brogers@suse.com
- Address various security/stability issues
  * Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.6
  * Fix OOB access in MIPSnet emulated controller CVE-2016-4002 (bsc#975136)
  0042-net-mipsnet-check-packet-length-aga.patch
  * Fix possible host data leakage to guest from TPR access
  CVE-2016-4020 (bsc#975700)
  0043-i386-kvmvapic-initialise-imm32-vari.patch
  * Avoid OOB access in 53C9X emulation CVE-2016-4439 (bsc#980711)
  0044-esp-check-command-buffer-length-bef.patch
  * Avoid OOB access in 53C9X emulation CVE-2016-4441 (bsc#980723)
  0045-esp-check-dma-length-before-reading.patch
  * Avoid OOB access in Vmware PV SCSI emulation CVE-2016-4952 (bsc#981266)
  0046-scsi-pvscsi-check-command-descripto.patch
  * Avoid potential DoS in LSI SAS1068 emulation CVE-2016-4964 (bsc#981399)
  0047-scsi-mptsas-infinite-loop-while-fet.patch
  * Fix regression in vga behavior - introduced in v2.6.0 CVE-2016-3712 (bsc#978160)
  0048-vga-add-sr_vbe-register-set.patch
* Wed May 25 2016 brogers@suse.com
- Update to v2.6.0: See http://wiki.qemu-project.org/ChangeLog/2.6
- Enable SDL2, virglrenderer (for use with virtio-gpu), xfsctl, and
  tracing using default log backend
- Build efi pxe roms on x86_64
* Thu Apr 28 2016 afaerber@suse.de
- Check modules for conflicting release versions
- Suggest recently added block modules
* Thu Apr 28 2016 afaerber@suse.de
- Bump copyright in qemu.spec.in
- Enable libiscsi for Factory
- Enable seccomp for ppc64le as well
* Fri Apr 22 2016 afaerber@suse.de
- Update to v2.6.0-rc3: See http://wiki.qemu-project.org/ChangeLog/2.6
  * Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.6
  * Patches dropped (upstreamed):
  0041-tests-Use-correct-config-param-for-.patch
  * Patches renamed:
  0042-build-link-with-libatomic-on-powerp.patch -> 0041-build-link-with-libatomic-on-powerp.patch
* Wed Apr 20 2016 afaerber@suse.de
- Partially revert the last change's cleanup
- Indicate SUSE version
* Fri Apr  1 2016 olaf@aepfle.de
- Update to v2.6.0-rc0: See http://wiki.qemu-project.org/ChangeLog/2.6
  * Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.6
  * Accept every size in DISCARD request from a guest (bsc#964427)
  0039-block-split-large-discard-requests-.patch
  * Recognize libxl flag to disable flush in block device (bsc#879425)
  0040-xen_disk-Add-suse-specific-flush-di.patch
  * Use correct flag for crypto tests
  0041-tests-Use-correct-config-param-for-.patch
  * Fix build on powerpc:
  0042-build-link-with-libatomic-on-powerp.patch
  * Patches dropped (upstreamed):
  seabios_checkrom_typo.patch
  seabios_avoid_smbios_signature_string.patch
* Tue Feb  9 2016 afaerber@suse.de
- Disable vte for Leap, fixing build
* Wed Jan 27 2016 afaerber@suse.de
- Don't drop u-boot.e500 yet - breaks testsuite
* Thu Jan 21 2016 afaerber@suse.de
- Re-enable libcacard support
- Clean up configured features
* Thu Jan 21 2016 afaerber@suse.de
- Clean up qemu-tools libcacard Provides/Obsoletes - separate again
- Drop u-boot.e500 - being packaged as u-boot-ppce500
* Fri Jan  8 2016 afaerber@suse.de
- Update to v2.5.0: See http://wiki.qemu-project.org/ChangeLog/2.5
  * Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.5
  * Patches dropped (upstreamed):
  0039-tests-Fix-check-report-qtest-target.patch
* Tue Dec 29 2015 stefan.bruens@rwth-aachen.de
- Fix build on openSUSE 13.2
* Tue Dec  1 2015 agraf@suse.com
- Fix testsuite on 32bit systems (bsc#957379)
* Sun Nov 22 2015 afaerber@suse.de
- Update to v2.5.0-rc1: See http://wiki.qemu-project.org/ChangeLog/2.5
  * Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.5
  * Rebase libseccomp enablement:
  0037-Revert-Revert-seccomp-tests-that-al.patch -> 0037-configure-Enable-libseccomp-for-ppc.patch
  * Provide qemu-ga and qemu-ipxe for qemu-testsuite
- Clean up qemu-ksm recommendation
* Sun Nov 22 2015 afaerber@suse.de
- Fix SLE11 build by fixing systemd conditionalization (from olh)
* Thu Nov 19 2015 afaerber@suse.de
- Update to v2.5.0-rc0: See http://wiki.qemu-project.org/ChangeLog/2.5
  * Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.5
  * Patches dropped (upstream):
  0038-tcg-aarch64-Fix-tcg_out_qemu_-ld-st.patch
  0039-tests-Unique-test-path-for-string-v.patch
  gcc5-ipxe-add-missing-const-qualifiers.patch
  gcc5-ipxe-ath9k-Remove-confusing-logic-inversion-in-an-ANI-var.patch
  SLOF_ppc64le.patch
  * Patch renamed:
  0040-dictzip-Fix-on-big-endian-systems.patch -> 0038-dictzip-Fix-on-big-endian-systems.patch
  * --enable-smartcard-nss -> --enable-smartcard
  Needs an external libcacard, so drop it for now.
  * Drop --enable-vnc-tls
  * Require xz-devel for ipxe build
  * Package qemu-ga(8) man page
  * Package ivshmem-{client,server}
  * Patches added:
  0039-tests-Fix-check-report-qtest-target.patch
* Thu Nov 19 2015 tampakrap@opensuse.org
- Add systemd unit file and udev rules for qemu guest agent
  - taken from the SLE12 / Leap package, see boo#955707
* Wed Oct 28 2015 afaerber@suse.de
- Add _constraints file (based on work by kenljohnson)
* Tue Oct 20 2015 afaerber@suse.de
- Enable SLOF build for ppc64le, too, now (bsc#949000, bsc#949016)
* Mon Oct 12 2015 afaerber@suse.de
- Allow building SLOF on ppc64le (bsc#949016)
  SLOF_ppc64le.patch
- Add two checks for DictZip and tar qemu-img behavior (bsc#945778)
  * Clean up qemu-testsuite build/installation
* Thu Oct  8 2015 afaerber@suse.de
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.4
  * Fix endianness issues in DictZip block driver (bsc#937572, bsc#945778)
  0027-block-Add-support-for-DictZip-enabl.patch
  0028-block-Add-tar-container-format.patch
  0040-dictzip-Fix-on-big-endian-systems.patch
* Thu Oct  8 2015 afaerber@suse.de
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.4
  * Fix qemu-testsuite for glib2-2.46.0 by assuring uniqueness of paths
  0039-tests-Unique-test-path-for-string-v.patch
* Thu Oct  8 2015 afaerber@suse.de
- Build SLOF on ppc64 (bsc#949016, thanks to k0da)
  * Simplify x86 fw logic while at it
- No need to enable KVM for armv6hl
- Add notice about pre_checkin.sh to update_git.sh
* Thu Sep  3 2015 afaerber@suse.de
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.4
  * Fix aarch64 TCG:
  0038-tcg-aarch64-Fix-tcg_out_qemu_-ld-st.patch
* Wed Aug 12 2015 afaerber@suse.de
- Update to v2.4.0: See http://wiki.qemu-project.org/ChangeLog/2.4
* Fri Jul 24 2015 afaerber@suse.de
- Update to v2.4.0-rc2: See http://wiki.qemu-project.org/ChangeLog/2.4
  * Provide qemu-img symlink instead of passing QTEST_QEMU_IMG
* Fri Jul 17 2015 afaerber@suse.de
- Update to v2.4.0-rc1: See http://wiki.qemu-project.org/ChangeLog/2.4
  * Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.4
  * Patches dropped:
  0037-linux-user-Allocate-thunk-size-dyna.patch
  0039-s390x-Fix-stoc-direction.patch
  0040-s390x-Add-interlocked-access-facili.patch
  0041-fdc-force-the-fifo-access-to-be-in-.patch
  0042-rules.mak-Force-CFLAGS-for-all-obje.patch
  0043-qcow2-Set-MIN_L2_CACHE_SIZE-to-2.patch
  0044-hw-arm-boot-Increase-fdt-alignment.patch
  * Patches renamed:
  0038-Revert-Revert-seccomp-tests-that-al.patch -> 0037-Revert-Revert-seccomp-tests-that-al.patch
  * Package new vgabios-virtio.bin
  * target-x86_64.conf was dropped
  * Add qemu-block-dmg module sub-package
  * Set QTEST_QEMU_IMG variable for ahci-test
  * --enable-quorum and --enable-vnc-ws are no longer available
* Mon Jul 13 2015 agraf@suse.com
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.3
- Fix -kernel boot for AArch64
  * Patches added:
  0044-hw-arm-boot-Increase-fdt-alignment.patch
* Sat Jul  4 2015 crrodriguez@opensuse.org
- Use libusb-1_0-devel as buildrequires, not the old
  unused compatibility layer in libusb-devel
* Mon Jun  1 2015 agraf@suse.com
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.3
- Fix qemu2 cow caching (bsc#933132)
  * Patches added:
  0043-qcow2-Set-MIN_L2_CACHE_SIZE-to-2.patch
* Tue May 26 2015 jslaby@suse.com
- Patch queue updated from git://github.com/jirislaby/qemu.git opensuse-2.3
  * Patches added:
  0042-rules.mak-Force-CFLAGS-for-all-obje.patch
  gcc5-ipxe-add-missing-const-qualifiers.patch
  gcc5-ipxe-ath9k-Remove-confusing-logic-inversion-in-an-ANI-var.patch
* Thu May 14 2015 afaerber@suse.de
- Fix CVE-2015-3456 (boo#929339)
  0041-fdc-force-the-fifo-access-to-be-in-.patch
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.3
* Tue May 12 2015 agraf@suse.com
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.3
  * Patches added:
  0040-s390x-Add-interlocked-access-facili.patch
- Disable dependency on libnuma for s390x (not available in SLE12)
* Wed Apr 29 2015 afaerber@suse.de
- Update to v2.3.0: See http://wiki.qemu-project.org/ChangeLog/2.3
- Disable iotests for now
* Wed Apr 22 2015 afaerber@suse.de
- Update to v2.3.0-rc4: See http://wiki.qemu-project.org/ChangeLog/2.3
* Mon Apr 20 2015 afaerber@suse.de
- Update seabios_avoid_smbios_signature_string.patch with version
  applied upstream
* Wed Apr 15 2015 agraf@suse.com
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.3
- Fix s390x stoc instructions
  0039-s390x-Fix-stoc-direction.patch
* Tue Apr 14 2015 afaerber@suse.de
- Update to v2.3.0-rc3: See http://wiki.qemu-project.org/ChangeLog/2.3
  * Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.3
  * Patches dropped (applied upstream):
  0037-tcg-tcg-op.c-Fix-ld-st-of-64-bit-va.patch
  * Patches renamed:
  0038-linux-user-Allocate-thunk-size-dyna.patch -> 0037-linux-user-Allocate-thunk-size-dyna.patch
  * Revert -rc3 change to disable seccomp on non-x86 architectures
  0038-Revert-Revert-seccomp-tests-that-al.patch
* Tue Apr 14 2015 agraf@suse.com
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.3
- Fix qemu-linux-user on powerpc
  * Patches added:
  0038-linux-user-Allocate-thunk-size-dyna.patch
* Sat Apr 11 2015 afaerber@suse.de
- Split off qemu-testsuite.spec
  * Package check-report.html and check-report.xml
  * Enable quick iotests
- Dropped 0030-net-Warn-about-default-MAC-address.patch
  The warning is relevant only for bridged setups, not for the
  default SLIRP based -net user / -netdev user setup,
  and it breaks output expectations of some iotests.
  * Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.3
  * Patches renamed:
  0031-console-add-question-mark-escape-op.patch -> 0030-console-add-question-mark-escape-op.patch
  0032-Make-char-muxer-more-robust-wrt-sma.patch -> 0031-Make-char-muxer-more-robust-wrt-sma.patch
  0033-linux-user-lseek-explicitly-cast-no.patch -> 0032-linux-user-lseek-explicitly-cast-no.patch
  0034-virtfs-proxy-helper-Provide-__u64-f.patch -> 0033-virtfs-proxy-helper-Provide-__u64-f.patch
  0035-configure-Enable-PIE-for-ppc-and-pp.patch -> 0034-configure-Enable-PIE-for-ppc-and-pp.patch
  0036-qtest-Increase-socket-timeout.patch       -> 0035-qtest-Increase-socket-timeout.patch
  0037-AIO-Reduce-number-of-threads-for-32.patch -> 0036-AIO-Reduce-number-of-threads-for-32.patch
  0038-tcg-tcg-op.c-Fix-ld-st-of-64-bit-va.patch -> 0037-tcg-tcg-op.c-Fix-ld-st-of-64-bit-va.patch
- Re-enable glusterfs on Factory (updated from v3.6.1 to v3.6.2)
- Re-enable seccomp for armv7l (libseccomp submission pending)
* Fri Apr 10 2015 afaerber@suse.de
- Suppress seccomp for Factory armv7l (broken in libseccomp v2.2.0)
- Disable glusterfs explicitly on Factory, SLE12 and before 13.1
* Thu Apr  9 2015 afaerber@suse.de
- Enable glusterfs and package as qemu-block-gluster
  glusterfs post-v3.5.3 and v3.6.1/v3.6.2 have switched the
  glusterfs-api.pc version incompatibly, so only 13.1+13.2 for now
- Use macro for module Conflicts
* Thu Apr  9 2015 afaerber@suse.de
- Tidy configure options:
  * Move --enable-modules to build options
  * Sort libusb alphabetically
  * Explicitly enable attr, bluez, fdt, lzo, tpm, vhdx, vhost-net,
  vnc, xen-pci-passthrough
  * Enable bzip2
  * Enable libssh2 where possible and package as qemu-block-ssh
  * Enable numa where a compatible numactl is available
  * Enable quorum where a compatible gnutls is available
  * Enable snappy where possible
  * Prepare to enable glusterfs
  * Explicitly enable the nop tracing backend (to be revisited)
  * Explicitly disable Archipelago, as we don't have libxseg and it's
  incompatibly GPL-3.0+
  * Explicitly disable libiscsi, libnfs, netmap and rbd
  as we don't have packages
  * Drop deprecated --enable-virtio-blk-data-plane (now default)
* Wed Apr  8 2015 afaerber@suse.de
- Fix 64-bit TCG stores on 32-bit Big Endian hosts (ppc)
  0038-tcg-tcg-op.c-Fix-ld-st-of-64-bit-va.patch
  * Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.3
* Wed Apr  8 2015 afaerber@suse.de
- Update to v2.3.0-rc2: See http://wiki.qemu-project.org/ChangeLog/2.3
  * Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.3
  * Patches dropped (upstreamed):
  0038-fw_cfg-test-Fix-test-path-to-includ.patch
  0039-rcu-tests-fix-compilation-on-32-bit.patch
* Sat Mar 28 2015 brogers@suse.com
- make check was failing due to a bogus SMBIOS signature being
  encountered within SeaBIOS. Avoid having that signature stored
  randomly within the SeaBIOS image.
  * seabios_avoid_smbios_signature_string.patch
* Thu Mar 26 2015 afaerber@suse.de
- Build x86 firmware only from 13.1 on (11.4 was broken, surpassing 128 KB)
- Update to v2.3.0-rc1: See http://wiki.qemu-project.org/ChangeLog/2.3
  * Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.3
  * Patches dropped (upstreamed):
  0038-linux-user-Fix-emulation-of-splice-.patch
  0039-ide-fix-cmd_write_pio-when-nsectors.patch
  0040-ide-fix-cmd_read_pio-when-nsectors-.patch
  0041-ahci-Fix-sglist-offset-manipulation.patch
  0042-ahci-test-improve-rw-buffer-pattern.patch
  0045-linux-user-fix-broken-cpu_copy.patch
  * Patches renamed:
  0043-fw_cfg-test-Fix-test-path-to-includ.patch -> 0038-fw_cfg-test-Fix-test-path-to-includ.patch
  0044-rcu-tests-fix-compilation-on-32-bit.patch -> 0039-rcu-tests-fix-compilation-on-32-bit.patch
* Mon Mar 23 2015 afaerber@suse.de
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.3
  * Patches added:
  0045-linux-user-fix-broken-cpu_copy.patch
* Sat Mar 21 2015 afaerber@suse.de
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.3
  * Make test path for fw_cfg-test unique (including architecture)
  0043-fw_cfg-test-Fix-test-path-to-includ.patch
  * Fix rcu tests build on ppc (undefined reference to `__sync_fetch_and_add_8')
  0044-rcu-tests-fix-compilation-on-32-bit.patch
- Fix typo in SeaBIOS size check
  seabios_checkrom_typo.patch
* Fri Mar 20 2015 afaerber@suse.de
- Update to v2.3.0-rc0: See http://wiki.qemu-project.org/ChangeLog/2.3
  * Updated update_git.sh accordingly
  * Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.3
  * seabios_128kb.patch: Added patch to squeeze SeaBIOS into 128 KB
  with our gcc 4.8.3 (brogers@suse.com)
- Renamed 0030-Legacy-Patch-kvm-qemu-preXX-report-.patch to
  0030-net-Warn-about-default-MAC-address.patch:
  Suppress warning for accel=qtest, to sanitize make check results.
- Added patches to fix ahci-test:
  0039-ide-fix-cmd_write_pio-when-nsectors.patch
  0040-ide-fix-cmd_read_pio-when-nsectors-.patch
  0041-ahci-Fix-sglist-offset-manipulation.patch
  0042-ahci-test-improve-rw-buffer-pattern.patch
* Sun Mar  1 2015 afaerber@suse.de
- Update company name in spec file templates
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.2
  * Patches added:
  0038-linux-user-Fix-emulation-of-splice-.patch
* Sun Feb 22 2015 mpluskal@suse.com
- Add user kvm when installing guest-agent.
- Use macro to update udev_rules when available
* Wed Feb 11 2015 afaerber@suse.de
- Fix packaging of e500 U-Boot
- Don't rely on wildcard with explicit excludes
* Wed Jan 14 2015 agraf@suse.com
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.2
  * Patches added:
  0037-AIO-Reduce-number-of-threads-for-32.patch
* Mon Jan 12 2015 afaerber@suse.de
- Update to v2.2.0: See http://wiki.qemu-project.org/ChangeLog/2.2
  * Updated DictZip and Tar block backends accordingly
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.2
  * Patches dropped:
  0015-target-arm-linux-user-no-tb_flush-o.patch (tb_flush() not called)
  0037-tests-Don-t-run-qom-test-twice.patch (superseded)
  0039-linux-user-Cast-validity-checks-on-.patch (helper function introduced)
  0040-linux-user-Convert-blkpg-to-use-a-s.patch (upstreamed)
  * Patched renumbered:
  0016-linux-user-Ignore-broken-loop-ioctl.patch -> 0015-linux-user-Ignore-broken-loop-ioctl.patch
  0017-linux-user-lock-tcg.patch                 -> 0016-linux-user-lock-tcg.patch
  0018-linux-user-Run-multi-threaded-code-.patch -> 0017-linux-user-Run-multi-threaded-code-.patch
  0019-linux-user-lock-tb-flushing-too.patch     -> 0018-linux-user-lock-tb-flushing-too.patch
  0020-linux-user-Fake-proc-cpuinfo.patch        -> 0019-linux-user-Fake-proc-cpuinfo.patch
  0021-linux-user-implement-FS_IOC_GETFLAG.patch -> 0020-linux-user-implement-FS_IOC_GETFLAG.patch
  0022-linux-user-implement-FS_IOC_SETFLAG.patch -> 0021-linux-user-implement-FS_IOC_SETFLAG.patch
  0023-linux-user-XXX-disable-fiemap.patch       -> 0022-linux-user-XXX-disable-fiemap.patch
  0024-slirp-nooutgoing.patch                    -> 0023-slirp-nooutgoing.patch
  0025-vnc-password-file-and-incoming-conn.patch -> 0024-vnc-password-file-and-incoming-conn.patch
  0026-linux-user-add-more-blk-ioctls.patch      -> 0025-linux-user-add-more-blk-ioctls.patch
  0027-linux-user-use-target_ulong.patch         -> 0026-linux-user-use-target_ulong.patch
  0028-block-Add-support-for-DictZip-enabl.patch -> 0027-block-Add-support-for-DictZip-enabl.patch
  0029-block-Add-tar-container-format.patch      -> 0028-block-Add-tar-container-format.patch
  0030-Legacy-Patch-kvm-qemu-preXX-dictzip.patch -> 0029-Legacy-Patch-kvm-qemu-preXX-dictzip.patch
  0031-Legacy-Patch-kvm-qemu-preXX-report-.patch -> 0030-Legacy-Patch-kvm-qemu-preXX-report-.patch
  0032-console-add-question-mark-escape-op.patch -> 0031-console-add-question-mark-escape-op.patch
  0033-Make-char-muxer-more-robust-wrt-sma.patch -> 0032-Make-char-muxer-more-robust-wrt-sma.patch
  0034-linux-user-lseek-explicitly-cast-no.patch -> 0033-linux-user-lseek-explicitly-cast-no.patch
  0035-virtfs-proxy-helper-Provide-__u64-f.patch -> 0034-virtfs-proxy-helper-Provide-__u64-f.patch
  0036-configure-Enable-PIE-for-ppc-and-pp.patch -> 0035-configure-Enable-PIE-for-ppc-and-pp.patch
  0038-qtest-Increase-socket-timeout.patch       -> 0036-qtest-Increase-socket-timeout.patch
* Mon Sep  1 2014 afaerber@suse.de
- Enable RDMA support for SLE12 and later
* Thu Aug 28 2014 afaerber@suse.de
- Copy the freshly built s390-ccw.img over to the location
  from which it is installed (bnc#893535)
* Fri Aug 22 2014 agraf@suse.com
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.1
  * Patches added:
  0039-linux-user-Cast-validity-checks-on-.patch
  0040-linux-user-Convert-blkpg-to-use-a-s.patch
* Thu Aug 21 2014 afaerber@suse.de
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.1
- Revert use of %%{version} in Source line as it breaks update_git.sh
* Thu Aug  7 2014 coolo@suse.com
- build libcaca from a different spec file to avoid cycles with
  spice
* Wed Aug  6 2014 coolo@suse.com
- update to v2.1.0: http://wiki.qemu-project.org/ChangeLog/2.1
- Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.1
  * Patches dropped:
  0039-fix-full-frame-updates-for-VNC-clie.patch
  0040-vnc-update-fix.patch
* Thu Jul 31 2014 dimstar@opensuse.org
- Rename rpmlintrc to %%{name}-rpmlintrc.
  Follow the packaging guidelines.
* Fri Jul 25 2014 afaerber@suse.de
- Backported VNC fixes for openQA (bnc#888142):
  * Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.1
  * Patches added:
  0039-fix-full-frame-updates-for-VNC-clie.patch
  0040-vnc-update-fix.patch
* Wed Jul 23 2014 afaerber@suse.de
- Update to v2.1.0-rc3: http://wiki.qemu-project.org/ChangeLog/2.1
  * Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.1
  * Patches dropped:
  0039-module-Simplify-module_load.patch
  0040-module-Don-t-complain-when-a-module.patch
  * Updated update_git.sh script accordingly
* Wed Jul 16 2014 afaerber@suse.de
- Update to v2.1.0-rc2: http://wiki.qemu-project.org/ChangeLog/2.1
  * Patch queue updated from git://github.com/openSUSE/qemu.git opensuse-2.1
  * Patches dropped:
  0041-tests-Fix-unterminated-string-outpu.patch
  0042-libqos-Fix-PC-PCI-endianness-glitch.patch
  0043-qtest-fix-vhost-user-test-compilati.patch
  * Updated update_git.sh accordingly
  * Extended update_git.sh to add generated patch names to change log
* Wed Jul  9 2014 afaerber@suse.de
- Update to v2.1.0-rc0: http://wiki.qemu-project.org/ChangeLog/2.1
  * xen_disk discard support now upstream
  0037-xen_disk-add-discard-support.patch dropped
  * PowerPC ELF v2 support now upstream
  0045-tcg-ppc64-Support-the-ELFv2-ABI.patch dropped
  * Patches renamed during rebase:
  0015-linux-user-arm-no-tb_flush-on-reset.patch -> 0015-target-arm-linux-user-no-tb_flush-o.patch
  * Patches reordered due to rebase:
  0038-tests-Don-t-run-qom-test-twice.patch -> 0037-tests-Don-t-run-qom-test-twice.patch
  0041-qtest-Increase-socket-timeout.patch -> 0038-qtest-Increase-socket-timeout.patch
  0071-module-Simplify-module_load.patch -> 0039-module-Simplify-module_load.patch
  0072-module-Don-t-complain-when-a-module.patch -> 0040-module-Don-t-complain-when-a-module.patch
  * Patches automatically dropped on rebase:
  0039-qtest-Assure-that-init_socket-s-lis.patch
  0040-qtest-Add-error-reporting-to-socket.patch
  0042-qtest-Be-paranoid-about-accept-addr.patch
  0043-arm-translate.c-Fix-smlald-Instruct.patch
  0044-target-arm-A64-fix-unallocated-test.patch
  0046-vmstate-add-VMS_MUST_EXIST.patch
  0047-vmstate-add-VMSTATE_VALIDATE.patch
  0048-virtio-net-fix-buffer-overflow-on-i.patch
  0049-virtio-net-out-of-bounds-buffer-wri.patch
  0050-virtio-out-of-bounds-buffer-write-o.patch
  0051-ahci-fix-buffer-overrun-on-invalid-.patch
  0052-hpet-fix-buffer-overrun-on-invalid-.patch
  0053-hw-pci-pcie_aer.c-fix-buffer-overru.patch
  0054-pl022-fix-buffer-overun-on-invalid-.patch
  0055-vmstate-fix-buffer-overflow-in-targ.patch
  0056-virtio-avoid-buffer-overrun-on-inco.patch
  0057-virtio-validate-num_sg-when-mapping.patch
  0058-pxa2xx-avoid-buffer-overrun-on-inco.patch
  0059-ssd0323-fix-buffer-overun-on-invali.patch
  0060-tsc210x-fix-buffer-overrun-on-inval.patch
  0061-zaurus-fix-buffer-overrun-on-invali.patch
  0062-virtio-scsi-fix-buffer-overrun-on-i.patch
  0063-vmstate-s-VMSTATE_INT32_LE-VMSTATE_.patch
  0064-usb-sanity-check-setup_index-setup_.patch
  0065-savevm-Ignore-minimum_version_id_ol.patch
  0066-ssi-sd-fix-buffer-overrun-on-invali.patch
  0067-openpic-avoid-buffer-overrun-on-inc.patch
  0068-virtio-net-out-of-bounds-buffer-wri.patch
  0069-virtio-validate-config_len-on-load.patch
  0070-virtio-allow-mapping-up-to-max-queu.patch
  * iPXE fixes now included
  ipxe-build-Work-around-bug-in-gcc-4.8.patch dropped
  ipxe-build-Avoid-strict-aliasing-warning.patch dropped
  ipxe-zbin-Fix-size-used-for-memset-in-al.patch dropped
  * SeaVGABIOS fix now included
  vgabios-Make-sure-stdvga_list_modes-doesn-t-overrun-.patch dropped
  * Modified update_git.sh accordingly
- Update to v2.1.0-rc1: http://wiki.qemu-project.org/ChangeLog/2.1
  * 0041-tests-Fix-unterminated-string-outpu.patch: Test fix
  * 0042-libqos-Fix-PC-PCI-endianness-glitch.patch: Test fix for ppc
  * 0043-qtest-fix-vhost-user-test-compilati.patch: Test fix for SP3
  * Modified update_git.sh accordingly
* Mon Jun 23 2014 afaerber@suse.de
- Move block-curl.so to new qemu-block-curl package, avoiding a
  dependency of qemu-tools on qemu for Xen (bnc#882008)
  * Avoid the reported error message about missing modules
  0071-module-Simplify-module_load.patch
  0072-module-Don-t-complain-when-a-module.patch
- Improve update_git.sh performance (backport from SLE12)
* Mon Jun  2 2014 brogers@suse.com
- only build the x86 firmware on x86, ensuring consistent binaries
- remove un-needed pmtools BuildRequires
- Fix vga related problem doing direct kernel boot for SLES 10 and
  SLES 11 guests (bnc#880625)
  vgabios-Make-sure-stdvga_list_modes-doesn-t-overrun-.patch
- Probe vhost-net module on SLE11 (copied from SP3 package)
* Sat May 24 2014 afaerber@suse.de
- Fix CVE-2013-4148 (bnc#864812)
  0048-virtio-net-fix-buffer-overflow-on-i.patch
- Fix CVE-2013-4149 (bnc#864649)
  0068-virtio-net-out-of-bounds-buffer-wri.patch
- Fix CVE-2013-4150 (bnc#864650)
  0049-virtio-net-out-of-bounds-buffer-wri.patch
- Fix CVE-2013-4151 (bnc#864653)
  0050-virtio-out-of-bounds-buffer-write-o.patch
- Fix CVE-2013-4526 (bnc#864671)
  0051-ahci-fix-buffer-overrun-on-invalid-.patch
- Fix CVE-2013-4527 (bnc#864673)
  0052-hpet-fix-buffer-overrun-on-invalid-.patch
- Fix CVE-2013-4529 (bnc#864678)
  0053-hw-pci-pcie_aer.c-fix-buffer-overru.patch
- Fix CVE-2013-4530 (bnc#864682)
  0054-pl022-fix-buffer-overun-on-invalid-.patch
- Fix CVE-2013-4531 (bnc#864796)
  0055-vmstate-fix-buffer-overflow-in-targ.patch
- Fix CVE-2013-4533 (bnc#864655)
  0058-pxa2xx-avoid-buffer-overrun-on-inco.patch
- Fix CVE-2013-4534 (bnc#864811)
  0067-openpic-avoid-buffer-overrun-on-inc.patch
- Fix CVE-2013-4535 / CVE-2013-4536 (bnc#864665)
  0057-virtio-validate-num_sg-when-mapping.patch
- Fix CVE-2013-4537 (bnc#864391)
  0066-ssi-sd-fix-buffer-overrun-on-invali.patch
- Fix CVE-2013-4538 (bnc#864769)
  0059-ssd0323-fix-buffer-overun-on-invali.patch
- Fix CVE-2013-4539 (bnc#864805)
  0060-tsc210x-fix-buffer-overrun-on-inval.patch
- Fix CVE-2013-4540 (bnc#864801)
  0061-zaurus-fix-buffer-overrun-on-invali.patch
- Fix CVE-2013-4541 (bnc#864802)
  0064-usb-sanity-check-setup_index-setup_.patch
- Fix CVE-2013-4542 (bnc#864804)
  0062-virtio-scsi-fix-buffer-overrun-on-i.patch
- Fix CVE-2013-6399 (bnc#864814)
  0067-virtio-avoid-buffer-overrun-on-inco.patch
- Fix CVE-2014-0182 (bnc#874788)
  0069-virtio-validate-config_len-on-load.patch
- Backported patch from stable branch (brogers):
  0070-virtio-allow-mapping-up-to-max-queu.patch
* Wed May 14 2014 afaerber@suse.de
- Fix conflict with libcacard 0.1.2 from project security
  * Adopt description and group for libcacard-devel from there
- Drop libcacard.{a,la} from libcacard-devel
  If needed, libcacard.a should go into a libcacard-devel-static.
- Change qemu-extra to explicitly list qemu-system-* binaries
* Tue May 13 2014 afaerber@suse.de
- ABI fixes for ppc64le TCG. Addresses acpi-test failures.
  0045-tcg-ppc64-Support-the-ELFv2-ABI.patch
* Sat May 10 2014 afaerber@suse.de
- Split 60-kvm.rules into 60-kvm.rules and 60-kvm.x86.rules
- Install udev rules on s390x, ppc and arm, too
  * Clean up KVM vs. qemu-kvm conditionals
- Package qemu-system-ppcemb in qemu-ppc and qemu-system-aarch64
  in qemu-arm
* Fri Apr 18 2014 afaerber@suse.de
- Fix aarch64 emulation issues:
  0043-arm-translate.c-Fix-smlald-Instruct.patch
  0044-target-arm-A64-fix-unallocated-test.patch
* Thu Apr 17 2014 afaerber@suse.de
- Attempt to solve or debug occasional qtest fd assertions
  0039-qtest-Assure-that-init_socket-s-lis.patch
  0040-qtest-Add-error-reporting-to-socket.patch
  0041-qtest-Increase-socket-timeout.patch
  0042-qtest-Be-paranoid-about-accept-addr.patch
* Thu Apr 17 2014 afaerber@suse.de
- Update to v2.0.0: cf. http://wiki.qemu-project.org/ChangeLog/2.0
  * Update update_git.sh script accordingly
* Mon Apr 14 2014 afaerber@suse.de
- Update to v2.0.0-rc3: cf. http://wiki.qemu-project.org/ChangeLog/2.0
  * Update update_git.sh script accordingly
  * Addresses CVE-2014-0150 (bnc#873235), CVE-2013-4544 (bnc#873613)
* Fri Apr 11 2014 afaerber@suse.de
- Update to v2.0.0-rc2: cf. http://wiki.qemu-project.org/ChangeLog/2.0
  * Update update_git.sh script accordingly
  * Resolves Gtk mouse issues (bnc#849587)
  * Fixes endianness-test failures for prep machine on Big Endian host
* Fri Apr  4 2014 afaerber@suse.de
- Fix qemu-kvm script for s390x (bnc#869021) (bfrogers)
- Fix ksm.service file permissions to not be executable w/o shebang
- Re-enable Gtk build on 11.4 (fixes unpackaged qemu.mo files error)
- Fix SLE11 SP3 build by making sub-package "noarch" conditional
  * Do not exclude qemu-ifup from qemu package when not installing
- Speed up %%check by not running qom-test twice for some targets
  0038-tests-Don-t-run-qom-test-twice.patch
* Thu Apr  3 2014 afaerber@suse.de
- Updated to v2.0.0-rc1: cf. http://wiki.qemu-project.org/ChangeLog/2.0
  * Update update_git.sh script accordingly
  * Resolves breakage of live migration (bnc#871302)
  * Addresses CVE-2014-0142, CVE-2014-0143, CVE-2014-0144,
  CVE-2014-0145, CVE-2014-0146, CVE-2014-0147 (bnc#870439)
- Drop 0037-roms-Build-vgabios.bin.patch (unneeded)
* Tue Apr  1 2014 afaerber@suse.de
- Enhance update_git.sh script
  * Fix duplicate .patch for short filenames
  * Read SeaBIOS version from QEMU tarball
* Mon Mar 31 2014 afaerber@suse.de
- Re-create qemu-2.0.0-rc0.tar.bz2 to include SeaBIOS .version
* Fri Mar 28 2014 afaerber@suse.de
- Improve update_git.sh script to use a user-agnostic git symlink
* Wed Mar 26 2014 brogers@suse.com
- Tidy up qemu.spec a bit: white space, line length, use macros
  as appropriate
- Improve update_git.sh to not change files in OBS if the file
  remains unchanged
* Wed Mar 26 2014 afaerber@suse.de
- Fix library directory
- Set default file attributes for qemu-ksm
- Enable Smartcard emulation
  * Reorder package definitions to avoid %%version troubles
* Wed Mar 19 2014 afaerber@suse.de
- Update to v2.0.0-rc0: Cf. http://wiki.qemu-project.org/ChangeLog/2.0
  * Adapt update_git.sh script accordingly
  * Generate package version automatically from VERSION file
  * Switch from LGPL VGABIOS to SeaVGABIOS
  * Enable modules
  * Explicitly enable virtio-blk-data-plane (no functional change)
  * Package new firmware blobs
* Fri Mar  7 2014 speilicke@suse.com
- Add qemu-ksm subpackage providing a systemd service file for
  enabling / disabling Kernel Samepage Merging. It is independent of
  qemu (but recommended by it).
* Fri Feb 28 2014 afaerber@suse.de
- Move target-x86_64.conf to qemu-x86 sub-package
* Thu Feb 20 2014 afaerber@suse.de
- Split out KVM architectures to per-architecture subpackages
- Suggest qemu-lang package
* Tue Feb 18 2014 brogers@suse.com
- enable usb passthrough via --enable-libusb config option
* Sat Feb 15 2014 brogers@suse.com
- Fix improperly formatted wrapper script (bnc#864094)
* Fri Feb 14 2014 afaerber@suse.de
- Enable make check
* Thu Feb 13 2014 brogers@suse.com
- Use Provides/Obsoletes as means to transition kvm users to the
  qemu-kvm package
* Wed Feb 12 2014 brogers@suse.com
- Migrate kvm package contents to a new qemu-kvm sub-package, and
  move the kvm related udev rules to the qemu package. (The kvm
  package is being retired).
  Add qemu-ifup
  Add qemu-kvm.1.gz
  Add kvm_stat
  Add 60-kvm.rules
  Add 80-kvm.rules
* Mon Feb  3 2014 afaerber@suse.de
- Implement discard support for xen_disk (olh) (fate#316071)
  0042-xen_disk-add-discard-support.patch.patch
* Fri Jan 24 2014 afaerber@suse.de
- Simplified VMDK SCSI support handling (backport from 13.1/v1.6.2)
  * Increases the number of heads for SCSI from 16 to 255
* Fri Jan 17 2014 afaerber@suse.de
- Bump copyright in qemu.spec.in
- Use vte-devel instead of vte2-devel and force using Gtk3,
  reported by Takashi and Dirk (bnc#840607, bnc#849587).
* Mon Jan 13 2014 coolo@suse.com
- xen dropped support for i586, so don't buildrequire it
* Thu Jan  2 2014 afaerber@suse.de
- Update version number of SeaBIOS, spotted by BdMdesigN
* Fri Dec  6 2013 afaerber@suse.de
- Update to v1.7.0: See http://wiki.qemu-project.org/ChangeLog/1.7
  * Adapted update_git.sh accordingly
  * Dropped --enable-mixemu (no longer available)
  * Added a work around for roms/Makefile issue
  * patch 0038-gdbstub-Fix-gdb_register_coprocesso.patch dropped
* Wed Oct  9 2013 dmueller@suse.com
- use --enable-pie to fix arm build
* Tue Oct  8 2013 afaerber@suse.de
- Undo workaround for sasl - fixed in Factory
* Wed Oct  2 2013 afaerber@suse.de
- Work around Factory sasl breakage by allowing build w/o vnc-sasl
- Move build_and_copy_vgabios_bin.patch to Git patch queue
  * Clearly mark in-.spec patch queue as for roms/ipxe/ only
- Synchronized qemu.spec.in from qemu.spec:
  * Added ipxe-build-Avoid-strict-aliasing-warning.patch
  * Modified SeaBIOS summary
  * Added Conflicts for firmware subpackages
  * Dropped %%dir
- Add patch enabling PIE support for ppc/ppc64 (k0da)
* Fri Sep 13 2013 brogers@suse.com
- Build x86 firmware from tarball source. The firmware components
  produced are equivalent to the upstream provided binary blobs
  currently being used. They are delivered in noarch subpackages as
  follows (addresses bnc#830002):
  * qemu-seabios v1.7.2.2
  * qemu-vgabios v0.6c
  * qemu-sgabios v8 (svn rev 8)
  * qemu-ipxe v1.0.0
- Add following patches to enable x86 firmware to build:
  ipxe-build-Work-around-bug-in-gcc-4.8.patch
  ipxe-zbin-Fix-size-used-for-memset-in-al.patch
  build_and_copy_vgabios_bin.patch
  ipxe-build-Avoid-strict-aliasing-warning.patch
* Mon Aug 26 2013 rob.verduijn@gmail.com
- Enabled usb-redirection
  * Fixes starting VMs in gnome-boxes (bnc#839703)
* Sat Aug 17 2013 afaerber@suse.de
- Enable Xen support to allow testing our QEMU in place of Xen's
  This prepares for consolidating QEMU packaging further.
* Sat Aug 17 2013 afaerber@suse.de
- Fix gdbstub for arm/m68k/ppc guests (from pending upstream pull)
* Fri Aug 16 2013 afaerber@suse.de
- Update to v1.6.0: cf. http://wiki.qemu.org/ChangeLog/1.6
  * Adapt update_git.sh script accordingly
* Tue Aug 13 2013 afaerber@suse.de
- Update to v1.6.0-rc3: cf. http://wiki.qemu.org/ChangeLog/1.6
  * Adapt update_git.sh script accordingly
- Changed back versioning scheme to 1.6.0rcX (saschpe, brogers).
* Fri Aug  9 2013 afaerber@suse.de
- Update to v1.6.0-rc2: cf. http://wiki.qemu.org/ChangeLog/1.6
  * Adapt update_git.sh script accordingly
- Changed version from 1.6.0rcX to 1.5.9X (reported by lnussel)
* Fri Aug  2 2013 afaerber@suse.de
- Update to v1.6.0-rc1: cf. http://wiki.qemu.org/ChangeLog/1.6
  * Adapt update_git.sh script accordingly
* Thu Aug  1 2013 afaerber@suse.de
- Use libfdt build dependency not just for ppc.
  QEMU 1.6 needs it for ppc, arm and microblaze guests and uses a
  bundled copy if one of those targets is built and the system does
  not have a compatible libfdt.
* Tue Jul 30 2013 afaerber@suse.de
- Update to v1.6.0-rc0: cf. http://wiki.qemu.org/ChangeLog/1.6
  * Fixes build against recent dtc (SLE_11_SP2 ppc64 w/ Base:System)
  * Includes Gtk patches for bnc#830459
  * Adapt update_git.sh script accordingly
* Mon Jul 22 2013 brogers@suse.com
- Fix segfault that occurs from using -vga qxl (bnc#830459)
* Tue Jun 18 2013 afaerber@suse.de
- Clarify through spec file comments that the patch queue is
  auto-generated through update_git.sh.
  Please do not submit SRs that bypass our patch handling.
* Sat May 25 2013 afaerber@suse.de
- Really enable pulseaudio and oss support (bnc#737070)
- Amend licenses list with GPL-2.0 (adopted from kvm package)
- Be explicit in enabling features our packaging relies on
- Disable stripping at QEMU level
- Use %%optflags rather than $QEMU_OPT_FLAGS
* Fri May 24 2013 afaerber@suse.de
- Move /emul/ia32-linux to qemu-linux-user
* Fri May 24 2013 afaerber@suse.de
- Drop dependency on bison
  * Reorder BuildRequires alphabetically
- Fix summary of qemu-lang sub-package
- Drop post-build-checks suppression (qemu-linux-user only)
* Fri May 24 2013 afaerber@suse.de
- Enable brlapi as done for kvm package
* Fri May 24 2013 afaerber@suse.de
- Use %%{?_smp_mflags} rather than %%{?jobs:-j%%jobs}
* Thu May 23 2013 afaerber@suse.de
- Extend virtfs typedef workaround to ppc64 for SLE_11_SP2
* Tue May 21 2013 afaerber@suse.de
- Split off qemu-linux-user package into separate .spec file.
  * Simplified %%build and %%install sections
  * Dropped *-static dependencies
* Mon May 20 2013 afaerber@suse.de
- Update to v1.5.0 final: Cf. http://wiki.qemu.org/ChangeLog/1.5
  * Adapt update_git.sh accordingly
  * Add BuildRequires for gtk2/3 and vte2 to enable new frontend
  * Place translations into new qemu-lang package
  * Really enable SPICE support from 11.4 in configure, too
* Fri May 17 2013 afaerber@suse.de
- Add BuildRequires for pixman, to avoid using internal version
  that fails to build on ppc
* Thu May 16 2013 afaerber@suse.de
- Update to v1.5.0-rc2: Cf. http://wiki.qemu.org/ChangeLog/1.5
  * Adapt update_git.sh accordingly
- Add patch by brogers to fix SLE 11 SP2 build (from kvm package)
* Tue May 14 2013 afaerber@suse.de
- Update to v1.5.0-rc1: Cf. http://wiki.qemu.org/ChangeLog/1.5
  * Adapt update_git.sh accordingly
  * TODO file no longer exists, don't try to package it
  * Package qemu-mips64{,el}
* Tue May 14 2013 ohering@suse.de
- Enable spice support starting from 11.4
* Tue May  7 2013 afaerber@suse.de
- Update to v1.5.0-rc0: Cf. http://wiki.qemu.org/ChangeLog/1.5
  * Adapt update_git.sh script
  * Audio cards are now all enabled by default
* Wed Apr 24 2013 afaerber@suse.de
- Enable pa and oss audio drivers, as in kvm package (bnc#737070)
- Enable hda audio card, as done in kvm package (bnc#801794)
* Mon Apr 22 2013 dimstar@opensuse.org
- Add explicit alsa-devel BuildRequires: this used to (wrongly) be
  pulled in by libSDL-devel.
* Sun Feb 17 2013 afaerber@suse.de
- Update licenses from new format_spec_file
* Sat Feb 16 2013 afaerber@suse.de
- Update to v1.4.0: See http://wiki.qemu.org/ChangeLog/1.4
  * Adapt update_git.sh accordingly
* Sun Feb 10 2013 afaerber@suse.de
- Update to v1.4.0-rc1: See http://wiki.qemu.org/ChangeLog/1.4
  * Adapt update_git.sh accordingly
- Update update_git.sh to reference github.com/openSUSE/qemu.git
* Thu Dec 13 2012 agraf@suse.com
- linux-user: fix lseek with SEEK_END
* Thu Dec 13 2012 agraf@suse.com
- Cherry pick patches from qemu-kvm into the qemu package:
  - 0033-Add-support-for-DictZip-enabled-gzi.patch
  - 0034-Add-tar-container-format.patch.patch
  - 0035-Legacy-Patch-kvm-qemu-preXX-dictzip.patch
  - 0036-Legacy-Patch-kvm-qemu-preXX-report-.patch
  - 0037-console-add-question-mark-escape-op.patch
  - 0038-Make-char-muxer-more-robust-wrt-sma.patch
* Wed Dec  5 2012 afaerber@suse.de
- Revert memory size constraint, the workaround patch got applied
* Wed Dec  5 2012 afaerber@suse.de
- Update to v1.3.0: see http://wiki.qemu.org/ChangeLog/1.3
  * Update update_git.sh script accordingly
* Wed Nov 28 2012 afaerber@suse.de
- Replace -fno-gcse patch with v3
* Tue Nov 27 2012 afaerber@suse.de
- Add patch to workaround GCC PR55489 via -fno-gcse,
  another step to avoid builds failing with out-of-memory
* Tue Nov 27 2012 afaerber@suse.de
- Add _constraints to request adequate memory to succeed building
* Tue Nov 27 2012 afaerber@suse.de
- Update to v1.3.0-rc1: see http://wiki.qemu.org/ChangeLog/1.3
  * Update update_git.sh accordingly
* Tue Nov 20 2012 afaerber@suse.de
- Update to v1.3.0-rc0: see http://wiki.qemu.org/ChangeLog/1.3
  * Update update_git.sh accordingly
  * Disable SPICE for 11.4 since it can no longer be built lacking
  pyparsing
* Wed Oct 10 2012 agraf@suse.com
- use ulong for syscall parameters. fixes lseek with big files
* Wed Oct 10 2012 agraf@suse.com
- Add more block ioctl handlers for kiwi
* Tue Oct  9 2012 afaerber@suse.de
- Sync qemu.spec.in file, please don't forget!
* Fri Sep 14 2012 schwab@linux-m68k.org
- Don't strip binaries
* Thu Sep 13 2012 idonmez@suse.com
- Add explicit dependency on makeinfo for manpage/doc generation
* Fri Sep  7 2012 vuntz@suse.com
- Add libcap-devel BuildRequires since it's needed to enable VirtFS
  support.
- Add libcap-ng-devel BuildRequires to be able to drop privileges
  in qemu-bridge-helper.
- Add libjpeg-devel and libpng-devel BuildRequires to enable VNC
  jpeg and png support.
- Pass --enable-virtfs to configure to detect changes that will
  cause VirtFS support to not be built.
* Wed Sep  5 2012 afaerber@suse.de
- update to v1.2.0: see http://wiki.qemu.org/ChangeLog/1.2
  * adapt update_git.sh script
  * this includes the fix for CVE-2012-3515
* Wed Sep  5 2012 afaerber@suse.de
- add bounds checks in console VT100 emulation (bnc#777084,
  CVE-2012-3515)
* Wed Sep  5 2012 afaerber@suse.de
- update to v1.2.0-rc3: http://wiki.qemu.org/ChangeLog/1.2
  * adapt update_git.sh script
- add two patches from kvm for SUSE Studio, forwarded by Dinar:
  * add -nooutgoing option for slirp networking
  * add allowed-connections= and password-file= options for VNC
* Tue Aug 28 2012 afaerber@suse.de
- update to v1.2.0-rc1: http://wiki.qemu.org/ChangeLog/1.2
  * adapt update_git.sh script
* Tue Aug 21 2012 agraf@suse.com
- fix getdent in linux-user
- fix statfs in linux-user
- disable FIEMAP support in linux-user
* Tue Aug 21 2012 agraf@suse.com
- add support for FS_IOC_[G|S]ETFLAG ioctls
* Fri Aug 17 2012 afaerber@suse.de
- update to v1.2.0-rc0: http://wiki.qemu.org/ChangeLog/1.2
  * adapt update_git.sh script
  * package new qemu-or32 executable into linux-user
* Wed Aug 15 2012 dmueller@suse.com
- avoid installing qemu on a qemu build
* Tue Jul 31 2012 agraf@suse.com
- add hijack for /proc/cpuinfo
- add compatibility patch for glibc 2.16 (Thanks AJ!)
* Wed Jul 11 2012 agraf@suse.com
- fix segfault even better
* Wed Jul 11 2012 agraf@suse.com
- linux-user: improve locking even across tb flushes
* Tue Jul 10 2012 agraf@suse.com
- linux-user: pin multi-threaded applications to a single host cpu
* Tue Jul 10 2012 agraf@suse.com
- linux-user: improve lock
* Thu Jul  5 2012 agraf@suse.com
- linux-user: add tcg lock for multi-threaded guest programs
* Mon Jun 25 2012 afaerber@suse.de
- include qemu-img and qemu-nbd in qemu-tools package
* Mon Jun 25 2012 afaerber@suse.de
- add qemu-guest-agent package
* Mon Jun 25 2012 agraf@suse.com
- add fix for segv in linux-user (fixes java)
* Tue Jun 19 2012 afaerber@suse.de
- revert LIBRARY_PATH patch by Alex
* Mon Jun 18 2012 afaerber@suse.de
- add patch by Alex to force gcc being emulated for LIBRARY_PATH
* Mon Jun 18 2012 dvaleev@suse.com
- add libfdt1 to buildrequires on ppc
* Thu Jun 14 2012 afaerber@suse.de
- revert qemu-bridge-helper mode change (cf. BNC#765948)
* Wed Jun 13 2012 brogers@suse.com
- don't use permissions.d
* Wed Jun 13 2012 agraf@suse.com
- Add fix for bogus losetup ioctl to not warn us about it
* Sat Jun  9 2012 brogers@suse.com
- Set up bridge helper as a setuid executable
* Sat Jun  9 2012 afaerber@suse.de
- Add patch for specifying --libexecdir
  * Fixes having to specify the path to qemu-bridge-helper manually
- Drop superfluous `make qemu-img`
- Move qemu-bridge-helper to its own subpackage, to share with KVM
* Tue Jun  5 2012 afaerber@suse.de
- Update to v1.1.0: see http://wiki.qemu.org/ChangeLog/1.1
  * Adapt update_git.sh script to new tag
* Fri Jun  1 2012 afaerber@suse.de
- Update to v1.1.0-rc4: see http://wiki.qemu.org/ChangeLog/1.1
  * Adapt update_git.sh script to new tag
* Thu May 31 2012 agraf@suse.com
- fix /proc/self/maps faking with secondary threads
* Wed May 30 2012 agraf@suse.de
- revert commit a84fac1426 to fix automoc4 for arm
* Tue May 29 2012 agraf@suse.com
- remove manual interp prefix, which made all linux-user binaries
  point to an i386 LD_PREFIX path
* Tue May 29 2012 agraf@suse.com
- add patch to fix building of go (swi in thumb ignores imm)
* Tue May 22 2012 afaerber@suse.de
- Update to v1.1.0-rc3; see http://wiki.qemu.org/ChangeLog/1.1
  * Adapt update_git.sh script to new tag
* Thu May 17 2012 afaerber@suse.de
- Update to v1.1.0-rc2; see http://wiki.qemu.org/ChangeLog/1.1
  * Adapt update_git.sh script to new tag
* Wed May 16 2012 afaerber@suse.de
- Remove workaround for forcing cURL detection
* Sun May 13 2012 afaerber@suse.de
- Update to v1.1-rc1: see http://wiki.qemu.org/ChangeLog/1.1
  * Add workaround for 11.4: add -lrt after -lglib-2.0
- Adapt update_git.sh script to new tag
* Tue May  8 2012 afaerber@suse.de
- Update to v1.1-rc0; see http://wiki.qemu.org/ChangeLog/1.1
  * Do not try to mv qemu, the executable is gone, there's a dir now
  * Do not try to mv */qemu-*, there's a pc-bios/qemu-icon.bmp
  * Be more specific in temporary copying of files to dynamic/
  * Drop duplicate %%_bindir/qemu_binfmt_misc.sh, only in sbin now
  * Use dynamic version of new executable qemu-bridge-helper
- Update update_git.sh script to pull patches from suse-1.1 branch
* Mon May  7 2012 afaerber@suse.de
- Update URL to project homepage
* Mon May  7 2012 agraf@suse.com
- compile fixes for ppc64abi32
- fix update_fit.sh script if wc -c is 40
- fix stale tbs after mmap
* Wed Mar  7 2012 agraf@suse.com
- cut off patch file names after 40 bytes, support older git versions
  for update_git.sh
* Mon Mar  5 2012 agraf@suse.com
- resolve VMAs downwards, fixes arm git build for real
* Sat Mar  3 2012 agraf@suse.com
- update update_git.sh for 1.0.1
- add fixes for reserved_va mmap(NULL) case, fixes git build on arm
* Tue Feb 28 2012 idonmez@suse.com
- Enable spice support
* Wed Feb 22 2012 brogers@suse.com
- update to latest stable release: 1.0.1. For changes see:
  http://wiki.qemu.org/ChangeLog/1.0
* Mon Feb  6 2012 brogers@suse.com
- avoid buffer overflow in e1000 device emulation (bnc#740165)
* Mon Feb  6 2012 agraf@suse.de
- qemu-user: fix fallocate
* Fri Feb  3 2012 agraf@suse.de
- qemu-user: fix previous patch
* Thu Feb  2 2012 agraf@suse.de
- qemu-user: add host binary support for binfmt
* Thu Feb  2 2012 agraf@suse.de
- qemu-user: add fix for /proc/self/maps to only show mappable regions
* Wed Feb  1 2012 agraf@suse.de
- qemu-user: add wrapper for BLKBSZGET
* Tue Jan 31 2012 agraf@suse.de
- qemu-user: add fix for deadloop with ulimit constraints
- qemu-user: add device mapper and loopback ioctls, enabling kpatkx
* Tue Jan 10 2012 agraf@suse.de
- backport all ppc-1.0 stable patches to our package
- add fix for PPC HV KVM on 3.1
* Tue Jan 10 2012 agraf@suse.de
- add fix for PPC in HV KVM mode
* Fri Jan  6 2012 agraf@suse.de
- add workaround for KVM on PPC to support hugetlbfs
- add upstream patches to fake $0 in /proc/self/maps (fixes ps)
* Wed Dec 14 2011 agraf@suse.com
- add update_git.sh to simplify maintenance of the tree
- use -R automatically in linux-user to gain more address space
- drop MAP_32BIT patch (deprecated by the one above)
* Mon Dec 12 2011 agraf@suse.com
- add linker patch for ppc hosts
* Mon Dec 12 2011 agraf@suse.com
- remove exclusivearch - we should be able to build everywhere!
* Sun Dec 11 2011 agraf@suse.com
- fix compilation for 11.4
* Sun Dec 11 2011 agraf@suse.com
- fix compilation on sle11
- add %%fdupes for duplicate file checking
* Sun Dec 11 2011 agraf@suse.com
- update to 1.0 from upstream. for changelogs please see:
  - 0.14 -> 0.15: http://wiki.qemu.org/ChangeLog/0.15
  - 0.15 -> 1.0:  http://wiki.qemu.org/ChangeLog/1.0
- the binary "qemu" is now called qemu-system-i386
* Tue Nov  8 2011 agraf@suse.com
- only ignore specific timer_create syscall, no all of them
* Sun Nov  6 2011 agraf@suse.com
- don't print out unsupported syscalls but rather silently ignore
  them - some programs catch stderr and get confused
* Thu Nov  3 2011 adrian@suse.de
- add a number of patches to fake /proc entries (from Alex)
* Fri Oct 28 2011 adrian@suse.de
- split linux-user binaries into qemu-linux-user package
  (this static package can be used in cross build in OBS)
* Tue Oct 18 2011 adrian@suse.de
- add xattr support for openSUSE 12.1
- added syscall numbers to fix prlimit64 support (from upstream)
* Wed Oct  5 2011 joop.boonen@opensuse.org
- Fixed glibc arm build issue according to
  https://bugs.meego.com/show_bug.cgi?format=multiple&id=20418
  added a reverse patch
* Tue Oct  4 2011 adrian@suse.de
- add binfmt misc handler binary patch from alex. This is a handler
  which calls qemu with -0 argument and right parameters to avoid
  broken runs due to different set $0 as on real hardware
- move arch specific patches at the end to avoid arch dep conflicts
  with other patches
- enable linux-user build to get binfmt handler
* Thu Sep 29 2011 adrian@suse.de
- add 0021-implement-prlimit64-syscall.patch from upstream to avoid
  output and confused configure scripts when testing gcc on armv7hl
- add 0022-fixing-smp-races.patch from alex to fix some SMP races
* Tue Sep 27 2011 adrian@suse.de
- add 0020-linux-user-implement-reboot-syscall.patch from alex
- extend 0010 mmap patch for files
* Mon Sep 26 2011 adrian@suse.de
- add fix from alex for openat file permission handling
* Mon Jun 20 2011 aj@suse.de
- Add BuildRequires: glibc-devel-static
* Thu May 19 2011 andrea@opensuse.org
- removed un-required vde buildrequires
* Tue May 17 2011 brogers@novell.com
- build correctly for SLE11
- reset revision to zero after version change
* Wed May 11 2011 andrea@opensuse.org
- added 0018-qemu-0.14.1-mcast-udp.patch to enable multicast and
  udp tunneling support
* Tue May 10 2011 andrea@opensuse.org
- New Upstream version 0.14.1
    See http://wiki.qemu.org/Changelog/0.14 for changes in 0.14
- removed 0014-qemu-0.11-git-ioctl_mount.patch not required anymore
- enabled vde support
- some spec clean up
* Mon Feb 21 2011 coolo@novell.com
- disable parallel build for a while, it takes too much memory
  for our workers
* Thu Feb 10 2011 uli@suse.de
- update -> 0.14.0-rc1
  See http://wiki.qemu.org/Changelog/0.14 for changes in 0.14.
  (There is no such page for the 0.13 tree. The ChangeLog file ends at
  0.12.0, http://wiki.qemu.org/ChangeLog at 0.12.5 and suggests to look at
  the git log. High-level information of what has changed in 0.13 is
  apparently not available.)
* Fri May 28 2010 uli@suse.de
- update -> 0.12.4
  see http://wiki.qemu.org/ChangeLog for changes
* Wed May 19 2010 brogers@novell.com
- co-exist with virt-utils, which provides some qemu components
  (bnc#586115)
* Sun Apr 18 2010 crrodriguez@opensuse.org
- requires timezone
* Mon Oct  5 2009 uli@suse.de
- update -> 0.11.0 final
  - mips: fix conditional move off fp conditions codes (Nath Froyd)
  - fix migration to obey -S (Paolo Bonzini)
  - remove pc-0-10 machine type (Mark McLoughlin)
  - vnc: fix copyrect screen corruption (Gerd Hoffman)
  - fix vm state change handlers running order (Markus Armbruster)
  - e1000: fix eerc and ics emulation (Bill Paul)
  - fix sdl zooming with pl110 (Blue Swirl)
  - sparc64: flush pending conditional evaluations (Igor Kovalenko)
  - esp: fix interrupt register read (Blue Swirl)
  - option rom makefile fixes (Paul Brook)
  - fix sparse warnings (Blue Swirl)
  - fix symfind (Laurent Desnogues)
  - win32: fix default prefix (Stefan Weil)
  - fix checksum writing in signboot (Alex Graf)
  - fix sdl window resize (Stefano Stabellini)
  - do not resize the screen on hw_invalidate (Stefano Stabellini)
  - Add checks for -smbios option (Beth Kon)
  - fix do_set_link (Luiz Capitulino)
  - fix do_commit behavior (Luiz Capitulino)
  - make windows notice media change (Gleb Natapov)
  - check for PR_SET_NAME being defined (Nathan Froyd)
  - fix migration for ide devices (Anthony Liguori)
  - Use correct depth in vmware vga (Reimar Doffiner)
  - support 32bpp cursors in sdl (Reimar Doffinger)
  - fix device name completion for eject (Blue Swirl)
  - make screendump use DisplayState properly (Stefano Stabellini)
  - fix autostart with live migration (Avi Kivity)
  - fix detached migration with exec (Chris Lalancette)
  - fix segv when changing vnc password in sdl (Zach Amsden)
  - fix vnc password clearing with stdio monitor (Zach Amsden)
  - clean up VGA type selection (Zach Amsden)
  - add missing linefeed in error message (Stefan Weil)
  - fix rtc polling mode (Bernhard Kauer)
  - qcow2: order concurrent aio requests (Kevin Wolf)
  - qemu-io: port to win32 (Stefan Weil)
  - alpha: fix extlh instruction (Vince Weaver)
  - tcg: fix size of local variables in tcg_gen_bswap64_i64 (Stefan Weil)
  - net: fix send ordering (Jan Kiszka)
  - escc: fix IRQ routing (Aurelien Jarno)
  - versatile: fix Linux task preemption (Aurelien Jarno)
  - curses: reduce memory usage by 250MB (Aurelien Jarno)
- disabled ppc64 host, seems to be very broken
* Thu Sep 10 2009 uli@suse.de
- package missing tools (qemu-nbd, qemu-io)
* Thu Sep  3 2009 uli@suse.de
- disable S/390 target on 32-bit platforms
* Tue Aug 11 2009 uli@suse.de
- update -> 0.11.0 RC1
  - add machine aliasing support (Mark McLoughlin)
  - add getfd/closefd monitor commands (Mark McLoughlin)
  - fix live migration (various)
  - HPET fixes (various)
  - slirp: fix guestfwd for incoming data (Jan Kiszka)
  - fix qemu-img convert to copy unallocated parts of the image
    (Akkarit Sangpetch)
  - vmdk: fix backing file handling (Kevin Wolf)
  - scsi: add save/restore support (Nolan Leake)
  - e1000: ignore reset command (Kevin Wolf)
  - allow monitor interaction when using -incoming exec: (Chris Lalancette)
  - fix -net socket,listen (Jan Kiszka)
  - serial: fix lost characters after sysrq (Jason Wessel)
  - lsi53c895a: Implement additional registers (Sebastian Herbszt)
  - i386: fix cpu reset (Nitin Kamble)
  - Prevent CD-ROM eject while device is locked (Mark McLoughlin)
  - Fix memory leaks (various)
  - Reset PS2 keyboard/mouse on reset (Dinesh Subraveti)
  - Fix -initrd with > 4GB guests (Glauber Costa)
  - qcow2 fixes (various)
  - Enable power button event generation (Gleb Natapov)
  - fix serial option in -drive
  - Fix non-ACPI timer interrupt routing (Beth Kon)
  - Fix keyboard mapping on newer Xords with non-default keymaps (balrog)
  - Make PCI config status register read-only (Anthony Liguori)
  - Fix crash on resolution change -> screen dump -> vga redraw (Avi Kivity)
  - fix savevm/loadvm (various)
  - virtio: fix guest oops with 2.6.25 kernels (Rusty Russell)
  - SH4: add support for -kernel (Takashi Yoshii, Aurelien Jarno)
  - hotplug: fix closing of char devices (Jan Kiszka)
  - hotplug: remove incorrect check for device name (Eduardo Habkost)
  - fix VGA regression (malc)
- added S/390 host and target
- maintaining patches in git
- several linux-user improvements for compatibility and speed
- enabled curl block driver, VNC TLS and SASL, AIO, and Bluetooth
* Wed Apr 15 2009 ro@suse.de
- buildfix: refresh patches
* Fri Mar 27 2009 crrodriguez@suse.de
- BuildRequire zlib-devel-static
* Tue Mar 24 2009 uli@suse.de
- update -> release (0.10.1)
  - fixes bnc #435135 (CVE-2008-4539, SWAMP #22179), #362956
    (CVE-2008-1945, CVE-2008-0928), #461565 (CVE-2008-2382)
  - Kernel Virtual Machine acceleration support
  - GDB XML register description support
  - Intel e1000 emulation
  - HPET emulation
  - VirtIO paravirtual device support
  - Marvell 88w8618 / MusicPal emulation
  - Nokia N-series tablet emulation / OMAP2 processor emulation
  - PCI hotplug support
  - Curses display support
  - qemu-nbd utility to mount supported block formats
  - Altivec support in PPC emulation and new firmware (OpenBIOS)
  - Multiple VNC clients are now supported
  - TLS encryption is now supported in VNC
  - MIPS Magnum R4000 machine (Hervé Poussineau)
  - Braille support (Samuel Thibault)
  - Freecom MusicPal system emulation (Jan Kiszka)
  - OMAP242x and Nokia N800, N810 machines (Andrzej Zaborowski)
  - EsounD audio driver (Frederick Reeve)
  - Gravis Ultrasound GF1 sound card (Tibor "TS" Schütz)
  - Many, many, bug fixes and new features
  - virtio-net: allow masking of notifications on empty queue (Alex
    Williamson)
  - e1000: fix rx descriptor low threshold logic (Alex Willaimson)
  - x86 tcg: add NULL checks to lsl instruction (Jan Kiszka)
  - kvm vga: fix screen corruption with -std-vga and Windows (Avi Kivity)
  - kvm vga: fix screen corruption with Ubuntu installations (Glauber Costa)
  - virtio-net: check right return size on sg list (Alex Williamson)
  - Make qemu_announce_self handle holes (live migration after hotplug)
    (Marcelo Tosatti)
  - Revert r6804-r6808 (qcow2 allocation info).  This series of changes
    added a high cost to startup for large qcow2 images (Anthony Liguori)
  - qemu-img: fix help message (Aurelien Jarno)
  - Fix build for non-default installs of SDL (Anthony Liguori)
  - Fix race condition in env->interrupt_request.  When using TCG and a
    dynticks host timer, this condition could cause TCG to get stuck in an
    infinite loop (Aurelien Jarno)
  - Fix reading encrypted hard disk passwords during early startup (Jan
    Kiszka)
  - Fix encrypted disk reporting in 'info block' (Jan Kiszka)
  - Fix console size with tiny displays (MusicPal) (Jan Kiszka)
  - Improve error handling in bdrv_open2 (Jan Kiszka)
  - Avoid leaking data in mux'ed character devices (Jan Kiszka)
  - Fix initial character device reset (no banner in monitor) (Jan Kiszka)
  - Fix cpuid KVM crash on i386 host (Lubomir Rintel)
  - Fix SLES10sp2 installation by adding ISTAT1 register to LSI SCSI
    emulation (Ryan Harper)
- s390x TCG backend
- i386-linux-user NPTL support
* Wed Oct  8 2008 uli@suse.de
- update -> current SVN
  - fixes PPC hosts
  - fixes SH4 targets (reenabled), adds NPTL support
  - SSSE3, SSE4.* instructions, "unreal" mode fixed
  - Alpha target converted to TCG
  - live migration
  - Bluetooth emulation
  - SCSI tape support
  - OpenBIOS binary updated
  - UHCI emulation rewritten
  - lots of bugfixes
* Mon Aug  4 2008 ro@suse.de
- fix build by adding return statements for two nonvoid functions
  (qemu-nonvoid_return.patch), actually unreached code
* Thu Jul 24 2008 uli@suse.de
- update -> current SVN
  - MIPS Magnum R4000 machine (Hervé Poussineau)
  - Braille support (Samuel Thibault)
  - Freecom MusicPal system emulation (Jan Kiszka)
  - OMAP242x and Nokia N800, N810 machines (Andrzej Zaborowski)
  - EsounD audio driver (Frederick Reeve)
  - Gravis Ultrasound GF1 sound card (Tibor "TS" Schütz)
- added pcap networking
- ditched GCC3 thanks to TCG. Hooray!
  (downside: no PPC and SH4 emulation, no S/390 host)
* Thu May 15 2008 cthiel@suse.de
- use lsilogic instead of buslogic for SCSI VMDK images
* Thu May  8 2008 uli@suse.de
- add format option to disable autodetection of disk image format
  (bnc#380828)
* Fri Apr 25 2008 uli@suse.de
- revert secfix (causes data corruption, no known good patch
  available yet)
- support creation of SCSI VMDK images
* Tue Mar 11 2008 uli@suse.de
- secfix (unchecked block read/write vulnerability, bug #362956)
* Thu Jan 17 2008 uli@suse.de
- update -> 0.9.1
  - TFTP booting from host directory (Anthony Liguori, Erwan Velu)
  - Tap device emulation for Solaris (Sittichai Palanisong)
  - Monitor multiplexing to several I/O channels (Jason Wessel)
  - ds1225y nvram support (Herve Poussineau)
  - CPU model selection support (J. Mayer, Paul Brook, Herve Poussineau)
  - Several Sparc fixes (Aurelien Jarno, Blue Swirl, Robert Reif)
  - MIPS 64-bit FPU support (Thiemo Seufer)
  - Xscale PDA emulation (Andrzej Zaborowski)
  - ColdFire system emulation (Paul Brook)
  - Improved SH4 support (Magnus Damm)
  - MIPS64 support (Aurelien Jarno, Thiemo Seufer)
  - Preliminary Alpha guest support (J. Mayer)
  - Gumstix boards: connex and verdex emulation (Thorsten Zitterell)
  - Intel mainstone II board emulation (Armin Kuster)
  - VMware SVGA II graphics card support (Andrzej Zaborowski)
- revert s390-dis.c to last GPLv2 version
- reenabled qemu-system-ppcemb
* Mon Dec 10 2007 uli@suse.de
- fixed open() usage with O_CREAT
* Tue Nov 13 2007 uli@suse.de
- update -> current CVS:
  - Read-only support for Parallels disk images (Alex Beregszaszi)
  - CRIS emulation (Edgar E. Iglesias)
  - SPARC32PLUS execution support (Blue Swirl)
  - MIPS mipssim pseudo machine (Thiemo Seufer)
  - Strace for Linux userland emulation (Stuart Anderson, Thayne Harbaugh)
  - OMAP310 MPU emulation plus Palm T|E machine (Andrzej Zaborowski)
  - ARM v6, v7, NEON SIMD and SMP emulation (Paul Brook/CodeSourcery)
* Thu Oct 11 2007 ro@suse.de
- hack to fix build: undef DEBUG_BLOCK for now
* Fri Sep 28 2007 agraf@suse.de
- fix SVM support (Alexander Graf)
* Thu Sep 13 2007 agraf@suse.de
- add SVM emulation support (Alexander Graf)
- fix a misassumption in the s390 fix (uli)
- allow more IDE power management (Ben Guthro)
- log any I/O error and perform automatic read retry for CDrom (Ben Guthro)
- fix Coherent guest support (Jan Jezabek)
- fix several Darwin guest issues (Filip Navara)
* Mon Aug 20 2007 agraf@suse.de
- fix ATAPI bug when using libata (Brandon Philips)  (#291775)
* Sat Aug 11 2007 olh@suse.de
- disable only SNDRV_SB_CSP_IOCTL_LOAD_CODE for _IOC_SIZEBITS < 14
* Thu Aug  9 2007 olh@suse.de
- disable some alsa SB ioctl declarations
* Mon Aug  6 2007 olh@suse.de
- remove inclusion of linux/compiler.h
* Mon Jul 30 2007 uli@suse.de
- fixed for S/390
* Tue Jul 10 2007 schwab@suse.de
- Add (incomplete) m68k emulation.
* Mon Jul  9 2007 agraf@suse.de
- included alsa support in qemu-user
- update to current cvs
  - TFTP booting from host directory (Anthony Liguori, Erwan Velu)
  - Tap device emulation for Solaris (Sittichai Palanisong)
  - Monitor multiplexing to several I/O channels (Jason Wessel)
  - ds1225y nvram support (Herve Poussineau)
  - CPU model selection support (J. Mayer, Paul Brook, Herve Poussineau)
  - Several Sparc fixes (Aurelien Jarno, Blue Swirl)
  - MIPS 64-bit FPU support (Thiemo Seufer)
  - Xscale PDA emulation (Andrzei Zaborowski)
  - ColdFire system emulation (Paul Brook)
  - Improved SH4 support (Magnus Damm)
  - MIPS64 support (Aurelien Jarno, Thiemo Seufer)
  - Preliminary Alpha guest support (J. Mayer)
  - IPC fixes
* Wed Jun 20 2007 agraf@suse.de
- applied proper fix for x86_64 and the MAP_32BIT flag
* Wed Jun 20 2007 uli@suse.de
- added secfixes (bug #252519)
* Thu Jun 14 2007 agraf@suse.de
- made wine work (set FS register to 0 on init)
- suppressed robust_list warnings
* Wed Jun 13 2007 agraf@suse.de
- made flash player 9 work on ppc
- fixed FUTEX_WAKE_OP on machines where endianness differs
- made mmap on x86_64 use the MAP_32BIT flag
- removed a useless spin_lock
- removed an annoying debug message for forking
- implemented sched_getaffinity syscall
- fixed configure call so it takes gcc3 again
* Wed Jun 13 2007 uli@suse.de
- support "vga=" parameter (Pascal Terjan)
- direct jump support for x86-64 (Gwenole Beauchesne)
- fix chaining of CPU instances (Gwenole Beauchesne)
- live migration support (Anthony Liguori)
- fix fprem/fprem1 insns (Julian Seward)
- KVM support
- Z80/ZX Spectrum emulation (Stuart Brady)
- GCC4 support postponed (breaks x86-64 on i386)
* Mon Jun 11 2007 agraf@suse.de
- implemented TLS support on i386 so qemu-user can be used to run
  with current libc versions (partly done by David Woodhouse,
  fixed by Alexander Graf)
- added a dirty hack for an mmap page table bug on qemu-user x86
  emulation
- disable AF_NETLINK in qemu-user (endianness problems)
- applied fast path mangling patch from Kirill A. Shutemov
- applied strace patch for debugging (by Stuart R. Anderson)
* Wed Apr  4 2007 agraf@suse.de
- fixed initrd loading on x86
* Thu Mar 29 2007 ro@suse.de
- added bison to BuildRequires
* Tue Feb 20 2007 uli@suse.de
- added better fix by Robert Schiele (bug #241950)
- update -> 0.9.0
  - Support for relative paths in backing files for disk images
  - Async file I/O API
  - New qcow2 disk image format
  - Support of multiple VM snapshots
  - Linux: specific host CDROM and floppy support
  - SMM support
  - Moved PCI init, MP table init and ACPI table init to Bochs BIOS
  - Support for MIPS32 Release 2 instruction set (Thiemo Seufer)
  - MIPS Malta system emulation (Aurelien Jarno, Stefan Weil)
  - Darwin userspace emulation (Pierre d'Herbemont)
  - m68k user support (Paul Brook)
  - several x86 and x86_64 emulation fixes
  - Mouse relative offset VNC extension (Anthony Liguori)
  - PXE boot support (Anthony Liguori)
  - '-daemonize' option (Anthony Liguori)
* Tue Feb  6 2007 uli@suse.de
- added fix by Robert Schiele to work without usbdevfs
  (bug #241950)
* Thu Feb  1 2007 ro@suse.de
- remove -fstack-protector from CFLAGS
* Fri Oct 27 2006 schwab@suse.de
- Fix char signedness.
* Mon Sep 11 2006 uli@suse.de
- re-added ARM FPE patch
- fixed SPARC on PPC
- install missing openbios-sparc32
* Sat Sep  9 2006 dmueller@suse.de
- update to 0.8.2:
  - ACPI support
  - PC VGA BIOS fixes
  - MIPS FPU support (Marius Groeger)
  - PPC breakpoints and single stepping (Jason Wessel)
  - USB updates (Paul Brook)
  - UDP/TCP/telnet character devices (Jason Wessel)
  - Windows sparse file support (Frediano Ziglio)
  - RTL8139 NIC TCP segmentation offloading (Igor Kovalenko)
  - PCNET NIC support (Antony T Curtis)
  - Support for variable frequency host CPUs
  - Workaround for win32 SMP hosts
  - Support for AMD Flash memories (Jocelyn Mayer)
  - Audio capture to WAV files support (malc)
* Tue May 30 2006 uli@suse.de
- fixed to build on PPC with new glibc
* Mon May 29 2006 uli@suse.de
- fixed to build with new kernel headers
* Wed Mar  8 2006 uli@suse.de
- split giant patch
- added NWFPE glue code fix
* Tue Mar  7 2006 schwab@suse.de
- More fixes for ia64 port.
* Mon Mar  6 2006 schwab@suse.de
- Remove obsolete hunk from ia64 patch.
* Wed Jan 25 2006 mls@suse.de
- converted neededforbuild to BuildRequires
* Tue Dec 20 2005 uli@suse.de
- update -> 0.8.0
* Thu Dec  8 2005 uli@suse.de
- update to current CVS (MIPS userspace, ARM system, SMP, USB,
  NX, VLAN, serial, parallel, ES1370, ALSA backend)
- build i386 and AMD64 emus with kqemu support if possible
- install missing PPC video driver, SPARC boot ROM
- install missing keymaps
* Mon Nov  7 2005 uli@suse.de
- updated linker scripts for new binutils release
* Sat Sep 17 2005 dmueller@suse.de
- update to 0.7.2
* Mon Aug 15 2005 schwab@suse.de
- Don't package /emul/ia32-linux on ia64.
* Mon Aug 15 2005 schwab@suse.de
- Fix compilation on ia64.
* Mon Aug  1 2005 uli@suse.de
- update -> 0.7.1
- enabled x86_64-system
* Mon Jul 11 2005 uli@suse.de
- update -> CVS (MIPS emulation)
- build with throwaway GCC3 (still no GCC4-compatible QEMU in
  sight)
- enabled armeb-user
- link *-user statically (necessary for chrooting into non-native
  systems)
- disabled s390, alpha, armv4l build (broken and unmaintained)
- disabled qemu-fast (broken and deprecated)
- disabled i386-user on x86_64 (useless)
- build with %%jobs
* Wed Jun  1 2005 ro@suse.de
- update to 0.7.0, still no success with gcc-4
* Thu Jan 27 2005 uli@suse.de
- baselibs-x86 link is unnecessary on x86/AMD64
- two ARM emulation fixes (RRX operand fix by Paul Brook, shifter
  carry fix by me)
- SDL 1.2.8 can be linked statically -> enabled qemu-fast
* Mon Nov 29 2004 uli@suse.de
- update -> 0.6.1
- build softmmu binaries on s390 (still doesn't work though)
- pack /emul/ia32-linux directory and symlink from /usr/share/qemu
  so it's possible to use the IA64 baselibs-x86 packages
* Wed Sep 15 2004 uli@suse.de
- removed unnecessary dependency on private glibc symbol
  (bug #44864)
* Fri Aug  6 2004 uli@suse.de
- update -> 0.6.0 (fixes for several OSs, improvements in emulated
  hardware (IDE, PCI, network, VGA, APM, PowerMac), minor fixes,
  tool for conversion of VMware images)
* Tue May 11 2004 uli@suse.de
- update -> 0.5.5 (contains ARM, utime() fixes and several others)
- make uname() return target machine name (fixes config.guess,
  rpm etc.)
- fake sigaltstack() (fixes m4)
- enabled x86-64
* Wed May  5 2004 uli@suse.de
- fixed ARM emulation bug
* Wed Apr 28 2004 uli@suse.de
- build with -fno-unit-at-a-time (fixes PPC system emulator)
* Wed Apr 28 2004 uli@suse.de
- update -> 0.5.4 (runs Linux/PPC, several fixes)
* Mon Apr 19 2004 uli@suse.de
- initial package
