// Copyright 2024 Simo Sorce
// See LICENSE.txt file for terms

//! This module defines PKCS#11 attribute type constants specific to or
//! commonly used within NSS databases, including standard, vendor-defined
//! (NSS), and trust object attributes. It also provides static lists grouping
//! these attributes for different handling purposes (known, authenticated,
//! sensitive, vendor, skippable) and helper functions to check attribute
//! classifications.

use crate::pkcs11::*;

const DEPRECATED_CKA_SECONDARY_AUTH: CK_ULONG = 512;
const DEPRECATED_CKA_AUTH_PIN_FLAGS: CK_ULONG = 513;

/* we use the CKA_NSS prefix instead of the preferred NSS_CKA one
 * to allow for easy search in the NSS codebase */

/* off the regular NSS vendor offset for historical reasons */
const CKA_NSS_DB: CK_ATTRIBUTE_TYPE = 0xD5A0DB00;
const CKA_NSS_TRUST: CK_ATTRIBUTE_TYPE = 0x80000001;

/* NSS Vendor Offset attributes */
const NSS_VENDOR_OFFSET: CK_ULONG = CKA_VENDOR_DEFINED + 0x4E534350;
const CKA_NSS_URL: CK_ATTRIBUTE_TYPE = NSS_VENDOR_OFFSET + 1;
const CKA_NSS_EMAIL: CK_ATTRIBUTE_TYPE = NSS_VENDOR_OFFSET + 2;
const CKA_NSS_SMIME_INFO: CK_ATTRIBUTE_TYPE = NSS_VENDOR_OFFSET + 3;
const CKA_NSS_SMIME_TIMESTAMP: CK_ATTRIBUTE_TYPE = NSS_VENDOR_OFFSET + 4;
const CKA_NSS_PKCS8_SALT: CK_ATTRIBUTE_TYPE = NSS_VENDOR_OFFSET + 5;
const CKA_NSS_PASSWORD_CHECK: CK_ATTRIBUTE_TYPE = NSS_VENDOR_OFFSET + 6;
const CKA_NSS_EXPIRES: CK_ATTRIBUTE_TYPE = NSS_VENDOR_OFFSET + 7;
const CKA_NSS_KRL: CK_ATTRIBUTE_TYPE = NSS_VENDOR_OFFSET + 8;

const CKA_NSS_PQG_COUNTER: CK_ATTRIBUTE_TYPE = NSS_VENDOR_OFFSET + 20;
const CKA_NSS_PQG_SEED: CK_ATTRIBUTE_TYPE = NSS_VENDOR_OFFSET + 21;
const CKA_NSS_PQG_H: CK_ATTRIBUTE_TYPE = NSS_VENDOR_OFFSET + 22;
const CKA_NSS_PQG_SEED_BITS: CK_ATTRIBUTE_TYPE = NSS_VENDOR_OFFSET + 23;
const CKA_NSS_MODULE_SPEC: CK_ATTRIBUTE_TYPE = NSS_VENDOR_OFFSET + 24;
const CKA_NSS_OVERRIDE_EXTENSIONS: CK_ATTRIBUTE_TYPE = NSS_VENDOR_OFFSET + 25;

const CKA_NSS_SERVER_DISTRUST_AFTER: CK_ATTRIBUTE_TYPE = NSS_VENDOR_OFFSET + 35;
const CKA_NSS_EMAIL_DISTRUST_AFTER: CK_ATTRIBUTE_TYPE = NSS_VENDOR_OFFSET + 36;

const NSS_VENDOR_TRUST: CK_ULONG = NSS_VENDOR_OFFSET + 0x2000;

const CKA_TRUST_DIGITAL_SIGNATURE: CK_ATTRIBUTE_TYPE = NSS_VENDOR_TRUST + 1;
const CKA_TRUST_NON_REPUDIATION: CK_ATTRIBUTE_TYPE = NSS_VENDOR_TRUST + 2;
const CKA_TRUST_KEY_ENCIPHERMENT: CK_ATTRIBUTE_TYPE = NSS_VENDOR_TRUST + 3;
const CKA_TRUST_DATA_ENCIPHERMENT: CK_ATTRIBUTE_TYPE = NSS_VENDOR_TRUST + 4;
const CKA_TRUST_KEY_AGREEMENT: CK_ATTRIBUTE_TYPE = NSS_VENDOR_TRUST + 5;
const CKA_TRUST_KEY_CERT_SIGN: CK_ATTRIBUTE_TYPE = NSS_VENDOR_TRUST + 6;
const CKA_TRUST_CRL_SIGN: CK_ATTRIBUTE_TYPE = NSS_VENDOR_TRUST + 7;
const CKA_TRUST_SERVER_AUTH: CK_ATTRIBUTE_TYPE = NSS_VENDOR_TRUST + 8;
const CKA_TRUST_CLIENT_AUTH: CK_ATTRIBUTE_TYPE = NSS_VENDOR_TRUST + 9;
const CKA_TRUST_CODE_SIGNING: CK_ATTRIBUTE_TYPE = NSS_VENDOR_TRUST + 10;
const CKA_TRUST_EMAIL_PROTECTION: CK_ATTRIBUTE_TYPE = NSS_VENDOR_TRUST + 11;
const CKA_TRUST_IPSEC_END_SYSTEM: CK_ATTRIBUTE_TYPE = NSS_VENDOR_TRUST + 12;
const CKA_TRUST_IPSEC_TUNNEL: CK_ATTRIBUTE_TYPE = NSS_VENDOR_TRUST + 13;
const CKA_TRUST_IPSEC_USER: CK_ATTRIBUTE_TYPE = NSS_VENDOR_TRUST + 14;
const CKA_TRUST_TIME_STAMPING: CK_ATTRIBUTE_TYPE = NSS_VENDOR_TRUST + 15;
const CKA_TRUST_STEP_UP_APPROVED: CK_ATTRIBUTE_TYPE = NSS_VENDOR_TRUST + 16;

const CKA_CERT_SHA1_HASH: CK_ATTRIBUTE_TYPE = NSS_VENDOR_TRUST + 100;
const CKA_CERT_MD5_HASH: CK_ATTRIBUTE_TYPE = NSS_VENDOR_TRUST + 101;

const NSS_KA_LEN: usize = 121;

/// Static array listing all standard and NSS vendor-specific attributes
/// known and potentially stored by this NSS backend implementation.
pub static NSS_KNOWN_ATTRIBUTES: [CK_ATTRIBUTE_TYPE; NSS_KA_LEN] = [
    CKA_CLASS,
    CKA_TOKEN,
    CKA_PRIVATE,
    CKA_LABEL,
    CKA_APPLICATION,
    CKA_VALUE,
    CKA_OBJECT_ID,
    CKA_CERTIFICATE_TYPE,
    CKA_ISSUER,
    CKA_SERIAL_NUMBER,
    CKA_AC_ISSUER,
    CKA_OWNER,
    CKA_ATTR_TYPES,
    CKA_TRUSTED,
    CKA_CERTIFICATE_CATEGORY,
    CKA_JAVA_MIDP_SECURITY_DOMAIN,
    CKA_URL,
    CKA_HASH_OF_SUBJECT_PUBLIC_KEY,
    CKA_HASH_OF_ISSUER_PUBLIC_KEY,
    CKA_CHECK_VALUE,
    CKA_KEY_TYPE,
    CKA_SUBJECT,
    CKA_ID,
    CKA_SENSITIVE,
    CKA_ENCRYPT,
    CKA_DECRYPT,
    CKA_WRAP,
    CKA_UNWRAP,
    CKA_SIGN,
    CKA_SIGN_RECOVER,
    CKA_VERIFY,
    CKA_VERIFY_RECOVER,
    CKA_DERIVE,
    CKA_START_DATE,
    CKA_END_DATE,
    CKA_MODULUS,
    CKA_MODULUS_BITS,
    CKA_PUBLIC_EXPONENT,
    CKA_PRIVATE_EXPONENT,
    CKA_PRIME_1,
    CKA_PRIME_2,
    CKA_EXPONENT_1,
    CKA_EXPONENT_2,
    CKA_COEFFICIENT,
    CKA_PUBLIC_KEY_INFO,
    CKA_PRIME,
    CKA_SUBPRIME,
    CKA_BASE,
    CKA_PRIME_BITS,
    CKA_SUB_PRIME_BITS,
    CKA_VALUE_BITS,
    CKA_VALUE_LEN,
    CKA_EXTRACTABLE,
    CKA_LOCAL,
    CKA_NEVER_EXTRACTABLE,
    CKA_ALWAYS_SENSITIVE,
    CKA_KEY_GEN_MECHANISM,
    CKA_MODIFIABLE,
    CKA_EC_PARAMS,
    CKA_EC_POINT,
    DEPRECATED_CKA_SECONDARY_AUTH,
    DEPRECATED_CKA_AUTH_PIN_FLAGS,
    CKA_ALWAYS_AUTHENTICATE,
    CKA_WRAP_WITH_TRUSTED,
    CKA_HW_FEATURE_TYPE,
    CKA_RESET_ON_INIT,
    CKA_HAS_RESET,
    CKA_PIXEL_X,
    CKA_PIXEL_Y,
    CKA_RESOLUTION,
    CKA_CHAR_ROWS,
    CKA_CHAR_COLUMNS,
    CKA_COLOR,
    CKA_BITS_PER_PIXEL,
    CKA_CHAR_SETS,
    CKA_ENCODING_METHODS,
    CKA_MIME_TYPES,
    CKA_MECHANISM_TYPE,
    CKA_REQUIRED_CMS_ATTRIBUTES,
    CKA_DEFAULT_CMS_ATTRIBUTES,
    CKA_SUPPORTED_CMS_ATTRIBUTES,
    CKA_WRAP_TEMPLATE,
    CKA_UNWRAP_TEMPLATE,
    CKA_NSS_TRUST,
    CKA_NSS_URL,
    CKA_NSS_EMAIL,
    CKA_NSS_SMIME_INFO,
    CKA_NSS_SMIME_TIMESTAMP,
    CKA_NSS_PKCS8_SALT,
    CKA_NSS_PASSWORD_CHECK,
    CKA_NSS_EXPIRES,
    CKA_NSS_KRL,
    CKA_NSS_PQG_COUNTER,
    CKA_NSS_PQG_SEED,
    CKA_NSS_PQG_H,
    CKA_NSS_PQG_SEED_BITS,
    CKA_NSS_MODULE_SPEC,
    CKA_NSS_OVERRIDE_EXTENSIONS,
    CKA_NSS_SERVER_DISTRUST_AFTER,
    CKA_NSS_EMAIL_DISTRUST_AFTER,
    CKA_TRUST_DIGITAL_SIGNATURE,
    CKA_TRUST_NON_REPUDIATION,
    CKA_TRUST_KEY_ENCIPHERMENT,
    CKA_TRUST_DATA_ENCIPHERMENT,
    CKA_TRUST_KEY_AGREEMENT,
    CKA_TRUST_KEY_CERT_SIGN,
    CKA_TRUST_CRL_SIGN,
    CKA_TRUST_SERVER_AUTH,
    CKA_TRUST_CLIENT_AUTH,
    CKA_TRUST_CODE_SIGNING,
    CKA_TRUST_EMAIL_PROTECTION,
    CKA_TRUST_IPSEC_END_SYSTEM,
    CKA_TRUST_IPSEC_TUNNEL,
    CKA_TRUST_IPSEC_USER,
    CKA_TRUST_TIME_STAMPING,
    CKA_TRUST_STEP_UP_APPROVED,
    CKA_CERT_SHA1_HASH,
    CKA_CERT_MD5_HASH,
    CKA_NSS_DB,
    CKA_ENCAPSULATE,
    CKA_DECAPSULATE,
];

/// Static array listing attributes that NSS protects with an internal
/// signature mechanism (see `storage/nssdb/ci.rs`) when stored in the
/// key database (keyN.db).
///
/// NSS has a hardcoded list of attributes that are authenticated,
/// some are vendor defined attributes */
pub static AUTHENTICATED_ATTRIBUTES: [CK_ATTRIBUTE_TYPE; 10] = [
    CKA_MODULUS,
    CKA_PUBLIC_EXPONENT,
    CKA_CERT_SHA1_HASH,
    CKA_CERT_MD5_HASH,
    CKA_TRUST_SERVER_AUTH,
    CKA_TRUST_CLIENT_AUTH,
    CKA_TRUST_EMAIL_PROTECTION,
    CKA_TRUST_CODE_SIGNING,
    CKA_TRUST_STEP_UP_APPROVED,
    CKA_NSS_OVERRIDE_EXTENSIONS,
];

/// Static array listing only the NSS vendor-specific attribute types
/// (excluding standard PKCS#11 attributes).
static NSS_VENDOR_ATTRIBUTES: [CK_ATTRIBUTE_TYPE; 36] = [
    CKA_NSS_TRUST,
    CKA_NSS_URL,
    CKA_NSS_EMAIL,
    CKA_NSS_SMIME_INFO,
    CKA_NSS_SMIME_TIMESTAMP,
    CKA_NSS_PKCS8_SALT,
    CKA_NSS_PASSWORD_CHECK,
    CKA_NSS_EXPIRES,
    CKA_NSS_KRL,
    CKA_NSS_PQG_COUNTER,
    CKA_NSS_PQG_SEED,
    CKA_NSS_PQG_H,
    CKA_NSS_PQG_SEED_BITS,
    CKA_NSS_MODULE_SPEC,
    CKA_NSS_OVERRIDE_EXTENSIONS,
    CKA_NSS_SERVER_DISTRUST_AFTER,
    CKA_NSS_EMAIL_DISTRUST_AFTER,
    CKA_TRUST_DIGITAL_SIGNATURE,
    CKA_TRUST_NON_REPUDIATION,
    CKA_TRUST_KEY_ENCIPHERMENT,
    CKA_TRUST_DATA_ENCIPHERMENT,
    CKA_TRUST_KEY_AGREEMENT,
    CKA_TRUST_KEY_CERT_SIGN,
    CKA_TRUST_CRL_SIGN,
    CKA_TRUST_SERVER_AUTH,
    CKA_TRUST_CLIENT_AUTH,
    CKA_TRUST_CODE_SIGNING,
    CKA_TRUST_EMAIL_PROTECTION,
    CKA_TRUST_IPSEC_END_SYSTEM,
    CKA_TRUST_IPSEC_TUNNEL,
    CKA_TRUST_IPSEC_USER,
    CKA_TRUST_TIME_STAMPING,
    CKA_TRUST_STEP_UP_APPROVED,
    CKA_CERT_SHA1_HASH,
    CKA_CERT_MD5_HASH,
    CKA_NSS_DB,
];

/// Checks if an attribute type is an NSS vendor-defined attribute or
/// deprecated and should generally be ignored.
pub fn ignore_attribute(attr: CK_ATTRIBUTE_TYPE) -> bool {
    if NSS_VENDOR_ATTRIBUTES.contains(&attr) {
        return true;
    }
    if attr == DEPRECATED_CKA_SECONDARY_AUTH
        || attr == DEPRECATED_CKA_AUTH_PIN_FLAGS
    {
        return true;
    }
    return false;
}

/// Static array listing attributes that NSS considers sensitive and encrypts
/// when storing private or secret key objects in the key database (keyN.db).
pub static NSS_SENSITIVE_ATTRIBUTES: [CK_ATTRIBUTE_TYPE; 7] = [
    CKA_VALUE,
    CKA_PRIVATE_EXPONENT,
    CKA_PRIME_1,
    CKA_PRIME_2,
    CKA_EXPONENT_1,
    CKA_EXPONENT_2,
    CKA_COEFFICIENT,
];

/// Checks if an attribute type is considered sensitive by NSS (and should be
/// encrypted).
pub fn is_sensitive_attribute(attr: CK_ATTRIBUTE_TYPE) -> bool {
    NSS_SENSITIVE_ATTRIBUTES.contains(&attr)
}

/// Checks if an attribute type is known and potentially stored in the NSS DB.
pub fn is_db_attribute(attr: CK_ATTRIBUTE_TYPE) -> bool {
    NSS_KNOWN_ATTRIBUTES.contains(&attr)
}

/// Static array listing attributes that are typically managed internally
/// (like `CKA_UNIQUE_ID`) or derived/implicit (`CKA_ALLOWED_MECHANISMS`)
/// and are not directly stored as columns in the NSS database tables.
pub static NSS_SKIP_ATTRIBUTES: [CK_ATTRIBUTE_TYPE; 4] = [
    CKA_UNIQUE_ID,
    CKA_COPYABLE,
    CKA_DESTROYABLE,
    CKA_ALLOWED_MECHANISMS,
];

/// Checks if an attribute type is one that should be skipped (not directly
/// stored or retrieved as a column) when interacting with the NSS DB.
pub fn is_skippable_attribute(attr: CK_ATTRIBUTE_TYPE) -> bool {
    NSS_SKIP_ATTRIBUTES.contains(&attr)
}
