# Security Policy

Thank you for your interest in Libauth's security!

## Supported Versions

Only the latest major version is regularly supported.

Critical security updates may be backported to previous versions, but we recommend all users migrate to new major versions within 6 months of release.

## Reporting a Vulnerability

If you have found a security issue in Libauth, please notify us via email at security@bitauth.com.

## Acknowledgements

Thank you to the following contributors for reporting security issues in Libauth.

| Contributor | Issue                                      | Resolution                                                                                     |
| ----------- | ------------------------------------------ | ---------------------------------------------------------------------------------------------- |
| Jim Hamill  | Secp256k1: improper sanitization of inputs | [7fc75c90](https://github.com/bitauth/libauth/commit/7fc75c90be441cf22f3bb7946363e78fa0a61b17) |

## PGP Key

Files and messages may be encrypted with the following PGP key.

**Fingerprint**: `B9BE 5C81 9327 99DD F37D 74B2 97A1 A7DF 9EB6 9AB8`

```pgp
-----BEGIN PGP PUBLIC KEY BLOCK-----
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=37P9
-----END PGP PUBLIC KEY BLOCK-----

```
