import type {
  AuthenticationInstruction,
  Output,
  TransactionCommon,
} from '../lib.js';

export type AuthenticationProgramStateMinimum = {
  /**
   * The full list of instructions to be evaluated by the virtual machine.
   */
  instructions: AuthenticationInstruction[];
  /**
   * Instruction Pointer – the array index of `instructions` that will be read
   * to identify the next instruction. Once `ip` exceeds the last index of
   * `instructions` (`ip === instructions.length`), evaluation is complete.
   */
  ip: number;
};

export type AuthenticationProgramStateStack<StackType = Uint8Array> = {
  /**
   * The stack is the primary workspace of the virtual machine. Most virtual
   * machine operations create, read, update, or delete bytecode values
   * held on the stack.
   */
  stack: StackType[];
};

export type AuthenticationProgramStateAlternateStack<StackType = Uint8Array> = {
  /**
   * The "alternate stack" is separate stack on which `OP_TOALTSTACK` and
   * `OP_FROMALTSTACK` operate in bitcoin virtual machines.
   */
  alternateStack: StackType[];
};

export type AuthenticationProgramStateControlStack<ItemType = boolean> = {
  /**
   * An array of boolean values representing the current execution status of the
   * program. This allows the state to track nested conditional branches.
   *
   * The `OP_IF` and `OP_NOTIF` operations push a new boolean onto the
   * `controlStack`, `OP_ELSE` flips the top boolean, and `OP_ENDIF` removes
   * the top boolean from the `controlStack`.
   *
   * Other instructions are only evaluated if `controlStack` contains no
   * `false` items.
   *
   * A.K.A. `vfExec` in the C++ implementation.
   */
  controlStack: ItemType[];
};

export type AuthenticationProgramStateError = {
  /**
   * If present, the error returned by the most recent virtual machine
   * operation.
   */
  error?: string;
};

/**
 * A complete view of the information necessary to validate a transaction.
 */
export type ResolvedTransactionCommon = {
  sourceOutputs: Output[];
  transaction: TransactionCommon;
};

/**
 * A complete view of the information necessary to validate a specified input in
 * a transaction.
 */
export type AuthenticationProgramCommon = ResolvedTransactionCommon & {
  inputIndex: number;
};

export type AuthenticationProgramStateCodeSeparator = {
  /**
   * The `lastCodeSeparator` indicates the index of the most recently executed
   * `OP_CODESEPARATOR` instruction. In each of the signing serialization
   * algorithms, the `instructions` are sliced at `lastCodeSeparator`, and the
   * subarray is re-encoded. The resulting bytecode is called the
   * `coveredBytecode` (A.K.A. `scriptCode`), and is part of the data hashed to
   * create the signing serialization digest.
   *
   * By default, this is `-1`, which indicates that the whole `instructions`
   * array is included in the signing serialization.
   */
  lastCodeSeparator: number;
};

export type AuthenticationProgramStateSignatureAnalysis = {
  /**
   * An array of the `Uint8Array` values used in signature verification over the
   * course of this program. Each raw signing serialization and data signature
   * message should be pushed to this array in the order it was computed.
   *
   * This property is not used within any {@link AuthenticationVirtualMachine},
   * but it is provided in the program state to assist with analysis. Because
   * these messages must always be computed and hashed during evaluation,
   * recording them in the state does not meaningfully affect performance.
   */
  signedMessages: (
    | {
        /**
         * The final digest signed by the signature. Because this is a
         * transaction signature, the provided `serialization` is hashed using
         * two rounds of sha256 to produce this digest.
         */
        digest: Uint8Array;
        /**
         * The transaction signing serialization generated by the `OP_CHECKSIG`
         * or `OP_CHECKSIGVERIFY` operation.
         */
        serialization: Uint8Array;
      }
    | {
        /**
         * The final digest signed by the signature. Because this is a data
         * signature, the provided `message` is hashed using one round of sha256
         * to produce this digest.
         */
        digest: Uint8Array;
        /**
         * The message provided to the `OP_CHECKDATASIG` or
         * `OP_CHECKDATASIGVERIFY` operation.
         */
        message: Uint8Array;
      }
  )[];
};

export type AuthenticationProgramStateResourceLimits = {
  operationCount: number;
  signatureOperationsCount: number;
};

export type AuthenticationProgramStateTransactionContext = {
  program: AuthenticationProgramCommon;
};

export type AuthenticationProgramStateCommon =
  AuthenticationProgramStateAlternateStack &
    AuthenticationProgramStateCodeSeparator &
    AuthenticationProgramStateControlStack &
    AuthenticationProgramStateError &
    AuthenticationProgramStateMinimum &
    AuthenticationProgramStateResourceLimits &
    AuthenticationProgramStateSignatureAnalysis &
    AuthenticationProgramStateStack &
    AuthenticationProgramStateTransactionContext;
